// Sales System JavaScript
class SalesManager {
    constructor() {
        this.currentInvoiceId = null;
        this.rows = [];
        this.customers = [];
        this.products = [];
        this.currentRow = 0;
        this.selectedSuggestionIndex = -1;
        this.currentProductInput = null;
        this.modalSelectedIndex = -1;
        this.modalProducts = [];
        this.currentModalRow = null;
        this.isScanMode = false;
        this.floatingTotal = null;
        this.allInvoiceIds = []; // Store all invoice IDs for navigation
        this.currentInvoiceIndex = -1; // Current position in the invoice list
        this.isSaving = false; // Prevent multiple saves
        this.init();
    }

    init() {
        this.loadCustomers();
        this.addInitialRow();
        this.bindEvents();
        this.loadRecentSales();
        
        // Create floating total after initial setup
        setTimeout(() => {
            this.createFloatingTotal();
        }, 200);
        
        // Focus on sale counter (rateType) initially
        setTimeout(() => {
            this.focusOnSaleCounter();
        }, 500);
    }

    bindEvents() {
        const self = this;
        
        // Save button
        $('#saveInvoice').on('click', function() {
            self.saveInvoice();
        });
        
        // New invoice button
        $('#newInvoice').on('click', function() {
            self.newInvoice();
            // Focus on rate type box when new invoice is created
            setTimeout(() => {
                $('#rateType').focus();
            }, 100);
        });
        
        // Print button
        $('#printInvoice').on('click', function() {
            self.printInvoice();
        });
        
        // Previous Invoice button
        $('#prevInvoice').on('click', function() {
            self.navigateInvoice('prev');
        });
        
        // Next Invoice button
        $('#nextInvoice').on('click', function() {
            self.navigateInvoice('next');
        });
        
        // Add row button
        $('#addRow').on('click', function() {
            self.addRow();
        });
        
        // Search Products button
        $('#searchProductBtn').on('click', function() {
            // If there are no rows, add one first
            if (self.rows.length === 0) {
                self.addRow();
            }
            // Find the first empty product row or create a new one
            let targetRow = self.rows.findIndex(row => !row.product);
            if (targetRow === -1) {
                self.addRow();
                targetRow = self.rows.length - 1;
            }
            self.openProductSearchModal(targetRow);
        });
        
        // Rate type change and keyboard navigation
        $('#rateType').on('change', function() {
            self.updateAllRates();
            // Focus to first code input after rate type change
            if ($(this).val() === '2') {
                setTimeout(() => {
                    self.focusOnFirstCodeInput();
                }, 100);
            }
        });

        // Rate type keydown events
        $('#rateType').on('keydown', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                self.focusOnFirstCodeInput();
            } else if (e.key === '1' || e.key === '2' || e.key === '3') {
                e.preventDefault();
                const values = ['CustomerRate', 'ShopRate', 'WholesaleRate'];
                $(this).val(values[parseInt(e.key) - 1]);
                $(this).trigger('change');
                if (e.key === '2') {
                    setTimeout(() => {
                        self.focusOnFirstCodeInput();
                    }, 100);
                }
            }
        });

        // Rate type focus events for highlighting
        $('#rateType').on('focus', function() {
            $(this).addClass('rate-type-focused');
        });

        $('#rateType').on('blur', function() {
            $(this).removeClass('rate-type-focused');
        });

        // Global escape key handler
        $(document).on('keydown', function(e) {
            // Global keyboard shortcuts (work anywhere)
            if (e.ctrlKey) {
                switch(e.key) {
                    case 's':
                    case 'S':
                        e.preventDefault();
                        self.saveInvoice();
                        return false;
                    case 'n':
                    case 'N':
                        e.preventDefault();
                        self.newInvoice();
                        // Focus on rate type box when new invoice is created via Ctrl+N
                        setTimeout(() => {
                            $('#rateType').focus();
                        }, 100);
                        return false;
                    case 'p':
                    case 'P':
                        e.preventDefault();
                        self.printInvoice();
                        return false;
                    case 'ArrowLeft':
                        e.preventDefault();
                        self.navigateInvoice('prev');
                        return false;
                    case 'ArrowRight':
                        e.preventDefault();
                        self.navigateInvoice('next');
                        return false;
                }
            }
            
            // Keyboard shortcuts (only when not in input fields)
            if (!$(e.target).is('input, textarea, select')) {
                if (e.key === 'F2') {
                    e.preventDefault();
                    $('#discountPercent').focus().select();
                    return false;
                } else if (e.key === 'F3') {
                    e.preventDefault();
                    $('#cashTenderReceived').focus().select();
                    return false;
                } else if (e.key === 'F4') {
                    e.preventDefault();
                    $('#upiAmount').focus().select();
                    return false;
                }
            }
            
            if (e.key === 'Escape') {
                // First check if product search modal is open
                if ($('#productSearchModal').hasClass('show')) {
                    e.preventDefault();
                    e.stopPropagation();
                    self.closeProductSearchModal();
                    return false;
                }
                
                // Then check for other suggestions
                $('#customerSuggestions').hide();
                $('.product-suggestions').hide();
                
                // If nothing is open, ask before closing the page
                if (!$('#productSearchModal').hasClass('show') && 
                    !$('#customerSuggestions').is(':visible') && 
                    !$('.product-suggestions').is(':visible')) {
                    e.preventDefault();
                    if (confirm('Are you sure you want to close this page?')) {
                        window.location.href = '../dashboard.php';
                    }
                }
            }
        });
        
        // Customer name input for suggestions
        $('#customerName').on('input', function() {
            self.searchCustomers($(this).val());
        });
        
        // Customer phone input for suggestions
        $('#customerPhone').on('input', function() {
            self.searchCustomersByPhone($(this).val());
        });
        
        // Hide suggestions when clicking outside
        $(document).on('click', function(e) {
            if (!$(e.target).closest('.customer-suggestions, #customerName, #customerPhone, .product-suggestions, .product-input-container').length) {
                $('#customerSuggestions').hide();
                $('.product-suggestions').hide();
            }
        });
        
        // Payment input change events for real-time calculation
        $('#upiAmount').on('input', function() {
            self.calculateCashChange();
        });
        
        // Discount calculation handlers
        $('#discountPercent').on('input', function() {
            self.calculateDiscountFromPercent();
        });
        
        $('#discountPercent').on('change', function() {
            self.calculateDiscountFromPercent();
        });
        
        $('#discountAmount').on('input', function() {
            self.calculateDiscountFromAmount();
        });
        
        $('#discountAmount').on('change', function() {
            self.calculateDiscountFromAmount();
        });
        
        // Cash tender calculation handler
        $('#cashTenderReceived').on('input', function() {
            self.handleCashTenderInput();
        });
        
        // Grid keyboard navigation
        $(document).on('keydown', '.cell-input', function(e) {
            self.handleGridKeyNavigation(e, this);
        });
        
        // Cell focus event for highlighting and text selection
        $(document).on('focus', '.cell-input', function() {
            $(this).select();
            $(this).addClass('focused-cell');
        });

        // Cell blur event to remove highlighting
        $(document).on('blur', '.cell-input', function() {
            $(this).removeClass('focused-cell');
        });

        // Product input focus event to open modal
        $(document).on('focus', '.product-input', function() {
            const rowIndex = $(this).data('row');
            const productValue = $(this).val().trim();
            if (!productValue) {
                self.openProductSearchModal(rowIndex);
            }
        });

        // Product search modal events
        $('#closeProductModal').on('click', function() {
            self.closeProductSearchModal();
        });

        $('.product-search-modal').on('click', function(e) {
            // Close modal when clicking outside the modal content
            if (e.target === this) {
                self.closeProductSearchModal();
            }
        });

        $('#modalProductSearch').on('input', function() {
            self.searchProductsInModal($(this).val());
        });

        $('#modalProductSearch').on('keydown', function(e) {
            if (e.key === 'Escape') {
                const searchValue = $(this).val().trim();
                if (!searchValue) {
                    e.preventDefault();
                    self.closeProductSearchModal();
                } else {
                    $(this).val('');
                    self.searchProductsInModal('');
                }
            } else if (e.key === 'Backspace') {
                const searchValue = $(this).val().trim();
                if (!searchValue) {
                    e.preventDefault();
                    self.closeProductSearchModal();
                }
            } else {
                self.handleModalKeyNavigation(e);
            }
        });

        // Ensure focus when modal becomes visible
        $('#productSearchModal').on('shown.bs.modal', function() {
            $('#modalProductSearch').focus();
        });

        $(document).on('click', '.modal-product-item', function() {
            const index = $(this).data('index');
            self.selectProductFromModal(index);
        });

        // Scan Mode toggle
        $('#scanMode').on('change', function() {
            self.toggleScanMode($(this).is(':checked'));
        });

        // Collect Payment button
        $('#collectPaymentBtn').on('click', function() {
            self.openPaymentCollectionModal();
        });

        // Save Payment button in modal
        $('#savePaymentBtn').on('click', function() {
            self.saveCustomerPayment();
        });

        // Payment amount input - calculate remaining balance
        $('#paymentAmount').on('input', function() {
            self.calculateRemainingBalance();
        });

        // Customer name change - fetch outstanding balance
        $('#customerName').on('change', function() {
            const customerName = $(this).val().trim();
            if (customerName) {
                self.fetchCustomerOutstanding(customerName);
            } else {
                self.clearCustomerOutstanding();
            }
        });

        // Also fetch outstanding on blur event for manual typing
        $('#customerName').on('blur', function() {
            const customerName = $(this).val().trim();
            if (customerName) {
                self.fetchCustomerOutstanding(customerName);
            } else {
                self.clearCustomerOutstanding();
            }
        });

        // Quick save button (duplicate functionality for floating panel)
        $('#quickSave').on('click', function() {
            self.saveInvoice();
        });

        $('#quickPrint').on('click', function() {
            self.printInvoice();
        });

        $('#quickNew').on('click', function() {
            self.newInvoice();
        });
    }

    addInitialRow() {
        this.addRow();
    }

    focusOnSaleCounter() {
        // Focus on sale counter (rateType dropdown)
        $('#rateType').focus().addClass('rate-type-focused');
        
        // Remove highlight after a delay to show initial focus
        setTimeout(() => {
            if (!$('#rateType').is(':focus')) {
                $('#rateType').removeClass('rate-type-focused');
            }
        }, 2000);
    }

    focusOnFirstCodeInput() {
        // Focus on the first code input field
        const firstCodeInput = $('.code-input[data-row="0"]');
        if (firstCodeInput.length > 0) {
            firstCodeInput.focus().select();
        }
    }

    createFloatingTotal() {
        // Remove existing floating total if it exists
        $('#floatingTotal').remove();
        
        // Create floating total element
        const floatingTotalHtml = `
            <div id="floatingTotal" class="floating-total">
                <div class="floating-total-content">
                    <span class="floating-total-label">Total:</span>
                    <span class="floating-total-amount">₹0.00</span>
                </div>
            </div>
        `;
        
        // Append to sales grid container instead of body
        $('.sales-grid').append(floatingTotalHtml);
        this.floatingTotal = $('#floatingTotal');
        
        // Initial position update with a delay to ensure DOM is ready
        setTimeout(() => {
            this.updateFloatingTotalPosition();
        }, 100);
    }

    updateFloatingTotalPosition() {
        // Check if floating total exists before trying to position it
        if (!this.floatingTotal || this.floatingTotal.length === 0) {
            return;
        }

        const rowCount = this.rows.length;
        
        if (rowCount >= 8) {
            // Position near the action column after 8th item
            const lastRow = $('#salesGrid tbody tr:last');
            if (lastRow.length > 0) {
                const actionColumn = lastRow.find('td:last');
                const salesGridTable = $('#salesGrid');
                
                // Get position relative to the table
                const tablePosition = salesGridTable.position();
                const actionColumnPosition = actionColumn.position();
                
                this.floatingTotal.css({
                    position: 'absolute',
                    top: tablePosition.top + actionColumnPosition.top + 5,
                    left: tablePosition.left + actionColumnPosition.left + actionColumn.outerWidth() + 15,
                    right: 'auto',
                    display: 'block',
                    animation: 'floatAnimation 2s ease-in-out infinite alternate'
                });
            }
        } else {
            // Hide floating total for less than 8 items
            this.floatingTotal.css({
                display: 'none'
            });
        }
    }

    createRowTotalLabel(rowIndex) {
        // Move existing row total label to new position if it exists
        const existingLabel = $('.row-total-label');
        if (existingLabel.length > 0) {
            // Update the data-row attribute and move to new row
            existingLabel.attr('data-row', rowIndex);
            this.updateRowTotalLabelPosition(rowIndex);
            return;
        }
        
        // Create floating "Total Amount" label for this specific row
        const rowTotalLabelHtml = `
            <div class="row-total-label" data-row="${rowIndex}">
                <span class="row-total-amount">₹0.00</span>
                <div class="row-total-arrow"></div>
            </div>
        `;
        
        // Append to sales grid container
        $('.sales-grid').append(rowTotalLabelHtml);
        
        // Position the label near the amount column for this row
        setTimeout(() => {
            this.updateRowTotalLabelPosition(rowIndex);
        }, 100);
    }

    updateRowTotalLabelPosition(rowIndex) {
        const rowTotalLabel = $(`.row-total-label[data-row="${rowIndex}"]`);
        const targetRow = $(`#salesGrid tbody tr[data-row="${rowIndex}"]`);
        
        if (rowTotalLabel.length === 0 || targetRow.length === 0) {
            return;
        }
        
        const amountColumn = targetRow.find('.col-amount');
        const salesGridTable = $('#salesGrid');
        
        if (amountColumn.length > 0) {
            // Get position relative to the table
            const tablePosition = salesGridTable.position();
            const amountColumnPosition = amountColumn.position();
            
            rowTotalLabel.css({
                position: 'absolute',
                top: tablePosition.top + amountColumnPosition.top - 5,
                left: tablePosition.left + amountColumnPosition.left + amountColumn.outerWidth() + 10,
                display: 'block',
                zIndex: 1000
            });
            
            // Update the total amount display
            this.updateRowTotalAmount();
        }
    }

    updateRowTotalAmount() {
        const totalAmount = $('#totalAmount').text() || '₹0.00';
        $('.row-total-amount').text(totalAmount);
    }

    autoScrollIfNeeded() {
        const rowCount = this.rows.length;
        if (rowCount >= 9) {
            const lastRow = $('#salesGrid tbody tr:last');
            if (lastRow.length > 0) {
                try {
                    lastRow[0].scrollIntoView({ behavior: 'smooth', block: 'center' });
                } catch (e) {
                    // Fallback for browsers that don't support smooth scrolling
                    lastRow[0].scrollIntoView();
                }
            }
        }
    }

    // Scan Mode Methods
    toggleScanMode(enabled) {
        this.isScanMode = enabled;
        if (enabled) {
            $('#scanModeIndicator').fadeIn();
            this.showAlert('Scan Mode Activated - Ready for barcode scanning', 'success');
        } else {
            $('#scanModeIndicator').fadeOut();
            this.showAlert('Scan Mode Deactivated', 'info');
        }
    }

    handleScanModeCode(currentRow, code) {
        // Check if this code already exists in the cart
        const existingRowIndex = this.findProductByCode(code);
        
        if (existingRowIndex !== -1) {
            // Product exists, increment quantity
            this.incrementProductQuantity(existingRowIndex);
        } else {
            // New product, load it in current row
            this.loadProductByCodeInScanMode(currentRow, code);
        }
    }

    findProductByCode(code) {
        for (let i = 0; i < this.rows.length; i++) {
            if (this.rows[i].code === code && this.rows[i].product) {
                return i;
            }
        }
        return -1;
    }

    incrementProductQuantity(rowIndex) {
        const currentQty = parseFloat($(`.qty-input[data-row="${rowIndex}"]`).val()) || 0;
        const newQty = currentQty + 1;
        
        $(`.qty-input[data-row="${rowIndex}"]`).val(newQty);
        this.rows[rowIndex].qty = newQty;
        this.calculateAmount(rowIndex);
        
        // Highlight the row briefly
        this.highlightRow(rowIndex);
        
        // Show notification
        const productName = this.rows[rowIndex].product;
        this.showAlert(`${productName} quantity increased to ${newQty}`, 'success');
        
        // Focus back to first empty code input or add new row
        this.focusToNextScanPosition();
    }

    loadProductByCodeInScanMode(rowIndex, code) {
        const rateType = $('#rateType').val();
        
        $.ajax({
            url: 'ajax/get_product.php',
            method: 'POST',
            data: { 
                code: code,
                rate_type: rateType
            },
            dataType: 'json',
            success: (response) => {
                if (response.success) {
                    const product = response.data;
                    this.rows[rowIndex] = {
                        ...this.rows[rowIndex],
                        code: product.code,
                        product: product.name,
                        tamil_name: product.tamil_name || '',
                        unit: product.unit,
                        rate: product.selling_rate
                    };
                    
                    $(`.code-input[data-row="${rowIndex}"]`).val(product.code);
                    
                    // Display only Tamil name in cart if available, otherwise English name
                    const displayName = product.tamil_name || product.name;
                    $(`.product-input[data-row="${rowIndex}"]`).val(displayName);
                    $(`.unit-input[data-row="${rowIndex}"]`).val(product.unit);
                    $(`.rate-input[data-row="${rowIndex}"]`).val(product.selling_rate);
                    $(`.qty-input[data-row="${rowIndex}"]`).val('1');
                    
                    this.rows[rowIndex].qty = 1;
                    this.calculateAmount(rowIndex);
                    
                    // Highlight the row briefly
                    this.highlightRow(rowIndex);
                    
                    // Show notification
                    this.showAlert(`${product.name} added to cart`, 'success');
                    
                    // Focus to next scan position
                    this.focusToNextScanPosition();
                } else {
                    this.showAlert(response.message || 'Product not found!', 'error');
                    $(`.code-input[data-row="${rowIndex}"]`).val('').focus();
                }
            },
            error: (xhr, status, error) => {
                this.showAlert('Error loading product: ' + error, 'error');
                $(`.code-input[data-row="${rowIndex}"]`).val('').focus();
            }
        });
    }

    highlightRow(rowIndex) {
        const $row = $(`tr[data-row="${rowIndex}"]`);
        $row.addClass('scan-highlight');
        setTimeout(() => {
            $row.removeClass('scan-highlight');
        }, 1500);
    }

    focusToNextScanPosition() {
        // Find first empty row or add new one
        let emptyRowIndex = -1;
        for (let i = 0; i < this.rows.length; i++) {
            if (!this.rows[i].code) {
                emptyRowIndex = i;
                break;
            }
        }
        
        if (emptyRowIndex === -1) {
            // No empty row, add new one
            this.addRow();
            emptyRowIndex = this.rows.length - 1;
        }
        
        // Focus on the empty code input
        setTimeout(() => {
            $(`.code-input[data-row="${emptyRowIndex}"]`).focus().select();
        }, 100);
    }

    addRow() {
        // Check if last row has code and product before adding new row
        if (this.rows.length > 0) {
            const lastRowIndex = this.rows.length - 1;
            const lastCode = $(`.code-input[data-row="${lastRowIndex}"]`).val().trim();
            const lastProduct = $(`.product-input[data-row="${lastRowIndex}"]`).val().trim();
            
            if (!lastCode && !lastProduct) {
                this.showAlert('Please fill code and product name before adding new row!', 'warning');
                $(`.code-input[data-row="${lastRowIndex}"]`).focus().select();
                return;
            }
        }

        const rowIndex = this.rows.length;
        const rowHtml = `
            <tr data-row="${rowIndex}">
                <td class="col-code">
                    <input type="text" class="cell-input code-input" data-row="${rowIndex}" data-col="code" 
                           placeholder="Code">
                </td>
                <td class="col-product">
                    <div class="product-input-container">
                        <input type="text" class="cell-input product-input" data-row="${rowIndex}" data-col="product" 
                               placeholder="Product Name">
                        <div class="product-suggestions" id="suggestions-${rowIndex}"></div>
                    </div>
                </td>
                <td class="col-unit">
                    <input type="text" class="cell-input unit-input readonly" data-row="${rowIndex}" data-col="unit" 
                           placeholder="Unit" readonly>
                </td>
                <td class="col-qty">
                    <input type="number" class="cell-input qty-input" data-row="${rowIndex}" data-col="qty" 
                           placeholder="Qty" step="0.001" value="1">
                </td>
                <td class="col-rate">
                    <input type="number" class="cell-input rate-input" data-row="${rowIndex}" data-col="rate" 
                           placeholder="Rate" step="0.01">
                </td>
                <td class="col-amount">
                    <input type="number" class="cell-input amount-input readonly" data-row="${rowIndex}" data-col="amount" 
                           placeholder="Amount" readonly>
                </td>
                <td class="col-action text-center">
                    <button type="button" class="action-btn btn-delete" onclick="salesManager.deleteRow(${rowIndex})" title="Delete Row">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>
        `;
        
        $('#salesGrid tbody').append(rowHtml);
        this.rows.push({
            code: '',
            product: '',
            unit: '',
            qty: 1,
            rate: 0,
            amount: 0
        });
        
        // Bind enhanced keyboard events for this row
        this.bindRowKeyboardEvents(rowIndex);
        
        // Bind change events for calculation
        $(`.qty-input[data-row="${rowIndex}"], .rate-input[data-row="${rowIndex}"]`).on('input change', () => {
            this.calculateAmount(rowIndex);
        });
        
        // Set default quantity to 1 when quantity field loses focus if empty
        $(`.qty-input[data-row="${rowIndex}"]`).on('blur', function() {
            const qtyValue = $(this).val();
            if (!qtyValue || qtyValue === '' || parseFloat(qtyValue) <= 0) {
                $(this).val('1');
                salesManager.rows[rowIndex].qty = 1;
                salesManager.calculateAmount(rowIndex);
            }
        });
        
        // Bind product input events
        $(`.product-input[data-row="${rowIndex}"]`).on('input', (e) => {
            this.searchProducts(e.target.value, rowIndex);
        });
        
        $(`.code-input[data-row="${rowIndex}"]`).on('blur', (e) => {
            this.loadProductByCode(rowIndex, e.target.value);
        });

        // Auto-scroll and update floating total position
        this.autoScrollIfNeeded();
        if (this.floatingTotal && this.floatingTotal.length > 0) {
            this.updateFloatingTotalPosition();
        }
        
        // Add floating "Total Amount" label for the new row
        this.createRowTotalLabel(rowIndex);
        
        // Focus on the new row's code input with a slight delay to ensure DOM is ready
        setTimeout(() => {
            $(`.code-input[data-row="${rowIndex}"]`).focus().select();
        }, 50);
    }

    bindRowKeyboardEvents(rowIndex) {
        // Code input keyboard events
        $(`.code-input[data-row="${rowIndex}"]`).on('keydown', (e) => {
            if (e.key === 'Backspace') {
                const codeValue = $(e.target).val().trim();
                if (!codeValue) {
                    e.preventDefault();
                    this.deleteRowAndFocusPrevious(rowIndex);
                }
            } else if (e.key === 'ArrowUp') {
                e.preventDefault();
                this.focusPreviousRowSameColumn(rowIndex, 'code');
            } else if (e.key === 'ArrowDown') {
                e.preventDefault();
                this.focusNextRowSameColumn(rowIndex, 'code');
            }
        });

        // Product input keyboard events
        $(`.product-input[data-row="${rowIndex}"]`).on('keydown', (e) => {
            if (e.key === 'ArrowUp') {
                e.preventDefault();
                this.focusPreviousRowSameColumn(rowIndex, 'product');
            } else if (e.key === 'ArrowDown') {
                e.preventDefault();
                this.focusNextRowSameColumn(rowIndex, 'product');
            }
        });

        // Quantity input keyboard events
        $(`.qty-input[data-row="${rowIndex}"]`).on('keydown', (e) => {
            if (e.key === 'ArrowUp') {
                e.preventDefault();
                this.focusPreviousRowSameColumn(rowIndex, 'qty');
            } else if (e.key === 'ArrowDown') {
                e.preventDefault();
                this.focusNextRowSameColumn(rowIndex, 'qty');
            }
        });

        // Rate input keyboard events with autocomplete
        $(`.rate-input[data-row="${rowIndex}"]`).on('input', (e) => {
            const inputValue = $(e.target).val();
            if (inputValue.length === 1) {
                // Auto-complete rate based on first digit
                this.autoCompleteRate(rowIndex, inputValue);
            }
        });

        $(`.rate-input[data-row="${rowIndex}"]`).on('keydown', (e) => {
            if (e.key === 'ArrowUp') {
                e.preventDefault();
                this.focusPreviousRowSameColumn(rowIndex, 'rate');
            } else if (e.key === 'ArrowDown') {
                e.preventDefault();
                this.focusNextRowSameColumn(rowIndex, 'rate');
            }
        });
    }

    deleteRowAndFocusPrevious(rowIndex) {
        if (this.rows.length <= 1) {
            this.showAlert('At least one row is required!', 'error');
            return;
        }
        
        // Focus on previous row's rate input
        if (rowIndex > 0) {
            setTimeout(() => {
                $(`.rate-input[data-row="${rowIndex - 1}"]`).focus().select();
            }, 50);
        }
        
        // Delete the row
        this.deleteRow(rowIndex);
    }

    focusPreviousRowSameColumn(currentRow, column) {
        if (currentRow > 0) {
            $(`.${column}-input[data-row="${currentRow - 1}"]`).focus().select();
        }
    }

    focusNextRowSameColumn(currentRow, column) {
        const nextRow = currentRow + 1;
        if (nextRow < this.rows.length) {
            $(`.${column}-input[data-row="${nextRow}"]`).focus().select();
        } else {
            // Check if current row has code and product before creating new row
            const currentCode = $(`.code-input[data-row="${currentRow}"]`).val().trim();
            const currentProduct = $(`.product-input[data-row="${currentRow}"]`).val().trim();
            
            if (currentCode && currentProduct) {
                // If current row is valid, create new one
                this.addRow();
                setTimeout(() => {
                    $(`.${column}-input[data-row="${nextRow}"]`).focus().select();
                }, 100);
            } else {
                this.showAlert('Please fill code and product name first!', 'warning');
                // Focus on the missing field
                if (!currentCode) {
                    $(`.code-input[data-row="${currentRow}"]`).focus().select();
                } else {
                    $(`.product-input[data-row="${currentRow}"]`).focus().select();
                }
            }
        }
    }

    autoCompleteRate(rowIndex, firstDigit) {
        const code = $(`.code-input[data-row="${rowIndex}"]`).val().trim();
        if (!code) return;

        // Get existing product data to autocomplete rate
        const existingRate = $(`.rate-input[data-row="${rowIndex}"]`).data('suggested-rate');
        if (existingRate && existingRate.toString().startsWith(firstDigit)) {
            $(`.rate-input[data-row="${rowIndex}"]`).val(existingRate);
        }
    }

    focusNext(rowIndex, colType) {
        setTimeout(() => {
            $(`.${colType}-input[data-row="${rowIndex}"]`).focus().select();
        }, 100);
    }

    loadProductByCode(rowIndex, code) {
        if (!code) return;
        
        const rateType = $('#rateType').val();
        
        $.ajax({
            url: 'ajax/get_product.php',
            method: 'POST',
            data: { 
                code: code,
                rate_type: rateType
            },
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    const product = response.data;
                    
                    // Check for duplicate products when scan mode is disabled
                    if (!salesManager.isScanMode) {
                        const existingRowIndex = salesManager.findExistingProduct(product.code, rowIndex);
                        if (existingRowIndex !== -1) {
                            const shouldUpdate = confirm(
                                `🔄 PRODUCT ALREADY IN CART\n\n` +
                                `Product: ${product.tamil_name || product.name}\n` +
                                `Code: ${product.code}\n\n` +
                                `This product is already in row ${existingRowIndex + 1}.\n\n` +
                                `Do you want to:\n\n` +
                                `✅ YES - Update the existing cart entry (add to quantity)\n` +
                                `❌ NO - Add as duplicate entry in current row\n\n` +
                                `Choose your option:`
                            );
                            
                            if (shouldUpdate) {
                                // Clear the current row's code input since we're not using it
                                $(`.code-input[data-row="${rowIndex}"]`).val('');
                                salesManager.updateExistingProductQuantity(existingRowIndex);
                                return; // Don't proceed with adding to current row
                            }
                            // If user chooses NO, continue with adding duplicate entry
                        }
                    }
                    
                    salesManager.rows[rowIndex] = {
                        ...salesManager.rows[rowIndex],
                        code: product.code,
                        product: product.name,
                        tamil_name: product.tamil_name || '',
                        unit: product.unit,
                        rate: product.selling_rate
                    };
                    
                    // Display only Tamil name in cart if available, otherwise English name
                    const displayName = product.tamil_name || product.name;
                    $(`.product-input[data-row="${rowIndex}"]`).val(displayName);
                    $(`.unit-input[data-row="${rowIndex}"]`).val(product.unit);
                    $(`.rate-input[data-row="${rowIndex}"]`).val(product.selling_rate);
                    
                    // Store suggested rate for autocomplete
                    $(`.rate-input[data-row="${rowIndex}"]`).data('suggested-rate', product.selling_rate);
                } else {
                    console.log('Product error:', response);
                    salesManager.showAlert(response.message || 'Product not found!', 'error');
                    $(`.code-input[data-row="${rowIndex}"]`).val('').focus();
                }
            },
            error: function(xhr, status, error) {
                console.log('AJAX Error:', xhr.responseText);
                salesManager.showAlert('Error loading product: ' + error, 'error');
            }
        });
    }

    searchProducts(searchTerm, rowIndex) {
        if (!searchTerm || searchTerm.length < 2) {
            $(`#suggestions-${rowIndex}`).hide();
            return;
        }
        
        const rateType = $('#rateType').val();
        
        $.ajax({
            url: 'ajax/search_products.php',
            method: 'POST',
            data: { 
                search: searchTerm,
                rate_type: rateType
            },
            dataType: 'json',
            success: function(response) {
                if (response.success && response.data.length > 0) {
                    salesManager.showProductSuggestions(response.data, rowIndex);
                } else {
                    $(`#suggestions-${rowIndex}`).hide();
                }
            },
            error: function(xhr, status, error) {
                console.log('Search Error:', error);
                $(`#suggestions-${rowIndex}`).hide();
            }
        });
    }

    showProductSuggestions(products, rowIndex) {
        const suggestionsHtml = products.map((product, index) => {
            const tamilName = product.tamil_name ? `<br><small style="color: #666;">${product.tamil_name}</small>` : '';
            return `
                <div class="product-suggestion" data-index="${index}" data-row="${rowIndex}">
                    <div class="suggestion-name">${product.name}${tamilName}</div>
                    <div class="suggestion-details">Code: ${product.code} | Unit: ${product.unit} | Rate: ₹${product.selling_rate}</div>
                </div>
            `;
        }).join('');
        
        const $suggestions = $(`#suggestions-${rowIndex}`);
        $suggestions.html(suggestionsHtml).css({
            'z-index': '999999',
            'position': 'absolute',
            'top': '100%',
            'left': '0',
            'right': '0'
        }).show();
        
        this.selectedSuggestionIndex = -1;
        this.currentProductInput = $(`.product-input[data-row="${rowIndex}"]`);
        this.suggestionProducts = products;
        this.suggestionRowIndex = rowIndex;
        
        // Bind click events
        $suggestions.find('.product-suggestion').on('click', function() {
            const index = $(this).data('index');
            salesManager.selectProduct(products[index], rowIndex);
        });
    }

    selectProduct(product, rowIndex) {
        // Check for duplicate products when scan mode is disabled
        if (!this.isScanMode) {
            const existingRowIndex = this.findExistingProduct(product.code, rowIndex);
            if (existingRowIndex !== -1) {
                const shouldUpdate = confirm(
                    `🔄 PRODUCT ALREADY IN CART\n\n` +
                    `Product: ${product.tamil_name || product.name}\n` +
                    `Code: ${product.code}\n\n` +
                    `This product is already in row ${existingRowIndex + 1}.\n\n` +
                    `Do you want to:\n\n` +
                    `✅ YES - Update the existing cart entry (add to quantity)\n` +
                    `❌ NO - Add as duplicate entry in current row\n\n` +
                    `Choose your option:`
                );
                
                if (shouldUpdate) {
                    this.updateExistingProductQuantity(existingRowIndex);
                    return; // Don't proceed with adding to current row
                }
                // If user chooses NO, continue with adding duplicate entry
            }
        }
        
        this.rows[rowIndex] = {
            ...this.rows[rowIndex],
            code: product.code,
            product: product.name,
            tamil_name: product.tamil_name || '',
            unit: product.unit,
            rate: product.selling_rate
        };
        
        $(`.code-input[data-row="${rowIndex}"]`).val(product.code);
        
        // Display only Tamil name in cart if available, otherwise English name
        const displayName = product.tamil_name || product.name;
        $(`.product-input[data-row="${rowIndex}"]`).val(displayName);
        $(`.unit-input[data-row="${rowIndex}"]`).val(product.unit);
        $(`.rate-input[data-row="${rowIndex}"]`).val(product.selling_rate);
        
        // Store suggested rate for autocomplete
        $(`.rate-input[data-row="${rowIndex}"]`).data('suggested-rate', product.selling_rate);
        
        $(`#suggestions-${rowIndex}`).hide();
        
        // Focus on quantity input
        setTimeout(() => {
            $(`.qty-input[data-row="${rowIndex}"]`).focus().select();
        }, 100);
    }

    // Helper function to find if product already exists in cart
    findExistingProduct(productCode, excludeRowIndex) {
        for (let i = 0; i < this.rows.length; i++) {
            if (i !== excludeRowIndex && this.rows[i] && this.rows[i].code === productCode) {
                // Also check if the row actually has the product code in the input
                const existingCode = $(`.code-input[data-row="${i}"]`).val();
                if (existingCode === productCode) {
                    return i;
                }
            }
        }
        return -1;
    }

    // Helper function to update existing product quantity
    updateExistingProductQuantity(rowIndex) {
        const currentQty = parseFloat($(`.qty-input[data-row="${rowIndex}"]`).val()) || 0;
        const newQty = currentQty + 1; // Add 1 to existing quantity
        
        $(`.qty-input[data-row="${rowIndex}"]`).val(newQty);
        this.rows[rowIndex].qty = newQty;
        
        // Recalculate amount for this row
        const rate = parseFloat($(`.rate-input[data-row="${rowIndex}"]`).val()) || 0;
        const amount = newQty * rate;
        $(`.amount-input[data-row="${rowIndex}"]`).val(amount.toFixed(2));
        this.rows[rowIndex].amount = amount;
        
        // Update totals
        this.updateTotals();
        
        // Highlight the updated row briefly
        $(`.sales-grid tbody tr[data-row="${rowIndex}"]`).addClass('update-highlight');
        setTimeout(() => {
            $(`.sales-grid tbody tr[data-row="${rowIndex}"]`).removeClass('update-highlight');
        }, 2000);
        
        // Focus on the updated row's quantity field
        setTimeout(() => {
            $(`.qty-input[data-row="${rowIndex}"]`).focus().select();
        }, 100);
        
        this.showAlert(`Product quantity updated to ${newQty} in row ${rowIndex + 1}`, 'success');
    }

    updateAllRates() {
        // Update all existing products with new rate type
        for (let i = 0; i < this.rows.length; i++) {
            const code = $(`.code-input[data-row="${i}"]`).val();
            if (code) {
                this.loadProductByCode(i, code);
            }
        }
    }

    calculateAmount(rowIndex) {
        const qty = parseFloat($(`.qty-input[data-row="${rowIndex}"]`).val()) || 0;
        const rate = parseFloat($(`.rate-input[data-row="${rowIndex}"]`).val()) || 0;
        const amount = qty * rate;
        
        $(`.amount-input[data-row="${rowIndex}"]`).val(amount.toFixed(2));
        this.rows[rowIndex].qty = qty;
        this.rows[rowIndex].rate = rate;
        this.rows[rowIndex].amount = amount;
        
        this.updateTotals();
    }

    updateTotals() {
        let subtotal = 0;
        this.rows.forEach(row => {
            subtotal += row.amount || 0;
        });
        
        // Get discount values - use actual input values, not hidden fields
        const discountPercent = parseFloat($('#discountPercent').val()) || 0;
        const discountAmountInput = parseFloat($('#discountAmount').val()) || 0;
        
        // Calculate discount amount
        let discountAmount = 0;
        if (discountPercent > 0 && subtotal > 0) {
            // If percentage is entered, calculate from percentage
            discountAmount = (subtotal * discountPercent) / 100;
            // Update the discount amount field to match
            $('#discountAmount').val(discountAmount.toFixed(2));
        } else if (discountAmountInput > 0) {
            // If only amount is entered, use that
            discountAmount = discountAmountInput;
            // Calculate and update percentage
            if (subtotal > 0) {
                const calculatedPercent = (discountAmount / subtotal) * 100;
                $('#discountPercent').val(calculatedPercent.toFixed(2));
            }
        }
        
        // Ensure discount doesn't exceed subtotal
        if (discountAmount > subtotal) {
            discountAmount = subtotal;
            $('#discountAmount').val(discountAmount.toFixed(2));
            $('#discountPercent').val('100.00');
        }
        
        // Calculate final total
        const total = Math.max(0, subtotal - discountAmount);
        
        // Update display
        $('#subtotalAmount').text('₹' + subtotal.toFixed(2));
        $('#totalAmount').text('₹' + total.toFixed(2));
        
        // Update floating total
        if (this.floatingTotal && this.floatingTotal.length > 0) {
            this.floatingTotal.find('.floating-total-amount').text('₹' + total.toFixed(2));
            this.updateFloatingTotalPosition();
        }
        
        // Update row total amount in floating label
        this.updateRowTotalAmount();
        
        // Update hidden fields for saving
        $('#subtotalHidden').val(subtotal.toFixed(2));
        $('#discountAmountHidden').val(discountAmount.toFixed(2));
        $('#taxAmountHidden').val('0.00');
        $('#totalAmountHidden').val(total.toFixed(2));
        
        // Update payment amounts based on current payment mode
        this.updatePaymentAmounts();
        
        // Recalculate cash change if cash tender is entered
        this.calculateCashChange();
    }

    // Update payment amounts when total changes
    updatePaymentAmounts() {
        const paymentMode = $('#paymentMode').val();
        const totalAmount = parseFloat($('#totalAmountHidden').val()) || 0;
        
        if (paymentMode === 'cash') {
            const currentCash = parseFloat($('#cashTenderReceived').val()) || 0;
            if (currentCash === 0 || currentCash !== totalAmount) {
                $('#cashTenderReceived').val(totalAmount.toFixed(2));
            }
            $('#upiAmount').val('0.00');
        } else if (paymentMode === 'upi') {
            const currentUpi = parseFloat($('#upiAmount').val()) || 0;
            if (currentUpi === 0 || currentUpi !== totalAmount) {
                $('#upiAmount').val(totalAmount.toFixed(2));
            }
            $('#cashTenderReceived').val('0.00');
        }
        // For mixed mode, don't auto-update amounts
    }

    // Calculate discount amount from percentage
    calculateDiscountFromPercent() {
        // First calculate current subtotal from rows
        let subtotal = 0;
        this.rows.forEach(row => {
            subtotal += row.amount || 0;
        });
        
        const discountPercent = parseFloat($('#discountPercent').val()) || 0;
        
        if (discountPercent > 100) {
            $('#discountPercent').val(100);
            return;
        }
        
        if (subtotal > 0 && discountPercent > 0) {
            const discountAmount = (subtotal * discountPercent) / 100;
            $('#discountAmount').val(discountAmount.toFixed(2));
        } else if (discountPercent === 0) {
            $('#discountAmount').val('0.00');
        }
        
        this.updateTotals();
    }

    // Calculate discount percentage from amount
    calculateDiscountFromAmount() {
        // First calculate current subtotal from rows
        let subtotal = 0;
        this.rows.forEach(row => {
            subtotal += row.amount || 0;
        });
        
        const discountAmount = parseFloat($('#discountAmount').val()) || 0;
        
        if (discountAmount > subtotal && subtotal > 0) {
            $('#discountAmount').val(subtotal.toFixed(2));
            $('#discountPercent').val('100.00');
        } else if (subtotal > 0 && discountAmount >= 0) {
            const discountPercent = (discountAmount / subtotal) * 100;
            $('#discountPercent').val(discountPercent.toFixed(2));
        } else if (discountAmount === 0) {
            $('#discountPercent').val('0.00');
        }
        
        this.updateTotals();
    }

    // Calculate cash change
    calculateCashChange() {
        const totalAmount = parseFloat($('#totalAmountHidden').val()) || 0;
        const cashTenderReceived = parseFloat($('#cashTenderReceived').val()) || 0;
        const upiAmount = parseFloat($('#upiAmount').val()) || 0;
        const paymentMode = $('#paymentMode').val();
        
        let cashChange = 0;
        
        if (paymentMode === 'mixed') {
            // In mixed mode, calculate total payment received
            const totalPaid = cashTenderReceived + upiAmount;
            if (totalPaid > totalAmount) {
                // If total payment exceeds invoice amount, return excess as cash change
                cashChange = totalPaid - totalAmount;
            }
        } else if (paymentMode === 'cash') {
            // In cash mode, change is cash tender minus total amount
            cashChange = Math.max(0, cashTenderReceived - totalAmount);
        } else {
            // UPI or credit mode - no cash change
            cashChange = 0;
        }
        
        $('#cashChange').val(cashChange.toFixed(2));
        
        // Update the background color based on whether change is needed
        if (cashChange > 0) {
            $('#cashChange').css('background-color', 'rgba(255, 235, 59, 0.3)');
        } else {
            $('#cashChange').css('background-color', 'rgba(255, 235, 59, 0.1)');
        }
        
        // Debug log
        console.log('Cash Change Calculation:', {
            paymentMode: paymentMode,
            totalAmount: totalAmount,
            cashTenderReceived: cashTenderReceived,
            upiAmount: upiAmount,
            totalPaid: cashTenderReceived + upiAmount,
            cashChange: cashChange
        });
    }

    // Handle payment mode changes
    handlePaymentModeChange(paymentMode) {
        const totalAmount = parseFloat($('#totalAmountHidden').val()) || 0;
        const paymentModeSelect = $('#paymentMode');
        
        // Remove any existing credit styling
        paymentModeSelect.removeClass('payment-mode-credit');
        
        if (paymentMode === 'cash') {
            // Clear UPI amount and set total to cash
            $('#upiAmount').val('0.00');
            $('#cashTenderReceived').val(totalAmount.toFixed(2));
            this.calculateCashChange();
        } else if (paymentMode === 'upi') {
            // Clear cash amount and set total to UPI
            $('#cashTenderReceived').val('0.00');
            $('#upiAmount').val(totalAmount.toFixed(2));
            $('#cashChange').val('0.00');
        } else if (paymentMode === 'mixed') {
            // Clear both amounts for manual entry
            $('#cashTenderReceived').val('');
            $('#upiAmount').val('');
            $('#cashChange').val('0.00');
        } else if (paymentMode === 'credit') {
            // Clear all payment amounts for credit sale
            $('#cashTenderReceived').val('0.00');
            $('#upiAmount').val('0.00');
            $('#cashChange').val('0.00');
            // Set invoice type to credit automatically
            $('#invoiceType').val('credit');
            $('#invoiceType').trigger('change');
            // Add visual styling for credit mode
            paymentModeSelect.addClass('payment-mode-credit');
            // Show alert about credit payment
            this.showAlert('Credit payment mode selected. Invoice will be saved as credit sale.', 'info');
        }
    }

    // Handle cash amount changes in mixed payment mode
    handleCashAmountChange() {
        const paymentMode = $('#paymentMode').val();
        
        if (paymentMode === 'mixed') {
            // Just recalculate change, don't auto-fill UPI
            this.calculateCashChange();
        } else {
            this.calculateCashChange();
        }
    }

    // Handle UPI amount changes in mixed payment mode
    handleUpiAmountChange() {
        const paymentMode = $('#paymentMode').val();
        
        if (paymentMode === 'mixed') {
            // Just recalculate change, don't auto-fill cash
            this.calculateCashChange();
        } else {
            this.calculateCashChange();
        }
    }

    // Handle cash tender input in mixed payment mode
    handleCashTenderInput() {
        const paymentMode = $('#paymentMode').val();
        const totalAmount = parseFloat($('#totalAmountHidden').val()) || 0;
        const cashTenderReceived = parseFloat($('#cashTenderReceived').val()) || 0;
        
        if (paymentMode === 'mixed' && cashTenderReceived > 0 && totalAmount > 0) {
            // Calculate remaining balance and auto-fill UPI amount
            const remainingBalance = Math.max(0, totalAmount - cashTenderReceived);
            $('#upiAmount').val(remainingBalance.toFixed(2));
        }
        
        // Always recalculate cash change
        this.calculateCashChange();
    }

    deleteRow(rowIndex) {
        if (this.rows.length <= 1) {
            this.showAlert('At least one row is required!', 'error');
            return;
        }
        
        // Remove the row total label for this row
        $(`.row-total-label[data-row="${rowIndex}"]`).remove();
        
        $(`tr[data-row="${rowIndex}"]`).remove();
        this.rows.splice(rowIndex, 1);
        this.reindexRows();
        this.updateTotals();
    }

    reindexRows() {
        $('#salesGrid tbody tr').each((index, row) => {
            $(row).attr('data-row', index);
            $(row).find('input, button').each((i, element) => {
                const $el = $(element);
                if ($el.attr('data-row')) {
                    $el.attr('data-row', index);
                }
                if ($el.attr('onkeydown')) {
                    $el.attr('onkeydown', $el.attr('onkeydown').replace(/\d+/g, index));
                }
                if ($el.attr('onchange')) {
                    $el.attr('onchange', $el.attr('onchange').replace(/\d+/g, index));
                }
                if ($el.attr('onclick')) {
                    $el.attr('onclick', $el.attr('onclick').replace(/\d+/g, index));
                }
            });
        });
    }

    loadCustomers() {
        const self = this;
        $.ajax({
            url: 'ajax/get_customers.php',
            method: 'GET',
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    self.customers = response.data;
                }
            }
        });
    }

    loadProducts() {
        // Products are loaded dynamically when needed by product code
        // This function is kept for future enhancement if needed
        this.products = [];
    }

    setSaveButtonState(enabled) {
        const saveButton = $('#saveInvoiceBtn');
        if (enabled) {
            saveButton.prop('disabled', false).text('Save Invoice (Ctrl+S)');
        } else {
            saveButton.prop('disabled', true).text('Saving...');
        }
    }

    saveInvoice() {
        // Prevent multiple saves
        if (this.isSaving) {
            this.showAlert('Save in progress, please wait...', 'warning');
            return;
        }
        
        if (!this.validateInvoice()) {
            return;
        }
        
        // Set saving state and disable save button
        this.isSaving = true;
        this.setSaveButtonState(false);
        
        const invoiceData = this.getInvoiceData();
        
        $.ajax({
            url: 'ajax/save_invoice.php',
            method: 'POST',
            data: JSON.stringify(invoiceData),
            contentType: 'application/json',
            dataType: 'json',
            success: (response) => {
                // Reset saving state
                this.isSaving = false;
                this.setSaveButtonState(true);
                
                if (response.success) {
                    // Determine if this was truly an edit operation (not just a save of a new invoice)
                    const wasEditingExistingInvoice = this.currentInvoiceId !== null && (typeof editInvoiceData !== 'undefined' && editInvoiceData);
                    
                    // Set the current invoice ID for printing/other operations
                    const savedInvoiceId = response.invoice_id;
                    
                    // Check if cash tender was provided to show appropriate message
                    const cashTenderReceived = parseFloat($('#cashTenderReceived').val()) || 0;
                    const totalAmount = parseFloat($('#totalAmountHidden').val()) || 0;
                    
                    let message = wasEditingExistingInvoice ? 'Invoice updated successfully!' : 'Invoice saved successfully!';
                    if (cashTenderReceived > 0) {
                        const cashChange = Math.max(0, cashTenderReceived - totalAmount);
                        if (cashChange > 0) {
                            message += ` Change to return: ₹${cashChange.toFixed(2)}`;
                        }
                        if (cashTenderReceived >= totalAmount) {
                            message += ' (Payment completed)';
                        } else {
                            const remainingBalance = totalAmount - cashTenderReceived;
                            message += ` Remaining balance: ₹${remainingBalance.toFixed(2)}`;
                        }
                    }
                    
                    this.showAlert(message, 'success');
                    
                    // Set current invoice ID temporarily for printing
                    this.currentInvoiceId = savedInvoiceId;
                    
                    this.loadRecentSales();
                    
                    // Print 3-inch receipt
                    this.printThermalInvoice();
                    
                    if (wasEditingExistingInvoice) {
                        // For edit mode, stay on the same page and show success message
                        setTimeout(() => {
                            this.showAlert('Invoice updated successfully! You can continue editing or go back to reports.', 'info');
                        }, 2000);
                    } else {
                        // For new invoice mode, clear and start new (including credit invoices)
                        setTimeout(() => {
                            this.newInvoice();
                            // Focus on rate type box after invoice save
                            $('#rateType').focus();
                        }, 1000);
                    }
                } else {
                    this.showAlert(response.message || 'Error saving invoice!', 'error');
                }
            },
            error: () => {
                // Reset saving state
                this.isSaving = false;
                this.setSaveButtonState(true);
                this.showAlert('Error saving invoice!', 'error');
            }
        });
    }

    validateInvoice() {
        const validRows = this.rows.filter(row => row.code && row.qty > 0);
        if (validRows.length === 0) {
            this.showAlert('At least one product is required!', 'error');
            return false;
        }
        
        // Check for overpayment in mixed mode
        const paymentMode = $('#paymentMode').val();
        const totalAmount = parseFloat($('#totalAmountHidden').val()) || 0;
        const cashAmount = parseFloat($('#cashTenderReceived').val()) || 0;
        const upiAmount = parseFloat($('#upiAmount').val()) || 0;
        const totalPaid = cashAmount + upiAmount;
        
        if (paymentMode === 'mixed' && totalPaid > totalAmount) {
            const overpayment = totalPaid - totalAmount;
            const confirmed = confirm(
                `Total payment (₹${totalPaid.toFixed(2)}) exceeds invoice amount (₹${totalAmount.toFixed(2)}) by ₹${overpayment.toFixed(2)}.\n\n` +
                `Change to return: ₹${overpayment.toFixed(2)}\n\n` +
                `Do you want to continue?`
            );
            if (!confirmed) {
                return false;
            }
        }
        
        return true;
    }

    getInvoiceData() {
        const validRows = this.rows.filter(row => row.code && row.qty > 0);
        
        // Save new customer if phone number is provided and new
        const customerData = {
            customer_name: $('#customerName').val(),
            phone: $('#customerPhone').val(),
            gst_number: $('#customerGst').val(),
            address: $('#customerAddress').val()
        };
        
        if (customerData.phone && customerData.phone.length >= 10) {
            this.saveNewCustomer(customerData);
        }

        const invoiceType = $('#invoiceType').val();
        const creditDays = $('#creditDays').val();
        
        let dueDate = null;
        if (invoiceType === 'credit' && creditDays) {
            const today = new Date();
            today.setDate(today.getDate() + parseInt(creditDays));
            dueDate = today.toISOString().split('T')[0];
        }
        
        const invoiceData = {
            invoice_id: this.currentInvoiceId,
            customer_id: $('#customerName').data('customer-id') || null,
            customer_name: customerData.customer_name,
            customer_phone: customerData.phone,
            customer_address: customerData.address,
            subtotal: parseFloat($('#subtotalHidden').val()) || 0,
            discount_percent: parseFloat($('#discountPercent').val()) || 0,
            discount_amount: parseFloat($('#discountAmountHidden').val()) || 0,
            tax_percent: 0,
            tax_amount: parseFloat($('#taxAmountHidden').val()) || 0,
            total_amount: parseFloat($('#totalAmountHidden').val()) || 0,
            cash_tender_received: parseFloat($('#cashTenderReceived').val()) || 0,
            upi_amount: parseFloat($('#upiAmount').val()) || 0,
            cash_change: parseFloat($('#cashChange').val()) || 0,
            payment_mode: $('#paymentMode').val() || 'cash',
            notes: $('#notes').val(),
            invoice_type: invoiceType,
            credit_days: creditDays ? parseInt(creditDays) : null,
            due_date: dueDate,
            items: validRows
        };
        
        // Debug: Log the data being sent
        console.log('=== INVOICE DATA BEING SENT ===');
        console.log('Cash Tender Received:', invoiceData.cash_tender_received);
        console.log('UPI Amount:', invoiceData.upi_amount);
        console.log('Cash Change:', invoiceData.cash_change);
        console.log('Payment Mode:', invoiceData.payment_mode);
        console.log('Invoice Type:', invoiceData.invoice_type);
        console.log('Total Amount:', invoiceData.total_amount);
        console.log('==========================');
        
        return invoiceData;
    }

    newInvoice() {
        this.currentInvoiceId = null;
        // Reset saving state
        this.isSaving = false;
        this.setSaveButtonState(true);
        
        // Hide editing badge
        this.hideEditingBadge();
        
        // Clear edit invoice data flag
        if (typeof editInvoiceData !== 'undefined') {
            window.editInvoiceData = null;
        }
        
        $('#salesGrid tbody').empty();
        this.rows = [];
        $('#customerName, #customerPhone, #customerGst, #customerAddress, #notes').val('');
        $('#discountPercent, #discountAmount, #cashTenderReceived, #upiAmount, #cashChange').val('');
        $('#paymentMode').val('cash');
        $('#invoiceType').val('cash'); // Reset invoice type to cash
        $('#creditDays').val('30'); // Reset credit days to default
        $('#creditDaysGroup').hide(); // Hide credit days group
        $('#customerSuggestions').hide();
        $('.product-suggestions').hide();
        this.selectedSuggestionIndex = -1;
        this.currentProductInput = null;
        
        // Clear all row total labels
        $('.row-total-label').remove();
        
        // Clear customer outstanding data
        this.clearCustomerOutstanding();
        
        this.addInitialRow();
        this.updateTotals();
        this.generateInvoiceNumber();
        this.updateNavigationButtons(); // Hide navigation buttons for new invoice
    }

    generateInvoiceNumber() {
        $.ajax({
            url: 'ajax/generate_invoice_no.php',
            method: 'GET',
            dataType: 'json',
            success: (response) => {
                if (response.success) {
                    $('#invoiceNo').text(response.invoice_no);
                }
            }
        });
    }

    loadRecentSales() {
        $.ajax({
            url: 'ajax/get_recent_sales.php',
            method: 'GET',
            dataType: 'json',
            success: (response) => {
                if (response.success) {
                    this.populateRecentSales(response.data);
                    // Populate invoice IDs for navigation (sorted by ID descending - newest first)
                    this.allInvoiceIds = response.data.map(sale => sale.id);
                    this.updateNavigationButtons();
                }
            }
        });
    }

    populateRecentSales(sales) {
        const $container = $('#recentSales');
        $container.empty();
        
        sales.forEach(sale => {
            const statusClass = `status-${sale.payment_status}`;
            const cashReceived = parseFloat(sale.cash_received || 0);
            const upiReceived = parseFloat(sale.upi_received || 0);
            const totalReceived = cashReceived + upiReceived;
            const balanceAmount = parseFloat(sale.balance_amount || 0);
            
            // Create payment status display
            let paymentDetails = '';
            if (sale.payment_status === 'paid') {
                paymentDetails = `<div class="small text-success">Paid: ₹${totalReceived.toFixed(2)}</div>`;
            } else if (sale.payment_status === 'partial') {
                paymentDetails = `<div class="small text-warning">Paid: ₹${totalReceived.toFixed(2)} | Balance: ₹${balanceAmount.toFixed(2)}</div>`;
            } else {
                paymentDetails = `<div class="small text-danger">Balance: ₹${balanceAmount.toFixed(2)}</div>`;
            }
            
            const $item = $(`
                <div class="sales-item" data-invoice-id="${sale.id}" style="cursor: pointer;" onclick="salesManager.editInvoice(${sale.id})" title="Click to edit invoice">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <div class="invoice-no">${sale.invoice_no}</div>
                            <div class="small text-muted">${sale.customer_name || 'Walk-in Customer'}</div>
                            <div class="small">${sale.invoice_date}</div>
                            ${paymentDetails}
                        </div>
                        <div class="text-right">
                            <div class="invoice-amount">₹${parseFloat(sale.total_amount).toFixed(2)}</div>
                            <div class="invoice-status ${statusClass}">${sale.payment_status.toUpperCase()}</div>
                        </div>
                    </div>
                    <div class="mt-2">
                        <button class="action-btn btn-edit" onclick="event.stopPropagation(); salesManager.editInvoice(${sale.id})" title="Edit Invoice">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="action-btn btn-delete" onclick="event.stopPropagation(); salesManager.deleteInvoice(${sale.id})" title="Delete Invoice">
                            <i class="fas fa-trash"></i>
                        </button>
                        <button class="action-btn btn-print" onclick="event.stopPropagation(); salesManager.printInvoiceById(${sale.id})" title="Print Invoice">
                            <i class="fas fa-print"></i>
                        </button>
                    </div>
                </div>
            `);
            $container.append($item);
        });
    }

    editInvoice(invoiceId) {
        $.ajax({
            url: 'ajax/get_invoice.php',
            method: 'POST',
            data: { invoice_id: invoiceId },
            dataType: 'json',
            success: (response) => {
                if (response.success) {
                    this.loadInvoiceData(response.data);
                } else {
                    this.showAlert('Error loading invoice!', 'error');
                }
            }
        });
    }

    deleteInvoice(invoiceId) {
        // Show comprehensive confirmation dialog
        if (!confirm('⚠️ DELETE INVOICE CONFIRMATION ⚠️\n\nThis will permanently delete:\n• Invoice and all its line items\n• Payment allocations to this invoice\n• Customer ledger entries for this invoice\n• Related payment receipts (if no other allocations)\n• Recalculate customer outstanding balance\n\nThis action CANNOT be undone!\n\nAre you absolutely sure you want to proceed?')) {
            return;
        }

        $.ajax({
            url: 'ajax/delete_invoice.php',
            method: 'POST',
            data: JSON.stringify({ invoice_id: invoiceId }),
            contentType: 'application/json',
            dataType: 'json',
            success: (response) => {
                if (response.success) {
                    let successMessage = 'Invoice deleted successfully!';
                    
                    // Add details about cleanup if available
                    if (response.details) {
                        if (response.details.payment_allocations_cleaned > 0) {
                            successMessage += `\n\n✓ Cleaned up ${response.details.payment_allocations_cleaned} payment allocation(s)`;
                        }
                        if (response.details.customer_balance_recalculated) {
                            successMessage += '\n✓ Customer balance recalculated';
                        }
                        successMessage += '\n✓ All related ledger entries removed';
                    }
                    
                    this.showAlert(successMessage, 'success');
                    this.loadRecentSales(); // Refresh the recent sales list
                    
                    // If the deleted invoice is currently being edited, clear the form
                    if (this.currentInvoiceId == invoiceId) {
                        this.newInvoice();
                    }
                } else {
                    this.showAlert('Error deleting invoice: ' + (response.message || 'Unknown error'), 'error');
                }
            },
            error: () => {
                this.showAlert('Error deleting invoice!', 'error');
            }
        });
    }

    loadInvoiceData(invoiceData) {
        this.currentInvoiceId = invoiceData.invoice.id;
        
        // Show editing badge
        this.showEditingBadge(invoiceData.invoice.invoice_no);
        
        // Reset saving state
        this.isSaving = false;
        this.setSaveButtonState(true);
        
        // Clear existing data
        $('#salesGrid tbody').empty();
        this.rows = [];
        
        // Load customer data
        $('#customerName').val(invoiceData.invoice.customer_name);
        $('#customerPhone').val(invoiceData.invoice.customer_phone);
        $('#customerAddress').val(invoiceData.invoice.customer_address);
        $('#notes').val(invoiceData.invoice.notes);
        $('#discountPercent').val(invoiceData.invoice.discount_percent);
        $('#discountAmount').val(invoiceData.invoice.discount_amount || 0);
        $('#taxPercent').val(invoiceData.invoice.tax_percent);
        $('#invoiceNo').text(invoiceData.invoice.invoice_no);
        
        // Fetch customer outstanding balance
        if (invoiceData.invoice.customer_name) {
            this.fetchCustomerOutstanding(invoiceData.invoice.customer_name);
        }
        
        // Load cash tender and change data with backward compatibility
        let cashTenderReceived = parseFloat(invoiceData.invoice.cash_tender_received || 0);
        const upiReceived = parseFloat(invoiceData.invoice.upi_received || 0);
        const cashChange = parseFloat(invoiceData.invoice.cash_change || 0);
        
        // Backward compatibility: if cash_tender_received is 0 but cash_received has value, use cash_received
        if (cashTenderReceived === 0 && invoiceData.invoice.cash_received) {
            cashTenderReceived = parseFloat(invoiceData.invoice.cash_received || 0);
        }
        
        // Debug logging
        console.log('Loading invoice data:', {
            cash_tender_received: invoiceData.invoice.cash_tender_received,
            upi_received: invoiceData.invoice.upi_received,
            cash_change: invoiceData.invoice.cash_change,
            cashTenderReceived: cashTenderReceived,
            upiReceived: upiReceived,
            cashChange: cashChange
        });
        
        $('#cashTenderReceived').val(cashTenderReceived.toFixed(2));
        $('#upiAmount').val(upiReceived.toFixed(2));
        $('#cashChange').val(cashChange.toFixed(2));
        
        // Determine payment mode - first check if it's a credit invoice
        let paymentMode = 'cash';
        if (invoiceData.invoice.invoice_type === 'credit') {
            paymentMode = 'credit';
        } else if (cashTenderReceived > 0 && upiReceived > 0) {
            paymentMode = 'mixed';
        } else if (upiReceived > 0 && cashTenderReceived === 0) {
            paymentMode = 'upi';
        } else if (cashTenderReceived > 0 && upiReceived === 0) {
            paymentMode = 'cash';
        }
        
        $('#paymentMode').val(paymentMode);
        
        // Trigger payment mode change to update UI properly
        this.handlePaymentModeChange(paymentMode);
        
        // Load invoice type and credit days
        $('#invoiceType').val(invoiceData.invoice.invoice_type || 'cash');
        if (invoiceData.invoice.credit_days) {
            $('#creditDays').val(invoiceData.invoice.credit_days);
        }
        
        // Show/hide credit days based on invoice type
        if (invoiceData.invoice.invoice_type === 'credit') {
            $('#creditDaysGroup').show();
        } else {
            $('#creditDaysGroup').hide();
        }
        
        // Load items
        invoiceData.items.forEach((item, index) => {
            this.addRow();
            $(`.code-input[data-row="${index}"]`).val(item.product_code);
            $(`.product-input[data-row="${index}"]`).val(item.product_name);
            $(`.unit-input[data-row="${index}"]`).val(item.unit);
            $(`.qty-input[data-row="${index}"]`).val(item.quantity);
            $(`.rate-input[data-row="${index}"]`).val(item.rate);
            $(`.amount-input[data-row="${index}"]`).val(item.amount);
            
            this.rows[index] = {
                code: item.product_code,
                product: item.product_name,
                unit: item.unit,
                qty: parseFloat(item.quantity),
                rate: parseFloat(item.rate),
                amount: parseFloat(item.amount)
            };
        });
        
        this.updateTotals();
        this.showAlert('Invoice loaded for editing!', 'success');
        
        // Update navigation button states
        this.updateNavigationButtons();
    }

    navigateInvoice(direction) {
        if (this.allInvoiceIds.length === 0) {
            this.showAlert('No invoices available for navigation. Please create some invoices first.', 'info');
            return;
        }

        let targetIndex = -1;
        
        if (this.currentInvoiceId) {
            // Find current invoice in the list
            this.currentInvoiceIndex = this.allInvoiceIds.findIndex(id => id == this.currentInvoiceId);
            
            if (direction === 'prev') {
                targetIndex = this.currentInvoiceIndex - 1;
                if (targetIndex < 0) {
                    this.showAlert('This is the first invoice!', 'info');
                    return;
                }
            } else if (direction === 'next') {
                targetIndex = this.currentInvoiceIndex + 1;
                if (targetIndex >= this.allInvoiceIds.length) {
                    this.showAlert('This is the last invoice!', 'info');
                    return;
                }
            }
        } else {
            // No current invoice, start from beginning or end
            if (direction === 'prev') {
                targetIndex = this.allInvoiceIds.length - 1; // Go to last invoice
            } else {
                targetIndex = 0; // Go to first invoice
            }
        }
        
        if (targetIndex >= 0 && targetIndex < this.allInvoiceIds.length) {
            const targetInvoiceId = this.allInvoiceIds[targetIndex];
            this.editInvoice(targetInvoiceId);
            this.showAlert(`Loading ${direction === 'prev' ? 'previous' : 'next'} invoice...`, 'info');
        }
    }

    updateNavigationButtons() {
        const hasInvoices = this.allInvoiceIds.length > 0;
        const isEditing = this.currentInvoiceId !== null;
        
        if (hasInvoices && isEditing) {
            $('#prevInvoice, #nextInvoice').show();
            
            // Update current position in the list
            this.currentInvoiceIndex = this.allInvoiceIds.findIndex(id => id == this.currentInvoiceId);
            
            // Update button states
            $('#prevInvoice').prop('disabled', this.currentInvoiceIndex <= 0);
            $('#nextInvoice').prop('disabled', this.currentInvoiceIndex >= this.allInvoiceIds.length - 1);
            
            // Update button titles with position info
            const position = this.currentInvoiceIndex + 1;
            const total = this.allInvoiceIds.length;
            $('#prevInvoice').attr('title', `Previous Invoice (${position > 1 ? position - 1 : 'First'} of ${total}) - Ctrl+←`);
            $('#nextInvoice').attr('title', `Next Invoice (${position < total ? position + 1 : 'Last'} of ${total}) - Ctrl+→`);
            
            // Update position indicator
            $('#invoicePosition').text(`${position} of ${total}`).show();
        } else {
            $('#prevInvoice, #nextInvoice').hide();
            $('#invoicePosition').hide();
        }
    }

    printInvoice() {
        if (!this.currentInvoiceId) {
            this.showAlert('Please save the invoice first!', 'error');
            return;
        }
        this.printInvoiceById(this.currentInvoiceId);
    }

    printThermalInvoice() {
        if (!this.currentInvoiceId) {
            this.showAlert('Please save the invoice first!', 'error');
            return;
        }
        window.open(`print_invoice_thermal.php?id=${this.currentInvoiceId}`, '_blank');
    }

    printInvoiceById(invoiceId) {
        window.open(`print_invoice.php?id=${invoiceId}`, '_blank');
    }

    showAlert(message, type) {
        const alertClass = type === 'success' ? 'alert-success' : 'alert-error';
        const $alert = $(`
            <div class="alert ${alertClass}">
                ${message}
            </div>
        `);
        
        $('#alertContainer').empty().append($alert);
        
        setTimeout(() => {
            $alert.fadeOut();
        }, 3000);
    }

    showEditingBadge(invoiceNo) {
        // Update the badge with invoice number
        const $badge = $('.badge-warning');
        if ($badge.length > 0) {
            $badge.find('i').removeClass('fa-edit').addClass('fa-edit');
            $badge.contents().filter(function() {
                return this.nodeType === 3;
            }).remove();
            $badge.append(`Editing Invoice: ${invoiceNo}`);
            $badge.show();
        }
        
        // Also show the back to reports button
        $('a[href="../reports/bill_wise_report.php"].btn-success').show();
    }

    hideEditingBadge() {
        const $badge = $('.badge-warning');
        if ($badge.length > 0) {
            $badge.hide();
        }
        
        // Hide the back to reports button
        $('a[href="../reports/bill_wise_report.php"].btn-success').hide();
    }

    // Customer outstanding balance functions
    fetchCustomerOutstanding(customerName) {
        if (!customerName) return;
        
        $.ajax({
            url: 'ajax/get_customer_outstanding.php',
            method: 'POST',
            data: JSON.stringify({ customer_name: customerName }),
            contentType: 'application/json',
            dataType: 'json',
            success: (response) => {
                if (response.success) {
                    const data = response.data;
                    $('#customerOutstanding').val(data.outstanding_balance.toFixed(2));
                    $('#customerCreditLimit').val(data.credit_limit.toFixed(2));
                    
                    // Show collect payment button if there's outstanding balance
                    if (data.outstanding_balance > 0) {
                        $('#collectPaymentBtn').show();
                    } else {
                        $('#collectPaymentBtn').hide();
                    }
                    
                    // Store customer ID for payment collection
                    $('#customerName').data('customer-id', data.customer_id);
                } else {
                    this.clearCustomerOutstanding();
                }
            },
            error: () => {
                this.clearCustomerOutstanding();
            }
        });
    }

    clearCustomerOutstanding() {
        $('#customerOutstanding').val('0.00');
        $('#customerCreditLimit').val('0.00');
        $('#collectPaymentBtn').hide();
        $('#customerName').removeData('customer-id');
    }

    openPaymentCollectionModal() {
        const customerName = $('#customerName').val().trim();
        const customerId = $('#customerName').data('customer-id');
        const outstandingBalance = parseFloat($('#customerOutstanding').val()) || 0;
        
        if (!customerName || !customerId || outstandingBalance <= 0) {
            this.showAlert('Please select a customer with outstanding balance!', 'error');
            return;
        }
        
        // Populate modal with customer data
        $('#paymentCustomerName').val(customerName);
        $('#paymentCustomerId').val(customerId);
        $('#paymentOutstandingBalance').val(outstandingBalance.toFixed(2));
        $('#paymentRemainingBalance').val(outstandingBalance.toFixed(2));
        
        // Set today's date
        const today = new Date().toISOString().split('T')[0];
        $('#paymentDate').val(today);
        
        // Clear form fields
        $('#paymentAmount').val('');
        $('#paymentMethod').val('');
        $('#paymentReference').val('');
        $('#paymentNotes').val('');
        
        // Load outstanding invoices
        this.loadOutstandingInvoices(customerId);
        
        // Show modal
        $('#paymentCollectionModal').modal('show');
        
        // Focus on payment amount after modal is shown
        $('#paymentCollectionModal').on('shown.bs.modal', function() {
            $('#paymentAmount').focus().select();
        });
    }
    
    loadOutstandingInvoices(customerId) {
        $.ajax({
            url: 'ajax/get_outstanding_invoices.php',
            method: 'POST',
            data: JSON.stringify({ customer_id: customerId }),
            contentType: 'application/json',
            dataType: 'json',
            success: (response) => {
                if (response.success && response.invoices.length > 0) {
                    let invoicesHtml = '<table class="table table-sm table-striped mb-0">';
                    invoicesHtml += '<thead><tr><th>Invoice</th><th>Date</th><th>Total</th><th>Balance</th><th>Status</th></tr></thead><tbody>';
                    
                    response.invoices.forEach(invoice => {
                        const statusBadge = this.getPaymentStatusBadge(invoice.payment_status);
                        invoicesHtml += `
                            <tr>
                                <td>${invoice.invoice_no}</td>
                                <td>${invoice.invoice_date}</td>
                                <td>₹${invoice.total_amount.toFixed(2)}</td>
                                <td>₹${invoice.balance_amount.toFixed(2)}</td>
                                <td>${statusBadge}</td>
                            </tr>
                        `;
                    });
                    
                    invoicesHtml += '</tbody></table>';
                    $('#outstandingInvoicesList').html(invoicesHtml);
                } else {
                    $('#outstandingInvoicesList').html('<p class="text-muted mb-0">No outstanding invoices found.</p>');
                }
            },
            error: () => {
                $('#outstandingInvoicesList').html('<p class="text-danger mb-0">Error loading outstanding invoices.</p>');
            }
        });
    }
    
    getPaymentStatusBadge(status) {
        switch(status) {
            case 'paid':
                return '<span class="badge badge-success">Paid</span>';
            case 'partial':
                return '<span class="badge badge-warning">Partial</span>';
            case 'unpaid':
            default:
                return '<span class="badge badge-danger">Unpaid</span>';
        }
    }

    calculateRemainingBalance() {
        const outstandingBalance = parseFloat($('#paymentOutstandingBalance').val()) || 0;
        const paymentAmount = parseFloat($('#paymentAmount').val()) || 0;
        const remainingBalance = Math.max(0, outstandingBalance - paymentAmount);
        
        $('#paymentRemainingBalance').val(remainingBalance.toFixed(2));
        
        // Validate payment amount
        if (paymentAmount > outstandingBalance) {
            $('#paymentAmount').addClass('is-invalid');
            if (!$('#paymentAmount').siblings('.invalid-feedback').length) {
                $('#paymentAmount').after('<div class="invalid-feedback">Payment amount cannot exceed outstanding balance</div>');
            }
        } else {
            $('#paymentAmount').removeClass('is-invalid');
            $('#paymentAmount').siblings('.invalid-feedback').remove();
        }
    }

    saveCustomerPayment() {
        // Validate form
        const customerId = $('#paymentCustomerId').val();
        const paymentAmount = parseFloat($('#paymentAmount').val()) || 0;
        const paymentMethod = $('#paymentMethod').val();
        const paymentDate = $('#paymentDate').val();
        const outstandingBalance = parseFloat($('#paymentOutstandingBalance').val()) || 0;
        
        if (!customerId || !paymentAmount || !paymentMethod || !paymentDate) {
            this.showAlert('Please fill all required fields!', 'error');
            return;
        }
        
        if (paymentAmount <= 0) {
            this.showAlert('Payment amount must be greater than 0!', 'error');
            return;
        }
        
        if (paymentAmount > outstandingBalance) {
            this.showAlert('Payment amount cannot exceed outstanding balance!', 'error');
            return;
        }
        
        // Disable save button
        $('#savePaymentBtn').prop('disabled', true).text('Saving...');
        
        const paymentData = {
            customer_id: customerId,
            payment_amount: paymentAmount,
            payment_mode: paymentMethod,
            payment_date: paymentDate,
            reference_no: $('#paymentReference').val().trim(),
            notes: $('#paymentNotes').val().trim()
        };
        
        console.log('Sending payment data:', paymentData);
        
        $.ajax({
            url: 'ajax/collect_customer_payment.php',
            method: 'POST',
            data: JSON.stringify(paymentData),
            contentType: 'application/json',
            dataType: 'json',
            success: (response) => {
                console.log('Payment response:', response);
                if (response.success) {
                    let successMessage = `Payment of ₹${paymentAmount.toFixed(2)} collected successfully! Receipt: ${response.data.receipt_no}`;
                    
                    // Add invoice allocation details if available
                    if (response.data.allocated_invoices && response.data.allocated_invoices.length > 0) {
                        successMessage += '\n\nPayment allocated to:';
                        response.data.allocated_invoices.forEach(invoice => {
                            successMessage += `\n• ${invoice.invoice_no}: ₹${invoice.allocated_amount.toFixed(2)} (Status: ${invoice.payment_status.toUpperCase()})`;
                        });
                    }
                    
                    this.showAlert(successMessage, 'success');
                    
                    // Update customer outstanding balance
                    $('#customerOutstanding').val(response.data.new_balance.toFixed(2));
                    
                    // Hide collect payment button if balance is 0
                    if (response.data.new_balance <= 0) {
                        $('#collectPaymentBtn').hide();
                    }
                    
                    // Close modal
                    $('#paymentCollectionModal').modal('hide');
                } else {
                    this.showAlert(response.message || 'Error saving payment!', 'error');
                }
            },
            error: (xhr, status, error) => {
                console.error('Payment AJAX error:', xhr.responseText);
                this.showAlert(`Error saving payment: ${error}`, 'error');
            },
            complete: () => {
                // Re-enable save button
                $('#savePaymentBtn').prop('disabled', false).text('Save Payment');
            }
        });
    }

    // Customer search functionality
    searchCustomers(query) {
        if (query.length < 2) {
            $('#customerSuggestions').hide();
            return;
        }

        const filtered = this.customers.filter(customer => 
            customer.customer_name.toLowerCase().includes(query.toLowerCase()) ||
            customer.phone.includes(query) ||
            (customer.gst_number && customer.gst_number.toLowerCase().includes(query.toLowerCase()))
        );

        this.showCustomerSuggestions(filtered);
    }

    searchCustomersByPhone(query) {
        if (query.length < 3) {
            $('#customerSuggestions').hide();
            return;
        }

        const filtered = this.customers.filter(customer => 
            customer.phone.includes(query)
        );

        this.showCustomerSuggestions(filtered);
    }

    showCustomerSuggestions(customers) {
        const $suggestions = $('#customerSuggestions');
        $suggestions.empty();

        if (customers.length === 0) {
            $suggestions.hide();
            return;
        }

        customers.forEach(customer => {
            const gstDisplay = customer.gst_number ? `<div class="suggestion-gst">GST: ${customer.gst_number}</div>` : '';
            const $item = $(`
                <div class="customer-suggestion-item" data-id="${customer.id}">
                    <div class="suggestion-name">${customer.customer_name}</div>
                    <div class="suggestion-phone">${customer.phone}</div>
                    ${gstDisplay}
                    <div class="suggestion-address">${customer.address || ''}</div>
                </div>
            `);

            $item.on('click', () => {
                this.selectCustomerFromSuggestion(customer);
            });

            $suggestions.append($item);
        });

        $suggestions.show();
    }

    selectCustomerFromSuggestion(customer) {
        $('#customerName').val(customer.customer_name);
        $('#customerPhone').val(customer.phone);
        $('#customerGst').val(customer.gst_number || '');
        $('#customerAddress').val(customer.address || '');
        $('#customerSuggestions').hide();
        
        // Fetch outstanding balance for selected customer
        this.fetchCustomerOutstanding(customer.customer_name);
    }

    // Grid navigation with arrow keys
    handleGridKeyNavigation(e, element) {
        const $element = $(element);
        const row = parseInt($element.data('row'));
        const col = $element.data('col');

        // Handle product suggestions navigation
        if (col === 'product' && this.currentProductInput && $(`#suggestions-${row}`).is(':visible')) {
            if (e.key === 'ArrowDown') {
                e.preventDefault();
                this.navigateProductSuggestions('down', row);
                return;
            } else if (e.key === 'ArrowUp') {
                e.preventDefault();
                this.navigateProductSuggestions('up', row);
                return;
            } else if (e.key === 'Enter') {
                e.preventDefault();
                this.selectCurrentSuggestion(row);
                return;
            } else if (e.key === 'Escape') {
                e.preventDefault();
                $(`#suggestions-${row}`).hide();
                return;
            }
        }

        // Handle Delete key for row deletion
        if (e.key === 'Delete') {
            e.preventDefault();
            this.deleteRow(row);
            return;
        }

        if (e.key === 'ArrowLeft') {
            e.preventDefault();
            this.moveToPreviousCell(row, col);
        } else if (e.key === 'ArrowRight') {
            e.preventDefault();
            this.moveToNextCell(row, col);
        } else if (e.key === 'Enter') {
            this.handleEnterKey(e, row, col);
        }
    }

    navigateProductSuggestions(direction, rowIndex) {
        const $suggestions = $(`#suggestions-${rowIndex} .product-suggestion`);
        const totalSuggestions = $suggestions.length;
        
        if (totalSuggestions === 0) return;

        $suggestions.removeClass('highlighted');

        if (direction === 'down') {
            this.selectedSuggestionIndex = (this.selectedSuggestionIndex + 1) % totalSuggestions;
        } else {
            this.selectedSuggestionIndex = this.selectedSuggestionIndex <= 0 ? totalSuggestions - 1 : this.selectedSuggestionIndex - 1;
        }

        $suggestions.eq(this.selectedSuggestionIndex).addClass('highlighted');
    }

    selectCurrentSuggestion(rowIndex) {
        if (this.selectedSuggestionIndex >= 0 && this.suggestionProducts && this.suggestionProducts[this.selectedSuggestionIndex]) {
            this.selectProduct(this.suggestionProducts[this.selectedSuggestionIndex], rowIndex);
        }
    }

    moveToPreviousCell(row, col) {
        const columns = ['code', 'product', 'qty', 'rate'];
        const currentIndex = columns.indexOf(col);
        
        if (currentIndex > 0) {
            const prevCol = columns[currentIndex - 1];
            $(`.${prevCol}-input[data-row="${row}"]`).focus().select();
        } else if (row > 0) {
            $(`.rate-input[data-row="${row - 1}"]`).focus().select();
        }
    }

    moveToNextCell(row, col) {
        const columns = ['code', 'product', 'qty', 'rate'];
        const currentIndex = columns.indexOf(col);
        
        if (currentIndex < columns.length - 1) {
            const nextCol = columns[currentIndex + 1];
            $(`.${nextCol}-input[data-row="${row}"]`).focus().select();
        } else if (row < this.rows.length - 1) {
            $(`.code-input[data-row="${row + 1}"]`).focus().select();
        }
    }

    handleEnterKey(e, row, col) {
        e.preventDefault();
        
        if (col === 'code') {
            const codeValue = e.target.value.trim();
            if (codeValue) {
                if (this.isScanMode) {
                    // Scan mode behavior
                    this.handleScanModeCode(row, codeValue);
                } else {
                    // Normal mode behavior
                    this.loadProductByCode(row, codeValue);
                    this.focusNext(row, 'qty');
                }
            } else {
                // If code cell is empty, focus to product column
                this.focusNext(row, 'product');
            }
        } else if (col === 'product') {
            // After product selection, focus to quantity
            this.focusNext(row, 'qty');
        } else if (col === 'qty') {
            const qtyValue = $(`.qty-input[data-row="${row}"]`).val();
            if (!qtyValue || qtyValue === '') {
                // Set default quantity to 1 if empty
                $(`.qty-input[data-row="${row}"]`).val('1');
                this.rows[row].qty = 1;
            }
            this.calculateAmount(row);
            
            // Move to next product's code input instead of rate
            // But first validate current row has code and product
            const currentCode = $(`.code-input[data-row="${row}"]`).val().trim();
            const currentProduct = $(`.product-input[data-row="${row}"]`).val().trim();
            
            if (currentCode && currentProduct) {
                if (row === this.rows.length - 1) {
                    this.addRow();
                }
                this.focusNext(row + 1, 'code');
            } else {
                this.showAlert('Please fill code and product name first!', 'warning');
                if (!currentCode) {
                    this.focusNext(row, 'code');
                } else {
                    this.focusNext(row, 'product');
                }
            }
        } else if (col === 'rate') {
            this.calculateAmount(row);
            
            // Move to next row or add new row
            // But first validate current row has code and product
            const currentCode = $(`.code-input[data-row="${row}"]`).val().trim();
            const currentProduct = $(`.product-input[data-row="${row}"]`).val().trim();
            
            if (currentCode && currentProduct) {
                if (row === this.rows.length - 1) {
                    this.addRow();
                }
                this.focusNext(row + 1, 'code');
            } else {
                this.showAlert('Please fill code and product name first!', 'warning');
                if (!currentCode) {
                    this.focusNext(row, 'code');
                } else {
                    this.focusNext(row, 'product');
                }
            }
        }
    }

    // Save new customer if mobile number is new
    saveNewCustomer(customerData) {
        if (!customerData.phone || customerData.phone.length < 10) {
            return;
        }

        // Check if customer already exists
        const existingCustomer = this.customers.find(c => c.phone === customerData.phone);
        if (existingCustomer) {
            return;
        }

        $.ajax({
            url: 'ajax/save_customer.php',
            method: 'POST',
            data: JSON.stringify(customerData),
            contentType: 'application/json',
            dataType: 'json',
            success: (response) => {
                if (response.success) {
                    // Reload customers list
                    this.loadCustomers();
                    this.showAlert('New customer saved successfully!', 'success');
                }
            },
            error: () => {
                console.log('Error saving customer');
            }
        });
    }

    // Product Search Modal Methods
    openProductSearchModal(rowIndex) {
        this.currentModalRow = rowIndex;
        this.modalSelectedIndex = -1;
        this.modalProducts = [];
        
        $('#productSearchModal').addClass('show').fadeIn(300, function() {
            // Focus after animation completes to ensure it works
            $('#modalProductSearch').focus();
        });
        
        // Clear and focus search input immediately (for faster response)
        $('#modalProductSearch').val('');
        
        // Also try to focus with a slight delay to ensure modal is fully visible
        setTimeout(function() {
            $('#modalProductSearch').focus();
        }, 50);
        
        // Clear results
        $('#modalSearchResults').html(`
            <div class="no-results">
                <i class="fas fa-box-open"></i>
                <p>Start typing to search for products...</p>
            </div>
        `);
        
        // Prevent body scroll
        $('body').addClass('modal-open');
    }

    closeProductSearchModal() {
        $('#productSearchModal').fadeOut(300, function() {
            $(this).removeClass('show');
        });
        $('body').removeClass('modal-open');
        
        // Return focus to the code input instead of product input
        if (this.currentModalRow !== null) {
            setTimeout(() => {
                $(`.code-input[data-row="${this.currentModalRow}"]`).focus().select();
            }, 300);
        }
        
        this.currentModalRow = null;
        this.modalSelectedIndex = -1;
        this.modalProducts = [];
    }

    searchProductsInModal(searchTerm) {
        if (!searchTerm || searchTerm.length < 1) {
            $('#modalSearchResults').html(`
                <div class="no-results">
                    <i class="fas fa-box-open"></i>
                    <p>Start typing to search for products...</p>
                </div>
            `);
            this.modalProducts = [];
            return;
        }
        
        const rateType = $('#rateType').val();
        
        $.ajax({
            url: 'ajax/search_products.php',
            method: 'POST',
            data: { 
                search: searchTerm,
                rate_type: rateType,
                limit: 4  // Limit to 4 suggestions for compact display
            },
            dataType: 'json',
            success: (response) => {
                if (response.success && response.data.length > 0) {
                    this.displayModalSearchResults(response.data);
                } else {
                    $('#modalSearchResults').html(`
                        <div class="no-results">
                            <i class="fas fa-search"></i>
                            <p>No products found for "${searchTerm}"</p>
                        </div>
                    `);
                    this.modalProducts = [];
                }
            },
            error: (xhr, status, error) => {
                console.log('Search Error:', error);
                $('#modalSearchResults').html(`
                    <div class="no-results">
                        <i class="fas fa-exclamation-triangle"></i>
                        <p>Error searching products. Please try again.</p>
                    </div>
                `);
                this.modalProducts = [];
            }
        });
    }

    displayModalSearchResults(products) {
        this.modalProducts = products;
        this.modalSelectedIndex = 0; // Auto-select first product
        
        const resultsHtml = products.map((product, index) => {
            const tamilName = product.tamil_name ? `<br><small style="color: #666;">${product.tamil_name}</small>` : '';
            return `
                <div class="modal-product-item ${index === 0 ? 'selected' : ''}" data-index="${index}">
                    <div class="modal-product-icon">
                        <i class="fas fa-leaf"></i>
                    </div>
                    <div class="modal-product-details">
                        <div class="modal-product-name">${product.name}${tamilName}</div>
                        <div class="modal-product-info">
                            <span class="modal-product-code">${product.code}</span>
                            <span class="modal-product-unit">${product.unit}</span>
                            <span class="modal-product-price">₹${product.selling_rate}</span>
                        </div>
                    </div>
                </div>
            `;
        }).join('');
        
        $('#modalSearchResults').html(resultsHtml);
    }

    handleModalKeyNavigation(e) {
        if (e.key === 'ArrowDown') {
            e.preventDefault();
            if (this.modalProducts.length === 0) return;
            this.modalSelectedIndex = (this.modalSelectedIndex + 1) % this.modalProducts.length;
            this.highlightModalSelection();
        } else if (e.key === 'ArrowUp') {
            e.preventDefault();
            if (this.modalProducts.length === 0) return;
            this.modalSelectedIndex = this.modalSelectedIndex <= 0 ? 
                this.modalProducts.length - 1 : this.modalSelectedIndex - 1;
            this.highlightModalSelection();
        } else if (e.key === 'Enter') {
            e.preventDefault();
            if (this.modalProducts.length > 0) {
                // If no product is selected, select the first one
                if (this.modalSelectedIndex < 0) {
                    this.modalSelectedIndex = 0;
                }
                this.selectProductFromModal(this.modalSelectedIndex);
            }
        } else if (e.key === 'Escape') {
            e.preventDefault();
            this.closeProductSearchModal();
        }
    }

    highlightModalSelection() {
        $('.modal-product-item').removeClass('selected');
        if (this.modalSelectedIndex >= 0) {
            $(`.modal-product-item[data-index="${this.modalSelectedIndex}"]`).addClass('selected');
            
            // Scroll to selected item
            const $selected = $(`.modal-product-item[data-index="${this.modalSelectedIndex}"]`);
            const $container = $('#modalSearchResults');
            
            if ($selected.length && $container.length) {
                const containerTop = $container.scrollTop();
                const containerHeight = $container.height();
                const itemTop = $selected.position().top + containerTop;
                const itemHeight = $selected.outerHeight();
                
                if (itemTop < containerTop) {
                    $container.scrollTop(itemTop);
                } else if (itemTop + itemHeight > containerTop + containerHeight) {
                    $container.scrollTop(itemTop + itemHeight - containerHeight);
                }
            }
        }
    }

    selectProductFromModal(index) {
        if (index >= 0 && index < this.modalProducts.length && this.currentModalRow !== null) {
            const product = this.modalProducts[index];
            this.selectProduct(product, this.currentModalRow);
            this.closeProductSearchModal();
        }
    }

    loadInvoiceForEdit(invoiceData) {
        if (!invoiceData) return;
        
        // Set invoice ID for editing mode
        this.currentInvoiceId = invoiceData.id;
        
        // Load customer data
        $('#customerName').val(invoiceData.customer_name || '');
        $('#customerPhone').val(invoiceData.customer_phone || '');
        $('#customerGst').val(invoiceData.customer_gst || '');
        $('#customerAddress').val(invoiceData.customer_address || '');
        
        // Load invoice details
        $('#invoiceNo').text(invoiceData.invoice_no);
        $('#discountPercent').val(invoiceData.discount_percent || '');
        $('#discountAmount').val(invoiceData.discount_amount || '');
        $('#cashTenderReceived').val(invoiceData.cash_received || '');
        $('#upiAmount').val(invoiceData.upi_received || '');
        $('#paymentMode').val(invoiceData.payment_mode || 'cash');
        
        // Clear existing rows
        $('#salesGrid tbody').empty();
        this.rows = [];
        
        // Load invoice items
        if (invoiceData.items && invoiceData.items.length > 0) {
            invoiceData.items.forEach((item, index) => {
                this.addRowForEdit(index, item);
            });
        } else {
            // Add at least one empty row
            this.addRow();
        }
        
        // Update totals
        this.updateTotals();
        
        // Show edit mode indicator
        this.showAlert('Invoice loaded for editing. Make changes and save to update.', 'info');
    }

    addRowForEdit(rowIndex, itemData) {
        const rowHtml = `
            <tr data-row="${rowIndex}">
                <td class="col-code">
                    <input type="text" class="cell-input code-input" data-row="${rowIndex}" data-col="code" 
                           placeholder="Code" value="${itemData.product_code || ''}">
                </td>
                <td class="col-product">
                    <div class="product-input-container">
                        <input type="text" class="cell-input product-input" data-row="${rowIndex}" data-col="product" 
                               placeholder="Product Name" value="${itemData.VegetableName || ''}">
                        <div class="product-suggestions" id="suggestions-${rowIndex}"></div>
                    </div>
                </td>
                <td class="col-unit">
                    <input type="text" class="cell-input unit-input readonly" data-row="${rowIndex}" data-col="unit" 
                           placeholder="Unit" readonly value="${itemData.Unit || ''}">
                </td>
                <td class="col-qty">
                    <input type="number" class="cell-input qty-input" data-row="${rowIndex}" data-col="qty" 
                           placeholder="Qty" step="0.001" value="${itemData.quantity || 1}">
                </td>
                <td class="col-rate">
                    <input type="number" class="cell-input rate-input" data-row="${rowIndex}" data-col="rate" 
                           placeholder="Rate" step="0.01" value="${itemData.rate || 0}">
                </td>
                <td class="col-amount">
                    <input type="number" class="cell-input amount-input readonly" data-row="${rowIndex}" data-col="amount" 
                           placeholder="Amount" readonly value="${itemData.amount || 0}">
                </td>
                <td class="col-action text-center">
                    <button type="button" class="action-btn btn-delete" onclick="salesManager.deleteRow(${rowIndex})" title="Delete Row">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>
        `;
        
        $('#salesGrid tbody').append(rowHtml);
        this.rows.push({
            code: itemData.product_code || '',
            product: itemData.VegetableName || '',
            unit: itemData.Unit || '',
            qty: parseFloat(itemData.quantity) || 1,
            rate: parseFloat(itemData.rate) || 0,
            amount: parseFloat(itemData.amount) || 0
        });
        
        // Bind events for this row
        this.bindRowKeyboardEvents(rowIndex);
        
        // Bind change events for calculation
        $(`.qty-input[data-row="${rowIndex}"], .rate-input[data-row="${rowIndex}"]`).on('input change', () => {
            this.calculateAmount(rowIndex);
        });
        
        // Set default quantity to 1 when quantity field loses focus if empty
        $(`.qty-input[data-row="${rowIndex}"]`).on('blur', function() {
            const qtyValue = $(this).val();
            if (!qtyValue || qtyValue === '' || parseFloat(qtyValue) <= 0) {
                $(this).val('1');
                salesManager.rows[rowIndex].qty = 1;
                salesManager.calculateAmount(rowIndex);
            }
        });
        
        // Bind product input events
        $(`.product-input[data-row="${rowIndex}"]`).on('input', (e) => {
            this.searchProducts(e.target.value, rowIndex);
        });
        
        $(`.code-input[data-row="${rowIndex}"]`).on('blur', (e) => {
            this.loadProductByCode(rowIndex, e.target.value);
        });
    }
}

// Initialize sales manager when document is ready
let salesManager;
$(document).ready(function() {
    salesManager = new SalesManager();
    
    // Check if we're editing an invoice
    if (typeof editInvoiceData !== 'undefined' && editInvoiceData) {
        salesManager.loadInvoiceForEdit(editInvoiceData);
    } else {
        salesManager.generateInvoiceNumber();
    }
    
    // Additional focus handling after page load
    $(window).on('load', function() {
        setTimeout(() => {
            if (salesManager) {
                if (typeof editInvoiceData !== 'undefined' && editInvoiceData) {
                    // Focus on first row when editing
                    salesManager.focusOnFirstCodeInput();
                } else {
                    // Focus on rate type for new invoice
                    $('#rateType').focus();
                }
            }
        }, 100);
    });
});