<?php
session_start();
include_once('../../includes/config.php');

header('Content-Type: application/json');

if (!isset($_SESSION['uid'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit();
}

if (!$con) {
    echo json_encode(['success' => false, 'message' => 'Database connection failed']);
    exit();
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        echo json_encode(['success' => false, 'message' => 'Invalid JSON data']);
        exit();
    }
    
    // Start transaction
    mysqli_autocommit($con, false);
    
    // Extract and validate payment data
    $cashTenderReceived = floatval($input['cash_tender_received'] ?? 0);
    $upiAmount = floatval($input['upi_amount'] ?? 0);
    $cashChange = floatval($input['cash_change'] ?? 0);
    $totalAmount = floatval($input['total_amount'] ?? 0);
    
    // Calculate actual cash received (tender minus change)
    $cashReceived = max(0, $cashTenderReceived - $cashChange);
    $upiReceived = $upiAmount;
    $totalReceived = round($cashReceived + $upiReceived, 2);
    
    // Debug: Log payment calculations
    error_log('DEBUG: Payment Processing - cashTender: ' . $cashTenderReceived . ', upiAmount: ' . $upiAmount . ', cashChange: ' . $cashChange);
    error_log('DEBUG: Payment Calculated - cashReceived: ' . $cashReceived . ', upiReceived: ' . $upiReceived . ', totalReceived: ' . $totalReceived);
    
    // Determine payment status and invoice type based on payment, not form input
    $balanceAmount = round(max(0, $totalAmount - $totalReceived), 2);
    
    if ($balanceAmount == 0 && $totalReceived >= $totalAmount && $totalAmount > 0) {
        // Fully paid, so it's a cash sale regardless of the initial selection.
        $paymentStatus = 'paid';
        $actualInvoiceType = 'cash';
    } else if ($totalReceived > 0 && $totalReceived < $totalAmount) {
        // Partially paid, so it's a credit sale.
        $paymentStatus = 'partial';
        $actualInvoiceType = 'credit';
    } else {
        // Unpaid, so it's a credit sale.
        $paymentStatus = 'unpaid';
        $actualInvoiceType = 'credit';
        $balanceAmount = $totalAmount;
    }
    
    // Debug: Log final payment status
    error_log('DEBUG: Final Status - paymentStatus: ' . $paymentStatus . ', invoiceType: ' . $actualInvoiceType . ', balance: ' . $balanceAmount);
    
    $isUpdate = !empty($input['invoice_id']);
    
    if ($isUpdate) {
        // Update existing invoice
        $invoiceId = intval($input['invoice_id']);
        $customerIdValue = $input['customer_id'] ? intval($input['customer_id']) : 'NULL';
        
        $updateQuery = "UPDATE sales_invoices SET 
            customer_id = " . $customerIdValue . ", 
            customer_name = '" . mysqli_real_escape_string($con, $input['customer_name'] ?? '') . "', 
            customer_phone = '" . mysqli_real_escape_string($con, $input['customer_phone'] ?? '') . "', 
            customer_address = '" . mysqli_real_escape_string($con, $input['customer_address'] ?? '') . "',
            subtotal = " . floatval($input['subtotal'] ?? 0) . ", 
            discount_percent = " . floatval($input['discount_percent'] ?? 0) . ", 
            discount_amount = " . floatval($input['discount_amount'] ?? 0) . ", 
            tax_percent = " . floatval($input['tax_percent'] ?? 0) . ", 
            tax_amount = " . floatval($input['tax_amount'] ?? 0) . ", 
            total_amount = " . $totalAmount . ", 
            balance_amount = " . $balanceAmount . ", 
            cash_received = " . $cashReceived . ",
            upi_received = " . $upiReceived . ",
            payment_status = '" . $paymentStatus . "',
            cash_tender_received = " . $cashTenderReceived . ",
            cash_change = " . $cashChange . ",
            notes = '" . mysqli_real_escape_string($con, $input['notes'] ?? '') . "',
            invoice_type = '" . $actualInvoiceType . "',
            credit_days = " . (isset($input['credit_days']) && $actualInvoiceType === 'credit' ? intval($input['credit_days']) : 'NULL') . ",
            due_date = " . (isset($input['due_date']) && $actualInvoiceType === 'credit' ? "'" . mysqli_real_escape_string($con, $input['due_date']) . "'" : 'NULL') . ",
            updated_date = NOW()
            WHERE id = " . $invoiceId;
        
        if (!mysqli_query($con, $updateQuery)) {
            throw new Exception('Failed to update invoice: ' . mysqli_error($con));
        }
        
        // Delete existing items for update
        $deleteQuery = "DELETE FROM sales_invoice_details WHERE invoice_id = " . $invoiceId;
        if (!mysqli_query($con, $deleteQuery)) {
            throw new Exception('Failed to delete existing items: ' . mysqli_error($con));
        }
        
    } else {
        // Create new invoice
        $date = date('Y-m-d');
        $year = date('Y');
        $month = date('m');
        
        // Generate invoice number
        $lastInvoiceQuery = "SELECT invoice_no FROM sales_invoices WHERE invoice_no LIKE 'INV-" . $year . $month . "-%' ORDER BY CAST(SUBSTRING(invoice_no, -4) AS UNSIGNED) DESC LIMIT 1";
        $lastInvoiceResult = mysqli_query($con, $lastInvoiceQuery);
        
        if ($lastInvoiceResult && mysqli_num_rows($lastInvoiceResult) > 0) {
            $lastInvoice = mysqli_fetch_assoc($lastInvoiceResult);
            $lastNumber = (int)substr($lastInvoice['invoice_no'], -4);
            $newNumber = $lastNumber + 1;
        } else {
            $newNumber = 1;
        }
        
        $invoiceNo = 'INV-' . $year . $month . '-' . str_pad($newNumber, 4, '0', STR_PAD_LEFT);
        $customerIdValue = $input['customer_id'] ? intval($input['customer_id']) : 'NULL';
        
        $insertQuery = "INSERT INTO sales_invoices 
            (invoice_no, invoice_date, invoice_type, sale_counter, customer_id, customer_name, customer_phone, customer_address,
             subtotal, discount_percent, discount_amount, tax_percent, tax_amount, total_amount, 
             balance_amount, cash_received, upi_received, payment_status,
             cash_tender_received, cash_change, notes, credit_days, due_date, created_by) 
            VALUES (
                '" . mysqli_real_escape_string($con, $invoiceNo) . "',
                '$date',
                '" . $actualInvoiceType . "',
                'customer',
                " . $customerIdValue . ",
                '" . mysqli_real_escape_string($con, $input['customer_name'] ?? '') . "',
                '" . mysqli_real_escape_string($con, $input['customer_phone'] ?? '') . "',
                '" . mysqli_real_escape_string($con, $input['customer_address'] ?? '') . "',
                " . floatval($input['subtotal'] ?? 0) . ",
                " . floatval($input['discount_percent'] ?? 0) . ",
                " . floatval($input['discount_amount'] ?? 0) . ",
                " . floatval($input['tax_percent'] ?? 0) . ",
                " . floatval($input['tax_amount'] ?? 0) . ",
                " . $totalAmount . ",
                " . $balanceAmount . ",
                " . $cashReceived . ",
                " . $upiReceived . ",
                '" . $paymentStatus . "',
                " . $cashTenderReceived . ",
                " . $cashChange . ",
                '" . mysqli_real_escape_string($con, $input['notes'] ?? '') . "',
                " . (isset($input['credit_days']) && $actualInvoiceType === 'credit' ? intval($input['credit_days']) : 'NULL') . ",
                " . (isset($input['due_date']) && $actualInvoiceType === 'credit' ? "'" . mysqli_real_escape_string($con, $input['due_date']) . "'" : 'NULL') . ",
                " . intval($_SESSION['uid']) . "
            )";
        
        if (!mysqli_query($con, $insertQuery)) {
            throw new Exception('Failed to create invoice: ' . mysqli_error($con));
        }
        
        $invoiceId = mysqli_insert_id($con);
    }
    
    // Insert invoice items with purchase rate tracking
    foreach ($input['items'] as $item) {
        // Get current purchase rate from vegetables table
        $purchaseRateQuery = "SELECT PurchasedRate FROM tblvegetables WHERE VegetableCode = '" . mysqli_real_escape_string($con, $item['code']) . "' AND Status = 'Active'";
        $purchaseRateResult = mysqli_query($con, $purchaseRateQuery);
        $purchaseRate = 0.00;
        
        if ($purchaseRateResult && mysqli_num_rows($purchaseRateResult) > 0) {
            $purchaseRateRow = mysqli_fetch_assoc($purchaseRateResult);
            $purchaseRate = floatval($purchaseRateRow['PurchasedRate']);
        }
        
        // Calculate profit amount and margin
        $qty = floatval($item['qty']);
        $rate = floatval($item['rate']);
        $profitAmount = ($rate - $purchaseRate) * $qty;
        $profitMarginPercent = $rate > 0 ? (($rate - $purchaseRate) / $rate) * 100 : 0;
        
        $itemQuery = "INSERT INTO sales_invoice_details 
            (invoice_id, product_code, product_name, unit, quantity, rate, purchase_rate, profit_amount, profit_margin_percent, amount) 
            VALUES (
                " . intval($invoiceId) . ",
                '" . mysqli_real_escape_string($con, $item['code']) . "',
                '" . mysqli_real_escape_string($con, $item['product']) . "',
                '" . mysqli_real_escape_string($con, $item['unit']) . "',
                " . $qty . ",
                " . $rate . ",
                " . $purchaseRate . ",
                " . $profitAmount . ",
                " . $profitMarginPercent . ",
                " . floatval($item['amount']) . "
            )";
        
        if (!mysqli_query($con, $itemQuery)) {
            throw new Exception('Failed to insert item: ' . mysqli_error($con));
        }
    }
    
    // Insert payment history if any payment received
    if ($totalReceived > 0) {
        // Determine payment method
        $paymentMethod = 'cash';
        if ($cashReceived > 0 && $upiReceived > 0) {
            $paymentMethod = 'mixed';
        } else if ($upiReceived > 0 && $cashReceived == 0) {
            $paymentMethod = 'upi';
        }
        
        // Check if payment_history table exists
        $tableCheckQuery = "SHOW TABLES LIKE 'payment_history'";
        $tableCheckResult = mysqli_query($con, $tableCheckQuery);

        if ($tableCheckResult && mysqli_num_rows($tableCheckResult) > 0) {
            // For updates, delete existing payment history first
            if ($isUpdate) {
                $deletePaymentQuery = "DELETE FROM payment_history WHERE invoice_id = " . $invoiceId;
                mysqli_query($con, $deletePaymentQuery);
            }
            
            // Insert payment history
            $paymentQuery = "INSERT INTO payment_history 
                (invoice_id, payment_date, cash_amount, upi_amount, total_payment, 
                 cash_tender_received, cash_change, payment_method, created_by) 
                VALUES (
                    " . $invoiceId . ",
                    '" . date('Y-m-d') . "',
                    " . $cashReceived . ",
                    " . $upiReceived . ",
                    " . $totalReceived . ",
                    " . $cashTenderReceived . ",
                    " . $cashChange . ",
                    '" . $paymentMethod . "',
                    " . intval($_SESSION['uid']) . "
                )";
            
            if (!mysqli_query($con, $paymentQuery)) {
                error_log("Payment history insert failed: " . mysqli_error($con));
                throw new Exception('Failed to insert payment history: ' . mysqli_error($con));
            }
            
            error_log("SUCCESS: Payment History inserted - cash: $cashReceived, upi: $upiReceived, method: $paymentMethod");
        }
    }

    // Add customer ledger entry for credit invoices
    if ($actualInvoiceType === 'credit' && !empty($input['customer_id']) && $balanceAmount > 0) {
        // Get current customer balance
        $balanceQuery = "SELECT balance FROM customer_ledger WHERE customer_id = " . intval($input['customer_id']) . " ORDER BY id DESC LIMIT 1";
        $balanceResult = mysqli_query($con, $balanceQuery);
        $currentBalance = 0.00;
        
        if ($balanceResult && mysqli_num_rows($balanceResult) > 0) {
            $balanceRow = mysqli_fetch_assoc($balanceResult);
            $currentBalance = floatval($balanceRow['balance']);
        }
        
        // Calculate new balance (debit increases customer's outstanding)
        $newBalance = $currentBalance + $balanceAmount;
        
        // For updates, check if ledger entry already exists
        if ($isUpdate) {
            $existingLedgerQuery = "SELECT id FROM customer_ledger WHERE reference_id = " . $invoiceId . " AND transaction_type = 'invoice'";
            $existingLedgerResult = mysqli_query($con, $existingLedgerQuery);
            
            if ($existingLedgerResult && mysqli_num_rows($existingLedgerResult) > 0) {
                // Update existing ledger entry
                $ledgerUpdateQuery = "UPDATE customer_ledger SET 
                    debit_amount = " . $balanceAmount . ",
                    balance = " . $newBalance . ",
                    description = 'Credit Invoice: " . ($isUpdate ? "Updated " : "") . ($invoiceNo ?? 'INV-' . $invoiceId) . "'
                    WHERE reference_id = " . $invoiceId . " AND transaction_type = 'invoice'";
                
                if (!mysqli_query($con, $ledgerUpdateQuery)) {
                    error_log("Customer ledger update failed: " . mysqli_error($con));
                }
            } else {
                // Insert new ledger entry for update
                $ledgerInsertQuery = "INSERT INTO customer_ledger 
                    (customer_id, transaction_date, transaction_type, reference_no, reference_id, 
                     debit_amount, credit_amount, balance, description, created_by) 
                    VALUES (
                        " . intval($input['customer_id']) . ",
                        '" . date('Y-m-d') . "',
                        'invoice',
                        '" . mysqli_real_escape_string($con, $invoiceNo ?? 'INV-' . $invoiceId) . "',
                        " . $invoiceId . ",
                        " . $balanceAmount . ",
                        0.00,
                        " . $newBalance . ",
                        'Credit Invoice: " . mysqli_real_escape_string($con, $invoiceNo ?? 'INV-' . $invoiceId) . "',
                        " . intval($_SESSION['uid']) . "
                    )";
                
                if (!mysqli_query($con, $ledgerInsertQuery)) {
                    error_log("Customer ledger insert failed: " . mysqli_error($con));
                }
            }
        } else {
            // Insert new ledger entry
            $ledgerInsertQuery = "INSERT INTO customer_ledger 
                (customer_id, transaction_date, transaction_type, reference_no, reference_id, 
                 debit_amount, credit_amount, balance, description, created_by) 
                VALUES (
                    " . intval($input['customer_id']) . ",
                    '" . date('Y-m-d') . "',
                    'invoice',
                    '" . mysqli_real_escape_string($con, $invoiceNo) . "',
                    " . $invoiceId . ",
                    " . $balanceAmount . ",
                    0.00,
                    " . $newBalance . ",
                    'Credit Invoice: " . mysqli_real_escape_string($con, $invoiceNo) . "',
                    " . intval($_SESSION['uid']) . "
                )";
            
            if (!mysqli_query($con, $ledgerInsertQuery)) {
                error_log("Customer ledger insert failed: " . mysqli_error($con));
                throw new Exception('Failed to create customer ledger entry: ' . mysqli_error($con));
            }
            
            error_log("SUCCESS: Customer ledger entry created for credit invoice");
        }
        
        // Update customer outstanding balance
        $updateCustomerQuery = "UPDATE customers SET outstanding_balance = " . $newBalance . " WHERE id = " . intval($input['customer_id']);
        if (!mysqli_query($con, $updateCustomerQuery)) {
            error_log("Customer outstanding balance update failed: " . mysqli_error($con));
        }
    }

    // Commit transaction
    mysqli_commit($con);
    mysqli_autocommit($con, true);

    echo json_encode([
        'success' => true, 
        'message' => $isUpdate ? 'Invoice updated successfully' : 'Invoice saved successfully',
        'invoice_id' => $invoiceId
    ]);

} catch (Exception $e) {
    // Rollback transaction
    mysqli_rollback($con);
    mysqli_autocommit($con, true);

    // Log detailed error information
    error_log("=== SAVE INVOICE ERROR ===");
    error_log("Error message: " . $e->getMessage());
    error_log("Error file: " . $e->getFile());
    error_log("Error line: " . $e->getLine());
    error_log("Stack trace: " . $e->getTraceAsString());
    error_log("Input data: " . json_encode($input));

    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}
