<?php
session_start();
include_once('../../includes/config.php');

// Set proper headers
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, must-revalidate');

// Disable error display for production
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Authentication check
if (!isset($_SESSION['uid'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Authentication required']);
    exit();
}

// Method validation
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

// Database connection check
if (!$con) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database connection failed']);
    exit();
}

try {
    // Input validation and sanitization
    $customer_id = filter_input(INPUT_POST, 'customer_id', FILTER_VALIDATE_INT) ?? 0;
    $customer_name = trim($_POST['customer_name'] ?? '');
    $payment_date = $_POST['payment_date'] ?? date('Y-m-d');
    $cash_amount = filter_input(INPUT_POST, 'cash_amount', FILTER_VALIDATE_FLOAT) ?? 0;
    $upi_amount = filter_input(INPUT_POST, 'upi_amount', FILTER_VALIDATE_FLOAT) ?? 0;
    $total_payment = filter_input(INPUT_POST, 'total_payment', FILTER_VALIDATE_FLOAT) ?? 0;
    $payment_method = $_POST['payment_method'] ?? 'cash';
    $reference_no = trim($_POST['reference_no'] ?? '');
    $notes = trim($_POST['notes'] ?? '');
    
    // Business logic validation
    if ($total_payment <= 0) {
        echo json_encode(['success' => false, 'message' => 'Payment amount must be greater than zero']);
        exit();
    }
    
    if (abs(($cash_amount + $upi_amount) - $total_payment) > 0.01) {
        echo json_encode(['success' => false, 'message' => 'Cash and UPI amounts must equal total payment']);
        exit();
    }
    
    // Start database transaction
    mysqli_begin_transaction($con);
    
    // Fetch outstanding invoices
    $invoices = [];
    
    if ($customer_id > 0) {
        $query = "SELECT id, invoice_no, total_amount, balance_amount, invoice_date 
                  FROM sales_invoices 
                  WHERE customer_id = ? 
                  AND invoice_type = 'credit' 
                  AND balance_amount > 0 
                  AND (status IS NULL OR status != 'cancelled')
                  ORDER BY invoice_date ASC, id ASC";
        
        $stmt = mysqli_prepare($con, $query);
        mysqli_stmt_bind_param($stmt, 'i', $customer_id);
    } else {
        $query = "SELECT id, invoice_no, total_amount, balance_amount, invoice_date 
                  FROM sales_invoices 
                  WHERE (customer_id IS NULL OR customer_id = 0) 
                  AND customer_name = ? 
                  AND invoice_type = 'credit' 
                  AND balance_amount > 0 
                  AND (status IS NULL OR status != 'cancelled')
                  ORDER BY invoice_date ASC, id ASC";
        
        $stmt = mysqli_prepare($con, $query);
        mysqli_stmt_bind_param($stmt, 's', $customer_name);
    }
    
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    while ($row = mysqli_fetch_assoc($result)) {
        $invoices[] = [
            'id' => intval($row['id']),
            'invoice_no' => $row['invoice_no'],
            'total_amount' => floatval($row['total_amount']),
            'balance_amount' => floatval($row['balance_amount']),
            'invoice_date' => $row['invoice_date']
        ];
    }
    mysqli_stmt_close($stmt);
    
    if (empty($invoices)) {
        mysqli_rollback($con);
        echo json_encode(['success' => false, 'message' => 'No outstanding invoices found']);
        exit();
    }
    
    // Process payment allocation (FIFO)
    $remaining_payment = $total_payment;
    $allocations = [];
    
    foreach ($invoices as $invoice) {
        if ($remaining_payment <= 0.01) break;
        
        $invoice_id = $invoice['id'];
        $invoice_no = $invoice['invoice_no'];
        $balance_amount = $invoice['balance_amount'];
        
        // Calculate payment for this invoice
        $payment_for_invoice = min($remaining_payment, $balance_amount);
        
        if ($payment_for_invoice > 0.01) {
            // Calculate proportional cash and UPI
            $cash_portion = ($total_payment > 0) ? ($cash_amount * $payment_for_invoice / $total_payment) : 0;
            $upi_portion = ($total_payment > 0) ? ($upi_amount * $payment_for_invoice / $total_payment) : 0;
            
            // Calculate new balance
            $new_balance = $balance_amount - $payment_for_invoice;
            $payment_status = ($new_balance <= 0.01) ? 'paid' : 'partial';
            
            // Update invoice
            $update_query = "UPDATE sales_invoices 
                           SET cash_received = COALESCE(cash_received, 0) + ?,
                               upi_received = COALESCE(upi_received, 0) + ?,
                               balance_amount = ?,
                               payment_status = ?
                           WHERE id = ?";
            
            $update_stmt = mysqli_prepare($con, $update_query);
            mysqli_stmt_bind_param($update_stmt, 'dddsi', $cash_portion, $upi_portion, $new_balance, $payment_status, $invoice_id);
            
            if (!mysqli_stmt_execute($update_stmt)) {
                throw new Exception('Failed to update invoice: ' . mysqli_stmt_error($update_stmt));
            }
            mysqli_stmt_close($update_stmt);
            
            // Record payment history
            $history_query = "INSERT INTO payment_history 
                            (invoice_id, payment_date, cash_amount, upi_amount, total_payment, 
                             payment_method, reference_no, notes, created_by) 
                            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $history_stmt = mysqli_prepare($con, $history_query);
            mysqli_stmt_bind_param($history_stmt, 'isdddsssi', 
                $invoice_id, $payment_date, $cash_portion, $upi_portion, 
                $payment_for_invoice, $payment_method, $reference_no, $notes, $_SESSION['uid']);
            
            if (mysqli_stmt_execute($history_stmt)) {
                mysqli_stmt_close($history_stmt);
            }
            
            // Add to allocations
            $allocations[] = [
                'invoice_no' => $invoice_no,
                'amount' => round($payment_for_invoice, 2),
                'new_balance' => round($new_balance, 2)
            ];
            
            $remaining_payment -= $payment_for_invoice;
        }
    }
    
    // Commit transaction
    mysqli_commit($con);
    
    // Generate receipt
    $receipt_no = 'PAY-' . date('YmdHis') . '-' . str_pad($_SESSION['uid'], 3, '0', STR_PAD_LEFT);
    
    // Prepare response
    $allocated_amount = $total_payment - $remaining_payment;
    $message = "Payment processed successfully! Receipt: {$receipt_no}";
    
    if ($remaining_payment > 0.01) {
        $message .= "\nExcess: ₹" . number_format($remaining_payment, 2);
    }
    
    echo json_encode([
        'success' => true,
        'message' => $message,
        'receipt_no' => $receipt_no,
        'allocated_amount' => round($allocated_amount, 2),
        'excess_amount' => round($remaining_payment, 2),
        'allocations' => $allocations,
        'invoices_updated' => count($allocations)
    ]);
    
} catch (Exception $e) {
    // Rollback transaction
    mysqli_rollback($con);
    
    // Log error
    error_log("Payment processing error: " . $e->getMessage());
    
    // Return error response
    echo json_encode([
        'success' => false,
        'message' => 'Payment processing failed: ' . $e->getMessage()
    ]);
}
?>
