<?php
session_start();
include_once('../../includes/config.php');

header('Content-Type: application/json');

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 0); // Don't display errors in output
ini_set('log_errors', 1);

if (!isset($_SESSION['uid'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit();
}

if (!$con) {
    echo json_encode(['success' => false, 'message' => 'Database connection failed: ' . mysqli_connect_error()]);
    exit();
}

try {
    // Get payment data
    $customerId = intval($_POST['customer_id']);
    $customerName = $_POST['customer_name'];
    $paymentDate = $_POST['payment_date'];
    $cashAmount = floatval($_POST['cash_amount']);
    $upiAmount = floatval($_POST['upi_amount']);
    $totalPayment = floatval($_POST['total_payment']);
    $paymentMethod = $_POST['payment_method'];
    $referenceNo = $_POST['reference_no'] ?? '';
    $notes = $_POST['notes'] ?? '';
    
    if ($totalPayment <= 0) {
        echo json_encode(['success' => false, 'message' => 'Invalid payment amount']);
        exit();
    }
    
    // Validate payment amounts match total
    if (abs(($cashAmount + $upiAmount) - $totalPayment) > 0.01) {
        echo json_encode(['success' => false, 'message' => 'Payment amounts do not match total']);
        exit();
    }
    
    // Start transaction
    mysqli_autocommit($con, false);
    
    // Generate receipt number
    $receiptQuery = "SELECT MAX(CAST(SUBSTRING(receipt_no, -4) AS UNSIGNED)) as max_receipt 
                     FROM payment_receipts 
                     WHERE receipt_no LIKE 'RCP-" . date('Ym') . "-%'";
    $receiptResult = mysqli_query($con, $receiptQuery);
    
    if (!$receiptResult) {
        throw new Exception('Failed to query receipt numbers: ' . mysqli_error($con));
    }
    
    $receiptRow = mysqli_fetch_assoc($receiptResult);
    $nextReceiptNum = ($receiptRow['max_receipt'] ?? 0) + 1;
    $receiptNo = 'RCP-' . date('Ym') . '-' . str_pad($nextReceiptNum, 4, '0', STR_PAD_LEFT);
    
    // Insert payment receipt
    $receiptQuery = "INSERT INTO payment_receipts 
        (receipt_no, receipt_date, customer_id, customer_name, payment_mode, 
         cash_amount, upi_amount, total_amount, notes, created_by)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $stmt = $con->prepare($receiptQuery);
    if (!$stmt) {
        throw new Exception('Failed to prepare receipt query: ' . mysqli_error($con));
    }
    
    $stmt->bind_param('ssissdddsi', 
        $receiptNo, $paymentDate, $customerId, $customerName, $paymentMethod,
        $cashAmount, $upiAmount, $totalPayment, $notes, $_SESSION['uid']);
    
    if (!$stmt->execute()) {
        throw new Exception('Failed to insert payment receipt: ' . $stmt->error);
    }
    
    $receiptId = $con->insert_id;
    
    // Get outstanding invoices for this customer (oldest first)
    if ($customerId > 0) {
        // Query by customer ID
        $invoicesQuery = "SELECT id, invoice_no, balance_amount 
                          FROM sales_invoices 
                          WHERE customer_id = ? 
                          AND invoice_type = 'credit' 
                          AND balance_amount > 0 
                          AND status = 'active'
                          ORDER BY invoice_date ASC";
        
        $stmt = $con->prepare($invoicesQuery);
        $stmt->bind_param('i', $customerId);
    } else {
        // Query by customer name for customers without proper customer_id
        $invoicesQuery = "SELECT id, invoice_no, balance_amount 
                          FROM sales_invoices 
                          WHERE (customer_id IS NULL OR customer_id = 0)
                          AND customer_name = ?
                          AND invoice_type = 'credit' 
                          AND balance_amount > 0 
                          AND status = 'active'
                          ORDER BY invoice_date ASC";
        
        $stmt = $con->prepare($invoicesQuery);
        $stmt->bind_param('s', $customerName);
    }
    $stmt->execute();
    $invoicesResult = $stmt->get_result();
    
    $remainingPayment = $totalPayment;
    $allocations = [];
    
    // Allocate payment to invoices (FIFO - First In, First Out)
    while ($invoice = $invoicesResult->fetch_assoc() && $remainingPayment > 0) {
        $invoiceId = $invoice['id'];
        $invoiceNo = $invoice['invoice_no'];
        $outstandingAmount = floatval($invoice['balance_amount']);
        
        $allocationAmount = min($remainingPayment, $outstandingAmount);
        
        if ($allocationAmount > 0) {
            // Update invoice
            $newBalance = $outstandingAmount - $allocationAmount;
            $newCashReceived = 0;
            $newUpiReceived = 0;
            
            // Proportionally allocate cash and UPI
            if ($totalPayment > 0) {
                $cashProportion = $cashAmount / $totalPayment;
                $upiProportion = $upiAmount / $totalPayment;
                
                $newCashReceived = $allocationAmount * $cashProportion;
                $newUpiReceived = $allocationAmount * $upiProportion;
            }
            
            // Determine payment status
            $paymentStatus = ($newBalance <= 0.01) ? 'paid' : 'partial';
            
            $updateInvoiceQuery = "UPDATE sales_invoices SET 
                cash_received = cash_received + ?,
                upi_received = upi_received + ?,
                balance_amount = ?,
                payment_status = ?
                WHERE id = ?";
            
            $stmt = $con->prepare($updateInvoiceQuery);
            $stmt->bind_param('dddsi', $newCashReceived, $newUpiReceived, $newBalance, $paymentStatus, $invoiceId);
            
            if (!$stmt->execute()) {
                throw new Exception('Failed to update invoice: ' . $stmt->error);
            }
            
            // Insert payment allocation
            $allocationQuery = "INSERT INTO payment_allocations 
                (payment_receipt_id, invoice_id, allocated_amount) 
                VALUES (?, ?, ?)";
            
            $stmt = $con->prepare($allocationQuery);
            $stmt->bind_param('iid', $receiptId, $invoiceId, $allocationAmount);
            
            if (!$stmt->execute()) {
                throw new Exception('Failed to insert payment allocation: ' . $stmt->error);
            }
            
            // Insert payment history
            $paymentHistoryQuery = "INSERT INTO payment_history 
                (invoice_id, payment_date, cash_amount, upi_amount, total_payment, payment_method, reference_no, notes, created_by) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $stmt = $con->prepare($paymentHistoryQuery);
            $stmt->bind_param('isdddssi', 
                $invoiceId, $paymentDate, $newCashReceived, $newUpiReceived, 
                $allocationAmount, $paymentMethod, $referenceNo, $notes, $_SESSION['uid']);
            
            if (!$stmt->execute()) {
                throw new Exception('Failed to insert payment history: ' . $stmt->error);
            }
            
            $allocations[] = [
                'invoice_no' => $invoiceNo,
                'amount' => $allocationAmount,
                'remaining_balance' => $newBalance
            ];
            
            $remainingPayment -= $allocationAmount;
        }
    }
    
    // Update customer outstanding balance
    if ($customerId > 0) {
        $customerUpdateQuery = "UPDATE customers 
            SET outstanding_balance = outstanding_balance - ? 
            WHERE id = ?";
        
        $stmt = $con->prepare($customerUpdateQuery);
        $paidAmount = $totalPayment - $remainingPayment;
        $stmt->bind_param('di', $paidAmount, $customerId);
        
        if (!$stmt->execute()) {
            throw new Exception('Failed to update customer outstanding: ' . $stmt->error);
        }
        
        // Add entry to customer ledger
        $ledgerQuery = "INSERT INTO customer_ledger 
            (customer_id, transaction_date, transaction_type, reference_no, reference_id, credit_amount, description, created_by) 
            VALUES (?, ?, 'payment', ?, ?, ?, ?, ?)";
        
        $description = "Payment received - Receipt: $receiptNo";
        if ($paymentMethod === 'mixed') {
            $description .= " (Cash: ₹$cashAmount, UPI: ₹$upiAmount)";
        } else {
            $description .= " (" . ucfirst($paymentMethod) . ": ₹$totalPayment)";
        }
        
        $stmt = $con->prepare($ledgerQuery);
        $stmt->bind_param('issiisi', 
            $customerId, $paymentDate, $receiptNo, $receiptId, $paidAmount, $description, $_SESSION['uid']);
        
        if (!$stmt->execute()) {
            throw new Exception('Failed to update customer ledger: ' . $stmt->error);
        }
    }
    
    // Commit transaction
    mysqli_commit($con);
    mysqli_autocommit($con, true);
    
    $message = "Payment processed successfully! Receipt No: $receiptNo";
    if ($remainingPayment > 0) {
        $message .= " (Excess amount: ₹" . number_format($remainingPayment, 2) . " - consider refund or credit note)";
    }
    
    echo json_encode([
        'success' => true, 
        'message' => $message,
        'receipt_no' => $receiptNo,
        'allocations' => $allocations,
        'excess_amount' => $remainingPayment
    ]);
    
} catch (Exception $e) {
    // Rollback transaction
    mysqli_rollback($con);
    mysqli_autocommit($con, true);
    
    echo json_encode(['success' => false, 'message' => 'Error processing payment: ' . $e->getMessage()]);
}
?>
