<?php
header('Content-Type: application/json');
include_once('../../includes/config.php');

$response = ['success' => false, 'invoices' => [], 'debug' => []];

try {
    // Handle both JSON and POST data
    $customerId = 0;
    $customerName = '';
    
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (isset($_POST['customer_id'])) {
            $customerId = intval($_POST['customer_id']);
        }
        if (isset($_POST['customer_name'])) {
            $customerName = trim($_POST['customer_name']);
        }
    } else {
        $input = json_decode(file_get_contents('php://input'), true);
        $customerId = intval($input['customer_id'] ?? 0);
        $customerName = trim($input['customer_name'] ?? '');
    }
    
    // Debug information
    $response['debug'] = [
        'customer_id' => $customerId,
        'customer_name' => $customerName,
        'method' => $_SERVER['REQUEST_METHOD']
    ];
    
    $invoices = [];
    $total_outstanding = 0;
    
    if ($customerId > 0) {
        // Query by customer ID with payment allocation info
        $sql = "SELECT 
                    si.id, si.invoice_no, si.invoice_date, si.due_date, 
                    si.total_amount, si.balance_amount, si.payment_status,
                    (si.total_amount - si.balance_amount) as paid_amount,
                    COALESCE(SUM(pa.allocated_amount), 0) as total_allocated
                FROM sales_invoices si
                LEFT JOIN payment_allocations pa ON si.id = pa.invoice_id
                WHERE si.customer_id = ? 
                AND si.invoice_type = 'credit'
                AND si.balance_amount > 0
                AND si.status = 'active'
                GROUP BY si.id
                ORDER BY si.invoice_date ASC";
        
        $stmt = $con->prepare($sql);
        $stmt->bind_param('i', $customerId);
        $stmt->execute();
        $result = $stmt->get_result();
        
        while ($row = $result->fetch_assoc()) {
            $invoices[] = [
                'id' => $row['id'],
                'invoice_no' => $row['invoice_no'],
                'invoice_date' => date('d-m-Y', strtotime($row['invoice_date'])),
                'due_date' => date('d-m-Y', strtotime($row['due_date'])),
                'total_amount' => (float)$row['total_amount'],
                'paid_amount' => (float)$row['paid_amount'],
                'balance_amount' => (float)$row['balance_amount'],
                'payment_status' => $row['payment_status'],
                'total_allocated' => (float)$row['total_allocated']
            ];
            $total_outstanding += (float)$row['balance_amount'];
        }
        
        $response['debug']['query_type'] = 'by_customer_id';
        $response['debug']['rows_found'] = count($invoices);
        $response['summary'] = [
            'total_outstanding' => $total_outstanding,
            'invoice_count' => count($invoices)
        ];
        
    } elseif (!empty($customerName)) {
        // Query by customer name for customers without proper customer_id
        $sql = "SELECT id, invoice_no, invoice_date, due_date, total_amount, 
                       balance_amount
                FROM sales_invoices 
                WHERE (customer_id IS NULL OR customer_id = 0)
                AND customer_name = ?
                AND invoice_type = 'credit'
                AND balance_amount > 0
                AND status = 'active'
                ORDER BY invoice_date ASC";
        
        $stmt = $con->prepare($sql);
        $stmt->bind_param('s', $customerName);
        $stmt->execute();
        $result = $stmt->get_result();
        
        while ($row = $result->fetch_assoc()) {
            $invoices[] = [
                'id' => $row['id'],
                'invoice_no' => $row['invoice_no'],
                'invoice_date' => date('d-m-Y', strtotime($row['invoice_date'])),
                'due_date' => date('d-m-Y', strtotime($row['due_date'])),
                'total_amount' => $row['total_amount'],
                'balance_amount' => $row['balance_amount']
            ];
        }
        
        $response['debug']['query_type'] = 'by_customer_name';
        $response['debug']['rows_found'] = count($invoices);
        
    } else {
        $response['debug']['query_type'] = 'no_valid_criteria';
        $response['message'] = 'No valid customer ID or customer name provided';
    }
    
    $response['success'] = true;
    $response['invoices'] = $invoices;
    
} catch (Exception $e) {
    $response['message'] = 'Error loading outstanding invoices: ' . $e->getMessage();
    $response['debug']['error'] = $e->getMessage();
}

echo json_encode($response);
?>
