<?php
session_start();
include_once('../../includes/config.php');

header('Content-Type: application/json');

if (!isset($_SESSION['uid'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit();
}

if (!$con) {
    echo json_encode(['success' => false, 'message' => 'Database connection failed']);
    exit();
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['invoice_id'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid data: invoice_id is required']);
        exit();
    }
    
    $invoiceId = intval($input['invoice_id']);
    
    if ($invoiceId <= 0) {
        echo json_encode(['success' => false, 'message' => 'Invalid invoice ID']);
        exit();
    }
    
    // Start transaction
    mysqli_autocommit($con, false);
    
    // First, get invoice details to check if it's a credit invoice and get customer info
    $invoiceQuery = "SELECT customer_id, invoice_type, balance_amount, total_amount FROM sales_invoices WHERE id = " . $invoiceId;
    $invoiceResult = mysqli_query($con, $invoiceQuery);
    
    if (!$invoiceResult || mysqli_num_rows($invoiceResult) == 0) {
        throw new Exception('Invoice not found');
    }
    
    $invoice = mysqli_fetch_assoc($invoiceResult);
    $customerId = $invoice['customer_id'];
    $invoiceType = $invoice['invoice_type'];
    $balanceAmount = floatval($invoice['balance_amount']);
    
    // If it's a credit invoice with outstanding balance, clean up all related payment records
    if ($invoiceType === 'credit' && $customerId) {
        
        // Step 1: Get all payment allocations for this invoice
        $paymentAllocQuery = "SELECT pa.payment_receipt_id, pa.allocated_amount, pr.receipt_no 
                             FROM payment_allocations pa 
                             JOIN payment_receipts pr ON pa.payment_receipt_id = pr.id 
                             WHERE pa.invoice_id = " . $invoiceId;
        $paymentAllocResult = mysqli_query($con, $paymentAllocQuery);
        
        $affectedReceipts = [];
        if ($paymentAllocResult && mysqli_num_rows($paymentAllocResult) > 0) {
            while ($allocRow = mysqli_fetch_assoc($paymentAllocResult)) {
                $affectedReceipts[] = [
                    'receipt_id' => $allocRow['payment_receipt_id'],
                    'receipt_no' => $allocRow['receipt_no'],
                    'allocated_amount' => $allocRow['allocated_amount']
                ];
            }
        }
        
        // Step 2: Delete payment allocations for this invoice
        $deleteAllocationsQuery = "DELETE FROM payment_allocations WHERE invoice_id = " . $invoiceId;
        if (!mysqli_query($con, $deleteAllocationsQuery)) {
            error_log("Failed to delete payment allocations: " . mysqli_error($con));
        }
        
        // Step 3: For each affected payment receipt, check if it has other allocations
        foreach ($affectedReceipts as $receipt) {
            // Check if this receipt has other allocations
            $checkOtherAllocQuery = "SELECT COUNT(*) as count FROM payment_allocations WHERE payment_receipt_id = " . $receipt['receipt_id'];
            $checkResult = mysqli_query($con, $checkOtherAllocQuery);
            
            if ($checkResult) {
                $checkRow = mysqli_fetch_assoc($checkResult);
                if ($checkRow['count'] == 0) {
                    // No other allocations, delete the entire payment receipt and its ledger entry
                    $deleteReceiptQuery = "DELETE FROM payment_receipts WHERE id = " . $receipt['receipt_id'];
                    mysqli_query($con, $deleteReceiptQuery);
                    
                    // Delete the corresponding customer ledger entry
                    $deleteLedgerQuery = "DELETE FROM customer_ledger 
                                         WHERE reference_no = '" . $receipt['receipt_no'] . "' 
                                         AND customer_id = " . $customerId . " 
                                         AND transaction_type = 'payment'";
                    mysqli_query($con, $deleteLedgerQuery);
                } else {
                    // Adjust the payment receipt amount (reduce by the allocated amount)
                    $adjustReceiptQuery = "UPDATE payment_receipts 
                                          SET total_amount = total_amount - " . $receipt['allocated_amount'] . " 
                                          WHERE id = " . $receipt['receipt_id'];
                    mysqli_query($con, $adjustReceiptQuery);
                    
                    // Adjust the customer ledger entry
                    $adjustLedgerQuery = "UPDATE customer_ledger 
                                         SET credit_amount = credit_amount - " . $receipt['allocated_amount'] . " 
                                         WHERE reference_no = '" . $receipt['receipt_no'] . "' 
                                         AND customer_id = " . $customerId . " 
                                         AND transaction_type = 'payment'";
                    mysqli_query($con, $adjustLedgerQuery);
                }
            }
        }
        
        // Step 4: Remove the invoice ledger entry
        $deleteInvoiceLedgerQuery = "DELETE FROM customer_ledger 
                                    WHERE reference_id = " . $invoiceId . " 
                                    AND transaction_type = 'invoice' 
                                    AND customer_id = " . $customerId;
        if (!mysqli_query($con, $deleteInvoiceLedgerQuery)) {
            error_log("Failed to delete invoice ledger entry: " . mysqli_error($con));
        }
        
        // Step 5: Recalculate customer outstanding balance
        $recalcBalanceQuery = "SELECT COALESCE(SUM(debit_amount - credit_amount), 0) as balance 
                               FROM customer_ledger 
                               WHERE customer_id = " . $customerId;
        $recalcResult = mysqli_query($con, $recalcBalanceQuery);
        
        if ($recalcResult) {
            $recalcRow = mysqli_fetch_assoc($recalcResult);
            $newBalance = floatval($recalcRow['balance']);
            
            // Update customer outstanding balance
            $updateCustomerQuery = "UPDATE customers SET outstanding_balance = " . $newBalance . " WHERE id = " . $customerId;
            if (!mysqli_query($con, $updateCustomerQuery)) {
                error_log("Failed to update customer outstanding balance: " . mysqli_error($con));
            }
        }
    }
    
    // Delete invoice items first (foreign key constraint)
    $deleteItemsQuery = "DELETE FROM sales_invoice_details WHERE invoice_id = " . $invoiceId;
    if (!mysqli_query($con, $deleteItemsQuery)) {
        throw new Exception('Failed to delete invoice items: ' . mysqli_error($con));
    }
    
    // Delete payment history if exists
    $deletePaymentQuery = "DELETE FROM payment_history WHERE invoice_id = " . $invoiceId;
    mysqli_query($con, $deletePaymentQuery); // Don't throw error if table doesn't exist
    
    // Delete the main invoice
    $deleteInvoiceQuery = "DELETE FROM sales_invoices WHERE id = " . $invoiceId;
    if (!mysqli_query($con, $deleteInvoiceQuery)) {
        throw new Exception('Failed to delete invoice: ' . mysqli_error($con));
    }
    
    // Check if any rows were actually deleted
    if (mysqli_affected_rows($con) == 0) {
        throw new Exception('Invoice not found or already deleted');
    }
    
    // Commit transaction
    mysqli_commit($con);
    mysqli_autocommit($con, true);
    
    // Log successful deletion with details
    $logMessage = "Invoice ID {$invoiceId} deleted successfully.";
    if (!empty($affectedReceipts)) {
        $logMessage .= " Cleaned up " . count($affectedReceipts) . " payment allocation(s).";
    }
    error_log($logMessage);
    
    echo json_encode([
        'success' => true, 
        'message' => 'Invoice and all related records deleted successfully',
        'details' => [
            'invoice_id' => $invoiceId,
            'payment_allocations_cleaned' => count($affectedReceipts ?? []),
            'customer_balance_recalculated' => ($invoiceType === 'credit' && $customerId) ? true : false
        ]
    ]);
    
} catch (Exception $e) {
    // Rollback transaction
    mysqli_rollback($con);
    mysqli_autocommit($con, true);
    
    // Log detailed error information
    error_log("=== DELETE INVOICE ERROR ===");
    error_log("Error message: " . $e->getMessage());
    error_log("Error file: " . $e->getFile());
    error_log("Error line: " . $e->getLine());
    error_log("Input data: " . json_encode($input));
    
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}
?>
