<?php
session_start();
include_once('../includes/config.php');
if (!isset($_SESSION['uid'])) {
    header('location:../logout.php');
    exit();
}

// Handle CSV Export
if (isset($_GET['export']) && $_GET['export'] == 'csv') {
    $days_filter = $_GET['days'] ?? '';
    $customer_id = $_GET['customer_id'] ?? '';
    $amount_filter = $_GET['amount_filter'] ?? '';
    
    // Build query for export
    $query = "SELECT 
                si.invoice_no,
                si.invoice_date,
                COALESCE(c.customer_name, si.customer_name, 'Walk-in Customer') as customer_name,
                COALESCE(c.phone, si.customer_phone, '') as phone,
                COALESCE(c.email, '') as email,
                si.total_amount,
                si.cash_received,
                si.upi_received,
                si.balance_amount,
                si.payment_status,
                DATEDIFF(CURDATE(), si.invoice_date) as days_overdue,
                si.notes
              FROM sales_invoices si
              LEFT JOIN customers c ON si.customer_id = c.id
              WHERE si.status = 'active' 
              AND si.payment_status IN ('unpaid', 'partial') 
              AND si.balance_amount > 0";
    
    $conditions = [];
    if ($days_filter) {
        switch($days_filter) {
            case '7': $conditions[] = "DATEDIFF(CURDATE(), si.invoice_date) <= 7"; break;
            case '30': $conditions[] = "DATEDIFF(CURDATE(), si.invoice_date) <= 30"; break;
            case '60': $conditions[] = "DATEDIFF(CURDATE(), si.invoice_date) <= 60"; break;
            case '90': $conditions[] = "DATEDIFF(CURDATE(), si.invoice_date) > 90"; break;
        }
    }
    if ($customer_id) $conditions[] = "si.customer_id = '$customer_id'";
    if ($amount_filter) {
        switch($amount_filter) {
            case '500': $conditions[] = "si.balance_amount <= 500"; break;
            case '1000': $conditions[] = "si.balance_amount BETWEEN 501 AND 1000"; break;
            case '5000': $conditions[] = "si.balance_amount BETWEEN 1001 AND 5000"; break;
            case '5000+': $conditions[] = "si.balance_amount > 5000"; break;
        }
    }
    
    if (!empty($conditions)) {
        $query .= " AND " . implode(" AND ", $conditions);
    }
    
    $query .= " ORDER BY si.invoice_date ASC";
    $result = mysqli_query($con, $query);
    
    // Generate CSV
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="pending_payments_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    fputcsv($output, ['Invoice No', 'Date', 'Customer Name', 'Phone', 'Email', 'Total Amount', 'Cash Received', 'UPI Received', 'Balance Amount', 'Payment Status', 'Days Overdue', 'Notes']);
    
    while ($row = mysqli_fetch_assoc($result)) {
        fputcsv($output, [
            $row['invoice_no'],
            $row['invoice_date'],
            $row['customer_name'],
            $row['phone'],
            $row['email'],
            $row['total_amount'],
            $row['cash_received'],
            $row['upi_received'],
            $row['balance_amount'],
            $row['payment_status'],
            $row['days_overdue'],
            $row['notes']
        ]);
    }
    
    fclose($output);
    exit();
}

// Get filter values
$days_filter = $_GET['days'] ?? '';
$customer_id = $_GET['customer_id'] ?? '';
$amount_filter = $_GET['amount_filter'] ?? '';

// Build main query
$query = "SELECT 
            si.id,
            si.invoice_no,
            si.invoice_date,
            COALESCE(c.customer_name, si.customer_name, 'Walk-in Customer') as customer_name,
            COALESCE(c.phone, si.customer_phone, '') as phone,
            COALESCE(c.email, '') as email,
            si.total_amount,
            si.cash_received,
            si.upi_received,
            si.balance_amount,
            si.payment_status,
            DATEDIFF(CURDATE(), si.invoice_date) as days_overdue,
            si.notes,
            si.customer_id
          FROM sales_invoices si
          LEFT JOIN customers c ON si.customer_id = c.id
          WHERE si.status = 'active' 
          AND si.payment_status IN ('unpaid', 'partial') 
          AND si.balance_amount > 0";

$conditions = [];
if ($days_filter) {
    switch($days_filter) {
        case '7': $conditions[] = "DATEDIFF(CURDATE(), si.invoice_date) <= 7"; break;
        case '30': $conditions[] = "DATEDIFF(CURDATE(), si.invoice_date) <= 30"; break;
        case '60': $conditions[] = "DATEDIFF(CURDATE(), si.invoice_date) <= 60"; break;
        case '90': $conditions[] = "DATEDIFF(CURDATE(), si.invoice_date) > 90"; break;
    }
}
if ($customer_id) $conditions[] = "si.customer_id = '$customer_id'";
if ($amount_filter) {
    switch($amount_filter) {
        case '500': $conditions[] = "si.balance_amount <= 500"; break;
        case '1000': $conditions[] = "si.balance_amount BETWEEN 501 AND 1000"; break;
        case '5000': $conditions[] = "si.balance_amount BETWEEN 1001 AND 5000"; break;
        case '5000+': $conditions[] = "si.balance_amount > 5000"; break;
    }
}

if (!empty($conditions)) {
    $query .= " AND " . implode(" AND ", $conditions);
}

$query .= " ORDER BY si.invoice_date ASC";
$result = mysqli_query($con, $query);

// Get customers for filter dropdown
$customers = mysqli_query($con, "SELECT DISTINCT c.id, c.customer_name 
                                FROM customers c 
                                INNER JOIN sales_invoices si ON c.id = si.customer_id 
                                WHERE si.payment_status IN ('unpaid', 'partial') 
                                AND si.balance_amount > 0 
                                AND si.status = 'active'
                                ORDER BY c.customer_name");

// Get summary totals
$summary_query = "SELECT 
                    COUNT(si.id) as total_pending_invoices,
                    SUM(si.balance_amount) as total_outstanding,
                    COUNT(DISTINCT COALESCE(si.customer_id, CONCAT('walk_', si.customer_name))) as unique_customers,
                    AVG(DATEDIFF(CURDATE(), si.invoice_date)) as avg_days_overdue,
                    SUM(CASE WHEN DATEDIFF(CURDATE(), si.invoice_date) > 30 THEN si.balance_amount ELSE 0 END) as overdue_30_amount,
                    SUM(CASE WHEN DATEDIFF(CURDATE(), si.invoice_date) > 60 THEN si.balance_amount ELSE 0 END) as overdue_60_amount
                  FROM sales_invoices si
                  WHERE si.status = 'active' 
                  AND si.payment_status IN ('unpaid', 'partial') 
                  AND si.balance_amount > 0";

if (!empty($conditions)) {
    $summary_query .= " AND " . implode(" AND ", $conditions);
}

$summary = mysqli_fetch_assoc(mysqli_query($con, $summary_query));
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Pending Payments Report | Vegetable Shop</title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    
    <!-- Custom fonts for this template-->
    <link href="../vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">

    <!-- Custom styles for this template-->
    <link href="../css/sb-admin-2.min.css" rel="stylesheet">
    <link href="../css/enhanced-admin.css" rel="stylesheet">
    
    <!-- DataTables CSS -->
    <link href="../vendor/datatables/dataTables.bootstrap4.min.css" rel="stylesheet">
    
    <style>
        @media print {
            .no-print { display: none !important; }
            .card { border: none !important; box-shadow: none !important; }
            .card-body { padding: 0 !important; }
            body { font-size: 12px !important; }
            .table th, .table td { padding: 0.3rem !important; }
            
            /* 3-inch printer styles */
            @page { size: 80mm auto; margin: 5mm; }
            body { width: 70mm; font-size: 10px; }
            .table { font-size: 9px; }
            .card-header { font-size: 11px; }
        }
        
        .overdue-urgent { background-color: #f8d7da !important; }
        .overdue-warning { background-color: #fff3cd !important; }
        .overdue-mild { background-color: #d4edda !important; }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('../includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('../includes/topbar.php');?>
                <div class="container-fluid">
                    
                    <!-- Page Heading -->
                    <div class="d-sm-flex align-items-center justify-content-between mb-4 no-print">
                        <h1 class="h2 mb-0 text-gradient font-weight-bold">
                            <i class="fas fa-exclamation-triangle mr-2 text-warning"></i>Pending Payments Report
                        </h1>
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb">
                                <li class="breadcrumb-item"><a href="../dashboard.php">Dashboard</a></li>
                                <li class="breadcrumb-item"><a href="index.php">Reports</a></li>
                                <li class="breadcrumb-item active">Pending Payments</li>
                            </ol>
                        </nav>
                    </div>

                    <!-- Filters Card -->
                    <div class="card shadow mb-4 no-print">
                        <div class="card-header py-3 bg-warning text-dark">
                            <h6 class="m-0 font-weight-bold"><i class="fas fa-filter mr-2"></i>Filters & Export</h6>
                        </div>
                        <div class="card-body">
                            <form method="GET" class="row">
                                <div class="col-md-3">
                                    <label for="days" class="form-label">Days Overdue</label>
                                    <select class="form-control" id="days" name="days">
                                        <option value="">All</option>
                                        <option value="7" <?php echo $days_filter == '7' ? 'selected' : ''; ?>>Within 7 days</option>
                                        <option value="30" <?php echo $days_filter == '30' ? 'selected' : ''; ?>>Within 30 days</option>
                                        <option value="60" <?php echo $days_filter == '60' ? 'selected' : ''; ?>>Within 60 days</option>
                                        <option value="90" <?php echo $days_filter == '90' ? 'selected' : ''; ?>>Over 90 days</option>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <label for="customer_id" class="form-label">Customer</label>
                                    <select class="form-control" id="customer_id" name="customer_id">
                                        <option value="">All Customers</option>
                                        <?php while($customer = mysqli_fetch_assoc($customers)) { ?>
                                        <option value="<?php echo $customer['id']; ?>" <?php echo $customer_id == $customer['id'] ? 'selected' : ''; ?>><?php echo $customer['customer_name']; ?></option>
                                        <?php } ?>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <label for="amount_filter" class="form-label">Amount Range</label>
                                    <select class="form-control" id="amount_filter" name="amount_filter">
                                        <option value="">All Amounts</option>
                                        <option value="500" <?php echo $amount_filter == '500' ? 'selected' : ''; ?>>Up to ₹500</option>
                                        <option value="1000" <?php echo $amount_filter == '1000' ? 'selected' : ''; ?>>₹501 - ₹1,000</option>
                                        <option value="5000" <?php echo $amount_filter == '5000' ? 'selected' : ''; ?>>₹1,001 - ₹5,000</option>
                                        <option value="5000+" <?php echo $amount_filter == '5000+' ? 'selected' : ''; ?>>Over ₹5,000</option>
                                    </select>
                                </div>
                                <div class="col-md-3 d-flex align-items-end">
                                    <button type="submit" class="btn btn-primary mr-2"><i class="fas fa-search mr-1"></i>Filter</button>
                                    <a href="?" class="btn btn-secondary mr-2"><i class="fas fa-refresh mr-1"></i>Reset</a>
                                    <div class="dropdown">
                                        <button class="btn btn-success dropdown-toggle" type="button" data-toggle="dropdown">
                                            <i class="fas fa-download mr-1"></i>Export
                                        </button>
                                        <div class="dropdown-menu">
                                            <a class="dropdown-item" href="?<?php echo http_build_query(array_merge($_GET, ['export' => 'csv'])); ?>">
                                                <i class="fas fa-file-csv mr-1"></i>Export to CSV
                                            </a>
                                            <a class="dropdown-item" href="javascript:window.print()">
                                                <i class="fas fa-print mr-1"></i>Print Report
                                            </a>
                                            <a class="dropdown-item" href="print_thermal_report.php?<?php echo http_build_query(array_merge($_GET, ['report_type' => 'pending_payments', 'title' => 'Pending Payments Report'])); ?>" target="_blank">
                                                <i class="fas fa-receipt mr-1"></i>3" Thermal Print
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>

                    <!-- Summary Cards -->
                    <div class="row mb-4">
                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card border-left-danger shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">Total Outstanding</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($summary['total_outstanding'] ?? 0, 2); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-exclamation-triangle fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card border-left-warning shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">Pending Invoices</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $summary['total_pending_invoices'] ?? 0; ?></div>
                                            <div class="text-xs text-muted"><?php echo $summary['unique_customers'] ?? 0; ?> customers</div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-file-invoice fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card border-left-info shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">Avg Days Overdue</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo round($summary['avg_days_overdue'] ?? 0, 1); ?></div>
                                            <div class="text-xs text-muted">days</div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-calendar-times fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card border-left-danger shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">Over 60 Days</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($summary['overdue_60_amount'] ?? 0, 2); ?></div>
                                            <div class="text-xs text-muted">Critical</div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-skull-crossbones fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Alert for Critical Overdue -->
                    <?php if(($summary['overdue_60_amount'] ?? 0) > 0) { ?>
                    <div class="alert alert-danger alert-dismissible fade show no-print" role="alert">
                        <i class="fas fa-exclamation-triangle mr-2"></i>
                        <strong>Critical Alert!</strong> You have ₹<?php echo number_format($summary['overdue_60_amount'], 2); ?> in payments overdue by more than 60 days. 
                        Immediate action required for debt recovery.
                        <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                            <span aria-hidden="true">&times;</span>
                        </button>
                    </div>
                    <?php } ?>

                    <!-- Pending Payments Table -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3 bg-gradient-warning text-dark">
                            <h6 class="m-0 font-weight-bold"><i class="fas fa-table mr-2"></i>Pending Payments Details</h6>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered table-hover" id="dataTable">
                                    <thead class="thead-light">
                                        <tr>
                                            <th>Invoice No</th>
                                            <th>Date</th>
                                            <th>Customer</th>
                                            <th>Phone</th>
                                            <th>Total Amount</th>
                                            <th>Received</th>
                                            <th>Balance</th>
                                            <th>Status</th>
                                            <th>Days</th>
                                            <th class="no-print">Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php 
                                        $grand_total = 0;
                                        $grand_received = 0;
                                        $grand_balance = 0;
                                        
                                        while($row = mysqli_fetch_assoc($result)) { 
                                            $total_received = $row['cash_received'] + $row['upi_received'];
                                            $grand_total += $row['total_amount'];
                                            $grand_received += $total_received;
                                            $grand_balance += $row['balance_amount'];
                                            
                                            // Determine row class based on days overdue
                                            $row_class = '';
                                            if($row['days_overdue'] > 60) {
                                                $row_class = 'overdue-urgent';
                                            } elseif($row['days_overdue'] > 30) {
                                                $row_class = 'overdue-warning';
                                            } elseif($row['days_overdue'] > 7) {
                                                $row_class = 'overdue-mild';
                                            }
                                        ?>
                                        <tr class="<?php echo $row_class; ?>">
                                            <td><span class="badge badge-warning"><?php echo $row['invoice_no']; ?></span></td>
                                            <td><?php echo date('d-m-Y', strtotime($row['invoice_date'])); ?></td>
                                            <td>
                                                <div>
                                                    <strong><?php echo htmlspecialchars($row['customer_name']); ?></strong>
                                                    <?php if($row['email']) { ?>
                                                    <div class="text-muted small"><?php echo htmlspecialchars($row['email']); ?></div>
                                                    <?php } ?>
                                                </div>
                                            </td>
                                            <td><?php echo htmlspecialchars($row['phone']) ?: '-'; ?></td>
                                            <td>₹<?php echo number_format($row['total_amount'], 2); ?></td>
                                            <td>
                                                <div>₹<?php echo number_format($total_received, 2); ?></div>
                                                <div class="text-muted small">
                                                    Cash: ₹<?php echo number_format($row['cash_received'], 2); ?> | 
                                                    UPI: ₹<?php echo number_format($row['upi_received'], 2); ?>
                                                </div>
                                            </td>
                                            <td>
                                                <strong class="text-danger">₹<?php echo number_format($row['balance_amount'], 2); ?></strong>
                                            </td>
                                            <td>
                                                <span class="badge badge-<?php echo $row['payment_status'] == 'partial' ? 'warning' : 'danger'; ?>">
                                                    <?php echo ucfirst($row['payment_status']); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <span class="badge badge-<?php echo $row['days_overdue'] > 60 ? 'danger' : ($row['days_overdue'] > 30 ? 'warning' : 'info'); ?>">
                                                    <?php echo $row['days_overdue']; ?> days
                                                </span>
                                            </td>
                                            <td class="no-print">
                                                <div class="btn-group" role="group">
                                                    <a href="../sales/print_invoice.php?id=<?php echo $row['id']; ?>" class="btn btn-sm btn-outline-primary" target="_blank" title="Print Invoice">
                                                        <i class="fas fa-print"></i>
                                                    </a>
                                                    <?php if($row['customer_id']) { ?>
                                                    <a href="tel:<?php echo $row['phone']; ?>" class="btn btn-sm btn-outline-success" title="Call Customer">
                                                        <i class="fas fa-phone"></i>
                                                    </a>
                                                    <?php } ?>
                                                    <a href="../sales/invoices.php?payment_status=pending&customer=<?php echo $row['customer_id']; ?>" class="btn btn-sm btn-outline-info" title="View All Bills">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php } ?>
                                    </tbody>
                                    <tfoot class="thead-dark">
                                        <tr>
                                            <th colspan="4">TOTAL</th>
                                            <th>₹<?php echo number_format($grand_total, 2); ?></th>
                                            <th>₹<?php echo number_format($grand_received, 2); ?></th>
                                            <th class="text-danger">₹<?php echo number_format($grand_balance, 2); ?></th>
                                            <th colspan="3" class="no-print">-</th>
                                        </tr>
                                    </tfoot>
                                </table>
                            </div>
                        </div>
                    </div>

                    <!-- Collection Tips -->
                    <div class="card shadow mb-4 no-print">
                        <div class="card-header py-3 bg-gradient-info text-white">
                            <h6 class="m-0 font-weight-bold"><i class="fas fa-lightbulb mr-2"></i>Collection Tips</h6>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-6">
                                    <h6 class="text-success"><i class="fas fa-check-circle mr-1"></i>Best Practices:</h6>
                                    <ul class="list-unstyled">
                                        <li><i class="fas fa-chevron-right text-success mr-2"></i>Contact customers within 7 days of overdue</li>
                                        <li><i class="fas fa-chevron-right text-success mr-2"></i>Send payment reminders via phone/SMS</li>
                                        <li><i class="fas fa-chevron-right text-success mr-2"></i>Offer payment plans for large amounts</li>
                                        <li><i class="fas fa-chevron-right text-success mr-2"></i>Follow up weekly for amounts over ₹1000</li>
                                    </ul>
                                </div>
                                <div class="col-md-6">
                                    <h6 class="text-danger"><i class="fas fa-exclamation-triangle mr-1"></i>Red Flags:</h6>
                                    <ul class="list-unstyled">
                                        <li><i class="fas fa-times text-danger mr-2"></i>Payments overdue by 60+ days need immediate action</li>
                                        <li><i class="fas fa-times text-danger mr-2"></i>Customers with multiple pending invoices</li>
                                        <li><i class="fas fa-times text-danger mr-2"></i>Large amounts (₹5000+) overdue by 30+ days</li>
                                        <li><i class="fas fa-times text-danger mr-2"></i>No response to payment reminders</li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                    </div>

                </div>
                <!-- /.container-fluid -->
            </div>
            <!-- End of Main Content -->
            <?php include_once('../includes/footer.php');?>
        </div>
        <!-- End of Content Wrapper -->
    </div>
    <!-- End of Page Wrapper -->

    <!-- Bootstrap core JavaScript-->
    <script src="../vendor/jquery/jquery.min.js"></script>
    <script src="../vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <!-- Core plugin JavaScript-->
    <script src="../vendor/jquery-easing/jquery.easing.min.js"></script>
    <!-- Custom scripts for all pages-->
    <script src="../js/sb-admin-2.min.js"></script>
    <!-- DataTables JavaScript-->
    <script src="../vendor/datatables/jquery.dataTables.min.js"></script>
    <script src="../vendor/datatables/dataTables.bootstrap4.min.js"></script>
    
    <script>
        $(document).ready(function() {
            $('#dataTable').DataTable({
                "pageLength": 25,
                "order": [[ 8, "desc" ]], // Sort by days overdue
                "columnDefs": [
                    { "orderable": false, "targets": -1 }
                ]
            });
        });
    </script>
</body>
</html>
