<?php
session_start();
include_once('../includes/config.php');

if (!isset($_SESSION['uid'])) {
    header('location:../logout.php');
    exit();
}

// Check database connection
if (!$con) {
    die("Database connection failed. Please contact administrator.");
}

// Get filter parameters
$from_date = isset($_GET['from_date']) ? $_GET['from_date'] : date('Y-m-01');
$to_date = isset($_GET['to_date']) ? $_GET['to_date'] : date('Y-m-d');
$payment_method = isset($_GET['payment_method']) ? $_GET['payment_method'] : 'all';
$transaction_type = isset($_GET['transaction_type']) ? $_GET['transaction_type'] : 'all';

// Function to get cash inflows
function getCashInflows($conn, $from_date, $to_date, $payment_method) {
    $inflows = [];
    
    // 1. Sales revenue (cash and UPI from payment_history)
    $sales_query = "SELECT 
                        ph.payment_date as transaction_date,
                        'Sales Revenue' as description,
                        'sales' as type,
                        si.invoice_no as reference,
                        ph.payment_method,
                        (ph.cash_amount + ph.upi_amount) as amount
                    FROM payment_history ph
                    JOIN sales_invoices si ON ph.invoice_id = si.id
                    WHERE ph.payment_date BETWEEN ? AND ?";
    
    if ($payment_method != 'all') {
        if ($payment_method == 'cash') {
            $sales_query .= " AND ph.cash_amount > 0";
        } elseif ($payment_method == 'upi') {
            $sales_query .= " AND ph.upi_amount > 0";
        }
    }
    
    $sales_query .= " ORDER BY ph.payment_date DESC";
    
    $stmt = $conn->prepare($sales_query);
    if (!$stmt) {
        error_log("Prepare failed: " . $conn->error);
        return $inflows;
    }
    $stmt->bind_param("ss", $from_date, $to_date);
    if (!$stmt->execute()) {
        error_log("Execute failed: " . $stmt->error);
        return $inflows;
    }
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        $inflows[] = $row;
    }
    
    // 2. Customer payments from payment_receipts
    $payment_query = "SELECT 
                        receipt_date as transaction_date,
                        CONCAT('Customer Payment - ', customer_name) as description,
                        'customer_payment' as type,
                        receipt_no as reference,
                        payment_mode as payment_method,
                        total_amount as amount
                    FROM payment_receipts
                    WHERE receipt_date BETWEEN ? AND ?
                    AND status = 'active'";
    
    if ($payment_method != 'all') {
        if ($payment_method == 'cash') {
            $payment_query .= " AND cash_amount > 0";
        } elseif ($payment_method == 'upi') {
            $payment_query .= " AND upi_amount > 0";
        }
    }
    
    $payment_query .= " ORDER BY receipt_date DESC";
    
    $stmt = $conn->prepare($payment_query);
    $stmt->bind_param("ss", $from_date, $to_date);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        $inflows[] = $row;
    }
    
    // 3. Income from income_expense table
    $income_query = "SELECT 
                        transaction_date,
                        CONCAT('Income - ', description) as description,
                        'other_income' as type,
                        reference_no as reference,
                        payment_method,
                        amount
                    FROM income_expense
                    WHERE type = 'income'
                    AND transaction_date BETWEEN ? AND ?";
    
    if ($payment_method != 'all') {
        $income_query .= " AND payment_method = ?";
        $stmt = $conn->prepare($income_query);
        $stmt->bind_param("sss", $from_date, $to_date, $payment_method);
    } else {
        $stmt = $conn->prepare($income_query);
        $stmt->bind_param("ss", $from_date, $to_date);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        $inflows[] = $row;
    }
    
    return $inflows;
}

// Function to get cash outflows
function getCashOutflows($conn, $from_date, $to_date, $payment_method) {
    $outflows = [];
    
    // 1. Supplier payments
    $supplier_query = "SELECT 
                        payment_date as transaction_date,
                        CONCAT('Supplier Payment - ', supplier_name) as description,
                        'supplier_payment' as type,
                        payment_no as reference,
                        payment_mode as payment_method,
                        amount
                    FROM supplier_payments
                    WHERE payment_date BETWEEN ? AND ?";
    
    if ($payment_method != 'all') {
        $supplier_query .= " AND payment_mode = ?";
        $stmt = $conn->prepare($supplier_query);
        $stmt->bind_param("sss", $from_date, $to_date, $payment_method);
    } else {
        $stmt = $conn->prepare($supplier_query);
        $stmt->bind_param("ss", $from_date, $to_date);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        $outflows[] = $row;
    }
    
    // 2. Expenses from income_expense table
    $expense_query = "SELECT 
                        transaction_date,
                        CONCAT('Expense - ', description) as description,
                        'expense' as type,
                        reference_no as reference,
                        payment_method,
                        amount
                    FROM income_expense
                    WHERE type = 'expense'
                    AND transaction_date BETWEEN ? AND ?";
    
    if ($payment_method != 'all') {
        $expense_query .= " AND payment_method = ?";
        $stmt = $conn->prepare($expense_query);
        $stmt->bind_param("sss", $from_date, $to_date, $payment_method);
    } else {
        $stmt = $conn->prepare($expense_query);
        $stmt->bind_param("ss", $from_date, $to_date);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        $outflows[] = $row;
    }
    
    return $outflows;
}

// Get data
$inflows = getCashInflows($con, $from_date, $to_date, $payment_method);
$outflows = getCashOutflows($con, $from_date, $to_date, $payment_method);

// Calculate totals
$total_inflow = array_sum(array_column($inflows, 'amount'));
$total_outflow = array_sum(array_column($outflows, 'amount'));
$net_flow = $total_inflow - $total_outflow;

// Combine and sort all transactions
$all_transactions = [];

// Filter inflows if needed
if ($transaction_type == 'all' || $transaction_type == 'inflow') {
    foreach ($inflows as $inflow) {
        $inflow['flow_type'] = 'inflow';
        $all_transactions[] = $inflow;
    }
}

// Filter outflows if needed
if ($transaction_type == 'all' || $transaction_type == 'outflow') {
    foreach ($outflows as $outflow) {
        $outflow['flow_type'] = 'outflow';
        $all_transactions[] = $outflow;
    }
}

// Sort by date (newest first)
usort($all_transactions, function($a, $b) {
    return strtotime($b['transaction_date']) - strtotime($a['transaction_date']);
});
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <meta name="description" content="">
    <meta name="author" content="">
    <title>Cash Flow Report - Vegetable Shop Management System</title>
    
    <!-- Custom fonts for this template-->
    <link href="../vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="https://fonts.googleapis.com/css?family=Nunito:200,200i,300,300i,400,400i,600,600i,700,700i,800,800i,900,900i" rel="stylesheet">
    
    <!-- Custom styles for this template-->
    <link href="../css/sb-admin-2.min.css" rel="stylesheet">
    
    <!-- Custom styles for this page -->
    <link href="../vendor/datatables/dataTables.bootstrap4.min.css" rel="stylesheet">
    
    <style>
        .summary-card {
            border-radius: 10px;
            box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
            border: 1px solid #e3e6f0;
        }
        
        .inflow-card {
            border-left: 4px solid #1cc88a;
        }
        
        .outflow-card {
            border-left: 4px solid #e74a3b;
        }
        
        .netflow-card {
            border-left: 4px solid #36b9cc;
        }
        
        .negative {
            color: #e74a3b;
        }
        
        .positive {
            color: #1cc88a;
        }
        
        .inflow-row {
            background-color: #f8fff9;
        }
        
        .outflow-row {
            background-color: #fff8f8;
        }
        
        .filter-section {
            background: #f8f9fc;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            border: 1px solid #e3e6f0;
        }
    </style>
</head>

<body id="page-top">
    <!-- Page Wrapper -->
    <div id="wrapper">
        <!-- Sidebar -->
        <?php include_once('../includes/sidebar.php'); ?>
        <!-- End of Sidebar -->

        <!-- Content Wrapper -->
        <div id="content-wrapper" class="d-flex flex-column">
            <!-- Main Content -->
            <div id="content">
                <!-- Topbar -->
                <?php include_once('../includes/topbar.php'); ?>
                <!-- End of Topbar -->

                <!-- Begin Page Content -->
                <div class="container-fluid">
                    <!-- Page Heading -->
                    <div class="d-sm-flex align-items-center justify-content-between mb-4">
                        <h1 class="h3 mb-0 text-gray-800">
                            <i class="fas fa-exchange-alt mr-2"></i>Cash Flow Report
                        </h1>
                        <div class="d-sm-flex">
                            <button class="btn btn-primary btn-sm mr-2" onclick="window.print()">
                                <i class="fas fa-print mr-1"></i>Print Report
                            </button>
                            <button class="btn btn-success btn-sm" onclick="exportToExcel()">
                                <i class="fas fa-file-excel mr-1"></i>Export Excel
                            </button>
                        </div>
                    </div>

                    <!-- Filter Section -->
                    <div class="filter-section">
                        <form method="GET" action="">
                            <div class="row">
                                <div class="col-md-3">
                                    <label for="from_date">From Date:</label>
                                    <input type="date" class="form-control" id="from_date" name="from_date" 
                                           value="<?php echo $from_date; ?>" required>
                                </div>
                                <div class="col-md-3">
                                    <label for="to_date">To Date:</label>
                                    <input type="date" class="form-control" id="to_date" name="to_date" 
                                           value="<?php echo $to_date; ?>" required>
                                </div>
                                <div class="col-md-2">
                                    <label for="payment_method">Payment Method:</label>
                                    <select class="form-control" id="payment_method" name="payment_method">
                                        <option value="all" <?php echo ($payment_method == 'all') ? 'selected' : ''; ?>>All Methods</option>
                                        <option value="cash" <?php echo ($payment_method == 'cash') ? 'selected' : ''; ?>>Cash</option>
                                        <option value="upi" <?php echo ($payment_method == 'upi') ? 'selected' : ''; ?>>UPI</option>
                                        <option value="bank" <?php echo ($payment_method == 'bank') ? 'selected' : ''; ?>>Bank</option>
                                        <option value="cheque" <?php echo ($payment_method == 'cheque') ? 'selected' : ''; ?>>Cheque</option>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <label for="transaction_type">Transaction Type:</label>
                                    <select class="form-control" id="transaction_type" name="transaction_type">
                                        <option value="all" <?php echo ($transaction_type == 'all') ? 'selected' : ''; ?>>All</option>
                                        <option value="inflow" <?php echo ($transaction_type == 'inflow') ? 'selected' : ''; ?>>Inflow Only</option>
                                        <option value="outflow" <?php echo ($transaction_type == 'outflow') ? 'selected' : ''; ?>>Outflow Only</option>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <label>&nbsp;</label>
                                    <button type="submit" class="btn btn-primary btn-block">
                                        <i class="fas fa-search mr-1"></i>Filter
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>

                    <!-- Summary Cards -->
                    <div class="row mb-4">
                        <div class="col-xl-4 col-md-6 mb-4">
                            <div class="card summary-card inflow-card h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                                Total Cash Inflow
                                            </div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                                ₹<?php echo number_format($total_inflow, 2); ?>
                                            </div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-arrow-down fa-2x text-success"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-4 col-md-6 mb-4">
                            <div class="card summary-card outflow-card h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">
                                                Total Cash Outflow
                                            </div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                                ₹<?php echo number_format($total_outflow, 2); ?>
                                            </div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-arrow-up fa-2x text-danger"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-4 col-md-6 mb-4">
                            <div class="card summary-card netflow-card h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">
                                                Net Cash Flow
                                            </div>
                                            <div class="h5 mb-0 font-weight-bold <?php echo ($net_flow >= 0) ? 'positive' : 'negative'; ?>">
                                                ₹<?php echo number_format($net_flow, 2); ?>
                                            </div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-balance-scale fa-2x text-info"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Detailed Transactions -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary">
                                <i class="fas fa-list mr-2"></i>Detailed Cash Flow Transactions
                                <span class="badge badge-info ml-2"><?php echo count($all_transactions); ?> records</span>
                            </h6>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered" id="cashFlowTable" width="100%" cellspacing="0">
                                    <thead class="thead-light">
                                        <tr>
                                            <th>Date</th>
                                            <th>Type</th>
                                            <th>Description</th>
                                            <th>Reference</th>
                                            <th>Payment Method</th>
                                            <th class="text-right">Amount</th>
                                            <th>Flow Type</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if (count($all_transactions) > 0): ?>
                                            <?php foreach ($all_transactions as $transaction): ?>
                                                <tr class="<?php echo ($transaction['flow_type'] == 'inflow') ? 'inflow-row' : 'outflow-row'; ?>">
                                                    <td><?php echo date('d-m-Y', strtotime($transaction['transaction_date'])); ?></td>
                                                    <td>
                                                        <span class="badge badge-<?php echo ($transaction['flow_type'] == 'inflow') ? 'success' : 'danger'; ?>">
                                                            <?php echo ucfirst(str_replace('_', ' ', $transaction['type'])); ?>
                                                        </span>
                                                    </td>
                                                    <td><?php echo htmlspecialchars($transaction['description']); ?></td>
                                                    <td><?php echo htmlspecialchars($transaction['reference'] ?? '-'); ?></td>
                                                    <td>
                                                        <span class="badge badge-info">
                                                            <?php echo strtoupper($transaction['payment_method']); ?>
                                                        </span>
                                                    </td>
                                                    <td class="text-right font-weight-bold">
                                                        <span class="<?php echo ($transaction['flow_type'] == 'inflow') ? 'positive' : 'negative'; ?>">
                                                            <?php echo ($transaction['flow_type'] == 'inflow') ? '+' : '-'; ?>₹<?php echo number_format($transaction['amount'], 2); ?>
                                                        </span>
                                                    </td>
                                                    <td>
                                                        <i class="fas fa-arrow-<?php echo ($transaction['flow_type'] == 'inflow') ? 'down text-success' : 'up text-danger'; ?> mr-1"></i>
                                                        <?php echo ucfirst($transaction['flow_type']); ?>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        <?php else: ?>
                                            <tr>
                                                <td colspan="7" class="text-center text-muted">
                                                    <i class="fas fa-info-circle mr-2"></i>No transactions found for the selected period.
                                                </td>
                                            </tr>
                                        <?php endif; ?>
                                    </tbody>
                                    <?php if (count($all_transactions) > 0): ?>
                                        <tfoot class="thead-dark">
                                            <tr>
                                                <th colspan="5" class="text-right">Total:</th>
                                                <th class="text-right">
                                                    <span class="<?php echo ($net_flow >= 0) ? 'positive' : 'negative'; ?>">
                                                        ₹<?php echo number_format($net_flow, 2); ?>
                                                    </span>
                                                </th>
                                                <th>Net Flow</th>
                                            </tr>
                                        </tfoot>
                                    <?php endif; ?>
                                </table>
                            </div>
                        </div>
                    </div>

                </div>
                <!-- /.container-fluid -->
            </div>
            <!-- End of Main Content -->

            <!-- Footer -->
            <?php include_once('../includes/footer.php'); ?>
            <!-- End of Footer -->
        </div>
        <!-- End of Content Wrapper -->
    </div>
    <!-- End of Page Wrapper -->

    <!-- Scroll to Top Button-->
    <a class="scroll-to-top rounded" href="#page-top">
        <i class="fas fa-angle-up"></i>
    </a>

    <!-- Bootstrap core JavaScript-->
    <script src="../vendor/jquery/jquery.min.js"></script>
    <script src="../vendor/bootstrap/js/bootstrap.bundle.min.js"></script>

    <!-- Core plugin JavaScript-->
    <script src="../vendor/jquery-easing/jquery.easing.min.js"></script>

    <!-- Custom scripts for all pages-->
    <script src="../js/sb-admin-2.min.js"></script>

    <!-- Page level plugins -->
    <script src="../vendor/datatables/jquery.dataTables.min.js"></script>
    <script src="../vendor/datatables/dataTables.bootstrap4.min.js"></script>

    <script>
        $(document).ready(function() {
            $('#cashFlowTable').DataTable({
                "pageLength": 25,
                "order": [[ 0, "desc" ]],
                "columnDefs": [
                    { "orderable": false, "targets": [6] }
                ],
                "language": {
                    "search": "Search transactions:",
                    "lengthMenu": "Show _MENU_ transactions per page",
                    "info": "Showing _START_ to _END_ of _TOTAL_ transactions",
                    "infoEmpty": "No transactions available",
                    "infoFiltered": "(filtered from _MAX_ total transactions)",
                    "zeroRecords": "No matching transactions found"
                }
            });
        });

        function exportToExcel() {
            // Create a temporary form
            var form = document.createElement('form');
            form.method = 'POST';
            form.action = 'export_cash_flow.php';
            
            // Add current filter parameters
            var params = ['from_date', 'to_date', 'payment_method', 'transaction_type'];
            params.forEach(function(param) {
                var input = document.createElement('input');
                input.type = 'hidden';
                input.name = param;
                input.value = document.getElementById(param).value;
                form.appendChild(input);
            });
            
            document.body.appendChild(form);
            form.submit();
            document.body.removeChild(form);
        }

        // Auto-submit form when dates change
        document.getElementById('from_date').addEventListener('change', function() {
            if (document.getElementById('to_date').value) {
                document.querySelector('form').submit();
            }
        });

        document.getElementById('to_date').addEventListener('change', function() {
            if (document.getElementById('from_date').value) {
                document.querySelector('form').submit();
            }
        });
    </script>
</body>
</html>
