/**
 * Enhanced Admin Panel JavaScript
 * Modern interactive features and animations
 */

(function($) {
    "use strict";

    // Global error handling
    window.addEventListener('error', function(e) {
        console.error('🚨 JavaScript Error:', e.error);
        console.error('📍 Location:', e.filename + ':' + e.lineno);
        return false; // Don't prevent default error handling
    });

    // Global variables
    let currentTheme = localStorage.getItem('admin-theme') || 'light';
    let notifications = [];

    console.log('🎨 Theme system initializing. Current theme from storage:', currentTheme);

    // Immediate mobile sidebar initialization (before document ready)
    if (typeof window !== 'undefined' && typeof $ !== 'undefined') {
        // Immediate check for mobile sidebar
        $(function() {
            console.log('🚀 Immediate mobile sidebar check...');
            console.log('📏 Window width:', $(window).width());
            
            // Force initialization if on mobile
            if ($(window).width() < 768) {
                console.log('📱 Mobile device detected, forcing sidebar initialization');
                
                // Use mobile classes for hiding sidebar initially
                $('.sidebar').addClass('mobile-hidden').removeClass('mobile-visible');
                $('body').removeClass('mobile-sidebar-open');
                
                // Ensure the toggle button is visible and working
                const toggleBtn = $("#sidebarToggleTop");
                if (toggleBtn.length > 0) {
                    toggleBtn.show().css({
                        'display': 'block !important',
                        'visibility': 'visible !important'
                    });
                    console.log('✅ Toggle button found and made visible');
                } else {
                    console.error('❌ Toggle button #sidebarToggleTop not found!');
                    // Try to find alternative toggle buttons
                    const altButtons = $('.btn-link, [data-toggle="sidebar"]');
                    console.log('🔍 Found alternative buttons:', altButtons.length);
                }
            } else {
                // Desktop mode - ensure sidebar is visible
                $('.sidebar').removeClass('mobile-hidden mobile-visible');
                $('body').removeClass('mobile-sidebar-open');
                console.log('🖥️ Desktop mode - sidebar should be visible');
            }
        });
    }

    // Initialize when document is ready
    $(document).ready(function() {
        initializeTheme();
        initializeEnhancements();
        initializeAnimations();
        initializeTooltips();
        initializeCounters();
        initializeCharts();
        initializeNotifications();
        initializeLoadingStates();
        addThemeToggle();
        addFloatingActionButton();
        initializeAdvancedFeatures();
        initializeSidebarToggle();
        initializeMobileMenu();
    });

    // Initialize theme
    function initializeTheme() {
        console.log('🎨 Initializing theme. Current theme:', currentTheme);
        
        if (currentTheme === 'dark') {
            document.documentElement.setAttribute('data-theme', 'dark');
            console.log('🌙 Dark theme initialized');
        } else {
            console.log('☀️ Light theme initialized');
        }
        updateThemeIcon();
    }

    // Add theme toggle button
    function addThemeToggle() {
        // Remove existing theme toggle if present
        $('.theme-toggle').remove();
        
        const themeToggle = `
            <div class="theme-toggle" id="themeToggle" title="Toggle Dark Mode">
                <i class="fas fa-moon theme-icon"></i>
            </div>
        `;
        $('body').append(themeToggle);

        // Handle fixed theme toggle button
        $('#themeToggle').on('click', function() {
            console.log('🎨 Fixed theme toggle clicked');
            toggleTheme();
        });
        
        // Handle topbar theme toggle button
        $(document).on('click', '#topbarThemeToggle', function(e) {
            e.preventDefault();
            console.log('🎨 Topbar theme toggle clicked');
            toggleTheme();
        });
        
        console.log('✅ Theme toggle buttons added and initialized');
    }

    // Toggle theme function
    function toggleTheme() {
        console.log('🎨 Toggling theme from:', currentTheme);
        currentTheme = currentTheme === 'light' ? 'dark' : 'light';
        localStorage.setItem('admin-theme', currentTheme);
        
        console.log('🎨 New theme:', currentTheme);
        
        if (currentTheme === 'dark') {
            document.documentElement.setAttribute('data-theme', 'dark');
            console.log('🌙 Dark theme applied');
        } else {
            document.documentElement.removeAttribute('data-theme');
            console.log('☀️ Light theme applied');
        }
        
        updateThemeIcon();
        
        // Check if showNotification function exists
        if (typeof showNotification === 'function') {
            showNotification('Theme changed to ' + currentTheme + ' mode', 'success');
        } else {
            console.log('✅ Theme changed to ' + currentTheme + ' mode');
        }
    }

    // Update theme icon
    function updateThemeIcon() {
        const icon = $('.theme-icon');
        const topbarIcon = $('#topbarThemeIcon');
        console.log('🎨 Updating theme icon. Current theme:', currentTheme, 'Fixed icon found:', icon.length, 'Topbar icon found:', topbarIcon.length);
        
        if (currentTheme === 'dark') {
            icon.removeClass('fa-moon').addClass('fa-sun');
            topbarIcon.removeClass('fa-moon').addClass('fa-sun');
            console.log('🌙 Icons changed to sun');
        } else {
            icon.removeClass('fa-sun').addClass('fa-moon');
            topbarIcon.removeClass('fa-sun').addClass('fa-moon');
            console.log('☀️ Icons changed to moon');
        }
    }

    // Add floating action button
    function addFloatingActionButton() {
        const fab = `
            <button class="fab" id="quickActions" title="Quick Actions">
                <i class="fas fa-plus"></i>
            </button>
        `;
        $('body').append(fab);

        $('#quickActions').on('click', function() {
            showQuickActionsMenu();
        });
    }

    // Show quick actions menu
    function showQuickActionsMenu() {
        const menu = `
            <div class="quick-actions-menu" style="
                position: fixed;
                bottom: 100px;
                right: 30px;
                background: var(--white);
                border-radius: var(--rounded-xl);
                box-shadow: var(--shadow-2xl);
                padding: 1rem;
                z-index: 1001;
                min-width: 200px;
            ">
                <div class="mb-2">
                    <button class="btn btn-primary btn-sm w-100 mb-2">
                        <i class="fas fa-plus mr-2"></i>Add Product
                    </button>
                </div>
                <div class="mb-2">
                    <button class="btn btn-success btn-sm w-100 mb-2">
                        <i class="fas fa-shopping-cart mr-2"></i>New Order
                    </button>
                </div>
                <div class="mb-2">
                    <button class="btn btn-info btn-sm w-100 mb-2">
                        <i class="fas fa-user-plus mr-2"></i>Add Customer
                    </button>
                </div>
                <div>
                    <button class="btn btn-warning btn-sm w-100" onclick="closeQuickActions()">
                        <i class="fas fa-times mr-2"></i>Close
                    </button>
                </div>
            </div>
        `;
        
        // Remove existing menu
        $('.quick-actions-menu').remove();
        
        // Add new menu
        $('body').append(menu);
        
        // Auto-close after 5 seconds
        setTimeout(() => {
            $('.quick-actions-menu').fadeOut(() => {
                $('.quick-actions-menu').remove();
            });
        }, 5000);
    }

    // Close quick actions menu
    window.closeQuickActions = function() {
        $('.quick-actions-menu').fadeOut(() => {
            $('.quick-actions-menu').remove();
        });
    };

    // Initialize modern enhancements
    function initializeEnhancements() {
        
        // Enhanced mobile sidebar functionality
        initializeSidebarToggle();
        initializeMobileMenu();

        // Close sidebar on window resize if mobile
        $(window).resize(function() {
            const isMobile = $(window).width() < 768;
            
            if (isMobile) {
                // On mobile, ensure sidebar starts hidden
                $('.sidebar').addClass('mobile-hidden').removeClass('mobile-visible');
                $('body').removeClass('mobile-sidebar-open');
                console.log('📱 Resized to mobile - hiding sidebar');
            } else {
                // On desktop, remove mobile classes and use default behavior
                $('.sidebar').removeClass('mobile-hidden mobile-visible');
                $('body').removeClass('mobile-sidebar-open');
                console.log('🖥️ Resized to desktop - using default sidebar');
            }
        });

        // Enhanced dropdown animations
        $('.dropdown-toggle').on('click', function() {
            $(this).next('.dropdown-menu').addClass('animated fadeIn');
        });

        // Card hover effects
        $('.card').hover(
            function() { 
                $(this).addClass('hover-lift');
            },
            function() { 
                $(this).removeClass('hover-lift');
            }
        );

        // Button ripple effect
        $('.btn').addClass('btn-ripple');

        // Enhanced form focus
        $('.form-control').addClass('modern-focus');

        // Smooth scrolling for anchor links (exclude Bootstrap collapse toggles)
        $('a[href^="#"]:not([data-toggle="collapse"])').on('click', function(e) {
            const href = $(this).attr('href');
            // Check if href is valid and not just "#"
            if (href && href.length > 1 && href !== '#') {
                const target = $(href);
                if (target.length) {
                    e.preventDefault();
                    $('html, body').animate({
                        scrollTop: target.offset().top - 100
                    }, 500);
                }
            }
        });

        // Auto-hide alerts after 5 seconds
        $('.alert').each(function() {
            const alert = $(this);
            setTimeout(() => {
                alert.fadeOut();
            }, 5000);
        });

        // Enhanced search functionality
        $('.navbar-search input').on('focus', function() {
            $(this).parent().addClass('shadow-lg');
        }).on('blur', function() {
            $(this).parent().removeClass('shadow-lg');
        });
    }

    // Initialize animations
    function initializeAnimations() {
        // Intersection Observer for scroll animations
        const observerOptions = {
            threshold: 0.1,
            rootMargin: '0px 0px -50px 0px'
        };

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const element = entry.target;
                    element.style.opacity = '1';
                    element.style.transform = 'translateY(0)';
                }
            });
        }, observerOptions);

        // Observe all cards and stats
        $('.card, .stats-card').each(function(index) {
            $(this).css({
                'opacity': '0',
                'transform': 'translateY(30px)',
                'transition': 'all 0.6s ease-out ' + (index * 0.1) + 's'
            });
            observer.observe(this);
        });

        // Page load animations
        setTimeout(() => {
            $('.slide-in-left').each(function(index) {
                setTimeout(() => {
                    $(this).addClass('fade-in-scale');
                }, index * 200);
            });
        }, 100);

        // Stats icons animation on hover
        $('.stats-icon').on('mouseenter', function() {
            $(this).addClass('bounce');
        }).on('animationend', function() {
            $(this).removeClass('bounce');
        });
    }

    // Initialize tooltips
    function initializeTooltips() {
        // Initialize Bootstrap tooltips
        $('[data-toggle="tooltip"]').tooltip({
            trigger: 'hover',
            animation: true,
            html: true
        });

        // Add tooltips to common elements
        $('.btn').each(function() {
            if (!$(this).attr('title') && !$(this).attr('data-original-title')) {
                const text = $(this).text().trim() || $(this).find('i').attr('class');
                if (text) {
                    $(this).attr('title', text).tooltip();
                }
            }
        });
    }

    // Initialize counters
    function initializeCounters() {
        const counters = $('.stats-card .h4, .counter');
        
        counters.each(function() {
            const $this = $(this);
            const countTo = parseInt($this.text().replace(/[^0-9]/g, ''));
            
            if (countTo && countTo > 0) {
                $this.text('0');
                
                const observer = new IntersectionObserver((entries) => {
                    entries.forEach(entry => {
                        if (entry.isIntersecting) {
                            animateCounter($this, countTo);
                            observer.unobserve(entry.target);
                        }
                    });
                });
                
                observer.observe(this);
            }
        });
    }

    // Animate counter
    function animateCounter($element, target) {
        const duration = 2000;
        const start = 0;
        const increment = target / (duration / 16);
        let current = start;

        const timer = setInterval(() => {
            current += increment;
            if (current >= target) {
                current = target;
                clearInterval(timer);
            }
            
            const formatted = Math.floor(current).toLocaleString();
            const originalText = $element.text();
            const prefix = originalText.match(/^[^0-9]*/)[0];
            const suffix = originalText.match(/[^0-9]*$/)[0];
            
            $element.text(prefix + formatted + suffix);
        }, 16);
    }

    // Initialize charts (placeholder for future chart implementation)
    function initializeCharts() {
        // Chart.js initialization would go here
        console.log('Charts initialized');
    }

    // Initialize notification system
    function initializeNotifications() {
        // Create notification container
        if (!$('#notificationContainer').length) {
            $('body').append('<div id="notificationContainer" style="position: fixed; top: 20px; right: 20px; z-index: 1060;"></div>');
        }
    }

    // Show notification
    function showNotification(message, type = 'info', duration = 5000) {
        const id = 'notification_' + Date.now();
        const icons = {
            success: 'fa-check-circle',
            error: 'fa-exclamation-circle',
            warning: 'fa-exclamation-triangle',
            info: 'fa-info-circle'
        };

        const notification = `
            <div class="notification ${type}" id="${id}">
                <div class="d-flex align-items-center">
                    <i class="fas ${icons[type]} mr-3" style="font-size: 1.2rem;"></i>
                    <div class="flex-grow-1">
                        <div class="font-weight-bold">${type.charAt(0).toUpperCase() + type.slice(1)}</div>
                        <div class="text-muted">${message}</div>
                    </div>
                    <button class="btn btn-link btn-sm p-0 ml-3" onclick="closeNotification('${id}')">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            </div>
        `;

        $('#notificationContainer').append(notification);
        
        // Show notification
        setTimeout(() => {
            $('#' + id).addClass('show');
        }, 100);

        // Auto-hide notification
        if (duration > 0) {
            setTimeout(() => {
                closeNotification(id);
            }, duration);
        }

        notifications.push(id);
    }

    // Close notification
    window.closeNotification = function(id) {
        $('#' + id).removeClass('show');
        setTimeout(() => {
            $('#' + id).remove();
            notifications = notifications.filter(n => n !== id);
        }, 300);
    };

    // Initialize loading states
    function initializeLoadingStates() {
        // Add loading class to buttons on form submit
        $('form').on('submit', function() {
            $(this).find('.btn[type="submit"]').addClass('loading').prop('disabled', true);
        });

        // Simulate loading for demo
        $('.btn-primary').on('click', function(e) {
            if ($(this).hasClass('demo-loading')) {
                e.preventDefault();
                const btn = $(this);
                btn.addClass('loading').prop('disabled', true);
                
                setTimeout(() => {
                    btn.removeClass('loading').prop('disabled', false);
                    showNotification('Action completed successfully!', 'success');
                }, 2000);
            }
        });
    }

    // Initialize advanced features
    function initializeAdvancedFeatures() {
        // Keyboard shortcuts
        $(document).on('keydown', function(e) {
            // Ctrl/Cmd + D for dark mode toggle
            if ((e.ctrlKey || e.metaKey) && e.which === 68) {
                e.preventDefault();
                toggleTheme();
            }
            
            // Ctrl/Cmd + K for search focus
            if ((e.ctrlKey || e.metaKey) && e.which === 75) {
                e.preventDefault();
                $('.navbar-search input').focus();
            }
        });

        // Auto-save functionality (for forms)
        let autoSaveTimeout;
        $('.auto-save').on('input', function() {
            clearTimeout(autoSaveTimeout);
            autoSaveTimeout = setTimeout(() => {
                // Auto-save logic would go here
                showNotification('Draft saved automatically', 'info', 2000);
            }, 3000);
        });

        // Enhanced tables
        $('.table tbody tr').on('click', function() {
            $('.table tbody tr').removeClass('table-active');
            $(this).addClass('table-active');
        });

        // Progress bars animation
        $('.progress-bar').each(function() {
            const $this = $(this);
            const width = $this.attr('style') || $this.attr('aria-valuenow') + '%';
            $this.css('width', '0%');
            
            setTimeout(() => {
                $this.css('width', width);
            }, 500);
        });

        // Enhanced file upload (if file inputs exist)
        $('input[type="file"]').on('change', function() {
            const files = this.files;
            if (files.length > 0) {
                showNotification(`${files.length} file(s) selected`, 'info', 2000);
            }
        });

        // Dynamic breadcrumbs
        updateBreadcrumbs();
    }

    // Update breadcrumbs based on current page
    function updateBreadcrumbs() {
        const currentPage = window.location.pathname.split('/').pop();
        const breadcrumb = $('.breadcrumb');
        
        if (breadcrumb.length && currentPage) {
            const pageName = currentPage.replace('.php', '').replace('-', ' ');
            const formattedName = pageName.charAt(0).toUpperCase() + pageName.slice(1);
            breadcrumb.find('.breadcrumb-item.active').text(formattedName);
        }
    }

    // --- LEGACY SIDEBAR SYSTEM (DISABLED) ---
    // This professional sidebar system is disabled in favor of the mobile-responsive version above
    // Professional sidebar system has been replaced with mobile-first approach

    // Initialize sidebar toggle functionality
    function initializeSidebarToggle() {
        console.log('🔧 Initializing sidebar toggle functionality...');
        
        // Check if we're on mobile
        const isMobile = $(window).width() < 768;
        console.log('📱 Mobile check:', isMobile, 'Width:', $(window).width());
        
        // Initialize sidebar state for mobile
        if (isMobile) {
            $('.sidebar').addClass('mobile-hidden').removeClass('mobile-visible');
            $('body').removeClass('mobile-sidebar-open');
            console.log('📱 Mobile mode: Sidebar hidden');
        } else {
            $('.sidebar').removeClass('mobile-hidden mobile-visible');
            $('body').removeClass('mobile-sidebar-open');
            console.log('🖥️ Desktop mode: Sidebar visible');
        }

        // Handle sidebar toggle button click
        $(document).off('click', '#sidebarToggleTop').on('click', '#sidebarToggleTop', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 Sidebar toggle clicked');
            
            const sidebar = $('.sidebar');
            const body = $('body');
            
            // Toggle sidebar visibility
            if (sidebar.hasClass('mobile-hidden')) {
                sidebar.removeClass('mobile-hidden').addClass('mobile-visible');
                body.addClass('mobile-sidebar-open');
                console.log('📂 Sidebar opened');
            } else {
                sidebar.addClass('mobile-hidden').removeClass('mobile-visible');
                body.removeClass('mobile-sidebar-open');
                console.log('📁 Sidebar closed');
            }
        });

        // Handle clicking outside sidebar to close it on mobile
        $(document).on('click', function(e) {
            const sidebar = $('.sidebar');
            const toggleBtn = $('#sidebarToggleTop');
            
            // Only on mobile and if sidebar is visible
            if ($(window).width() < 768 && sidebar.hasClass('mobile-visible')) {
                // If click is not on sidebar or toggle button
                if (!sidebar.is(e.target) && sidebar.has(e.target).length === 0 && 
                    !toggleBtn.is(e.target) && toggleBtn.has(e.target).length === 0) {
                    sidebar.addClass('mobile-hidden').removeClass('mobile-visible');
                    $('body').removeClass('mobile-sidebar-open');
                    console.log('📁 Sidebar closed by outside click');
                }
            }
        });

        // Handle window resize
        $(window).on('resize', function() {
            const isMobile = $(window).width() < 768;
            const sidebar = $('.sidebar');
            const body = $('body');
            
            if (!isMobile) {
                // Desktop: remove mobile classes
                sidebar.removeClass('mobile-hidden mobile-visible');
                body.removeClass('mobile-sidebar-open');
            } else {
                // Mobile: ensure sidebar is hidden initially
                if (!sidebar.hasClass('mobile-visible')) {
                    sidebar.addClass('mobile-hidden');
                }
            }
        });

        // Handle Bootstrap collapse functionality for sidebar accordion menus
        $('.sidebar .nav-link[data-toggle="collapse"]').on('click', function(e) {
            e.preventDefault();
            const target = $(this).attr('data-target');
            if (target && target.length > 1 && target !== '#') {
                $(target).collapse('toggle');
                
                // Toggle the collapsed class on the link
                $(this).toggleClass('collapsed');
                
                console.log('📂 Accordion menu toggled:', target);
            } else {
                console.warn('⚠️ Invalid data-target for accordion:', target);
            }
        });

        console.log('✅ Sidebar toggle functionality initialized');
    }

    // Initialize mobile menu functionality
    function initializeMobileMenu() {
        console.log('📱 Initializing mobile menu functionality...');
        
        // Add loading state on mobile menu button click
        $('#mobileMenuDropdown').on('click', function() {
            $(this).addClass('loading');
            setTimeout(() => {
                $(this).removeClass('loading');
            }, 300);
        });
        
        // Close mobile menu when clicking outside
        $(document).on('click', function(e) {
            const mobileMenu = $('.mobile-nav-menu');
            const dropdown = $('.mobile-menu-dropdown');
            
            if ($(window).width() < 768 && dropdown.hasClass('show')) {
                if (!mobileMenu.is(e.target) && mobileMenu.has(e.target).length === 0) {
                    $('#mobileMenuDropdown').dropdown('hide');
                    console.log('📱 Mobile menu closed by outside click');
                }
            }
        });
        
        // Close mobile menu when a link is clicked (except headers)
        $('.mobile-menu-dropdown .dropdown-item').not('.dropdown-header').on('click', function() {
            const $this = $(this);
            
            // Add active state briefly
            $this.addClass('active');
            setTimeout(() => {
                $this.removeClass('active');
            }, 200);
            
            $('#mobileMenuDropdown').dropdown('hide');
            console.log('📱 Mobile menu closed after navigation to:', $this.attr('href'));
        });
        
        // Add smooth scrolling for better UX
        $('.mobile-menu-dropdown .dropdown-item[href]').on('click', function(e) {
            const href = $(this).attr('href');
            if (href && href.startsWith('#')) {
                e.preventDefault();
                const target = $(href);
                if (target.length) {
                    $('html, body').animate({
                        scrollTop: target.offset().top - 70
                    }, 500);
                }
            }
        });
        
        // Highlight current page in mobile menu
        const currentPath = window.location.pathname.split('/').pop();
        $('.mobile-menu-dropdown .dropdown-item').each(function() {
            const href = $(this).attr('href');
            if (href && href.includes(currentPath) && currentPath !== '') {
                $(this).addClass('current-page').prepend('<i class="fas fa-chevron-right mr-1 text-primary"></i>');
            }
        });
        
        console.log('✅ Mobile menu functionality initialized');
    }

    // Export functions for global use
    window.showNotification = showNotification;
    window.toggleTheme = toggleTheme;

    // Performance monitoring
    window.addEventListener('load', function() {
        const loadTime = window.performance.timing.domContentLoadedEventEnd - window.performance.timing.navigationStart;
        console.log('Page load time:', loadTime + 'ms');
        
        if (loadTime > 3000) {
            showNotification('Page loaded in ' + Math.round(loadTime/1000) + 's. Consider optimizing for better performance.', 'warning', 8000);
        }
    });

    // Service worker registration (for future PWA features)
    if ('serviceWorker' in navigator) {
        navigator.serviceWorker.register('./sw.js').catch(err => {
            console.log('Service worker registration failed:', err);
        });
    }

})(jQuery);

// Additional CSS injection for dynamic styles
const additionalStyles = `
    <style>
        /* Quick Actions Menu */
        .quick-actions-menu {
            animation: slideInUp 0.3s ease-out;
        }
        
        /* Enhanced loading states */
        .loading {
            position: relative;
            color: transparent !important;
        }
        
        .loading::after {
            content: '';
            position: absolute;
            width: 20px;
            height: 20px;
            top: 50%;
            left: 50%;
            margin-left: -10px;
            margin-top: -10px;
            border: 2px solid #ffffff;
            border-top: 2px solid transparent;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        /* Table active row */
        .table-active {
            background-color: rgba(99, 102, 241, 0.1) !important;
            transform: scale(1.01);
            box-shadow: 0 2px 8px rgba(99, 102, 241, 0.2);
        }
        
        /* Enhanced focus states */
        .modern-focus:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.2);
        }
        
        /* Notification animations */
        .notification {
            animation: slideInRight 0.3s ease-out;
        }
        
        .notification.show {
            animation: slideInRight 0.3s ease-out;
        }
        
        @keyframes slideInRight {
            from {
                transform: translateX(100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }
        
        /* Dark theme specific styles */
        [data-theme="dark"] .notification {
            background: var(--gray-100);
            color: var(--gray-300);
        }
        
        [data-theme="dark"] .quick-actions-menu {
            background: var(--gray-100) !important;
            color: var(--gray-300);
        }
        
        /* Mobile optimizations */
        @media (max-width: 768px) {
            .fab {
                width: 50px;
                height: 50px;
                bottom: 20px;
                right: 20px;
                font-size: 1.2rem;
            }
            
            .theme-toggle {
                width: 40px;
                height: 40px;
                top: 10px;
                right: 10px;
            }
            
            .notification {
                left: 10px;
                right: 10px;
                max-width: none;
            }
            
            .quick-actions-menu {
                right: 10px !important;
                left: 10px !important;
                bottom: 80px !important;
            }
        }
        
        /* Enhanced ripple effect */
        .btn-ripple {
            position: relative;
            overflow: hidden;
        }
        
        .btn-ripple::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 0;
            height: 0;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.5);
            transform: translate(-50%, -50%);
            transition: width 0.3s, height 0.3s;
        }
        
        .btn-ripple:active::before {
            width: 300px;
            height: 300px;
        }
    </style>
`;

document.head.insertAdjacentHTML('beforeend', additionalStyles);
