<?php
session_start();
if (strlen($_SESSION['uid']) == 0) {
    header('location:../index.php');
    exit;
}

include('../includes/config.php');

$supplierId = isset($_GET['id']) ? $_GET['id'] : 0;

// Get date filters
$fromDate = isset($_GET['from_date']) ? $_GET['from_date'] : date('Y-m-01'); // First day of current month
$toDate = isset($_GET['to_date']) ? $_GET['to_date'] : date('Y-m-d'); // Today

// Get supplier details
$supplierSql = "SELECT * FROM suppliers WHERE id = ?";
$stmt = $con->prepare($supplierSql);
$stmt->bind_param("i", $supplierId);
$stmt->execute();
$supplierResult = $stmt->get_result();
$supplier = $supplierResult->fetch_assoc();

if (!$supplier) {
    header('location: index.php');
    exit;
}

// Calculate opening balance (transactions before from_date)
$openingBalanceSql = "SELECT SUM(debit_amount - credit_amount) as opening_balance 
                     FROM supplier_ledger 
                     WHERE supplier_id = ? AND transaction_date < ?";
$openingStmt = $con->prepare($openingBalanceSql);
$openingStmt->bind_param("is", $supplierId, $fromDate);
$openingStmt->execute();
$openingResult = $openingStmt->get_result();
$openingBalance = $openingResult->fetch_assoc()['opening_balance'] ?: 0;

// Calculate totals for the period
$periodSql = "SELECT 
                SUM(debit_amount) as total_debit,
                SUM(credit_amount) as total_credit,
                SUM(debit_amount - credit_amount) as period_movement
              FROM supplier_ledger 
              WHERE supplier_id = ? AND transaction_date BETWEEN ? AND ?";
$periodStmt = $con->prepare($periodSql);
$periodStmt->bind_param("iss", $supplierId, $fromDate, $toDate);
$periodStmt->execute();
$periodResult = $periodStmt->get_result();
$periodData = $periodResult->fetch_assoc();

$totalDebit = $periodData['total_debit'] ?: 0;
$totalCredit = $periodData['total_credit'] ?: 0;
$periodMovement = $periodData['period_movement'] ?: 0;

// Calculate closing balance
$closingBalance = $openingBalance + $periodMovement;

// Handle delete transaction
if (isset($_GET['delete_transaction'])) {
    $transactionId = $_GET['delete_transaction'];
    
    // Get transaction details before deleting
    $getTransactionSql = "SELECT * FROM supplier_ledger WHERE id = ? AND supplier_id = ?";
    $getStmt = $con->prepare($getTransactionSql);
    $getStmt->bind_param("ii", $transactionId, $supplierId);
    $getStmt->execute();
    $transactionResult = $getStmt->get_result();
    
    if ($transaction = $transactionResult->fetch_assoc()) {
        // Delete the transaction
        $deleteSql = "DELETE FROM supplier_ledger WHERE id = ?";
        $deleteStmt = $con->prepare($deleteSql);
        $deleteStmt->bind_param("i", $transactionId);
        
        if ($deleteStmt->execute()) {
            // Recalculate all balances for this supplier
            $recalculateSql = "SELECT * FROM supplier_ledger WHERE supplier_id = ? ORDER BY transaction_date ASC, id ASC";
            $recalcStmt = $con->prepare($recalculateSql);
            $recalcStmt->bind_param("i", $supplierId);
            $recalcStmt->execute();
            $recalcResult = $recalcStmt->get_result();
            
            $runningBalance = 0;
            while ($row = $recalcResult->fetch_assoc()) {
                $runningBalance = $runningBalance + $row['debit_amount'] - $row['credit_amount'];
                $updateBalanceSql = "UPDATE supplier_ledger SET balance = ? WHERE id = ?";
                $updateStmt = $con->prepare($updateBalanceSql);
                $updateStmt->bind_param("di", $runningBalance, $row['id']);
                $updateStmt->execute();
            }
            
            // Update supplier outstanding balance
            $updateSupplierSql = "UPDATE suppliers SET outstanding_balance = ? WHERE id = ?";
            $updateSupplierStmt = $con->prepare($updateSupplierSql);
            $updateSupplierStmt->bind_param("di", $runningBalance, $supplierId);
            $updateSupplierStmt->execute();
            
            echo "<script>alert('Transaction deleted successfully'); window.location.href='ledger.php?id=$supplierId';</script>";
        } else {
            echo "<script>alert('Error deleting transaction');</script>";
        }
    } else {
        echo "<script>alert('Transaction not found');</script>";
    }
}

// Handle edit transaction
if (isset($_POST['update_transaction'])) {
    $transactionId = $_POST['transaction_id'];
    $transactionDate = $_POST['edit_transaction_date'];
    $transactionType = $_POST['edit_transaction_type'];
    $referenceNo = $_POST['edit_reference_no'];
    $debitAmount = $_POST['edit_debit_amount'] ?: 0;
    $creditAmount = $_POST['edit_credit_amount'] ?: 0;
    $description = $_POST['edit_description'];

    // Update transaction
    $updateSql = "UPDATE supplier_ledger SET transaction_date=?, transaction_type=?, reference_no=?, debit_amount=?, credit_amount=?, description=? WHERE id=? AND supplier_id=?";
    $updateStmt = $con->prepare($updateSql);
    $updateStmt->bind_param("sssddsii", $transactionDate, $transactionType, $referenceNo, $debitAmount, $creditAmount, $description, $transactionId, $supplierId);
    
    if ($updateStmt->execute()) {
        // Recalculate all balances for this supplier
        $recalculateSql = "SELECT * FROM supplier_ledger WHERE supplier_id = ? ORDER BY transaction_date ASC, id ASC";
        $recalcStmt = $con->prepare($recalculateSql);
        $recalcStmt->bind_param("i", $supplierId);
        $recalcStmt->execute();
        $recalcResult = $recalcStmt->get_result();
        
        $runningBalance = 0;
        while ($row = $recalcResult->fetch_assoc()) {
            $runningBalance = $runningBalance + $row['debit_amount'] - $row['credit_amount'];
            $updateBalanceSql = "UPDATE supplier_ledger SET balance = ? WHERE id = ?";
            $updateBalanceStmt = $con->prepare($updateBalanceSql);
            $updateBalanceStmt->bind_param("di", $runningBalance, $row['id']);
            $updateBalanceStmt->execute();
        }
        
        // Update supplier outstanding balance
        $updateSupplierSql = "UPDATE suppliers SET outstanding_balance = ? WHERE id = ?";
        $updateSupplierStmt = $con->prepare($updateSupplierSql);
        $updateSupplierStmt->bind_param("di", $runningBalance, $supplierId);
        $updateSupplierStmt->execute();
        
        echo "<script>alert('Transaction updated successfully'); window.location.href='ledger.php?id=$supplierId';</script>";
    } else {
        echo "<script>alert('Error updating transaction');</script>";
    }
}

// Handle transaction submission
if (isset($_POST['submit'])) {
    $transactionDate = $_POST['transaction_date'];
    $transactionType = $_POST['transaction_type'];
    $referenceNo = $_POST['reference_no'];
    $debitAmount = $_POST['debit_amount'] ?: 0;
    $creditAmount = $_POST['credit_amount'] ?: 0;
    $description = $_POST['description'];
    $createdBy = $_SESSION['uid'];

    // Calculate new balance
    $balanceSql = "SELECT balance FROM supplier_ledger WHERE supplier_id = ? ORDER BY id DESC LIMIT 1";
    $balanceStmt = $con->prepare($balanceSql);
    $balanceStmt->bind_param("i", $supplierId);
    $balanceStmt->execute();
    $balanceResult = $balanceStmt->get_result();
    $currentBalance = 0;
    if ($balanceRow = $balanceResult->fetch_assoc()) {
        $currentBalance = $balanceRow['balance'];
    }

    $newBalance = $currentBalance + $debitAmount - $creditAmount;

    // Insert transaction
    $insertSql = "INSERT INTO supplier_ledger (supplier_id, transaction_date, transaction_type, reference_no, debit_amount, credit_amount, balance, description, created_by) 
                  VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
    $insertStmt = $con->prepare($insertSql);
    $insertStmt->bind_param("isssdddsi", $supplierId, $transactionDate, $transactionType, $referenceNo, $debitAmount, $creditAmount, $newBalance, $description, $createdBy);
    
    if ($insertStmt->execute()) {
        // Update supplier outstanding balance
        $updateSql = "UPDATE suppliers SET outstanding_balance = ? WHERE id = ?";
        $updateStmt = $con->prepare($updateSql);
        $updateStmt->bind_param("di", $newBalance, $supplierId);
        $updateStmt->execute();
        
        echo "<script>alert('Transaction added successfully'); window.location.href='ledger.php?id=$supplierId';</script>";
    } else {
        echo "<script>alert('Error adding transaction');</script>";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <title>Supplier Ledger - <?php echo htmlspecialchars($supplier['supplier_name']); ?></title>
    <link href="../vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="../css/sb-admin-2.min.css" rel="stylesheet">
    <link href="../vendor/datatables/dataTables.bootstrap4.min.css" rel="stylesheet">
</head>

<body id="page-top">
    <div id="wrapper">
        <?php include('../includes/sidebar.php'); ?>
        
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include('../includes/topbar.php'); ?>
                
                <div class="container-fluid">
                    <!-- Page Heading -->
                    <div class="d-sm-flex align-items-center justify-content-between mb-4">
                        <h1 class="h3 mb-0 text-gray-800">Supplier Ledger - <?php echo htmlspecialchars($supplier['supplier_name']); ?></h1>
                        <div>
                            <button class="btn btn-success btn-sm mr-2" onclick="exportToCSV()">
                                <i class="fas fa-file-csv"></i> Export to CSV
                            </button>
                            <a href="index.php" class="btn btn-secondary btn-sm">
                                <i class="fas fa-arrow-left"></i> Back to Suppliers
                            </a>
                        </div>
                    </div>

                    <!-- Supplier Info Card -->
                    <div class="row mb-4">
                        <div class="col-lg-6">
                            <div class="card shadow">
                                <div class="card-header py-3">
                                    <h6 class="m-0 font-weight-bold text-primary">Supplier Information</h6>
                                </div>
                                <div class="card-body">
                                    <p><strong>Code:</strong> <?php echo htmlspecialchars($supplier['supplier_code']); ?></p>
                                    <p><strong>Contact Person:</strong> <?php echo htmlspecialchars($supplier['contact_person']); ?></p>
                                    <p><strong>Phone:</strong> <?php echo htmlspecialchars($supplier['phone']); ?></p>
                                    <p><strong>Address:</strong> <?php echo htmlspecialchars($supplier['address']); ?></p>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-6">
                            <div class="card shadow">
                                <div class="card-header py-3">
                                    <h6 class="m-0 font-weight-bold text-primary">Account Summary</h6>
                                </div>
                                <div class="card-body">
                                    <p><strong>Outstanding Balance:</strong> 
                                        <span class="h5 <?php echo $supplier['outstanding_balance'] > 0 ? 'text-danger' : 'text-success'; ?>">
                                            ₹<?php echo number_format($supplier['outstanding_balance'], 2); ?>
                                        </span>
                                    </p>
                                    <button class="btn btn-primary btn-sm" data-toggle="modal" data-target="#addTransactionModal">
                                        <i class="fas fa-plus"></i> Add Transaction
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Date Filter Card -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary">Date Filter</h6>
                        </div>
                        <div class="card-body">
                            <form method="GET" action="">
                                <input type="hidden" name="id" value="<?php echo $supplierId; ?>">
                                <div class="row">
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label>From Date:</label>
                                            <input type="date" class="form-control" name="from_date" value="<?php echo $fromDate; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label>To Date:</label>
                                            <input type="date" class="form-control" name="to_date" value="<?php echo $toDate; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label>&nbsp;</label>
                                            <div>
                                                <button type="submit" class="btn btn-primary btn-sm">
                                                    <i class="fas fa-filter"></i> Filter
                                                </button>
                                                <a href="ledger.php?id=<?php echo $supplierId; ?>" class="btn btn-secondary btn-sm">
                                                    <i class="fas fa-refresh"></i> Reset
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>

                    <!-- Balance Summary Card -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary">
                                Balance Summary (<?php echo date('d-m-Y', strtotime($fromDate)); ?> to <?php echo date('d-m-Y', strtotime($toDate)); ?>)
                            </h6>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-3">
                                    <div class="card border-left-info">
                                        <div class="card-body">
                                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">Opening Balance</div>
                                            <div class="h5 mb-0 font-weight-bold <?php echo $openingBalance > 0 ? 'text-danger' : 'text-success'; ?>">
                                                ₹<?php echo number_format(abs($openingBalance), 2); ?> <?php echo $openingBalance >= 0 ? 'Dr' : 'Cr'; ?>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="card border-left-danger">
                                        <div class="card-body">
                                            <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">Total Debit</div>
                                            <div class="h5 mb-0 font-weight-bold text-danger">₹<?php echo number_format($totalDebit, 2); ?></div>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="card border-left-success">
                                        <div class="card-body">
                                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">Total Credit</div>
                                            <div class="h5 mb-0 font-weight-bold text-success">₹<?php echo number_format($totalCredit, 2); ?></div>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="card border-left-warning">
                                        <div class="card-body">
                                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">Net Movement</div>
                                            <div class="h5 mb-0 font-weight-bold <?php echo $periodMovement > 0 ? 'text-danger' : 'text-success'; ?>">
                                                ₹<?php echo number_format(abs($periodMovement), 2); ?> <?php echo $periodMovement >= 0 ? 'Dr' : 'Cr'; ?>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="card border-left-primary">
                                        <div class="card-body">
                                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">Closing Balance</div>
                                            <div class="h5 mb-0 font-weight-bold <?php echo $closingBalance > 0 ? 'text-danger' : 'text-success'; ?>">
                                                ₹<?php echo number_format(abs($closingBalance), 2); ?> <?php echo $closingBalance >= 0 ? 'Dr' : 'Cr'; ?>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Ledger Table -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary">Transaction History</h6>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered" id="dataTable" width="100%" cellspacing="0">
                                    <thead>
                                        <tr>
                                            <th>Date</th>
                                            <th>Reference</th>
                                            <th>Type</th>
                                            <th>Description</th>
                                            <th>Debit</th>
                                            <th>Credit</th>
                                            <th>Balance</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php
                                        $ledgerSql = "SELECT * FROM supplier_ledger WHERE supplier_id = ? AND transaction_date BETWEEN ? AND ? ORDER BY transaction_date DESC, id DESC";
                                        $ledgerStmt = $con->prepare($ledgerSql);
                                        $ledgerStmt->bind_param("iss", $supplierId, $fromDate, $toDate);
                                        $ledgerStmt->execute();
                                        $ledgerResult = $ledgerStmt->get_result();
                                        
                                        // Calculate running balance starting from opening balance
                                        $runningBalance = $openingBalance;
                                        $transactions = [];
                                        
                                        // First, get all transactions and calculate balances
                                        while ($row = $ledgerResult->fetch_assoc()) {
                                            $transactions[] = $row;
                                        }
                                        
                                        // Reverse to calculate from oldest to newest, then reverse back for display
                                        $transactions = array_reverse($transactions);
                                        foreach ($transactions as &$transaction) {
                                            $runningBalance += ($transaction['debit_amount'] - $transaction['credit_amount']);
                                            $transaction['running_balance'] = $runningBalance;
                                        }
                                        $transactions = array_reverse($transactions);
                                        
                                        foreach ($transactions as $row) {
                                        ?>
                                        <tr>
                                            <td><?php echo date('d-m-Y', strtotime($row['transaction_date'])); ?></td>
                                            <td><?php echo htmlspecialchars($row['reference_no']); ?></td>
                                            <td>
                                                <span class="badge badge-<?php 
                                                    echo $row['transaction_type'] == 'purchase' ? 'danger' : 
                                                        ($row['transaction_type'] == 'payment' ? 'success' : 'info'); 
                                                ?>">
                                                    <?php echo ucfirst($row['transaction_type']); ?>
                                                </span>
                                            </td>
                                            <td><?php echo htmlspecialchars($row['description']); ?></td>
                                            <td class="text-danger">
                                                <?php echo $row['debit_amount'] > 0 ? '₹' . number_format($row['debit_amount'], 2) : '-'; ?>
                                            </td>
                                            <td class="text-success">
                                                <?php echo $row['credit_amount'] > 0 ? '₹' . number_format($row['credit_amount'], 2) : '-'; ?>
                                            </td>
                                            <td class="<?php echo $row['running_balance'] > 0 ? 'text-danger' : 'text-success'; ?>">
                                                ₹<?php echo number_format(abs($row['running_balance']), 2); ?> <?php echo $row['running_balance'] >= 0 ? 'Dr' : 'Cr'; ?>
                                            </td>
                                            <td>
                                                <button class="btn btn-warning btn-sm" onclick="editTransaction(<?php echo $row['id']; ?>)" data-toggle="modal" data-target="#editTransactionModal">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                                <a href="?id=<?php echo $supplierId; ?>&delete_transaction=<?php echo $row['id']; ?>&from_date=<?php echo $fromDate; ?>&to_date=<?php echo $toDate; ?>" 
                                                   class="btn btn-danger btn-sm" 
                                                   onclick="return confirm('Are you sure you want to delete this transaction?')">
                                                    <i class="fas fa-trash"></i>
                                                </a>
                                            </td>
                                        </tr>
                                        <?php } ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php include('../includes/footer.php'); ?>
        </div>
    </div>

    <!-- Add Transaction Modal -->
    <div class="modal fade" id="addTransactionModal" tabindex="-1" role="dialog">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Add Transaction</h5>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Transaction Date *</label>
                                    <input type="date" class="form-control" name="transaction_date" value="<?php echo date('Y-m-d'); ?>" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Transaction Type *</label>
                                    <select class="form-control" name="transaction_type" required>
                                        <option value="">Select Type</option>
                                        <option value="purchase">Purchase (Debit)</option>
                                        <option value="payment">Payment (Credit)</option>
                                        <option value="adjustment">Adjustment</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Reference No.</label>
                                    <input type="text" class="form-control" name="reference_no">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Description</label>
                                    <input type="text" class="form-control" name="description">
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Debit Amount (₹)</label>
                                    <input type="number" step="0.01" class="form-control" name="debit_amount" min="0">
                                    <small class="text-muted">Amount we owe to supplier</small>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Credit Amount (₹)</label>
                                    <input type="number" step="0.01" class="form-control" name="credit_amount" min="0">
                                    <small class="text-muted">Amount paid to supplier</small>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
                        <button type="submit" name="submit" class="btn btn-primary">Add Transaction</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit Transaction Modal -->
    <div class="modal fade" id="editTransactionModal" tabindex="-1" role="dialog">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Edit Transaction</h5>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="transaction_id" id="edit_transaction_id">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Transaction Date *</label>
                                    <input type="date" class="form-control" name="edit_transaction_date" id="edit_transaction_date" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Transaction Type *</label>
                                    <select class="form-control" name="edit_transaction_type" id="edit_transaction_type" required>
                                        <option value="">Select Type</option>
                                        <option value="purchase">Purchase (Debit)</option>
                                        <option value="payment">Payment (Credit)</option>
                                        <option value="adjustment">Adjustment</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Reference No.</label>
                                    <input type="text" class="form-control" name="edit_reference_no" id="edit_reference_no">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Description</label>
                                    <input type="text" class="form-control" name="edit_description" id="edit_description">
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Debit Amount (₹)</label>
                                    <input type="number" step="0.01" class="form-control" name="edit_debit_amount" id="edit_debit_amount" min="0">
                                    <small class="text-muted">Amount we owe to supplier</small>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Credit Amount (₹)</label>
                                    <input type="number" step="0.01" class="form-control" name="edit_credit_amount" id="edit_credit_amount" min="0">
                                    <small class="text-muted">Amount paid to supplier</small>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
                        <button type="submit" name="update_transaction" class="btn btn-primary">Update Transaction</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="../vendor/jquery/jquery.min.js"></script>
    <script src="../vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="../vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="../js/sb-admin-2.min.js"></script>
    <script src="../vendor/datatables/jquery.dataTables.min.js"></script>
    <script src="../vendor/datatables/dataTables.bootstrap4.min.js"></script>
    
    <script>
        $(document).ready(function() {
            // Simple table initialization with proper error handling
            try {
                if (typeof $.fn.DataTable !== 'undefined') {
                    $('#dataTable').DataTable({
                        "order": [[ 0, "desc" ]],
                        "pageLength": 25
                    });
                }
            } catch (e) {
                console.log('DataTable initialization error:', e);
                // Table will work as regular HTML table without DataTable features
            }
        });
        
        function editTransaction(id) {
            // Get transaction data via AJAX
            $.ajax({
                url: 'get_transaction.php',
                type: 'GET',
                data: { id: id },
                dataType: 'json',
                success: function(data) {
                    $('#edit_transaction_id').val(data.id);
                    $('#edit_transaction_date').val(data.transaction_date);
                    $('#edit_transaction_type').val(data.transaction_type);
                    $('#edit_reference_no').val(data.reference_no);
                    $('#edit_description').val(data.description);
                    $('#edit_debit_amount').val(data.debit_amount > 0 ? data.debit_amount : '');
                    $('#edit_credit_amount').val(data.credit_amount > 0 ? data.credit_amount : '');
                },
                error: function() {
                    alert('Error loading transaction data');
                }
            });
        }
        
        function exportToCSV() {
            // Get supplier information
            var supplierName = '<?php echo htmlspecialchars($supplier['supplier_name']); ?>';
            var supplierCode = '<?php echo htmlspecialchars($supplier['supplier_code']); ?>';
            var contactPerson = '<?php echo htmlspecialchars($supplier['contact_person']); ?>';
            var phone = '<?php echo htmlspecialchars($supplier['phone']); ?>';
            var fromDate = '<?php echo date('d-m-Y', strtotime($fromDate)); ?>';
            var toDate = '<?php echo date('d-m-Y', strtotime($toDate)); ?>';
            var openingBalance = '<?php echo number_format(abs($openingBalance), 2); ?>';
            var openingBalanceType = '<?php echo $openingBalance >= 0 ? 'Dr' : 'Cr'; ?>';
            var totalDebit = '<?php echo number_format($totalDebit, 2); ?>';
            var totalCredit = '<?php echo number_format($totalCredit, 2); ?>';
            var closingBalance = '<?php echo number_format(abs($closingBalance), 2); ?>';
            var closingBalanceType = '<?php echo $closingBalance >= 0 ? 'Dr' : 'Cr'; ?>';
            
            // Create CSV content
            var csvContent = "data:text/csv;charset=utf-8,";
            
            // Add header information
            csvContent += "Supplier Ledger Report\n";
            csvContent += "Generated on: " + new Date().toLocaleDateString() + "\n";
            csvContent += "Period: " + fromDate + " to " + toDate + "\n\n";
            csvContent += "Supplier Information:\n";
            csvContent += "Supplier Name," + supplierName + "\n";
            csvContent += "Supplier Code," + supplierCode + "\n";
            csvContent += "Contact Person," + contactPerson + "\n";
            csvContent += "Phone," + phone + "\n\n";
            
            // Add balance summary
            csvContent += "Balance Summary:\n";
            csvContent += "Opening Balance,₹" + openingBalance + " " + openingBalanceType + "\n";
            csvContent += "Total Debit,₹" + totalDebit + "\n";
            csvContent += "Total Credit,₹" + totalCredit + "\n";
            csvContent += "Closing Balance,₹" + closingBalance + " " + closingBalanceType + "\n\n";
            
            // Add table headers
            csvContent += "Date,Reference,Type,Description,Debit Amount,Credit Amount,Balance\n";
            
            // Add table data
            $('#dataTable tbody tr').each(function() {
                var row = [];
                $(this).find('td').each(function(index) {
                    if (index < 7) { // Only first 7 columns (excluding Actions)
                        var cellText = $(this).text().trim();
                        
                        // Remove currency symbols and clean up text
                        if (cellText.includes('₹')) {
                            cellText = cellText.replace('₹', '').replace(/,/g, '');
                        }
                        
                        // Handle commas in text
                        if (cellText.includes(',')) {
                            cellText = '"' + cellText + '"';
                        }
                        
                        row.push(cellText);
                    }
                });
                csvContent += row.join(',') + '\n';
            });
            
            // Create and download file
            var encodedUri = encodeURI(csvContent);
            var link = document.createElement("a");
            link.setAttribute("href", encodedUri);
            link.setAttribute("download", "Supplier_Ledger_" + supplierCode + "_" + fromDate.replace(/\//g, '-') + "_to_" + toDate.replace(/\//g, '-') + ".csv");
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }
        
    </script>
</body>
</html>
