// Customer Management JavaScript
class CustomerManager {
    constructor() {
        this.init();
    }

    init() {
        // Initialize any customer-specific functionality
        console.log('Customer Manager initialized');
    }

    // Validate form data
    validateCustomerForm(formData) {
        const errors = [];
        
        if (!formData.get('customer_name') || formData.get('customer_name').trim() === '') {
            errors.push('Customer name is required');
        }
        
        if (!formData.get('customer_code') || formData.get('customer_code').trim() === '') {
            errors.push('Customer code is required');
        }
        
        const phone = formData.get('phone');
        if (phone && phone.length > 0) {
            // Basic phone validation
            const phoneRegex = /^[0-9+\-\s()]+$/;
            if (!phoneRegex.test(phone)) {
                errors.push('Please enter a valid phone number');
            }
        }
        
        const email = formData.get('email');
        if (email && email.length > 0) {
            // Basic email validation
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(email)) {
                errors.push('Please enter a valid email address');
            }
        }
        
        const gst = formData.get('gst_number');
        if (gst && gst.length > 0) {
            // Basic GST validation (15 characters)
            if (gst.length !== 15) {
                errors.push('GST number must be exactly 15 characters');
            }
        }
        
        const creditLimit = formData.get('credit_limit');
        if (creditLimit && parseFloat(creditLimit) < 0) {
            errors.push('Credit limit cannot be negative');
        }
        
        return errors;
    }

    // Format phone number for display
    formatPhoneNumber(phone) {
        if (!phone) return '';
        
        // Remove any non-digit characters
        const cleaned = phone.replace(/\D/g, '');
        
        // Format as Indian mobile number if 10 digits
        if (cleaned.length === 10) {
            return `+91 ${cleaned.slice(0, 5)} ${cleaned.slice(5)}`;
        }
        
        return phone;
    }

    // Format currency for display
    formatCurrency(amount) {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR'
        }).format(amount);
    }

    // Show success message
    showSuccess(message) {
        // You can customize this to use your preferred notification system
        if (typeof toastr !== 'undefined') {
            toastr.success(message);
        } else {
            alert(message);
        }
    }

    // Show error message
    showError(message) {
        // You can customize this to use your preferred notification system
        if (typeof toastr !== 'undefined') {
            toastr.error(message);
        } else {
            alert('Error: ' + message);
        }
    }

    // Export customer data
    exportToCSV(customers) {
        const headers = ['Code', 'Name', 'Phone', 'Email', 'GST Number', 'Credit Limit', 'Outstanding', 'Status', 'Created Date'];
        const csvContent = [
            headers.join(','),
            ...customers.map(customer => [
                customer.customer_code,
                `"${customer.customer_name}"`,
                customer.phone || '',
                customer.email || '',
                customer.gst_number || '',
                customer.credit_limit,
                customer.outstanding_balance,
                customer.status,
                customer.created_date
            ].join(','))
        ].join('\n');

        const blob = new Blob([csvContent], { type: 'text/csv' });
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `customers_${new Date().toISOString().split('T')[0]}.csv`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        window.URL.revokeObjectURL(url);
    }
}

// Initialize customer manager when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    window.customerManager = new CustomerManager();
});

// Utility functions for customer operations
function generateRandomColor() {
    const colors = ['#007bff', '#28a745', '#ffc107', '#dc3545', '#6f42c1', '#fd7e14', '#20c997'];
    return colors[Math.floor(Math.random() * colors.length)];
}

function getInitials(name) {
    return name
        .split(' ')
        .map(word => word.charAt(0))
        .join('')
        .toUpperCase()
        .slice(0, 2);
}
