<?php
session_start();
include_once('../includes/config.php');

if (!isset($_SESSION['uid'])) {
    header('location:../login.php');
    exit();
}

// Get search parameters
$search = isset($_GET['search']) ? mysqli_real_escape_string($con, $_GET['search']) : '';
$status_filter = isset($_GET['status']) ? mysqli_real_escape_string($con, $_GET['status']) : '';

// Build search query
$where_conditions = ["c.id IS NOT NULL"];

if (!empty($search)) {
    $where_conditions[] = "(c.customer_name LIKE '%$search%' OR c.phone LIKE '%$search%' OR c.customer_code LIKE '%$search%' OR c.email LIKE '%$search%')";
}

if (!empty($status_filter)) {
    $where_conditions[] = "c.status = '$status_filter'";
}

$where_clause = implode(' AND ', $where_conditions);

// Get customers with their outstanding balance
$query = "SELECT c.*, 
          COALESCE(SUM(CASE WHEN cl.transaction_type = 'invoice' THEN cl.debit_amount ELSE 0 END), 0) - 
          COALESCE(SUM(CASE WHEN cl.transaction_type = 'payment' THEN cl.credit_amount ELSE 0 END), 0) as calculated_outstanding
          FROM customers c 
          LEFT JOIN customer_ledger cl ON c.id = cl.customer_id 
          WHERE $where_clause 
          GROUP BY c.id 
          ORDER BY c.customer_name ASC";

$result = mysqli_query($con, $query);

// Get statistics
$stats_query = "SELECT 
    COUNT(*) as total_customers,
    SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active_customers,
    SUM(CASE WHEN status = 'inactive' THEN 1 ELSE 0 END) as inactive_customers,
    SUM(outstanding_balance) as total_outstanding
    FROM customers";
$stats_result = mysqli_query($con, $stats_query);
$stats = mysqli_fetch_assoc($stats_result);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <title>Customer Management - Vegetable Shop</title>

    <!-- Fonts -->
    <link href="../vendor/fontawesome-free/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">

    <!-- Styles -->
    <link href="../css/sb-admin-2.min.css" rel="stylesheet">
    <link href="../css/enhanced-admin.css" rel="stylesheet">
    <link href="../vendor/datatables/dataTables.bootstrap4.min.css" rel="stylesheet">
    <link href="css/sales.css" rel="stylesheet">

    <style>
        .customer-card {
            transition: all 0.3s ease;
            border: 1px solid #e3e6f0;
        }
        .customer-card:hover {
            box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
            transform: translateY(-2px);
        }
        .customer-status {
            font-size: 0.8rem;
            padding: 0.25rem 0.5rem;
            border-radius: 0.25rem;
        }
        .customer-status.active {
            background-color: #d4edda;
            color: #155724;
        }
        .customer-status.inactive {
            background-color: #f8d7da;
            color: #721c24;
        }
        .outstanding-positive {
            color: #e74a3b;
            font-weight: 600;
        }
        .outstanding-zero {
            color: #5a5c69;
        }
        .quick-stats {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .stat-item {
            text-align: center;
        }
        .stat-number {
            font-size: 2rem;
            font-weight: 700;
            display: block;
        }
        .stat-label {
            font-size: 0.9rem;
            opacity: 0.9;
        }
        .search-section {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .customer-table th {
            background-color: #f8f9fa;
            border-top: none;
            font-weight: 600;
            font-size: 0.85rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            color: #5a5c69;
        }
        .btn-action {
            padding: 0.25rem 0.5rem;
            font-size: 0.75rem;
            margin: 0 0.1rem;
        }
        .modal-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        .modal-header .close {
            color: white;
            opacity: 0.8;
        }
        .modal-header .close:hover {
            opacity: 1;
        }
    </style>
</head>

<body class="bg-light">
    <!-- Navigation -->
    <div class="container-fluid">
        <div class="row">
            <div class="col-12">
                <div class="d-flex justify-content-between align-items-center py-3">
                    <div>
                        <h3 class="mb-0">
                            <i class="fas fa-users mr-2 text-primary"></i>Customer Management
                        </h3>
                        <small class="text-muted">Manage your customers and their information</small>
                    </div>
                    <div>
                        <button type="button" class="btn btn-success mr-2" data-toggle="modal" data-target="#customerModal" onclick="resetCustomerForm()">
                            <i class="fas fa-user-plus mr-2"></i>Add New Customer
                        </button>
                        <a href="index.php" class="btn btn-primary">
                            <i class="fas fa-arrow-left mr-2"></i>Back to Sales
                        </a>
                        <a href="../dashboard.php" class="btn btn-secondary">
                            <i class="fas fa-home mr-2"></i>Dashboard
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Quick Statistics -->
        <div class="quick-stats">
            <div class="row">
                <div class="col-md-3">
                    <div class="stat-item">
                        <span class="stat-number"><?php echo $stats['total_customers']; ?></span>
                        <span class="stat-label">Total Customers</span>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="stat-item">
                        <span class="stat-number"><?php echo $stats['active_customers']; ?></span>
                        <span class="stat-label">Active Customers</span>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="stat-item">
                        <span class="stat-number"><?php echo $stats['inactive_customers']; ?></span>
                        <span class="stat-label">Inactive Customers</span>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="stat-item">
                        <span class="stat-number">₹<?php echo number_format($stats['total_outstanding'], 2); ?></span>
                        <span class="stat-label">Total Outstanding</span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Search and Filters -->
        <div class="search-section">
            <form method="GET" id="searchForm">
                <div class="row align-items-end">
                    <div class="col-md-4">
                        <label for="search" class="form-label"><i class="fas fa-search mr-1"></i>Search Customers</label>
                        <input type="text" class="form-control" id="search" name="search" 
                               placeholder="Name, phone, code, or email..." value="<?php echo htmlspecialchars($search); ?>">
                    </div>
                    <div class="col-md-3">
                        <label for="status" class="form-label"><i class="fas fa-filter mr-1"></i>Status Filter</label>
                        <select class="form-control" id="status" name="status">
                            <option value="">All Status</option>
                            <option value="active" <?php echo $status_filter === 'active' ? 'selected' : ''; ?>>Active</option>
                            <option value="inactive" <?php echo $status_filter === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                        </select>
                    </div>
                    <div class="col-md-5">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-search mr-2"></i>Search
                        </button>
                        <a href="customers.php" class="btn btn-secondary">
                            <i class="fas fa-undo mr-2"></i>Clear
                        </a>
                        <button type="button" class="btn btn-info" onclick="exportCustomers()">
                            <i class="fas fa-file-export mr-2"></i>Export
                        </button>
                        <button type="button" class="btn btn-warning" onclick="printCustomerList()">
                            <i class="fas fa-print mr-2"></i>Print
                        </button>
                    </div>
                </div>
            </form>
        </div>

        <!-- Customer Table -->
        <div class="card shadow">
            <div class="card-header py-3">
                <h6 class="m-0 font-weight-bold text-primary">
                    <i class="fas fa-table mr-2"></i>Customer List
                    <?php if (mysqli_num_rows($result) > 0): ?>
                        <span class="badge badge-primary ml-2"><?php echo mysqli_num_rows($result); ?> customers</span>
                    <?php endif; ?>
                </h6>
            </div>
            <div class="card-body">
                <?php if (mysqli_num_rows($result) > 0): ?>
                <div class="table-responsive">
                    <table class="table table-bordered customer-table" id="customerTable">
                        <thead>
                            <tr>
                                <th>Code</th>
                                <th>Customer Name</th>
                                <th>Phone</th>
                                <th>Email</th>
                                <th>Credit Limit</th>
                                <th>Outstanding</th>
                                <th>Status</th>
                                <th>Created Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while ($customer = mysqli_fetch_assoc($result)): ?>
                            <tr>
                                <td><span class="badge badge-secondary"><?php echo htmlspecialchars($customer['customer_code']); ?></span></td>
                                <td>
                                    <strong><?php echo htmlspecialchars($customer['customer_name']); ?></strong>
                                    <?php if (!empty($customer['gst_number'])): ?>
                                        <br><small class="text-muted">GST: <?php echo htmlspecialchars($customer['gst_number']); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if (!empty($customer['phone'])): ?>
                                        <a href="tel:<?php echo $customer['phone']; ?>" class="text-decoration-none">
                                            <i class="fas fa-phone mr-1"></i><?php echo htmlspecialchars($customer['phone']); ?>
                                        </a>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if (!empty($customer['email'])): ?>
                                        <a href="mailto:<?php echo $customer['email']; ?>" class="text-decoration-none">
                                            <i class="fas fa-envelope mr-1"></i><?php echo htmlspecialchars($customer['email']); ?>
                                        </a>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                                <td>₹<?php echo number_format($customer['credit_limit'], 2); ?></td>
                                <td>
                                    <?php 
                                    $outstanding = $customer['calculated_outstanding'];
                                    $class = $outstanding > 0 ? 'outstanding-positive' : 'outstanding-zero';
                                    ?>
                                    <span class="<?php echo $class; ?>">₹<?php echo number_format($outstanding, 2); ?></span>
                                </td>
                                <td>
                                    <span class="customer-status <?php echo $customer['status']; ?>">
                                        <?php echo ucfirst($customer['status']); ?>
                                    </span>
                                </td>
                                <td><?php echo date('d M Y', strtotime($customer['created_date'])); ?></td>
                                <td>
                                    <button type="button" class="btn btn-info btn-action" 
                                            onclick="viewCustomer(<?php echo $customer['id']; ?>)" title="View Details">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                    <button type="button" class="btn btn-warning btn-action" 
                                            onclick="editCustomer(<?php echo $customer['id']; ?>)" title="Edit Customer">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <?php if ($outstanding > 0): ?>
                                    <button type="button" class="btn btn-success btn-action" 
                                            onclick="collectPayment(<?php echo $customer['id']; ?>)" title="Collect Payment">
                                        <i class="fas fa-hand-holding-usd"></i>
                                    </button>
                                    <?php endif; ?>
                                    <button type="button" class="btn btn-danger btn-action" 
                                            onclick="deleteCustomer(<?php echo $customer['id']; ?>)" title="Delete Customer">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </td>
                            </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
                <?php else: ?>
                <div class="text-center py-5">
                    <i class="fas fa-users fa-3x text-muted mb-3"></i>
                    <h5 class="text-muted">No customers found</h5>
                    <p class="text-muted">Start by adding your first customer!</p>
                    <button type="button" class="btn btn-success" data-toggle="modal" data-target="#customerModal" onclick="resetCustomerForm()">
                        <i class="fas fa-user-plus mr-2"></i>Add New Customer
                    </button>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Customer Modal -->
    <div class="modal fade" id="customerModal" tabindex="-1" role="dialog" aria-labelledby="customerModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="customerModalLabel">
                        <i class="fas fa-user-plus mr-2"></i>Add New Customer
                    </h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <form id="customerForm">
                    <div class="modal-body">
                        <input type="hidden" id="customerId" name="customer_id">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="customerCode"><i class="fas fa-barcode mr-2"></i>Customer Code *</label>
                                    <input type="text" class="form-control" id="customerCode" name="customer_code" required>
                                    <small class="form-text text-muted">Unique identifier for the customer</small>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="customerName"><i class="fas fa-user mr-2"></i>Customer Name *</label>
                                    <input type="text" class="form-control" id="customerName" name="customer_name" required>
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="customerPhone"><i class="fas fa-phone mr-2"></i>Phone Number</label>
                                    <input type="tel" class="form-control" id="customerPhone" name="phone">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="customerEmail"><i class="fas fa-envelope mr-2"></i>Email Address</label>
                                    <input type="email" class="form-control" id="customerEmail" name="email">
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="customerGst"><i class="fas fa-file-invoice mr-2"></i>GST Number</label>
                                    <input type="text" class="form-control" id="customerGst" name="gst_number" maxlength="15">
                                    <small class="form-text text-muted">15-digit GST number (optional)</small>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="customerCreditLimit"><i class="fas fa-credit-card mr-2"></i>Credit Limit</label>
                                    <input type="number" class="form-control" id="customerCreditLimit" name="credit_limit" step="0.01" min="0" value="0">
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-8">
                                <div class="form-group">
                                    <label for="customerAddress"><i class="fas fa-map-marker-alt mr-2"></i>Address</label>
                                    <textarea class="form-control" id="customerAddress" name="address" rows="3"></textarea>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="customerCity"><i class="fas fa-city mr-2"></i>City</label>
                                    <input type="text" class="form-control" id="customerCity" name="city">
                                </div>
                                <div class="form-group">
                                    <label for="customerStatus"><i class="fas fa-toggle-on mr-2"></i>Status</label>
                                    <select class="form-control" id="customerStatus" name="status">
                                        <option value="active">Active</option>
                                        <option value="inactive">Inactive</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">
                            <i class="fas fa-times mr-2"></i>Cancel
                        </button>
                        <button type="submit" class="btn btn-success">
                            <i class="fas fa-save mr-2"></i>Save Customer
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Customer View Modal -->
    <div class="modal fade" id="customerViewModal" tabindex="-1" role="dialog" aria-labelledby="customerViewModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="customerViewModalLabel">
                        <i class="fas fa-user mr-2"></i>Customer Details
                    </h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body" id="customerViewContent">
                    <!-- Customer details will be loaded here -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">
                        <i class="fas fa-times mr-2"></i>Close
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="../vendor/jquery/jquery.min.js"></script>
    <script src="../vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="../vendor/datatables/jquery.dataTables.min.js"></script>
    <script src="../vendor/datatables/dataTables.bootstrap4.min.js"></script>
    <script src="js/customers.js"></script>

    <script>
        // Initialize DataTable
        $(document).ready(function() {
            $('#customerTable').DataTable({
                "order": [[ 1, "asc" ]],
                "pageLength": 25,
                "responsive": true,
                "language": {
                    "search": "Search customers:",
                    "lengthMenu": "Show _MENU_ customers per page",
                    "info": "Showing _START_ to _END_ of _TOTAL_ customers",
                    "paginate": {
                        "first": "First",
                        "last": "Last",
                        "next": "Next",
                        "previous": "Previous"
                    }
                }
            });

            // Auto-generate customer code
            generateCustomerCode();
        });

        // Generate customer code
        function generateCustomerCode() {
            fetch('ajax/generate_customer_code.php')
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('customerCode').value = data.customer_code;
                }
            })
            .catch(error => console.error('Error:', error));
        }

        // Reset customer form
        function resetCustomerForm() {
            document.getElementById('customerForm').reset();
            document.getElementById('customerId').value = '';
            document.getElementById('customerModalLabel').innerHTML = '<i class="fas fa-user-plus mr-2"></i>Add New Customer';
            generateCustomerCode();
        }

        // Form submission
        document.getElementById('customerForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            const isEdit = document.getElementById('customerId').value !== '';
            
            fetch('ajax/save_customer.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert(isEdit ? 'Customer updated successfully!' : 'Customer added successfully!');
                    $('#customerModal').modal('hide');
                    location.reload();
                } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred while saving the customer.');
            });
        });

        // Edit customer
        function editCustomer(customerId) {
            fetch('ajax/get_customer.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({customer_id: customerId})
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const customer = data.customer;
                    document.getElementById('customerId').value = customer.id;
                    document.getElementById('customerCode').value = customer.customer_code;
                    document.getElementById('customerName').value = customer.customer_name;
                    document.getElementById('customerPhone').value = customer.phone || '';
                    document.getElementById('customerEmail').value = customer.email || '';
                    document.getElementById('customerGst').value = customer.gst_number || '';
                    document.getElementById('customerCreditLimit').value = customer.credit_limit;
                    document.getElementById('customerAddress').value = customer.address || '';
                    document.getElementById('customerCity').value = customer.city || '';
                    document.getElementById('customerStatus').value = customer.status;
                    
                    document.getElementById('customerModalLabel').innerHTML = '<i class="fas fa-user-edit mr-2"></i>Edit Customer';
                    $('#customerModal').modal('show');
                } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred while loading customer data.');
            });
        }

        // View customer
        function viewCustomer(customerId) {
            fetch('ajax/get_customer_details.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({customer_id: customerId})
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('customerViewContent').innerHTML = data.html;
                    $('#customerViewModal').modal('show');
                } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred while loading customer details.');
            });
        }

        // Delete customer
        function deleteCustomer(customerId) {
            if (confirm('Are you sure you want to delete this customer? This action cannot be undone.')) {
                fetch('ajax/delete_customer.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({customer_id: customerId})
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('Customer deleted successfully!');
                        location.reload();
                    } else {
                        alert('Error: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('An error occurred while deleting the customer.');
                });
            }
        }

        // Collect payment
        function collectPayment(customerId) {
            // Open payment collection modal by redirecting to sales page with auto-open modal
            window.open('index.php?open_payment_modal=' + customerId, '_blank');
        }

        // Export customers
        function exportCustomers() {
            const searchParams = new URLSearchParams(window.location.search);
            window.location.href = 'ajax/export_customers.php?' + searchParams.toString();
        }

        // Print customer list
        function printCustomerList() {
            window.print();
        }

        // Keyboard shortcuts
        document.addEventListener('keydown', function(e) {
            // Ctrl+N - Add New Customer
            if (e.ctrlKey && e.key === 'n') {
                e.preventDefault();
                resetCustomerForm();
                $('#customerModal').modal('show');
            }
            // Ctrl+F - Focus search
            else if (e.ctrlKey && e.key === 'f') {
                e.preventDefault();
                document.getElementById('search').focus();
            }
            // Esc - Go back to sales
            else if (e.key === 'Escape') {
                e.preventDefault();
                window.location.href = 'index.php';
            }
        });
    </script>
</body>
</html>
