-- ALTER TABLE statements for Bill Hold functionality and Integer Amounts
-- Run these commands in your MySQL database

-- 1. Add hold functionality columns to sales_invoices table
ALTER TABLE `sales_invoices` 
ADD COLUMN `is_hold` TINYINT(1) DEFAULT 0 COMMENT 'Whether this invoice is on hold (0=normal, 1=hold)',
ADD COLUMN `hold_number` VARCHAR(20) NULL COMMENT 'Hold reference number like HOLD-001',
ADD COLUMN `hold_date` TIMESTAMP NULL COMMENT 'When the bill was put on hold',
ADD COLUMN `hold_notes` TEXT NULL COMMENT 'Notes for the hold',
ADD COLUMN `released_from_hold_date` TIMESTAMP NULL COMMENT 'When the bill was released from hold';

-- 2. Add indexes for better performance on hold operations
ALTER TABLE `sales_invoices` 
ADD INDEX `idx_hold_status` (`is_hold`),
ADD INDEX `idx_hold_number` (`hold_number`),
ADD INDEX `idx_hold_date` (`hold_date`);

-- 3. Create bill_holds table for managing multiple holds
CREATE TABLE IF NOT EXISTS `bill_holds` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `hold_number` VARCHAR(20) NOT NULL,
  `customer_name` VARCHAR(100) NULL,
  `customer_phone` VARCHAR(15) NULL,
  `customer_address` TEXT NULL,
  `customer_gst` VARCHAR(20) NULL,
  `subtotal` INT NOT NULL DEFAULT 0 COMMENT 'Subtotal as integer (no decimals)',
  `discount_percent` DECIMAL(5,2) DEFAULT 0.00,
  `discount_amount` INT DEFAULT 0 COMMENT 'Discount amount as integer',
  `total_amount` INT NOT NULL DEFAULT 0 COMMENT 'Final total as integer (no decimals)',
  `payment_mode` ENUM('cash','upi','mixed','credit') DEFAULT 'cash',
  `cash_tender_received` INT DEFAULT 0 COMMENT 'Cash amount as integer',
  `upi_amount` INT DEFAULT 0 COMMENT 'UPI amount as integer',
  `cash_change` INT DEFAULT 0 COMMENT 'Cash change as integer',
  `notes` TEXT NULL,
  `invoice_type` ENUM('cash','credit') DEFAULT 'cash',
  `credit_days` INT NULL,
  `rate_type` ENUM('CustomerRate','ShopRate','WholesaleRate') DEFAULT 'CustomerRate',
  `hold_date` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `status` ENUM('active','released','cancelled') DEFAULT 'active',
  `created_by` INT NOT NULL,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_hold_number` (`hold_number`),
  KEY `idx_hold_status` (`status`),
  KEY `idx_hold_date` (`hold_date`),
  KEY `idx_hold_customer` (`customer_name`),
  KEY `idx_hold_created_by` (`created_by`)
) ENGINE=MyISAM DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci COMMENT='Table to store multiple bill holds with integer amounts';

-- 4. Create bill_hold_details table for storing hold line items
CREATE TABLE IF NOT EXISTS `bill_hold_details` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `hold_id` INT NOT NULL,
  `product_code` VARCHAR(20) NOT NULL,
  `product_name` VARCHAR(100) NOT NULL,
  `unit` VARCHAR(20) NOT NULL,
  `quantity` DECIMAL(10,3) NOT NULL,
  `rate` INT NOT NULL COMMENT 'Rate as integer (no decimals)',
  `amount` INT NOT NULL COMMENT 'Line amount as integer (no decimals)',
  `created_date` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_hold_details_hold` (`hold_id`),
  KEY `idx_hold_details_product` (`product_code`)
) ENGINE=MyISAM DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci COMMENT='Line items for bill holds with integer amounts';

-- 5. Create system_settings table for configuration
CREATE TABLE IF NOT EXISTS `system_settings` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `setting_key` VARCHAR(50) NOT NULL,
  `setting_value` TEXT NOT NULL,
  `setting_description` TEXT NULL,
  `created_date` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_date` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_setting_key` (`setting_key`)
) ENGINE=MyISAM DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci COMMENT='System configuration settings';

-- 6. Insert default settings for hold and rounding
INSERT IGNORE INTO `system_settings` (`setting_key`, `setting_value`, `setting_description`) VALUES
('max_hold_bills', '20', 'Maximum number of bills that can be held simultaneously'),
('auto_round_amounts', '1', 'Automatically round all amounts to integers (1=enabled, 0=disabled)'),
('hold_auto_increment', '1', 'Auto increment counter for hold numbers');

-- 7. OPTIONAL: If you want to modify existing sales_invoices table to use integers
-- WARNING: This will change existing data - backup first!
-- ALTER TABLE `sales_invoices` 
-- MODIFY COLUMN `subtotal` INT NOT NULL DEFAULT 0 COMMENT 'Subtotal as integer',
-- MODIFY COLUMN `discount_amount` INT DEFAULT 0 COMMENT 'Discount amount as integer', 
-- MODIFY COLUMN `total_amount` INT NOT NULL DEFAULT 0 COMMENT 'Total amount as integer',
-- MODIFY COLUMN `cash_received` INT DEFAULT 0 COMMENT 'Cash received as integer',
-- MODIFY COLUMN `upi_received` INT DEFAULT 0 COMMENT 'UPI received as integer',
-- MODIFY COLUMN `balance_amount` INT DEFAULT 0 COMMENT 'Balance amount as integer',
-- MODIFY COLUMN `cash_tender_received` INT DEFAULT 0 COMMENT 'Cash tender as integer',
-- MODIFY COLUMN `cash_change` INT DEFAULT 0 COMMENT 'Cash change as integer';

-- 8. OPTIONAL: Modify sales_invoice_details for integer amounts
-- WARNING: This will change existing data - backup first!
-- ALTER TABLE `sales_invoice_details`
-- MODIFY COLUMN `rate` INT NOT NULL COMMENT 'Rate as integer',
-- MODIFY COLUMN `amount` INT NOT NULL COMMENT 'Amount as integer';

-- 9. Add a function to generate hold numbers
DELIMITER //
CREATE FUNCTION IF NOT EXISTS GenerateHoldNumber() 
RETURNS VARCHAR(20)
READS SQL DATA
DETERMINISTIC
BEGIN
    DECLARE next_num INT;
    DECLARE hold_num VARCHAR(20);
    
    -- Get the next hold number
    SELECT COALESCE(MAX(CAST(SUBSTRING(hold_number, 6) AS UNSIGNED)), 0) + 1 
    INTO next_num 
    FROM bill_holds 
    WHERE hold_number LIKE 'HOLD-%';
    
    -- Format as HOLD-001, HOLD-002, etc.
    SET hold_num = CONCAT('HOLD-', LPAD(next_num, 3, '0'));
    
    RETURN hold_num;
END//
DELIMITER ;

-- Verification queries (run these to check if everything was created successfully):
-- SELECT * FROM system_settings WHERE setting_key LIKE '%hold%' OR setting_key LIKE '%round%';
-- DESCRIBE bill_holds;
-- DESCRIBE bill_hold_details;
-- SHOW COLUMNS FROM sales_invoices WHERE Field LIKE '%hold%';
-- SELECT GenerateHoldNumber() as next_hold_number;
