<?php
header('Content-Type: application/json');
include_once('../../includes/config.php');

$response = ['success' => false, 'message' => '', 'receipt_id' => null];

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    $requiredFields = ['receipt_no', 'receipt_date', 'customer_id', 'total_amount'];
    foreach ($requiredFields as $field) {
        if (empty($input[$field])) {
            throw new Exception("Required field missing: $field");
        }
    }
    
    $con->begin_transaction();
    
    // Insert payment receipt
    $sql = "INSERT INTO payment_receipts (
                receipt_no, receipt_date, customer_id, customer_name, payment_mode,
                cash_amount, upi_amount, cheque_amount, bank_amount, 
                cheque_no, cheque_date, bank_details, total_amount, notes, created_by
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $stmt = $con->prepare($sql);
    $stmt->bind_param('ssissddddsssdi',
        $input['receipt_no'],
        $input['receipt_date'],
        $input['customer_id'],
        $input['customer_name'],
        $input['payment_mode'],
        $input['cash_amount'],
        $input['upi_amount'],
        $input['cheque_amount'],
        $input['bank_amount'],
        $input['cheque_no'],
        $input['cheque_date'],
        $input['bank_details'],
        $input['total_amount'],
        $input['notes'],
        $_SESSION['uid']
    );
    
    if (!$stmt->execute()) {
        throw new Exception('Error saving payment receipt');
    }
    
    $receiptId = $con->insert_id;
    
    // Process allocations
    if (!empty($input['allocations'])) {
        foreach ($input['allocations'] as $allocation) {
            $invoiceId = intval($allocation['invoice_id']);
            $allocatedAmount = floatval($allocation['allocated_amount']);
            
            if ($invoiceId > 0 && $allocatedAmount > 0) {
                // Insert allocation record
                $sqlAlloc = "INSERT INTO payment_allocations (payment_receipt_id, invoice_id, allocated_amount) 
                            VALUES (?, ?, ?)";
                $stmtAlloc = $con->prepare($sqlAlloc);
                $stmtAlloc->bind_param('iid', $receiptId, $invoiceId, $allocatedAmount);
                
                if (!$stmtAlloc->execute()) {
                    throw new Exception('Error saving payment allocation');
                }
                
                // Update invoice payment status
                $sqlUpdate = "UPDATE sales_invoices 
                             SET cash_received = cash_received + ?,
                                 payment_status = CASE 
                                     WHEN (cash_received + upi_received + ?) >= total_amount THEN 'paid'
                                     WHEN (cash_received + upi_received + ?) > 0 THEN 'partial'
                                     ELSE 'unpaid'
                                 END,
                                 balance_amount = total_amount - (cash_received + upi_received + ?)
                             WHERE id = ?";
                
                $stmtUpdate = $con->prepare($sqlUpdate);
                $stmtUpdate->bind_param('ddddi', $allocatedAmount, $allocatedAmount, $allocatedAmount, $allocatedAmount, $invoiceId);
                
                if (!$stmtUpdate->execute()) {
                    throw new Exception('Error updating invoice payment status');
                }
                
                // Add to customer ledger
                $sqlLedger = "INSERT INTO customer_ledger (
                                customer_id, transaction_date, transaction_type, reference_no, reference_id,
                                credit_amount, description, created_by
                             ) VALUES (?, ?, 'payment', ?, ?, ?, ?, ?)";
                
                $description = "Payment received against invoice " . $allocation['invoice_no'] ?? $invoiceId;
                $stmtLedger = $con->prepare($sqlLedger);
                $stmtLedger->bind_param('issiisi',
                    $input['customer_id'],
                    $input['receipt_date'],
                    $input['receipt_no'],
                    $receiptId,
                    $allocatedAmount,
                    $description,
                    $_SESSION['uid']
                );
                
                if (!$stmtLedger->execute()) {
                    throw new Exception('Error updating customer ledger');
                }
            }
        }
    }
    
    // Update customer outstanding balance
    $sqlCustomer = "UPDATE customers 
                   SET outstanding_balance = (
                       SELECT COALESCE(SUM(total_amount - cash_received - upi_received), 0)
                       FROM sales_invoices 
                       WHERE customer_id = ? AND payment_status IN ('unpaid', 'partial')
                   )
                   WHERE id = ?";
    
    $stmtCustomer = $con->prepare($sqlCustomer);
    $stmtCustomer->bind_param('ii', $input['customer_id'], $input['customer_id']);
    
    if (!$stmtCustomer->execute()) {
        throw new Exception('Error updating customer outstanding balance');
    }
    
    $con->commit();
    
    $response['success'] = true;
    $response['message'] = 'Payment receipt saved successfully';
    $response['receipt_id'] = $receiptId;
    
} catch (Exception $e) {
    $con->rollback();
    $response['message'] = $e->getMessage();
}

echo json_encode($response);
?>
