<?php
session_start();
if (strlen($_SESSION['uid']) == 0) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

include('../../includes/config.php');

header('Content-Type: application/json');

try {
    // Get form data (both POST and JSON input)
    $input_data = [];
    
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST)) {
        // Form data
        $input_data = $_POST;
    } else {
        // JSON input
        $input_data = json_decode(file_get_contents('php://input'), true);
    }
    
    if (!$input_data) {
        throw new Exception('No payment data received');
    }
    
    // Extract and validate data
    $customer_id = intval($input_data['customer_id'] ?? 0);
    $payment_amount = floatval($input_data['payment_amount'] ?? $input_data['total_amount'] ?? 0);
    $payment_mode = mysqli_real_escape_string($con, $input_data['payment_mode'] ?? '');
    $payment_date = mysqli_real_escape_string($con, $input_data['payment_date'] ?? $input_data['receipt_date'] ?? date('Y-m-d'));
    $reference_no = mysqli_real_escape_string($con, $input_data['reference_no'] ?? '');
    $notes = mysqli_real_escape_string($con, $input_data['notes'] ?? '');
    
    // Debug logging
    error_log("Payment collection debug - Customer ID: $customer_id, Amount: $payment_amount, Mode: $payment_mode");
    error_log("Input data: " . json_encode($input_data));
    
    // Enhanced payment mode data
    $cash_amount = floatval($input_data['cash_amount'] ?? 0);
    $upi_amount = floatval($input_data['upi_amount'] ?? 0);
    $cheque_amount = floatval($input_data['cheque_amount'] ?? 0);
    $bank_amount = floatval($input_data['bank_amount'] ?? 0);
    $cheque_no = mysqli_real_escape_string($con, $input_data['cheque_no'] ?? '');
    $cheque_date = mysqli_real_escape_string($con, $input_data['cheque_date'] ?? '');
    $bank_details = mysqli_real_escape_string($con, $input_data['bank_details'] ?? '');
    
    $created_by = $_SESSION['uid'];
    
    // Validation with detailed error messages
    if (empty($customer_id) || $customer_id <= 0) {
        throw new Exception("Invalid customer ID: $customer_id");
    }
    if ($payment_amount <= 0) {
        throw new Exception("Invalid payment amount: $payment_amount");
    }
    if (empty($payment_mode)) {
        throw new Exception("Payment mode is required");
    }
    
    // Set amounts based on payment mode
    if ($payment_mode === 'mixed') {
        // For mixed payment, amounts are already set
        $total_mixed = $cash_amount + $upi_amount + $cheque_amount + $bank_amount;
        if (abs($total_mixed - $payment_amount) > 0.01) {
            throw new Exception('Mixed payment amounts do not match total');
        }
    } else {
        // For single payment mode
        $cash_amount = ($payment_mode === 'cash') ? $payment_amount : 0;
        $upi_amount = ($payment_mode === 'upi') ? $payment_amount : 0;
        $cheque_amount = ($payment_mode === 'cheque') ? $payment_amount : 0;
        $bank_amount = ($payment_mode === 'bank_transfer') ? $payment_amount : 0;
    }
    
    // Get customer data
    $customerSql = "SELECT customer_name, outstanding_balance FROM customers WHERE id = $customer_id AND status = 'active'";
    $customerResult = mysqli_query($con, $customerSql);
    
    if (!$customerResult || mysqli_num_rows($customerResult) == 0) {
        throw new Exception('Customer not found');
    }
    
    $customer = mysqli_fetch_assoc($customerResult);
    $current_outstanding = floatval($customer['outstanding_balance']);
    
    // Check if payment exceeds outstanding (optional validation)
    if ($payment_amount > $current_outstanding && $current_outstanding > 0) {
        // Allow overpayment but show warning
        $notes .= " [Overpayment: ₹" . ($payment_amount - $current_outstanding) . "]";
    }
    
    // Start transaction
    mysqli_autocommit($con, false);
    
    // Get outstanding invoices
    $invoicesSql = "SELECT id, invoice_no, total_amount, balance_amount, invoice_date
                    FROM sales_invoices 
                    WHERE customer_id = $customer_id AND balance_amount > 0 AND payment_status != 'paid' AND status = 'active'
                    ORDER BY invoice_date ASC, id ASC";
    $invoicesResult = mysqli_query($con, $invoicesSql);
    
    // Generate receipt number
    $receiptNoSql = "SELECT COALESCE(MAX(CAST(SUBSTRING(receipt_no, 13) AS UNSIGNED)), 0) + 1 as next_no 
                     FROM payment_receipts 
                     WHERE receipt_no LIKE 'RCP-" . date('Ym') . "-%'";
    $receiptNoResult = mysqli_query($con, $receiptNoSql);
    $receiptNoRow = mysqli_fetch_assoc($receiptNoResult);
    $receiptNo = 'RCP-' . date('Ym') . '-' . str_pad($receiptNoRow['next_no'], 4, '0', STR_PAD_LEFT);
    
    // Insert payment receipt
    $insertReceiptSql = "INSERT INTO payment_receipts (
        receipt_no, receipt_date, customer_id, customer_name, payment_mode,
        cash_amount, upi_amount, cheque_amount, bank_amount, total_amount,
        cheque_no, cheque_date, bank_details, notes, created_by, created_date
    ) VALUES (
        '$receiptNo', '$payment_date', $customer_id, '" . mysqli_real_escape_string($con, $customer['customer_name']) . "', '$payment_mode',
        $cash_amount, $upi_amount, $cheque_amount, $bank_amount, $payment_amount,
        '$cheque_no', " . ($cheque_date ? "'$cheque_date'" : "NULL") . ", '$bank_details', '$notes', 
        $created_by, NOW()
    )";
    
    if (!mysqli_query($con, $insertReceiptSql)) {
        throw new Exception('Failed to create payment receipt: ' . mysqli_error($con));
    }
    
    $receipt_id = mysqli_insert_id($con);
    
    // Allocate payment to invoices
    $remaining_payment = $payment_amount;
    $allocated_invoices = [];
    
    if ($invoicesResult && mysqli_num_rows($invoicesResult) > 0) {
        while ($invoice = mysqli_fetch_assoc($invoicesResult)) {
            if ($remaining_payment <= 0) break;
            
            $invoice_balance = floatval($invoice['balance_amount']);
            $allocation_amount = min($remaining_payment, $invoice_balance);
            
            // Create payment allocation
            $allocationSql = "INSERT INTO payment_allocations (payment_receipt_id, invoice_id, allocated_amount) 
                             VALUES ($receipt_id, {$invoice['id']}, $allocation_amount)";
            
            if (!mysqli_query($con, $allocationSql)) {
                throw new Exception('Failed to create payment allocation: ' . mysqli_error($con));
            }
            
            // Update invoice
            $new_invoice_balance = $invoice_balance - $allocation_amount;
            $new_payment_status = 'unpaid';
            
            if ($new_invoice_balance <= 0) {
                $new_payment_status = 'paid';
                $new_invoice_balance = 0;
            } else if ($new_invoice_balance < $invoice['total_amount']) {
                $new_payment_status = 'partial';
            }
            
            $updateInvoiceSql = "UPDATE sales_invoices 
                                SET balance_amount = $new_invoice_balance, payment_status = '$new_payment_status'
                                WHERE id = {$invoice['id']}";
            
            if (!mysqli_query($con, $updateInvoiceSql)) {
                throw new Exception('Failed to update invoice: ' . mysqli_error($con));
            }
            
            $allocated_invoices[] = [
                'invoice_no' => $invoice['invoice_no'],
                'allocated_amount' => $allocation_amount,
                'new_balance' => $new_invoice_balance,
                'payment_status' => $new_payment_status
            ];
            
            $remaining_payment -= $allocation_amount;
        }
    }
    
    // Calculate new customer balance
    $new_balance = $current_outstanding - $payment_amount;
    
    // Create ledger entry
    $description = "Payment received via " . strtoupper($payment_mode);
    if (!empty($reference_no)) {
        $description .= " (Ref: $reference_no)";
    }
    if (!empty($notes)) {
        $description .= " - $notes";
    }
    if (!empty($allocated_invoices)) {
        $description .= " | Allocated to " . count($allocated_invoices) . " invoice(s)";
    }
    
    $ledgerSql = "INSERT INTO customer_ledger (
        customer_id, transaction_date, transaction_type, reference_no, reference_id,
        debit_amount, credit_amount, balance, description, created_by, created_date
    ) VALUES (
        $customer_id, '$payment_date', 'payment', '$receiptNo', $receipt_id,
        0.00, $payment_amount, $new_balance, '$description', $created_by, NOW()
    )";
    
    if (!mysqli_query($con, $ledgerSql)) {
        throw new Exception('Failed to create ledger entry: ' . mysqli_error($con));
    }
    
    // Update customer outstanding balance
    $updateCustomerSql = "UPDATE customers SET outstanding_balance = $new_balance WHERE id = $customer_id";
    
    if (!mysqli_query($con, $updateCustomerSql)) {
        throw new Exception('Failed to update customer balance: ' . mysqli_error($con));
    }
    
    // Commit transaction
    mysqli_commit($con);
    
    echo json_encode([
        'success' => true,
        'message' => 'Payment collected successfully',
        'data' => [
            'receipt_no' => $receiptNo,
            'payment_amount' => $payment_amount,
            'new_balance' => $new_balance,
            'customer_name' => $customer['customer_name'],
            'allocated_invoices' => $allocated_invoices,
            'remaining_payment' => $remaining_payment
        ]
    ]);
    
} catch (Exception $e) {
    // Rollback transaction
    mysqli_rollback($con);
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
} finally {
    // Reset autocommit
    mysqli_autocommit($con, true);
}
?>
