<?php
session_start();
include_once('../includes/config.php');

if (!isset($_SESSION['uid'])) {
    header('location:../login.php');
    exit();
}

// Get report type and parameters
$report_type = $_GET['report_type'] ?? '';
$title = $_GET['title'] ?? 'Report';
$from_date = $_GET['from_date'] ?? '';
$to_date = $_GET['to_date'] ?? '';
$payment_status = $_GET['payment_status'] ?? '';
$customer_id = $_GET['customer_id'] ?? '';

// Build query based on report type
$data = [];
$headers = [];

switch($report_type) {
    case 'bill_wise':
        $title = 'Bill Wise Report';
        $query = "SELECT si.invoice_no, si.invoice_date, COALESCE(c.customer_name, si.customer_name, 'Walk-in') as customer_name, 
                  si.total_amount, si.payment_status, si.cash_received, si.upi_received, si.balance_amount
                  FROM sales_invoices si 
                  LEFT JOIN customers c ON si.customer_id = c.id 
                  WHERE si.status = 'active'";
        $headers = ['Invoice', 'Date', 'Customer', 'Amount', 'Status', 'Cash', 'UPI', 'Balance'];
        break;
        
    case 'item_wise':
        $title = 'Item Wise Report';
        $query = "SELECT v.VegetableName, SUM(sid.quantity) as total_qty, 
                  AVG(sid.rate) as avg_rate, SUM(sid.amount) as total_amount
                  FROM sales_invoice_details sid
                  LEFT JOIN tblvegetables v ON sid.product_code = v.VegetableCode
                  LEFT JOIN sales_invoices si ON sid.invoice_id = si.id
                  WHERE si.status = 'active'";
        $headers = ['Item', 'Qty', 'Avg Rate', 'Total'];
        break;
        
    case 'customer_wise':
        $title = 'Customer Wise Report';
        $query = "SELECT COALESCE(c.customer_name, si.customer_name, 'Walk-in') as customer_name,
                  COUNT(si.id) as total_bills, SUM(si.total_amount) as total_amount,
                  SUM(si.balance_amount) as total_balance
                  FROM sales_invoices si 
                  LEFT JOIN customers c ON si.customer_id = c.id 
                  WHERE si.status = 'active'";
        $headers = ['Customer', 'Bills', 'Amount', 'Balance'];
        break;
        
    case 'daily_summary':
        $title = 'Daily Summary Report';
        $query = "SELECT si.invoice_date, COUNT(si.id) as total_bills, 
                  SUM(si.total_amount) as total_sales, SUM(si.cash_received) as cash_sales,
                  SUM(si.upi_received) as upi_sales, SUM(si.balance_amount) as pending_amount
                  FROM sales_invoices si 
                  WHERE si.status = 'active'";
        $headers = ['Date', 'Bills', 'Sales', 'Cash', 'UPI', 'Pending'];
        break;
        
    case 'pending_payments':
        $title = 'Pending Payments Report';
        $query = "SELECT si.invoice_no, si.invoice_date, COALESCE(c.customer_name, si.customer_name, 'Walk-in') as customer_name,
                  si.total_amount, si.cash_received, si.upi_received, si.balance_amount
                  FROM sales_invoices si 
                  LEFT JOIN customers c ON si.customer_id = c.id 
                  WHERE si.status = 'active' AND si.balance_amount > 0";
        $headers = ['Invoice', 'Date', 'Customer', 'Total', 'Cash', 'UPI', 'Balance'];
        break;
        
    case 'profit_analysis':
        $title = 'Profit Analysis Report';
        $query = "SELECT si.invoice_no, si.invoice_date, si.total_amount,
                  (SELECT SUM(sid.quantity * v.PurchasedRate) FROM sales_invoice_details sid 
                   LEFT JOIN tblvegetables v ON sid.product_code = v.VegetableCode 
                   WHERE sid.invoice_id = si.id) as cost_price,
                  (si.total_amount - (SELECT SUM(sid.quantity * v.PurchasedRate) FROM sales_invoice_details sid 
                   LEFT JOIN tblvegetables v ON sid.product_code = v.VegetableCode 
                   WHERE sid.invoice_id = si.id)) as profit
                  FROM sales_invoices si 
                  WHERE si.status = 'active'";
        $headers = ['Invoice', 'Date', 'Sales', 'Cost', 'Profit'];
        break;
        
    case 'inventory_status':
        $title = 'Inventory Status Report';
        $query = "SELECT VegetableName, VegetableCode, PurchasedRate, SellingPrice, 
                  (SellingPrice - PurchasedRate) as profit_margin, Status
                  FROM tblvegetables 
                  WHERE Status = 'Active'";
        $headers = ['Item', 'Code', 'Cost', 'Price', 'Margin', 'Status'];
        break;
        
    case 'category_wise':
        $title = 'Category Wise Report';
        $query = "SELECT v.CategoryName, COUNT(DISTINCT v.id) as item_count,
                  SUM(sid.quantity) as total_qty, SUM(sid.amount) as total_sales
                  FROM tblvegetables v
                  LEFT JOIN sales_invoice_details sid ON v.VegetableCode = sid.product_code
                  LEFT JOIN sales_invoices si ON sid.invoice_id = si.id
                  WHERE si.status = 'active' OR si.status IS NULL";
        $headers = ['Category', 'Items', 'Qty Sold', 'Sales'];
        break;
        
    default:
        echo "Invalid report type";
        exit();
}

// Add common filters
$conditions = [];
if ($from_date && in_array($report_type, ['bill_wise', 'daily_summary', 'pending_payments', 'profit_analysis'])) {
    $conditions[] = "si.invoice_date >= '$from_date'";
}
if ($to_date && in_array($report_type, ['bill_wise', 'daily_summary', 'pending_payments', 'profit_analysis'])) {
    $conditions[] = "si.invoice_date <= '$to_date'";
}
if ($payment_status && in_array($report_type, ['bill_wise', 'pending_payments'])) {
    $conditions[] = "si.payment_status = '$payment_status'";
}
if ($customer_id && in_array($report_type, ['bill_wise', 'customer_wise', 'pending_payments'])) {
    $conditions[] = "si.customer_id = '$customer_id'";
}

if (!empty($conditions)) {
    $query .= " AND " . implode(" AND ", $conditions);
}

// Add GROUP BY for aggregate reports
if (in_array($report_type, ['item_wise', 'customer_wise', 'daily_summary', 'category_wise'])) {
    switch($report_type) {
        case 'item_wise':
            $query .= " GROUP BY v.VegetableName ORDER BY total_amount DESC";
            break;
        case 'customer_wise':
            $query .= " GROUP BY COALESCE(c.customer_name, si.customer_name, 'Walk-in') ORDER BY total_amount DESC";
            break;
        case 'daily_summary':
            $query .= " GROUP BY si.invoice_date ORDER BY si.invoice_date DESC";
            break;
        case 'category_wise':
            $query .= " GROUP BY v.CategoryName ORDER BY total_sales DESC";
            break;
    }
} else {
    $query .= " ORDER BY si.invoice_date DESC, si.created_date DESC";
}

$result = mysqli_query($con, $query);
if ($result) {
    $data = mysqli_fetch_all($result, MYSQLI_ASSOC);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $title; ?> - Thermal Print</title>
    <!-- Stylish font for shop name -->
    <link href="https://fonts.googleapis.com/css2?family=Pacifico&display=swap" rel="stylesheet">
    <style>
        @media print {
            @page { 
                size: 80mm auto; 
                margin: 0; 
            }
            body { 
                margin: 0; 
                padding: 0; 
            }
        }
        
        body {
            font-family: 'Segoe UI', 'Arial Unicode MS', Arial, sans-serif;
            font-size: 15px;
            line-height: 1.15;
            margin: 0;
            padding: 6px 18px 6px 8px;
            width: 80mm;
            background: white;
            color: #222;
            box-sizing: border-box;
        }
        
        .receipt {
            width: 100%;
            max-width: 80mm;
            margin: 0 auto;
            padding: 0 2px 0 2px;
            padding-right: 10px;
        }
        
        .center {
            text-align: center;
        }
        
        .bold {
            font-weight: bold;
        }
        
        .header {
            text-align: center;
            border-bottom: 1px dashed #888;
            padding-bottom: 4px;
            margin-bottom: 4px;
            padding-left: 2px;
            padding-right: 2px;
        }
        
        .shop-name {
            font-size: 20px;
            font-weight: bold;
            margin-bottom: 3px;
            font-family: 'Times New Roman', Times, serif;
            color: #222;
            text-align: center;
            letter-spacing: 1px;
        }
        
        .shop-details {
            font-size: 13px;
            margin-bottom: 4px;
            font-family: 'Segoe UI', 'Arial Unicode MS', Arial, sans-serif;
        }
        
        .report-title {
            font-size: 18px;
            font-weight: bold;
            margin: 6px 0;
            text-align: center;
            color: #222;
            letter-spacing: 1px;
        }
        
        .report-info {
            margin: 4px 2px;
            font-size: 13px;
            text-align: center;
            display: flex;
            justify-content: space-between;
            flex-wrap: wrap;
            gap: 4px;
            padding: 2px 0;
        }
        
        .report-info > div {
            flex: 1;
            min-width: 60px;
            font-size: 12px;
            font-weight: bold;
        }
        
        .data-table {
            width: 100%;
            border-collapse: collapse;
            margin: 4px 2px;
            font-size: 14px;
        }
        
        .data-table th,
        .data-table td {
            padding: 2px 4px;
            text-align: left;
            border-bottom: none;
        }
        
        .data-table th {
            font-weight: bold;
            font-size: 13px;
            border-bottom: 1px dashed #888;
        }
        
        .data-table td:first-child {
            font-size: 14px;
            font-weight: 500;
        }
        
        .data-table th.number,
        .data-table td.number {
            text-align: right;
        }
        
        .total-section {
            margin-top: 4px;
            padding: 4px 2px;
            border-top: 1px dashed #888;
            font-size: 14px;
        }
        
        .total-line {
            display: flex;
            justify-content: space-between;
            margin: 1px 0;
        }
        
        .grand-total {
            font-weight: bold;
            font-size: 16px;
            border-top: 1px solid #222;
            border-bottom: 1px solid #222;
            padding: 3px 0;
            margin: 3px 0;
            background: #f7f7f7;
            letter-spacing: 1px;
        }
        
        .footer {
            text-align: center;
            margin-top: 6px;
            font-size: 13px;
            border-top: 1px dashed #888;
            padding-top: 4px;
            font-family: 'Segoe UI', 'Arial Unicode MS', Arial, sans-serif;
        }
        
        .instagram {
            display: inline-flex;
            align-items: center;
            gap: 4px;
            margin-top: 2px;
            font-size: 13px;
        }
        
        .instagram-logo {
            width: 16px;
            height: 16px;
            vertical-align: middle;
        }
        
        .english-thanks {
            font-size: 12px;
            margin: 2px 0;
            font-weight: bold;
        }
        
        .no-data {
            text-align: center;
            margin: 20px 0;
            padding: 10px;
            background: #fff3cd;
            border: 1px dashed #856404;
            border-radius: 3px;
        }
        
        .summary-cards {
            margin: 4px 0;
            border-top: 1px dashed #888;
            padding-top: 4px;
        }
        
        .summary-card {
            display: flex;
            justify-content: space-between;
            margin: 2px 0;
            padding: 2px 4px;
            background: #f8f9fa;
            border-radius: 3px;
            font-size: 13px;
        }
    </style>
</head>
<body>
    <div class="receipt">
        <!-- Header -->
        <div class="header">
            <div class="shop-name" style="font-family: 'Times New Roman', Times, serif; font-size: 24px; text-transform: uppercase;">AANA VEG CORNER</div>
            <div class="shop-details">
                Opp. Bus Stand<br>
                Andipatti - 625512<br>
                <!-- Mobile SVG logo and number -->
                <span style="display:inline-flex;align-items:center;gap:4px;">
                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" style="vertical-align:middle;">
                        <rect x="6" y="2" width="12" height="20" rx="2" stroke="#222" stroke-width="1.5" fill="none"/>
                        <rect x="10" y="18" width="4" height="2" rx="1" fill="#222"/>
                    </svg>
                    95858 29948
                </span>
            </div>
        </div>
        
        <!-- Report Title -->
        <div class="report-title"><?php echo strtoupper($title); ?></div>
        
        <!-- Report Info -->
        <div class="report-info">
            <div><strong>Date:</strong> <?php echo date('d/m/Y'); ?></div>
            <div><strong>Time:</strong> <?php echo date('H:i'); ?></div>
        </div>
        <!-- Dotted line after report info -->
        <div style="border-bottom:1px dashed #888; margin: 2px 0;"></div>
        
        <?php if ($from_date || $to_date || $payment_status): ?>
        <div class="report-info">
            <?php if ($from_date || $to_date): ?>
            <div><strong>Period:</strong> <?php echo ($from_date ?: 'Start') . ' to ' . ($to_date ?: 'End'); ?></div>
            <?php endif; ?>
            <?php if ($payment_status): ?>
            <div><strong>Status:</strong> <?php echo ucfirst($payment_status); ?></div>
            <?php endif; ?>
        </div>
        <div style="border-bottom:1px dashed #888; margin: 2px 0;"></div>
        <?php endif; ?>
        
        <!-- Data Content -->
        <?php if (!empty($data)): ?>
        
        <!-- Summary Cards -->
        <?php 
        $totals = [];
        $record_count = count($data);
        foreach ($data as $row) {
            $colIndex = 0;
            foreach ($row as $key => $value) {
                if ($colIndex < count($headers) && is_numeric($value) && $colIndex > 0) {
                    if (!isset($totals[$colIndex])) $totals[$colIndex] = 0;
                    $totals[$colIndex] += $value;
                }
                $colIndex++;
            }
        }
        ?>
        
        <div class="summary-cards">
            <div class="summary-card">
                <span><strong>Total Records:</strong></span>
                <span><?php echo $record_count; ?></span>
            </div>
            <?php foreach ($totals as $index => $total): ?>
            <?php if ($total > 0 && isset($headers[$index])): ?>
            <div class="summary-card">
                <span><strong>Total <?php echo $headers[$index]; ?>:</strong></span>
                <span>₹<?php echo number_format($total, 2); ?></span>
            </div>
            <?php endif; ?>
            <?php endforeach; ?>
        </div>
        
        <!-- Data Table -->
        <table class="data-table">
            <thead>
                <tr>
                    <?php foreach ($headers as $index => $header): ?>
                    <th class="<?php echo ($index > 0 && is_numeric(reset($data)[array_keys(reset($data))[$index]] ?? '')) ? 'number' : ''; ?>">
                        <?php echo $header; ?>
                    </th>
                    <?php endforeach; ?>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($data as $row): ?>
                <tr>
                    <?php 
                    $colIndex = 0;
                    foreach ($row as $key => $value): 
                        if ($colIndex < count($headers)):
                    ?>
                    <td class="<?php echo is_numeric($value) ? 'number' : ''; ?>">
                        <?php 
                        if (is_numeric($value) && $value != (int)$value) {
                            echo '₹' . number_format($value, 2);
                        } elseif (is_numeric($value)) {
                            echo number_format($value);
                        } else {
                            echo htmlspecialchars(substr($value, 0, 20)) . (strlen($value) > 20 ? '...' : '');
                        }
                        ?>
                    </td>
                    <?php 
                        $colIndex++;
                        endif;
                    endforeach; 
                    ?>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        
        <!-- Total Items Count -->
        <div style="text-align: right; font-size: 13px; font-weight: bold; margin: 3px 0; padding: 2px 0; border-bottom: 1px dashed #888;">
            Total Records: <?php echo count($data); ?>
        </div>
        
        <!-- Totals Section -->
        <?php if (!empty($totals)): ?>
        <div class="total-section">
            <?php foreach ($totals as $index => $total): ?>
            <?php if ($total > 0 && isset($headers[$index])): ?>
            <div class="total-line">
                <span>Total <?php echo $headers[$index]; ?>:</span>
                <span>₹<?php echo number_format($total, 2); ?></span>
            </div>
            <?php endif; ?>
            <?php endforeach; ?>
            
            <div class="total-line grand-total">
                <span>Grand Total Records:</span>
                <span><?php echo count($data); ?></span>
            </div>
        </div>
        <?php endif; ?>
        
        <?php else: ?>
        <div class="no-data">
            <div class="bold">NO DATA FOUND</div>
            <div style="font-size: 12px; margin-top: 4px;">No records match the selected criteria</div>
        </div>
        <?php endif; ?>
        
        <!-- Footer -->
        <div class="footer">
            <div class="english-thanks">Your healthy way with us!</div>
            <div class="instagram">
                <!-- Instagram SVG logo -->
                <span>
                    <svg class="instagram-logo" viewBox="0 0 24 24" fill="none">
                        <rect x="2" y="2" width="20" height="20" rx="5" stroke="#222" stroke-width="2"/>
                        <circle cx="12" cy="12" r="5" stroke="#222" stroke-width="2"/>
                        <circle cx="17" cy="7" r="1.2" fill="#222"/>
                    </svg>
                </span>
                <span>aana_vegcorner</span>
            </div>
            <div class="shop-name" style="font-size: 12px; margin-top: 5px;">அ - காய்கறி கடை</div>
            <div style="font-size: 11px; margin-top: 4px;">Generated: <?php echo date('d-m-Y H:i:s'); ?></div>
        </div>
    </div>
    
    <script>
        // Auto print on load
        window.onload = function() {
            window.print();
            // Close window after printing
            setTimeout(function() {
                window.close();
            }, 1000);
        };
    </script>
</body>
</html>
