<?php
// Quick Cash Flow Widget for Dashboard
// Include this in dashboard.php to show cash flow summary

function getQuickCashFlow($con, $days = 7) {
    $from_date = date('Y-m-d', strtotime('-' . $days . ' days'));
    $to_date = date('Y-m-d');
    
    // Get total inflow
    $inflow_query = "SELECT 
                        COALESCE(SUM(ph.cash_amount + ph.upi_amount), 0) as sales_amount
                    FROM payment_history ph
                    WHERE ph.payment_date BETWEEN ? AND ?
                    UNION ALL
                    SELECT 
                        COALESCE(SUM(total_amount), 0) as payment_amount
                    FROM payment_receipts
                    WHERE receipt_date BETWEEN ? AND ? AND status = 'active'
                    UNION ALL
                    SELECT 
                        COALESCE(SUM(amount), 0) as income_amount
                    FROM income_expense
                    WHERE type = 'income' AND transaction_date BETWEEN ? AND ?";
    
    $stmt = $con->prepare($inflow_query);
    $stmt->bind_param("ssssss", $from_date, $to_date, $from_date, $to_date, $from_date, $to_date);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $total_inflow = 0;
    while ($row = $result->fetch_assoc()) {
        $total_inflow += $row['sales_amount'] ?? $row['payment_amount'] ?? $row['income_amount'] ?? 0;
    }
    
    // Get total outflow
    $outflow_query = "SELECT 
                        COALESCE(SUM(amount), 0) as supplier_amount
                    FROM supplier_payments
                    WHERE payment_date BETWEEN ? AND ?
                    UNION ALL
                    SELECT 
                        COALESCE(SUM(amount), 0) as expense_amount
                    FROM income_expense
                    WHERE type = 'expense' AND transaction_date BETWEEN ? AND ?";
    
    $stmt = $con->prepare($outflow_query);
    $stmt->bind_param("ssss", $from_date, $to_date, $from_date, $to_date);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $total_outflow = 0;
    while ($row = $result->fetch_assoc()) {
        $total_outflow += $row['supplier_amount'] ?? $row['expense_amount'] ?? 0;
    }
    
    return [
        'inflow' => $total_inflow,
        'outflow' => $total_outflow,
        'net_flow' => $total_inflow - $total_outflow,
        'period_days' => $days
    ];
}

// Usage in dashboard.php:
// $cash_flow = getQuickCashFlow($con, 7); // Last 7 days
// $cash_flow_30 = getQuickCashFlow($con, 30); // Last 30 days
?>

<!-- Cash Flow Widget for Dashboard -->
<div class="row mb-4">
    <div class="col-xl-12">
        <div class="card shadow mb-4">
            <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between">
                <h6 class="m-0 font-weight-bold text-primary">
                    <i class="fas fa-exchange-alt mr-2"></i>Quick Cash Flow Overview
                </h6>
                <div class="dropdown no-arrow">
                    <a class="dropdown-toggle" href="#" role="button" id="dropdownMenuLink"
                        data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                        <i class="fas fa-ellipsis-v fa-sm fa-fw text-gray-400"></i>
                    </a>
                    <div class="dropdown-menu dropdown-menu-right shadow animated--fade-in"
                        aria-labelledby="dropdownMenuLink">
                        <div class="dropdown-header">Quick Actions:</div>
                        <a class="dropdown-item" href="reports/cash_flow_report.php">
                            <i class="fas fa-chart-line mr-2"></i>Detailed Report
                        </a>
                        <a class="dropdown-item" href="finance/index.php">
                            <i class="fas fa-money-bill-wave mr-2"></i>Finance Dashboard
                        </a>
                    </div>
                </div>
            </div>
            <div class="card-body">
                <div class="row">
                    <!-- 7 Days Cash Flow -->
                    <div class="col-md-6">
                        <h6 class="text-xs font-weight-bold text-primary text-uppercase mb-3">
                            Last 7 Days
                        </h6>
                        <div class="row no-gutters">
                            <div class="col-4">
                                <div class="text-center">
                                    <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                        Inflow
                                    </div>
                                    <div class="h6 mb-0 font-weight-bold text-gray-800">
                                        ₹<?php echo number_format($cash_flow['inflow'], 0); ?>
                                    </div>
                                </div>
                            </div>
                            <div class="col-4">
                                <div class="text-center">
                                    <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">
                                        Outflow
                                    </div>
                                    <div class="h6 mb-0 font-weight-bold text-gray-800">
                                        ₹<?php echo number_format($cash_flow['outflow'], 0); ?>
                                    </div>
                                </div>
                            </div>
                            <div class="col-4">
                                <div class="text-center">
                                    <div class="text-xs font-weight-bold text-info text-uppercase mb-1">
                                        Net Flow
                                    </div>
                                    <div class="h6 mb-0 font-weight-bold <?php echo ($cash_flow['net_flow'] >= 0) ? 'text-success' : 'text-danger'; ?>">
                                        ₹<?php echo number_format($cash_flow['net_flow'], 0); ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- 30 Days Cash Flow -->
                    <div class="col-md-6">
                        <h6 class="text-xs font-weight-bold text-primary text-uppercase mb-3">
                            Last 30 Days
                        </h6>
                        <div class="row no-gutters">
                            <div class="col-4">
                                <div class="text-center">
                                    <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                        Inflow
                                    </div>
                                    <div class="h6 mb-0 font-weight-bold text-gray-800">
                                        ₹<?php echo number_format($cash_flow_30['inflow'], 0); ?>
                                    </div>
                                </div>
                            </div>
                            <div class="col-4">
                                <div class="text-center">
                                    <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">
                                        Outflow
                                    </div>
                                    <div class="h6 mb-0 font-weight-bold text-gray-800">
                                        ₹<?php echo number_format($cash_flow_30['outflow'], 0); ?>
                                    </div>
                                </div>
                            </div>
                            <div class="col-4">
                                <div class="text-center">
                                    <div class="text-xs font-weight-bold text-info text-uppercase mb-1">
                                        Net Flow
                                    </div>
                                    <div class="h6 mb-0 font-weight-bold <?php echo ($cash_flow_30['net_flow'] >= 0) ? 'text-success' : 'text-danger'; ?>">
                                        ₹<?php echo number_format($cash_flow_30['net_flow'], 0); ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <hr class="my-3">
                
                <div class="text-center">
                    <a href="reports/cash_flow_report.php" class="btn btn-primary btn-sm">
                        <i class="fas fa-chart-line mr-1"></i>View Detailed Cash Flow Report
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>
