<?php
require_once '../includes/config.php';

if(!isset($_SESSION)) {
    session_start();
}

if($_SERVER['REQUEST_METHOD'] == 'POST') {
    
    // Sanitize and validate input
    $id = intval($_POST['id']);
    $type = mysqli_real_escape_string($con, $_POST['type']);
    $category = mysqli_real_escape_string($con, $_POST['category']);
    $description = mysqli_real_escape_string($con, $_POST['description']);
    $amount = floatval($_POST['amount']);
    $transaction_date = mysqli_real_escape_string($con, $_POST['transaction_date']);
    $payment_method = mysqli_real_escape_string($con, $_POST['payment_method']);
    $reference_no = isset($_POST['reference_no']) ? mysqli_real_escape_string($con, $_POST['reference_no']) : '';
    $notes = isset($_POST['notes']) ? mysqli_real_escape_string($con, $_POST['notes']) : '';
    
    // Validate required fields
    if($id <= 0 || empty($type) || empty($category) || empty($description) || $amount <= 0 || empty($transaction_date) || empty($payment_method)) {
        $_SESSION['error'] = "All required fields must be filled and amount must be greater than 0.";
        header("Location: edit_transaction.php?id=" . $id);
        exit();
    }
    
    // Validate type
    if(!in_array($type, ['income', 'expense'])) {
        $_SESSION['error'] = "Invalid transaction type.";
        header("Location: edit_transaction.php?id=" . $id);
        exit();
    }
    
    // Validate payment method
    if(!in_array($payment_method, ['cash', 'bank', 'upi', 'card'])) {
        $_SESSION['error'] = "Invalid payment method.";
        header("Location: edit_transaction.php?id=" . $id);
        exit();
    }
    
    try {
        // Check if transaction exists
        $check_query = "SELECT id FROM income_expense WHERE id = ?";
        $check_stmt = mysqli_prepare($con, $check_query);
        mysqli_stmt_bind_param($check_stmt, "i", $id);
        mysqli_stmt_execute($check_stmt);
        $check_result = mysqli_stmt_get_result($check_stmt);
        
        if(mysqli_num_rows($check_result) == 0) {
            $_SESSION['error'] = "Transaction not found.";
            header("Location: index.php");
            exit();
        }
        
        mysqli_stmt_close($check_stmt);
        
        // Update transaction in database
        $query = "UPDATE income_expense SET 
                  category = ?, 
                  description = ?, 
                  amount = ?, 
                  transaction_date = ?, 
                  payment_method = ?, 
                  reference_no = ?, 
                  notes = ?,
                  updated_at = NOW()
                  WHERE id = ?";
        
        $stmt = mysqli_prepare($con, $query);
        mysqli_stmt_bind_param($stmt, "ssdssssi", $category, $description, $amount, $transaction_date, $payment_method, $reference_no, $notes, $id);
        
        if(mysqli_stmt_execute($stmt)) {
            $_SESSION['success'] = ucfirst($type) . " transaction has been updated successfully.";
            header("Location: index.php");
        } else {
            $_SESSION['error'] = "Error updating transaction: " . mysqli_error($con);
            header("Location: edit_transaction.php?id=" . $id);
        }
        
        mysqli_stmt_close($stmt);
        
    } catch (Exception $e) {
        $_SESSION['error'] = "Database error: " . $e->getMessage();
        header("Location: edit_transaction.php?id=" . $id);
    }
    
} else {
    $_SESSION['error'] = "Invalid request method.";
    header("Location: index.php");
}

exit();
?>
