<?php
session_start();

// Check if config file exists
$configPath = __DIR__ . '/includes/config.php';
if (!file_exists($configPath)) {
    die("Error: Database configuration file not found. Please create includes/config.php");
}

include_once($configPath);

// Check if database connection exists
if (!isset($con) || !$con) {
    die("Error: Database connection failed. Please check your database configuration.");
}
// Load branding helper
include_once(__DIR__ . '/includes/brand.php');

if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$msg = "";
$userId = $_SESSION['uid'];

// Search and pagination variables
$search = isset($_GET['search']) ? mysqli_real_escape_string($con, $_GET['search']) : '';
$customerSearch = isset($_GET['customer_search']) ? mysqli_real_escape_string($con, $_GET['customer_search']) : '';
$limit = 10; // Records per page
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// Create uploads directory for pawn items (use absolute path on disk, store web-relative path)
$pawnUploadDirWeb = 'uploads/pawn_items/';
$pawnUploadDirAbs = __DIR__ . '/' . $pawnUploadDirWeb;
if (!file_exists($pawnUploadDirAbs)) {
    @mkdir($pawnUploadDirAbs, 0777, true);
}

// Function to handle file upload
function handleFileUpload($file, $uploadDirWeb, $prefix, $pawnCode = '') {
    // Check if file was uploaded
    if (!isset($file) || !is_array($file) || $file['error'] != UPLOAD_ERR_OK) {
        return '';
    }
    
    // Check file size (5MB limit)
    if ($file['size'] > 5 * 1024 * 1024) {
        return false;
    }
    
    // Check file extension
    $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif'];
    
    if (!in_array($fileExtension, $allowedExtensions)) {
        return false;
    }
    
    // Generate unique filename
    $fileName = $prefix . '_' . $pawnCode . '_' . time() . '.' . $fileExtension;
    
    // Ensure upload directory exists
    $uploadDirAbs = __DIR__ . '/' . rtrim($uploadDirWeb, '/') . '/';
    if (!file_exists($uploadDirAbs)) {
        if (!mkdir($uploadDirAbs, 0777, true)) {
            return false; // Failed to create directory
        }
    }
    
    // File paths
    $filePathWeb = rtrim($uploadDirWeb, '/') . '/' . $fileName; // what we store in DB and use in <img src>
    $filePathAbs = __DIR__ . '/' . $filePathWeb; // where we actually move the file to
    
    // Move uploaded file
    if (move_uploaded_file($file['tmp_name'], $filePathAbs)) {
        return $filePathWeb; // Return web-relative path for database storage
    }
    
    return false;
}

// Handle form submissions
if ($_POST) {
    if (isset($_POST['add_pawn'])) {
        $pawnCode = "PAWN" . date('Y') . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        $customerID = (int)$_POST['customer_id'];
        $loanAmount = (float)$_POST['loan_amount'];
        $interestRate = (float)$_POST['interest_rate'];
        $loanDate = mysqli_real_escape_string($con, $_POST['loan_date']);
        $dueDate = mysqli_real_escape_string($con, $_POST['due_date']);
        $notes = mysqli_real_escape_string($con, $_POST['notes']);
        
        // Calculate totals from items
        $totalItems = isset($_POST['item_type']) ? count($_POST['item_type']) : 0;
        $totalWeight = 0;
        $totalEstimatedValue = 0;
        $firstItemType = '';
        $firstItemDescription = '';
        
        // Calculate totals and get first item details for backward compatibility
        for ($i = 0; $i < $totalItems; $i++) {
            $totalWeight += (float)$_POST['item_weight'][$i];
            $totalEstimatedValue += (float)$_POST['estimated_value'][$i];
            if ($i == 0) {
                $firstItemType = mysqli_real_escape_string($con, $_POST['item_type'][$i]);
                $firstItemDescription = mysqli_real_escape_string($con, $_POST['item_description'][$i]);
            }
        }
        
        // Start transaction
        mysqli_begin_transaction($con);
        
        try {
            // Insert main pawn transaction with totals
            $query = "INSERT INTO tblpawntransactions (PawnCode, CustomerID, ItemType, ItemDescription, ItemWeight, ItemPurity, EstimatedValue, LoanAmount, OriginalLoanAmount, CurrentLoanAmount, InterestRate, LoanDate, DueDate, TotalItems, TotalWeight, TotalEstimatedValue, Notes, CreatedBy) 
                      VALUES ('$pawnCode', $customerID, '$firstItemType', '$firstItemDescription', $totalWeight, '', $totalEstimatedValue, $loanAmount, $loanAmount, $loanAmount, $interestRate, '$loanDate', '$dueDate', $totalItems, $totalWeight, $totalEstimatedValue, '$notes', $userId)";
            
            if (!mysqli_query($con, $query)) {
                throw new Exception("Error adding pawn transaction: " . mysqli_error($con));
            }
            
            $insertedId = mysqli_insert_id($con);
            
            // Insert individual items
            for ($i = 0; $i < $totalItems; $i++) {
                $itemType = mysqli_real_escape_string($con, $_POST['item_type'][$i]);
                $itemDescription = mysqli_real_escape_string($con, $_POST['item_description'][$i]);
                $itemWeight = (float)$_POST['item_weight'][$i];
                $itemPurity = mysqli_real_escape_string($con, $_POST['item_purity'][$i]);
                $estimatedValue = (float)$_POST['estimated_value'][$i];
                
                // Handle individual item photo upload
                $photoPath = '';
                if (isset($_FILES['item_photo']['tmp_name'][$i]) && $_FILES['item_photo']['error'][$i] == 0) {
                    $fileArray = array(
                        'tmp_name' => $_FILES['item_photo']['tmp_name'][$i],
                        'name' => $_FILES['item_photo']['name'][$i],
                        'error' => $_FILES['item_photo']['error'][$i],
                        'size' => $_FILES['item_photo']['size'][$i]
                    );
                    $photoUploadResult = handleFileUpload($fileArray, $pawnUploadDirWeb, 'item', $pawnCode . '_' . ($i + 1));
                    if ($photoUploadResult !== false) {
                        $photoPath = $photoUploadResult;
                    }
                }
                
                $photoPathEscaped = mysqli_real_escape_string($con, $photoPath);
                $itemQuery = "INSERT INTO tblpawnitems (PawnTransactionID, ItemSequence, ItemType, ItemDescription, ItemWeight, ItemPurity, EstimatedValue, ItemPhotoPath) 
                              VALUES ($insertedId, " . ($i + 1) . ", '$itemType', '$itemDescription', $itemWeight, '$itemPurity', $estimatedValue, '$photoPathEscaped')";
                
                if (!mysqli_query($con, $itemQuery)) {
                    throw new Exception("Error adding item " . ($i + 1) . ": " . mysqli_error($con));
                }
            }
            
            // Commit transaction
            mysqli_commit($con);
            
            // Redirect to pawn transaction receipt after creation
            header("Location: pawn_receipt.php?id=" . $insertedId);
            exit();
            
        } catch (Exception $e) {
            // Rollback transaction on error
            mysqli_rollback($con);
            $msg = $e->getMessage();
        }
    }
    
    if (isset($_POST['update_pawn'])) {
        $id = (int)$_POST['pawn_id'];
        $customerID = (int)$_POST['customer_id'];
        $loanAmount = (float)$_POST['loan_amount'];
        $interestRate = (float)$_POST['interest_rate'];
        $loanDate = mysqli_real_escape_string($con, $_POST['loan_date']);
        $dueDate = mysqli_real_escape_string($con, $_POST['due_date']);
        $status = mysqli_real_escape_string($con, $_POST['status']);
        $notes = mysqli_real_escape_string($con, $_POST['notes']);
        
        // Calculate totals from items
        $totalItems = isset($_POST['item_type']) ? count($_POST['item_type']) : 0;
        $totalWeight = 0;
        $totalEstimatedValue = 0;
        $firstItemType = '';
        $firstItemDescription = '';
        
        // Calculate totals and get first item details for backward compatibility
        for ($i = 0; $i < $totalItems; $i++) {
            $totalWeight += (float)$_POST['item_weight'][$i];
            $totalEstimatedValue += (float)$_POST['estimated_value'][$i];
            if ($i == 0) {
                $firstItemType = mysqli_real_escape_string($con, $_POST['item_type'][$i]);
                $firstItemDescription = mysqli_real_escape_string($con, $_POST['item_description'][$i]);
            }
        }
        
        // Start transaction
        mysqli_begin_transaction($con);
        
        try {
            // Update main pawn transaction
            $query = "UPDATE tblpawntransactions SET CustomerID=$customerID, ItemType='$firstItemType', ItemDescription='$firstItemDescription',
                      ItemWeight=$totalWeight, EstimatedValue=$totalEstimatedValue, LoanAmount=$loanAmount, 
                      InterestRate=$interestRate, LoanDate='$loanDate', DueDate='$dueDate', Status='$status', 
                      TotalItems=$totalItems, TotalWeight=$totalWeight, TotalEstimatedValue=$totalEstimatedValue, Notes='$notes' WHERE ID=$id";
            
            if (!mysqli_query($con, $query)) {
                throw new Exception("Error updating pawn transaction: " . mysqli_error($con));
            }
            
            // Preserve existing photos by sequence before deleting
            $existingPhotos = [];
            $photosRes = mysqli_query($con, "SELECT ItemPhotoPath FROM tblpawnitems WHERE PawnTransactionID=$id ORDER BY ItemSequence");
            if ($photosRes) {
                while ($p = mysqli_fetch_assoc($photosRes)) { $existingPhotos[] = $p['ItemPhotoPath']; }
            }

            // Delete existing items
            mysqli_query($con, "DELETE FROM tblpawnitems WHERE PawnTransactionID=$id");
            
            // Insert updated items
            for ($i = 0; $i < $totalItems; $i++) {
                $itemType = mysqli_real_escape_string($con, $_POST['item_type'][$i]);
                $itemDescription = mysqli_real_escape_string($con, $_POST['item_description'][$i]);
                $itemWeight = (float)$_POST['item_weight'][$i];
                $itemPurity = mysqli_real_escape_string($con, $_POST['item_purity'][$i]);
                $estimatedValue = (float)$_POST['estimated_value'][$i];
                
                // Handle individual item photo upload
                $photoPath = '';
                if (isset($_FILES['item_photo']['tmp_name'][$i]) && $_FILES['item_photo']['error'][$i] == 0) {
                    $fileArray = array(
                        'tmp_name' => $_FILES['item_photo']['tmp_name'][$i],
                        'name' => $_FILES['item_photo']['name'][$i],
                        'error' => $_FILES['item_photo']['error'][$i],
                        'size' => $_FILES['item_photo']['size'][$i]
                    );
                    $photoUploadResult = handleFileUpload($fileArray, $pawnUploadDirWeb, 'item', 'UPD' . $id . '_' . ($i + 1));
                    if ($photoUploadResult !== false) {
                        $photoPath = $photoUploadResult;
                    }
                } else {
                    // No new upload; check for existing photo from form
                    if (isset($_POST['existing_photo'][$i]) && !empty($_POST['existing_photo'][$i])) {
                        $photoPath = mysqli_real_escape_string($con, $_POST['existing_photo'][$i]);
                    } elseif (isset($existingPhotos[$i]) && !empty($existingPhotos[$i])) {
                        // Fallback to sequence-based existing photos
                        $photoPath = $existingPhotos[$i];
                    }
                }
                
                $photoPathEscaped = mysqli_real_escape_string($con, $photoPath);
                $itemQuery = "INSERT INTO tblpawnitems (PawnTransactionID, ItemSequence, ItemType, ItemDescription, ItemWeight, ItemPurity, EstimatedValue, ItemPhotoPath) 
                              VALUES ($id, " . ($i + 1) . ", '$itemType', '$itemDescription', $itemWeight, '$itemPurity', $estimatedValue, '$photoPathEscaped')";
                
                if (!mysqli_query($con, $itemQuery)) {
                    throw new Exception("Error updating item " . ($i + 1) . ": " . mysqli_error($con));
                }
            }
            
            // Commit transaction
            mysqli_commit($con);
            $msg = "Pawn transaction updated successfully!";
            
        } catch (Exception $e) {
            // Rollback transaction on error
            mysqli_rollback($con);
            $msg = $e->getMessage();
        }
    }
}

// Delete pawn transaction
if (isset($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    
    // Start transaction
    mysqli_begin_transaction($con);
    
    try {
        // Get file paths before deletion
        $fileQuery = mysqli_query($con, "SELECT ItemPhotoPath FROM tblpawnitems WHERE PawnTransactionID=$id");
        while ($fileData = mysqli_fetch_assoc($fileQuery)) {
            if (!empty($fileData['ItemPhotoPath'])) {
                $absPath = __DIR__ . '/' . ltrim($fileData['ItemPhotoPath'], '/');
                if (file_exists($absPath)) {
                    @unlink($absPath);
                }
            }
        }
        
        // Delete items first (foreign key constraint)
        mysqli_query($con, "DELETE FROM tblpawnitems WHERE PawnTransactionID=$id");
        
        // Delete main transaction
        $query = "DELETE FROM tblpawntransactions WHERE ID=$id";
        if (!mysqli_query($con, $query)) {
            throw new Exception("Error deleting pawn transaction: " . mysqli_error($con));
        }
        
        // Commit transaction
        mysqli_commit($con);
        $msg = "Pawn transaction deleted successfully!";
        
    } catch (Exception $e) {
        // Rollback transaction on error
        mysqli_rollback($con);
        $msg = $e->getMessage();
    }
}

// Get pawn transaction for editing
$editPawn = null;
$editItems = array();
if (isset($_GET['edit'])) {
    $id = (int)$_GET['edit'];
    $result = mysqli_query($con, "SELECT p.*, c.Name as CustomerName, c.MobileNumber FROM tblpawntransactions p 
                                  JOIN tblcustomers c ON p.CustomerID = c.ID WHERE p.ID=$id");
    $editPawn = mysqli_fetch_assoc($result);
    
    // Get items for this transaction
    $itemsResult = mysqli_query($con, "SELECT * FROM tblpawnitems WHERE PawnTransactionID=$id ORDER BY ItemSequence");
    while ($item = mysqli_fetch_assoc($itemsResult)) {
        $editItems[] = $item;
    }
    
    // If no items found, create one from main transaction data for backward compatibility
    if (empty($editItems) && $editPawn['ItemDescription']) {
        $editItems[] = array(
            'ItemType' => $editPawn['ItemType'],
            'ItemDescription' => $editPawn['ItemDescription'],
            'ItemWeight' => $editPawn['ItemWeight'],
            'ItemPurity' => $editPawn['ItemPurity'],
            'EstimatedValue' => $editPawn['EstimatedValue'],
            'ItemPhotoPath' => $editPawn['ItemPhotoPath']
        );
    }
}

// Build search query
$whereClause = "";
if (!empty($search)) {
    $whereClause = "WHERE (p.PawnCode LIKE '%$search%' OR c.Name LIKE '%$search%' OR c.MobileNumber LIKE '%$search%' OR p.ItemType LIKE '%$search%')";
}

// Get total count for pagination
$countQuery = "SELECT COUNT(*) as total FROM tblpawntransactions p 
               JOIN tblcustomers c ON p.CustomerID = c.ID $whereClause";
$countResult = mysqli_query($con, $countQuery);
$totalRecords = mysqli_fetch_assoc($countResult)['total'];
$totalPages = ceil($totalRecords / $limit);

// Fetch pawn transactions with search and pagination - Updated to show current loan amount
$pawnQuery = mysqli_query($con, "SELECT p.*, c.Name as CustomerName, c.MobileNumber, c.CustomerCode,
                                 COALESCE(p.CurrentLoanAmount, p.LoanAmount) as DisplayLoanAmount,
                                 COALESCE(p.TotalItems, 1) as DisplayTotalItems,
                                 COALESCE(p.TotalWeight, p.ItemWeight) as DisplayTotalWeight
                                 FROM tblpawntransactions p 
                                 JOIN tblcustomers c ON p.CustomerID = c.ID 
                                 $whereClause 
                                 ORDER BY p.CreatedDate DESC LIMIT $limit OFFSET $offset");

// Get customers for dropdown
$customersQuery = mysqli_query($con, "SELECT ID, CustomerCode, Name, MobileNumber FROM tblcustomers WHERE Status='Active' ORDER BY Name");
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?php echo page_title('Pawn Transactions - Multiple Items'); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/compact.css" rel="stylesheet">
    <style>
        /* Compact Design - Reduced padding and spacing */
        .search-container {
            background: #f8f9fc;
            padding: 10px;
            border-radius: 5px;
            margin-bottom: 10px;
        }
        
        .customer-search {
            background: #e3f2fd;
            padding: 10px;
            border-radius: 5px;
            margin-bottom: 10px;
        }
        
        .item-row {
            border-left: 3px solid #007bff;
            margin-bottom: 10px !important;
        }
        
        .item-row .card-header {
            background-color: #f8f9fc;
            padding: 8px 10px !important;
        }
        
        .item-row .card-body {
            padding: 10px !important;
        }
        
        .totals-display {
            background-color: #e9ecef;
            padding: 8px 10px;
            border-radius: 4px;
            margin-bottom: 10px;
            font-size: 0.85rem;
        }
        
        /* Compact form groups */
        .form-group {
            margin-bottom: 8px !important;
        }
        
        .form-group label {
            margin-bottom: 3px;
            font-size: 0.85rem;
            font-weight: 600;
        }
        
        /* Compact cards */
        .card {
            margin-bottom: 12px !important;
        }
        
        .card-header {
            padding: 8px 12px !important;
        }
        
        .card-header h6 {
            font-size: 0.95rem;
            margin-bottom: 0 !important;
        }
        
        .card-body {
            padding: 12px !important;
        }
        
        /* Compact table */
        .table {
            font-size: 0.85rem;
        }
        
        .table th, .table td {
            padding: 6px 8px !important;
            vertical-align: middle;
        }
        
        .table th {
            font-size: 0.8rem;
            font-weight: 600;
            white-space: nowrap;
        }
        
        /* Mobile Responsive Styles */
        @media (max-width: 768px) {
            .container-fluid {
                padding: 5px !important;
            }
            
            /* Stack form columns on mobile */
            .row > [class*="col-"] {
                padding-left: 5px;
                padding-right: 5px;
            }
            
            /* Make buttons full width on mobile */
            .search-container .btn,
            .customer-search .btn {
                margin-bottom: 5px;
            }
            
            /* Horizontal scroll for table */
            .table-responsive {
                display: block;
                width: 100%;
                overflow-x: auto;
                -webkit-overflow-scrolling: touch;
            }
            
            .table {
                min-width: 800px;
                font-size: 0.75rem;
            }
            
            .table th, .table td {
                padding: 4px 6px !important;
            }
            
            /* Compact action buttons on mobile */
            .btn-sm {
                padding: 2px 6px !important;
                font-size: 0.7rem !important;
            }
            
            .btn-sm i {
                font-size: 0.8rem;
            }
            
            /* Stack item rows on mobile */
            .item-row .row > div {
                margin-bottom: 5px;
            }
            
            /* Smaller badges on mobile */
            .badge {
                font-size: 0.7rem;
                padding: 2px 6px;
            }
            
            /* Card titles smaller on mobile */
            .card-header h6 {
                font-size: 0.85rem;
            }
            
            /* Totals display compact on mobile */
            .totals-display {
                font-size: 0.75rem;
                padding: 6px 8px;
            }
            
            .totals-display .row > div {
                margin-bottom: 3px;
            }
            
            /* Form controls compact on mobile */
            .form-control, select.form-control, textarea.form-control {
                font-size: 0.85rem;
                padding: 5px 8px !important;
                height: auto !important;
            }
            
            textarea.form-control {
                min-height: 60px;
            }
            
            /* Pagination compact on mobile */
            .pagination {
                font-size: 0.8rem;
            }
            
            .page-link {
                padding: 4px 8px;
            }
            
            /* Alert compact on mobile */
            .alert {
                padding: 8px 10px;
                font-size: 0.85rem;
                margin-bottom: 8px;
            }
        }
        
        /* Extra small devices */
        @media (max-width: 576px) {
            .table {
                min-width: 700px;
                font-size: 0.7rem;
            }
            
            .btn {
                font-size: 0.8rem;
                padding: 4px 8px;
            }
            
            /* Hide less critical columns on very small screens */
            .table .hide-xs {
                display: none;
            }
        }
        
        /* Touch-friendly action buttons */
        .action-buttons {
            display: flex;
            gap: 3px;
            flex-wrap: nowrap;
        }
        
        .action-buttons .btn {
            flex: 0 0 auto;
        }
        
        /* Better spacing for mobile form sections */
        .mobile-section-title {
            margin-top: 12px;
            margin-bottom: 8px;
            padding-bottom: 5px;
            border-bottom: 2px solid #4e73df;
        }
        
        /* Compact customer info in table */
        .customer-info {
            line-height: 1.3;
        }
        
        .customer-info strong {
            font-size: 0.9em;
        }
        
        .customer-info small {
            font-size: 0.75em;
            display: block;
        }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid">
                    
                    <?php if($msg) { ?>
                        <div class="alert alert-info"><?php echo $msg; ?></div>
                    <?php } ?>
                    
                    <!-- Add/Edit Pawn Form -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary">
                                <?php echo $editPawn ? 'Edit Pawn Transaction' : 'New Pawn Transaction'; ?>
                                <?php if (!$editPawn) { ?>
                                    <span class="badge badge-success ml-2">Multiple Items Supported</span>
                                <?php } ?>
                            </h6>
                        </div>
                        <div class="card-body">
                            <!-- Customer Search Section -->
                            <div class="customer-search">
                                <h6 class="font-weight-bold mb-2">Customer Search</h6>
                                <div class="row align-items-end">
                                    <div class="col-md-8 col-12 mb-2">
                                        <input type="text" id="customerSearch" class="form-control" 
                                               placeholder="Customer name or mobile" title="Customer name or mobile number">
                                    </div>
                                    <div class="col-md-4 col-12 mb-2">
                                        <button type="button" id="searchCustomer" class="btn btn-info btn-block" title="Search">
                                            <i class="fas fa-search"></i> Search
                                        </button>
                                    </div>
                                </div>
                                <div id="customerResults" class="mt-2"></div>
                            </div>
                            
                            <form method="POST" enctype="multipart/form-data">
                                <?php if($editPawn) { ?>
                                    <input type="hidden" name="pawn_id" value="<?php echo $editPawn['ID']; ?>">
                                <?php } ?>
                                
                                <div class="row">
                                    <div class="col-12">
                                        <div class="form-group">
                                            <label title="Customer">Customer <span class="text-danger">*</span></label>
                                            <select name="customer_id" id="selectedCustomer" class="form-control" required title="Customer">
                                                <option value="">Select Customer</option>
                                                <?php 
                                                // Reset the query for customers dropdown
                                                $customersQuery = mysqli_query($con, "SELECT ID, CustomerCode, Name, MobileNumber FROM tblcustomers WHERE Status='Active' ORDER BY Name");
                                                while($customer = mysqli_fetch_assoc($customersQuery)) { ?>
                                                    <option value="<?php echo $customer['ID']; ?>" 
                                                            <?php echo ($editPawn && $editPawn['CustomerID'] == $customer['ID']) ? 'selected' : ''; ?>>
                                                        <?php echo $customer['CustomerCode'] . ' - ' . $customer['Name'] . ' (' . $customer['MobileNumber'] . ')'; ?>
                                                    </option>
                                                <?php } ?>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Items Section -->
                                <div class="row mb-2 mt-3">
                                    <div class="col-12">
                                        <h5 class="text-primary mobile-section-title">Items Details</h5>
                                        <button type="button" id="addItemBtn" class="btn btn-success btn-sm mb-2">
                                            <i class="fas fa-plus"></i> Add Item
                                        </button>
                                    </div>
                                </div>
                                
                                <!-- Totals Display -->
                                <div class="totals-display">
                                    <div class="row">
                                        <div class="col-md-4">
                                            <strong>Total Items: <span id="totalItemsDisplay">0</span></strong>
                                        </div>
                                        <div class="col-md-4">
                                            <strong>Total Weight: <span id="totalWeightDisplay">0.000</span> grams</strong>
                                        </div>
                                        <div class="col-md-4">
                                            <strong>Total Estimated Value: ₹<span id="totalValueDisplay">0.00</span></strong>
                                        </div>
                                    </div>
                                </div>
                                
                                <div id="itemsContainer">
                                    <!-- Items will be added here dynamically -->
                                </div>
                                
                                <!-- Loan Details Section -->
                                <div class="row mb-2 mt-3">
                                    <div class="col-12">
                                        <h5 class="text-primary mobile-section-title">Loan Details</h5>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-lg-3 col-md-6 col-6">
                                        <div class="form-group">
                                            <label title="Interest Rate (%)">Interest Rate (%) <span class="text-danger">*</span></label>
                                            <input type="number" step="0.01" name="interest_rate" class="form-control" required title="Interest Rate (%)"
                                                   value="<?php echo $editPawn ? $editPawn['InterestRate'] : '2.00'; ?>">
                                        </div>
                                    </div>
                                    <div class="col-lg-3 col-md-6 col-6">
                                        <div class="form-group">
                                            <label title="Loan Amount">Loan Amount <span class="text-danger">*</span></label>
                                            <input type="number" step="0.01" name="loan_amount" class="form-control" required title="Loan Amount"
                                                   value="<?php echo $editPawn ? $editPawn['LoanAmount'] : ''; ?>">
                                        </div>
                                    </div>
                                    <div class="col-lg-3 col-md-6 col-6">
                                        <div class="form-group">
                                            <label title="Loan Date">Loan Date <span class="text-danger">*</span></label>
                                            <input type="date" name="loan_date" class="form-control" required title="Loan Date"
                                                   value="<?php echo $editPawn ? $editPawn['LoanDate'] : date('Y-m-d'); ?>">
                                        </div>
                                    </div>
                                    <div class="col-lg-3 col-md-6 col-6">
                                        <div class="form-group">
                                            <label title="Due Date">Due Date <span class="text-danger">*</span></label>
                                            <input type="date" name="due_date" class="form-control" required title="Due Date"
                                                   value="<?php echo $editPawn ? $editPawn['DueDate'] : date('Y-m-d', strtotime('+3 months')); ?>">
                                        </div>
                                    </div>
                                </div>
                                
                                <?php if($editPawn) { ?>
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label title="Status">Status</label>
                                                <select name="status" class="form-control" title="Status">
                                                    <option value="Active" <?php echo ($editPawn && $editPawn['Status'] == 'Active') ? 'selected' : ''; ?>>Active</option>
                                                    <option value="Redeemed" <?php echo ($editPawn && $editPawn['Status'] == 'Redeemed') ? 'selected' : ''; ?>>Redeemed</option>
                                                    <option value="Extended" <?php echo ($editPawn && $editPawn['Status'] == 'Extended') ? 'selected' : ''; ?>>Extended</option>
                                                    <option value="Auctioned" <?php echo ($editPawn && $editPawn['Status'] == 'Auctioned') ? 'selected' : ''; ?>>Auctioned</option>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                <?php } ?>
                                
                                <div class="form-group">
                                    <label title="Notes">Notes</label>
                                    <textarea name="notes" class="form-control" rows="3" title="Notes"><?php echo $editPawn ? htmlspecialchars($editPawn['Notes']) : ''; ?></textarea>
                                </div>
                                
                                <button type="submit" name="<?php echo $editPawn ? 'update_pawn' : 'add_pawn'; ?>" 
                                        class="btn btn-primary" title="<?php echo $editPawn ? 'Update' : 'Add'; ?>">
                                    <i class="fas fa-save"></i> <?php echo $editPawn ? 'Update' : 'Add'; ?>
                                </button>
                                <?php if($editPawn) { ?>
                                    <a href="pawn_transactions.php" class="btn btn-secondary" title="Cancel">Cancel</a>
                                <?php } ?>
                            </form>
                        </div>
                    </div>
                    
                    <!-- Search Section -->
                    <div class="search-container">
                        <form method="GET" class="row align-items-end">
                            <div class="col-lg-5 col-md-6 col-12 mb-2">
                                <input type="text" name="search" class="form-control" 
                                       placeholder="Search pawn code, customer, mobile, item" title="Search"
                                       value="<?php echo htmlspecialchars($search); ?>">
                            </div>
                            <div class="col-lg-2 col-md-3 col-6 mb-2">
                                <button type="submit" class="btn btn-primary btn-block" title="Search">
                                    <i class="fas fa-search"></i> <span class="d-none d-md-inline">Search</span>
                                </button>
                            </div>
                            <div class="col-lg-2 col-md-3 col-6 mb-2">
                                <a href="pawn_transactions.php" class="btn btn-secondary btn-block" title="Clear">
                                    <i class="fas fa-redo"></i> <span class="d-none d-md-inline">Clear</span>
                                </a>
                            </div>
                            <div class="col-lg-3 col-md-12 col-12 mb-2">
                                <a href="pawn_report_detailed.php" class="btn btn-info btn-block" title="Detailed Report">
                                    <i class="fas fa-chart-bar"></i> Report
                                </a>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Pawn Transactions List -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-2 d-flex justify-content-between align-items-center flex-wrap">
                            <h6 class="m-0 font-weight-bold text-primary">All Pawn Transactions</h6>
                            <small class="text-muted">Total: <?php echo $totalRecords; ?></small>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered table-hover">
                                    <thead class="thead-light">
                                        <tr>
                                            <th title="Number">#</th>
                                            <th title="Pawn Code">Code</th>
                                            <th title="Customer">Customer</th>
                                            <th title="Items" class="hide-xs">Items</th>
                                            <th title="Total Weight" class="hide-xs">Weight</th>
                                            <th title="Loan Amount">Loan</th>
                                            <th title="Loan Date" class="hide-xs">Date</th>
                                            <th title="Due Date">Due</th>
                                            <th title="Status">Status</th>
                                            <th title="Actions">Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php 
                                        if (mysqli_num_rows($pawnQuery) > 0) {
                                            $i = $offset + 1; 
                                            while($pawn = mysqli_fetch_assoc($pawnQuery)) { 
                                        ?>
                                            <tr>
                                                <td><?php echo $i++; ?></td>
                                                <td><small><strong><?php echo htmlspecialchars($pawn['PawnCode']); ?></strong></small></td>
                                                <td>
                                                    <div class="customer-info">
                                                        <strong><?php echo htmlspecialchars($pawn['CustomerName']); ?></strong>
                                                        <small class="text-muted d-block"><?php echo htmlspecialchars($pawn['CustomerCode']); ?></small>
                                                        <small class="text-muted d-block d-md-none"><?php echo htmlspecialchars($pawn['MobileNumber']); ?></small>
                                                    </div>
                                                </td>
                                                <td class="hide-xs">
                                                    <span class="badge badge-info"><?php echo $pawn['DisplayTotalItems']; ?></span>
                                                    <small class="d-block text-muted"><?php echo htmlspecialchars($pawn['ItemType']); ?></small>
                                                </td>
                                                <td class="hide-xs"><small><?php echo number_format($pawn['DisplayTotalWeight'], 2); ?>g</small></td>
                                                <td><strong>₹<?php echo number_format($pawn['DisplayLoanAmount'], 0); ?></strong></td>
                                                <td class="hide-xs"><small><?php echo date('d/m/y', strtotime($pawn['LoanDate'])); ?></small></td>
                                                <td><small><?php echo date('d/m/y', strtotime($pawn['DueDate'])); ?></small></td>
                                                <td>
                                                    <span class="badge badge-<?php 
                                                        echo $pawn['Status'] == 'Active' ? 'success' : 
                                                            ($pawn['Status'] == 'Redeemed' ? 'primary' : 
                                                            ($pawn['Status'] == 'Extended' ? 'warning' : 'danger')); 
                                                    ?>">
                                                        <?php echo htmlspecialchars($pawn['Status']); ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <div class="action-buttons">
                                                        <a href="pawn_transactions.php?edit=<?php echo $pawn['ID']; ?>" 
                                                           class="btn btn-sm btn-primary" title="Edit">
                                                            <i class="fas fa-edit"></i>
                                                        </a>
                                                        <a href="due_collection.php?pawn_id=<?php echo $pawn['ID']; ?>" 
                                                           class="btn btn-sm btn-success" title="Collection">
                                                            <i class="fas fa-money-bill"></i>
                                                        </a>
                                                        <a href="pawn_transactions.php?delete=<?php echo $pawn['ID']; ?>" 
                                                           onclick="return confirm('Are you sure you want to delete?')" 
                                                           class="btn btn-sm btn-danger" title="Delete">
                                                            <i class="fas fa-trash"></i>
                                                        </a>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php 
                                            }
                                        } else { 
                                        ?>
                                            <tr>
                                                <td colspan="10" class="text-center">No pawn transactions found</td>
                                            </tr>
                                        <?php } ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <!-- Pagination -->
                            <?php if ($totalPages > 1) { ?>
                                <nav aria-label="Page navigation" class="mt-2">
                                    <ul class="pagination pagination-sm justify-content-center mb-0">
                                        <?php if ($page > 1) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?php echo ($page-1); ?>&search=<?php echo urlencode($search); ?>" title="Previous">
                                                    <i class="fas fa-chevron-left"></i>
                                                </a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php 
                                        // Show limited page numbers on mobile
                                        $start = max(1, $page - 2);
                                        $end = min($totalPages, $page + 2);
                                        
                                        if ($start > 1) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=1&search=<?php echo urlencode($search); ?>">1</a>
                                            </li>
                                            <?php if ($start > 2) { ?>
                                                <li class="page-item disabled"><span class="page-link">...</span></li>
                                            <?php } ?>
                                        <?php }
                                        
                                        for ($i = $start; $i <= $end; $i++) { ?>
                                            <li class="page-item <?php echo ($i == $page) ? 'active' : ''; ?>">
                                                <a class="page-link" href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>" title="Page <?php echo $i; ?>"><?php echo $i; ?></a>
                                            </li>
                                        <?php } 
                                        
                                        if ($end < $totalPages) { 
                                            if ($end < $totalPages - 1) { ?>
                                                <li class="page-item disabled"><span class="page-link">...</span></li>
                                            <?php } ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?php echo $totalPages; ?>&search=<?php echo urlencode($search); ?>"><?php echo $totalPages; ?></a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php if ($page < $totalPages) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?php echo ($page+1); ?>&search=<?php echo urlencode($search); ?>" title="Next">
                                                    <i class="fas fa-chevron-right"></i>
                                                </a>
                                            </li>
                                        <?php } ?>
                                    </ul>
                                </nav>
                            <?php } ?>
                        </div>
                    </div>
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    
    <!-- Customer Search AJAX and Multiple Items JavaScript -->
    <script src="vendor/jquery/jquery.min.js"></script>
    <script>
    $(document).ready(function() {
        let itemCounter = 0;
        
        // Customer search functionality
        $('#searchCustomer').click(function() {
            var search = $('#customerSearch').val();
            if(search.length > 2) {
                $.ajax({
                    url: 'ajax_search_customers.php',
                    method: 'POST',
                    data: {search: search},
                    success: function(response) {
                        $('#customerResults').html(response);
                    }
                });
            }
        });
        
        $('#customerSearch').keyup(function(e) {
            if(e.keyCode == 13) {
                $('#searchCustomer').click();
            }
        });
        
        // Add first item on page load
        addItem();
        
        // Load existing items for editing
        <?php if (!empty($editItems)) { ?>
            // Clear the default item first
            $('#itemsContainer').empty();
            itemCounter = 0;
            
            <?php foreach ($editItems as $index => $item) { ?>
                addItem();
                var currentIndex = itemCounter - 1;
                $('[name="item_type[]"]').eq(currentIndex).val('<?php echo $item['ItemType']; ?>');
                $('[name="item_description[]"]').eq(currentIndex).val('<?php echo htmlspecialchars($item['ItemDescription']); ?>');
                $('[name="item_weight[]"]').eq(currentIndex).val('<?php echo $item['ItemWeight']; ?>');
                $('[name="item_purity[]"]').eq(currentIndex).val('<?php echo htmlspecialchars($item['ItemPurity']); ?>');
                $('[name="estimated_value[]"]').eq(currentIndex).val('<?php echo $item['EstimatedValue']; ?>');
                
                // Handle existing photo
                <?php if (!empty($item['ItemPhotoPath'])) { ?>
                    var photoContainer = $('.item-row').eq(currentIndex).find('.current-photo');
                    photoContainer.find('img').attr('src', '<?php echo htmlspecialchars($item['ItemPhotoPath'], ENT_QUOTES); ?>');
                    photoContainer.find('input[name="existing_photo[]"]').val('<?php echo htmlspecialchars($item['ItemPhotoPath'], ENT_QUOTES); ?>');
                    photoContainer.show();
                <?php } ?>
            <?php } ?>
            
            updateTotals();
        <?php } ?>
        
        // Add item button click
        $('#addItemBtn').click(function() {
            addItem();
        });
        
        // Function to add a new item row
        function addItem() {
            itemCounter++;
            
            var itemHtml = `
                <div class="item-row card mb-2" data-item="${itemCounter}">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h6 class="mb-0">Item #${itemCounter}</h6>
                        <button type="button" class="btn btn-danger btn-sm remove-item" ${itemCounter === 1 ? 'style="display: none;"' : ''}>
                            <i class="fas fa-times"></i><span class="d-none d-md-inline"> Remove</span>
                        </button>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6 col-12">
                                <div class="form-group">
                                    <label title="Item Type">Item Type <span class="text-danger">*</span></label>
                                    <select name="item_type[]" class="form-control" required title="Item Type">
                                        <option value="">Select Type</option>
                                        <option value="Gold">Gold</option>
                                        <option value="Silver">Silver</option>
                                        <option value="Diamond">Diamond</option>
                                        <option value="Platinum">Platinum</option>
                                        <option value="Other">Other</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6 col-12">
                                <div class="form-group">
                                    <label title="Item Photo">Item Photo</label>
                                    <input type="file" name="item_photo[]" class="form-control-file" accept="image/*" title="Item Photo">
                                    <div class="current-photo mt-1" style="display: none;">
                                        <small class="text-muted">Current:</small><br>
                                        <img src="" alt="Current Item Photo" style="max-height: 60px; border: 1px solid #ccc; padding: 2px; border-radius: 3px;">
                                        <input type="hidden" name="existing_photo[]" value="">
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-12">
                                <div class="form-group">
                                    <label title="Item Description">Description <span class="text-danger">*</span></label>
                                    <textarea name="item_description[]" class="form-control" rows="2" required title="Item Description"></textarea>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-lg-3 col-md-6 col-6">
                                <div class="form-group">
                                    <label title="Weight (grams)">Weight (g) <span class="text-danger">*</span></label>
                                    <input type="number" step="0.001" name="item_weight[]" class="form-control item-weight" required title="Weight (grams)">
                                </div>
                            </div>
                            <div class="col-lg-3 col-md-6 col-6">
                                <div class="form-group">
                                    <label title="Purity">Purity</label>
                                    <input type="text" name="item_purity[]" class="form-control" placeholder="22K, 18K" title="Purity">
                                </div>
                            </div>
                            <div class="col-lg-6 col-12">
                                <div class="form-group">
                                    <label title="Estimated Value">Estimated Value (₹) <span class="text-danger">*</span></label>
                                    <input type="number" step="0.01" name="estimated_value[]" class="form-control item-value" required title="Estimated Value">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            $('#itemsContainer').append(itemHtml);
            updateItemNumbers();
            updateRemoveButtons();
            
            // Bind events to new item
            bindItemEvents();
        }
        
        // Function to remove item
        $(document).on('click', '.remove-item', function() {
            $(this).closest('.item-row').remove();
            updateItemNumbers();
            updateRemoveButtons();
            updateTotals();
        });
        
        // Function to update item numbers
        function updateItemNumbers() {
            $('.item-row').each(function(index) {
                $(this).find('.card-header h6').text('Item #' + (index + 1));
                $(this).attr('data-item', index + 1);
            });
        }
        
        // Function to update remove buttons visibility
        function updateRemoveButtons() {
            var itemCount = $('.item-row').length;
            if (itemCount === 1) {
                $('.remove-item').hide();
            } else {
                $('.remove-item').show();
            }
        }
        
        // Function to bind events to items
        function bindItemEvents() {
            $('.item-weight, .item-value').off('input').on('input', function() {
                updateTotals();
            });
        }
        
        // Function to update totals
        function updateTotals() {
            var totalItems = $('.item-row').length;
            var totalWeight = 0;
            var totalValue = 0;
            
            $('.item-weight').each(function() {
                var weight = parseFloat($(this).val()) || 0;
                totalWeight += weight;
            });
            
            $('.item-value').each(function() {
                var value = parseFloat($(this).val()) || 0;
                totalValue += value;
            });
            
            $('#totalItemsDisplay').text(totalItems);
            $('#totalWeightDisplay').text(totalWeight.toFixed(3));
            $('#totalValueDisplay').text(totalValue.toFixed(2));
        }
        
        // Initial bind of events
        bindItemEvents();
    });
    
    function selectCustomer(id, name, mobile, code) {
        $('#selectedCustomer').val(id);
        $('#customerResults').html('<div class="alert alert-success">Selected: ' + code + ' - ' + name + ' (' + mobile + ')</div>');
    }
    </script>
    
    <?php include_once('includes/footer2.php');?>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
</body>
</html>