<?php
session_start();

// Check if config file exists
$configPath = __DIR__ . '/includes/config.php';
if (!file_exists($configPath)) {
    die("Error: Database configuration file not found. Please create includes/config.php");
}

include_once($configPath);

// Check if database connection exists
if (!isset($con) || !$con) {
    die("Error: Database connection failed. Please check your database configuration.");
}
// Load branding helper
include_once(__DIR__ . '/includes/brand.php');

if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$msg = "";
$userId = $_SESSION['uid'];

// Filter variables
$dateFrom = isset($_GET['date_from']) ? mysqli_real_escape_string($con, $_GET['date_from']) : '';
$dateTo = isset($_GET['date_to']) ? mysqli_real_escape_string($con, $_GET['date_to']) : '';
$status = isset($_GET['status']) ? mysqli_real_escape_string($con, $_GET['status']) : '';
$itemType = isset($_GET['item_type']) ? mysqli_real_escape_string($con, $_GET['item_type']) : '';
$customerSearch = isset($_GET['customer_search']) ? mysqli_real_escape_string($con, $_GET['customer_search']) : '';
$amountFrom = isset($_GET['amount_from']) ? (float)$_GET['amount_from'] : 0;
$amountTo = isset($_GET['amount_to']) ? (float)$_GET['amount_to'] : 0;
$sortBy = isset($_GET['sort_by']) ? mysqli_real_escape_string($con, $_GET['sort_by']) : 'p.CreatedDate';
$sortOrder = isset($_GET['sort_order']) ? mysqli_real_escape_string($con, $_GET['sort_order']) : 'DESC';
$showOverdue = isset($_GET['show_overdue']) ? true : false;

// Pagination
$limit = isset($_GET['records_per_page']) ? (int)$_GET['records_per_page'] : 25;
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// Handle bulk operations
if ($_POST) {
    if (isset($_POST['bulk_action']) && isset($_POST['selected_records'])) {
        $selectedIds = $_POST['selected_records'];
        $bulkAction = $_POST['bulk_action'];
        
        if (!empty($selectedIds) && is_array($selectedIds)) {
            $ids = implode(',', array_map('intval', $selectedIds));
            
            switch($bulkAction) {
                case 'activate':
                    $updateQuery = "UPDATE tblpawntransactions SET Status='Active' WHERE ID IN ($ids)";
                    if (mysqli_query($con, $updateQuery)) {
                        $msg = "Selected records activated";
                    }
                    break;
                    
                case 'redeem':
                    $updateQuery = "UPDATE tblpawntransactions SET Status='Redeemed' WHERE ID IN ($ids)";
                    if (mysqli_query($con, $updateQuery)) {
                        $msg = "Selected records marked as redeemed";
                    }
                    break;
                    
                case 'extend':
                    $updateQuery = "UPDATE tblpawntransactions SET Status='Extended', DueDate=DATE_ADD(DueDate, INTERVAL 3 MONTH) WHERE ID IN ($ids)";
                    if (mysqli_query($con, $updateQuery)) {
                        $msg = "Selected records extended";
                    }
                    break;
                    
                case 'auction':
                    $updateQuery = "UPDATE tblpawntransactions SET Status='Auctioned' WHERE ID IN ($ids)";
                    if (mysqli_query($con, $updateQuery)) {
                        $msg = "Selected records sent to auction";
                    }
                    break;
                    
                case 'delete':
                    // Delete associated files first
                    $fileQuery = mysqli_query($con, "SELECT ItemPhotoPath FROM tblpawntransactions WHERE ID IN ($ids)");
                    while($file = mysqli_fetch_assoc($fileQuery)) {
                        if ($file['ItemPhotoPath'] && file_exists($file['ItemPhotoPath'])) {
                            unlink($file['ItemPhotoPath']);
                        }
                    }
                    
                    $deleteQuery = "DELETE FROM tblpawntransactions WHERE ID IN ($ids)";
                    if (mysqli_query($con, $deleteQuery)) {
                        $msg = "Selected records deleted";
                    }
                    break;
            }
        }
    }
    
    // Individual status change
    if (isset($_POST['change_status'])) {
        $id = (int)$_POST['record_id'];
        $newStatus = mysqli_real_escape_string($con, $_POST['new_status']);
        
        $updateQuery = "UPDATE tblpawntransactions SET Status='$newStatus' WHERE ID=$id";
        if (mysqli_query($con, $updateQuery)) {
            $msg = "Status updated successfully";
        }
    }
}

// Build WHERE clause for filtering
$whereConditions = [];

if (!empty($dateFrom)) {
    $whereConditions[] = "p.LoanDate >= '$dateFrom'";
}
if (!empty($dateTo)) {
    $whereConditions[] = "p.LoanDate <= '$dateTo'";
}
if (!empty($status)) {
    $whereConditions[] = "p.Status = '$status'";
}
if (!empty($itemType)) {
    $whereConditions[] = "p.ItemType = '$itemType'";
}
if (!empty($customerSearch)) {
    $whereConditions[] = "(c.Name LIKE '%$customerSearch%' OR c.MobileNumber LIKE '%$customerSearch%' OR c.CustomerCode LIKE '%$customerSearch%')";
}
if ($amountFrom > 0) {
    $whereConditions[] = "COALESCE(p.CurrentLoanAmount, p.LoanAmount) >= $amountFrom";
}
if ($amountTo > 0) {
    $whereConditions[] = "COALESCE(p.CurrentLoanAmount, p.LoanAmount) <= $amountTo";
}
if ($showOverdue) {
    $whereConditions[] = "p.DueDate < CURDATE() AND p.Status = 'Active'";
}

$whereClause = !empty($whereConditions) ? "WHERE " . implode(" AND ", $whereConditions) : "";

// Get total count for pagination
$countQuery = "SELECT COUNT(*) as total FROM tblpawntransactions p 
               JOIN tblcustomers c ON p.CustomerID = c.ID $whereClause";
$countResult = mysqli_query($con, $countQuery);
$totalRecords = mysqli_fetch_assoc($countResult)['total'];
$totalPages = ceil($totalRecords / $limit);

// Get summary statistics
$summaryQuery = "SELECT 
    COUNT(*) as total_transactions,
    SUM(CASE WHEN p.Status = 'Active' THEN 1 ELSE 0 END) as active_count,
    SUM(CASE WHEN p.Status = 'Redeemed' THEN 1 ELSE 0 END) as redeemed_count,
    SUM(CASE WHEN p.Status = 'Extended' THEN 1 ELSE 0 END) as extended_count,
    SUM(CASE WHEN p.Status = 'Auctioned' THEN 1 ELSE 0 END) as auctioned_count,
    SUM(COALESCE(p.CurrentLoanAmount, p.LoanAmount)) as total_loan_amount,
    SUM(CASE WHEN p.Status = 'Active' THEN COALESCE(p.CurrentLoanAmount, p.LoanAmount) ELSE 0 END) as active_loan_amount,
    COUNT(CASE WHEN p.DueDate < CURDATE() AND p.Status = 'Active' THEN 1 END) as overdue_count,
    SUM(CASE WHEN p.DueDate < CURDATE() AND p.Status = 'Active' THEN COALESCE(p.CurrentLoanAmount, p.LoanAmount) ELSE 0 END) as overdue_amount
    FROM tblpawntransactions p 
    JOIN tblcustomers c ON p.CustomerID = c.ID $whereClause";
$summaryResult = mysqli_query($con, $summaryQuery);
$summary = mysqli_fetch_assoc($summaryResult);

// Fetch detailed data
$mainQuery = "SELECT p.*, c.Name as CustomerName, c.MobileNumber, c.CustomerCode,
              COALESCE(p.CurrentLoanAmount, p.LoanAmount) as DisplayLoanAmount,
              COALESCE(p.TotalWeight, p.ItemWeight) as DisplayTotalWeight,
              DATEDIFF(CURDATE(), p.DueDate) as DaysOverdue,
              (SELECT SUM(AmountPaid) FROM tblpaymentcollections WHERE PawnTransactionID = p.ID) as TotalPaid,
              (SELECT ItemPhotoPath FROM tblpawnitems WHERE PawnTransactionID = p.ID ORDER BY ItemSequence LIMIT 1) as FirstItemPhoto,
              (SELECT COUNT(*) FROM tblpawnitems WHERE PawnTransactionID = p.ID) as ItemsCount,
              (SELECT GROUP_CONCAT(CONCAT(ItemType, ': ', ItemDescription, ' (', ItemWeight, 'g)') ORDER BY ItemSequence SEPARATOR '; ') FROM tblpawnitems WHERE PawnTransactionID = p.ID) as ItemsSummary
              FROM tblpawntransactions p 
              JOIN tblcustomers c ON p.CustomerID = c.ID 
              $whereClause 
              ORDER BY $sortBy $sortOrder 
              LIMIT $limit OFFSET $offset";
$detailResult = mysqli_query($con, $mainQuery);

// Get filter options
$statusOptions = ['Active', 'Redeemed', 'Extended', 'Auctioned'];
$itemTypeOptions = ['Gold', 'Silver', 'Diamond', 'Platinum', 'Other'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?php echo page_title('Detailed Pawn Report'); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/compact.css" rel="stylesheet">
    <style>
        .filter-section {
            background: #f8f9fc;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .summary-cards {
            margin-bottom: 20px;
        }
        .summary-card {
            background: linear-gradient(45deg, #4e73df, #224abe);
            color: white;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 15px;
        }
        .overdue-row {
            background-color: #f8d7da !important;
        }
        .extended-row {
            background-color: #fff3cd !important;
        }
        .redeemed-row {
            background-color: #d1edff !important;
        }
        .bulk-actions {
            background: #e3f2fd;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .action-buttons {
            white-space: nowrap;
        }
        .table-responsive {
            max-height: 70vh;
            overflow-y: auto;
        }
        @media print {
            .no-print { display: none !important; }
            .table { font-size: 12px; }
        }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid">
                    
                    <?php if($msg) { ?>
                        <div class="alert alert-info alert-dismissible fade show">
                            <?php echo $msg; ?>
                            <button type="button" class="close" data-dismiss="alert">&times;</button>
                        </div>
                    <?php } ?>
                    
                    <!-- Summary Cards -->
                    <div class="summary-cards no-print">
                        <div class="row">
                            <div class="col-xl-3 col-md-6 mb-4">
                                <div class="summary-card">
                                    <div class="card-body">
                                        <div class="row no-gutters align-items-center">
                                            <div class="col mr-2">
                                                <div class="text-xs font-weight-bold text-white text-uppercase mb-1">Total Transactions</div>
                                                <div class="h5 mb-0 font-weight-bold text-white"><?php echo number_format($summary['total_transactions']); ?></div>
                                            </div>
                                            <div class="col-auto">
                                                <i class="fas fa-clipboard-list fa-2x text-white"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-xl-3 col-md-6 mb-4">
                                <div class="summary-card" style="background: linear-gradient(45deg, #28a745, #1e7e34);">
                                    <div class="card-body">
                                        <div class="row no-gutters align-items-center">
                                            <div class="col mr-2">
                                                <div class="text-xs font-weight-bold text-white text-uppercase mb-1">Active</div>
                                                <div class="h5 mb-0 font-weight-bold text-white"><?php echo number_format($summary['active_count']); ?></div>
                                                <div class="text-xs text-white">₹<?php echo number_format($summary['active_loan_amount'], 2); ?></div>
                                            </div>
                                            <div class="col-auto">
                                                <i class="fas fa-circle fa-2x text-white"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-xl-3 col-md-6 mb-4">
                                <div class="summary-card" style="background: linear-gradient(45deg, #dc3545, #c82333);">
                                    <div class="card-body">
                                        <div class="row no-gutters align-items-center">
                                            <div class="col mr-2">
                                                <div class="text-xs font-weight-bold text-white text-uppercase mb-1">Overdue</div>
                                                <div class="h5 mb-0 font-weight-bold text-white"><?php echo number_format($summary['overdue_count']); ?></div>
                                                <div class="text-xs text-white">₹<?php echo number_format($summary['overdue_amount'], 2); ?></div>
                                            </div>
                                            <div class="col-auto">
                                                <i class="fas fa-exclamation-triangle fa-2x text-white"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-xl-3 col-md-6 mb-4">
                                <div class="summary-card" style="background: linear-gradient(45deg, #17a2b8, #138496);">
                                    <div class="card-body">
                                        <div class="row no-gutters align-items-center">
                                            <div class="col mr-2">
                                                <div class="text-xs font-weight-bold text-white text-uppercase mb-1">Redeemed</div>
                                                <div class="h5 mb-0 font-weight-bold text-white"><?php echo number_format($summary['redeemed_count']); ?></div>
                                            </div>
                                            <div class="col-auto">
                                                <i class="fas fa-check-circle fa-2x text-white"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Advanced Filters -->
                    <div class="filter-section no-print">
                        <h6 class="font-weight-bold mb-3">Advanced Filters</h6>
                        <form method="GET" id="filterForm">
                            <div class="row">
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label>From Date</label>
                                        <input type="date" name="date_from" class="form-control" value="<?php echo $dateFrom; ?>">
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label>To Date</label>
                                        <input type="date" name="date_to" class="form-control" value="<?php echo $dateTo; ?>">
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label>Status</label>
                                        <select name="status" class="form-control">
                                            <option value="">All</option>
                                            <?php foreach($statusOptions as $opt) { ?>
                                                <option value="<?php echo $opt; ?>" <?php echo ($status == $opt) ? 'selected' : ''; ?>><?php echo $opt; ?></option>
                                            <?php } ?>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label>Item Type</label>
                                        <select name="item_type" class="form-control">
                                            <option value="">All</option>
                                            <?php foreach($itemTypeOptions as $opt) { ?>
                                                <option value="<?php echo $opt; ?>" <?php echo ($itemType == $opt) ? 'selected' : ''; ?>><?php echo $opt; ?></option>
                                            <?php } ?>
                                        </select>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label>Customer Search</label>
                                        <input type="text" name="customer_search" class="form-control" 
                                               placeholder="Name, mobile or code" value="<?php echo $customerSearch; ?>">
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label>Min Amount</label>
                                        <input type="number" name="amount_from" class="form-control" value="<?php echo $amountFrom; ?>">
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label>Max Amount</label>
                                        <input type="number" name="amount_to" class="form-control" value="<?php echo $amountTo; ?>">
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label>Sort By</label>
                                        <select name="sort_by" class="form-control">
                                            <option value="p.CreatedDate" <?php echo ($sortBy == 'p.CreatedDate') ? 'selected' : ''; ?>>Created Date</option>
                                            <option value="p.LoanDate" <?php echo ($sortBy == 'p.LoanDate') ? 'selected' : ''; ?>>Loan Date</option>
                                            <option value="p.DueDate" <?php echo ($sortBy == 'p.DueDate') ? 'selected' : ''; ?>>Due Date</option>
                                            <option value="DisplayLoanAmount" <?php echo ($sortBy == 'DisplayLoanAmount') ? 'selected' : ''; ?>>Loan Amount</option>
                                            <option value="c.Name" <?php echo ($sortBy == 'c.Name') ? 'selected' : ''; ?>>Customer Name</option>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label>Order</label>
                                        <select name="sort_order" class="form-control">
                                            <option value="DESC" <?php echo ($sortOrder == 'DESC') ? 'selected' : ''; ?>>Descending</option>
                                            <option value="ASC" <?php echo ($sortOrder == 'ASC') ? 'selected' : ''; ?>>Ascending</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label>Records per page</label>
                                        <select name="records_per_page" class="form-control">
                                            <option value="25" <?php echo ($limit == 25) ? 'selected' : ''; ?>>25</option>
                                            <option value="50" <?php echo ($limit == 50) ? 'selected' : ''; ?>>50</option>
                                            <option value="100" <?php echo ($limit == 100) ? 'selected' : ''; ?>>100</option>
                                            <option value="500" <?php echo ($limit == 500) ? 'selected' : ''; ?>>500</option>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <div class="form-check mt-4">
                                            <input type="checkbox" name="show_overdue" class="form-check-input" 
                                                   <?php echo $showOverdue ? 'checked' : ''; ?>>
                                            <label class="form-check-label">Show overdue only</label>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label>&nbsp;</label><br>
                                        <button type="submit" class="btn btn-primary">
                                            <i class="fas fa-search"></i> Filter
                                        </button>
                                        <a href="pawn_report_detailed.php" class="btn btn-secondary">
                                            <i class="fas fa-refresh"></i> Reset
                                        </a>
                                        <button type="button" onclick="window.print()" class="btn btn-info">
                                            <i class="fas fa-print"></i> Print
                                        </button>
                                        <button type="button" onclick="exportToCSV()" class="btn btn-success">
                                            <i class="fas fa-download"></i> Export CSV
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Bulk Actions -->
                    <div class="bulk-actions no-print">
                        <form method="POST" id="bulkForm">
                            <div class="row align-items-end">
                                <div class="col-md-3">
                                    <select name="bulk_action" class="form-control" required>
                                        <option value="">Select Bulk Action</option>
                                        <option value="activate">Set Active</option>
                                        <option value="redeem">Mark Redeemed</option>
                                        <option value="extend">Extend</option>
                                        <option value="auction">Send to Auction</option>
                                        <option value="delete" style="color: red;">Delete</option>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <button type="submit" class="btn btn-warning" onclick="return confirmBulkAction()">
                                        <i class="fas fa-tasks"></i> Apply to Selected
                                    </button>
                                </div>
                                <div class="col-md-6 text-right">
                                    <span id="selectedCount">0</span> records selected
                                </div>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Detailed Report Table -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3 d-flex justify-content-between align-items-center">
                            <h6 class="m-0 font-weight-bold text-primary">Detailed Pawn Transaction Report</h6>
                            <small class="text-muted">Total: <?php echo $totalRecords; ?> records</small>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered table-hover" id="dataTable">
                                    <thead class="thead-light">
                                        <tr>
                                            <th class="no-print">
                                                <input type="checkbox" id="selectAll">
                                            </th>
                                            <th>#</th>
                                            <th>Pawn Code</th>
                                            <th>Customer</th>
                                            <th>Item Details</th>
                                            <th>Weight</th>
                                            <th>Loan Amount</th>
                                            <th>Paid</th>
                                            <th>Balance</th>
                                            <th>Interest Rate</th>
                                            <th>Daily Interest</th>
                                            <th>Loan Date</th>
                                            <th>Due Date</th>
                                            <th>Status</th>
                                            <th class="no-print">Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php 
                                        if (mysqli_num_rows($detailResult) > 0) {
                                            $i = $offset + 1; 
                                            while($row = mysqli_fetch_assoc($detailResult)) { 
                                                $rowClass = '';
                                                if ($row['Status'] == 'Active' && $row['DaysOverdue'] > 0) {
                                                    $rowClass = 'overdue-row';
                                                } elseif ($row['Status'] == 'Extended') {
                                                    $rowClass = 'extended-row';
                                                } elseif ($row['Status'] == 'Redeemed') {
                                                    $rowClass = 'redeemed-row';
                                                }
                                                
                                                $balanceAmount = $row['DisplayLoanAmount'] - ($row['TotalPaid'] ?? 0);
                                
                                // Calculate daily interest
                                $dailyRate = 0.15 / 365;
                                $daysSinceLoan = max(0, (strtotime(date('Y-m-d')) - strtotime($row['LoanDate'])) / 86400);
                                $dailyInterestAmount = $row['DisplayLoanAmount'] * $dailyRate;
                                $totalInterest = $dailyInterestAmount * $daysSinceLoan;
                                ?>
                                            <tr class="<?php echo $rowClass; ?>">
                                                <td class="no-print">
                                                    <input type="checkbox" name="selected_records[]" value="<?php echo $row['ID']; ?>" class="record-checkbox">
                                                </td>
                                                <td><?php echo $i++; ?></td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($row['PawnCode']); ?></strong>
                                                    <?php if ($row['DaysOverdue'] > 0 && $row['Status'] == 'Active') { ?>
                                                        <br><span class="badge badge-danger"><?php echo $row['DaysOverdue']; ?> days overdue</span>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($row['CustomerName']); ?></strong><br>
                                                    <small><?php echo htmlspecialchars($row['CustomerCode']); ?></small><br>
                                                    <small><?php echo htmlspecialchars($row['MobileNumber']); ?></small>
                                                </td>
                                                <td>
                                                    <div class="d-flex align-items-start">
                                                        <?php if (!empty($row['FirstItemPhoto'])) { ?>
                                                            <a href="<?php echo $row['FirstItemPhoto']; ?>" target="_blank" style="margin-right:8px;">
                                                                <img src="<?php echo $row['FirstItemPhoto']; ?>" alt="Item Photo" style="width:48px;height:48px;object-fit:cover;border-radius:4px;border:1px solid #ddd;">
                                                            </a>
                                                        <?php } ?>
                                                        <div>
                                                            <strong><?php echo htmlspecialchars($row['ItemType']); ?></strong>
                                                            <?php if (!empty($row['ItemsCount']) && $row['ItemsCount'] > 1) { ?>
                                                                <span class="badge badge-info"><?php echo (int)$row['ItemsCount']; ?> items</span>
                                                            <?php } ?>
                                                            <br>
                                                            <small>
                                                                <?php 
                                                                $desc = $row['ItemsSummary'] ?: $row['ItemDescription'];
                                                                $short = mb_strimwidth($desc, 0, 60, '...');
                                                                echo htmlspecialchars($short);
                                                                ?>
                                                            </small>
                                                            <?php if ($row['ItemPurity']) { ?>
                                                                <br><span class="badge badge-info"><?php echo htmlspecialchars($row['ItemPurity']); ?></span>
                                                            <?php } ?>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td><?php echo number_format($row['DisplayTotalWeight'], 3); ?>g</td>
                                                <td>
                                                    ₹<?php echo number_format($row['DisplayLoanAmount'], 2); ?>
                                                    <?php if ($row['TotalLoanIncreases'] > 0) { ?>
                                                        <br><small class="text-info">(Inc: +₹<?php echo number_format($row['TotalLoanIncreases'], 2); ?>)</small>
                                                    <?php } ?>
                                                </td>
                                                <td>₹<?php echo number_format($row['TotalPaid'] ?? 0, 2); ?></td>
                                                <td>
                                                    <strong>₹<?php echo number_format($balanceAmount, 2); ?></strong>
                                                    <?php if ($balanceAmount <= 0) { ?>
                                                        <br><span class="badge badge-success">Fully paid</span>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <strong><?php echo number_format($row['InterestRate'], 2); ?>%</strong>
                                                    <br><small class="text-muted">Monthly</small>
                                                </td>
                                                <td>
                                                    <strong>₹<?php echo number_format($dailyInterestAmount, 2); ?></strong>
                                                    <br><small class="text-muted"><?php echo $daysSinceLoan; ?> days = ₹<?php echo number_format($totalInterest, 2); ?></small>
                                                </td>
                                                <td><?php echo date('d/m/Y', strtotime($row['LoanDate'])); ?></td>
                                                <td>
                                                    <?php echo date('d/m/Y', strtotime($row['DueDate'])); ?>
                                                    <?php if ($row['DaysOverdue'] > 0 && $row['Status'] == 'Active') { ?>
                                                        <br><span class="text-danger"><i class="fas fa-exclamation-triangle"></i> Overdue</span>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="record_id" value="<?php echo $row['ID']; ?>">
                                                        <select name="new_status" class="form-control form-control-sm" onchange="this.form.submit()">
                                                            <option value="Active" <?php echo ($row['Status'] == 'Active') ? 'selected' : ''; ?>>Active</option>
                                                            <option value="Redeemed" <?php echo ($row['Status'] == 'Redeemed') ? 'selected' : ''; ?>>Redeemed</option>
                                                            <option value="Extended" <?php echo ($row['Status'] == 'Extended') ? 'selected' : ''; ?>>Extended</option>
                                                            <option value="Auctioned" <?php echo ($row['Status'] == 'Auctioned') ? 'selected' : ''; ?>>Auctioned</option>
                                                        </select>
                                                        <input type="hidden" name="change_status" value="1">
                                                    </form>
                                                </td>
                                                <td class="no-print action-buttons">
                                                    <a href="pawn_transactions.php?edit=<?php echo $row['ID']; ?>" 
                                                       class="btn btn-sm btn-primary" title="Edit">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <a href="due_collection.php?pawn_id=<?php echo $row['ID']; ?>" 
                                                       class="btn btn-sm btn-success" title="Collection">
                                                        <i class="fas fa-money-bill"></i>
                                                    </a>
                                                    <a href="bank_pledge.php?pawn_id=<?php echo $row['ID']; ?>" 
                                                       class="btn btn-sm btn-warning" title="Bank Pledge">
                                                        <i class="fas fa-university"></i>
                                                    </a>
                                                    <?php if (!empty($row['FirstItemPhoto'])) { ?>
                                                        <a href="<?php echo $row['FirstItemPhoto']; ?>" target="_blank" 
                                                           class="btn btn-sm btn-info" title="View Photo">
                                                            <i class="fas fa-image"></i>
                                                        </a>
                                                    <?php } ?>
                                                    <a href="pawn_transactions.php?delete=<?php echo $row['ID']; ?>" 
                                                       onclick="return confirm('Are you sure you want to delete?')" 
                                                       class="btn btn-sm btn-danger" title="Delete">
                                                        <i class="fas fa-trash"></i>
                                                    </a>
                                                </td>
                                            </tr>
                                        <?php 
                                            }
                                        } else { 
                                        ?>
                                            <tr>
                                                <td colspan="13" class="text-center">No records found</td>
                                            </tr>
                                        <?php } ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <!-- Pagination -->
                            <?php if ($totalPages > 1) { ?>
                                <nav class="no-print">
                                    <ul class="pagination justify-content-center">
                                        <?php if ($page > 1) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page-1])); ?>">Previous</a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php 
                                        $start = max(1, $page - 2);
                                        $end = min($totalPages, $page + 2);
                                        for ($i = $start; $i <= $end; $i++) { 
                                        ?>
                                            <li class="page-item <?php echo ($i == $page) ? 'active' : ''; ?>">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>"><?php echo $i; ?></a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php if ($page < $totalPages) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page+1])); ?>">Next</a>
                                            </li>
                                        <?php } ?>
                                    </ul>
                                </nav>
                            <?php } ?>
                        </div>
                    </div>
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
    
    <script>
    $(document).ready(function() {
        // Select all checkbox functionality
        $('#selectAll').change(function() {
            $('.record-checkbox').prop('checked', this.checked);
            updateSelectedCount();
        });
        
        $('.record-checkbox').change(function() {
            updateSelectedCount();
            if (!this.checked) {
                $('#selectAll').prop('checked', false);
            }
        });
        
        function updateSelectedCount() {
            var count = $('.record-checkbox:checked').length;
            $('#selectedCount').text(count);
        }
        
        // Bulk form submission
        $('#bulkForm').submit(function(e) {
            var selectedCount = $('.record-checkbox:checked').length;
            if (selectedCount === 0) {
                e.preventDefault();
                alert('Please select at least one record');
                return false;
            }
            
            // Add selected checkboxes to bulk form
            $('.record-checkbox:checked').each(function() {
                $('#bulkForm').append('<input type="hidden" name="selected_records[]" value="' + $(this).val() + '">');
            });
        });
    });
    
    function confirmBulkAction() {
        var action = $('select[name="bulk_action"]').val();
        var count = $('.record-checkbox:checked').length;
        
        if (count === 0) {
            alert('Please select at least one record');
            return false;
        }
        
        var message = 'Are you sure you want to perform this action on ' + count + ' records?';
        if (action === 'delete') {
            message = 'Warning: This will permanently delete ' + count + ' records. Are you sure?';
        }
        
        return confirm(message);
    }
    
    function exportToCSV() {
        var currentUrl = window.location.href;
        var separator = currentUrl.includes('?') ? '&' : '?';
        window.open(currentUrl + separator + 'export=csv');
    }
    </script>
    
    <?php
    // Handle CSV export
    if (isset($_GET['export']) && $_GET['export'] == 'csv') {
        // Remove pagination for export
        $exportQuery = "SELECT p.*, c.Name as CustomerName, c.MobileNumber, c.CustomerCode,
                        COALESCE(p.CurrentLoanAmount, p.LoanAmount) as DisplayLoanAmount,
                        DATEDIFF(CURDATE(), p.DueDate) as DaysOverdue,
                        (SELECT SUM(AmountPaid) FROM tblpaymentcollections WHERE PawnTransactionID = p.ID) as TotalPaid
                        FROM tblpawntransactions p 
                        JOIN tblcustomers c ON p.CustomerID = c.ID 
                        $whereClause 
                        ORDER BY $sortBy $sortOrder";
        $exportResult = mysqli_query($con, $exportQuery);
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="pawn_transactions_report_' . date('Y-m-d') . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        // CSV headers
        fputcsv($output, [
            'Pawn Code', 'Customer Name', 'Customer Code', 'Mobile', 'Item Type', 'Item Description',
            'Weight', 'Purity', 'Estimated Value', 'Loan Amount', 'Total Paid', 'Balance',
            'Interest Rate (Monthly)', 'Daily Interest Rate', 'Days Since Loan', 'Daily Interest Amount', 'Total Interest',
            'Loan Date', 'Due Date', 'Days Overdue', 'Status', 'Notes'
        ]);
        
        while ($row = mysqli_fetch_assoc($exportResult)) {
            $balanceAmount = $row['DisplayLoanAmount'] - ($row['TotalPaid'] ?? 0);
            
            // Calculate daily interest details
            $dailyRate = 0.15 / 365;
            $daysSinceLoan = max(0, (strtotime(date('Y-m-d')) - strtotime($row['LoanDate'])) / 86400);
            $dailyInterestAmount = $row['DisplayLoanAmount'] * $dailyRate;
            $totalInterest = $dailyInterestAmount * $daysSinceLoan;
            
            fputcsv($output, [
                $row['PawnCode'],
                $row['CustomerName'],
                $row['CustomerCode'],
                $row['MobileNumber'],
                $row['ItemType'],
                $row['ItemDescription'],
                $row['ItemWeight'],
                $row['ItemPurity'],
                $row['EstimatedValue'],
                $row['DisplayLoanAmount'],
                $row['TotalPaid'] ?? 0,
                $balanceAmount,
                $row['InterestRate'] . '%',
                number_format((0.15/365) * 100, 4) . '%',
                $daysSinceLoan,
                number_format($dailyInterestAmount, 2),
                number_format($totalInterest, 2),
                $row['LoanDate'],
                $row['DueDate'],
                max(0, $row['DaysOverdue']),
                $row['Status'],
                $row['Notes']
            ]);
        }
        
        fclose($output);
        exit();
    }
    ?>
</body>
</html>
