<?php
session_start();

// Check if config file exists
$configPath = __DIR__ . '/includes/config.php';
if (!file_exists($configPath)) {
    die("Error: Database configuration file not found. Please create includes/config.php");
}

include_once($configPath);

// Check if database connection exists
if (!isset($con) || !$con) {
    die("Error: Database connection failed. Please check your database configuration.");
}
// Load branding helper
include_once(__DIR__ . '/includes/brand.php');

if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$msg = "";
$userId = $_SESSION['uid'];

// Get user permissions
$userQuery = mysqli_query($con, "SELECT UserType, CanEdit FROM tblusers WHERE ID = $userId");
$userData = mysqli_fetch_assoc($userQuery);
$canEdit = ($userData['UserType'] == 'admin' || $userData['CanEdit'] == 1);

// Filter variables
$dateFrom = isset($_GET['date_from']) ? mysqli_real_escape_string($con, $_GET['date_from']) : '';
$dateTo = isset($_GET['date_to']) ? mysqli_real_escape_string($con, $_GET['date_to']) : '';
$paymentMethod = isset($_GET['payment_method']) ? mysqli_real_escape_string($con, $_GET['payment_method']) : '';
$customerSearch = isset($_GET['customer_search']) ? mysqli_real_escape_string($con, $_GET['customer_search']) : '';
$authorizationType = isset($_GET['authorization_type']) ? mysqli_real_escape_string($con, $_GET['authorization_type']) : '';
$returnCondition = isset($_GET['return_condition']) ? mysqli_real_escape_string($con, $_GET['return_condition']) : '';
$amountFrom = isset($_GET['amount_from']) ? (float)$_GET['amount_from'] : 0;
$amountTo = isset($_GET['amount_to']) ? (float)$_GET['amount_to'] : 0;
$sortBy = isset($_GET['sort_by']) ? mysqli_real_escape_string($con, $_GET['sort_by']) : 'r.CreatedDate';
$sortOrder = isset($_GET['sort_order']) ? mysqli_real_escape_string($con, $_GET['sort_order']) : 'DESC';
$showHighValue = isset($_GET['show_high_value']) ? true : false;
$showWithDiscount = isset($_GET['show_with_discount']) ? true : false;

// Pagination
$limit = isset($_GET['records_per_page']) ? (int)$_GET['records_per_page'] : 25;
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// Handle bulk operations
if ($_POST) {
    if (isset($_POST['bulk_action']) && isset($_POST['selected_records']) && $canEdit) {
        $selectedIds = $_POST['selected_records'];
        $bulkAction = $_POST['bulk_action'];
        
        if (!empty($selectedIds) && is_array($selectedIds)) {
            $ids = implode(',', array_map('intval', $selectedIds));
            
            switch($bulkAction) {
                case 'mark_verified':
                    $updateQuery = "UPDATE tblredemptions SET ReturnNotes = CONCAT(COALESCE(ReturnNotes, ''), ' [Verified by Admin]') WHERE ID IN ($ids)";
                    if (mysqli_query($con, $updateQuery)) {
                        $msg = "Selected redemptions verified";
                    }
                    break;
                    
                case 'lock_edit':
                    $updateQuery = "UPDATE tblredemptions SET IsEditable = 0 WHERE ID IN ($ids)";
                    if (mysqli_query($con, $updateQuery)) {
                        $msg = "Selected redemptions locked from editing";
                    }
                    break;
                    
                case 'unlock_edit':
                    $updateQuery = "UPDATE tblredemptions SET IsEditable = 1 WHERE ID IN ($ids)";
                    if (mysqli_query($con, $updateQuery)) {
                        $msg = "Selected redemptions unlocked for editing";
                    }
                    break;
                    
                case 'cancel':
                    $updateQuery = "UPDATE tblredemptions SET Status = 'Cancelled' WHERE ID IN ($ids) AND IsEditable = 1";
                    if (mysqli_query($con, $updateQuery)) {
                        $affectedRows = mysqli_affected_rows($con);
                        $msg = "$affectedRows redemption records cancelled";
                    }
                    break;
            }
        }
    }
}

// Build WHERE clause for filtering
$whereConditions = ["r.Status = 'Completed'"]; // Show completed redemptions by default

if (!empty($dateFrom)) {
    $whereConditions[] = "r.RedemptionDate >= '$dateFrom'";
}
if (!empty($dateTo)) {
    $whereConditions[] = "r.RedemptionDate <= '$dateTo'";
}
if (!empty($paymentMethod)) {
    $whereConditions[] = "r.PaymentMethod = '$paymentMethod'";
}
if (!empty($customerSearch)) {
    $whereConditions[] = "(c.Name LIKE '%$customerSearch%' OR c.MobileNumber LIKE '%$customerSearch%' OR c.CustomerCode LIKE '%$customerSearch%' OR p.PawnCode LIKE '%$customerSearch%' OR r.RedemptionCode LIKE '%$customerSearch%')";
}
if (!empty($authorizationType)) {
    $whereConditions[] = "r.CustomerAuthorization = '$authorizationType'";
}
if (!empty($returnCondition)) {
    $whereConditions[] = "r.ReturnCondition = '$returnCondition'";
}
if ($amountFrom > 0) {
    $whereConditions[] = "r.TotalAmountDue >= $amountFrom";
}
if ($amountTo > 0) {
    $whereConditions[] = "r.TotalAmountDue <= $amountTo";
}
if ($showHighValue) {
    $whereConditions[] = "r.TotalAmountDue >= 50000";
}
if ($showWithDiscount) {
    $whereConditions[] = "r.DiscountGiven > 0";
}

$whereClause = "WHERE " . implode(" AND ", $whereConditions);

// Early CSV export before any HTML output
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    // Build export query without pagination
    $exportQuery = "SELECT r.*, p.PawnCode, p.ItemType, p.ItemDescription, p.ItemWeight, p.LoanAmount, p.LoanDate, p.DueDate,
                    COALESCE(p.CurrentLoanAmount, p.LoanAmount) as ActualLoanAmount,
                    c.Name as CustomerName, c.MobileNumber, c.CustomerCode,
                    u.FullName as ProcessedByName, eu.FullName as EditedByName,
                    DATEDIFF(r.RedemptionDate, p.DueDate) as DaysFromDue
                    FROM tblredemptions r 
                    JOIN tblpawntransactions p ON r.PawnTransactionID = p.ID
                    JOIN tblcustomers c ON r.CustomerID = c.ID 
                    LEFT JOIN tblusers u ON r.ProcessedBy = u.ID
                    LEFT JOIN tblusers eu ON r.EditedBy = eu.ID
                    $whereClause 
                    ORDER BY $sortBy $sortOrder";
    $exportResult = mysqli_query($con, $exportQuery);

    // Send headers before any output
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="jewel_redemption_report_' . date('Y-m-d') . '.csv"');

    $output = fopen('php://output', 'w');
    // CSV headers
    fputcsv($output, [
        'Redemption Code', 'Pawn Code', 'Customer Name', 'Customer Code', 'Mobile', 'Item Type', 'Item Description',
        'Item Weight', 'Loan Amount', 'Redemption Date', 'Total Amount Due', 'Total Amount Paid', 'Final Payment',
        'Discount Given', 'Payment Method', 'Reference Number', 'Receiver Name', 'Receiver Mobile', 'Relationship',
        'Authorization Type', 'Return Condition', 'Return Notes', 'Processed By', 'Edited By'
    ]);

    while ($row = mysqli_fetch_assoc($exportResult)) {
        $totalCollection = $row['TotalAmountPaid'] + $row['FinalPaymentAmount'];
        fputcsv($output, [
            $row['RedemptionCode'],
            $row['PawnCode'],
            $row['CustomerName'],
            $row['CustomerCode'],
            $row['MobileNumber'],
            $row['ItemType'],
            $row['ItemDescription'],
            $row['ItemWeight'],
            $row['ActualLoanAmount'],
            $row['RedemptionDate'],
            $row['TotalAmountDue'],
            $totalCollection,
            $row['FinalPaymentAmount'],
            $row['DiscountGiven'],
            $row['PaymentMethod'],
            $row['ReferenceNumber'],
            $row['ReceiverName'],
            $row['ReceiverMobileNumber'],
            $row['RelationshipToCustomer'],
            $row['CustomerAuthorization'],
            $row['ReturnCondition'],
            $row['ReturnNotes'],
            $row['ProcessedByName'],
            $row['EditedByName']
        ]);
    }

    fclose($output);
    exit();
}

// Get total count for pagination
$countQuery = "SELECT COUNT(*) as total FROM tblredemptions r 
               JOIN tblpawntransactions p ON r.PawnTransactionID = p.ID
               JOIN tblcustomers c ON r.CustomerID = c.ID $whereClause";
$countResult = mysqli_query($con, $countQuery);
$totalRecords = mysqli_fetch_assoc($countResult)['total'];
$totalPages = ceil($totalRecords / $limit);

// Get summary statistics
$summaryQuery = "SELECT 
    COUNT(*) as total_redemptions,
    SUM(r.TotalAmountDue) as total_amount_due,
    SUM(r.TotalAmountPaid) as total_amount_paid,
    SUM(r.FinalPaymentAmount) as total_final_payments,
    SUM(r.DiscountGiven) as total_discounts,
    AVG(r.TotalAmountDue) as average_redemption_value,
    COUNT(CASE WHEN r.PaymentMethod = 'Cash' THEN 1 END) as cash_redemptions,
    COUNT(CASE WHEN r.PaymentMethod = 'Bank Transfer' THEN 1 END) as bank_redemptions,
    COUNT(CASE WHEN r.PaymentMethod = 'UPI' THEN 1 END) as upi_redemptions,
    COUNT(CASE WHEN r.CustomerAuthorization = 'Authorized Person' THEN 1 END) as authorized_person_redemptions,
    COUNT(CASE WHEN r.TotalAmountDue >= 50000 THEN 1 END) as high_value_redemptions,
    COUNT(CASE WHEN r.DiscountGiven > 0 THEN 1 END) as discount_redemptions,
    COUNT(CASE WHEN r.IsEditable = 0 THEN 1 END) as locked_redemptions
    FROM tblredemptions r 
    JOIN tblpawntransactions p ON r.PawnTransactionID = p.ID
    JOIN tblcustomers c ON r.CustomerID = c.ID $whereClause";
$summaryResult = mysqli_query($con, $summaryQuery);
$summary = mysqli_fetch_assoc($summaryResult);

// Fetch detailed data
$mainQuery = "SELECT r.*, p.PawnCode, p.ItemType, p.ItemDescription, p.ItemWeight, p.LoanAmount, p.LoanDate, p.DueDate,
              COALESCE(p.CurrentLoanAmount, p.LoanAmount) as ActualLoanAmount,
              c.Name as CustomerName, c.MobileNumber, c.CustomerCode,
              u.FullName as ProcessedByName, eu.FullName as EditedByName,
              DATEDIFF(r.RedemptionDate, p.DueDate) as DaysFromDue
              FROM tblredemptions r 
              JOIN tblpawntransactions p ON r.PawnTransactionID = p.ID
              JOIN tblcustomers c ON r.CustomerID = c.ID 
              LEFT JOIN tblusers u ON r.ProcessedBy = u.ID
              LEFT JOIN tblusers eu ON r.EditedBy = eu.ID
              $whereClause 
              ORDER BY $sortBy $sortOrder 
              LIMIT $limit OFFSET $offset";
$detailResult = mysqli_query($con, $mainQuery);

// Get filter options
$paymentMethodOptions = ['Cash', 'Bank Transfer', 'Card', 'Cheque', 'UPI', 'No Payment'];
$authorizationOptions = ['Self', 'Authorized Person'];
$returnConditionOptions = ['Good', 'Fair', 'Damaged'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?php echo page_title('Jewel Redemption Report'); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/compact.css" rel="stylesheet">
    <style>
        .filter-section {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            padding: 25px;
            border-radius: 15px;
            margin-bottom: 25px;
            color: white;
        }
        .summary-cards {
            margin-bottom: 25px;
        }
        .summary-card {
            background: linear-gradient(45deg, #28a745, #20c997);
            color: white;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 15px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }
        .redemption-card { background: linear-gradient(45deg, #28a745, #20c997); }
        .value-card { background: linear-gradient(45deg, #007bff, #0056b3); }
        .discount-card { background: linear-gradient(45deg, #6f42c1, #5a32a3); }
        .authorized-card { background: linear-gradient(45deg, #fd7e14, #e55a00); }
        
        .bulk-actions {
            background: #e8f5e9;
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
            border-left: 4px solid #28a745;
        }
        .action-buttons {
            white-space: nowrap;
        }
        .table-responsive {
            max-height: 70vh;
            overflow-y: auto;
        }
        .verified-redemption {
            background-color: #d4edda !important;
        }
        .locked-redemption {
            background-color: #f8d7da !important;
        }
        .high-value-redemption {
            background-color: #fff3cd !important;
        }
        .discount-redemption {
            background-color: #e2e3e5 !important;
        }
        .authorized-redemption {
            background-color: #ffeaa7 !important;
        }
        @media print {
            .no-print { display: none !important; }
            .table { font-size: 11px; }
            .summary-card { 
                background: #f8f9fa !important; 
                color: #333 !important; 
                border: 1px solid #dee2e6 !important; 
            }
        }
        .quick-stats {
            display: flex;
            justify-content: space-around;
            margin-top: 15px;
        }
        .quick-stat {
            text-align: center;
            background: rgba(255,255,255,0.1);
            padding: 10px;
            border-radius: 8px;
            min-width: 120px;
        }
        .stat-value {
            font-size: 1.2em;
            font-weight: bold;
        }
        .stat-label {
            font-size: 0.8em;
            opacity: 0.9;
        }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid">
                    
                    <?php if($msg) { ?>
                        <div class="alert alert-info alert-dismissible fade show">
                            <?php echo $msg; ?>
                            <button type="button" class="close" data-dismiss="alert">&times;</button>
                        </div>
                    <?php } ?>
                    
                    <!-- Summary Cards -->
                    <div class="summary-cards no-print">
                        <div class="row">
                            <div class="col-xl-3 col-md-6 mb-4">
                                <div class="summary-card redemption-card">
                                    <div class="card-body">
                                        <div class="row no-gutters align-items-center">
                                            <div class="col mr-2">
                                                <div class="text-xs font-weight-bold text-white text-uppercase mb-1">Total Redemptions</div>
                                                <div class="h5 mb-0 font-weight-bold text-white"><?php echo number_format($summary['total_redemptions'] ?? 0); ?></div>
                                                <div class="text-xs text-white">₹<?php echo number_format($summary['total_amount_due'] ?? 0, 2); ?></div>
                                            </div>
                                            <div class="col-auto">
                                                <i class="fas fa-gem fa-2x text-white"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-xl-3 col-md-6 mb-4">
                                <div class="summary-card value-card">
                                    <div class="card-body">
                                        <div class="row no-gutters align-items-center">
                                            <div class="col mr-2">
                                                <div class="text-xs font-weight-bold text-white text-uppercase mb-1">Average Value</div>
                                                <div class="h5 mb-0 font-weight-bold text-white">₹<?php echo number_format($summary['average_redemption_value'] ?? 0, 0); ?></div>
                                                <div class="text-xs text-white">Average redemption value</div>
                                            </div>
                                            <div class="col-auto">
                                                <i class="fas fa-calculator fa-2x text-white"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-xl-3 col-md-6 mb-4">
                                <div class="summary-card discount-card">
                                    <div class="card-body">
                                        <div class="row no-gutters align-items-center">
                                            <div class="col mr-2">
                                                <div class="text-xs font-weight-bold text-white text-uppercase mb-1">Discounts Given</div>
                                                <div class="h5 mb-0 font-weight-bold text-white">₹<?php echo number_format($summary['total_discounts'] ?? 0, 0); ?></div>
                                                <div class="text-xs text-white"><?php echo number_format($summary['discount_redemptions'] ?? 0); ?> records</div>
                                            </div>
                                            <div class="col-auto">
                                                <i class="fas fa-percentage fa-2x text-white"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-xl-3 col-md-6 mb-4">
                                <div class="summary-card authorized-card">
                                    <div class="card-body">
                                        <div class="row no-gutters align-items-center">
                                            <div class="col mr-2">
                                                <div class="text-xs font-weight-bold text-white text-uppercase mb-1">Authorized Person</div>
                                                <div class="h5 mb-0 font-weight-bold text-white"><?php echo number_format($summary['authorized_person_redemptions'] ?? 0); ?></div>
                                                <div class="text-xs text-white"><?php echo ($summary['total_redemptions'] ?? 0) > 0 ? round((($summary['authorized_person_redemptions'] ?? 0)/($summary['total_redemptions'] ?? 1))*100, 1) : 0; ?>%</div>
                                            </div>
                                            <div class="col-auto">
                                                <i class="fas fa-user-check fa-2x text-white"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Quick Statistics -->
                        <div class="row">
                            <div class="col-12">
                                <div class="summary-card">
                                    <h6 class="text-white mb-3"><i class="fas fa-chart-bar"></i> Quick Statistics</h6>
                                    <div class="quick-stats">
                                        <div class="quick-stat">
                                            <div class="stat-value">₹<?php echo number_format($summary['total_final_payments'] ?? 0, 0); ?></div>
                                            <div class="stat-label">Final Payments</div>
                                        </div>
                                        <div class="quick-stat">
                                            <div class="stat-value"><?php echo number_format($summary['cash_redemptions'] ?? 0); ?></div>
                                            <div class="stat-label">Cash Redemptions</div>
                                        </div>
                                        <div class="quick-stat">
                                            <div class="stat-value"><?php echo number_format(($summary['bank_redemptions'] ?? 0) + ($summary['upi_redemptions'] ?? 0)); ?></div>
                                            <div class="stat-label">Digital Redemptions</div>
                                        </div>
                                        <div class="quick-stat">
                                            <div class="stat-value"><?php echo number_format($summary['high_value_redemptions'] ?? 0); ?></div>
                                            <div class="stat-label">High Value</div>
                                        </div>
                                        <div class="quick-stat">
                                            <div class="stat-value"><?php echo number_format($summary['locked_redemptions'] ?? 0); ?></div>
                                            <div class="stat-label">Locked</div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Advanced Filters -->
                    <div class="filter-section no-print">
                        <h5 class="mb-3"><i class="fas fa-filter"></i> Advanced Filters</h5>
                        <form method="GET" id="filterForm">
                            <div class="row">
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label class="text-white">From Date</label>
                                        <input type="date" name="date_from" class="form-control" value="<?php echo $dateFrom; ?>">
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label class="text-white">To Date</label>
                                        <input type="date" name="date_to" class="form-control" value="<?php echo $dateTo; ?>">
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label class="text-white">Payment Method</label>
                                        <select name="payment_method" class="form-control">
                                            <option value="">All Methods</option>
                                            <?php foreach($paymentMethodOptions as $method) { ?>
                                                <option value="<?php echo $method; ?>" <?php echo ($paymentMethod == $method) ? 'selected' : ''; ?>><?php echo $method; ?></option>
                                            <?php } ?>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label class="text-white">Customer Search</label>
                                        <input type="text" name="customer_search" class="form-control" 
                                               placeholder="Name, mobile or code" value="<?php echo $customerSearch; ?>">
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label class="text-white">Authorization</label>
                                        <select name="authorization_type" class="form-control">
                                            <option value="">All</option>
                                            <?php foreach($authorizationOptions as $auth) { ?>
                                                <option value="<?php echo $auth; ?>" <?php echo ($authorizationType == $auth) ? 'selected' : ''; ?>><?php echo $auth; ?></option>
                                            <?php } ?>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label class="text-white">Return Condition</label>
                                        <select name="return_condition" class="form-control">
                                            <option value="">All</option>
                                            <?php foreach($returnConditionOptions as $condition) { ?>
                                                <option value="<?php echo $condition; ?>" <?php echo ($returnCondition == $condition) ? 'selected' : ''; ?>><?php echo $condition; ?></option>
                                            <?php } ?>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label class="text-white">Min Amount</label>
                                        <input type="number" name="amount_from" class="form-control" value="<?php echo $amountFrom; ?>">
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label class="text-white">Max Amount</label>
                                        <input type="number" name="amount_to" class="form-control" value="<?php echo $amountTo; ?>">
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label class="text-white">Sort By</label>
                                        <select name="sort_by" class="form-control">
                                            <option value="r.CreatedDate" <?php echo ($sortBy == 'r.CreatedDate') ? 'selected' : ''; ?>>Created Date</option>
                                            <option value="r.RedemptionDate" <?php echo ($sortBy == 'r.RedemptionDate') ? 'selected' : ''; ?>>Redemption Date</option>
                                            <option value="r.TotalAmountDue" <?php echo ($sortBy == 'r.TotalAmountDue') ? 'selected' : ''; ?>>Total Amount</option>
                                            <option value="c.Name" <?php echo ($sortBy == 'c.Name') ? 'selected' : ''; ?>>Customer Name</option>
                                            <option value="p.PawnCode" <?php echo ($sortBy == 'p.PawnCode') ? 'selected' : ''; ?>>Pawn Code</option>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label class="text-white">Order</label>
                                        <select name="sort_order" class="form-control">
                                            <option value="DESC" <?php echo ($sortOrder == 'DESC') ? 'selected' : ''; ?>>Descending</option>
                                            <option value="ASC" <?php echo ($sortOrder == 'ASC') ? 'selected' : ''; ?>>Ascending</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label class="text-white">Records per page</label>
                                        <select name="records_per_page" class="form-control">
                                            <option value="25" <?php echo ($limit == 25) ? 'selected' : ''; ?>>25</option>
                                            <option value="50" <?php echo ($limit == 50) ? 'selected' : ''; ?>>50</option>
                                            <option value="100" <?php echo ($limit == 100) ? 'selected' : ''; ?>>100</option>
                                            <option value="500" <?php echo ($limit == 500) ? 'selected' : ''; ?>>500</option>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-check mt-4">
                                        <input type="checkbox" name="show_high_value" class="form-check-input" 
                                               <?php echo $showHighValue ? 'checked' : ''; ?>>
                                        <label class="form-check-label text-white">High Value Only</label>
                                    </div>
                                    <div class="form-check">
                                        <input type="checkbox" name="show_with_discount" class="form-check-input" 
                                               <?php echo $showWithDiscount ? 'checked' : ''; ?>>
                                        <label class="form-check-label text-white">With Discount</label>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-group mt-4">
                                        <button type="submit" class="btn btn-light">
                                            <i class="fas fa-search"></i> Filter
                                        </button>
                                        <a href="jewel_redemption_report.php" class="btn btn-outline-light ml-2">
                                            <i class="fas fa-refresh"></i> Reset
                                        </a>
                                        <button type="button" onclick="window.print()" class="btn btn-outline-light ml-2">
                                            <i class="fas fa-print"></i> Print
                                        </button>
                                        <button type="button" onclick="exportToCSV()" class="btn btn-outline-light ml-2">
                                            <i class="fas fa-download"></i> Export CSV
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Bulk Actions -->
                    <?php if ($canEdit) { ?>
                    <div class="bulk-actions no-print">
                        <form method="POST" id="bulkForm">
                            <div class="row align-items-end">
                                <div class="col-md-3">
                                    <label><strong>Bulk Action</strong></label>
                                    <select name="bulk_action" class="form-control" required>
                                        <option value="">Select Action</option>
                                        <option value="mark_verified">Mark as Verified</option>
                                        <option value="lock_edit">Lock Editing</option>
                                        <option value="unlock_edit">Unlock Editing</option>
                                        <option value="cancel" style="color: red;">Cancel</option>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <button type="submit" class="btn btn-warning" onclick="return confirmBulkAction()">
                                        <i class="fas fa-tasks"></i> Apply to Selected
                                    </button>
                                </div>
                                <div class="col-md-6 text-right">
                                    <span id="selectedCount">0</span> records selected
                                </div>
                            </div>
                        </form>
                    </div>
                    <?php } ?>
                    
                    <!-- Detailed Report Table -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3 d-flex justify-content-between align-items-center">
                            <h6 class="m-0 font-weight-bold text-primary">Detailed Jewel Redemption Report</h6>
                            <small class="text-muted">Total: <?php echo $totalRecords; ?> records</small>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered table-hover" id="dataTable">
                                    <thead class="thead-light">
                                        <tr>
                                            <?php if ($canEdit) { ?><th class="no-print"><input type="checkbox" id="selectAll"></th><?php } ?>
                                            <th>#</th>
                                            <th>Redemption Code</th>
                                            <th>Pawn Code</th>
                                            <th>Customer</th>
                                            <th>Item</th>
                                            <th>Redemption Date</th>
                                            <th>Total Amount</th>
                                            <th>Paid</th>
                                            <th>Discount</th>
                                            <th>Receiver</th>
                                            <th>Authorization</th>
                                            <th>Status</th>
                                            <th class="no-print">Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php 
                                        if (mysqli_num_rows($detailResult) > 0) {
                                            $i = $offset + 1; 
                                            while($row = mysqli_fetch_assoc($detailResult)) { 
                                                $rowClass = '';
                                                $statusBadge = 'secondary';
                                                $statusText = 'Normal';
                                                
                                                if ($row['ReturnNotes'] && strpos($row['ReturnNotes'], '[Verified by Admin]') !== false) {
                                                    $rowClass = 'verified-redemption';
                                                    $statusBadge = 'success';
                                                    $statusText = 'Verified';
                                                } elseif (!$row['IsEditable']) {
                                                    $rowClass = 'locked-redemption';
                                                    $statusBadge = 'danger';
                                                    $statusText = 'Locked';
                                                } elseif ($row['TotalAmountDue'] >= 50000) {
                                                    $rowClass = 'high-value-redemption';
                                                    $statusBadge = 'warning';
                                                    $statusText = 'High Value';
                                                } elseif ($row['DiscountGiven'] > 0) {
                                                    $rowClass = 'discount-redemption';
                                                    $statusBadge = 'info';
                                                    $statusText = 'With Discount';
                                                } elseif ($row['CustomerAuthorization'] == 'Authorized Person') {
                                                    $rowClass = 'authorized-redemption';
                                                    $statusBadge = 'primary';
                                                    $statusText = 'Authorized';
                                                }
                                                
                                                $totalCollection = $row['TotalAmountPaid'] + $row['FinalPaymentAmount'];
                                        ?>
                                            <tr class="<?php echo $rowClass; ?>">
                                                <?php if ($canEdit) { ?>
                                                <td class="no-print">
                                                    <input type="checkbox" name="selected_records[]" value="<?php echo $row['ID']; ?>" class="record-checkbox">
                                                </td>
                                                <?php } ?>
                                                <td><?php echo $i++; ?></td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($row['RedemptionCode']); ?></strong>
                                                    <?php if ($row['EditedBy']) { ?>
                                                        <br><small class="text-info"><i class="fas fa-edit"></i> Edited</small>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($row['PawnCode']); ?></strong>
                                                    <br><span class="badge badge-secondary"><?php echo htmlspecialchars($row['ItemType']); ?></span>
                                                </td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($row['CustomerName']); ?></strong><br>
                                                    <small><?php echo htmlspecialchars($row['CustomerCode']); ?></small><br>
                                                    <small><?php echo htmlspecialchars($row['MobileNumber']); ?></small>
                                                </td>
                                                <td>
                                                    <small><?php echo htmlspecialchars(substr($row['ItemDescription'], 0, 30)) . '...'; ?></small><br>
                                                    <span class="badge badge-primary"><?php echo $row['ItemWeight']; ?>g</span>
                                                    <span class="badge badge-info">₹<?php echo number_format($row['ActualLoanAmount'], 0); ?></span>
                                                </td>
                                                <td>
                                                    <?php echo date('d/m/Y', strtotime($row['RedemptionDate'])); ?>
                                                    <br><small class="text-muted"><?php echo date('H:i', strtotime($row['CreatedDate'])); ?></small>
                                                    <?php if ($row['ReceiverName'] && $row['ReceiverName'] != $row['CustomerName']) { ?>
                                                        <br><small class="text-primary">Receiver: <?php echo htmlspecialchars($row['ReceiverName']); ?></small>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <strong class="text-primary">₹<?php echo number_format($row['TotalAmountDue'], 2); ?></strong>
                                                    <br><small class="text-muted">Loan: ₹<?php echo number_format($row['ActualLoanAmount'], 0); ?></small>
                                                </td>
                                                <td>
                                                    <strong>₹<?php echo number_format($totalCollection, 2); ?></strong>
                                                    <?php if ($row['FinalPaymentAmount'] > 0) { ?>
                                                        <br><small class="text-success">Final: ₹<?php echo number_format($row['FinalPaymentAmount'], 2); ?></small>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <?php if ($row['DiscountGiven'] > 0) { ?>
                                                        <span class="text-success">₹<?php echo number_format($row['DiscountGiven'], 2); ?></span>
                                                    <?php } else { ?>
                                                        <span class="text-muted">-</span>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <?php if ($row['ReceiverName']) { ?>
                                                        <strong><?php echo htmlspecialchars($row['ReceiverName']); ?></strong>
                                                        <?php if ($row['ReceiverMobileNumber']) { ?>
                                                            <br><small><?php echo htmlspecialchars($row['ReceiverMobileNumber']); ?></small>
                                                        <?php } ?>
                                                        <?php if ($row['RelationshipToCustomer']) { ?>
                                                            <br><small class="text-info"><?php echo htmlspecialchars($row['RelationshipToCustomer']); ?></small>
                                                        <?php } ?>
                                                    <?php } else { ?>
                                                        <span class="text-muted">-</span>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <span class="badge badge-<?php 
                                                        echo $row['CustomerAuthorization'] == 'Self' ? 'success' : 'warning'; 
                                                    ?>">
                                                        <?php echo htmlspecialchars($row['CustomerAuthorization']); ?>
                                                    </span>
                                                    <?php if ($row['PaymentMethod'] && $row['PaymentMethod'] != 'No Payment') { ?>
                                                        <br><span class="badge badge-info"><?php echo htmlspecialchars($row['PaymentMethod']); ?></span>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <span class="badge badge-<?php echo $statusBadge; ?>"><?php echo $statusText; ?></span>
                                                    <?php if ($row['ReturnCondition']) { ?>
                                                        <br><span class="badge badge-<?php 
                                                            echo $row['ReturnCondition'] == 'Good' ? 'success' : 
                                                                ($row['ReturnCondition'] == 'Fair' ? 'warning' : 'danger'); 
                                                        ?>"><?php echo htmlspecialchars($row['ReturnCondition']); ?></span>
                                                    <?php } ?>
                                                </td>
                                                <td class="no-print action-buttons">
                                                    <a href="redemption_receipt.php?id=<?php echo $row['ID']; ?>" 
                                                       class="btn btn-sm btn-info" title="Receipt">
                                                        <i class="fas fa-receipt"></i>
                                                    </a>
                                                    <a href="jewel_redemption.php?pawn_id=<?php echo $row['PawnTransactionID']; ?>" 
                                                       class="btn btn-sm btn-primary" title="Redemption Page">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                    <?php if ($canEdit && $row['IsEditable']) { ?>
                                                        <button class="btn btn-sm btn-warning" title="Edit"
                                                                onclick="editRedemption(<?php echo $row['ID']; ?>)">
                                                            <i class="fas fa-edit"></i>
                                                        </button>
                                                    <?php } ?>
                                                </td>
                                            </tr>
                                        <?php 
                                            }
                                        } else { 
                                        ?>
                                            <tr>
                                                <td colspan="<?php echo $canEdit ? '14' : '13'; ?>" class="text-center">No records found</td>
                                            </tr>
                                        <?php } ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <!-- Pagination -->
                            <?php if ($totalPages > 1) { ?>
                                <nav class="no-print">
                                    <ul class="pagination justify-content-center">
                                        <?php if ($page > 1) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page-1])); ?>">Previous</a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php 
                                        $start = max(1, $page - 2);
                                        $end = min($totalPages, $page + 2);
                                        for ($i = $start; $i <= $end; $i++) { 
                                        ?>
                                            <li class="page-item <?php echo ($i == $page) ? 'active' : ''; ?>">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>"><?php echo $i; ?></a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php if ($page < $totalPages) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page+1])); ?>">Next</a>
                                            </li>
                                        <?php } ?>
                                    </ul>
                                </nav>
                            <?php } ?>
                        </div>
                    </div>
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
    
    <script>
    $(document).ready(function() {
        // Select all checkbox functionality
        $('#selectAll').change(function() {
            $('.record-checkbox').prop('checked', this.checked);
            updateSelectedCount();
        });
        
        $('.record-checkbox').change(function() {
            updateSelectedCount();
            if (!this.checked) {
                $('#selectAll').prop('checked', false);
            }
        });
        
        function updateSelectedCount() {
            var count = $('.record-checkbox:checked').length;
            $('#selectedCount').text(count);
        }
        
        // Bulk form submission
        $('#bulkForm').submit(function(e) {
            var selectedCount = $('.record-checkbox:checked').length;
            if (selectedCount === 0) {
                e.preventDefault();
                alert('Please select at least one record');
                return false;
            }
            
            // Add selected checkboxes to bulk form
            $('.record-checkbox:checked').each(function() {
                $('#bulkForm').append('<input type="hidden" name="selected_records[]" value="' + $(this).val() + '">');
            });
        });
    });
    
    function confirmBulkAction() {
        var action = $('select[name="bulk_action"]').val();
        var count = $('.record-checkbox:checked').length;
        
        if (count === 0) {
            alert('Please select at least one record');
            return false;
        }
        
        var message = 'Are you sure you want to perform this action on ' + count + ' records?';
        if (action === 'cancel') {
            message = 'Warning: This will cancel ' + count + ' redemptions. Are you sure?';
        }
        
        return confirm(message);
    }
    
    function exportToCSV() {
        var currentUrl = window.location.href;
        var separator = currentUrl.includes('?') ? '&' : '?';
        window.open(currentUrl + separator + 'export=csv');
    }
    
    function editRedemption(redemptionId) {
        // Redirect to edit page or open modal
        window.location.href = 'edit_redemption.php?id=' + redemptionId;
    }
    </script>
    
</body>
</html>
