<?php
session_start();

// Check if config file exists
$configPath = __DIR__ . '/includes/config.php';
if (!file_exists($configPath)) {
    die("Error: Database configuration file not found. Please create includes/config.php");
}

include_once($configPath);
include_once(__DIR__ . '/includes/brand.php');

// Check if database connection exists
if (!isset($con) || !$con) {
    die("Error: Database connection failed. Please check your database configuration.");
}

if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$msg = "";
$userId = $_SESSION['uid'];

// Function to upload files
function uploadFile($fileArray, $uploadDir, $allowedTypes = ['jpg', 'jpeg', 'png', 'pdf']) {
    if (!isset($fileArray['name']) || empty($fileArray['name'])) {
        return null;
    }
    
    $fileName = $fileArray['name'];
    $fileTemp = $fileArray['tmp_name'];
    $fileError = $fileArray['error'];
    
    if ($fileError !== UPLOAD_ERR_OK) {
        return false;
    }
    
    $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
    if (!in_array($fileExt, $allowedTypes)) {
        return false;
    }
    
    // Create upload directory if it doesn't exist
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0777, true);
    }
    
    $newFileName = uniqid() . '_' . time() . '.' . $fileExt;
    $uploadPath = $uploadDir . $newFileName;
    
    if (move_uploaded_file($fileTemp, $uploadPath)) {
        return $uploadPath;
    }
    
    return false;
}

// Handle redemption processing
if ($_POST && isset($_POST['process_redemption'])) {
    $pawnTransactionID = (int)$_POST['pawn_transaction_id'];
    $customerID = (int)$_POST['customer_id'];
    $redemptionDate = mysqli_real_escape_string($con, $_POST['redemption_date']);
    $finalPaymentAmount = (float)($_POST['final_payment_amount'] ?: 0);
    $discountGiven = (float)($_POST['discount_given'] ?: 0);
    $paymentMethod = mysqli_real_escape_string($con, $_POST['payment_method']);
    $referenceNumber = mysqli_real_escape_string($con, $_POST['reference_number']);
    
    // Receiver details
    $receiverName = mysqli_real_escape_string($con, $_POST['receiver_name']);
    $receiverMobile = mysqli_real_escape_string($con, $_POST['receiver_mobile']);
    $receiverIDType = mysqli_real_escape_string($con, $_POST['receiver_id_type']);
    $receiverIDNumber = mysqli_real_escape_string($con, $_POST['receiver_id_number']);
    $relationship = mysqli_real_escape_string($con, $_POST['relationship']);
    $customerAuth = mysqli_real_escape_string($con, $_POST['customer_authorization']);
    
    // Item details
    $itemsDescription = mysqli_real_escape_string($con, $_POST['items_description']);
    $returnCondition = mysqli_real_escape_string($con, $_POST['return_condition']);
    $returnNotes = mysqli_real_escape_string($con, $_POST['return_notes']);
    
    // File upload directories
    $uploadBaseDir = 'uploads/redemptions/';
    $idProofDir = $uploadBaseDir . 'id_proofs/';
    $photoDir = $uploadBaseDir . 'photos/';
    $signatureDir = $uploadBaseDir . 'signatures/';
    $jewelPhotoDir = $uploadBaseDir . 'jewel_photos/';
    $authLetterDir = $uploadBaseDir . 'authorization_letters/';
    
    // Upload files (all optional now)
    $receiverIDPath = uploadFile($_FILES['receiver_id_proof'], $idProofDir, ['jpg', 'jpeg', 'png', 'pdf']);
    $receiverPhotoPath = uploadFile($_FILES['receiver_photo'], $photoDir, ['jpg', 'jpeg', 'png']);
    $receiverSignaturePath = uploadFile($_FILES['receiver_signature'], $signatureDir, ['jpg', 'jpeg', 'png']);
    $jewelPhotoPath = uploadFile($_FILES['jewel_photo'], $jewelPhotoDir, ['jpg', 'jpeg', 'png']);
    $authLetterPath = uploadFile($_FILES['authorization_letter'], $authLetterDir, ['jpg', 'jpeg', 'png', 'pdf']);
    
    // Get pawn transaction details
    $pawnQuery = mysqli_query($con, "SELECT *, COALESCE(CurrentLoanAmount, LoanAmount) as ActualLoanAmount FROM tblpawntransactions WHERE ID = $pawnTransactionID");
    $pawnData = mysqli_fetch_assoc($pawnQuery);
    
    if ($pawnData) {
        // Calculate total amount due
        $dueCalc = calculateDueAmount($pawnData['LoanDate'], $pawnData['DueDate'], $pawnData['ActualLoanAmount'], $pawnData['InterestRate'], $redemptionDate);
        
        // Get total payments made
        $paidQuery = mysqli_query($con, "SELECT SUM(AmountPaid + DiscountAmount) as TotalPaid FROM tblpaymentcollections WHERE PawnTransactionID = $pawnTransactionID");
        $paidData = mysqli_fetch_assoc($paidQuery);
        $totalPaid = $paidData['TotalPaid'] ?: 0;
        
        $totalAmountDue = $dueCalc['totalDue'];
        $remainingBalance = $totalAmountDue - $totalPaid;
        
        // Generate redemption code
        $redemptionCode = "RED" . date('Y') . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        
        // Insert redemption record (handle NULL values for optional fields)
        $insertQuery = "INSERT INTO tblredemptions (
            RedemptionCode, PawnTransactionID, CustomerID, RedemptionDate, 
            TotalAmountDue, TotalAmountPaid, FinalPaymentAmount, DiscountGiven,
            PaymentMethod, ReferenceNumber,
            ReceiverName, ReceiverMobileNumber, ReceiverIDProofType, ReceiverIDProofNumber,
            ReceiverIDProofPath, ReceiverPhotoPath, ReceiverSignaturePath, RelationshipToCustomer,
            ItemsReturnedDescription, ReturnedItemPhotoPath, ReturnCondition, ReturnNotes,
            CustomerAuthorization, AuthorizationLetter, ProcessedBy
        ) VALUES (
            '$redemptionCode', $pawnTransactionID, $customerID, '$redemptionDate',
            $totalAmountDue, $totalPaid, $finalPaymentAmount, $discountGiven,
            '$paymentMethod', '$referenceNumber',
            '$receiverName', '$receiverMobile', '$receiverIDType', '$receiverIDNumber',
            " . ($receiverIDPath ? "'$receiverIDPath'" : "NULL") . ", 
            " . ($receiverPhotoPath ? "'$receiverPhotoPath'" : "NULL") . ", 
            " . ($receiverSignaturePath ? "'$receiverSignaturePath'" : "NULL") . ", '$relationship',
            '$itemsDescription', " . ($jewelPhotoPath ? "'$jewelPhotoPath'" : "NULL") . ", '$returnCondition', '$returnNotes',
            '$customerAuth', " . ($authLetterPath ? "'$authLetterPath'" : "NULL") . ", $userId
        )";
        
        if (mysqli_query($con, $insertQuery)) {
            $redemptionId = mysqli_insert_id($con);
            
            // Update pawn transaction status
            mysqli_query($con, "UPDATE tblpawntransactions SET Status = 'Redeemed' WHERE ID = $pawnTransactionID");
            
            // Record final payment if any
            if ($finalPaymentAmount > 0) {
                $collectionCode = "COL" . date('Y') . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
                $finalBalanceAmount = max(0, $remainingBalance - $finalPaymentAmount - $discountGiven);
                
                $paymentQuery = "INSERT INTO tblpaymentcollections (
                    CollectionCode, PawnTransactionID, CustomerID, CollectionDate,
                    DaysOverdue, InterestDays, InterestAmount, DailyInterestAmount, MonthlyInterestAmount,
                    TotalAmountDue, AmountPaid, DiscountAmount, BalanceAmount,
                    PaymentMethod, ReferenceNumber, Notes, CreatedBy
                ) VALUES (
                    '$collectionCode', $pawnTransactionID, $customerID, '$redemptionDate',
                    0, 0, 0, 0, 0,
                    $remainingBalance, $finalPaymentAmount, $discountGiven, $finalBalanceAmount,
                    '$paymentMethod', '$referenceNumber', 'Final redemption payment', $userId
                )";
                
                mysqli_query($con, $paymentQuery);
            }
            
            $msg = "Jewel redeemed successfully!";
            header("Location: redemption_receipt.php?id=$redemptionId");
            exit();
        } else {
            $msg = "Error processing redemption: " . mysqli_error($con);
        }
    }
}

// Get pawn transaction details for redemption
$pawnTransaction = null;
if (isset($_GET['pawn_id'])) {
    $pawnId = (int)$_GET['pawn_id'];
    
    $pawnQuery = mysqli_query($con, "SELECT p.*, c.Name, c.MobileNumber, c.CustomerCode, c.Address,
                                    COALESCE(p.CurrentLoanAmount, p.LoanAmount) as ActualLoanAmount
                                    FROM tblpawntransactions p 
                                    JOIN tblcustomers c ON p.CustomerID = c.ID 
                                    WHERE p.ID = $pawnId AND p.Status = 'Active'");
    $pawnTransaction = mysqli_fetch_assoc($pawnQuery);
    
    if ($pawnTransaction) {
        // Calculate due amount
        $dueCalc = calculateDueAmount($pawnTransaction['LoanDate'], $pawnTransaction['DueDate'], $pawnTransaction['ActualLoanAmount'], $pawnTransaction['InterestRate']);
        
        // Get total payments made
        $paidQuery = mysqli_query($con, "SELECT SUM(AmountPaid + DiscountAmount) as TotalPaid FROM tblpaymentcollections WHERE PawnTransactionID = $pawnId");
        $paidData = mysqli_fetch_assoc($paidQuery);
        $totalPaid = $paidData['TotalPaid'] ?: 0;
        
        $pawnTransaction['TotalDue'] = $dueCalc['totalDue'];
        $pawnTransaction['TotalPaid'] = $totalPaid;
        $pawnTransaction['RemainingBalance'] = $dueCalc['totalDue'] - $totalPaid;
        $pawnTransaction['DaysOverdue'] = $dueCalc['daysOverdue'];
    }
}

// Enhanced function to calculate interest and due amount with proper rounding
function calculateDueAmount($loanDate, $dueDate, $loanAmount, $interestRate, $collectionDate = null) {
    if (!$collectionDate) {
        $collectionDate = date('Y-m-d');
    }
    
    $loanDateTime = new DateTime($loanDate);
    $dueDatetime = new DateTime($dueDate);
    $collectionDateTime = new DateTime($collectionDate);
    
    $totalDays = $loanDateTime->diff($collectionDateTime)->days;
    $daysOverdue = max(0, $collectionDateTime->diff($dueDatetime)->days);
    if ($collectionDateTime < $dueDatetime) {
        $daysOverdue = 0;
    }
    
    // 15% per annum daily interest with proper rounding
    $annualRate = 0.15;
    $dailyRate = $annualRate / 365;
    $dailyInterestAmount = round($loanAmount * $dailyRate, 2); // Round daily interest
    $actualInterestAmount = round($dailyInterestAmount * $totalDays, 2); // Round total interest
    $monthlyInterestAmount = round($loanAmount * ($annualRate / 12), 2); // Round monthly interest
    
    $totalDue = round($loanAmount + $actualInterestAmount, 2); // Round total due
    
    return [
        'totalDays' => $totalDays,
        'daysOverdue' => $daysOverdue,
        'dailyInterestAmount' => $dailyInterestAmount, // Rounded daily interest
        'monthlyInterestAmount' => $monthlyInterestAmount, // Rounded monthly interest
        'interestAmount' => $actualInterestAmount, // Rounded total interest
        'totalDue' => $totalDue // Rounded total due
    ];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?php echo page_title('Jewel Redemption'); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/compact.css" rel="stylesheet">
    <style>
        .redemption-header {
            background: linear-gradient(45deg, #28a745, #20c997);
            color: white;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .form-section {
            background: #f8f9fc;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .optional {
            color: #6c757d;
            font-size: 0.8em;
        }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid">
                    
                    <?php if($msg) { ?>
                        <div class="alert alert-info"><?php echo $msg; ?></div>
                    <?php } ?>
                    
                    <?php if (!$pawnTransaction) { ?>
                        <div class="alert alert-warning">
                            <h5>No valid transaction for redemption</h5>
                            <p>Please select an active transaction from the <a href="due_collection.php">Due Collection</a> page.</p>
                        </div>
                    <?php } else { ?>
                        
                        <!-- Redemption Header -->
                        <div class="redemption-header">
                            <div class="row">
                                <div class="col-md-8">
                                    <h4><i class="fas fa-gem"></i> Jewel Redemption Process</h4>
                                    <p><strong>Customer:</strong> <?php echo htmlspecialchars($pawnTransaction['Name']); ?></p>
                                    <p><strong>Pawn Code:</strong> <?php echo htmlspecialchars($pawnTransaction['PawnCode']); ?></p>
                                    <p><strong>Item:</strong> <?php echo htmlspecialchars($pawnTransaction['ItemDescription']); ?></p>
                                </div>
                                <div class="col-md-4 text-right">
                                    <h3>₹<?php echo number_format($pawnTransaction['RemainingBalance'], 2); ?></h3>
                                    <p>Balance Due</p>
                                    <?php if ($pawnTransaction['DaysOverdue'] > 0) { ?>
                                        <span class="badge badge-danger">
                                            <?php echo $pawnTransaction['DaysOverdue']; ?> days overdue
                                        </span>
                                    <?php } ?>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Redemption Form -->
                        <form method="POST" enctype="multipart/form-data" id="redemptionForm">
                            <input type="hidden" name="pawn_transaction_id" value="<?php echo $pawnTransaction['ID']; ?>">
                            <input type="hidden" name="customer_id" value="<?php echo $pawnTransaction['CustomerID']; ?>">
                            
                            <div class="row">
                                <!-- Payment Details -->
                                <div class="col-md-6">
                                    <div class="form-section">
                                        <h5><i class="fas fa-money-bill"></i> Payment Details</h5>
                                        
                                        <div class="form-group">
                                            <label>Redemption Date</label>
                                            <input type="date" name="redemption_date" class="form-control" value="<?php echo date('Y-m-d'); ?>">
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Total Amount Due</label>
                                            <input type="text" class="form-control" value="₹<?php echo number_format($pawnTransaction['TotalDue'], 2); ?>" readonly>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Already Paid</label>
                                            <input type="text" class="form-control" value="₹<?php echo number_format($pawnTransaction['TotalPaid'], 2); ?>" readonly>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Final Payment Amount <span class="optional">(Optional)</span></label>
                                            <input type="number" step="0.01" name="final_payment_amount" id="finalPayment" class="form-control" 
                                                   max="<?php echo $pawnTransaction['RemainingBalance']; ?>">
                                            <small class="text-muted">Balance: ₹<?php echo number_format($pawnTransaction['RemainingBalance'], 2); ?></small>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Discount Given <span class="optional">(Optional)</span></label>
                                            <input type="number" step="0.01" name="discount_given" id="discountGiven" class="form-control" value="0">
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Payment Method</label>
                                            <select name="payment_method" class="form-control">
                                                <option value="">Select Method</option>
                                                <option value="Cash">Cash</option>
                                                <option value="Bank Transfer">Bank Transfer</option>
                                                <option value="Card">Card</option>
                                                <option value="Cheque">Cheque</option>
                                                <option value="UPI">UPI</option>
                                            </select>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Reference Number <span class="optional">(Optional)</span></label>
                                            <input type="text" name="reference_number" class="form-control">
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Receiver Details -->
                                <div class="col-md-6">
                                    <div class="form-section">
                                        <h5><i class="fas fa-user"></i> Receiver Details</h5>
                                        
                                        <div class="form-group">
                                            <label>Authorization</label>
                                            <select name="customer_authorization" id="authType" class="form-control">
                                                <option value="Self">Self Customer</option>
                                                <option value="Authorized Person">Authorized Person</option>
                                            </select>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Receiver Name <span class="optional">(Optional)</span></label>
                                            <input type="text" name="receiver_name" class="form-control" 
                                                   value="<?php echo htmlspecialchars($pawnTransaction['Name']); ?>">
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Mobile Number <span class="optional">(Optional)</span></label>
                                            <input type="text" name="receiver_mobile" class="form-control" 
                                                   value="<?php echo htmlspecialchars($pawnTransaction['MobileNumber']); ?>">
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>ID Proof Type <span class="optional">(Optional)</span></label>
                                            <select name="receiver_id_type" class="form-control">
                                                <option value="">Select Type</option>
                                                <option value="Aadhaar Card">Aadhaar Card</option>
                                                <option value="PAN Card">PAN Card</option>
                                                <option value="Driving License">Driving License</option>
                                                <option value="Voter ID">Voter ID</option>
                                                <option value="Passport">Passport</option>
                                            </select>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>ID Proof Number <span class="optional">(Optional)</span></label>
                                            <input type="text" name="receiver_id_number" class="form-control">
                                        </div>
                                        
                                        <div class="form-group" id="relationshipGroup" style="display: none;">
                                            <label>Relationship to Customer <span class="optional">(Optional)</span></label>
                                            <input type="text" name="relationship" class="form-control" 
                                                   placeholder="Example: Husband, Wife, Son, Daughter">
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <!-- File Uploads -->
                                <div class="col-md-6">
                                    <div class="form-section">
                                        <h5><i class="fas fa-upload"></i> File Uploads <span class="optional">(All Optional)</span></h5>
                                        
                                        <div class="form-group">
                                            <label>Receiver ID Proof <span class="optional">(Optional)</span></label>
                                            <input type="file" name="receiver_id_proof" class="form-control-file" 
                                                   accept=".jpg,.jpeg,.png,.pdf">
                                            <small class="text-muted">JPG, PNG, PDF formats only</small>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Receiver Photo <span class="optional">(Optional)</span></label>
                                            <input type="file" name="receiver_photo" class="form-control-file" 
                                                   accept=".jpg,.jpeg,.png">
                                            <small class="text-muted">JPG, PNG formats only</small>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Receiver Signature <span class="optional">(Optional)</span></label>
                                            <input type="file" name="receiver_signature" class="form-control-file" 
                                                   accept=".jpg,.jpeg,.png">
                                            <small class="text-muted">JPG, PNG formats only</small>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Jewel Photo <span class="optional">(Optional)</span></label>
                                            <input type="file" name="jewel_photo" class="form-control-file" 
                                                   accept=".jpg,.jpeg,.png">
                                            <small class="text-muted">Photo of redeemed jewelry</small>
                                        </div>
                                        
                                        <div class="form-group" id="authLetterGroup" style="display: none;">
                                            <label>Authorization Letter <span class="optional">(Optional)</span></label>
                                            <input type="file" name="authorization_letter" class="form-control-file" 
                                                   accept=".jpg,.jpeg,.png,.pdf">
                                            <small class="text-muted">For authorized person only</small>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Item Return Details -->
                                <div class="col-md-6">
                                    <div class="form-section">
                                        <h5><i class="fas fa-gem"></i> Item Return Details</h5>
                                        
                                        <div class="form-group">
                                            <label>Items Returned Description <span class="optional">(Optional)</span></label>
                                            <textarea name="items_description" class="form-control" rows="4" 
                                                      placeholder="Description of redeemed jewelry"><?php echo htmlspecialchars($pawnTransaction['ItemDescription'] . ' - Weight: ' . $pawnTransaction['ItemWeight'] . 'g'); ?></textarea>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Return Condition <span class="optional">(Optional)</span></label>
                                            <select name="return_condition" class="form-control">
                                                <option value="">Select Condition</option>
                                                <option value="Good">Good Condition</option>
                                                <option value="Fair">Fair Condition</option>
                                                <option value="Damaged">Damaged</option>
                                            </select>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Return Notes <span class="optional">(Optional)</span></label>
                                            <textarea name="return_notes" class="form-control" rows="3" 
                                                      placeholder="Any special notes about the return"></textarea>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-12 text-center">
                                    <button type="submit" name="process_redemption" class="btn btn-success btn-lg">
                                        <i class="fas fa-gem"></i> Complete Redemption
                                    </button>
                                    <a href="due_collection.php" class="btn btn-secondary btn-lg ml-2">
                                        <i class="fas fa-arrow-left"></i> Back
                                    </a>
                                </div>
                            </div>
                        </form>
                        
                    <?php } ?>
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    
    <script src="vendor/jquery/jquery.min.js"></script>
    <script>
        $(document).ready(function() {
            // Handle authorization type change
            $('#authType').change(function() {
                if ($(this).val() === 'Authorized Person') {
                    $('#relationshipGroup').show();
                    $('#authLetterGroup').show();
                } else {
                    $('#relationshipGroup').hide();
                    $('#authLetterGroup').hide();
                }
            });
            
            // Validate payment amounts
            $('#finalPayment, #discountGiven').on('input', function() {
                var finalPayment = parseFloat($('#finalPayment').val()) || 0;
                var discount = parseFloat($('#discountGiven').val()) || 0;
                var maxAmount = <?php echo $pawnTransaction ? $pawnTransaction['RemainingBalance'] : 0; ?>;
                
                if ((finalPayment + discount) > maxAmount) {
                    alert('Final payment + Discount cannot exceed the balance due!');
                    $(this).focus();
                }
            });
            
            // Form validation before submit - simplified
            $('#redemptionForm').submit(function(e) {
                return confirm('Are you sure you want to complete this redemption? This action cannot be undone.');
            });
        });
    </script>
    
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
</body>
</html>
