<?php
session_start();

$configPath = __DIR__ . '/includes/config.php';
if (!file_exists($configPath)) {
    die("Error: Database configuration file not found.");
}

include_once($configPath);
include_once(__DIR__ . '/includes/brand.php');

if (!isset($con) || !$con) {
    die("Error: Database connection failed.");
}

if (!isset($_GET['pawn_id'])) {
    die("Error: Pawn ID not provided.");
}

$pawnId = (int)$_GET['pawn_id'];

// Get shop details
$shopQuery = mysqli_query($con, "SELECT * FROM tblshopdetails WHERE IsActive = 1 LIMIT 1");
$shopDetails = mysqli_fetch_assoc($shopQuery);

// Get main pawn transaction with customer details
$query = "
SELECT 
    p.ID as PawnID,
    p.PawnCode,
    p.LoanDate,
    p.DueDate,
    p.Status,
    p.ItemType,
    p.ItemDescription,
    p.ItemWeight,
    p.ItemPurity,
    p.EstimatedValue,
    p.LoanAmount,
    p.InterestRate,
    p.CurrentLoanAmount,
    p.TotalLoanIncreases,
    p.ItemPhotoPath,
    p.Notes as PawnNotes,
    p.TotalItems,
    p.TotalWeight,
    p.TotalEstimatedValue,
    c.ID as CustomerID,
    c.CustomerCode,
    c.Name as CustomerName,
    c.FatherHusbandName,
    c.MobileNumber,
    c.Address,
    c.IDProofType,
    c.IDProofNumber,
    c.PhotoPath as CustomerPhotoPath
FROM tblpawntransactions p
INNER JOIN tblcustomers c ON p.CustomerID = c.ID
WHERE p.ID = $pawnId
";

// Get individual items for the pawn transaction
$itemsQuery = "SELECT * FROM tblpawnitems WHERE PawnTransactionID = $pawnId ORDER BY ItemSequence";
$itemsResult = mysqli_query($con, $itemsQuery);
$items = [];
while ($item = mysqli_fetch_assoc($itemsResult)) {
    $items[] = $item;
}

$result = mysqli_query($con, $query);
$txn = mysqli_fetch_assoc($result);

if (!$txn) {
    die("Error: Pawn transaction not found.");
}

// Get payment collections
$paymentQuery = "SELECT * FROM tblpaymentcollections WHERE PawnTransactionID = $pawnId ORDER BY CollectionDate ASC";
$paymentResult = mysqli_query($con, $paymentQuery);
$payments = [];
$totalPaid = 0;
while($payment = mysqli_fetch_assoc($paymentResult)) {
    $payments[] = $payment;
    $totalPaid += $payment['AmountPaid'];
}

// Get redemption details
$redemptionQuery = "SELECT * FROM tblredemptions WHERE PawnTransactionID = $pawnId";
$redemptionResult = mysqli_query($con, $redemptionQuery);
$redemption = mysqli_fetch_assoc($redemptionResult);

// Get loan increases
$increaseQuery = "SELECT * FROM tblloanincreases WHERE PawnTransactionID = $pawnId ORDER BY IncreaseDate ASC";
$increaseResult = mysqli_query($con, $increaseQuery);
$increases = [];
while($increase = mysqli_fetch_assoc($increaseResult)) {
    $increases[] = $increase;
}

// Get bank pledge details
$bankPledgeQuery = "SELECT bp.*, ba.BankName, ba.AccountHolderName, ba.AccountNumber 
                   FROM tblbankpledges bp 
                   LEFT JOIN tblbankaccounts ba ON bp.BankAccountID = ba.ID 
                   WHERE bp.PawnTransactionID = $pawnId";
$bankPledgeResult = mysqli_query($con, $bankPledgeQuery);
$bankPledge = mysqli_fetch_assoc($bankPledgeResult);

$bankPayments = [];
if ($bankPledge) {
    $bankPaymentQuery = "SELECT * FROM tblbankpledgepayments WHERE BankPledgeID = {$bankPledge['ID']} ORDER BY PaymentDate ASC";
    $bankPaymentResult = mysqli_query($con, $bankPaymentQuery);
    while($payment = mysqli_fetch_assoc($bankPaymentResult)) {
        $bankPayments[] = $payment;
    }
}

// Enhanced function to calculate interest and due amount with proper rounding
function calculateDueAmount($loanDate, $dueDate, $loanAmount, $interestRate, $collectionDate = null) {
    if (!$collectionDate) {
        $collectionDate = date('Y-m-d');
    }
    
    $loanDateTime = new DateTime($loanDate);
    $dueDatetime = new DateTime($dueDate);
    $collectionDateTime = new DateTime($collectionDate);
    
    $totalDays = $loanDateTime->diff($collectionDateTime)->days;
    $daysOverdue = max(0, $collectionDateTime->diff($dueDatetime)->days);
    if ($collectionDateTime < $dueDatetime) {
        $daysOverdue = 0;
    }
    
    // Use 15% per annum daily interest with proper rounding
    $annualRate = 0.15;
    $dailyRate = $annualRate / 365;
    $dailyInterestAmount = round($loanAmount * $dailyRate, 2); // Round daily interest
    $actualInterestAmount = round($dailyInterestAmount * $totalDays, 2); // Round total interest
    $monthlyInterestAmount = round($loanAmount * ($annualRate / 12), 2); // Round monthly interest
    
    $totalDue = round($loanAmount + $actualInterestAmount, 2); // Round total due
    
    return [
        'totalDays' => $totalDays,
        'daysOverdue' => $daysOverdue,
        'dailyInterestAmount' => $dailyInterestAmount, // Rounded daily interest
        'monthlyInterestAmount' => $monthlyInterestAmount, // Rounded monthly interest
        'interestAmount' => $actualInterestAmount, // Rounded total interest
        'totalDue' => $totalDue // Rounded total due
    ];
}

// Calculate detailed interest information
$currentLoanAmount = $txn['CurrentLoanAmount'] ?? $txn['LoanAmount'];
$interestCalc = calculateDueAmount($txn['LoanDate'], $txn['DueDate'], $currentLoanAmount, $txn['InterestRate']);

// Calculate balance
$balanceAmount = $interestCalc['totalDue'] - $totalPaid;
if ($redemption) {
    $balanceAmount = 0; // Fully redeemed
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?php echo page_title('Jewel History - ' . htmlspecialchars($txn['PawnCode'])); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <style>
        /* Optimized A4 with minimal margins for maximum space utilization */
        @page { size: A4; margin: 6mm 8mm 8mm 8mm; }
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Times New Roman', Times, serif; font-size: 9pt; line-height: 1.15; color: #333; }
        .receipt-container { width: 100%; max-width: 210mm; margin: 0 auto; padding: 2mm; border: 1px solid #999; background: #fff; }
        .header { text-align: center; border-bottom: 1px solid #333; padding-bottom: 2px; margin-bottom: 3px; }
        .receipt-title { font-size: 11pt; font-weight: bold; background: #f5f5f5; padding: 2px; border: 1px solid #bbb; margin-top: 2px; }
        .section { margin: 3px 0; border: 1px solid #e1e1e1; border-radius: 2px; overflow: hidden; }
        .section-header { background: #f8f9fa; padding: 2px 4px; font-weight: bold; border-bottom: 1px solid #e1e1e1; color: #495057; font-size: 9pt; }
        .section-content { padding: 3px 4px; }
        .info-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(160px, 1fr)); gap: 2px; margin-bottom: 2px; }
        .info-item { display: flex; justify-content: space-between; padding: 1px 0; border-bottom: 1px dotted #ccc; }
        .info-label { font-weight: bold; color: #6c757d; width: 45%; font-size: 8pt; }
        .info-value { color: #343a40; text-align: right; width: 55%; font-size: 8pt; }
        .payment-table { width: 100%; border-collapse: collapse; margin-top: 2px; }
        .payment-table th, .payment-table td { border: 1px solid #ddd; padding: 1px 2px; text-align: left; font-size: 7pt; }
        .payment-table th { background: #f8f9fa; font-weight: bold; }
        .payment-table .amount { text-align: right; font-weight: bold; }
        .status-badge { display: inline-block; padding: 1px 3px; border-radius: 8px; font-size: 7pt; font-weight: bold; color: white; }
        .status-active { background: #28a745; } .status-redeemed { background: #17a2b8; } .status-extended { background: #ffc107; color: #333; } .status-auctioned { background: #dc3545; }
        .amount-positive { color: #28a745; font-weight: bold; }
        .amount-negative { color: #dc3545; font-weight: bold; }
        .amount-neutral { color: #6c757d; font-weight: bold; }
        .summary-box { background: #f8f9fa; border: 1px solid #007bff; border-radius: 3px; padding: 4px; margin: 3px 0; }
        .summary-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(120px, 1fr)); gap: 3px; }
        .summary-item { text-align: center; padding: 3px; background: white; border-radius: 2px; border: 1px solid #dee2e6; }
        .summary-label { font-size: 7pt; color: #6c757d; margin-bottom: 1px; }
        .summary-value { font-size: 9pt; font-weight: bold; color: #495057; }
        .bank-pledge-box { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border-radius: 3px; padding: 4px; margin: 3px 0; }
        .footer { margin-top: 4px; text-align: center; font-size: 7pt; border-top: 1px dotted #666; padding-top: 3px; }
        .signature-section { display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-top: 6px; }
        .signature-box { text-align: center; font-size: 8pt; }
        .signature-line { border-bottom: 1px solid #333; height: 15px; margin-bottom: 2px; }
        .customer-photo { max-height: 60px; max-width: 50px; border: 1px solid #ccc; padding: 1px; border-radius: 2px; object-fit: cover; }
        .back-button { position: fixed; top: 10px; left: 10px; padding: 6px 12px; background: #6c757d; color: white; border: none; border-radius: 4px; cursor: pointer; text-decoration: none; font-size: 12px; }
        .print-button { position: fixed; top: 10px; right: 10px; padding: 6px 12px; background: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; font-size: 12px; }

        /* Compact Pocket Card */
        .pocket-card { width: 85mm; border: 1px dashed #333; padding: 3mm; border-radius: 2px; margin: 4px 0 0 0; page-break-inside: avoid; }
        .pc-title { font-weight: bold; font-size: 9pt; text-align: center; margin-bottom: 2mm; }
        .pc-grid { display: grid; grid-template-columns: 1fr 1.2fr; gap: 2mm; align-items: center; }
        .pc-photo { display: grid; grid-template-columns: 1fr 1fr; gap: 2mm; }
        .pc-photo img { width: 25mm; height: 25mm; object-fit: cover; border: 1px solid #bbb; border-radius: 2px; }
        .pc-row { display: flex; justify-content: space-between; border-bottom: 1px dotted #bbb; padding: 1px 0; font-size: 8pt; }
        .pc-label { font-weight: 600; color: #555; }
        .pc-value { font-weight: 700; color: #222; text-align: right; }
        .cut-here { text-align: center; font-size: 7pt; color: #666; margin-top: 2mm; }
        @media print { 
            .no-print { display: none; } 
            body { margin: 0; padding: 0; font-size: 8pt; } 
            .receipt-container { border: none; margin: 0; padding: 1mm; } 
            .customer-photo { max-height: 50px; max-width: 40px; } 
            .payment-table th, .payment-table td { padding: 1px; font-size: 6pt; }
            .info-label, .info-value { font-size: 7pt; }
            .section-header { font-size: 8pt; padding: 1px 2px; }
            .receipt-title { font-size: 10pt; padding: 1px; }
            .summary-label { font-size: 6pt; }
            .summary-value { font-size: 8pt; }
            .section { margin: 2px 0; }
            .section-content { padding: 2px; }
            .info-grid { gap: 1px; }
            .pc-photo img { width: 20mm; height: 20mm; }
            .pc-row { font-size: 7pt; }
            .pc-title { font-size: 8pt; }
        }
    </style>
</head>
<body>
    <a href="customer-jewel-history-report.php" class="back-button no-print">← Back</a>
    <button class="print-button no-print" onclick="window.print()">🖨️ Print</button>

    <div class="receipt-container">
        <!-- Header -->
        <div class="header">
            <div class="receipt-title">
                Jewel History Summary
                <div style="font-size: 10pt; margin-top: 2px;">
                    <?php echo htmlspecialchars($txn['PawnCode']); ?> - <?php echo htmlspecialchars($txn['CustomerName']); ?>
                    <span class="status-badge status-<?php echo strtolower($txn['Status']); ?>">
                        <?php echo $txn['Status']; ?>
                    </span>
                </div>
            </div>
        </div>

        <!-- Customer Information -->
        <div class="section">
            <div class="section-header">Customer Details</div>
            <div class="section-content">
                <div class="info-grid">
                    <div class="info-item">
                        <span class="info-label">Name:</span>
                        <span class="info-value"><?php echo htmlspecialchars($txn['CustomerName']); ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">Customer Code:</span>
                        <span class="info-value"><?php echo htmlspecialchars($txn['CustomerCode']); ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">Parent/Spouse Name:</span>
                        <span class="info-value"><?php echo htmlspecialchars($txn['FatherHusbandName']); ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">Mobile Number:</span>
                        <span class="info-value"><?php echo htmlspecialchars($txn['MobileNumber']); ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">ID Proof:</span>
                        <span class="info-value"><?php echo htmlspecialchars($txn['IDProofType'] . ' - ' . $txn['IDProofNumber']); ?></span>
                    </div>
                </div>
                <div class="info-item">
                    <span class="info-label">Address:</span>
                    <span class="info-value"><?php echo htmlspecialchars($txn['Address']); ?></span>
                </div>
                <?php if (!empty($txn['CustomerPhotoPath'])) { 
                    $customerPhotoPath = ltrim($txn['CustomerPhotoPath'], '/');
                ?>
                <div class="info-item" style="border-bottom: none; display: flex; align-items: center; padding: 3px 0;">
                    <span class="info-label">Customer Photo:</span>
                    <span class="info-value" style="text-align: center;">
                        <?php if (file_exists(__DIR__ . '/' . $customerPhotoPath)) { ?>
                            <img src="<?php echo $customerPhotoPath; ?>" alt="Customer Photo" class="customer-photo">
                        <?php } else { ?>
                            <span style="color: #999; font-size: 7pt;">Photo not available</span>
                        <?php } ?>
                    </span>
                </div>
                <?php } ?>
            </div>
        </div>

        <!-- Jewel Information -->
        <div class="section">
            <div class="section-header">
                Jewel Details 
                <?php if (count($items) > 1) { ?>
                    (<?php echo count($items); ?> items)
                <?php } ?>
            </div>
            <div class="section-content">
                <?php if (count($items) > 0) { ?>
                    <?php foreach ($items as $index => $item) { ?>
                        <?php if (count($items) > 1) { ?><h4 style="margin: 3px 0 2px; color: #007bff; font-size: 9pt;">Item #<?php echo ($index + 1); ?></h4><?php } ?>
                        <div class="info-grid">
                            <div class="info-item">
                                <span class="info-label">Type:</span>
                                <span class="info-value"><?php echo htmlspecialchars($item['ItemType']); ?></span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">Weight:</span>
                                <span class="info-value"><?php echo number_format($item['ItemWeight'], 3); ?> g</span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">Purity:</span>
                                <span class="info-value"><?php echo htmlspecialchars($item['ItemPurity']); ?></span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">Estimated Value:</span>
                                <span class="info-value amount-positive">₹<?php echo number_format($item['EstimatedValue'], 2); ?></span>
                            </div>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Description:</span>
                            <span class="info-value"><?php echo htmlspecialchars($item['ItemDescription']); ?></span>
                        </div>
                        <?php if (!empty($item['ItemPhotoPath'])) { ?>
                        <div class="info-item" style="border-bottom: none; display: flex; align-items: center; padding: 3px 0;">
                            <span class="info-label">Photo:</span>
                            <span class="info-value" style="text-align: center;">
                                <?php 
                                $photoPath = ltrim($item['ItemPhotoPath'], '/');
                                if (file_exists(__DIR__ . '/' . $photoPath)) {
                                ?>
                                    <img src="<?php echo $photoPath; ?>" alt="Item Photo" style="max-height: 60px; border: 1px solid #ccc; padding: 1px; border-radius: 2px;">
                                <?php } else { ?>
                                    <span style="color: #999; font-size: 7pt;">Photo not available</span>
                                <?php } ?>
                            </span>
                        </div>
                        <?php } ?>
                        
                        <?php if (count($items) > 1 && $index < count($items) - 1) { ?>
                            <hr style="margin: 4px 0; border: 1px solid #eee;">
                        <?php } ?>
                    <?php } ?>
                <?php } else { ?>
                    <!-- Fallback to main transaction data if no items -->
                    <div class="info-grid">
                        <div class="info-item">
                            <span class="info-label">Type:</span>
                            <span class="info-value"><?php echo htmlspecialchars($txn['ItemType']); ?></span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Weight:</span>
                            <span class="info-value"><?php echo number_format($txn['ItemWeight'], 3); ?> g</span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Purity:</span>
                            <span class="info-value"><?php echo htmlspecialchars($txn['ItemPurity']); ?></span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Estimated Value:</span>
                            <span class="info-value amount-positive">₹<?php echo number_format($txn['EstimatedValue'], 2); ?></span>
                        </div>
                    </div>
                    <div class="info-item">
                        <span class="info-label">Description:</span>
                        <span class="info-value"><?php echo htmlspecialchars($txn['ItemDescription']); ?></span>
                    </div>
                    <?php if (!empty($txn['ItemPhotoPath'])) { ?>
                    <div class="info-item" style="border-bottom: none; display: flex; align-items: center; padding: 10px 0;">
                        <span class="info-label">Photo:</span>
                        <span class="info-value" style="text-align: center;">
                            <?php 
                            $photoPath = ltrim($txn['ItemPhotoPath'], '/');
                            if (file_exists(__DIR__ . '/' . $photoPath)) {
                            ?>
                                <img src="<?php echo $photoPath; ?>" alt="Item Photo" style="max-height: 120px; border: 1px solid #ccc; padding: 2px; border-radius: 4px;">
                            <?php } else { ?>
                                <span style="color: #999;">Photo not available</span>
                            <?php } ?>
                        </span>
                    </div>
                    <?php } ?>
                <?php } ?>
            </div>
        </div>

        <!-- Loan Summary -->
        <div class="summary-box">
            <h3 style="text-align: center; margin-bottom: 4px; color: #007bff; font-size: 10pt;">Loan Summary</h3>
            <div class="summary-grid">
                <div class="summary-item">
                    <div class="summary-label">Loan Amount</div>
                    <div class="summary-value amount-neutral">₹<?php echo number_format($txn['LoanAmount'], 2); ?></div>
                </div>
                <?php if (count($items) > 1) { ?>
                <div class="summary-item">
                    <div class="summary-label">Total Items</div>
                    <div class="summary-value"><?php echo count($items); ?> items</div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Total Weight</div>
                    <div class="summary-value"><?php echo number_format($txn['TotalWeight'] ?: array_sum(array_column($items, 'ItemWeight')), 3); ?>g</div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Total Value</div>
                    <div class="summary-value">₹<?php echo number_format($txn['TotalEstimatedValue'] ?: array_sum(array_column($items, 'EstimatedValue')), 2); ?></div>
                </div>
                <?php } ?>
                <div class="summary-item">
                    <div class="summary-label">Interest Rate</div>
                    <div class="summary-value">15% per year (<?php echo number_format((0.15/365)*100, 4); ?>% per day)</div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Total Paid</div>
                    <div class="summary-value amount-positive">₹<?php echo number_format($totalPaid, 2); ?></div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Balance</div>
                    <div class="summary-value <?php echo $balanceAmount > 0 ? 'amount-negative' : 'amount-neutral'; ?>">₹<?php echo number_format($balanceAmount, 2); ?></div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Loan Date</div>
                    <div class="summary-value"><?php echo date('d/m/Y', strtotime($txn['LoanDate'])); ?></div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Due Date</div>
                    <div class="summary-value"><?php echo date('d/m/Y', strtotime($txn['DueDate'])); ?></div>
                </div>
            </div>
        </div>

        <!-- Detailed Interest Information -->
        <div class="section">
            <div class="section-header">Interest Details</div>
            <div class="section-content">
                <div class="info-grid">
                    <div class="info-item">
                        <span class="info-label">Total Days:</span>
                        <span class="info-value"><?php echo $interestCalc['totalDays']; ?> days</span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">Overdue Days:</span>
                        <span class="info-value <?php echo $interestCalc['daysOverdue'] > 0 ? 'amount-negative' : 'amount-positive'; ?>"><?php echo $interestCalc['daysOverdue']; ?> days</span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">Daily Interest:</span>
                        <span class="info-value">₹<?php echo number_format($interestCalc['dailyInterestAmount'], 2); ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">Monthly Interest:</span>
                        <span class="info-value">₹<?php echo number_format($interestCalc['monthlyInterestAmount'], 2); ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">Accrued Interest:</span>
                        <span class="info-value amount-neutral">₹<?php echo number_format($interestCalc['interestAmount'], 2); ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">Total Due:</span>
                        <span class="info-value amount-negative">₹<?php echo number_format($interestCalc['totalDue'], 2); ?></span>
                    </div>
                </div>
            </div>
        </div>

        <?php if (count($increases) > 0) { ?>
        <!-- Loan Increases -->
        <div class="section">
            <div class="section-header">Loan Increase History</div>
            <div class="section-content">
                <table class="payment-table">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Increase Amount</th>
                            <th>New Total Loan</th>
                            <th>Reason</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach($increases as $increase) { ?>
                        <tr>
                            <td><?php echo date('d/m/Y', strtotime($increase['IncreaseDate'])); ?></td>
                            <td class="amount">₹<?php echo number_format($increase['IncreaseAmount'], 2); ?></td>
                            <td class="amount">₹<?php echo number_format($increase['NewTotalLoanAmount'], 2); ?></td>
                            <td><?php echo htmlspecialchars($increase['Reason']); ?></td>
                        </tr>
                        <?php } ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php } ?>

        <?php if (count($payments) > 0) { ?>
        <!-- Payment History -->
        <div class="section">
            <div class="section-header">Payment History</div>
            <div class="section-content">
                <table class="payment-table">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Method</th>
                            <th>Interest Days</th>
                            <th>Interest Amount</th>
                            <th>Amount Paid</th>
                            <th>Discount</th>
                            <th>Balance</th>
                            <th>Collection Code</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach($payments as $payment) { ?>
                        <tr>
                            <td><?php echo date('d/m/Y', strtotime($payment['CollectionDate'])); ?></td>
                            <td><?php echo htmlspecialchars($payment['PaymentMethod']); ?></td>
                            <td><?php echo $payment['InterestDays']; ?></td>
                            <td class="amount">₹<?php echo number_format($payment['InterestAmount'], 2); ?></td>
                            <td class="amount amount-positive">₹<?php echo number_format($payment['AmountPaid'], 2); ?></td>
                            <td class="amount">₹<?php echo number_format($payment['DiscountAmount'], 2); ?></td>
                            <td class="amount">₹<?php echo number_format($payment['BalanceAmount'], 2); ?></td>
                            <td><?php echo htmlspecialchars($payment['CollectionCode']); ?></td>
                        </tr>
                        <?php } ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php } ?>

        <?php if ($bankPledge) { ?>
        <!-- Bank Pledge Details -->
        <div class="bank-pledge-box">
            <h3 style="text-align: center; margin-bottom: 4px; font-size: 9pt;">Bank Pledge Details</h3>
            <div class="info-grid">
                <div class="info-item">
                    <span class="info-label" style="color: white;">Bank:</span>
                    <span class="info-value" style="color: white;"><?php echo htmlspecialchars($bankPledge['BankName']); ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label" style="color: white;">Account Number:</span>
                    <span class="info-value" style="color: white;"><?php echo htmlspecialchars($bankPledge['AccountNumber']); ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label" style="color: white;">Pledge Amount:</span>
                    <span class="info-value" style="color: white;">₹<?php echo number_format($bankPledge['PledgeAmount'], 2); ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label" style="color: white;">Interest Rate:</span>
                    <span class="info-value" style="color: white;"><?php echo $bankPledge['InterestRate']; ?>%</span>
                </div>
                <div class="info-item">
                    <span class="info-label" style="color: white;">Pledge Date:</span>
                    <span class="info-value" style="color: white;"><?php echo date('d/m/Y', strtotime($bankPledge['PledgeDate'])); ?></span>
                </div>
                <div class="info-item">
                    <span class="info-label" style="color: white;">Maturity Date:</span>
                    <span class="info-value" style="color: white;"><?php echo date('d/m/Y', strtotime($bankPledge['MaturityDate'])); ?></span>
                </div>
            </div>

            <?php if (count($bankPayments) > 0) { ?>
            <h4 style="margin: 4px 0 3px; color: white; font-size: 8pt;">Bank Payment History</h4>
            <table class="payment-table" style="background: rgba(255,255,255,0.9); color: #333;">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Type</th>
                        <th>Amount</th>
                        <th>Receipt No</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach($bankPayments as $payment) { ?>
                    <tr>
                        <td><?php echo date('d/m/Y', strtotime($payment['PaymentDate'])); ?></td>
                        <td><?php echo htmlspecialchars($payment['PaymentType']); ?></td>
                        <td class="amount">₹<?php echo number_format($payment['PaymentAmount'], 2); ?></td>
                        <td><?php echo htmlspecialchars($payment['ReceiptNumber']); ?></td>
                    </tr>
                    <?php } ?>
                </tbody>
            </table>
            <?php } ?>
        </div>
        <?php } ?>

        <?php if ($redemption) { ?>
        <!-- Redemption Details -->
        <div class="section">
            <div class="section-header">Redemption Details</div>
            <div class="section-content">
                <div class="info-grid">
                    <div class="info-item">
                        <span class="info-label">Redemption Date:</span>
                        <span class="info-value"><?php echo date('d/m/Y', strtotime($redemption['RedemptionDate'])); ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">Final Payment:</span>
                        <span class="info-value amount-positive">₹<?php echo number_format($redemption['FinalPaymentAmount'], 2); ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">Discount:</span>
                        <span class="info-value amount-neutral">₹<?php echo number_format($redemption['DiscountGiven'], 2); ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">Receiver:</span>
                        <span class="info-value"><?php echo htmlspecialchars($redemption['ReceiverName']); ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">Return Condition:</span>
                        <span class="info-value"><?php echo htmlspecialchars($redemption['ReturnCondition']); ?></span>
                    </div>
                    <div class="info-item">
                        <span class="info-label">Authorization:</span>
                        <span class="info-value"><?php echo htmlspecialchars($redemption['CustomerAuthorization']); ?></span>
                    </div>
                </div>
            </div>
        </div>
        <?php } ?>

        <!-- Signatures -->
        <div class="signature-section">
            <div class="signature-box">
                <div class="signature-line"></div>
                <div>Customer Signature</div>
            </div>
            <div class="signature-box">
                <div class="signature-line"></div>
                <div>Authorized Signatory</div>
            </div>
        </div>

        <!-- Footer -->
        <div class="footer">
            <p><strong><?php echo $shopDetails && !empty($shopDetails['FooterNote']) ? htmlspecialchars($shopDetails['FooterNote']) : 'Thank you for your business!'; ?></strong></p>
            <p>This report was generated on <?php echo date('d/m/Y h:i A'); ?>.</p>
            <p>For reference only. Keep this document for your records.</p>
        </div>
    </div>
    
    <!-- Pocket Card (Detachable) -->
    <?php 
        $firstItem = count($items) > 0 ? $items[0] : null; 
        $custPhoto = !empty($txn['CustomerPhotoPath']) ? ltrim($txn['CustomerPhotoPath'], '/') : '';
        $itemPhoto = ($firstItem && !empty($firstItem['ItemPhotoPath'])) ? ltrim($firstItem['ItemPhotoPath'], '/') : (!empty($txn['ItemPhotoPath']) ? ltrim($txn['ItemPhotoPath'], '/') : '');
    ?>
    <div class="pocket-card">
        <div class="pc-title">Pawn Pocket Card</div>
        <div class="pc-grid">
            <div class="pc-photo">
                <div>
                    <?php if ($custPhoto && file_exists(__DIR__ . '/' . $custPhoto)) { ?>
                        <img src="<?php echo $custPhoto; ?>" alt="Customer Photo">
                    <?php } else { ?>
                        <div style="width:36mm;height:36mm;border:1px solid #bbb;border-radius:3px;display:flex;align-items:center;justify-content:center;font-size:9pt;color:#888;">No Photo</div>
                    <?php } ?>
                </div>
                <div>
                    <?php if ($itemPhoto && file_exists(__DIR__ . '/' . $itemPhoto)) { ?>
                        <img src="<?php echo $itemPhoto; ?>" alt="Item Photo">
                    <?php } else { ?>
                        <div style="width:36mm;height:36mm;border:1px solid #bbb;border-radius:3px;display:flex;align-items:center;justify-content:center;font-size:9pt;color:#888;">No Photo</div>
                    <?php } ?>
                </div>
            </div>
            <div>
                <div class="pc-row"><span class="pc-label">Pawn Code</span><span class="pc-value"><?php echo htmlspecialchars($txn['PawnCode']); ?></span></div>
                <div class="pc-row"><span class="pc-label">Customer</span><span class="pc-value"><?php echo htmlspecialchars($txn['CustomerName']); ?></span></div>
                <div class="pc-row"><span class="pc-label">Mobile</span><span class="pc-value"><?php echo htmlspecialchars($txn['MobileNumber']); ?></span></div>
                <div class="pc-row"><span class="pc-label">Loan Amt</span><span class="pc-value">₹<?php echo number_format($txn['LoanAmount'], 2); ?></span></div>
                <div class="pc-row"><span class="pc-label">Loan Date</span><span class="pc-value"><?php echo date('d/m/Y', strtotime($txn['LoanDate'])); ?></span></div>
                <div class="pc-row"><span class="pc-label">Due Date</span><span class="pc-value"><?php echo date('d/m/Y', strtotime($txn['DueDate'])); ?></span></div>
                <?php if ($firstItem) { ?>
                    <div class="pc-row"><span class="pc-label">Item</span><span class="pc-value"><?php echo htmlspecialchars($firstItem['ItemType']); ?></span></div>
                    <div class="pc-row"><span class="pc-label">Weight</span><span class="pc-value"><?php echo number_format($firstItem['ItemWeight'], 3); ?>g</span></div>
                <?php } ?>
                <div class="pc-row"><span class="pc-label">Status</span><span class="pc-value"><?php echo htmlspecialchars($txn['Status']); ?></span></div>
            </div>
        </div>
        <div class="cut-here">— — — — — — — — — — — — — — — — — — — — — — — (Cut here)</div>
    </div>
    
    <script>
        // Auto print when page loads
        window.onload = function() {
            setTimeout(function() { window.print(); }, 500);
        };
    </script>
</body>
</html>
