<?php
session_start();

// Check if config file exists
$configPath = __DIR__ . '/includes/config.php';
if (!file_exists($configPath)) {
    die("Error: Database configuration file not found. Please create includes/config.php");
}

include_once($configPath);
include_once(__DIR__ . '/includes/brand.php');

// Check if database connection exists
if (!isset($con) || !$con) {
    die("Error: Database connection failed. Please check your database configuration.");
}

if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$msg = "";
$userId = $_SESSION['uid'];

// Search and pagination variables
$search = isset($_GET['search']) ? mysqli_real_escape_string($con, $_GET['search']) : '';
$type_filter = isset($_GET['type']) ? mysqli_real_escape_string($con, $_GET['type']) : '';
$limit = 10; // Records per page
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// Create uploads directory structure if it doesn't exist
$baseUploadDir = 'uploads/';
$incomeExpenseUploadDir = $baseUploadDir . 'income-expenses/';
$receiptUploadDir = $incomeExpenseUploadDir . 'receipts/';

// Create directories if they don't exist
if (!file_exists($baseUploadDir)) {
    mkdir($baseUploadDir, 0777, true);
}
if (!file_exists($incomeExpenseUploadDir)) {
    mkdir($incomeExpenseUploadDir, 0777, true);
}
if (!file_exists($receiptUploadDir)) {
    mkdir($receiptUploadDir, 0777, true);
}

// Function to handle file upload
function handleFileUpload($file, $uploadDir, $prefix, $description = '') {
    if (!isset($file) || $file['error'] != 0) {
        return '';
    }
    
    // Validate file size (5MB max)
    if ($file['size'] > 5 * 1024 * 1024) {
        return false;
    }
    
    $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'pdf'];
    
    if (!in_array($fileExtension, $allowedExtensions)) {
        return false;
    }
    
    // Create filename with description and timestamp
    $descriptionSafe = preg_replace('/[^A-Za-z0-9_-]/', '_', $description);
    $fileName = $prefix . '_' . $descriptionSafe . '_' . time() . '.' . $fileExtension;
    $filePath = $uploadDir . $fileName;
    
    if (move_uploaded_file($file['tmp_name'], $filePath)) {
        return $filePath;
    }
    
    return false;
}

// Handle form submissions
if ($_POST) {
    if (isset($_POST['add_transaction'])) {
        $transactionCode = strtoupper($_POST['type']) . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        $type = mysqli_real_escape_string($con, $_POST['type']);
        $category = mysqli_real_escape_string($con, $_POST['category']);
        $description = mysqli_real_escape_string($con, $_POST['description']);
        $amount = (float)$_POST['amount'];
        $transactionDate = mysqli_real_escape_string($con, $_POST['transaction_date']);
        $paymentMethod = mysqli_real_escape_string($con, $_POST['payment_method']);
        $notes = mysqli_real_escape_string($con, $_POST['notes']);
        
        // Handle receipt upload
        $receiptPath = handleFileUpload($_FILES['receipt'], $receiptUploadDir, 'receipt', $description);
        if ($receiptPath === false && isset($_FILES['receipt']) && $_FILES['receipt']['error'] == 0) {
            $msg = "Error uploading receipt. Please check file size (max 5MB) and format (jpg, png, pdf).";
        } else {
            $query = "INSERT INTO tblincomeexpenses (TransactionCode, Type, Category, Description, Amount, TransactionDate, PaymentMethod, Receipt, Notes, CreatedBy) 
                      VALUES ('$transactionCode', '$type', '$category', '$description', '$amount', '$transactionDate', '$paymentMethod', '$receiptPath', '$notes', '$userId')";
            
            if (mysqli_query($con, $query)) {
                $msg = "Transaction added successfully!";
            } else {
                $msg = "Error adding transaction: " . mysqli_error($con);
            }
        }
    }
    
    if (isset($_POST['update_transaction'])) {
        $id = (int)$_POST['transaction_id'];
        $type = mysqli_real_escape_string($con, $_POST['type']);
        $category = mysqli_real_escape_string($con, $_POST['category']);
        $description = mysqli_real_escape_string($con, $_POST['description']);
        $amount = (float)$_POST['amount'];
        $transactionDate = mysqli_real_escape_string($con, $_POST['transaction_date']);
        $paymentMethod = mysqli_real_escape_string($con, $_POST['payment_method']);
        $notes = mysqli_real_escape_string($con, $_POST['notes']);
        $status = mysqli_real_escape_string($con, $_POST['status']);
        
        // Get existing transaction data
        $existingQuery = mysqli_query($con, "SELECT Receipt FROM tblincomeexpenses WHERE ID=$id");
        $existing = mysqli_fetch_assoc($existingQuery);
        
        $receiptPath = $existing['Receipt'];
        
        // Handle receipt upload
        if (isset($_FILES['receipt']) && $_FILES['receipt']['error'] == 0) {
            $newReceiptPath = handleFileUpload($_FILES['receipt'], $receiptUploadDir, 'receipt', $description);
            if ($newReceiptPath === false) {
                $msg = "Error uploading receipt. Please check file size and format.";
            } else {
                // Delete old receipt
                if ($receiptPath && file_exists($receiptPath)) {
                    unlink($receiptPath);
                }
                $receiptPath = $newReceiptPath;
            }
        }
        
        if (!$msg) { // Only update if no upload errors
            $query = "UPDATE tblincomeexpenses SET Type='$type', Category='$category', Description='$description', 
                      Amount='$amount', TransactionDate='$transactionDate', PaymentMethod='$paymentMethod', 
                      Receipt='$receiptPath', Notes='$notes', Status='$status' WHERE ID=$id";
            
            if (mysqli_query($con, $query)) {
                $msg = "Transaction updated successfully!";
            } else {
                $msg = "Error updating transaction: " . mysqli_error($con);
            }
        }
    }
}

// Delete transaction
if (isset($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    
    // Get file path before deletion
    $fileQuery = mysqli_query($con, "SELECT Receipt FROM tblincomeexpenses WHERE ID=$id");
    $fileData = mysqli_fetch_assoc($fileQuery);
    
    $query = "DELETE FROM tblincomeexpenses WHERE ID=$id";
    if (mysqli_query($con, $query)) {
        // Delete associated file
        if ($fileData['Receipt'] && file_exists($fileData['Receipt'])) {
            unlink($fileData['Receipt']);
        }
        $msg = "Transaction deleted successfully!";
    } else {
        $msg = "Error deleting transaction: " . mysqli_error($con);
    }
}

// Get transaction for editing
$editTransaction = null;
if (isset($_GET['edit'])) {
    $id = (int)$_GET['edit'];
    $result = mysqli_query($con, "SELECT * FROM tblincomeexpenses WHERE ID=$id");
    $editTransaction = mysqli_fetch_assoc($result);
}

// Build search query
$whereClause = "WHERE 1=1";
if (!empty($search)) {
    $whereClause .= " AND (Description LIKE '%$search%' OR Category LIKE '%$search%' OR TransactionCode LIKE '%$search%' OR Notes LIKE '%$search%')";
}
if (!empty($type_filter)) {
    $whereClause .= " AND Type = '$type_filter'";
}

// Get total count for pagination
$countQuery = "SELECT COUNT(*) as total FROM tblincomeexpenses $whereClause";
$countResult = mysqli_query($con, $countQuery);
$totalRecords = mysqli_fetch_assoc($countResult)['total'];
$totalPages = ceil($totalRecords / $limit);

// Fetch transactions with search and pagination
$transactionsQuery = mysqli_query($con, "SELECT * FROM tblincomeexpenses $whereClause ORDER BY TransactionDate DESC, CreatedDate DESC LIMIT $limit OFFSET $offset");

// Get summary statistics
$incomeQuery = mysqli_query($con, "SELECT SUM(Amount) as total FROM tblincomeexpenses WHERE Type='Income' AND Status='Active'");
$totalIncome = mysqli_fetch_assoc($incomeQuery)['total'] ?? 0;

$expenseQuery = mysqli_query($con, "SELECT SUM(Amount) as total FROM tblincomeexpenses WHERE Type='Expense' AND Status='Active'");
$totalExpense = mysqli_fetch_assoc($expenseQuery)['total'] ?? 0;

$netAmount = $totalIncome - $totalExpense;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?php echo page_title('Income & Expenses'); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/compact.css" rel="stylesheet">
    <style>
        .search-container {
            background: #f8f9fc;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .summary-cards {
            margin-bottom: 20px;
        }
        .income-card { border-left: 4px solid #1cc88a; }
        .expense-card { border-left: 4px solid #e74a3b; }
        .net-card { border-left: 4px solid #36b9cc; }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid">
                    
                    <?php if($msg) { ?>
                        <div class="alert alert-info"><?php echo $msg; ?></div>
                    <?php } ?>
                    
                    <!-- Summary Cards -->
                    <div class="row summary-cards">
                        <div class="col-xl-4 col-md-6 mb-4">
                            <div class="card income-card shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">Total Income</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($totalIncome, 2); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-arrow-up fa-2x text-success"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-4 col-md-6 mb-4">
                            <div class="card expense-card shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">Total Expenses</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($totalExpense, 2); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-arrow-down fa-2x text-danger"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-4 col-md-6 mb-4">
                            <div class="card net-card shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">Net Amount</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800 <?php echo $netAmount >= 0 ? 'text-success' : 'text-danger'; ?>">
                                                ₹<?php echo number_format($netAmount, 2); ?>
                                            </div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-calculator fa-2x text-info"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Add/Edit Transaction Form -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary">
                                <?php echo $editTransaction ? 'Edit Transaction' : 'Add New Transaction'; ?>
                            </h6>
                        </div>
                        <div class="card-body">
                            <form method="POST" enctype="multipart/form-data">
                                <?php if($editTransaction) { ?>
                                    <input type="hidden" name="transaction_id" value="<?php echo $editTransaction['ID']; ?>">
                                <?php } ?>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label>Type <span class="text-danger">*</span></label>
                                            <select name="type" class="form-control" required>
                                                <option value="">Select Type</option>
                                                <option value="Income" <?php echo ($editTransaction && $editTransaction['Type'] == 'Income') ? 'selected' : ''; ?>>Income</option>
                                                <option value="Expense" <?php echo ($editTransaction && $editTransaction['Type'] == 'Expense') ? 'selected' : ''; ?>>Expense</option>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label>Category <span class="text-danger">*</span></label>
                                            <select name="category" class="form-control" required>
                                                <option value="">Select Category</option>
                                                <optgroup label="Income Categories">
                                                    <option value="Service Charges" <?php echo ($editTransaction && $editTransaction['Category'] == 'Service Charges') ? 'selected' : ''; ?>>Service Charges</option>
                                                    <option value="Interest Income" <?php echo ($editTransaction && $editTransaction['Category'] == 'Interest Income') ? 'selected' : ''; ?>>Interest Income</option>
                                                    <option value="Commission" <?php echo ($editTransaction && $editTransaction['Category'] == 'Commission') ? 'selected' : ''; ?>>Commission</option>
                                                    <option value="Other Income" <?php echo ($editTransaction && $editTransaction['Category'] == 'Other Income') ? 'selected' : ''; ?>>Other Income</option>
                                                </optgroup>
                                                <optgroup label="Expense Categories">
                                                    <option value="Rent" <?php echo ($editTransaction && $editTransaction['Category'] == 'Rent') ? 'selected' : ''; ?>>Rent</option>
                                                    <option value="Utilities" <?php echo ($editTransaction && $editTransaction['Category'] == 'Utilities') ? 'selected' : ''; ?>>Utilities</option>
                                                    <option value="Staff Salary" <?php echo ($editTransaction && $editTransaction['Category'] == 'Staff Salary') ? 'selected' : ''; ?>>Staff Salary</option>
                                                    <option value="Office Supplies" <?php echo ($editTransaction && $editTransaction['Category'] == 'Office Supplies') ? 'selected' : ''; ?>>Office Supplies</option>
                                                    <option value="Marketing" <?php echo ($editTransaction && $editTransaction['Category'] == 'Marketing') ? 'selected' : ''; ?>>Marketing</option>
                                                    <option value="Maintenance" <?php echo ($editTransaction && $editTransaction['Category'] == 'Maintenance') ? 'selected' : ''; ?>>Maintenance</option>
                                                    <option value="Insurance" <?php echo ($editTransaction && $editTransaction['Category'] == 'Insurance') ? 'selected' : ''; ?>>Insurance</option>
                                                    <option value="Other Expenses" <?php echo ($editTransaction && $editTransaction['Category'] == 'Other Expenses') ? 'selected' : ''; ?>>Other Expenses</option>
                                                </optgroup>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label>Amount <span class="text-danger">*</span></label>
                                            <input type="number" step="0.01" name="amount" class="form-control" required
                                                   value="<?php echo $editTransaction ? $editTransaction['Amount'] : ''; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label>Transaction Date <span class="text-danger">*</span></label>
                                            <input type="date" name="transaction_date" class="form-control" required
                                                   value="<?php echo $editTransaction ? $editTransaction['TransactionDate'] : date('Y-m-d'); ?>">
                                        </div>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label>Payment Method</label>
                                            <select name="payment_method" class="form-control">
                                                <option value="">Select Payment Method</option>
                                                <option value="Cash" <?php echo ($editTransaction && $editTransaction['PaymentMethod'] == 'Cash') ? 'selected' : ''; ?>>Cash</option>
                                                <option value="Bank Transfer" <?php echo ($editTransaction && $editTransaction['PaymentMethod'] == 'Bank Transfer') ? 'selected' : ''; ?>>Bank Transfer</option>
                                                <option value="Cheque" <?php echo ($editTransaction && $editTransaction['PaymentMethod'] == 'Cheque') ? 'selected' : ''; ?>>Cheque</option>
                                                <option value="UPI" <?php echo ($editTransaction && $editTransaction['PaymentMethod'] == 'UPI') ? 'selected' : ''; ?>>UPI</option>
                                                <option value="Card" <?php echo ($editTransaction && $editTransaction['PaymentMethod'] == 'Card') ? 'selected' : ''; ?>>Card</option>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label>Receipt Upload</label>
                                            <input type="file" name="receipt" class="form-control-file" accept=".pdf,.jpg,.jpeg,.png">
                                            <small class="form-text text-muted">Max file size: 5MB. Allowed: PDF, JPG, PNG</small>
                                            <?php if($editTransaction && $editTransaction['Receipt']) { ?>
                                                <br><small class="text-muted">Current: <a href="<?php echo $editTransaction['Receipt']; ?>" target="_blank">View Receipt</a></small>
                                            <?php } ?>
                                        </div>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label>Description <span class="text-danger">*</span></label>
                                            <textarea name="description" class="form-control" rows="3" required><?php echo $editTransaction ? htmlspecialchars($editTransaction['Description']) : ''; ?></textarea>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label>Notes</label>
                                            <textarea name="notes" class="form-control" rows="3"><?php echo $editTransaction ? htmlspecialchars($editTransaction['Notes']) : ''; ?></textarea>
                                        </div>
                                    </div>
                                </div>
                                <?php if($editTransaction) { ?>
                                    <div class="form-group">
                                        <label>Status</label>
                                        <select name="status" class="form-control">
                                            <option value="Active" <?php echo ($editTransaction['Status'] == 'Active') ? 'selected' : ''; ?>>Active</option>
                                            <option value="Inactive" <?php echo ($editTransaction['Status'] == 'Inactive') ? 'selected' : ''; ?>>Inactive</option>
                                        </select>
                                    </div>
                                <?php } ?>
                                <button type="submit" name="<?php echo $editTransaction ? 'update_transaction' : 'add_transaction'; ?>" 
                                        class="btn btn-primary">
                                    <i class="fas fa-save"></i> <?php echo $editTransaction ? 'Update' : 'Add'; ?> Transaction
                                </button>
                                <?php if($editTransaction) { ?>
                                    <a href="income-expenses.php" class="btn btn-secondary">Cancel</a>
                                <?php } ?>
                            </form>
                        </div>
                    </div>
                    
                    <!-- Search Section -->
                    <div class="search-container">
                        <form method="GET" class="row">
                            <div class="col-md-6">
                                <input type="text" name="search" class="form-control" 
                                       placeholder="Search by description, category, code, or notes" 
                                       value="<?php echo htmlspecialchars($search); ?>">
                            </div>
                            <div class="col-md-2">
                                <select name="type" class="form-control">
                                    <option value="">All Types</option>
                                    <option value="Income" <?php echo ($type_filter == 'Income') ? 'selected' : ''; ?>>Income</option>
                                    <option value="Expense" <?php echo ($type_filter == 'Expense') ? 'selected' : ''; ?>>Expense</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <button type="submit" class="btn btn-primary btn-block">
                                    <i class="fas fa-search"></i> Search
                                </button>
                            </div>
                            <div class="col-md-2">
                                <a href="income-expenses.php" class="btn btn-secondary btn-block">
                                    <i class="fas fa-refresh"></i> Clear
                                </a>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Transactions List -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3 d-flex justify-content-between align-items-center">
                            <h6 class="m-0 font-weight-bold text-primary">Income & Expense Transactions</h6>
                            <small class="text-muted">Total: <?php echo $totalRecords; ?></small>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered">
                                    <thead>
                                        <tr>
                                            <th>No</th>
                                            <th>Code</th>
                                            <th>Type</th>
                                            <th>Category</th>
                                            <th>Description</th>
                                            <th>Amount</th>
                                            <th>Date</th>
                                            <th>Payment</th>
                                            <th>Receipt</th>
                                            <th>Status</th>
                                            <th>Action</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php 
                                        if (mysqli_num_rows($transactionsQuery) > 0) {
                                            $i = $offset + 1; 
                                            while($transaction = mysqli_fetch_assoc($transactionsQuery)) { 
                                        ?>
                                            <tr>
                                                <td><?php echo $i++; ?></td>
                                                <td><?php echo htmlspecialchars($transaction['TransactionCode']); ?></td>
                                                <td>
                                                    <span class="badge badge-<?php echo $transaction['Type'] == 'Income' ? 'success' : 'danger'; ?>">
                                                        <?php echo $transaction['Type']; ?>
                                                    </span>
                                                </td>
                                                <td><?php echo htmlspecialchars($transaction['Category']); ?></td>
                                                <td><?php echo htmlspecialchars(substr($transaction['Description'], 0, 50)) . (strlen($transaction['Description']) > 50 ? '...' : ''); ?></td>
                                                <td class="<?php echo $transaction['Type'] == 'Income' ? 'text-success' : 'text-danger'; ?>">
                                                    ₹<?php echo number_format($transaction['Amount'], 2); ?>
                                                </td>
                                                <td><?php echo date('d-m-Y', strtotime($transaction['TransactionDate'])); ?></td>
                                                <td><?php echo htmlspecialchars($transaction['PaymentMethod']); ?></td>
                                                <td>
                                                    <?php if($transaction['Receipt']) { ?>
                                                        <a href="<?php echo $transaction['Receipt']; ?>" target="_blank" class="btn btn-sm btn-info">
                                                            <i class="fas fa-file"></i>
                                                        </a>
                                                    <?php } else { ?>
                                                        <span class="text-muted">-</span>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <span class="badge badge-<?php echo $transaction['Status'] == 'Active' ? 'success' : 'secondary'; ?>">
                                                        <?php echo $transaction['Status']; ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <a href="?edit=<?php echo $transaction['ID']; ?>" class="btn btn-sm btn-primary" title="Edit">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <a href="?delete=<?php echo $transaction['ID']; ?>" 
                                                       onclick="return confirm('Are you sure to delete?')" 
                                                       class="btn btn-sm btn-danger" title="Delete">
                                                        <i class="fas fa-trash"></i>
                                                    </a>
                                                </td>
                                            </tr>
                                        <?php 
                                            }
                                        } else { 
                                        ?>
                                            <tr>
                                                <td colspan="11" class="text-center">No transactions found</td>
                                            </tr>
                                        <?php } ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <!-- Pagination -->
                            <?php if ($totalPages > 1) { ?>
                                <nav aria-label="Page navigation">
                                    <ul class="pagination justify-content-center">
                                        <?php if ($page > 1) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?php echo ($page-1); ?>&search=<?php echo urlencode($search); ?>&type=<?php echo urlencode($type_filter); ?>">Previous</a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php for ($i = 1; $i <= $totalPages; $i++) { ?>
                                            <li class="page-item <?php echo ($i == $page) ? 'active' : ''; ?>">
                                                <a class="page-link" href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&type=<?php echo urlencode($type_filter); ?>"><?php echo $i; ?></a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php if ($page < $totalPages) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?php echo ($page+1); ?>&search=<?php echo urlencode($search); ?>&type=<?php echo urlencode($type_filter); ?>">Next</a>
                                            </li>
                                        <?php } ?>
                                    </ul>
                                </nav>
                            <?php } ?>
                        </div>
                    </div>
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    <?php include_once('includes/footer2.php');?>
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
</body>
</html>
