<?php
session_start();

// Check if config file exists
$configPath = __DIR__ . '/includes/config.php';
if (!file_exists($configPath)) {
    die("Error: Database configuration file not found. Please create includes/config.php");
}

include_once($configPath);

// Check if database connection exists
if (!isset($con) || !$con) {
    die("Error: Database connection failed. Please check your database configuration.");
}
// Load branding helper
include_once(__DIR__ . '/includes/brand.php');

if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$msg = "";
$userId = $_SESSION['uid'];

// Filter variables
$dateFrom = isset($_GET['date_from']) ? mysqli_real_escape_string($con, $_GET['date_from']) : date('Y-m-01');
$dateTo = isset($_GET['date_to']) ? mysqli_real_escape_string($con, $_GET['date_to']) : date('Y-m-t');
$type_filter = isset($_GET['type']) ? mysqli_real_escape_string($con, $_GET['type']) : '';
$category_filter = isset($_GET['category']) ? mysqli_real_escape_string($con, $_GET['category']) : '';
$payment_method_filter = isset($_GET['payment_method']) ? mysqli_real_escape_string($con, $_GET['payment_method']) : '';
$status_filter = isset($_GET['status']) ? mysqli_real_escape_string($con, $_GET['status']) : 'Active';

// Pagination
$limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 25;
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// Build WHERE clause
$whereConditions = [];
$whereConditions[] = "TransactionDate BETWEEN '$dateFrom' AND '$dateTo'";

if (!empty($type_filter)) {
    $whereConditions[] = "Type = '$type_filter'";
}
if (!empty($category_filter)) {
    $whereConditions[] = "Category = '$category_filter'";
}
if (!empty($payment_method_filter)) {
    $whereConditions[] = "PaymentMethod = '$payment_method_filter'";
}
if (!empty($status_filter)) {
    $whereConditions[] = "Status = '$status_filter'";
}

$whereClause = "WHERE " . implode(" AND ", $whereConditions);

// Get categories for filter dropdown
$categoriesQuery = mysqli_query($con, "SELECT DISTINCT Category FROM tblincomeexpenses ORDER BY Category");
$categories = [];
while ($row = mysqli_fetch_assoc($categoriesQuery)) {
    $categories[] = $row['Category'];
}

// Get payment methods for filter dropdown
$paymentMethodsQuery = mysqli_query($con, "SELECT DISTINCT PaymentMethod FROM tblincomeexpenses WHERE PaymentMethod IS NOT NULL AND PaymentMethod != '' ORDER BY PaymentMethod");
$paymentMethods = [];
while ($row = mysqli_fetch_assoc($paymentMethodsQuery)) {
    $paymentMethods[] = $row['PaymentMethod'];
}

// Get summary statistics
$summaryQuery = "SELECT 
    SUM(CASE WHEN Type = 'Income' THEN Amount ELSE 0 END) as total_income,
    SUM(CASE WHEN Type = 'Expense' THEN Amount ELSE 0 END) as total_expense,
    COUNT(CASE WHEN Type = 'Income' THEN 1 END) as income_count,
    COUNT(CASE WHEN Type = 'Expense' THEN 1 END) as expense_count,
    COUNT(*) as total_transactions
    FROM tblincomeexpenses $whereClause";

$summaryResult = mysqli_query($con, $summaryQuery);
$summary = mysqli_fetch_assoc($summaryResult);

$totalIncome = $summary['total_income'] ?? 0;
$totalExpense = $summary['total_expense'] ?? 0;
$netAmount = $totalIncome - $totalExpense;
$incomeCount = $summary['income_count'] ?? 0;
$expenseCount = $summary['expense_count'] ?? 0;
$totalTransactions = $summary['total_transactions'] ?? 0;

// Get category-wise breakdown
$categoryBreakdownQuery = "SELECT 
    Category,
    Type,
    SUM(Amount) as total_amount,
    COUNT(*) as transaction_count
    FROM tblincomeexpenses $whereClause
    GROUP BY Category, Type
    ORDER BY Type, total_amount DESC";

$categoryBreakdown = mysqli_query($con, $categoryBreakdownQuery);

// Get payment method breakdown
$paymentBreakdownQuery = "SELECT 
    PaymentMethod,
    SUM(CASE WHEN Type = 'Income' THEN Amount ELSE 0 END) as income_amount,
    SUM(CASE WHEN Type = 'Expense' THEN Amount ELSE 0 END) as expense_amount,
    COUNT(*) as transaction_count
    FROM tblincomeexpenses $whereClause
    AND PaymentMethod IS NOT NULL AND PaymentMethod != ''
    GROUP BY PaymentMethod
    ORDER BY (income_amount + expense_amount) DESC";

$paymentBreakdown = mysqli_query($con, $paymentBreakdownQuery);

// Get daily trend data
$dailyTrendQuery = "SELECT 
    DATE(TransactionDate) as transaction_date,
    SUM(CASE WHEN Type = 'Income' THEN Amount ELSE 0 END) as daily_income,
    SUM(CASE WHEN Type = 'Expense' THEN Amount ELSE 0 END) as daily_expense
    FROM tblincomeexpenses $whereClause
    GROUP BY DATE(TransactionDate)
    ORDER BY transaction_date";

$dailyTrend = mysqli_query($con, $dailyTrendQuery);

// Get total count for pagination
$countQuery = "SELECT COUNT(*) as total FROM tblincomeexpenses $whereClause";
$countResult = mysqli_query($con, $countQuery);
$totalRecords = mysqli_fetch_assoc($countResult)['total'];
$totalPages = ceil($totalRecords / $limit);

// Fetch detailed transactions
$detailQuery = "SELECT * FROM tblincomeexpenses $whereClause ORDER BY TransactionDate DESC, CreatedDate DESC LIMIT $limit OFFSET $offset";
$transactions = mysqli_query($con, $detailQuery);

// Export functionality
if (isset($_GET['export']) && $_GET['export'] == 'csv') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="income_expense_report_' . $dateFrom . '_to_' . $dateTo . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // CSV Headers
    fputcsv($output, [
        'Transaction Code', 'Date', 'Type', 'Category', 'Description', 
        'Amount', 'Payment Method', 'Notes', 'Status', 'Created Date'
    ]);
    
    // Export all data (without pagination)
    $exportQuery = "SELECT * FROM tblincomeexpenses $whereClause ORDER BY TransactionDate DESC";
    $exportResult = mysqli_query($con, $exportQuery);
    
    while ($row = mysqli_fetch_assoc($exportResult)) {
        fputcsv($output, [
            $row['TransactionCode'],
            $row['TransactionDate'],
            $row['Type'],
            $row['Category'],
            $row['Description'],
            $row['Amount'],
            $row['PaymentMethod'],
            $row['Notes'],
            $row['Status'],
            $row['CreatedDate']
        ]);
    }
    
    fclose($output);
    exit();
}

// Opening balance: sum of all (Income - Expense) before $dateFrom
$openingQuery = "SELECT 
    SUM(CASE WHEN Type = 'Income' THEN Amount ELSE 0 END) -
    SUM(CASE WHEN Type = 'Expense' THEN Amount ELSE 0 END) AS opening_balance
    FROM tblincomeexpenses
    WHERE TransactionDate < '$dateFrom'";
$openingResult = mysqli_query($con, $openingQuery);
$openingBalance = mysqli_fetch_assoc($openingResult)['opening_balance'] ?? 0;

// Closing balance: opening + netAmount
$closingBalance = $openingBalance + $netAmount;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?php echo page_title('Income & Expense Report'); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/compact.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        .filter-container {
            background: #f8f9fc;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .summary-cards { margin-bottom: 20px; }
        .income-card { border-left: 4px solid #1cc88a; }
        .expense-card { border-left: 4px solid #e74a3b; }
        .net-card { border-left: 4px solid #36b9cc; }
        .transaction-card { border-left: 4px solid #f6c23e; }
        .chart-container { height: 400px; margin-bottom: 20px; }
        .breakdown-table { font-size: 0.875rem; }
        .print-section { page-break-inside: avoid; }
        .balance-card { border-left: 4px solid #4e73df; }
        .quick-filters .btn { margin-right: 5px; margin-bottom: 5px; }
        @media print {
            .no-print { display: none !important; }
            .print-section { page-break-inside: avoid; }
        }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid">
                    
                    <div class="d-sm-flex align-items-center justify-content-between mb-4">
                        <h1 class="h3 mb-0 text-gray-800">Income & Expense Report</h1>
                        <div class="no-print">
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['export' => 'csv'])); ?>" 
                               class="d-none d-sm-inline-block btn btn-sm btn-success shadow-sm">
                                <i class="fas fa-download fa-sm text-white-50"></i> Download CSV
                            </a>
                            <button onclick="window.print()" class="d-none d-sm-inline-block btn btn-sm btn-primary shadow-sm">
                                <i class="fas fa-print fa-sm text-white-50"></i> Print Report
                            </button>
                        </div>
                    </div>

                    <!-- Quick Filter Buttons -->
                    <div class="no-print mb-3 quick-filters">
                        <form method="GET" id="quickFilterForm">
                            <input type="hidden" name="date_from" id="qf_date_from">
                            <input type="hidden" name="date_to" id="qf_date_to">
                            <button type="button" class="btn btn-outline-primary btn-sm" onclick="setQuickFilter('today')">Today</button>
                            <button type="button" class="btn btn-outline-primary btn-sm" onclick="setQuickFilter('this_month')">This Month</button>
                            <button type="button" class="btn btn-outline-primary btn-sm" onclick="setQuickFilter('last_month')">Last Month</button>
                            <button type="button" class="btn btn-outline-primary btn-sm" onclick="setQuickFilter('this_year')">This Year</button>
                            <button type="button" class="btn btn-outline-secondary btn-sm" onclick="window.location='income-expense-report.php'">Clear</button>
                        </form>
                    </div>

                    <!-- Filter Section -->
                    <div class="filter-container no-print">
                        <form method="GET" class="row">
                            <div class="col-md-2">
                                <label>From Date</label>
                                <input type="date" name="date_from" class="form-control" value="<?php echo $dateFrom; ?>">
                            </div>
                            <div class="col-md-2">
                                <label>To Date</label>
                                <input type="date" name="date_to" class="form-control" value="<?php echo $dateTo; ?>">
                            </div>
                            <div class="col-md-2">
                                <label>Type</label>
                                <select name="type" class="form-control">
                                    <option value="">All Types</option>
                                    <option value="Income" <?php echo ($type_filter == 'Income') ? 'selected' : ''; ?>>Income</option>
                                    <option value="Expense" <?php echo ($type_filter == 'Expense') ? 'selected' : ''; ?>>Expense</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label>Category</label>
                                <select name="category" class="form-control">
                                    <option value="">All Categories</option>
                                    <?php foreach ($categories as $category) { ?>
                                        <option value="<?php echo $category; ?>" <?php echo ($category_filter == $category) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($category); ?>
                                        </option>
                                    <?php } ?>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label>Payment Method</label>
                                <select name="payment_method" class="form-control">
                                    <option value="">All Methods</option>
                                    <?php foreach ($paymentMethods as $method) { ?>
                                        <option value="<?php echo $method; ?>" <?php echo ($payment_method_filter == $method) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($method); ?>
                                        </option>
                                    <?php } ?>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label>Status</label>
                                <select name="status" class="form-control">
                                    <option value="">All Status</option>
                                    <option value="Active" <?php echo ($status_filter == 'Active') ? 'selected' : ''; ?>>Active</option>
                                    <option value="Inactive" <?php echo ($status_filter == 'Inactive') ? 'selected' : ''; ?>>Inactive</option>
                                </select>
                            </div>
                            <div class="col-md-12 mt-3">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-filter"></i> Apply Filters
                                </button>
                                <a href="income-expense-report.php" class="btn btn-secondary">
                                    <i class="fas fa-refresh"></i> Clear Filters
                                </a>
                                <div class="float-right">
                                    <label>Records per page:</label>
                                    <select name="limit" class="form-control d-inline-block" style="width: auto;" onchange="this.form.submit()">
                                        <option value="25" <?php echo ($limit == 25) ? 'selected' : ''; ?>>25</option>
                                        <option value="50" <?php echo ($limit == 50) ? 'selected' : ''; ?>>50</option>
                                        <option value="100" <?php echo ($limit == 100) ? 'selected' : ''; ?>>100</option>
                                        <option value="500" <?php echo ($limit == 500) ? 'selected' : ''; ?>>500</option>
                                    </select>
                                </div>
                            </div>
                        </form>
                    </div>

                    <!-- Report Period Info -->
                    <div class="print-section">
                        <div class="alert alert-info">
                            <strong>Report Period:</strong> <?php echo date('d M Y', strtotime($dateFrom)); ?> to <?php echo date('d M Y', strtotime($dateTo)); ?>
                            <?php if ($type_filter) echo " | Type: $type_filter"; ?>
                            <?php if ($category_filter) echo " | Category: $category_filter"; ?>
                            <?php if ($payment_method_filter) echo " | Payment: $payment_method_filter"; ?>
                        </div>
                    </div>

                    <!-- Opening/Closing Balance -->
                    <div class="row mb-3 print-section">
                        <div class="col-md-6">
                            <div class="card balance-card shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">Opening Balance</div>
                                    <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($openingBalance, 2); ?></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="card balance-card shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">Closing Balance</div>
                                    <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($closingBalance, 2); ?></div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Summary Cards -->
                    <div class="row summary-cards print-section">
                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card income-card shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">Total Income</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($totalIncome, 2); ?></div>
                                            <div class="text-xs text-gray-500"><?php echo $incomeCount; ?> transactions</div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-arrow-up fa-2x text-success"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card expense-card shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">Total Expenses</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($totalExpense, 2); ?></div>
                                            <div class="text-xs text-gray-500"><?php echo $expenseCount; ?> transactions</div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-arrow-down fa-2x text-danger"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card net-card shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">Net Amount</div>
                                            <div class="h5 mb-0 font-weight-bold <?php echo $netAmount >= 0 ? 'text-success' : 'text-danger'; ?>">
                                                ₹<?php echo number_format($netAmount, 2); ?>
                                            </div>
                                            <div class="text-xs text-gray-500">
                                                <?php echo $netAmount >= 0 ? 'Profit' : 'Loss'; ?>
                                            </div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-calculator fa-2x text-info"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card transaction-card shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">Total Transactions</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $totalTransactions; ?></div>
                                            <div class="text-xs text-gray-500">All transactions</div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-list fa-2x text-warning"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Charts Section -->
                    <div class="row no-print">
                        <div class="col-lg-6">
                            <div class="card shadow mb-4">
                                <div class="card-header py-3">
                                    <h6 class="m-0 font-weight-bold text-primary">Income vs Expense</h6>
                                </div>
                                <div class="card-body">
                                    <canvas id="pieChart" height="200"></canvas>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-6">
                            <div class="card shadow mb-4">
                                <div class="card-header py-3">
                                    <h6 class="m-0 font-weight-bold text-primary">Daily Trend</h6>
                                </div>
                                <div class="card-body">
                                    <canvas id="lineChart" height="200"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Category Breakdown -->
                    <div class="row print-section">
                        <div class="col-lg-6">
                            <div class="card shadow mb-4">
                                <div class="card-header py-3">
                                    <h6 class="m-0 font-weight-bold text-primary">Category Breakdown</h6>
                                </div>
                                <div class="card-body">
                                    <div class="table-responsive">
                                        <table class="table table-sm breakdown-table">
                                            <thead>
                                                <tr>
                                                    <th>Category</th>
                                                    <th>Type</th>
                                                    <th>Amount</th>
                                                    <th>Count</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php while ($row = mysqli_fetch_assoc($categoryBreakdown)) { ?>
                                                    <tr>
                                                        <td><?php echo htmlspecialchars($row['Category']); ?></td>
                                                        <td>
                                                            <span class="badge badge-<?php echo $row['Type'] == 'Income' ? 'success' : 'danger'; ?> badge-sm">
                                                                <?php echo $row['Type']; ?>
                                                            </span>
                                                        </td>
                                                        <td class="<?php echo $row['Type'] == 'Income' ? 'text-success' : 'text-danger'; ?>">
                                                            ₹<?php echo number_format($row['total_amount'], 2); ?>
                                                        </td>
                                                        <td><?php echo $row['transaction_count']; ?></td>
                                                    </tr>
                                                <?php } ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-6">
                            <div class="card shadow mb-4">
                                <div class="card-header py-3">
                                    <h6 class="m-0 font-weight-bold text-primary">Payment Method Breakdown</h6>
                                </div>
                                <div class="card-body">
                                    <div class="table-responsive">
                                        <table class="table table-sm breakdown-table">
                                            <thead>
                                                <tr>
                                                    <th>Payment Method</th>
                                                    <th>Income</th>
                                                    <th>Expense</th>
                                                    <th>Count</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php while ($row = mysqli_fetch_assoc($paymentBreakdown)) { ?>
                                                    <tr>
                                                        <td><?php echo htmlspecialchars($row['PaymentMethod']); ?></td>
                                                        <td class="text-success">₹<?php echo number_format($row['income_amount'], 2); ?></td>
                                                        <td class="text-danger">₹<?php echo number_format($row['expense_amount'], 2); ?></td>
                                                        <td><?php echo $row['transaction_count']; ?></td>
                                                    </tr>
                                                <?php } ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Detailed Transactions -->
                    <div class="card shadow mb-4 print-section">
                        <div class="card-header py-3 d-flex justify-content-between align-items-center">
                            <h6 class="m-0 font-weight-bold text-primary">Detailed Transactions</h6>
                            <small class="text-muted">Showing <?php echo min($limit, $totalRecords); ?> of <?php echo $totalRecords; ?> records</small>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered table-sm">
                                    <thead>
                                        <tr>
                                            <th>No</th>
                                            <th>Date</th>
                                            <th>Code</th>
                                            <th>Type</th>
                                            <th>Category</th>
                                            <th>Description</th>
                                            <th>Amount</th>
                                            <th>Payment</th>
                                            <th class="no-print">Receipt</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php 
                                        if (mysqli_num_rows($transactions) > 0) {
                                            $i = $offset + 1; 
                                            while($transaction = mysqli_fetch_assoc($transactions)) { 
                                        ?>
                                            <tr>
                                                <td><?php echo $i++; ?></td>
                                                <td><?php echo date('d-m-Y', strtotime($transaction['TransactionDate'])); ?></td>
                                                <td><small><?php echo htmlspecialchars($transaction['TransactionCode']); ?></small></td>
                                                <td>
                                                    <span class="badge badge-<?php echo $transaction['Type'] == 'Income' ? 'success' : 'danger'; ?> badge-sm">
                                                        <?php echo $transaction['Type']; ?>
                                                    </span>
                                                </td>
                                                <td><small><?php echo htmlspecialchars($transaction['Category']); ?></small></td>
                                                <td><small><?php echo htmlspecialchars(substr($transaction['Description'], 0, 30)) . (strlen($transaction['Description']) > 30 ? '...' : ''); ?></small></td>
                                                <td class="<?php echo $transaction['Type'] == 'Income' ? 'text-success' : 'text-danger'; ?>">
                                                    <strong>₹<?php echo number_format($transaction['Amount'], 2); ?></strong>
                                                </td>
                                                <td><small><?php echo htmlspecialchars($transaction['PaymentMethod']); ?></small></td>
                                                <td class="no-print">
                                                    <?php if($transaction['Receipt']) { ?>
                                                        <a href="<?php echo $transaction['Receipt']; ?>" target="_blank" class="btn btn-xs btn-info">
                                                            <i class="fas fa-file"></i>
                                                        </a>
                                                    <?php } else { ?>
                                                        <span class="text-muted">-</span>
                                                    <?php } ?>
                                                </td>
                                            </tr>
                                        <?php 
                                            }
                                        } else { 
                                        ?>
                                            <tr>
                                                <td colspan="9" class="text-center">No transactions found for the selected criteria</td>
                                            </tr>
                                        <?php } ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <!-- Pagination -->
                            <?php if ($totalPages > 1) { ?>
                                <nav aria-label="Page navigation" class="no-print">
                                    <ul class="pagination justify-content-center">
                                        <?php if ($page > 1) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page-1])); ?>">Previous</a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php 
                                        $start = max(1, $page - 2);
                                        $end = min($totalPages, $page + 2);
                                        for ($i = $start; $i <= $end; $i++) { 
                                        ?>
                                            <li class="page-item <?php echo ($i == $page) ? 'active' : ''; ?>">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>"><?php echo $i; ?></a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php if ($page < $totalPages) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page+1])); ?>">Next</a>
                                            </li>
                                        <?php } ?>
                                    </ul>
                                </nav>
                            <?php } ?>
                        </div>
                    </div>
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    <?php include_once('includes/footer2.php');?>

    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>

    <script>
        // Pie Chart for Income vs Expense
        const pieCtx = document.getElementById('pieChart').getContext('2d');
        new Chart(pieCtx, {
            type: 'doughnut',
            data: {
                labels: ['Income', 'Expense'],
                datasets: [{
                    data: [<?php echo $totalIncome; ?>, <?php echo $totalExpense; ?>],
                    backgroundColor: ['#1cc88a', '#e74a3b'],
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });

        // Line Chart for Daily Trend
        const lineCtx = document.getElementById('lineChart').getContext('2d');
        const dailyData = [
            <?php 
            mysqli_data_seek($dailyTrend, 0);
            $chartLabels = [];
            $chartIncome = [];
            $chartExpense = [];
            while ($row = mysqli_fetch_assoc($dailyTrend)) {
                $chartLabels[] = "'" . date('M d', strtotime($row['transaction_date'])) . "'";
                $chartIncome[] = $row['daily_income'];
                $chartExpense[] = $row['daily_expense'];
            }
            ?>
        ];
        
        new Chart(lineCtx, {
            type: 'line',
            data: {
                labels: [<?php echo implode(',', $chartLabels); ?>],
                datasets: [{
                    label: 'Income',
                    data: [<?php echo implode(',', $chartIncome); ?>],
                    borderColor: '#1cc88a',
                    backgroundColor: 'rgba(28, 200, 138, 0.1)',
                    tension: 0.1
                }, {
                    label: 'Expense',
                    data: [<?php echo implode(',', $chartExpense); ?>],
                    borderColor: '#e74a3b',
                    backgroundColor: 'rgba(231, 74, 59, 0.1)',
                    tension: 0.1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });

        // Quick Filter JS
        function setQuickFilter(type) {
            const today = new Date();
            let from, to;
            if (type === 'today') {
                from = to = today.toISOString().slice(0,10);
            } else if (type === 'this_month') {
                from = new Date(today.getFullYear(), today.getMonth(), 1).toISOString().slice(0,10);
                to = new Date(today.getFullYear(), today.getMonth() + 1, 0).toISOString().slice(0,10);
            } else if (type === 'last_month') {
                let first = new Date(today.getFullYear(), today.getMonth() - 1, 1);
                let last = new Date(today.getFullYear(), today.getMonth(), 0);
                from = first.toISOString().slice(0,10);
                to = last.toISOString().slice(0,10);
            } else if (type === 'this_year') {
                from = new Date(today.getFullYear(), 0, 1).toISOString().slice(0,10);
                to = new Date(today.getFullYear(), 11, 31).toISOString().slice(0,10);
            }
            document.getElementById('qf_date_from').value = from;
            document.getElementById('qf_date_to').value = to;
            document.getElementById('quickFilterForm').submit();
        }
    </script>
</body>
</html>
