<?php
session_start();

// Check if config file exists
$configPath = __DIR__ . '/includes/config.php';
if (!file_exists($configPath)) {
    die("Error: Database configuration file not found. Please create includes/config.php");
}

include_once($configPath);

// Check if database connection exists
if (!isset($con) || !$con) {
    die("Error: Database connection failed. Please check your database configuration.");
}
// Load branding helper
include_once(__DIR__ . '/includes/brand.php');

if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$msg = "";
$userId = $_SESSION['uid'];

// Get user permissions
$userQuery = mysqli_query($con, "SELECT UserType, CanEdit FROM tblusers WHERE ID = $userId");
$userData = mysqli_fetch_assoc($userQuery);
$canEdit = ($userData['UserType'] == 'admin' || $userData['CanEdit'] == 1);

// Filter variables
$dateFrom = isset($_GET['date_from']) ? mysqli_real_escape_string($con, $_GET['date_from']) : '';
$dateTo = isset($_GET['date_to']) ? mysqli_real_escape_string($con, $_GET['date_to']) : '';
$paymentMethod = isset($_GET['payment_method']) ? mysqli_real_escape_string($con, $_GET['payment_method']) : '';
$customerSearch = isset($_GET['customer_search']) ? mysqli_real_escape_string($con, $_GET['customer_search']) : '';
$amountFrom = isset($_GET['amount_from']) ? (float)$_GET['amount_from'] : 0;
$amountTo = isset($_GET['amount_to']) ? (float)$_GET['amount_to'] : 0;
$sortBy = isset($_GET['sort_by']) ? mysqli_real_escape_string($con, $_GET['sort_by']) : 'pc.CreatedDate';
$sortOrder = isset($_GET['sort_order']) ? mysqli_real_escape_string($con, $_GET['sort_order']) : 'DESC';
$showPendingOnly = isset($_GET['show_pending_only']) ? true : false;
$showHighValue = isset($_GET['show_high_value']) ? true : false;

// Pagination
$limit = isset($_GET['records_per_page']) ? (int)$_GET['records_per_page'] : 25;
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// Enhanced function to calculate interest and due amount - Daily Interest Calculation with proper rounding
function calculateDueAmount($loanDate, $dueDate, $loanAmount, $interestRate, $collectionDate = null) {
    if (!$collectionDate) {
        $collectionDate = date('Y-m-d');
    }
    
    $loanDateTime = new DateTime($loanDate);
    $dueDatetime = new DateTime($dueDate);
    $collectionDateTime = new DateTime($collectionDate);
    
    $totalDays = $loanDateTime->diff($collectionDateTime)->days;
    $daysOverdue = max(0, $collectionDateTime->diff($dueDatetime)->days);
    if ($collectionDateTime < $dueDatetime) {
        $daysOverdue = 0;
    }
    
    // Calculate daily interest rate from annual rate (fixed 15% per annum)
    $annualRate = 0.15; // 15% p.a.
    $dailyRate = $annualRate / 365;
    
    // Calculate interest amounts based on daily rate with proper rounding
    $dailyInterestAmount = round($loanAmount * $dailyRate, 2); // Round daily interest
    $totalInterest = round($dailyInterestAmount * $totalDays, 2); // Round total interest
    $monthlyInterestAmount = round($loanAmount * ($annualRate / 12), 2); // Round monthly interest
    
    $totalDue = round($loanAmount + $totalInterest, 2); // Round total due
    
    return [
        'totalDays' => $totalDays,
        'daysOverdue' => $daysOverdue,
        'dailyInterestAmount' => $dailyInterestAmount, // Rounded daily interest
        'monthlyInterestAmount' => $monthlyInterestAmount, // Rounded monthly interest
        'interestAmount' => $totalInterest, // Rounded total interest
        'totalDue' => $totalDue, // Rounded total due
        'dailyRate' => $dailyRate
    ];
}

// Handle bulk operations
if ($_POST) {
    if (isset($_POST['bulk_action']) && isset($_POST['selected_records']) && $canEdit) {
        $selectedIds = $_POST['selected_records'];
        $bulkAction = $_POST['bulk_action'];
        
        if (!empty($selectedIds) && is_array($selectedIds)) {
            $ids = implode(',', array_map('intval', $selectedIds));
            
            switch($bulkAction) {
                case 'mark_verified':
                    $updateQuery = "UPDATE tblpaymentcollections SET Notes = CONCAT(COALESCE(Notes, ''), ' [Verified by Admin]') WHERE ID IN ($ids)";
                    if (mysqli_query($con, $updateQuery)) {
                        $msg = "Selected payments verified";
                    }
                    break;
                    
                case 'lock_edit':
                    $updateQuery = "UPDATE tblpaymentcollections SET IsEditable = 0 WHERE ID IN ($ids)";
                    if (mysqli_query($con, $updateQuery)) {
                        $msg = "Selected payments locked from editing";
                    }
                    break;
                    
                case 'unlock_edit':
                    $updateQuery = "UPDATE tblpaymentcollections SET IsEditable = 1 WHERE ID IN ($ids)";
                    if (mysqli_query($con, $updateQuery)) {
                        $msg = "Selected payments unlocked for editing";
                    }
                    break;
                    
                case 'delete':
                    $deleteQuery = "DELETE FROM tblpaymentcollections WHERE ID IN ($ids) AND IsEditable = 1";
                    if (mysqli_query($con, $deleteQuery)) {
                        $affectedRows = mysqli_affected_rows($con);
                        $msg = "$affectedRows payment records deleted";
                    }
                    break;
            }
        }
    }
}

// Build WHERE clause for filtering
$whereConditions = ["1=1"]; // Always true condition to simplify query building

if (!empty($dateFrom)) {
    $whereConditions[] = "pc.CollectionDate >= '$dateFrom'";
}
if (!empty($dateTo)) {
    $whereConditions[] = "pc.CollectionDate <= '$dateTo'";
}
if (!empty($paymentMethod)) {
    $whereConditions[] = "pc.PaymentMethod = '$paymentMethod'";
}
if (!empty($customerSearch)) {
    $whereConditions[] = "(c.Name LIKE '%$customerSearch%' OR c.MobileNumber LIKE '%$customerSearch%' OR c.CustomerCode LIKE '%$customerSearch%' OR p.PawnCode LIKE '%$customerSearch%')";
}
if ($amountFrom > 0) {
    $whereConditions[] = "pc.AmountPaid >= $amountFrom";
}
if ($amountTo > 0) {
    $whereConditions[] = "pc.AmountPaid <= $amountTo";
}
if ($showPendingOnly) {
    $whereConditions[] = "p.Status IN ('Active', 'Extended')";
}
if ($showHighValue) {
    $whereConditions[] = "pc.AmountPaid >= 10000";
}

$whereClause = "WHERE " . implode(" AND ", $whereConditions);

// Get total count for pagination
$countQuery = "SELECT COUNT(*) as total FROM tblpaymentcollections pc 
               JOIN tblpawntransactions p ON pc.PawnTransactionID = p.ID
               JOIN tblcustomers c ON pc.CustomerID = c.ID $whereClause";
$countResult = mysqli_query($con, $countQuery);
$totalRecords = mysqli_fetch_assoc($countResult)['total'];
$totalPages = ceil($totalRecords / $limit);

// Get summary statistics
$summaryQuery = "SELECT 
    COALESCE(COUNT(*), 0) as total_collections,
    COALESCE(SUM(pc.AmountPaid), 0) as total_amount_collected,
    COALESCE(SUM(pc.DiscountAmount), 0) as total_discounts,
    COALESCE(SUM(pc.AmountPaid + pc.DiscountAmount), 0) as total_value,
    COALESCE(AVG(pc.AmountPaid), 0) as average_payment,
    COALESCE(COUNT(CASE WHEN pc.PaymentMethod = 'Cash' THEN 1 END), 0) as cash_payments,
    COALESCE(COUNT(CASE WHEN pc.PaymentMethod = 'Bank Transfer' THEN 1 END), 0) as bank_payments,
    COALESCE(COUNT(CASE WHEN pc.PaymentMethod = 'UPI' THEN 1 END), 0) as upi_payments,
    COALESCE(COUNT(CASE WHEN pc.AmountPaid >= 10000 THEN 1 END), 0) as high_value_payments,
    COALESCE(COUNT(CASE WHEN pc.IsEditable = 0 THEN 1 END), 0) as locked_payments
    FROM tblpaymentcollections pc 
    JOIN tblpawntransactions p ON pc.PawnTransactionID = p.ID
    JOIN tblcustomers c ON pc.CustomerID = c.ID $whereClause";
$summaryResult = mysqli_query($con, $summaryQuery);
$summary = mysqli_fetch_assoc($summaryResult);

// Ensure all summary values are properly set with defaults
$summary['total_collections'] = (int)($summary['total_collections'] ?? 0);
$summary['total_amount_collected'] = (float)($summary['total_amount_collected'] ?? 0);
$summary['total_discounts'] = (float)($summary['total_discounts'] ?? 0);
$summary['total_value'] = (float)($summary['total_value'] ?? 0);
$summary['average_payment'] = (float)($summary['average_payment'] ?? 0);
$summary['cash_payments'] = (int)($summary['cash_payments'] ?? 0);
$summary['bank_payments'] = (int)($summary['bank_payments'] ?? 0);
$summary['upi_payments'] = (int)($summary['upi_payments'] ?? 0);
$summary['high_value_payments'] = (int)($summary['high_value_payments'] ?? 0);
$summary['locked_payments'] = (int)($summary['locked_payments'] ?? 0);

// Fetch detailed data
$mainQuery = "SELECT pc.*, p.PawnCode, p.ItemType, p.ItemDescription, p.LoanAmount, p.LoanDate, p.DueDate, p.Status as PawnStatus,
              COALESCE(p.CurrentLoanAmount, p.LoanAmount) as ActualLoanAmount,
              c.Name as CustomerName, c.MobileNumber, c.CustomerCode,
              u.FullName as CreatedByName, eu.FullName as EditedByName,
              DATEDIFF(pc.CollectionDate, p.DueDate) as DaysFromDue
              FROM tblpaymentcollections pc 
              JOIN tblpawntransactions p ON pc.PawnTransactionID = p.ID
              JOIN tblcustomers c ON pc.CustomerID = c.ID 
              LEFT JOIN tblusers u ON pc.CreatedBy = u.ID
              LEFT JOIN tblusers eu ON pc.EditedBy = eu.ID
              $whereClause 
              ORDER BY $sortBy $sortOrder 
              LIMIT $limit OFFSET $offset";
$detailResult = mysqli_query($con, $mainQuery);

// Get filter options
$paymentMethodOptions = ['Cash', 'Bank Transfer', 'Card', 'Cheque', 'UPI'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?php echo page_title('Detailed Collection Report'); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/compact.css" rel="stylesheet">
    <style>
        .filter-section {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 25px;
            border-radius: 15px;
            margin-bottom: 25px;
            color: white;
        }
        .summary-cards {
            margin-bottom: 25px;
        }
        .summary-card {
            background: linear-gradient(45deg, #28a745, #20c997);
            color: white;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 15px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }
        .cash-card { background: linear-gradient(45deg, #28a745, #20c997); }
        .bank-card { background: linear-gradient(45deg, #007bff, #0056b3); }
        .digital-card { background: linear-gradient(45deg, #6f42c1, #5a32a3); }
        .high-value-card { background: linear-gradient(45deg, #fd7e14, #e55a00); }
        
        .bulk-actions {
            background: #e8f4f8;
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
            border-left: 4px solid #17a2b8;
        }
        .action-buttons {
            white-space: nowrap;
        }
        .table-responsive {
            max-height: 70vh;
            overflow-y: auto;
        }
        .verified-payment {
            background-color: #d4edda !important;
        }
        .locked-payment {
            background-color: #f8d7da !important;
        }
        .high-value-payment {
            background-color: #fff3cd !important;
        }
        .discount-payment {
            background-color: #e2e3e5 !important;
        }
        @media print {
            .no-print { display: none !important; }
            .table { font-size: 11px; }
            .summary-card { 
                background: #f8f9fa !important; 
                color: #333 !important; 
                border: 1px solid #dee2e6 !important; 
            }
        }
        .quick-stats {
            display: flex;
            justify-content: space-around;
            margin-top: 15px;
        }
        .quick-stat {
            text-align: center;
            background: rgba(255,255,255,0.1);
            padding: 10px;
            border-radius: 8px;
            min-width: 120px;
        }
        .stat-value {
            font-size: 1.2em;
            font-weight: bold;
        }
        .stat-label {
            font-size: 0.8em;
            opacity: 0.9;
        }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid">
                    
                    <?php if($msg) { ?>
                        <div class="alert alert-info alert-dismissible fade show">
                            <?php echo $msg; ?>
                            <button type="button" class="close" data-dismiss="alert">&times;</button>
                        </div>
                    <?php } ?>
                    
                    <!-- Summary Cards -->
                    <div class="summary-cards no-print">
                        <div class="row">
                            <div class="col-xl-3 col-md-6 mb-4">
                                <div class="summary-card">
                                    <div class="card-body">
                                        <div class="row no-gutters align-items-center">
                                            <div class="col mr-2">
                                                <div class="text-xs font-weight-bold text-white text-uppercase mb-1">Total Collections</div>
                                                <div class="h5 mb-0 font-weight-bold text-white"><?php echo number_format($summary['total_collections']); ?></div>
                                                <div class="text-xs text-white">₹<?php echo number_format($summary['total_amount_collected'], 2); ?></div>
                                            </div>
                                            <div class="col-auto">
                                                <i class="fas fa-money-bill-wave fa-2x text-white"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-xl-3 col-md-6 mb-4">
                                <div class="summary-card cash-card">
                                    <div class="card-body">
                                        <div class="row no-gutters align-items-center">
                                            <div class="col mr-2">
                                                <div class="text-xs font-weight-bold text-white text-uppercase mb-1">Cash Collections</div>
                                                <div class="h5 mb-0 font-weight-bold text-white"><?php echo number_format($summary['cash_payments']); ?></div>
                                                <div class="text-xs text-white"><?php echo $summary['total_collections'] > 0 ? round(($summary['cash_payments']/$summary['total_collections'])*100, 1) : 0; ?>%</div>
                                            </div>
                                            <div class="col-auto">
                                                <i class="fas fa-coins fa-2x text-white"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-xl-3 col-md-6 mb-4">
                                <div class="summary-card digital-card">
                                    <div class="card-body">
                                        <div class="row no-gutters align-items-center">
                                            <div class="col mr-2">
                                                <div class="text-xs font-weight-bold text-white text-uppercase mb-1">Digital Collections</div>
                                                <div class="h5 mb-0 font-weight-bold text-white"><?php echo number_format($summary['bank_payments'] + $summary['upi_payments']); ?></div>
                                                <div class="text-xs text-white"><?php echo $summary['total_collections'] > 0 ? round((($summary['bank_payments'] + $summary['upi_payments'])/$summary['total_collections'])*100, 1) : 0; ?>%</div>
                                            </div>
                                            <div class="col-auto">
                                                <i class="fas fa-mobile-alt fa-2x text-white"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-xl-3 col-md-6 mb-4">
                                <div class="summary-card high-value-card">
                                    <div class="card-body">
                                        <div class="row no-gutters align-items-center">
                                            <div class="col mr-2">
                                                <div class="text-xs font-weight-bold text-white text-uppercase mb-1">High Value Collections</div>
                                                <div class="h5 mb-0 font-weight-bold text-white"><?php echo number_format($summary['high_value_payments']); ?></div>
                                                <div class="text-xs text-white">₹10,000+ Collected</div>
                                            </div>
                                            <div class="col-auto">
                                                <i class="fas fa-gem fa-2x text-white"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Quick Statistics -->
                        <div class="row">
                            <div class="col-12">
                                <div class="summary-card">
                                    <h6 class="text-white mb-3"><i class="fas fa-chart-bar"></i> Quick Stats</h6>
                                    <div class="quick-stats">
                                        <div class="quick-stat">
                                            <div class="stat-value">₹<?php echo number_format($summary['average_payment'], 0); ?></div>
                                            <div class="stat-label">Average Collection</div>
                                        </div>
                                        <div class="quick-stat">
                                            <div class="stat-value">₹<?php echo number_format($summary['total_discounts'], 0); ?></div>
                                            <div class="stat-label">Total Discount</div>
                                        </div>
                                        <div class="quick-stat">
                                            <div class="stat-value">₹<?php echo number_format($summary['total_value'], 0); ?></div>
                                            <div class="stat-label">Total Value</div>
                                        </div>
                                        <div class="quick-stat">
                                            <div class="stat-value"><?php echo number_format($summary['locked_payments']); ?></div>
                                            <div class="stat-label">Locked</div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Advanced Filters -->
                    <div class="filter-section no-print">
                        <h5 class="mb-3"><i class="fas fa-filter"></i> Advanced Filters</h5>
                        <form method="GET" id="filterForm">
                            <div class="row">
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label class="text-white">Start Date</label>
                                        <input type="date" name="date_from" class="form-control" value="<?php echo $dateFrom; ?>">
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label class="text-white">End Date</label>
                                        <input type="date" name="date_to" class="form-control" value="<?php echo $dateTo; ?>">
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label class="text-white">Payment Type</label>
                                        <select name="payment_method" class="form-control">
                                            <option value="">All Types</option>
                                            <?php foreach($paymentMethodOptions as $method) { ?>
                                                <option value="<?php echo $method; ?>" <?php echo ($paymentMethod == $method) ? 'selected' : ''; ?>><?php echo $method; ?></option>
                                            <?php } ?>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label class="text-white">Customer Search</label>
                                        <input type="text" name="customer_search" class="form-control" 
                                               placeholder="Name, Mobile or Pawn Code" value="<?php echo $customerSearch; ?>">
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label class="text-white">Min Amount</label>
                                        <input type="number" name="amount_from" class="form-control" value="<?php echo $amountFrom; ?>">
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label class="text-white">Max Amount</label>
                                        <input type="number" name="amount_to" class="form-control" value="<?php echo $amountTo; ?>">
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label class="text-white">Sort By</label>
                                        <select name="sort_by" class="form-control">
                                            <option value="pc.CreatedDate" <?php echo ($sortBy == 'pc.CreatedDate') ? 'selected' : ''; ?>>Created Date</option>
                                            <option value="pc.CollectionDate" <?php echo ($sortBy == 'pc.CollectionDate') ? 'selected' : ''; ?>>Collection Date</option>
                                            <option value="pc.AmountPaid" <?php echo ($sortBy == 'pc.AmountPaid') ? 'selected' : ''; ?>>Amount Paid</option>
                                            <option value="c.Name" <?php echo ($sortBy == 'c.Name') ? 'selected' : ''; ?>>Customer Name</option>
                                            <option value="p.PawnCode" <?php echo ($sortBy == 'p.PawnCode') ? 'selected' : ''; ?>>Pawn Code</option>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label class="text-white">Sort Order</label>
                                        <select name="sort_order" class="form-control">
                                            <option value="DESC" <?php echo ($sortOrder == 'DESC') ? 'selected' : ''; ?>>Descending</option>
                                            <option value="ASC" <?php echo ($sortOrder == 'ASC') ? 'selected' : ''; ?>>Ascending</option>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <label class="text-white">Records Per Page</label>
                                        <select name="records_per_page" class="form-control">
                                            <option value="25" <?php echo ($limit == 25) ? 'selected' : ''; ?>>25</option>
                                            <option value="50" <?php echo ($limit == 50) ? 'selected' : ''; ?>>50</option>
                                            <option value="100" <?php echo ($limit == 100) ? 'selected' : ''; ?>>100</option>
                                            <option value="500" <?php echo ($limit == 500) ? 'selected' : ''; ?>>500</option>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-2">
                                    <div class="form-group">
                                        <div class="form-check mt-4">
                                            <input type="checkbox" name="show_pending_only" class="form-check-input" 
                                                   <?php echo $showPendingOnly ? 'checked' : ''; ?>>
                                            <label class="form-check-label text-white">Pending Only</label>
                                        </div>
                                        <div class="form-check">
                                            <input type="checkbox" name="show_high_value" class="form-check-input" 
                                                   <?php echo $showHighValue ? 'checked' : ''; ?>>
                                            <label class="form-check-label text-white">High Value</label>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-12">
                                    <button type="submit" class="btn btn-light">
                                        <i class="fas fa-search"></i> Filter
                                    </button>
                                    <a href="due_collection_report_detailed.php" class="btn btn-outline-light ml-2">
                                        <i class="fas fa-refresh"></i> Reset
                                    </a>
                                    <button type="button" onclick="window.print()" class="btn btn-outline-light ml-2">
                                        <i class="fas fa-print"></i> Print
                                    </button>
                                    <button type="button" onclick="exportToCSV()" class="btn btn-outline-light ml-2">
                                        <i class="fas fa-download"></i> Export CSV
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Bulk Actions -->
                    <?php if ($canEdit) { ?>
                    <div class="bulk-actions no-print">
                        <form method="POST" id="bulkForm">
                            <div class="row align-items-end">
                                <div class="col-md-3">
                                    <label><strong>Bulk Action</strong></label>
                                    <select name="bulk_action" class="form-control" required>
                                        <option value="">Select Action</option>
                                        <option value="mark_verified">Mark as Verified</option>
                                        <option value="lock_edit">Lock Editing</option>
                                        <option value="unlock_edit">Unlock Editing</option>
                                        <option value="delete" style="color: red;">Delete</option>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <button type="submit" class="btn btn-warning" onclick="return confirmBulkAction()">
                                        <i class="fas fa-tasks"></i> Apply to Selected
                                    </button>
                                </div>
                                <div class="col-md-6 text-right">
                                    <span id="selectedCount">0</span> records selected
                                </div>
                            </div>
                        </form>
                    </div>
                    <?php } ?>
                    
                    <!-- Detailed Report Table -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3 d-flex justify-content-between align-items-center">
                            <h6 class="m-0 font-weight-bold text-primary">Detailed Collection Report</h6>
                            <small class="text-muted">Total: <?php echo $totalRecords; ?> records</small>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered table-hover" id="dataTable">
                                    <thead class="thead-light">
                                        <tr>
                                            <?php if ($canEdit) { ?><th class="no-print"><input type="checkbox" id="selectAll"></th><?php } ?>
                                            <th>No</th>
                                            <th>Code</th>
                                            <th>Pawn Code</th>
                                            <th>Customer</th>
                                            <th>Item</th>
                                            <th>Collection Date</th>
                                            <th>Amount Paid</th>
                                            <th>Discount</th>
                                            <th>Total</th>
                                            <th>Method</th>
                                            <th>Days</th>
                                            <th>Status</th>
                                            <th class="no-print">Actions</th>
                                        </tr>
                                                    // Calculate daily interest rate from annual rate (fixed 15% per annum)
                                                    $annualRate = 0.15; // 15% p.a.
                                                    $dailyRate = $annualRate / 365;
        
                                                    // Per-day and total interest
                                                    $dailyInterestAmount = $loanAmount * $dailyRate; // one-day interest
                                                    $totalInterest = $dailyInterestAmount * $totalDays;
                                                    $monthlyInterestAmount = $loanAmount * ($annualRate / 12); // For reference
                                                $statusBadge = 'secondary';
                                                $statusText = 'Normal';
                                                
                                                if ($row['Notes'] && strpos($row['Notes'], '[Verified by Admin]') !== false) {
                                                    $rowClass = 'verified-payment';
                                                    $statusBadge = 'success';
                                                    $statusText = 'Verified';
                                                } elseif (!$row['IsEditable']) {
                                                    $rowClass = 'locked-payment';
                                                    $statusBadge = 'danger';
                                                    $statusText = 'Locked';
                                                } elseif ($row['AmountPaid'] >= 10000) {
                                                    $rowClass = 'high-value-payment';
                                                    $statusBadge = 'warning';
                                                    $statusText = 'High Value';
                                                } elseif ($row['DiscountAmount'] > 0) {
                                                    $rowClass = 'discount-payment';
                                                    $statusBadge = 'info';
                                                    $statusText = 'Discounted';
                                                }
                                                
                                                $totalCollection = $row['AmountPaid'] + $row['DiscountAmount'];
                                        ?>
                                            <tr class="<?php echo $rowClass; ?>">
                                                <?php if ($canEdit) { ?>
                                                <td class="no-print">
                                                    <input type="checkbox" name="selected_records[]" value="<?php echo $row['ID']; ?>" class="record-checkbox">
                                                </td>
                                                <?php } ?>
                                                <td><?php echo $i++; ?></td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($row['CollectionCode']); ?></strong>
                                                    <?php if ($row['EditedBy']) { ?>
                                                        <br><small class="text-info"><i class="fas fa-edit"></i> Edited</small>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($row['PawnCode']); ?></strong>
                                                    <br><span class="badge badge-secondary"><?php echo htmlspecialchars($row['ItemType']); ?></span>
                                                </td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($row['CustomerName']); ?></strong><br>
                                                    <small><?php echo htmlspecialchars($row['CustomerCode']); ?></small><br>
                                                    <small><?php echo htmlspecialchars($row['MobileNumber']); ?></small>
                                                </td>
                                                <td>
                                                    <small><?php echo htmlspecialchars(substr($row['ItemDescription'], 0, 30)) . '...'; ?></small><br>
                                                    <span class="badge badge-primary">₹<?php echo number_format($row['ActualLoanAmount'], 0); ?></span>
                                                </td>
                                                <td>
                                                    <?php echo date('d/m/Y', strtotime($row['CollectionDate'])); ?>
                                                    <br><small class="text-muted"><?php echo date('H:i', strtotime($row['CreatedDate'])); ?></small>
                                                </td>
                                                <td>
                                                    <strong>₹<?php echo number_format($row['AmountPaid'], 2); ?></strong>
                                                    <?php if ($row['PaymentMethod'] != 'Cash') { ?>
                                                        <br><small class="badge badge-info"><?php echo htmlspecialchars($row['PaymentMethod']); ?></small>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <?php if ($row['DiscountAmount'] > 0) { ?>
                                                        <span class="text-success">₹<?php echo number_format($row['DiscountAmount'], 2); ?></span>
                                                    <?php } else { ?>
                                                        <span class="text-muted">-</span>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <strong class="text-primary">₹<?php echo number_format($totalCollection, 2); ?></strong>
                                                </td>
                                                <td>
                                                    <span class="badge badge-<?php 
                                                        echo $row['PaymentMethod'] == 'Cash' ? 'success' : 
                                                            ($row['PaymentMethod'] == 'UPI' ? 'primary' : 'info'); 
                                                    ?>">
                                                        <?php echo htmlspecialchars($row['PaymentMethod']); ?>
                                                    </span>
                                                    <?php if ($row['ReferenceNumber']) { ?>
                                                        <br><small>Ref: <?php echo htmlspecialchars($row['ReferenceNumber']); ?></small>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <small>Due: <?php echo $row['InterestDays']; ?> days</small><br>
                                                    <?php if ($row['DaysOverdue'] > 0) { ?>
                                                        <span class="text-danger">Overdue: <?php echo $row['DaysOverdue']; ?> days</span>
                                                    <?php } else { ?>
                                                        <span class="text-success">On Time</span>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <span class="badge badge-<?php echo $statusBadge; ?>"><?php echo $statusText; ?></span>
                                                    <?php if ($row['PawnStatus'] == 'Redeemed') { ?>
                                                        <br><span class="badge badge-success">Released</span>
                                                    <?php } ?>
                                                </td>
                                                <td class="no-print action-buttons">
                                                    <a href="receipt.php?id=<?php echo $row['ID']; ?>" 
                                                       class="btn btn-sm btn-info" title="Receipt">
                                                        <i class="fas fa-receipt"></i>
                                                    </a>
                                                    <a href="due_collection.php?pawn_id=<?php echo $row['PawnTransactionID']; ?>" 
                                                       class="btn btn-sm btn-primary" title="View Collection">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                    <?php if ($canEdit && $row['IsEditable']) { ?>
                                                        <button class="btn btn-sm btn-warning" title="Edit"
                                                                onclick="editPayment(<?php echo $row['ID']; ?>)">
                                                            <i class="fas fa-edit"></i>
                                                        </button>
                                                        <button class="btn btn-sm btn-danger" title="Delete"
                                                                onclick="deletePayment(<?php echo $row['ID']; ?>)">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    <?php } ?>
                                                </td>
                                            </tr>
                                        <?php 
                                            }
                                        } else { 
                                        ?>
                                            <tr>
                                                <td colspan="<?php echo $canEdit ? '14' : '13'; ?>" class="text-center">No records found</td>
                                            </tr>
                                        <?php } ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <!-- Pagination -->
                            <?php if ($totalPages > 1) { ?>
                                <nav class="no-print">
                                    <ul class="pagination justify-content-center">
                                        <?php if ($page > 1) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page-1])); ?>">Previous</a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php 
                                        $start = max(1, $page - 2);
                                        $end = min($totalPages, $page + 2);
                                        for ($i = $start; $i <= $end; $i++) { 
                                        ?>
                                            <li class="page-item <?php echo ($i == $page) ? 'active' : ''; ?>">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>"><?php echo $i; ?></a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php if ($page < $totalPages) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page+1])); ?>">Next</a>
                                            </li>
                                        <?php } ?>
                                    </ul>
                                </nav>
                            <?php } ?>
                        </div>
                    </div>
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
    
    <script>
    $(document).ready(function() {
        // Select all checkbox functionality
        $('#selectAll').change(function() {
            $('.record-checkbox').prop('checked', this.checked);
            updateSelectedCount();
        });
        
        $('.record-checkbox').change(function() {
            updateSelectedCount();
            if (!this.checked) {
                $('#selectAll').prop('checked', false);
            }
        });
        
        function updateSelectedCount() {
            var count = $('.record-checkbox:checked').length;
            $('#selectedCount').text(count);
        }
        
        // Bulk form submission
        $('#bulkForm').submit(function(e) {
            var selectedCount = $('.record-checkbox:checked').length;
            if (selectedCount === 0) {
                e.preventDefault();
                alert('Please select at least one record');
                return false;
            }
            
            // Add selected checkboxes to bulk form
            $('.record-checkbox:checked').each(function() {
                $('#bulkForm').append('<input type="hidden" name="selected_records[]" value="' + $(this).val() + '">');
            });
        });
    });
    
    function confirmBulkAction() {
        var action = $('select[name="bulk_action"]').val();
        var count = $('.record-checkbox:checked').length;
        
        if (count === 0) {
            alert('Please select at least one record');
            return false;
        }
        
        var message = 'Are you sure you want to perform this action on ' + count + ' records?';
        if (action === 'delete') {
            message = 'Warning: This will permanently delete ' + count + ' records. Are you sure?';
        }
        
        return confirm(message);
    }
    
    function exportToCSV() {
        var currentUrl = window.location.href;
        var separator = currentUrl.includes('?') ? '&' : '?';
        window.open(currentUrl + separator + 'export=csv');
    }
    
    function editPayment(paymentId) {
        // Redirect to edit page or open modal
        window.location.href = 'edit_payment.php?id=' + paymentId;
    }
    
    function deletePayment(paymentId) {
        if (confirm('Are you sure you want to delete this payment record?')) {
            var form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = '<input type="hidden" name="collection_id" value="' + paymentId + '">' +
                            '<input type="hidden" name="delete_payment" value="1">';
            document.body.appendChild(form);
            form.submit();
        }
    }
    </script>
    
    <?php
    // Handle CSV export
    if (isset($_GET['export']) && $_GET['export'] == 'csv') {
        // Remove pagination for export
        $exportQuery = "SELECT pc.*, p.PawnCode, p.ItemType, p.ItemDescription, p.LoanAmount, p.LoanDate, p.DueDate, p.Status as PawnStatus,
                        COALESCE(p.CurrentLoanAmount, p.LoanAmount) as ActualLoanAmount,
                        c.Name as CustomerName, c.MobileNumber, c.CustomerCode,
                        u.FullName as CreatedByName, eu.FullName as EditedByName,
                        DATEDIFF(pc.CollectionDate, p.DueDate) as DaysFromDue
                        FROM tblpaymentcollections pc 
                        JOIN tblpawntransactions p ON pc.PawnTransactionID = p.ID
                        JOIN tblcustomers c ON pc.CustomerID = c.ID 
                        LEFT JOIN tblusers u ON pc.CreatedBy = u.ID
                        LEFT JOIN tblusers eu ON pc.EditedBy = eu.ID
                        $whereClause 
                        ORDER BY $sortBy $sortOrder";
        $exportResult = mysqli_query($con, $exportQuery);
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="due_collection_report_' . date('Y-m-d') . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        // CSV headers
        fputcsv($output, [
            'Collection Code', 'Pawn Code', 'Customer Name', 'Customer Code', 'Mobile', 'Item Type', 'Item Description',
            'Loan Amount', 'Collection Date', 'Amount Paid', 'Discount Amount', 'Total Collection',
            'Payment Method', 'Reference Number', 'Interest Days', 'Days Overdue', 'Notes', 'Created By', 'Edited By'
        ]);
        
        while ($row = mysqli_fetch_assoc($exportResult)) {
            $totalCollection = $row['AmountPaid'] + $row['DiscountAmount'];
            fputcsv($output, [
                $row['CollectionCode'],
                $row['PawnCode'],
                $row['CustomerName'],
                $row['CustomerCode'],
                $row['MobileNumber'],
                $row['ItemType'],
                $row['ItemDescription'],
                $row['ActualLoanAmount'],
                $row['CollectionDate'],
                $row['AmountPaid'],
                $row['DiscountAmount'],
                $totalCollection,
                $row['PaymentMethod'],
                $row['ReferenceNumber'],
                $row['InterestDays'],
                $row['DaysOverdue'],
                $row['Notes'],
                $row['CreatedByName'],
                $row['EditedByName']
            ]);
        }
        
        fclose($output);
        exit();
    }
    ?>
</body>
</html>
