<?php
session_start();

// Check if config file exists
$configPath = __DIR__ . '/includes/config.php';
if (!file_exists($configPath)) {
    die("Error: Database configuration file not found. Please create includes/config.php");
}

include_once($configPath);
include_once(__DIR__ . '/includes/brand.php');

// Check if database connection exists
if (!isset($con) || !$con) {
    die("Error: Database connection failed. Please check your database configuration.");
}

if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$msg = "";
$userId = $_SESSION['uid'];

// Get user permissions
$userQuery = mysqli_query($con, "SELECT UserType, CanEdit FROM tblusers WHERE ID = $userId");
$userData = mysqli_fetch_assoc($userQuery);
$canEdit = ($userData['UserType'] == 'admin' || $userData['CanEdit'] == 1);

// Enhanced function to calculate daily interest and due amount with proper rounding
function calculateDueAmount($loanDate, $dueDate, $loanAmount, $interestRate, $collectionDate = null) {
    if (!$collectionDate) {
        $collectionDate = date('Y-m-d');
    }
    
    $loanDateTime = new DateTime($loanDate);
    $dueDatetime = new DateTime($dueDate);
    $collectionDateTime = new DateTime($collectionDate);
    
    $totalDays = $loanDateTime->diff($collectionDateTime)->days;
    $daysOverdue = max(0, $collectionDateTime->diff($dueDatetime)->days);
    if ($collectionDateTime < $dueDatetime) {
        $daysOverdue = 0;
    }
    
    // Calculate daily interest rate from annual rate (15% per annum)
    // Daily rate = 0.15 / 365
    $annualRate = 0.15; // 15% per year
    $dailyRate = $annualRate / 365;

    // Per-day interest amount - ROUNDED to nearest rupee
    $perDayInterestAmount = round($loanAmount * $dailyRate, 0); // Round to nearest rupee
    
    // Total accrued interest - ROUNDED to nearest rupee
    $totalInterestAccrued = round($perDayInterestAmount * $totalDays, 0); // Round to nearest rupee

    // Monthly interest (for display/reference only): annual/12 - ROUNDED to nearest rupee
    $monthlyInterestAmount = round($loanAmount * ($annualRate / 12), 0);

    // Actual interest is the total accrued based on days
    $actualInterestAmount = $totalInterestAccrued;
    
    // Total due amount - ROUNDED to nearest rupee
    $totalDue = round($loanAmount + $actualInterestAmount, 0); // Round to nearest rupee
    
    return [
        'totalDays' => $totalDays,
        'daysOverdue' => $daysOverdue,
        'dailyRate' => $dailyRate,
        'dailyInterestAmount' => $perDayInterestAmount, // Rounded per-day interest amount (whole number)
        'monthlyInterestAmount' => $monthlyInterestAmount, // Rounded monthly interest (whole number)
        'interestAmount' => $actualInterestAmount, // Rounded total interest (whole number)
        'totalDue' => $totalDue // Rounded total due (whole number)
    ];
}

// Handle payment collection
if ($_POST && isset($_POST['collect_payment'])) {
    $pawnTransactionID = (int)$_POST['pawn_transaction_id'];
    $customerID = (int)$_POST['customer_id'];
    $collectionDate = mysqli_real_escape_string($con, $_POST['collection_date']);
    $amountPaid = (float)$_POST['amount_paid'];
    $discountAmount = (float)($_POST['discount_amount'] ?: 0);
    $paymentMethod = mysqli_real_escape_string($con, $_POST['payment_method']);
    $referenceNumber = mysqli_real_escape_string($con, $_POST['reference_number']);
    $notes = mysqli_real_escape_string($con, $_POST['notes']);
    
    // Get pawn transaction details with current loan amount
    $pawnQuery = mysqli_query($con, "SELECT *, COALESCE(CurrentLoanAmount, LoanAmount) as ActualLoanAmount FROM tblpawntransactions WHERE ID = $pawnTransactionID");
    $pawnData = mysqli_fetch_assoc($pawnQuery);
    
    if ($pawnData) {
        $dueCalc = calculateDueAmount($pawnData['LoanDate'], $pawnData['DueDate'], $pawnData['ActualLoanAmount'], $pawnData['InterestRate'], $collectionDate);
        
        // Calculate already paid amount
        $paidQuery = mysqli_query($con, "SELECT SUM(AmountPaid + DiscountAmount) as TotalPaid FROM tblpaymentcollections WHERE PawnTransactionID = $pawnTransactionID");
        $paidData = mysqli_fetch_assoc($paidQuery);
        $totalPaid = $paidData['TotalPaid'] ?: 0;
        
        $currentBalance = $dueCalc['totalDue'] - $totalPaid;
        $totalCollecting = $amountPaid + $discountAmount;
        
        // Validate payment amount doesn't exceed outstanding balance
        if ($totalCollecting > $currentBalance) {
            $msg = "Payment amount + Discount cannot exceed outstanding balance! Amount due: ₹" . number_format($currentBalance, 0);
        } else {
            $balanceAmount = $currentBalance - $totalCollecting;
            
            $collectionCode = "COL" . date('Y') . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
            
            $insertQuery = "INSERT INTO tblpaymentcollections (CollectionCode, PawnTransactionID, CustomerID, CollectionDate, DaysOverdue, InterestDays, InterestAmount, DailyInterestRate, DailyInterestAmount, MonthlyInterestAmount, TotalAmountDue, AmountPaid, DiscountAmount, BalanceAmount, PaymentMethod, ReferenceNumber, Notes, CreatedBy) 
                            VALUES ('$collectionCode', $pawnTransactionID, $customerID, '$collectionDate', {$dueCalc['daysOverdue']}, {$dueCalc['totalDays']}, {$dueCalc['interestAmount']}, {$dueCalc['dailyRate']}, {$dueCalc['dailyInterestAmount']}, {$dueCalc['monthlyInterestAmount']}, {$dueCalc['totalDue']}, $amountPaid, $discountAmount, $balanceAmount, '$paymentMethod', '$referenceNumber', '$notes', $userId)";
            
            if (mysqli_query($con, $insertQuery)) {
                $collectionId = mysqli_insert_id($con);
                
                // Update pawn transaction status if fully paid
                if ($balanceAmount <= 0) {
                    mysqli_query($con, "UPDATE tblpawntransactions SET Status = 'Redeemed' WHERE ID = $pawnTransactionID");
                }
                
                $msg = "Payment collected successfully!";
                header("Location: receipt.php?id=$collectionId");
                exit();
            } else {
                $msg = "Error collecting payment: " . mysqli_error($con);
            }
        }
    }
}

// Handle payment update (Admin only)
if ($_POST && isset($_POST['update_payment']) && $canEdit) {
    $collectionId = (int)$_POST['collection_id'];
    $amountPaid = (float)$_POST['amount_paid'];
    $discountAmount = (float)($_POST['discount_amount'] ?: 0);
    $paymentMethod = mysqli_real_escape_string($con, $_POST['payment_method']);
    $referenceNumber = mysqli_real_escape_string($con, $_POST['reference_number']);
    $notes = mysqli_real_escape_string($con, $_POST['notes']);
    
    // Check if record is still editable
    $checkQuery = mysqli_query($con, "SELECT IsEditable FROM tblpaymentcollections WHERE ID = $collectionId");
    $checkData = mysqli_fetch_assoc($checkQuery);
    
    if ($checkData && $checkData['IsEditable']) {
        $updateQuery = "UPDATE tblpaymentcollections SET 
                       AmountPaid = $amountPaid,
                       DiscountAmount = $discountAmount,
                       PaymentMethod = '$paymentMethod',
                       ReferenceNumber = '$referenceNumber',
                       Notes = '$notes',
                       EditedBy = $userId,
                       EditedDate = NOW()
                       WHERE ID = $collectionId";
        
        if (mysqli_query($con, $updateQuery)) {
            $msg = "Payment updated successfully!";
        } else {
            $msg = "Error updating payment: " . mysqli_error($con);
        }
    } else {
        $msg = "This record cannot be edited!";
    }
}

// Handle payment deletion (Admin only)
if ($_POST && isset($_POST['delete_payment']) && $canEdit) {
    $collectionId = (int)$_POST['collection_id'];
    
    // Check if record is still editable
    $checkQuery = mysqli_query($con, "SELECT IsEditable FROM tblpaymentcollections WHERE ID = $collectionId");
    $checkData = mysqli_fetch_assoc($checkQuery);
    
    if ($checkData && $checkData['IsEditable']) {
        if (mysqli_query($con, "DELETE FROM tblpaymentcollections WHERE ID = $collectionId")) {
            $msg = "Payment deleted successfully!";
        } else {
            $msg = "Error deleting payment: " . mysqli_error($con);
        }
    } else {
        $msg = "This record cannot be deleted!";
    }
}

// Handle loan increase
if ($_POST && isset($_POST['increase_loan'])) {
    $pawnTransactionID = (int)$_POST['pawn_transaction_id'];
    $customerID = (int)$_POST['customer_id'];
    $increaseAmount = (float)$_POST['increase_amount'];
    $newDueDate = mysqli_real_escape_string($con, $_POST['new_due_date']);
    $reason = mysqli_real_escape_string($con, $_POST['reason']);
    
    // Get current loan details
    $pawnQuery = mysqli_query($con, "SELECT *, COALESCE(CurrentLoanAmount, LoanAmount) as ActualLoanAmount FROM tblpawntransactions WHERE ID = $pawnTransactionID");
    $pawnData = mysqli_fetch_assoc($pawnQuery);
    
    if ($pawnData && $increaseAmount > 0) {
        $newTotalLoanAmount = $pawnData['ActualLoanAmount'] + $increaseAmount;
        $increaseCode = "INC" . date('Y') . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        
        // Insert loan increase record
        $insertIncreaseQuery = "INSERT INTO tblloanincreases (IncreaseCode, PawnTransactionID, CustomerID, IncreaseAmount, NewTotalLoanAmount, IncreaseDate, NewDueDate, Reason, ApprovedBy) 
                               VALUES ('$increaseCode', $pawnTransactionID, $customerID, $increaseAmount, $newTotalLoanAmount, CURDATE(), '$newDueDate', '$reason', $userId)";
        
        if (mysqli_query($con, $insertIncreaseQuery)) {
            $increaseId = mysqli_insert_id($con);
            
            // Update pawn transaction with new amounts
            $updatePawnQuery = "UPDATE tblpawntransactions SET 
                               CurrentLoanAmount = $newTotalLoanAmount,
                               TotalLoanIncreases = TotalLoanIncreases + $increaseAmount,
                               DueDate = '$newDueDate'
                               WHERE ID = $pawnTransactionID";
            
            if (mysqli_query($con, $updatePawnQuery)) {
                $msg = "Loan amount increased successfully!";
                header("Location: loan_increase_receipt.php?id=$increaseId");
                exit();
            } else {
                $msg = "Error updating loan amount: " . mysqli_error($con);
            }
        } else {
            $msg = "Error recording loan increase: " . mysqli_error($con);
        }
    }
}

// Handle loan increase update (Admin only)
if ($_POST && isset($_POST['update_loan_increase']) && $canEdit) {
    $increaseId = (int)$_POST['increase_id'];
    $increaseAmount = (float)$_POST['increase_amount'];
    $newDueDate = mysqli_real_escape_string($con, $_POST['new_due_date']);
    $reason = mysqli_real_escape_string($con, $_POST['reason']);
    
    // Check if record is still editable
    $checkQuery = mysqli_query($con, "SELECT IsEditable, PawnTransactionID, NewTotalLoanAmount, IncreaseAmount as OldIncreaseAmount FROM tblloanincreases WHERE ID = $increaseId");
    $checkData = mysqli_fetch_assoc($checkQuery);
    
    if ($checkData && $checkData['IsEditable']) {
        $pawnTransactionID = $checkData['PawnTransactionID'];
        $oldIncreaseAmount = $checkData['OldIncreaseAmount'];
        $newTotalLoanAmount = $checkData['NewTotalLoanAmount'] - $oldIncreaseAmount + $increaseAmount;
        
        $updateQuery = "UPDATE tblloanincreases SET 
                       IncreaseAmount = $increaseAmount,
                       NewTotalLoanAmount = $newTotalLoanAmount,
                       NewDueDate = '$newDueDate',
                       Reason = '$reason',
                       EditedBy = $userId,
                       EditedDate = NOW()
                       WHERE ID = $increaseId";
        
        if (mysqli_query($con, $updateQuery)) {
            // Update pawn transaction with new amounts
            $updatePawnQuery = "UPDATE tblpawntransactions SET 
                               CurrentLoanAmount = $newTotalLoanAmount,
                               TotalLoanIncreases = TotalLoanIncreases - $oldIncreaseAmount + $increaseAmount,
                               DueDate = '$newDueDate'
                               WHERE ID = $pawnTransactionID";
            
            mysqli_query($con, $updatePawnQuery);
            $msg = "Loan increase updated successfully!";
        } else {
            $msg = "Error updating loan increase: " . mysqli_error($con);
        }
    } else {
        $msg = "This record cannot be edited!";
    }
}

// Handle loan increase deletion (Admin only)
if ($_POST && isset($_POST['delete_loan_increase']) && $canEdit) {
    $increaseId = (int)$_POST['increase_id'];
    
    // Check if record is still editable and get details
    $checkQuery = mysqli_query($con, "SELECT IsEditable, PawnTransactionID, IncreaseAmount, NewTotalLoanAmount FROM tblloanincreases WHERE ID = $increaseId");
    $checkData = mysqli_fetch_assoc($checkQuery);
    
    if ($checkData && $checkData['IsEditable']) {
        $pawnTransactionID = $checkData['PawnTransactionID'];
        $increaseAmount = $checkData['IncreaseAmount'];
        
        // Get current loan details to calculate new amounts
        $pawnQuery = mysqli_query($con, "SELECT CurrentLoanAmount, TotalLoanIncreases FROM tblpawntransactions WHERE ID = $pawnTransactionID");
        $pawnData = mysqli_fetch_assoc($pawnQuery);
        
        if ($pawnData) {
            $newCurrentLoanAmount = $pawnData['CurrentLoanAmount'] - $increaseAmount;
            $newTotalLoanIncreases = $pawnData['TotalLoanIncreases'] - $increaseAmount;
            
            if (mysqli_query($con, "DELETE FROM tblloanincreases WHERE ID = $increaseId")) {
                // Update pawn transaction with corrected amounts
                $updatePawnQuery = "UPDATE tblpawntransactions SET 
                                   CurrentLoanAmount = $newCurrentLoanAmount,
                                   TotalLoanIncreases = $newTotalLoanIncreases
                                   WHERE ID = $pawnTransactionID";
                
                mysqli_query($con, $updatePawnQuery);
                $msg = "Loan increase deleted successfully!";
            } else {
                $msg = "Error deleting loan increase: " . mysqli_error($con);
            }
        }
    } else {
        $msg = "This record cannot be deleted!";
    }
}

// Search for pawn transactions with filters
$searchResults = [];
if (isset($_POST['search_customer']) && !empty($_POST['search_term'])) {
    $searchTerm = mysqli_real_escape_string($con, $_POST['search_term']);
    $statusFilter = isset($_POST['status_filter']) ? mysqli_real_escape_string($con, $_POST['status_filter']) : '';
    $overdueFilter = isset($_POST['overdue_filter']) ? mysqli_real_escape_string($con, $_POST['overdue_filter']) : '';
    
    $searchQuery = "SELECT p.*, c.Name, c.MobileNumber, c.CustomerCode,
                           COALESCE(p.CurrentLoanAmount, p.LoanAmount) as ActualLoanAmount
                    FROM tblpawntransactions p 
                    JOIN tblcustomers c ON p.CustomerID = c.ID 
                    WHERE (c.Name LIKE '%$searchTerm%' OR c.MobileNumber LIKE '%$searchTerm%' OR p.PawnCode LIKE '%$searchTerm%')";
    
    // Add status filter
    if (!empty($statusFilter)) {
        $searchQuery .= " AND p.Status = '$statusFilter'";
    } else {
        $searchQuery .= " AND p.Status IN ('Active', 'Extended')";
    }
    
    $searchQuery .= " ORDER BY p.DueDate ASC";
    
    $searchResult = mysqli_query($con, $searchQuery);
    while ($row = mysqli_fetch_assoc($searchResult)) {
        $dueCalc = calculateDueAmount($row['LoanDate'], $row['DueDate'], $row['ActualLoanAmount'], $row['InterestRate']);
        
        // Get already paid amount
        $paidQuery = mysqli_query($con, "SELECT SUM(AmountPaid + DiscountAmount) as TotalPaid FROM tblpaymentcollections WHERE PawnTransactionID = {$row['ID']}");
        $paidData = mysqli_fetch_assoc($paidQuery);
        $totalPaid = $paidData['TotalPaid'] ?: 0;
        
        $row['TotalDays'] = $dueCalc['totalDays'];
        $row['DaysOverdue'] = $dueCalc['daysOverdue'];
        $row['InterestAmount'] = $dueCalc['interestAmount'];
        $row['TotalDue'] = $dueCalc['totalDue'];
        $row['AlreadyPaid'] = $totalPaid;
        $row['BalanceDue'] = $dueCalc['totalDue'] - $totalPaid;
        
        // Apply overdue filter
        if (!empty($overdueFilter)) {
            if ($overdueFilter == 'overdue' && $row['DaysOverdue'] <= 0) continue;
            if ($overdueFilter == 'due_soon' && ($row['DaysOverdue'] > 0 || $row['DaysOverdue'] < -7)) continue;
            if ($overdueFilter == 'current' && $row['DaysOverdue'] > -7) continue;
        }
        
        $searchResults[] = $row;
    }
}

// Get selected pawn transaction details
$selectedPawn = null;
if (isset($_GET['pawn_id'])) {
    $pawnId = (int)$_GET['pawn_id'];
    $pawnQuery = mysqli_query($con, "SELECT p.*, c.Name, c.MobileNumber, c.CustomerCode,
                                    COALESCE(p.CurrentLoanAmount, p.LoanAmount) as ActualLoanAmount
                                    FROM tblpawntransactions p 
                                    JOIN tblcustomers c ON p.CustomerID = c.ID 
                                    WHERE p.ID = $pawnId");
    $selectedPawn = mysqli_fetch_assoc($pawnQuery);
    
    if ($selectedPawn) {
        $dueCalc = calculateDueAmount($selectedPawn['LoanDate'], $selectedPawn['DueDate'], $selectedPawn['ActualLoanAmount'], $selectedPawn['InterestRate']);
        
        // Get payment history with enhanced fields
        $paymentHistoryQuery = mysqli_query($con, "SELECT pc.*, u.FullName as CreatedByName, eu.FullName as EditedByName 
                                                  FROM tblpaymentcollections pc 
                                                  LEFT JOIN tblusers u ON pc.CreatedBy = u.ID 
                                                  LEFT JOIN tblusers eu ON pc.EditedBy = eu.ID 
                                                  WHERE pc.PawnTransactionID = $pawnId 
                                                  ORDER BY pc.CollectionDate DESC");
        $paymentHistory = [];
        $totalPaid = 0;
        $totalDiscount = 0;
        
        while ($payment = mysqli_fetch_assoc($paymentHistoryQuery)) {
            $paymentHistory[] = $payment;
            $totalPaid += $payment['AmountPaid'];
            $totalDiscount += $payment['DiscountAmount'];
        }
        
        // Get loan increase history with enhanced fields
        $increaseHistoryQuery = mysqli_query($con, "SELECT li.*, u.FullName as ApprovedByName, eu.FullName as EditedByName 
                                                   FROM tblloanincreases li 
                                                   LEFT JOIN tblusers u ON li.ApprovedBy = u.ID 
                                                   LEFT JOIN tblusers eu ON li.EditedBy = eu.ID 
                                                   WHERE li.PawnTransactionID = $pawnId 
                                                   ORDER BY li.IncreaseDate DESC");
        $increaseHistory = [];
        
        if ($increaseHistoryQuery) {
            while ($increase = mysqli_fetch_assoc($increaseHistoryQuery)) {
                $increaseHistory[] = $increase;
            }
        }
        
        $selectedPawn['TotalDays'] = $dueCalc['totalDays'];
        $selectedPawn['DaysOverdue'] = $dueCalc['daysOverdue'];
        $selectedPawn['DailyInterestAmount'] = $dueCalc['dailyInterestAmount'];
        $selectedPawn['MonthlyInterestAmount'] = $dueCalc['monthlyInterestAmount'];
        $selectedPawn['InterestAmount'] = $dueCalc['interestAmount'];
        $selectedPawn['TotalDue'] = $dueCalc['totalDue'];
        $selectedPawn['AlreadyPaid'] = $totalPaid;
        $selectedPawn['TotalDiscount'] = $totalDiscount;
        $selectedPawn['BalanceDue'] = $dueCalc['totalDue'] - $totalPaid - $totalDiscount;
        $selectedPawn['PaymentHistory'] = $paymentHistory;
        $selectedPawn['IncreaseHistory'] = $increaseHistory;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?php echo page_title('Due Collection'); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/compact.css" rel="stylesheet">
    <style>
        .filter-container {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            border-radius: 15px;
            margin-bottom: 20px;
            color: white;
        }
        .search-container {
            background: #f8f9fc;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .filter-badge {
            background: rgba(255,255,255,0.2);
            color: white;
            padding: 5px 10px;
            border-radius: 15px;
            margin: 2px;
            display: inline-block;
        }
        .clear-filters {
            background: rgba(255,255,255,0.1);
            border: 1px solid rgba(255,255,255,0.3);
            color: white;
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 12px;
        }
        .clear-filters:hover {
            background: rgba(255,255,255,0.2);
            color: white;
        }
        .pawn-card {
            border: 1px solid #e3e6f0;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 15px;
            cursor: pointer;
            transition: all 0.3s;
        }
        .pawn-card:hover {
            border-color: #4e73df;
            box-shadow: 0 0 10px rgba(78, 115, 223, 0.2);
        }
        .overdue {
            border-left: 4px solid #e74a3b;
        }
        .due-soon {
            border-left: 4px solid #f6c23e;
        }
        .active {
            border-left: 4px solid #1cc88a;
        }
        .details-card {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        /* Tamil tooltip styling */
        [title] {
            position: relative;
        }
        
        [title]:hover::after {
            content: attr(title);
            position: absolute;
            bottom: 100%;
            left: 50%;
            transform: translateX(-50%);
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 10px;
            white-space: nowrap;
            z-index: 1000;
            pointer-events: none;
        }
        
        [title]:hover::before {
            content: '';
            position: absolute;
            bottom: 95%;
            left: 50%;
            transform: translateX(-50%);
            border: 4px solid transparent;
            border-top-color: rgba(0, 0, 0, 0.8);
            z-index: 1000;
            pointer-events: none;
        }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid">
                    
                    <?php if($msg) { ?>
                        <div class="alert alert-info"><?php echo $msg; ?></div>
                    <?php } ?>
                    
                    <!-- Filter Section -->
                    <div class="filter-container">
                        <h5 class="mb-3"><i class="fas fa-filter"></i> Filter Options</h5>
                        <form method="POST" id="filterForm" class="row">
                            <div class="col-md-3">
                                <label class="small mb-1" title="Status">Status</label>
                                <select name="status_filter" class="form-control form-control-sm" onchange="updateFilters()" title="Status">
                                    <option value="">All Statuses</option>
                                    <option value="Active" <?php echo (isset($_POST['status_filter']) && $_POST['status_filter'] == 'Active') ? 'selected' : ''; ?>>Active</option>
                                    <option value="Extended" <?php echo (isset($_POST['status_filter']) && $_POST['status_filter'] == 'Extended') ? 'selected' : ''; ?>>Extended</option>
                                    <option value="Redeemed" <?php echo (isset($_POST['status_filter']) && $_POST['status_filter'] == 'Redeemed') ? 'selected' : ''; ?>>Redeemed</option>
                                    <option value="Auctioned" <?php echo (isset($_POST['status_filter']) && $_POST['status_filter'] == 'Auctioned') ? 'selected' : ''; ?>>Auctioned</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="small mb-1" title="Overdue Status">Overdue Status</label>
                                <select name="overdue_filter" class="form-control form-control-sm" onchange="updateFilters()" title="Overdue Status">
                                    <option value="">All</option>
                                    <option value="overdue" <?php echo (isset($_POST['overdue_filter']) && $_POST['overdue_filter'] == 'overdue') ? 'selected' : ''; ?>>Overdue</option>
                                    <option value="due_soon" <?php echo (isset($_POST['overdue_filter']) && $_POST['overdue_filter'] == 'due_soon') ? 'selected' : ''; ?>>Due Soon</option>
                                    <option value="current" <?php echo (isset($_POST['overdue_filter']) && $_POST['overdue_filter'] == 'current') ? 'selected' : ''; ?>>Current</option>
                                </select>
                            </div>
                            <div class="col-md-4">
                                <label class="small mb-1" title="Quick Filter">Quick Filter</label>
                                <div>
                                    <button type="button" class="btn btn-sm btn-outline-light mr-1 mb-1" onclick="setQuickFilter('today_due')" title="Due Today">
                                        Due Today
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-light mr-1 mb-1" onclick="setQuickFilter('overdue_only')" title="Overdue Only">
                                        Overdue Only
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-light mr-1 mb-1" onclick="setQuickFilter('high_value')" title="High Value">
                                        High Value
                                    </button>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <label class="small mb-1">&nbsp;</label>
                                <div>
                                    <button type="button" class="btn btn-sm clear-filters btn-block" onclick="clearAllFilters()" title="Clear">
                                        <i class="fas fa-times"></i> Clear
                                    </button>
                                </div>
                            </div>
                            <input type="hidden" name="search_term" value="<?php echo isset($_POST['search_term']) ? htmlspecialchars($_POST['search_term']) : ''; ?>">
                            <input type="hidden" name="search_customer" value="1">
                        </form>
                        
                        <!-- Active Filters Display -->
                        <div class="mt-3" id="activeFilters">
                            <?php if (isset($_POST['status_filter']) && !empty($_POST['status_filter'])) { ?>
                                <span class="filter-badge">
                                    <i class="fas fa-tag"></i> Status: <?php echo htmlspecialchars($_POST['status_filter']); ?>
                                </span>
                            <?php } ?>
                            <?php if (isset($_POST['overdue_filter']) && !empty($_POST['overdue_filter'])) { ?>
                                <span class="filter-badge">
                                    <i class="fas fa-clock"></i> Overdue: <?php echo htmlspecialchars($_POST['overdue_filter']); ?>
                                </span>
                            <?php } ?>
                        </div>
                    </div>
                    
                    <!-- Search Section -->
                    <div class="search-container">
                        <h5 class="mb-3">Customer Search</h5>
                        <form method="POST" class="row" id="searchForm">
                            <div class="col-md-6">
                                <input type="text" name="search_term" class="form-control" 
                                       placeholder="Name, Mobile, Code" title="Name, Mobile, Code"
                                       value="<?php echo isset($_POST['search_term']) ? htmlspecialchars($_POST['search_term']) : ''; ?>"
                                       required>
                            </div>
                            <div class="col-md-3">
                                <button type="submit" name="search_customer" class="btn btn-primary btn-block" title="Search">
                                    <i class="fas fa-search"></i> Search
                                </button>
                            </div>
                            <div class="col-md-3">
                                <a href="due_collection_report_detailed.php" class="btn btn-info btn-block" title="Collection Report">
                                    <i class="fas fa-chart-bar"></i> Collection Report
                                </a>
                            </div>
                            <div class="col-md-3">
                                <a href="jewel_redemption_report.php" class="btn btn-success btn-block" title="Jewel Redemption Report">
                                    <i class="fas fa-gem"></i> Jewel Redemption Report
                                </a>
                            </div>
                            <!-- Hidden fields to maintain filter state -->
                            <input type="hidden" name="status_filter" value="<?php echo isset($_POST['status_filter']) ? htmlspecialchars($_POST['status_filter']) : ''; ?>">
                            <input type="hidden" name="overdue_filter" value="<?php echo isset($_POST['overdue_filter']) ? htmlspecialchars($_POST['overdue_filter']) : ''; ?>">
                        </form>
                    </div>
                    
                    <!-- Search Results -->
                    <div class="card shadow mb-4">
                            <div class="card-header py-3 d-flex justify-content-between align-items-center">
                                <h6 class="m-0 font-weight-bold text-primary">
                                    Search Results - <?php echo count($searchResults); ?> records
                                </h6>
                                <div class="btn-group" role="group">
                                    <button type="button" class="btn btn-sm btn-outline-primary" onclick="sortResults('amount')" title="Sort by amount">
                                        <i class="fas fa-sort-amount-down"></i> Amount
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-primary" onclick="sortResults('date')" title="Sort by date">
                                        <i class="fas fa-sort-numeric-down"></i> Date
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-primary" onclick="sortResults('overdue')" title="Sort by overdue">
                                        <i class="fas fa-exclamation-triangle"></i> Overdue
                                    </button>
                                </div>
                            </div>
                            <div class="card-body">
                                <div class="row" id="searchResultsContainer">
                                    <?php foreach ($searchResults as $pawn) { 
                                        $cardClass = 'active';
                                        $statusBadge = 'success';
                                        $statusText = 'Current';
                                        
                                        if ($pawn['DaysOverdue'] > 0) {
                                            $cardClass = 'overdue';
                                            $statusBadge = 'danger';
                                            $statusText = $pawn['DaysOverdue'] . ' days overdue';
                                        } elseif ($pawn['DaysOverdue'] >= -7) {
                                            $cardClass = 'due-soon';
                                            $statusBadge = 'warning';
                                            $statusText = 'Due Soon';
                                        }
                                    ?>
                                        <div class="col-md-6 search-result-item" 
                                             data-amount="<?php echo $pawn['BalanceDue']; ?>"
                                             data-date="<?php echo strtotime($pawn['DueDate']); ?>"
                                             data-overdue="<?php echo $pawn['DaysOverdue']; ?>">
                                            <div class="pawn-card <?php echo $cardClass; ?>" onclick="selectPawn(<?php echo $pawn['ID']; ?>)">
                                                <div class="row">
                                                    <div class="col-8">
                                                        <strong><?php echo htmlspecialchars($pawn['Name']); ?></strong>
                                                        <span class="badge badge-<?php echo $statusBadge; ?> ml-2"><?php echo htmlspecialchars($pawn['Status']); ?></span>
                                                        <br>
                                                        <small><?php echo htmlspecialchars($pawn['MobileNumber']); ?></small><br>
                                                        <span class="badge badge-info"><?php echo htmlspecialchars($pawn['PawnCode']); ?></span>
                                                    </div>
                                                    <div class="col-4 text-right">
                                                        <strong>₹<?php echo number_format($pawn['BalanceDue'], 0); ?></strong><br>
                                                        <small><span class="badge badge-<?php echo $statusBadge; ?>"><?php echo $statusText; ?></span></small>
                                                    </div>
                                                </div>
                                                <div class="mt-2">
                                                    <small><strong>Item:</strong> <?php echo htmlspecialchars($pawn['ItemDescription']); ?></small><br>
                                                    <small><strong>Loan Amount:</strong> ₹<?php echo number_format($pawn['ActualLoanAmount'], 0); ?></small>
                                                    <small class="ml-3"><strong>Due Date:</strong> <?php echo date('d/m/Y', strtotime($pawn['DueDate'])); ?></small>
                                                </div>
                                            </div>
                                        </div>
                                    <?php } ?>
                                </div>
                            </div>
                        </div>
                    
                    <!-- Selected Pawn Details & Payment Collection -->
                    <?php if ($selectedPawn) { ?>
                        <div class="details-card">
                            <div class="row">
                                <div class="col-md-6">
                                    <h5><?php echo htmlspecialchars($selectedPawn['Name']); ?></h5>
                                    <p><i class="fas fa-phone"></i> <?php echo htmlspecialchars($selectedPawn['MobileNumber']); ?></p>
                                    <p><i class="fas fa-gem"></i> <?php echo htmlspecialchars($selectedPawn['PawnCode']); ?></p>
                                </div>
                                <div class="col-md-6 text-right">
                                    <h3>₹<?php echo number_format($selectedPawn['BalanceDue'], 0); ?></h3>
                                    <p>Balance Due</p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <!-- Pawn Details -->
                            <div class="col-md-6">
                                <div class="card shadow mb-4">
                                    <div class="card-header py-3">
                                        <h6 class="m-0 font-weight-bold text-primary">Pawn Details</h6>
                                    </div>
                                    <div class="card-body">
                                        <table class="table table-borderless">
                                            <tr><td><strong>Item Description:</strong></td><td><?php echo htmlspecialchars($selectedPawn['ItemDescription']); ?></td></tr>
                                            <tr><td><strong>Weight:</strong></td><td><?php echo $selectedPawn['ItemWeight']; ?> g</td></tr>
                                            <tr><td><strong>Loan Amount:</strong></td><td>₹<?php echo number_format($selectedPawn['LoanAmount'], 0); ?></td></tr>
                                            <tr><td><strong>Interest Rate:</strong></td><td>15% per year (<?php echo number_format((0.15/365)*100, 4); ?>% per day)</td></tr>
                                            <tr><td><strong>Loan Date:</strong></td><td><?php echo date('d/m/Y', strtotime($selectedPawn['LoanDate'])); ?></td></tr>
                                            <tr><td><strong>Due Date:</strong></td><td><?php echo date('d/m/Y', strtotime($selectedPawn['DueDate'])); ?></td></tr>
                                            <tr><td><strong>Total Days:</strong></td><td><?php echo $selectedPawn['TotalDays']; ?> days</td></tr>
                                            <tr><td><strong>Days Overdue:</strong></td><td><span class="<?php echo $selectedPawn['DaysOverdue'] > 0 ? 'text-danger' : 'text-success'; ?>"><?php echo $selectedPawn['DaysOverdue']; ?> days</span></td></tr>
                                        </table>
                                        
                                        <hr>
                                        <table class="table table-borderless">
                                            <tr><td><strong>Days Since Loan:</strong></td><td><?php echo $selectedPawn['TotalDays']; ?> days</td></tr>
                                            <tr><td><strong>Daily Interest Rate:</strong></td><td><?php echo number_format((0.15/365)*100, 4); ?>% per day</td></tr>
                                            <tr><td><strong>Daily Interest Amount:</strong></td><td>₹<?php echo number_format($selectedPawn['DailyInterestAmount'], 0); ?> per day</td></tr>
                                            <tr><td><strong>Total Interest (<?php echo $selectedPawn['TotalDays']; ?> days):</strong></td><td>₹<?php echo number_format($selectedPawn['InterestAmount'], 0); ?></td></tr>
                                            <tr><td><strong>Total Amount Due:</strong></td><td>₹<?php echo number_format($selectedPawn['TotalDue'], 0); ?></td></tr>
                                            <tr><td><strong>Amount Paid:</strong></td><td>₹<?php echo number_format($selectedPawn['AlreadyPaid'], 0); ?></td></tr>
                                            <tr><td><strong>Discount:</strong></td><td>₹<?php echo number_format($selectedPawn['TotalDiscount'], 0); ?></td></tr>
                                            <tr class="table-warning"><td><strong>Balance Amount:</strong></td><td><strong>₹<?php echo number_format($selectedPawn['BalanceDue'], 0); ?></strong></td></tr>
                                        </table>
                                    </div>
                                </div>
                                
                                <!-- Enhanced Payment History -->
                                <?php if (!empty($selectedPawn['PaymentHistory'])) { ?>
                                    <div class="card shadow mb-4">
                                        <div class="card-header py-3">
                                            <h6 class="m-0 font-weight-bold text-primary">Payment History</h6>
                                        </div>
                                        <div class="card-body">
                                            <div class="table-responsive">
                                                <table class="table table-bordered">
                                                    <thead>
                                                        <tr>
                                                            <th title="Date">Date</th>
                                                            <th title="Amount">Amount</th>
                                                            <th title="Discount">Discount</th>
                                                            <th title="Method">Method</th>
                                                            <?php if ($canEdit) { ?><th title="Actions">Actions</th><?php } ?>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        <?php foreach ($selectedPawn['PaymentHistory'] as $payment) { ?>
                                                            <tr>
                                                                <td><?php echo date('d/m/Y', strtotime($payment['CollectionDate'])); ?></td>
                                                                <td>₹<?php echo number_format($payment['AmountPaid'], 0); ?></td>
                                                                <td>₹<?php echo number_format($payment['DiscountAmount'], 0); ?></td>
                                                                <td><?php echo htmlspecialchars($payment['PaymentMethod']); ?></td>
                                                                <?php if ($canEdit) { ?>
                                                                <td>
                                                                    <?php if ($payment['IsEditable']) { ?>
                                                                        <button class="btn btn-sm btn-warning" onclick="editPayment(<?php echo $payment['ID']; ?>)">
                                                                            <i class="fas fa-edit"></i>
                                                                        </button>
                                                                        <button class="btn btn-sm btn-danger" onclick="deletePayment(<?php echo $payment['ID']; ?>)">
                                                                            <i class="fas fa-trash"></i>
                                                                        </button>
                                                                    <?php } else { ?>
                                                                        <span class="text-muted">Cannot Edit</span>
                                                                    <?php } ?>
                                                                </td>
                                                                <?php } ?>
                                                            </tr>
                                                        <?php } ?>
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>
                                    </div>
                                <?php } ?>
                            </div>
                            
                            <!-- Payment Collection Form -->
                            <div class="col-md-6">
                                <div class="card shadow mb-4">
                                    <div class="card-header py-3 d-flex justify-content-between">
                                        <h6 class="m-0 font-weight-bold text-primary">Collect Payment</h6>
                                        <div>
                                            <button type="button" class="btn btn-sm btn-info" data-toggle="modal" data-target="#increaseLoanModal" title="Increase Loan">
                                                <i class="fas fa-plus"></i> Increase Loan
                                            </button>
                                            <a href="jewel_redemption.php?pawn_id=<?php echo $selectedPawn['ID']; ?>" class="btn btn-sm btn-success ml-1" title="Redeem Jewel">
                                                <i class="fas fa-gem"></i> Redeem Jewel
                                            </a>
                                        </div>
                                    </div>
                                    <div class="card-body">
                                        <form method="POST" id="paymentForm">
                                            <input type="hidden" name="pawn_transaction_id" value="<?php echo $selectedPawn['ID']; ?>">
                                            <input type="hidden" name="customer_id" value="<?php echo $selectedPawn['CustomerID']; ?>">
                                            
                                            <div class="form-group">
                                                <label title="Collection Date">Collection Date <span class="text-danger">*</span></label>
                                                <input type="date" name="collection_date" class="form-control" value="<?php echo date('Y-m-d'); ?>" required title="Collection Date">
                                            </div>
                                            
                                            <div class="form-group">
                                                <label title="Amount Paid">Amount Paid <span class="text-danger">*</span></label>
                                                <input type="number" step="0.01" name="amount_paid" id="amountPaid" class="form-control" required title="Amount Paid">
                                            </div>
                                            
                                            <div class="form-group">
                                                <label title="Discount Amount">Discount Amount</label>
                                                <input type="number" step="0.01" name="discount_amount" id="discountAmount" class="form-control" value="0" title="Discount Amount">
                                                <small class="text-muted">Final minimum amount</small>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label title="Total Collection">Total Collection</label>
                                                <input type="text" id="totalCollection" class="form-control" readonly title="Total Collection">
                                            </div>
                                            
                                            <div class="form-group">
                                                <label title="Outstanding Summary">Outstanding Summary</label>
                                                <div class="card border-info">
                                                    <div class="card-body p-2">
                                                        <div class="d-flex justify-content-between mb-2">
                                                            <strong>Old Outstanding:</strong>
                                                            <span class="text-danger" id="oldOutstanding">₹<?php echo number_format($selectedPawn['BalanceDue'], 0); ?></span>
                                                        </div>
                                                        <div class="d-flex justify-content-between mb-2">
                                                            <strong>Payment + Discount:</strong>
                                                            <span class="text-info" id="paymentAmount">₹0</span>
                                                        </div>
                                                        <hr class="my-1">
                                                        <div class="d-flex justify-content-between">
                                                            <strong>New Outstanding:</strong>
                                                            <span class="text-success font-weight-bold" id="newOutstanding">₹<?php echo number_format($selectedPawn['BalanceDue'], 0); ?></span>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div id="amountError" class="text-danger mt-2" style="display: none;"></div>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label title="Payment Method">Payment Method <span class="text-danger">*</span></label>
                                                <select name="payment_method" class="form-control" required title="Payment Method">
                                                    <option value="Cash">Cash</option>
                                                    <option value="Bank Transfer">Bank Transfer</option>
                                                    <option value="Card">Card</option>
                                                    <option value="Cheque">Cheque</option>
                                                    <option value="UPI">UPI</option>
                                                </select>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label title="Reference Number">Reference Number</label>
                                                <input type="text" name="reference_number" class="form-control" title="Reference Number">
                                            </div>
                                            
                                            <div class="form-group">
                                                <label title="Notes">Notes</label>
                                                <textarea name="notes" class="form-control" rows="3" title="Notes"></textarea>
                                            </div>
                                            
                                            <button type="submit" name="collect_payment" class="btn btn-success btn-block" id="submitPayment" title="Collect Payment">
                                                <i class="fas fa-money-bill"></i> Collect Payment
                                            </button>
                                        </form>
                                    </div>
                                </div>
                                
                                <!-- Enhanced Loan Increase History -->
                                <?php if (!empty($selectedPawn['IncreaseHistory'])) { ?>
                                    <div class="card shadow mb-4">
                                        <div class="card-header py-3">
                                            <h6 class="m-0 font-weight-bold text-primary">Loan Increase History</h6>
                                        </div>
                                        <div class="card-body">
                                            <div class="table-responsive">
                                                <table class="table table-bordered">
                                                    <thead>
                                                        <tr>
                                                            <th title="Date">Date</th>
                                                            <th title="Amount">Amount</th>
                                                            <th title="New Total">New Total</th>
                                                            <th title="Reason">Reason</th>
                                                            <?php if ($canEdit) { ?><th title="Actions">Actions</th><?php } ?>
                                                        </tr>
                                                    </thead>
                                                    <tbody>
                                                        <?php foreach ($selectedPawn['IncreaseHistory'] as $increase) { ?>
                                                            <tr>
                                                                <td><?php echo date('d/m/Y', strtotime($increase['IncreaseDate'])); ?></td>
                                                                <td>₹<?php echo number_format($increase['IncreaseAmount'], 0); ?></td>
                                                                <td>₹<?php echo number_format($increase['NewTotalLoanAmount'], 0); ?></td>
                                                                <td><?php echo htmlspecialchars($increase['Reason']); ?></td>
                                                                <?php if ($canEdit) { ?>
                                                                <td>
                                                                    <?php if ($increase['IsEditable']) { ?>
                                                                        <button class="btn btn-sm btn-warning" onclick="editLoanIncrease(<?php echo $increase['ID']; ?>, '<?php echo $increase['IncreaseAmount']; ?>', '<?php echo $increase['NewDueDate']; ?>', '<?php echo htmlspecialchars($increase['Reason']); ?>')">
                                                                            <i class="fas fa-edit"></i>
                                                                        </button>
                                                                        <button class="btn btn-sm btn-danger" onclick="deleteLoanIncrease(<?php echo $increase['ID']; ?>)">
                                                                            <i class="fas fa-trash"></i>
                                                                        </button>
                                                                    <?php } else { ?>
                                                                        <span class="text-muted">Cannot Edit</span>
                                                                    <?php } ?>
                                                                    <?php if ($increase['EditedBy']) { ?>
                                                                        <br><small class="text-info">Edited by: <?php echo htmlspecialchars($increase['EditedByName']); ?></small>
                                                                    <?php } ?>
                                                                </td>
                                                                <?php } ?>
                                                            </tr>
                                                        <?php } ?>
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>
                                    </div>
                                <?php } ?>
                            </div>
                        </div>
                        
                        <!-- Increase Loan Modal -->
                        <div class="modal fade" id="increaseLoanModal" tabindex="-1">
                            <div class="modal-dialog">
                                <div class="modal-content">
                                    <form method="POST" id="increaseLoanForm">
                                        <div class="modal-header">
                                            <h5 class="modal-title">Increase Loan Amount</h5>
                                            <button type="button" class="close" data-dismiss="modal">&times;</button>
                                        </div>
                                        <div class="modal-body">
                                            <input type="hidden" name="pawn_transaction_id" value="<?php echo $selectedPawn['ID']; ?>">
                                            <input type="hidden" name="customer_id" value="<?php echo $selectedPawn['CustomerID']; ?>">
                                            
                                            <div class="form-group">
                                                <label>Current Loan Amount</label>
                                                <input type="text" class="form-control" value="₹<?php echo number_format($selectedPawn['ActualLoanAmount'], 0); ?>" readonly>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>Outstanding Balance</label>
                                                <input type="text" class="form-control" value="₹<?php echo number_format($selectedPawn['BalanceDue'], 0); ?>" readonly>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>Increase Amount <span class="text-danger">*</span></label>
                                                <input type="number" step="0.01" name="increase_amount" id="increaseAmount" class="form-control" required>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>New Total Loan Amount</label>
                                                <input type="text" id="newTotalAmount" class="form-control" readonly>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>New Due Date <span class="text-danger">*</span></label>
                                                <input type="date" name="new_due_date" class="form-control" 
                                                       value="<?php echo date('Y-m-d', strtotime('+3 months')); ?>" required>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>Reason</label>
                                                <textarea name="reason" class="form-control" rows="3" 
                                                          placeholder="Enter reason for loan increase"></textarea>
                                            </div>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                                            <button type="submit" name="increase_loan" class="btn btn-primary">
                                                <i class="fas fa-plus"></i> Increase Loan
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Edit Loan Increase Modal -->
                        <div class="modal fade" id="editLoanIncreaseModal" tabindex="-1">
                            <div class="modal-dialog">
                                <div class="modal-content">
                                    <form method="POST" id="editLoanIncreaseForm">
                                        <div class="modal-header">
                                            <h5 class="modal-title">Edit Loan Increase</h5>
                                            <button type="button" class="close" data-dismiss="modal">&times;</button>
                                        </div>
                                        <div class="modal-body">
                                            <input type="hidden" name="increase_id" id="editIncreaseId">
                                            
                                            <div class="form-group">
                                                <label>Increase Amount <span class="text-danger">*</span></label>
                                                <input type="number" step="0.01" name="increase_amount" id="editIncreaseAmount" class="form-control" required>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>New Due Date <span class="text-danger">*</span></label>
                                                <input type="date" name="new_due_date" id="editNewDueDate" class="form-control" required>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>Reason</label>
                                                <textarea name="reason" id="editReason" class="form-control" rows="3"></textarea>
                                            </div>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                                            <button type="submit" name="update_loan_increase" class="btn btn-warning">
                                                <i class="fas fa-save"></i> Update
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    <?php } ?>
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    
    <script src="vendor/jquery/jquery.min.js"></script>
    <script>
        function selectPawn(pawnId) {
            window.location.href = 'due_collection.php?pawn_id=' + pawnId;
        }
        
        // Update filters and submit form
        function updateFilters() {
            var searchTerm = document.querySelector('input[name="search_term"]').value;
            if (searchTerm.trim() !== '') {
                document.getElementById('filterForm').submit();
            }
        }
        
        // Clear all filters
        function clearAllFilters() {
            document.querySelector('select[name="status_filter"]').value = '';
            document.querySelector('select[name="overdue_filter"]').value = '';
            document.querySelector('input[name="search_term"]').value = '';
            
            // Clear hidden fields in search form
            document.querySelector('#searchForm input[name="status_filter"]').value = '';
            document.querySelector('#searchForm input[name="overdue_filter"]').value = '';
            
            // Update active filters display
            document.getElementById('activeFilters').innerHTML = '';
        }
        
        // Set quick filters
        function setQuickFilter(filterType) {
            switch(filterType) {
                case 'today_due':
                    document.querySelector('select[name="status_filter"]').value = 'Active';
                    document.querySelector('select[name="overdue_filter"]').value = 'due_soon';
                    break;
                case 'overdue_only':
                    document.querySelector('select[name="status_filter"]').value = 'Active';
                    document.querySelector('select[name="overdue_filter"]').value = 'overdue';
                    break;
                case 'high_value':
                    document.querySelector('select[name="status_filter"]').value = 'Active';
                    document.querySelector('select[name="overdue_filter"]').value = '';
                    break;
            }
            
            var searchTerm = document.querySelector('input[name="search_term"]').value;
            if (searchTerm.trim() !== '') {
                updateFilters();
            }
        }
        
        // Sort results
        function sortResults(sortBy) {
            var container = document.getElementById('searchResultsContainer');
            var items = Array.from(container.children);
            
            items.sort(function(a, b) {
                var aVal, bVal;
                switch(sortBy) {
                    case 'amount':
                        aVal = parseFloat(a.dataset.amount);
                        bVal = parseFloat(b.dataset.amount);
                        return bVal - aVal; // Descending
                    case 'date':
                        aVal = parseInt(a.dataset.date);
                        bVal = parseInt(b.dataset.date);
                        return aVal - bVal; // Ascending (earliest due first)
                    case 'overdue':
                        aVal = parseInt(a.dataset.overdue);
                        bVal = parseInt(b.dataset.overdue);
                        return bVal - aVal; // Most overdue first
                }
            });
            
            // Clear container and re-append sorted items
            container.innerHTML = '';
            items.forEach(function(item) {
                container.appendChild(item);
            });
        }
        
        $(document).ready(function() {
            // If navigated from sidebar Loan Increase (action=increase), focus on Increase Loan flow
            const params = new URLSearchParams(window.location.search);
            if (params.get('action') === 'increase') {
                // If a pawn is selected, open the modal; else guide user
                <?php if ($selectedPawn) { ?>
                    $('#increaseLoanModal').modal('show');
                <?php } else { ?>
                    // Provide a subtle hint to search/select a pawn first
                    const hint = $('<div class="alert alert-info">Use the search to select a pawn, then click "Increase Loan".</div>');
                    $('.container-fluid').prepend(hint);
                    setTimeout(() => hint.fadeOut(500, () => hint.remove()), 5000);
                <?php } ?>
            }
            // Enhanced payment amount validation with discount
            function validatePayment() {
                var amountPaid = parseFloat($('#amountPaid').val()) || 0;
                var discountAmount = parseFloat($('#discountAmount').val()) || 0;
                var totalCollection = amountPaid + discountAmount;
                var maxAmount = <?php echo $selectedPawn ? $selectedPawn['BalanceDue'] : 0; ?>;
                
                // Update total collection display
                $('#totalCollection').val('₹' + totalCollection.toLocaleString('en-IN', {minimumFractionDigits: 0, maximumFractionDigits: 0}));
                
                // Update payment amount display
                $('#paymentAmount').text('₹' + totalCollection.toLocaleString('en-IN', {minimumFractionDigits: 0, maximumFractionDigits: 0}));
                
                // Calculate and display new outstanding
                var newOutstanding = Math.max(0, maxAmount - totalCollection);
                $('#newOutstanding').text('₹' + newOutstanding.toLocaleString('en-IN', {minimumFractionDigits: 0, maximumFractionDigits: 0}));
                
                // Change color based on whether fully paid
                if (newOutstanding <= 0) {
                    $('#newOutstanding').removeClass('text-success').addClass('text-primary font-weight-bold');
                } else {
                    $('#newOutstanding').removeClass('text-primary').addClass('text-success font-weight-bold');
                }
                
                // Validate amount
                if (totalCollection > maxAmount) {
                    $('#amountError').text('Total collection cannot exceed outstanding balance!').show();
                    $('#submitPayment').prop('disabled', true);
                    $('#amountPaid, #discountAmount').addClass('is-invalid');
                } else {
                    $('#amountError').hide();
                    $('#submitPayment').prop('disabled', false);
                    $('#amountPaid, #discountAmount').removeClass('is-invalid');
                }
            }
            
            $('#amountPaid, #discountAmount').on('input', validatePayment);
            
            // Calculate new total loan amount
            $('#increaseAmount').on('input', function() {
                var currentAmount = <?php echo $selectedPawn ? $selectedPawn['ActualLoanAmount'] : 0; ?>;
                var increaseAmount = parseFloat($(this).val()) || 0;
                var newTotal = currentAmount + increaseAmount;
                $('#newTotalAmount').val('₹' + newTotal.toLocaleString('en-IN', {minimumFractionDigits: 0, maximumFractionDigits: 0}));
            });
            
            // Clear form after modal close
            $('#increaseLoanModal').on('hidden.bs.modal', function() {
                $('#increaseLoanForm')[0].reset();
                $('#newTotalAmount').val('');
            });
            
            // Sync filter values between forms
            function syncFilters() {
                var statusFilter = document.querySelector('#filterForm select[name="status_filter"]').value;
                var overdueFilter = document.querySelector('#filterForm select[name="overdue_filter"]').value;
                
                document.querySelector('#searchForm input[name="status_filter"]').value = statusFilter;
                document.querySelector('#searchForm input[name="overdue_filter"]').value = overdueFilter;
            }
            
            // Call sync on form changes
            $('#filterForm select').on('change', syncFilters);
        });
        
        // Edit payment function (Admin only)
        function editPayment(paymentId) {
            if (confirm('Do you want to edit this payment?')) {
                // Implementation for edit modal
                console.log('Edit payment:', paymentId);
            }
        }
        
        // Delete payment function (Admin only)
        function deletePayment(paymentId) {
            if (confirm('Do you want to delete this payment? This action cannot be undone!')) {
                var form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = '<input type="hidden" name="collection_id" value="' + paymentId + '">' +
                                '<input type="hidden" name="delete_payment" value="1">';
                document.body.appendChild(form);
                form.submit();
            }
        }
        
        // Edit loan increase function (Admin only)
        function editLoanIncrease(increaseId, amount, dueDate, reason) {
            $('#editIncreaseId').val(increaseId);
            $('#editIncreaseAmount').val(amount);
            $('#editNewDueDate').val(dueDate);
            $('#editReason').val(reason);
            $('#editLoanIncreaseModal').modal('show');
        }
        
        // Delete loan increase function (Admin only)
        function deleteLoanIncrease(increaseId) {
            if (confirm('Do you want to delete this loan increase? This will also affect the loan amount of the pawn transaction!')) {
                var form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = '<input type="hidden" name="increase_id" value="' + increaseId + '">' +
                                '<input type="hidden" name="delete_loan_increase" value="1">';
                document.body.appendChild(form);
                form.submit();
            }
        }
        
        $(document).ready(function() {
            // Clear edit loan increase form after modal close
            $('#editLoanIncreaseModal').on('hidden.bs.modal', function() {
                $('#editLoanIncreaseForm')[0].reset();
            });
        });
    </script>
    
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
</body>
</html>