<?php
session_start();
include_once('includes/config.php');
include_once(__DIR__ . '/includes/brand.php');
if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$userId = $_SESSION['uid'];
$userType = $_SESSION['usertype'];

// Get dashboard statistics
$totalCustomers = mysqli_num_rows(mysqli_query($con, "SELECT ID FROM tblcustomers WHERE Status='Active'"));
$totalActivePawns = mysqli_num_rows(mysqli_query($con, "SELECT ID FROM tblpawntransactions WHERE Status='Active'"));
$totalActiveLoanAmount = mysqli_fetch_assoc(mysqli_query($con, "SELECT SUM(CurrentLoanAmount) as total FROM tblpawntransactions WHERE Status='Active'"))['total'] ?? 0;
$totalRedemptions = mysqli_num_rows(mysqli_query($con, "SELECT ID FROM tblredemptions WHERE Status='Completed'"));

// Get financial metrics
$todayCollections = mysqli_fetch_assoc(mysqli_query($con, "SELECT SUM(AmountPaid) as total FROM tblpaymentcollections WHERE DATE(CollectionDate) = CURDATE()"))['total'] ?? 0;
$monthlyCollections = mysqli_fetch_assoc(mysqli_query($con, "SELECT SUM(AmountPaid) as total FROM tblpaymentcollections WHERE MONTH(CollectionDate) = MONTH(CURDATE()) AND YEAR(CollectionDate) = YEAR(CURDATE())"))['total'] ?? 0;
$totalInterestEarned = mysqli_fetch_assoc(mysqli_query($con, "SELECT SUM(InterestAmount) as total FROM tblpaymentcollections"))['total'] ?? 0;

// Get overdue statistics
$overdueLoans = mysqli_query($con, "
    SELECT COUNT(*) as count, SUM(CurrentLoanAmount) as amount 
    FROM tblpawntransactions 
    WHERE Status='Active' AND DueDate < CURDATE()
");
$overdueData = mysqli_fetch_assoc($overdueLoans);
$overdueCount = $overdueData['count'] ?? 0;
$overdueAmount = $overdueData['amount'] ?? 0;

// Get recent transactions
$recentPawns = mysqli_query($con, "
    SELECT p.PawnCode, p.LoanAmount, p.LoanDate, c.Name as CustomerName, p.ItemType
    FROM tblpawntransactions p 
    JOIN tblcustomers c ON p.CustomerID = c.ID 
    ORDER BY p.CreatedDate DESC 
    LIMIT 8
");

// Get recent collections
$recentCollections = mysqli_query($con, "
    SELECT pc.CollectionCode, pc.AmountPaid, pc.CollectionDate, c.Name as CustomerName, p.PawnCode
    FROM tblpaymentcollections pc
    JOIN tblcustomers c ON pc.CustomerID = c.ID
    JOIN tblpawntransactions p ON pc.PawnTransactionID = p.ID
    ORDER BY pc.CreatedDate DESC 
    LIMIT 8
");

// Get item type distribution
$itemTypeStats = mysqli_query($con, "
    SELECT ItemType, COUNT(*) as Count, SUM(CurrentLoanAmount) as TotalAmount
    FROM tblpawntransactions 
    WHERE Status='Active'
    GROUP BY ItemType
    ORDER BY TotalAmount DESC
");

// Get monthly collection trends (last 6 months)
$monthlyTrends = mysqli_query($con, "
    SELECT 
        DATE_FORMAT(CollectionDate, '%Y-%m') as Month,
        SUM(AmountPaid) as TotalCollection,
        COUNT(*) as TransactionCount
    FROM tblpaymentcollections 
    WHERE CollectionDate >= DATE_SUB(CURDATE(), INTERVAL 6 MONTH)
    GROUP BY DATE_FORMAT(CollectionDate, '%Y-%m')
    ORDER BY Month DESC
    LIMIT 6
");

// Get due loans for today and next 7 days
$upcomingDues = mysqli_query($con, "
    SELECT p.PawnCode, p.DueDate, p.CurrentLoanAmount, c.Name as CustomerName, c.MobileNumber
    FROM tblpawntransactions p
    JOIN tblcustomers c ON p.CustomerID = c.ID
    WHERE p.Status='Active' AND p.DueDate BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 7 DAY)
    ORDER BY p.DueDate ASC
    LIMIT 10
");

// Get bank pledge statistics
$bankPledgeStats = mysqli_query($con, "
    SELECT 
        COUNT(*) as TotalPledges,
        SUM(PledgeAmount) as TotalPledgeAmount,
        SUM(CASE WHEN Status='Active' THEN PledgeAmount ELSE 0 END) as ActivePledgeAmount
    FROM tblbankpledges
");
$pledgeData = mysqli_fetch_assoc($bankPledgeStats);

// Helper function for Indian number formatting
function formatNumber($number) {
    return number_format($number);
}

function formatCurrency($amount) {
    return '₹' . number_format($amount, 2);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?php echo page_title('Dashboard'); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/compact.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        .card { border-radius: 1rem; transition: transform 0.2s, box-shadow 0.2s; }
        .card:hover { transform: translateY(-2px); box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15) !important; }
        .card-header { border-radius: 1rem 1rem 0 0; }
        .summary-card { color: #fff; position: relative; overflow: hidden; }
        .summary-value { font-size: 2.2rem; font-weight: bold; }
        .summary-label { font-size: 0.9rem; opacity: 0.9; }
        .summary-icon { font-size: 4rem; opacity: 0.2; position: absolute; right: 15px; top: 15px; }
        .alert-card { border-left: 5px solid #dc3545; }
        .success-card { border-left: 5px solid #28a745; }
        .warning-card { border-left: 5px solid #ffc107; }
        .info-card { border-left: 5px solid #17a2b8; }
        .quick-action { 
            border-radius: 50px; 
            padding: 12px 24px; 
            font-weight: 600; 
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
        }
        .quick-action:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }
        .metric-card { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .collection-card { background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); }
        .loan-card { background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); }
        .customer-card { background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%); }
        
        /* Quick Access Panel Styles */
        .quick-access-btn {
            transition: all 0.2s ease;
            border-radius: 8px;
            font-weight: 500;
        }
        .quick-access-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 2px 6px rgba(0,0,0,0.15);
        }
        
        .d-grid .btn + .btn {
            margin-top: 0.5rem;
        }
        
        /* Section headers in quick access */
        .quick-access-section h6 {
            border-bottom: 2px solid;
            padding-bottom: 0.5rem;
            margin-bottom: 1rem;
        }
        .overdue-badge { background: linear-gradient(45deg, #ff6b6b, #ee5a24); }
        .due-today { background-color: #fff3cd; border-left: 4px solid #ffc107; }
        .due-tomorrow { background-color: #d1ecf1; border-left: 4px solid #17a2b8; }
        .overdue { background-color: #f8d7da; border-left: 4px solid #dc3545; }
        
        /* Tamil tooltip styling */
        [title] {
            position: relative;
        }
        
        [title]:hover::after {
            content: attr(title);
            position: absolute;
            bottom: 100%;
            left: 50%;
            transform: translateX(-50%);
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 10px;
            white-space: nowrap;
            z-index: 1000;
            pointer-events: none;
        }
        
        [title]:hover::before {
            content: '';
            position: absolute;
            bottom: 95%;
            left: 50%;
            transform: translateX(-50%);
            border: 4px solid transparent;
            border-top-color: rgba(0, 0, 0, 0.8);
            z-index: 1000;
            pointer-events: none;
        }
        
        @media (max-width: 767.98px) {
            .container-fluid { padding: 0.5rem; }
            .summary-value { font-size: 1.8rem; }
            .summary-icon { font-size: 3rem; }
        }
        
        .chart-container { position: relative; height: 300px; }
        .activity-item { border-bottom: 1px solid #e3e6f0; padding: 0.75rem 0; }
        .activity-item:last-child { border-bottom: none; }
        .notification-bell { animation: ring 2s infinite; }
        
        @keyframes ring {
            0%, 20%, 50%, 80%, 100% { transform: rotate(0deg); }
            10% { transform: rotate(10deg); }
            30% { transform: rotate(-10deg); }
            60% { transform: rotate(10deg); }
            90% { transform: rotate(-10deg); }
        }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid">
                    
                    <!-- Page Heading -->
                    <div class="d-sm-flex align-items-center justify-content-between mb-4">
                        <div>
                            <h1 class="h3 mb-0 text-gray-800" title="Dashboard">
                                <i class="fas fa-gem text-primary"></i> Dashboard
                            </h1>
                            <p class="mb-0 text-muted">Welcome back, <?php echo htmlspecialchars($_SESSION['fullname']); ?>!</p>
                        </div>
                        <div class="d-flex flex-wrap">
                            <a href="pawn_transactions.php" class="btn btn-primary quick-action mr-2 mb-2" title="Create new pawn transaction">
                                <i class="fas fa-plus"></i> New Pawn
                            </a>
                            <a href="due_collection.php" class="btn btn-success quick-action mr-2 mb-2" title="Collect payments and dues">
                                <i class="fas fa-money-bill-wave"></i> Collections
                            </a>
                            <a href="customers.php" class="btn btn-info quick-action mr-2 mb-2" title="Add new customer">
                                <i class="fas fa-user-plus"></i> Add Customer
                            </a>
                            <a href="pawn_report_detailed.php" class="btn btn-warning quick-action mr-2 mb-2" title="View detailed reports">
                                <i class="fas fa-chart-line"></i> Reports
                            </a>
                            <?php if($_SESSION['usertype'] == 'admin'): ?>
                            <a href="jewel_redemption.php" class="btn btn-secondary quick-action mr-2 mb-2" title="Redeem jewels">
                                <i class="fas fa-gem"></i> Redemption
                            </a>
                            <?php endif; ?>
                            <a href="income-expenses.php" class="btn btn-outline-primary quick-action mb-2" title="Manage income and expenses">
                                <i class="fas fa-calculator"></i> Finance
                            </a>
                        </div>
                    </div>

                    <!-- Alert for overdue loans -->
                    <?php if($overdueCount > 0): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert" title="Overdue loans">
                        <i class="fas fa-exclamation-triangle notification-bell mr-2"></i>
                        <strong>Attention!</strong> You have <strong><?php echo $overdueCount; ?></strong> overdue loans totaling 
                        <strong><?php echo formatCurrency($overdueAmount); ?></strong>. 
                        <a href="due_collection.php" class="alert-link" title="View details">View Details</a>
                        <button type="button" class="close" data-dismiss="alert" title="Close alert">
                            <span aria-hidden="true">&times;</span>
                        </button>
                    </div>
                    <?php endif; ?>

                    <!-- Quick Access Panel -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3 bg-gradient-primary text-white">
                            <h6 class="m-0 font-weight-bold">
                                <i class="fas fa-rocket mr-2"></i>Quick Access - Important Links
                            </h6>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <!-- Pawn Operations -->
                                <div class="col-lg-3 col-md-6 mb-3">
                                    <h6 class="text-primary mb-3"><i class="fas fa-gem"></i> Pawn Operations</h6>
                                    <div class="d-grid gap-2">
                                        <a href="pawn_transactions.php" class="btn btn-primary btn-sm quick-access-btn" title="Create new pawn transaction">
                                            <i class="fas fa-plus"></i> New Pawn Transaction
                                        </a>
                                        <a href="due_collection.php" class="btn btn-success btn-sm quick-access-btn" title="Collect payments">
                                            <i class="fas fa-money-bill-wave"></i> Due Collection
                                        </a>
                                        <a href="due_collection.php?action=increase" class="btn btn-warning btn-sm quick-access-btn" title="Increase loan amount">
                                            <i class="fas fa-arrow-up"></i> Loan Increase
                                        </a>
                                        <?php if($_SESSION['usertype'] == 'admin'): ?>
                                        <a href="jewel_redemption.php" class="btn btn-info btn-sm quick-access-btn" title="Redeem jewels">
                                            <i class="fas fa-gem"></i> Jewel Redemption
                                        </a>
                                        <?php endif; ?>
                                    </div>
                                </div>

                                <!-- Customer Management -->
                                <div class="col-lg-3 col-md-6 mb-3">
                                    <h6 class="text-success mb-3"><i class="fas fa-users"></i> Customer Management</h6>
                                    <div class="d-grid gap-2">
                                        <a href="customers.php" class="btn btn-success btn-sm quick-access-btn" title="Add new customer">
                                            <i class="fas fa-user-plus"></i> Add New Customer
                                        </a>
                                        <a href="manage-customers.php" class="btn btn-outline-success btn-sm quick-access-btn" title="Manage existing customers">
                                            <i class="fas fa-users-cog"></i> Manage Customers
                                        </a>
                                        <a href="customer-jewel-history-report.php" class="btn btn-outline-success btn-sm quick-access-btn" title="View customer history">
                                            <i class="fas fa-history"></i> Customer History
                                        </a>
                                    </div>
                                </div>

                                <!-- Reports & Analytics -->
                                <div class="col-lg-3 col-md-6 mb-3">
                                    <h6 class="text-info mb-3"><i class="fas fa-chart-line"></i> Reports & Analytics</h6>
                                    <div class="d-grid gap-2">
                                        <a href="pawn_report_detailed.php" class="btn btn-info btn-sm quick-access-btn" title="Detailed pawn report">
                                            <i class="fas fa-file-alt"></i> Pawn Report
                                        </a>
                                        <a href="due_collection_report_detailed.php" class="btn btn-info btn-sm quick-access-btn" title="Collection report">
                                            <i class="fas fa-money-check-alt"></i> Collection Report
                                        </a>
                                        <a href="income-expense-report.php" class="btn btn-outline-info btn-sm quick-access-btn" title="Income & expense report">
                                            <i class="fas fa-chart-pie"></i> Income/Expense
                                        </a>
                                        <a href="master-cashflow-report.php" class="btn btn-outline-info btn-sm quick-access-btn" title="Master cashflow report">
                                            <i class="fas fa-file-invoice-dollar"></i> Cashflow Report
                                        </a>
                                    </div>
                                </div>

                                <!-- Financial Management -->
                                <div class="col-lg-3 col-md-6 mb-3">
                                    <h6 class="text-warning mb-3"><i class="fas fa-university"></i> Financial Management</h6>
                                    <div class="d-grid gap-2">
                                        <a href="income-expenses.php" class="btn btn-warning btn-sm quick-access-btn" title="Manage income and expenses">
                                            <i class="fas fa-calculator"></i> Income & Expenses
                                        </a>
                                        <a href="bank_accounts.php" class="btn btn-warning btn-sm quick-access-btn" title="Manage bank accounts">
                                            <i class="fas fa-university"></i> Bank Accounts
                                        </a>
                                        <a href="bank_pledge_management.php" class="btn btn-outline-warning btn-sm quick-access-btn" title="Bank pledge management">
                                            <i class="fas fa-archive"></i> Bank Pledges
                                        </a>
                                        <?php if($_SESSION['usertype'] == 'admin'): ?>
                                        <a href="shop_details.php" class="btn btn-outline-warning btn-sm quick-access-btn" title="Shop configuration">
                                            <i class="fas fa-store"></i> Shop Details
                                        </a>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>

                            <!-- Additional Quick Actions Row -->
                            <hr class="mt-3 mb-3">
                            <div class="row">
                                <div class="col-12">
                                    <h6 class="text-secondary mb-3"><i class="fas fa-tools"></i> Additional Tools</h6>
                                    <div class="d-flex flex-wrap">
                                        <a href="bulk_pawn_upload.php" class="btn btn-secondary btn-sm mr-2 mb-2 quick-access-btn" title="Bulk upload transactions">
                                            <i class="fas fa-file-excel"></i> Bulk Upload
                                        </a>
                                        <?php if($_SESSION['usertype'] == 'admin'): ?>
                                        <a href="create-user.php" class="btn btn-secondary btn-sm mr-2 mb-2 quick-access-btn" title="Create new user">
                                            <i class="fas fa-user-plus"></i> Create User
                                        </a>
                                        <?php endif; ?>
                                        <a href="profile.php" class="btn btn-outline-secondary btn-sm mr-2 mb-2 quick-access-btn" title="View profile">
                                            <i class="fas fa-user"></i> My Profile
                                        </a>
                                        <a href="change-password.php" class="btn btn-outline-secondary btn-sm mr-2 mb-2 quick-access-btn" title="Change password">
                                            <i class="fas fa-key"></i> Change Password
                                        </a>
                                        <a href="jewel_redemption_report.php" class="btn btn-outline-secondary btn-sm mr-2 mb-2 quick-access-btn" title="Redemption report">
                                            <i class="fas fa-gem"></i> Redemption Report
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Key Metrics Cards -->
                    <div class="row mb-4">
                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card summary-card metric-card shadow h-100" title="Active loans">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="summary-label text-uppercase mb-1">Active Loans</div>
                                            <div class="summary-value"><?php echo formatNumber($totalActivePawns); ?></div>
                                            <div class="mt-2 small">Amount: <?php echo formatCurrency($totalActiveLoanAmount); ?></div>
                                        </div>
                                    </div>
                                    <i class="fas fa-gem summary-icon"></i>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card summary-card collection-card shadow h-100" title="Today's collections">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="summary-label text-uppercase mb-1">Today's Collections</div>
                                            <div class="summary-value"><?php echo formatCurrency($todayCollections); ?></div>
                                            <div class="mt-2 small">Monthly: <?php echo formatCurrency($monthlyCollections); ?></div>
                                        </div>
                                    </div>
                                    <i class="fas fa-money-bill-wave summary-icon"></i>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card summary-card loan-card shadow h-100" title="Total customers">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="summary-label text-uppercase mb-1">Total Customers</div>
                                            <div class="summary-value"><?php echo formatNumber($totalCustomers); ?></div>
                                            <div class="mt-2 small">Redemptions: <?php echo formatNumber($totalRedemptions); ?></div>
                                        </div>
                                    </div>
                                    <i class="fas fa-users summary-icon"></i>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-3 col-md-6 mb-4">
                            <div class="card summary-card customer-card shadow h-100" title="Interest earned">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="summary-label text-uppercase mb-1">Interest Earned</div>
                                            <div class="summary-value"><?php echo formatCurrency($totalInterestEarned); ?></div>
                                            <?php if($overdueCount > 0): ?>
                                            <div class="mt-2 small text-warning">
                                                <i class="fas fa-exclamation-triangle"></i> <?php echo $overdueCount; ?> Overdue
                                            </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <i class="fas fa-chart-line summary-icon"></i>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Charts and Statistics Row -->
                    <div class="row mb-4">
                        <!-- Item Type Distribution -->
                        <div class="col-xl-4 col-lg-6 mb-4">
                            <div class="card shadow" title="Item type distribution">
                                <div class="card-header py-3 bg-gradient-primary text-white">
                                    <h6 class="m-0 font-weight-bold">
                                        <i class="fas fa-chart-pie mr-2"></i>Item Type Distribution
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <div class="chart-container">
                                        <canvas id="itemTypeChart"></canvas>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Monthly Collection Trends -->
                        <div class="col-xl-8 col-lg-6 mb-4">
                            <div class="card shadow" title="Monthly collection trend">
                                <div class="card-header py-3 bg-gradient-success text-white">
                                    <h6 class="m-0 font-weight-bold">
                                        <i class="fas fa-chart-area mr-2"></i>Collection Trends (Last 6 Months)
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <div class="chart-container">
                                        <canvas id="collectionTrendChart"></canvas>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Upcoming Dues and Bank Pledges -->
                    <div class="row mb-4">
                        <!-- Upcoming Due Loans -->
                        <div class="col-xl-8 mb-4">
                            <div class="card shadow warning-card" title="Upcoming due loans">
                                <div class="card-header py-3 bg-gradient-warning text-white">
                                    <h6 class="m-0 font-weight-bold">
                                        <i class="fas fa-calendar-exclamation mr-2"></i>Upcoming Due Loans (Next 7 Days)
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <?php if(mysqli_num_rows($upcomingDues) > 0): ?>
                                    <div class="table-responsive">
                                        <table class="table table-hover">
                                            <thead class="thead-light">
                                                <tr>
                                                    <th title="Pawn code">Pawn Code</th>
                                                    <th title="Customer name">Customer</th>
                                                    <th title="Loan amount">Amount</th>
                                                    <th title="Due date">Due Date</th>
                                                    <th title="Action">Action</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php while($due = mysqli_fetch_assoc($upcomingDues)): 
                                                    $dueDate = new DateTime($due['DueDate']);
                                                    $today = new DateTime();
                                                    $diff = $today->diff($dueDate)->days;
                                                    
                                                    $rowClass = '';
                                                    if($dueDate < $today) $rowClass = 'overdue';
                                                    elseif($diff == 0) $rowClass = 'due-today';
                                                    elseif($diff <= 1) $rowClass = 'due-tomorrow';
                                                ?>
                                                <tr class="<?php echo $rowClass; ?>">
                                                    <td><strong><?php echo htmlspecialchars($due['PawnCode']); ?></strong></td>
                                                    <td>
                                                        <?php echo htmlspecialchars($due['CustomerName']); ?>
                                                        <br><small class="text-muted"><?php echo htmlspecialchars($due['MobileNumber']); ?></small>
                                                    </td>
                                                    <td><strong><?php echo formatCurrency($due['CurrentLoanAmount']); ?></strong></td>
                                                    <td>
                                                        <?php echo date('M j, Y', strtotime($due['DueDate'])); ?>
                                                        <?php if($dueDate < $today): ?>
                                                            <br><small class="text-danger"><i class="fas fa-exclamation-triangle"></i> Overdue</small>
                                                        <?php elseif($diff == 0): ?>
                                                            <br><small class="text-warning"><i class="fas fa-clock"></i> Due Today</small>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td>
                                                                          <a href="due_collection.php?pawn_code=<?php echo $due['PawnCode']; ?>" 
                                                                              class="btn btn-sm btn-primary" title="Collect payment">
                                                            <i class="fas fa-money-bill-wave"></i> Collect
                                                        </a>
                                                    </td>
                                                </tr>
                                                <?php endwhile; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                    <?php else: ?>
                                    <div class="text-center py-4">
                                        <i class="fas fa-calendar-check fa-3x text-success mb-3"></i>
                                        <p class="text-muted">No loans due in the next 7 days!</p>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <!-- Bank Pledge Summary -->
                        <div class="col-xl-4 mb-4">
                            <div class="card shadow info-card" title="Bank pledge summary">
                                <div class="card-header py-3 bg-gradient-info text-white">
                                    <h6 class="m-0 font-weight-bold">
                                        <i class="fas fa-university mr-2"></i>Bank Pledge Summary
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <div class="text-center">
                                        <div class="mb-3">
                                            <h4 class="text-primary"><?php echo formatNumber($pledgeData['TotalPledges'] ?? 0); ?></h4>
                                            <small class="text-muted">Total Pledges</small>
                                        </div>
                                        <div class="mb-3">
                                            <h4 class="text-success"><?php echo formatCurrency($pledgeData['ActivePledgeAmount'] ?? 0); ?></h4>
                                            <small class="text-muted">Active Pledge Amount</small>
                                        </div>
                                        <a href="bank_pledge_management.php" class="btn btn-info btn-block" title="View details">
                                            <i class="fas fa-eye"></i> View Details
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Recent Activities -->
                    <div class="row">
                        <!-- Recent Pawn Transactions -->
                        <div class="col-xl-6 mb-4">
                            <div class="card shadow success-card" title="Recent pawn transactions">
                                <div class="card-header py-3 bg-gradient-primary text-white">
                                    <h6 class="m-0 font-weight-bold">
                                        <i class="fas fa-gem mr-2"></i>Recent Pawn Transactions
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <?php if(mysqli_num_rows($recentPawns) > 0): ?>
                                    <?php while($pawn = mysqli_fetch_assoc($recentPawns)): ?>
                                    <div class="activity-item" title="Pawn details">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <strong><?php echo htmlspecialchars($pawn['PawnCode']); ?></strong>
                                                <div class="small text-muted">
                                                    <?php echo htmlspecialchars($pawn['CustomerName']); ?> - 
                                                    <?php echo htmlspecialchars($pawn['ItemType']); ?>
                                                </div>
                                            </div>
                                            <div class="text-right">
                                                <div class="font-weight-bold text-success">
                                                    <?php echo formatCurrency($pawn['LoanAmount']); ?>
                                                </div>
                                                <div class="small text-muted">
                                                    <?php echo date('M j, Y', strtotime($pawn['LoanDate'])); ?>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endwhile; ?>
                                    <?php else: ?>
                                    <div class="text-center py-4">
                                        <i class="fas fa-gem fa-3x text-muted mb-3"></i>
                                        <p class="text-muted">No recent transactions</p>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <!-- Recent Collections -->
                        <div class="col-xl-6 mb-4">
                            <div class="card shadow alert-card" title="Recent collections">
                                <div class="card-header py-3 bg-gradient-success text-white">
                                    <h6 class="m-0 font-weight-bold">
                                        <i class="fas fa-money-bill-wave mr-2"></i>Recent Collections
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <?php if(mysqli_num_rows($recentCollections) > 0): ?>
                                    <?php while($collection = mysqli_fetch_assoc($recentCollections)): ?>
                                    <div class="activity-item" title="Collection details">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <strong><?php echo htmlspecialchars($collection['CollectionCode']); ?></strong>
                                                <div class="small text-muted">
                                                    <?php echo htmlspecialchars($collection['CustomerName']); ?> - 
                                                    <?php echo htmlspecialchars($collection['PawnCode']); ?>
                                                </div>
                                            </div>
                                            <div class="text-right">
                                                <div class="font-weight-bold text-primary">
                                                    <?php echo formatCurrency($collection['AmountPaid']); ?>
                                                </div>
                                                <div class="small text-muted">
                                                    <?php echo date('M j, Y', strtotime($collection['CollectionDate'])); ?>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endwhile; ?>
                                    <?php else: ?>
                                    <div class="text-center py-4">
                                        <i class="fas fa-money-bill-wave fa-3x text-muted mb-3"></i>
                                        <p class="text-muted">No recent collections</p>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>

                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>

    <!-- Chart Data and Scripts -->
    <script>
        // Item Type Distribution Chart
        const itemTypeData = {
            labels: [
                <?php 
                mysqli_data_seek($itemTypeStats, 0);
                while($item = mysqli_fetch_assoc($itemTypeStats)) {
                    echo "'" . $item['ItemType'] . "',";
                }
                ?>
            ],
            datasets: [{
                data: [
                    <?php 
                    mysqli_data_seek($itemTypeStats, 0);
                    while($item = mysqli_fetch_assoc($itemTypeStats)) {
                        echo $item['TotalAmount'] . ",";
                    }
                    ?>
                ],
                backgroundColor: [
                    '#FF6384', '#36A2EB', '#FFCE56', '#4BC0C0', '#9966FF', '#FF9F40'
                ]
            }]
        };

        const itemTypeConfig = {
            type: 'doughnut',
            data: itemTypeData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        };

        // Collection Trend Chart
        const collectionTrendData = {
            labels: [
                <?php 
                $months = [];
                $amounts = [];
                while($trend = mysqli_fetch_assoc($monthlyTrends)) {
                    $months[] = "'" . date('M Y', strtotime($trend['Month'] . '-01')) . "'";
                    $amounts[] = $trend['TotalCollection'];
                }
                echo implode(',', array_reverse($months));
                ?>
            ],
            datasets: [{
                label: 'Collections (₹)',
                data: [<?php echo implode(',', array_reverse($amounts)); ?>],
                borderColor: '#28a745',
                backgroundColor: 'rgba(40, 167, 69, 0.1)',
                fill: true,
                tension: 0.4
            }]
        };

        const collectionTrendConfig = {
            type: 'line',
            data: collectionTrendData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return '₹' + value.toLocaleString();
                            }
                        }
                    }
                },
                plugins: {
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return 'Amount: ₹' + context.parsed.y.toLocaleString();
                            }
                        }
                    }
                }
            }
        };

        // Initialize Charts
        document.addEventListener('DOMContentLoaded', function() {
            new Chart(document.getElementById('itemTypeChart'), itemTypeConfig);
            new Chart(document.getElementById('collectionTrendChart'), collectionTrendConfig);
        });
    </script>

    <?php include_once('includes/footer2.php');?>
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
</body>
</html>