<?php
session_start();

// Check if config file exists
$configPath = __DIR__ . '/includes/config.php';
if (!file_exists($configPath)) {
    die("Error: Database configuration file not found. Please create includes/config.php");
}

include_once($configPath);

// Check if database connection exists
if (!isset($con) || !$con) {
    die("Error: Database connection failed. Please check your database configuration.");
}
// Load branding helper
include_once(__DIR__ . '/includes/brand.php');

if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$msg = "";
$userId = $_SESSION['uid'];

// Search and pagination variables
$search = isset($_GET['search']) ? mysqli_real_escape_string($con, $_GET['search']) : '';
$limit = 10; // Records per page
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// Create uploads directory structure if it doesn't exist
$baseUploadDir = 'uploads/';
$customerUploadDir = $baseUploadDir . 'customers/';
$photoUploadDir = $customerUploadDir . 'photos/';
$idProofUploadDir = $customerUploadDir . 'idproofs/';

// Create directories if they don't exist
if (!file_exists($baseUploadDir)) {
    mkdir($baseUploadDir, 0777, true);
}
if (!file_exists($customerUploadDir)) {
    mkdir($customerUploadDir, 0777, true);
}
if (!file_exists($photoUploadDir)) {
    mkdir($photoUploadDir, 0777, true);
}
if (!file_exists($idProofUploadDir)) {
    mkdir($idProofUploadDir, 0777, true);
}

// Enhanced function to handle file upload with detailed error reporting
function handleFileUpload($file, $uploadDir, $prefix, $customerName = '', &$errorMsg = '') {
    // Check if file was uploaded
    if (!isset($file) || $file['error'] != 0) {
        if (isset($file['error']) && $file['error'] != UPLOAD_ERR_NO_FILE) {
            switch ($file['error']) {
                case UPLOAD_ERR_INI_SIZE:
                case UPLOAD_ERR_FORM_SIZE:
                    $errorMsg = 'File size exceeds the maximum allowed limit.';
                    break;
                case UPLOAD_ERR_PARTIAL:
                    $errorMsg = 'File upload was incomplete.';
                    break;
                case UPLOAD_ERR_NO_TMP_DIR:
                    $errorMsg = 'Temporary folder not found.';
                    break;
                case UPLOAD_ERR_CANT_WRITE:
                    $errorMsg = 'Failed to write file to disk.';
                    break;
                default:
                    $errorMsg = 'Unknown file upload error.';
            }
            return false;
        }
        return ''; // No file uploaded, which is OK
    }
    
    // Validate file size (5MB max)
    if ($file['size'] > 5 * 1024 * 1024) {
        $errorMsg = 'File size must be less than 5MB. Uploaded file is ' . round($file['size'] / (1024 * 1024), 2) . 'MB.';
        return false;
    }
    
    // Validate file type
    $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'pdf'];
    
    if (!in_array($fileExtension, $allowedExtensions)) {
        $errorMsg = 'Invalid file type. Only JPG, PNG, GIF, and PDF files are allowed.';
        return false;
    }
    
    // Additional validation for image files
    if (in_array($fileExtension, ['jpg', 'jpeg', 'png', 'gif'])) {
        $imageInfo = @getimagesize($file['tmp_name']);
        if ($imageInfo === false) {
            $errorMsg = 'Invalid image file. Please upload a valid image.';
            return false;
        }
        
        // Check image dimensions (optional - can be adjusted)
        if ($imageInfo[0] > 5000 || $imageInfo[1] > 5000) {
            $errorMsg = 'Image dimensions too large. Maximum size is 5000x5000 pixels.';
            return false;
        }
    }
    
    // Create secure filename
    $customerNameSafe = preg_replace('/[^A-Za-z0-9_-]/', '_', $customerName);
    $customerNameSafe = substr($customerNameSafe, 0, 20); // Limit length
    $fileName = $prefix . '_' . $customerNameSafe . '_' . time() . '_' . uniqid() . '.' . $fileExtension;
    $filePath = $uploadDir . $fileName;
    
    // Ensure upload directory exists and is writable
    if (!is_dir($uploadDir)) {
        if (!mkdir($uploadDir, 0755, true)) {
            $errorMsg = 'Failed to create upload directory.';
            return false;
        }
    }
    
    if (!is_writable($uploadDir)) {
        $errorMsg = 'Upload directory is not writable.';
        return false;
    }
    
    // Move uploaded file
    if (move_uploaded_file($file['tmp_name'], $filePath)) {
        // Set file permissions
        chmod($filePath, 0644);
        return $filePath;
    }
    
    $errorMsg = 'Failed to save uploaded file.';
    return false;
}

// Handle form submissions
if ($_POST) {
    if (isset($_POST['add_customer'])) {
        $customerCode = "CUST" . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        // Safely read inputs, defaulting to empty strings to avoid undefined index warnings
        $name = mysqli_real_escape_string($con, $_POST['name'] ?? '');
        $fatherHusbandName = mysqli_real_escape_string($con, $_POST['father_husband_name'] ?? '');
        $mobileNumber = mysqli_real_escape_string($con, $_POST['mobile_number'] ?? '');
        $address = mysqli_real_escape_string($con, $_POST['address'] ?? '');
        $idProofType = mysqli_real_escape_string($con, $_POST['id_proof_type'] ?? '');
        // Derive IDProofNumber from specific fields if not explicitly provided
        $rawIdProofNumber = $_POST['id_proof_number'] ?? null;
        if ($rawIdProofNumber === null || $rawIdProofNumber === '') {
            if ($idProofType === 'PAN Card') {
                $rawIdProofNumber = $_POST['pan_number'] ?? '';
            } elseif ($idProofType === 'Aadhaar Card') {
                $rawIdProofNumber = $_POST['aadhar_number'] ?? '';
            } else {
                $rawIdProofNumber = '';
            }
        }
        $idProofNumber = mysqli_real_escape_string($con, $rawIdProofNumber);
        $panNumber = mysqli_real_escape_string($con, strtoupper($_POST['pan_number'] ?? ''));
        $aadharNumber = mysqli_real_escape_string($con, $_POST['aadhar_number'] ?? '');
        $nomineeName = mysqli_real_escape_string($con, $_POST['nominee_name'] ?? '');
        $nomineeRelationship = mysqli_real_escape_string($con, $_POST['nominee_relationship'] ?? '');
        
        // Handle photo upload
        $photoErrorMsg = '';
        $photoPath = handleFileUpload($_FILES['photo'] ?? null, $photoUploadDir, 'photo', $name, $photoErrorMsg);
        if ($photoPath === false) {
            $msg = "Photo upload error: " . $photoErrorMsg;
        } else {
            // Handle ID proof upload
            $idProofErrorMsg = '';
            $idProofPath = handleFileUpload($_FILES['id_proof'] ?? null, $idProofUploadDir, 'idproof', $name, $idProofErrorMsg);
            if ($idProofPath === false && isset($_FILES['id_proof']) && $_FILES['id_proof']['error'] == 0) {
                $msg = "ID Proof upload error: " . $idProofErrorMsg;
            } else {
                $query = "INSERT INTO tblcustomers (CustomerCode, Name, FatherHusbandName, MobileNumber, Address, IDProofType, IDProofNumber, PANNumber, AadharNumber, NomineeName, NomineeRelationship, PhotoPath, IDProofPath) 
                          VALUES ('$customerCode', '$name', '$fatherHusbandName', '$mobileNumber', '$address', '$idProofType', '$idProofNumber', '$panNumber', '$aadharNumber', '$nomineeName', '$nomineeRelationship', '$photoPath', '$idProofPath')";
                
                if (mysqli_query($con, $query)) {
                    $msg = "Customer added successfully!";
                } else {
                    $msg = "Error adding customer: " . mysqli_error($con);
                }
            }
        }
    }
    
    if (isset($_POST['update_customer'])) {
        $id = (int)($_POST['customer_id'] ?? 0);
        $name = mysqli_real_escape_string($con, $_POST['name'] ?? '');
        $fatherHusbandName = mysqli_real_escape_string($con, $_POST['father_husband_name'] ?? '');
        $mobileNumber = mysqli_real_escape_string($con, $_POST['mobile_number'] ?? '');
        $address = mysqli_real_escape_string($con, $_POST['address'] ?? '');
        $idProofType = mysqli_real_escape_string($con, $_POST['id_proof_type'] ?? '');
        $rawIdProofNumber = $_POST['id_proof_number'] ?? null;
        if ($rawIdProofNumber === null || $rawIdProofNumber === '') {
            if ($idProofType === 'PAN Card') {
                $rawIdProofNumber = $_POST['pan_number'] ?? '';
            } elseif ($idProofType === 'Aadhaar Card') {
                $rawIdProofNumber = $_POST['aadhar_number'] ?? '';
            } else {
                $rawIdProofNumber = '';
            }
        }
        $idProofNumber = mysqli_real_escape_string($con, $rawIdProofNumber);
        $panNumber = mysqli_real_escape_string($con, strtoupper($_POST['pan_number'] ?? ''));
        $aadharNumber = mysqli_real_escape_string($con, $_POST['aadhar_number'] ?? '');
        $nomineeName = mysqli_real_escape_string($con, $_POST['nominee_name'] ?? '');
        $nomineeRelationship = mysqli_real_escape_string($con, $_POST['nominee_relationship'] ?? '');
        $status = mysqli_real_escape_string($con, $_POST['status'] ?? 'Active');
        
        // Get existing customer data
        $existingQuery = mysqli_query($con, "SELECT PhotoPath, IDProofPath FROM tblcustomers WHERE ID=$id");
        $existing = mysqli_fetch_assoc($existingQuery);
        
        $photoPath = $existing['PhotoPath'];
        $idProofPath = $existing['IDProofPath'];
        
        // Handle photo upload
        if (isset($_FILES['photo']) && $_FILES['photo']['error'] == 0) {
            $photoErrorMsg = '';
            $newPhotoPath = handleFileUpload($_FILES['photo'], $photoUploadDir, 'photo', $name, $photoErrorMsg);
            if ($newPhotoPath === false) {
                $msg = "Photo upload error: " . $photoErrorMsg;
            } else {
                // Delete old photo
                if ($photoPath && file_exists($photoPath)) {
                    unlink($photoPath);
                }
                $photoPath = $newPhotoPath;
            }
        }
        
        // Handle ID proof upload
        if (isset($_FILES['id_proof']) && $_FILES['id_proof']['error'] == 0) {
            $idProofErrorMsg = '';
            $newIdProofPath = handleFileUpload($_FILES['id_proof'], $idProofUploadDir, 'idproof', $name, $idProofErrorMsg);
            if ($newIdProofPath === false) {
                $msg = "ID Proof upload error: " . $idProofErrorMsg;
            } else {
                // Delete old ID proof
                if ($idProofPath && file_exists($idProofPath)) {
                    unlink($idProofPath);
                }
                $idProofPath = $newIdProofPath;
            }
        }
        
        if (!$msg) { // Only update if no upload errors
            $query = "UPDATE tblcustomers SET Name='$name', FatherHusbandName='$fatherHusbandName', MobileNumber='$mobileNumber', 
                      Address='$address', IDProofType='$idProofType', IDProofNumber='$idProofNumber', PANNumber='$panNumber', 
                      AadharNumber='$aadharNumber', NomineeName='$nomineeName', NomineeRelationship='$nomineeRelationship',
                      PhotoPath='$photoPath', IDProofPath='$idProofPath', Status='$status' WHERE ID=$id";
            
            if (mysqli_query($con, $query)) {
                $msg = "Customer updated successfully!";
            } else {
                $msg = "Error updating customer: " . mysqli_error($con);
            }
        }
    }
}

// Delete customer
if (isset($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    
    // Get file paths before deletion
    $fileQuery = mysqli_query($con, "SELECT PhotoPath, IDProofPath FROM tblcustomers WHERE ID=$id");
    $fileData = mysqli_fetch_assoc($fileQuery);
    
    $query = "DELETE FROM tblcustomers WHERE ID=$id";
    if (mysqli_query($con, $query)) {
        // Delete associated files
        if ($fileData['PhotoPath'] && file_exists($fileData['PhotoPath'])) {
            unlink($fileData['PhotoPath']);
        }
        if ($fileData['IDProofPath'] && file_exists($fileData['IDProofPath'])) {
            unlink($fileData['IDProofPath']);
        }
        $msg = "Customer deleted successfully!";
    } else {
        $msg = "Error deleting customer: " . mysqli_error($con);
    }
}

// Get customer for editing
$editCustomer = null;
if (isset($_GET['edit'])) {
    $id = (int)$_GET['edit'];
    $result = mysqli_query($con, "SELECT * FROM tblcustomers WHERE ID=$id");
    $editCustomer = mysqli_fetch_assoc($result);
}

// Build search query
$whereClause = "";
if (!empty($search)) {
    $whereClause = "WHERE (Name LIKE '%$search%' OR MobileNumber LIKE '%$search%' OR CustomerCode LIKE '%$search%' OR FatherHusbandName LIKE '%$search%' OR PANNumber LIKE '%$search%' OR AadharNumber LIKE '%$search%' OR NomineeName LIKE '%$search%')";
}

// Get total count for pagination
$countQuery = "SELECT COUNT(*) as total FROM tblcustomers $whereClause";
$countResult = mysqli_query($con, $countQuery);
$totalRecords = mysqli_fetch_assoc($countResult)['total'];
$totalPages = ceil($totalRecords / $limit);

// Fetch customers with search and pagination
$customersQuery = mysqli_query($con, "SELECT * FROM tblcustomers $whereClause ORDER BY CreatedDate DESC LIMIT $limit OFFSET $offset");
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?php echo page_title('Customer Management'); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/compact.css" rel="stylesheet">
    <style>
        .search-container {
            background: #f8f9fc;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        
        /* Enhanced Image Upload Styling */
        .image-upload-container {
            border: 2px dashed #dee2e6;
            border-radius: 8px;
            padding: 15px;
            text-align: center;
            transition: all 0.3s ease;
        }
        
        .image-upload-container:hover {
            border-color: #007bff;
            background-color: #f8f9fa;
        }
        
        .image-preview-container {
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 10px;
            background-color: #f8f9fa;
        }
        
        .current-image-container {
            border: 1px solid #28a745;
            border-radius: 8px;
            padding: 10px;
            background-color: #f8fff9;
        }
        
        .image-thumbnail {
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            transition: transform 0.2s ease;
        }
        
        .image-thumbnail:hover {
            transform: scale(1.05);
        }
        
        .file-info {
            font-size: 0.85em;
            color: #6c757d;
            margin-top: 5px;
        }
        
        .upload-icon {
            font-size: 2em;
            color: #6c757d;
            margin-bottom: 10px;
        }
        
        /* Tamil tooltip styling */
        [title] {
            position: relative;
        }
        
        [title]:hover::after {
            content: attr(title);
            position: absolute;
            bottom: 100%;
            left: 50%;
            transform: translateX(-50%);
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 10px;
            white-space: nowrap;
            z-index: 1000;
            pointer-events: none;
        }
        
        [title]:hover::before {
            content: '';
            position: absolute;
            bottom: 95%;
            left: 50%;
            transform: translateX(-50%);
            border: 4px solid transparent;
            border-top-color: rgba(0, 0, 0, 0.8);
            z-index: 1000;
            pointer-events: none;
        }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid">
                     
                    <?php if($msg) { ?>
                        <div class="alert alert-info"><?php echo $msg; ?></div>
                    <?php } ?>
                    
                    
                    <!-- Add/Edit Customer Form -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary">
                                <?php echo $editCustomer ? 'Edit Customer' : 'Add New Customer'; ?>
                            </h6>
                        </div>
                        <div class="card-body">
                            <form method="POST" enctype="multipart/form-data">
                                <?php if($editCustomer) { ?>
                                    <input type="hidden" name="customer_id" value="<?php echo $editCustomer['ID']; ?>">
                                <?php } ?>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label title="Name">Name <span class="text-danger">*</span></label>
                                            <input type="text" name="name" class="form-control" required title="Name"
                                                   value="<?php echo $editCustomer ? htmlspecialchars($editCustomer['Name']) : ''; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label title="Father/Husband name">Father/Husband Name <span class="text-danger">*</span></label>
                                            <input type="text" name="father_husband_name" class="form-control" required title="Father/Husband name"
                                                   value="<?php echo $editCustomer ? htmlspecialchars($editCustomer['FatherHusbandName']) : ''; ?>">
                                        </div>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label title="Mobile number">Mobile Number <span class="text-danger">*</span></label>
                                            <input type="number" name="mobile_number" class="form-control" required title="Mobile number"
                                                   value="<?php echo $editCustomer ? htmlspecialchars($editCustomer['MobileNumber']) : ''; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label title="ID proof type">ID Proof Type <span class="text-danger">*</span></label>
                                            <select name="id_proof_type" class="form-control" required title="ID proof type">
                                                <option value="">Select ID Proof Type</option>
                                                <option value="Aadhaar Card" <?php echo ($editCustomer && $editCustomer['IDProofType'] == 'Aadhaar Card') ? 'selected' : ''; ?>>Aadhaar Card</option>
                                                <option value="PAN Card" <?php echo ($editCustomer && $editCustomer['IDProofType'] == 'PAN Card') ? 'selected' : ''; ?>>PAN Card</option>
                                                <option value="Driving License" <?php echo ($editCustomer && $editCustomer['IDProofType'] == 'Driving License') ? 'selected' : ''; ?>>Driving License</option>
                                                <option value="Voter ID" <?php echo ($editCustomer && $editCustomer['IDProofType'] == 'Voter ID') ? 'selected' : ''; ?>>Voter ID</option>
                                                <option value="Passport" <?php echo ($editCustomer && $editCustomer['IDProofType'] == 'Passport') ? 'selected' : ''; ?>>Passport</option>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label title="PAN Number">PAN Number <span class="text-danger">*</span></label>
                                            <input type="text" name="pan_number" class="form-control" required title="PAN Number" 
                                                   pattern="[A-Z]{5}[0-9]{4}[A-Z]{1}" maxlength="10" style="text-transform: uppercase;"
                                                   value="<?php echo $editCustomer ? htmlspecialchars($editCustomer['PANNumber']) : ''; ?>">
                                            <small class="form-text text-muted">Format: ABCDE1234F</small>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label title="Aadhar Number">Aadhar Number <span class="text-danger">*</span></label>
                                            <input type="text" name="aadhar_number" class="form-control" required title="Aadhar Number"
                                                   pattern="[0-9]{12}" maxlength="12"
                                                   value="<?php echo $editCustomer ? htmlspecialchars($editCustomer['AadharNumber']) : ''; ?>">
                                            <small class="form-text text-muted">12 digit Aadhar number</small>
                                        </div>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group" id="idProofNumberGroup" style="display:none;">
                                            <label title="ID Proof Number">ID Proof Number</label>
                                            <input type="text" name="id_proof_number" class="form-control" title="ID Proof Number"
                                                   value="<?php echo $editCustomer ? htmlspecialchars($editCustomer['IDProofNumber']) : ''; ?>">
                                            <small class="form-text text-muted">Required for Driving License, Voter ID, Passport</small>
                                        </div>
                                    </div>
                                    <div class="col-md-6"></div>
                                </div>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label title="Nominee Name">Nominee Name <span class="text-danger">*</span></label>
                                            <input type="text" name="nominee_name" class="form-control" required title="Nominee Name"
                                                   value="<?php echo $editCustomer ? htmlspecialchars($editCustomer['NomineeName']) : ''; ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label title="Nominee Relationship">Nominee Relationship <span class="text-danger">*</span></label>
                                            <select name="nominee_relationship" class="form-control" required title="Nominee Relationship">
                                                <option value="">Select Relationship</option>
                                                <option value="Father" <?php echo ($editCustomer && $editCustomer['NomineeRelationship'] == 'Father') ? 'selected' : ''; ?>>Father</option>
                                                <option value="Mother" <?php echo ($editCustomer && $editCustomer['NomineeRelationship'] == 'Mother') ? 'selected' : ''; ?>>Mother</option>
                                                <option value="Spouse" <?php echo ($editCustomer && $editCustomer['NomineeRelationship'] == 'Spouse') ? 'selected' : ''; ?>>Spouse</option>
                                                <option value="Son" <?php echo ($editCustomer && $editCustomer['NomineeRelationship'] == 'Son') ? 'selected' : ''; ?>>Son</option>
                                                <option value="Daughter" <?php echo ($editCustomer && $editCustomer['NomineeRelationship'] == 'Daughter') ? 'selected' : ''; ?>>Daughter</option>
                                                <option value="Brother" <?php echo ($editCustomer && $editCustomer['NomineeRelationship'] == 'Brother') ? 'selected' : ''; ?>>Brother</option>
                                                <option value="Sister" <?php echo ($editCustomer && $editCustomer['NomineeRelationship'] == 'Sister') ? 'selected' : ''; ?>>Sister</option>
                                                <option value="Other" <?php echo ($editCustomer && $editCustomer['NomineeRelationship'] == 'Other') ? 'selected' : ''; ?>>Other</option>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label title="Photo upload">Customer Photo</label>
                                            <input type="file" name="photo" id="photoInput" class="form-control-file" accept="image/*" title="Photo upload" onchange="previewImage(this, 'photoPreview')">
                                            <small class="form-text text-muted">Max file size: 5MB. Allowed: JPG, PNG, GIF</small>
                                            
                                            <!-- Current Photo Preview -->
                                            <div class="mt-3" id="currentPhotoContainer" <?php echo (!$editCustomer || !$editCustomer['PhotoPath']) ? 'style="display:none"' : ''; ?>>
                                                <label class="font-weight-bold">Current Photo:</label><br>
                                                <?php if($editCustomer && $editCustomer['PhotoPath'] && file_exists($editCustomer['PhotoPath'])) { ?>
                                                    <img src="<?php echo $editCustomer['PhotoPath']; ?>" class="img-thumbnail" style="max-width: 150px; max-height: 150px; object-fit: cover;" alt="Current Photo">
                                                    <br><small class="text-muted mt-1">
                                                        <a href="<?php echo $editCustomer['PhotoPath']; ?>" target="_blank" class="text-info">
                                                            <i class="fas fa-external-link-alt"></i> View Full Size
                                                        </a>
                                                    </small>
                                                <?php } elseif($editCustomer && $editCustomer['PhotoPath']) { ?>
                                                    <div class="alert alert-warning py-2">
                                                        <i class="fas fa-exclamation-triangle"></i> Photo file not found: <?php echo htmlspecialchars(basename($editCustomer['PhotoPath'])); ?>
                                                    </div>
                                                <?php } ?>
                                            </div>
                                            
                                            <!-- New Photo Preview -->
                                            <div class="mt-3" id="photoPreviewContainer" style="display:none">
                                                <label class="font-weight-bold">New Photo Preview:</label><br>
                                                <img id="photoPreview" class="img-thumbnail" style="max-width: 150px; max-height: 150px; object-fit: cover;" alt="Photo Preview">
                                                <br><button type="button" class="btn btn-sm btn-secondary mt-1" onclick="clearImagePreview('photoInput', 'photoPreview')">
                                                    <i class="fas fa-times"></i> Clear Selection
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <!-- Empty column for layout balance -->
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label title="Address">Address</label>
                                            <textarea name="address" class="form-control" rows="3" title="Address"><?php echo $editCustomer ? htmlspecialchars($editCustomer['Address']) : ''; ?></textarea>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label title="ID proof upload">ID Proof Document</label>
                                            <input type="file" name="id_proof" id="idProofInput" class="form-control-file" accept=".pdf,.jpg,.jpeg,.png" title="ID proof upload" onchange="previewDocument(this, 'idProofPreview')">
                                            <small class="form-text text-muted">Max file size: 5MB. Allowed: PDF, JPG, PNG</small>
                                            
                                            <!-- Current ID Proof Preview -->
                                            <div class="mt-3" id="currentIdProofContainer" <?php echo (!$editCustomer || !$editCustomer['IDProofPath']) ? 'style="display:none"' : ''; ?>>
                                                <label class="font-weight-bold">Current ID Proof:</label><br>
                                                <?php if($editCustomer && $editCustomer['IDProofPath'] && file_exists($editCustomer['IDProofPath'])) { ?>
                                                    <?php 
                                                    $idProofExt = strtolower(pathinfo($editCustomer['IDProofPath'], PATHINFO_EXTENSION));
                                                    if(in_array($idProofExt, ['jpg', 'jpeg', 'png', 'gif'])) { ?>
                                                        <img src="<?php echo $editCustomer['IDProofPath']; ?>" class="img-thumbnail" style="max-width: 150px; max-height: 150px; object-fit: cover;" alt="ID Proof">
                                                    <?php } else { ?>
                                                        <div class="border p-3 text-center" style="max-width: 150px;">
                                                            <i class="fas fa-file-pdf fa-3x text-danger"></i>
                                                            <br><small class="text-muted"><?php echo strtoupper($idProofExt); ?> Document</small>
                                                        </div>
                                                    <?php } ?>
                                                    <br><small class="text-muted mt-1">
                                                        <a href="<?php echo $editCustomer['IDProofPath']; ?>" target="_blank" class="text-info">
                                                            <i class="fas fa-external-link-alt"></i> View Document
                                                        </a>
                                                    </small>
                                                <?php } elseif($editCustomer && $editCustomer['IDProofPath']) { ?>
                                                    <div class="alert alert-warning py-2">
                                                        <i class="fas fa-exclamation-triangle"></i> Document file not found: <?php echo htmlspecialchars(basename($editCustomer['IDProofPath'])); ?>
                                                    </div>
                                                <?php } ?>
                                            </div>
                                            
                                            <!-- New ID Proof Preview -->
                                            <div class="mt-3" id="idProofPreviewContainer" style="display:none">
                                                <label class="font-weight-bold">New Document Preview:</label><br>
                                                <div id="idProofPreview"></div>
                                                <br><button type="button" class="btn btn-sm btn-secondary mt-1" onclick="clearDocumentPreview('idProofInput', 'idProofPreview')">
                                                    <i class="fas fa-times"></i> Clear Selection
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <?php if($editCustomer) { ?>
                                    <div class="form-group">
                                        <label title="Status">Status</label>
                                        <select name="status" class="form-control" title="Status">
                                            <option value="Active" <?php echo ($editCustomer['Status'] == 'Active') ? 'selected' : ''; ?>>Active</option>
                                            <option value="Inactive" <?php echo ($editCustomer['Status'] == 'Inactive') ? 'selected' : ''; ?>>Inactive</option>
                                        </select>
                                    </div>
                                <?php } ?>
                                <button type="submit" name="<?php echo $editCustomer ? 'update_customer' : 'add_customer'; ?>" 
                                        class="btn btn-primary">
                                    <i class="fas fa-save"></i> <?php echo $editCustomer ? 'Update' : 'Add'; ?> Customer
                                </button>
                                <?php if($editCustomer) { ?>
                                    <a href="customers.php" class="btn btn-secondary">Cancel</a>
                                <?php } ?>
                            </form>
                        </div>
                    </div>
                    
                    <!-- Search Section -->
                    <div class="search-container">
                        <form method="GET" class="row">
                            <div class="col-md-8">
                                <input type="text" name="search" class="form-control" 
                                       placeholder="Search by name, mobile, code" title="Search by name, mobile, code"
                                       value="<?php echo htmlspecialchars($search); ?>">
                            </div>
                            <div class="col-md-2">
                                <button type="submit" class="btn btn-primary btn-block" title="Search">
                                    <i class="fas fa-search"></i> Search
                                </button>
                            </div>
                            <div class="col-md-2">
                                <a href="customers.php" class="btn btn-secondary btn-block" title="Clear">
                                    <i class="fas fa-refresh"></i> Clear
                                </a>
                            </div>
                        </form>
                    </div>
                    <!-- Customers List -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3 d-flex justify-content-between align-items-center">
                            <h6 class="m-0 font-weight-bold text-primary">Customer List</h6>
                            <small class="text-muted">Total: <?php echo $totalRecords; ?></small>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered">
                                    <thead>
                                        <tr>
                                            <th title="Number">No</th>
                                            <th title="Customer code">Code</th>
                                            <th title="Customer name">Name</th>
                                            <th title="Father/Husband name">Father/Husband</th>
                                            <th title="Mobile number">Mobile</th>
                                            <th title="PAN/Aadhar">PAN/Aadhar</th>
                                            <th title="Nominee details">Nominee</th>
                                            <th title="Customer photo">Photo</th>
                                            <th title="Record status">Status</th>
                                            <th title="Actions">Action</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php 
                                        if (mysqli_num_rows($customersQuery) > 0) {
                                            $i = $offset + 1; 
                                            while($customer = mysqli_fetch_assoc($customersQuery)) { 
                                        ?>
                                            <tr>
                                                <td><?php echo $i++; ?></td>
                                                <td><?php echo htmlspecialchars($customer['CustomerCode']); ?></td>
                                                <td><?php echo htmlspecialchars($customer['Name']); ?></td>
                                                <td><?php echo htmlspecialchars($customer['FatherHusbandName']); ?></td>
                                                <td><?php echo htmlspecialchars($customer['MobileNumber']); ?></td>
                                                <td>
                                                    <small><strong>PAN:</strong> <?php echo htmlspecialchars($customer['PANNumber'] ?? 'N/A'); ?></small><br>
                                                    <small><strong>Aadhar:</strong> <?php echo htmlspecialchars($customer['AadharNumber'] ?? 'N/A'); ?></small>
                                                </td>
                                                <td>
                                                    <small><strong><?php echo htmlspecialchars($customer['NomineeName'] ?? 'N/A'); ?></strong></small><br>
                                                    <small class="text-muted"><?php echo htmlspecialchars($customer['NomineeRelationship'] ?? 'N/A'); ?></small>
                                                </td>
                                                <td>
                                                    <?php if($customer['PhotoPath']) { ?>
                                                        <img src="<?php echo $customer['PhotoPath']; ?>" width="50" height="50" class="rounded">
                                                    <?php } else { ?>
                                                        <span class="text-muted">No Photo</span>
                                                    <?php } ?>
                                                </td>
                                                <td>
                                                    <span class="badge badge-<?php echo $customer['Status'] == 'Active' ? 'success' : 'secondary'; ?>">
                                                        <?php echo $customer['Status'] == 'Active' ? 'Active' : 'Inactive'; ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <button onclick="viewCustomer(<?php echo $customer['ID']; ?>)" class="btn btn-sm btn-info" title="View">
                                                        <i class="fas fa-eye"></i>
                                                    </button>
                                                    <a href="?edit=<?php echo $customer['ID']; ?>" class="btn btn-sm btn-primary" title="Edit">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <a href="?delete=<?php echo $customer['ID']; ?>" 
                                                       onclick="return confirm('Are you sure to delete?')" 
                                                       class="btn btn-sm btn-danger" title="Delete">
                                                        <i class="fas fa-trash"></i>
                                                    </a>
                                                </td>
                                            </tr>
                                        <?php 
                                            }
                                        } else { 
                                        ?>
                                            <tr>
                                                <td colspan="10" class="text-center">No customers found</td>
                                            </tr>
                                        <?php } ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <!-- Pagination -->
                            <?php if ($totalPages > 1) { ?>
                                <nav aria-label="Page navigation">
                                    <ul class="pagination justify-content-center">
                                        <?php if ($page > 1) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?php echo ($page-1); ?>&search=<?php echo urlencode($search); ?>" title="Previous page">Previous</a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php for ($i = 1; $i <= $totalPages; $i++) { ?>
                                            <li class="page-item <?php echo ($i == $page) ? 'active' : ''; ?>">
                                                <a class="page-link" href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>" title="Page <?php echo $i; ?>"><?php echo $i; ?></a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php if ($page < $totalPages) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?php echo ($page+1); ?>&search=<?php echo urlencode($search); ?>" title="Next page">Next</a>
                                            </li>
                                        <?php } ?>
                                    </ul>
                                </nav>
                            <?php } ?>
                        </div>
                    </div>
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    <?php include_once('includes/footer2.php');?>
    <!-- Customer View Modal -->
    <div class="modal fade" id="customerViewModal" tabindex="-1" role="dialog" aria-labelledby="customerViewModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="customerViewModalLabel">Customer Details</h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body" id="customerDetailsContent">
                    <!-- Customer details will be loaded here -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-primary" onclick="printCustomerDetails()">
                        <i class="fas fa-print"></i> Print
                    </button>
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
    
    <script>
        // Toggle ID Proof Number field visibility and autofill
        function toggleIdProofNumber() {
            var type = document.querySelector('select[name=\'id_proof_type\']').value;
            var group = document.getElementById('idProofNumberGroup');
            var input = document.querySelector('input[name=\'id_proof_number\']');
            if (type === 'Aadhaar Card') {
                group.style.display = 'none';
                input.value = document.querySelector('input[name=\'aadhar_number\']').value;
            } else if (type === 'PAN Card') {
                group.style.display = 'none';
                input.value = document.querySelector('input[name=\'pan_number\']').value;
            } else if (type) {
                group.style.display = 'block';
            } else {
                group.style.display = 'none';
                input.value = '';
            }
        }
        document.addEventListener('DOMContentLoaded', function() {
            var idTypeSelect = document.querySelector('select[name=\'id_proof_type\']');
            if (idTypeSelect) {
                idTypeSelect.addEventListener('change', toggleIdProofNumber);
                toggleIdProofNumber();
            }
            // Keep IDProofNumber synced when PAN/Aadhaar changes
            var pan = document.querySelector('input[name=\'pan_number\']');
            var aadhaar = document.querySelector('input[name=\'aadhar_number\']');
            var idnum = document.querySelector('input[name=\'id_proof_number\']');
            if (pan) pan.addEventListener('input', function(){ if (idTypeSelect.value==='PAN Card') idnum.value = pan.value; });
            if (aadhaar) aadhaar.addEventListener('input', function(){ if (idTypeSelect.value==='Aadhaar Card') idnum.value = aadhaar.value; });
        });
        function viewCustomer(customerId) {
            // Show loading spinner
            $('#customerDetailsContent').html('<div class="text-center"><i class="fas fa-spinner fa-spin"></i> Loading...</div>');
            $('#customerViewModal').modal('show');
            
            // Fetch customer details via AJAX
            $.ajax({
                url: 'ajax_get_customer_details.php',
                method: 'GET',
                data: { id: customerId },
                success: function(response) {
                    $('#customerDetailsContent').html(response);
                },
                error: function() {
                    $('#customerDetailsContent').html('<div class="alert alert-danger">Error loading customer details.</div>');
                }
            });
        }
        
        function printCustomerDetails() {
            var printWindow = window.open('', '_blank');
            var customerContent = document.getElementById('customerDetailsContent').innerHTML;
            
            printWindow.document.write(`
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Customer Details</title>
                    <style>
                        @page {
                            size: A4;
                            margin: 20mm;
                        }
                        body {
                            font-family: Arial, sans-serif;
                            font-size: 12px;
                            line-height: 1.4;
                            margin: 0;
                            padding: 20px;
                        }
                        .print-header {
                            text-align: center;
                            margin-bottom: 30px;
                            border-bottom: 2px solid #333;
                            padding-bottom: 15px;
                        }
                        .customer-photo {
                            width: 120px;
                            height: 120px;
                            object-fit: cover;
                            border: 2px solid #ddd;
                            border-radius: 5px;
                        }
                        .details-table {
                            width: 100%;
                            border-collapse: collapse;
                            margin-top: 20px;
                        }
                        .details-table th,
                        .details-table td {
                            border: 1px solid #ddd;
                            padding: 8px;
                            text-align: left;
                        }
                        .details-table th {
                            background-color: #f8f9fa;
                            font-weight: bold;
                        }
                        .photo-section {
                            text-align: center;
                            margin: 20px 0;
                        }
                        @media print {
                            body { margin: 0; }
                            .no-print { display: none !important; }
                        }
                    </style>
                </head>
                <body>
                    <div class="print-header">
                        <h2>Customer Information</h2>
                        <p>Generated on: ${new Date().toLocaleDateString()}</p>
                    </div>
                    ${customerContent}
                </body>
                </html>
            `);
            
            printWindow.document.close();
            printWindow.focus();
            
            setTimeout(function() {
                printWindow.print();
                printWindow.close();
            }, 250);
        }
        
        // Image Preview Functions
        function previewImage(input, previewId) {
            const file = input.files[0];
            const previewContainer = document.getElementById(previewId + 'Container');
            const preview = document.getElementById(previewId);
            
            if (file) {
                // Validate file size (5MB)
                if (file.size > 5 * 1024 * 1024) {
                    alert('File size must be less than 5MB');
                    input.value = '';
                    previewContainer.style.display = 'none';
                    return;
                }
                
                // Validate file type
                const validTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
                if (!validTypes.includes(file.type)) {
                    alert('Please select a valid image file (JPG, PNG, GIF)');
                    input.value = '';
                    previewContainer.style.display = 'none';
                    return;
                }
                
                const reader = new FileReader();
                reader.onload = function(e) {
                    preview.src = e.target.result;
                    previewContainer.style.display = 'block';
                };
                reader.readAsDataURL(file);
            } else {
                previewContainer.style.display = 'none';
            }
        }
        
        function previewDocument(input, previewId) {
            const file = input.files[0];
            const previewContainer = document.getElementById(previewId + 'Container');
            const preview = document.getElementById(previewId);
            
            if (file) {
                // Validate file size (5MB)
                if (file.size > 5 * 1024 * 1024) {
                    alert('File size must be less than 5MB');
                    input.value = '';
                    previewContainer.style.display = 'none';
                    return;
                }
                
                // Validate file type
                const validTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'application/pdf'];
                if (!validTypes.includes(file.type)) {
                    alert('Please select a valid document file (JPG, PNG, PDF)');
                    input.value = '';
                    previewContainer.style.display = 'none';
                    return;
                }
                
                if (file.type === 'application/pdf') {
                    preview.innerHTML = `
                        <div class="border p-3 text-center" style="max-width: 150px;">
                            <i class="fas fa-file-pdf fa-3x text-danger"></i>
                            <br><small class="text-muted">PDF Document</small>
                            <br><small class="font-weight-bold">${file.name}</small>
                        </div>
                    `;
                } else {
                    const reader = new FileReader();
                    reader.onload = function(e) {
                        preview.innerHTML = `<img src="${e.target.result}" class="img-thumbnail" style="max-width: 150px; max-height: 150px; object-fit: cover;" alt="Document Preview">`;
                    };
                    reader.readAsDataURL(file);
                }
                previewContainer.style.display = 'block';
            } else {
                previewContainer.style.display = 'none';
            }
        }
        
        function clearImagePreview(inputId, previewId) {
            document.getElementById(inputId).value = '';
            document.getElementById(previewId + 'Container').style.display = 'none';
        }
        
        function clearDocumentPreview(inputId, previewId) {
            document.getElementById(inputId).value = '';
            document.getElementById(previewId + 'Container').style.display = 'none';
        }
        
        // Enhanced file validation on form submit
        document.addEventListener('DOMContentLoaded', function() {
            const form = document.querySelector('form[enctype="multipart/form-data"]');
            if (form) {
                form.addEventListener('submit', function(e) {
                    const photoInput = document.getElementById('photoInput');
                    const idProofInput = document.getElementById('idProofInput');
                    
                    // Validate photo if selected
                    if (photoInput && photoInput.files[0]) {
                        const photoFile = photoInput.files[0];
                        if (photoFile.size > 5 * 1024 * 1024) {
                            alert('Customer photo file size must be less than 5MB');
                            e.preventDefault();
                            return false;
                        }
                    }
                    
                    // Validate ID proof if selected
                    if (idProofInput && idProofInput.files[0]) {
                        const idProofFile = idProofInput.files[0];
                        if (idProofFile.size > 5 * 1024 * 1024) {
                            alert('ID Proof document file size must be less than 5MB');
                            e.preventDefault();
                            return false;
                        }
                    }
                    
                    return true;
                });
            }
        });
    </script>
</body>
</html>
