<?php
session_start();
$configPath = __DIR__ . '/includes/config.php';
if (!file_exists($configPath)) die("Error: Database configuration file not found.");
include_once($configPath);
if (!isset($_SESSION['uid'])) { header('location:logout.php'); exit(); }
$con = $con ?? null;
if (!$con) die("Error: Database connection failed.");

// Enhanced function to calculate interest and due amount - Daily Interest Calculation with proper rounding
function calculateDueAmount($loanDate, $dueDate, $loanAmount, $interestRate, $collectionDate = null) {
    if (!$collectionDate) {
        $collectionDate = date('Y-m-d');
    }
    
    $loanDateTime = new DateTime($loanDate);
    $dueDatetime = new DateTime($dueDate);
    $collectionDateTime = new DateTime($collectionDate);
    
    $totalDays = $loanDateTime->diff($collectionDateTime)->days;
    $daysOverdue = max(0, $collectionDateTime->diff($dueDatetime)->days);
    if ($collectionDateTime < $dueDatetime) {
        $daysOverdue = 0;
    }
    
    // Calculate daily interest rate from annual rate (fixed 15% per annum)
    $annualRate = 0.15; // 15% p.a.
    $dailyRate = $annualRate / 365;
    
    // Per-day interest amount and total accrued interest with proper rounding
    $dailyInterestAmount = round($loanAmount * $dailyRate, 2); // Round daily interest
    $totalInterest = round($dailyInterestAmount * $totalDays, 2); // Round total interest
    $monthlyInterestAmount = round($loanAmount * ($annualRate / 12), 2); // Round monthly interest
    
    $totalDue = round($loanAmount + $totalInterest, 2); // Round total due
    
    return [
        'totalDays' => $totalDays,
        'daysOverdue' => $daysOverdue,
        'dailyInterestAmount' => $dailyInterestAmount, // Rounded daily interest
        'monthlyInterestAmount' => $monthlyInterestAmount, // Rounded monthly interest
        'interestAmount' => $totalInterest, // Rounded total interest
        'totalDue' => $totalDue, // Rounded total due
        'dailyRate' => $dailyRate
    ];
}

// Filters
$search_customer = isset($_GET['search_customer']) ? mysqli_real_escape_string($con, $_GET['search_customer']) : '';
$search_mobile = isset($_GET['search_mobile']) ? mysqli_real_escape_string($con, $_GET['search_mobile']) : '';
$search_pawn_id = isset($_GET['search_pawn_id']) ? mysqli_real_escape_string($con, $_GET['search_pawn_id']) : '';
$item_type_filter = isset($_GET['item_type']) ? mysqli_real_escape_string($con, $_GET['item_type']) : '';
$status_filter = isset($_GET['status']) ? mysqli_real_escape_string($con, $_GET['status']) : '';
$date_from = isset($_GET['date_from']) ? mysqli_real_escape_string($con, $_GET['date_from']) : '';
$date_to = isset($_GET['date_to']) ? mysqli_real_escape_string($con, $_GET['date_to']) : '';
$min_amount = isset($_GET['min_amount']) ? mysqli_real_escape_string($con, $_GET['min_amount']) : '';
$max_amount = isset($_GET['max_amount']) ? mysqli_real_escape_string($con, $_GET['max_amount']) : '';

// Build WHERE clause
$where = "1=1";
if ($search_customer) $where .= " AND c.Name LIKE '%$search_customer%'";
if ($search_mobile) $where .= " AND c.MobileNumber LIKE '%$search_mobile%'";
if ($search_pawn_id) $where .= " AND p.PawnCode LIKE '%$search_pawn_id%'";
if ($item_type_filter) $where .= " AND p.ItemType = '$item_type_filter'";
if ($status_filter) $where .= " AND p.Status = '$status_filter'";
if ($date_from) $where .= " AND p.LoanDate >= '$date_from'";
if ($date_to) $where .= " AND p.LoanDate <= '$date_to'";
if ($min_amount) $where .= " AND p.LoanAmount >= '$min_amount'";
if ($max_amount) $where .= " AND p.LoanAmount <= '$max_amount'";

// Main query to get all pawn transactions with customer details
$query = "
SELECT 
    p.ID as PawnID,
    p.PawnCode,
    p.LoanDate,
    p.DueDate,
    p.Status,
    p.ItemType,
    p.ItemDescription,
    p.ItemWeight,
    p.ItemPurity,
    p.EstimatedValue,
    p.LoanAmount,
    p.InterestRate,
    p.CurrentLoanAmount,
    p.TotalLoanIncreases,
    p.ItemPhotoPath,
    p.Notes as PawnNotes,
    c.ID as CustomerID,
    c.CustomerCode,
    c.Name as CustomerName,
    c.FatherHusbandName,
    c.MobileNumber,
    c.Address,
    c.IDProofType,
    c.IDProofNumber,
    c.PhotoPath as CustomerPhotoPath,
    (SELECT ItemPhotoPath FROM tblpawnitems WHERE PawnTransactionID = p.ID ORDER BY ItemSequence LIMIT 1) as FirstItemPhoto,
    (SELECT COUNT(*) FROM tblpawnitems WHERE PawnTransactionID = p.ID) as ItemsCount
FROM tblpawntransactions p
INNER JOIN tblcustomers c ON p.CustomerID = c.ID
WHERE $where
ORDER BY p.LoanDate DESC, p.ID DESC
";

$result = mysqli_query($con, $query);
$transactions = [];
if ($result && mysqli_num_rows($result) > 0) {
    while($row = mysqli_fetch_assoc($result)) {
        $pawnID = $row['PawnID'];
        
        // Get payment collections for this pawn
        $paymentQuery = "SELECT * FROM tblpaymentcollections WHERE PawnTransactionID = $pawnID ORDER BY CollectionDate DESC";
        $paymentResult = mysqli_query($con, $paymentQuery);
        $payments = [];
        $totalPaid = 0;
        while($payment = mysqli_fetch_assoc($paymentResult)) {
            $payments[] = $payment;
            $totalPaid += $payment['AmountPaid'];
        }
        
        // Get redemption details
        $redemptionQuery = "SELECT * FROM tblredemptions WHERE PawnTransactionID = $pawnID";
        $redemptionResult = mysqli_query($con, $redemptionQuery);
        $redemption = mysqli_fetch_assoc($redemptionResult);
        
        // Get loan increases
        $increaseQuery = "SELECT * FROM tblloanincreases WHERE PawnTransactionID = $pawnID ORDER BY IncreaseDate DESC";
        $increaseResult = mysqli_query($con, $increaseQuery);
        $increases = [];
        while($increase = mysqli_fetch_assoc($increaseResult)) {
            $increases[] = $increase;
        }
        
        // Get bank pledge details
        $bankPledgeQuery = "SELECT bp.*, ba.BankName, ba.AccountHolderName, ba.AccountNumber 
                           FROM tblbankpledges bp 
                           LEFT JOIN tblbankaccounts ba ON bp.BankAccountID = ba.ID 
                           WHERE bp.PawnTransactionID = $pawnID";
        $bankPledgeResult = mysqli_query($con, $bankPledgeQuery);
        $bankPledge = mysqli_fetch_assoc($bankPledgeResult);
        
        $bankPayments = [];
        $bankRedemptions = [];
        if ($bankPledge) {
            // Get bank pledge payments
            $bankPaymentQuery = "SELECT * FROM tblbankpledgepayments WHERE BankPledgeID = {$bankPledge['ID']} ORDER BY PaymentDate DESC";
            $bankPaymentResult = mysqli_query($con, $bankPaymentQuery);
            while($payment = mysqli_fetch_assoc($bankPaymentResult)) {
                $bankPayments[] = $payment;
            }
            
            // Get bank pledge redemptions
            $bankRedemptionQuery = "SELECT * FROM tblbankpledgeredemptions WHERE BankPledgeID = {$bankPledge['ID']} ORDER BY RedemptionDate DESC";
            $bankRedemptionResult = mysqli_query($con, $bankRedemptionQuery);
            while($redemp = mysqli_fetch_assoc($bankRedemptionResult)) {
                $bankRedemptions[] = $redemp;
            }
        }
        
        // Calculate detailed interest information
        $currentLoanAmount = $row['CurrentLoanAmount'] ?? $row['LoanAmount'];
        $interestCalc = calculateDueAmount($row['LoanDate'], $row['DueDate'], $currentLoanAmount, $row['InterestRate']);
        
        // Calculate balance
        $balanceAmount = $interestCalc['totalDue'] - $totalPaid;
        if ($redemption) {
            $balanceAmount = 0; // Fully redeemed
        }
        
        // Add interest calculation details
        $row['TotalDays'] = $interestCalc['totalDays'];
        $row['DaysOverdue'] = $interestCalc['daysOverdue'];
        $row['DailyInterestAmount'] = $interestCalc['dailyInterestAmount'];
        $row['MonthlyInterestAmount'] = $interestCalc['monthlyInterestAmount'];
        $row['ActualInterestAmount'] = $interestCalc['interestAmount'];
        $row['TotalDueAmount'] = $interestCalc['totalDue'];
        
        $row['Payments'] = $payments;
        $row['TotalPaid'] = $totalPaid;
        $row['BalanceAmount'] = $balanceAmount;
        $row['Redemption'] = $redemption;
        $row['Increases'] = $increases;
        $row['BankPledge'] = $bankPledge;
        $row['BankPayments'] = $bankPayments;
        $row['BankRedemptions'] = $bankRedemptions;
        
        $transactions[] = $row;
    }
}

// Get filter options
$itemTypes = ['Gold', 'Silver', 'Diamond', 'Platinum', 'Other'];
$statuses = ['Active', 'Redeemed', 'Auctioned', 'Extended'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Customer Jewel History Report</title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/compact.css" rel="stylesheet">
    <style>
        .filter-container {
            background: #f8f9fc;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .jewel-card {
            border: 1px solid #e3e6f0;
            border-radius: 15px;
            margin-bottom: 30px;
            overflow: hidden;
            transition: all 0.3s ease;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .jewel-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 20px rgba(0,0,0,0.15);
        }
        .card-header-custom {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 15px 20px;
            border-bottom: none;
        }
        .status-badge {
            font-size: 0.9em;
            padding: 5px 10px;
        }
        .status-active { background: #1cc88a; }
        .status-redeemed { background: #36b9cc; }
        .status-extended { background: #f6c23e; }
        .status-auctioned { background: #e74a3b; }
        .info-section {
            border-bottom: 1px solid #e3e6f0;
            padding: 15px 20px;
        }
        .info-section:last-child { border-bottom: none; }
        .info-title {
            font-weight: bold;
            color: #5a5c69;
            margin-bottom: 10px;
            font-size: 1.1em;
            border-bottom: 2px solid #4e73df;
            padding-bottom: 5px;
        }
        .info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 10px;
        }
        .info-item {
            display: flex;
            justify-content: space-between;
            padding: 5px 0;
        }
        .info-label {
            font-weight: 600;
            color: #6c757d;
        }
        .info-value {
            color: #495057;
            font-weight: 500;
        }
        .amount-positive { color: #1cc88a; font-weight: bold; }
        .amount-negative { color: #e74a3b; font-weight: bold; }
        .amount-neutral { color: #858796; font-weight: bold; }
        .payment-history {
            max-height: 200px;
            overflow-y: auto;
        }
        .payment-item {
            border: 1px solid #e3e6f0;
            border-radius: 5px;
            padding: 8px;
            margin-bottom: 8px;
            background: #f8f9fc;
        }
        .customer-photo {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            object-fit: cover;
            border: 3px solid #fff;
            box-shadow: 0 2px 5px rgba(0,0,0,0.2);
            transition: transform 0.2s ease, box-shadow 0.2s ease;
        }
        .customer-photo:hover {
            transform: scale(1.1);
            box-shadow: 0 4px 10px rgba(0,0,0,0.3);
        }
        .jewel-photo {
            width: 80px;
            height: 80px;
            border-radius: 10px;
            object-fit: cover;
            border: 2px solid #e3e6f0;
            transition: transform 0.2s ease, box-shadow 0.2s ease;
        }
        .jewel-photo:hover {
            transform: scale(1.1);
            box-shadow: 0 4px 10px rgba(0,0,0,0.3);
        }
        .no-data {
            text-align: center;
            color: #858796;
            font-style: italic;
            padding: 20px;
        }
        @media print {
            .no-print { display: none !important; }
            .jewel-card { page-break-inside: avoid; }
        }
        .bank-pledge-section {
            background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            color: white;
            border-radius: 10px;
            padding: 15px;
            margin-top: 10px;
        }
        .bank-info {
            background: rgba(255,255,255,0.1);
            border-radius: 5px;
            padding: 10px;
            margin-top: 10px;
        }
    </style>
</head>
<body id="page-top">
<div id="wrapper">
    <?php include_once('includes/sidebar.php'); ?>
    <div id="content-wrapper" class="d-flex flex-column">
        <div id="content">
            <?php include_once('includes/topbar.php'); ?>
            <div class="container-fluid mt-4">
                <div class="d-sm-flex align-items-center justify-content-between mb-4">
                    <h3 class="mb-0">Customer Jewel History Report</h3>
                    <div class="no-print">
                        <button onclick="window.print()" class="btn btn-primary btn-sm">
                            <i class="fas fa-print"></i> Print
                        </button>
                    </div>
                </div>

                <!-- Success/Error Messages -->
                <?php if (isset($_SESSION['msg'])) { ?>
                <div class="alert alert-<?php echo $_SESSION['msg_type'] ?? 'info'; ?> alert-dismissible fade show no-print" role="alert">
                    <i class="fas fa-<?php echo $_SESSION['msg_type'] == 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                    <?php echo htmlspecialchars($_SESSION['msg']); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php 
                    unset($_SESSION['msg'], $_SESSION['msg_type']); 
                } ?>

                <!-- Filter Section -->
                <div class="filter-container no-print">
                    <form class="row" method="get">
                        <div class="col-md-3 mb-2">
                            <label>Customer Name</label>
                            <input type="text" name="search_customer" class="form-control" value="<?php echo htmlspecialchars($search_customer); ?>" placeholder="Search by name">
                        </div>
                        <div class="col-md-3 mb-2">
                            <label>Mobile Number</label>
                            <input type="text" name="search_mobile" class="form-control" value="<?php echo htmlspecialchars($search_mobile); ?>" placeholder="Search by mobile">
                        </div>
                        <div class="col-md-3 mb-2">
                            <label>Pawn ID</label>
                            <input type="text" name="search_pawn_id" class="form-control" value="<?php echo htmlspecialchars($search_pawn_id); ?>" placeholder="Search by pawn code">
                        </div>
                        <div class="col-md-3 mb-2">
                            <label>Item Type</label>
                            <select name="item_type" class="form-control">
                                <option value="">All Types</option>
                                <?php foreach($itemTypes as $type) { ?>
                                    <option value="<?php echo $type; ?>" <?php if($item_type_filter==$type) echo 'selected'; ?>><?php echo $type; ?></option>
                                <?php } ?>
                            </select>
                        </div>
                        <div class="col-md-2 mb-2">
                            <label>Status</label>
                            <select name="status" class="form-control">
                                <option value="">All Status</option>
                                <?php foreach($statuses as $status) { ?>
                                    <option value="<?php echo $status; ?>" <?php if($status_filter==$status) echo 'selected'; ?>><?php echo $status; ?></option>
                                <?php } ?>
                            </select>
                        </div>
                        <div class="col-md-2 mb-2">
                            <label>From Date</label>
                            <input type="date" name="date_from" class="form-control" value="<?php echo $date_from; ?>">
                        </div>
                        <div class="col-md-2 mb-2">
                            <label>To Date</label>
                            <input type="date" name="date_to" class="form-control" value="<?php echo $date_to; ?>">
                        </div>
                        <div class="col-md-2 mb-2">
                            <label>Min Amount</label>
                            <input type="number" name="min_amount" class="form-control" value="<?php echo $min_amount; ?>" placeholder="Min loan">
                        </div>
                        <div class="col-md-2 mb-2">
                            <label>Max Amount</label>
                            <input type="number" name="max_amount" class="form-control" value="<?php echo $max_amount; ?>" placeholder="Max loan">
                        </div>
                        <div class="col-md-2 mb-2 align-self-end">
                            <button type="submit" class="btn btn-primary btn-block"><i class="fas fa-search"></i> Search</button>
                        </div>
                        <div class="col-md-1 mb-2 align-self-end">
                            <a href="customer-jewel-history-report.php" class="btn btn-secondary btn-block"><i class="fas fa-refresh"></i> Clear</a>
                        </div>
                    </form>
                </div>

                <!-- Results Summary -->
                <div class="alert alert-info no-print">
                    <i class="fas fa-info-circle"></i> Found <strong><?php echo count($transactions); ?></strong> records
                </div>

                <!-- Cards Grid -->
                <div class="row">
                    <?php if (count($transactions) > 0) {
                        foreach($transactions as $txn) {
                            $statusClass = 'status-' . strtolower($txn['Status']);
                            $balanceClass = $txn['BalanceAmount'] > 0 ? 'amount-negative' : ($txn['BalanceAmount'] < 0 ? 'amount-positive' : 'amount-neutral');
                            ?>
                            <div class="col-lg-6 col-xl-4 mb-4">
                                <div class="jewel-card">
                                    <!-- Card Header -->
                                    <div class="card-header-custom">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <h6 class="mb-1"><?php echo htmlspecialchars($txn['PawnCode']); ?></h6>
                                                <small><?php echo htmlspecialchars($txn['CustomerName']); ?></small>
                                            </div>
                                            <div class="d-flex align-items-center">
                                                <span class="badge status-badge <?php echo $statusClass; ?> me-2"><?php echo $txn['Status']; ?></span>
                                                <div class="btn-group no-print">
                                                    <button onclick="printJewelHistory(<?php echo $txn['PawnID']; ?>)" class="btn btn-light btn-sm" title="Print History">
                                                        <i class="fas fa-print"></i>
                                                    </button>
                                                    <a href="delete-pawn.php?id=<?php echo $txn['PawnID']; ?>" class="btn btn-danger btn-sm" title="Delete Pawn Transaction" onclick="return confirm('Are you sure you want to delete this pawn transaction? Customer data will be preserved.')">
                                                        <i class="fas fa-trash"></i>
                                                    </a>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Customer Details -->
                                    <div class="info-section">
                                        <div class="info-title"><i class="fas fa-user"></i> Customer Details</div>
                                        <div class="d-flex align-items-center mb-2">
                                            <?php if ($txn['CustomerPhotoPath']) { 
                                                $customerPhotoPath = ltrim($txn['CustomerPhotoPath'], '/');
                                                if (file_exists(__DIR__ . '/' . $customerPhotoPath)) {
                                            ?>
                                                <a href="<?php echo htmlspecialchars($customerPhotoPath); ?>" target="_blank" title="Click to view full size">
                                                    <img src="<?php echo htmlspecialchars($customerPhotoPath); ?>" class="customer-photo me-3" alt="Customer" style="cursor: pointer;">
                                                </a>
                                            <?php } else { ?>
                                                <div class="customer-photo d-flex align-items-center justify-content-center bg-warning text-white me-3" title="Photo not found: <?php echo htmlspecialchars($customerPhotoPath); ?>">
                                                    <i class="fas fa-exclamation-triangle"></i>
                                                </div>
                                            <?php } } else { ?>
                                                <div class="customer-photo d-flex align-items-center justify-content-center bg-secondary text-white me-3">
                                                    <i class="fas fa-user"></i>
                                                </div>
                                            <?php } ?>
                                            <div>
                                                <strong><?php echo htmlspecialchars($txn['CustomerName']); ?></strong><br>
                                                <small class="text-muted"><?php echo htmlspecialchars($txn['CustomerCode']); ?></small>
                                            </div>
                                        </div>
                                        <div class="info-grid">
                                            <div class="info-item">
                                                <span class="info-label">Father/Husband:</span>
                                                <span class="info-value"><?php echo htmlspecialchars($txn['FatherHusbandName']); ?></span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Mobile:</span>
                                                <span class="info-value"><?php echo htmlspecialchars($txn['MobileNumber']); ?></span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">ID Proof:</span>
                                                <span class="info-value"><?php echo htmlspecialchars($txn['IDProofType'] . ' - ' . $txn['IDProofNumber']); ?></span>
                                            </div>
                                        </div>
                                        <div class="mt-2">
                                            <span class="info-label">Address:</span>
                                            <span class="info-value"><?php echo htmlspecialchars($txn['Address']); ?></span>
                                        </div>
                                    </div>

                                    <!-- Jewel Details -->
                                    <div class="info-section">
                                        <div class="info-title"><i class="fas fa-gem"></i> Jewel Details <?php if($txn['ItemsCount'] > 1) echo "({$txn['ItemsCount']} items)"; ?></div>
                                        <div class="d-flex align-items-center mb-2">
                                            <?php 
                                            // First try to get photo from items table, then from main transaction
                                            $displayPhotoPath = $txn['FirstItemPhoto'] ?: $txn['ItemPhotoPath'];
                                            if ($displayPhotoPath) { 
                                                $itemPhotoPath = ltrim($displayPhotoPath, '/');
                                                if (file_exists(__DIR__ . '/' . $itemPhotoPath)) {
                                            ?>
                                                <a href="<?php echo htmlspecialchars($itemPhotoPath); ?>" target="_blank" title="Click to view full size">
                                                    <img src="<?php echo htmlspecialchars($itemPhotoPath); ?>" class="jewel-photo me-3" alt="Jewel" style="cursor: pointer;">
                                                </a>
                                            <?php } else { ?>
                                                <div class="jewel-photo d-flex align-items-center justify-content-center bg-warning text-white me-3" title="Photo not found: <?php echo htmlspecialchars($itemPhotoPath); ?>">
                                                    <i class="fas fa-exclamation-triangle"></i>
                                                </div>
                                            <?php } } else { ?>
                                                <div class="jewel-photo d-flex align-items-center justify-content-center bg-secondary text-white me-3">
                                                    <i class="fas fa-gem"></i>
                                                </div>
                                            <?php } ?>
                                            <div>
                                                <strong><?php echo htmlspecialchars($txn['ItemType']); ?></strong><br>
                                                <small class="text-muted"><?php echo htmlspecialchars($txn['ItemDescription']); ?></small>
                                            </div>
                                        </div>
                                        <div class="info-grid">
                                            <div class="info-item">
                                                <span class="info-label">Weight:</span>
                                                <span class="info-value"><?php echo $txn['ItemWeight']; ?>g</span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Purity:</span>
                                                <span class="info-value"><?php echo htmlspecialchars($txn['ItemPurity']); ?></span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Estimated Value:</span>
                                                <span class="info-value amount-positive">₹<?php echo number_format($txn['EstimatedValue'], 2); ?></span>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Loan Details -->
                                    <div class="info-section">
                                        <div class="info-title"><i class="fas fa-money-bill-wave"></i> Loan Details</div>
                                        <div class="info-grid">
                                            <div class="info-item">
                                                <span class="info-label">Loan Date:</span>
                                                <span class="info-value"><?php echo date('d-m-Y', strtotime($txn['LoanDate'])); ?></span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Due Date:</span>
                                                <span class="info-value"><?php echo date('d-m-Y', strtotime($txn['DueDate'])); ?></span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Original Amount:</span>
                                                <span class="info-value amount-neutral">₹<?php echo number_format($txn['LoanAmount'], 2); ?></span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Interest Rate:</span>
                                                <span class="info-value">15% per year (<?php echo number_format((0.15/365)*100, 4); ?>% per day)</span>
                                            </div>
                                            <?php if ($txn['TotalLoanIncreases'] > 0) { ?>
                                            <div class="info-item">
                                                <span class="info-label">Loan Increases:</span>
                                                <span class="info-value amount-positive">₹<?php echo number_format($txn['TotalLoanIncreases'], 2); ?></span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Current Amount:</span>
                                                <span class="info-value amount-positive">₹<?php echo number_format($txn['CurrentLoanAmount'], 2); ?></span>
                                            </div>
                                            <?php } ?>
                                        </div>
                                    </div>

                                    <!-- Interest Calculation Details -->
                                    <div class="info-section">
                                        <div class="info-title"><i class="fas fa-calculator"></i> Interest Calculation</div>
                                        <div class="info-grid">
                                            <div class="info-item">
                                                <span class="info-label">Total Days:</span>
                                                <span class="info-value"><?php echo $txn['TotalDays']; ?> days</span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Days Overdue:</span>
                                                <span class="info-value <?php echo $txn['DaysOverdue'] > 0 ? 'amount-negative' : 'amount-positive'; ?>">
                                                    <?php echo $txn['DaysOverdue']; ?> days
                                                </span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Daily Interest:</span>
                                                <span class="info-value">₹<?php echo number_format($txn['DailyInterestAmount'], 2); ?></span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Monthly Interest:</span>
                                                <span class="info-value">₹<?php echo number_format($txn['MonthlyInterestAmount'], 2); ?></span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Actual Interest:</span>
                                                <span class="info-value amount-neutral">₹<?php echo number_format($txn['ActualInterestAmount'], 2); ?></span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Total Due Amount:</span>
                                                <span class="info-value amount-negative">₹<?php echo number_format($txn['TotalDueAmount'], 2); ?></span>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Payment History -->
                                    <div class="info-section">
                                        <div class="info-title"><i class="fas fa-receipt"></i> Payment History</div>
                                        <?php if (count($txn['Payments']) > 0) { ?>
                                            <div class="payment-history">
                                                <?php foreach($txn['Payments'] as $payment) { ?>
                                                    <div class="payment-item">
                                                        <div class="d-flex justify-content-between">
                                                            <span><strong><?php echo date('d-m-Y', strtotime($payment['CollectionDate'])); ?></strong></span>
                                                            <span class="amount-positive">₹<?php echo number_format($payment['AmountPaid'], 2); ?></span>
                                                        </div>
                                                        <div class="d-flex justify-content-between">
                                                            <small class="text-muted">Interest: ₹<?php echo number_format($payment['InterestAmount'], 2); ?></small>
                                                            <small class="text-muted">Days: <?php echo $payment['InterestDays']; ?></small>
                                                        </div>
                                                        <?php if ($payment['DiscountAmount'] > 0) { ?>
                                                        <div class="d-flex justify-content-between">
                                                            <small class="text-info">Discount: ₹<?php echo number_format($payment['DiscountAmount'], 2); ?></small>
                                                            <small class="text-muted">Balance: ₹<?php echo number_format($payment['BalanceAmount'], 2); ?></small>
                                                        </div>
                                                        <?php } ?>
                                                        <small class="text-muted"><?php echo $payment['PaymentMethod']; ?> | <?php echo $payment['CollectionCode']; ?></small>
                                                    </div>
                                                <?php } ?>
                                            </div>
                                            <div class="mt-2">
                                                <div class="info-item">
                                                    <span class="info-label">Total Paid:</span>
                                                    <span class="info-value amount-positive">₹<?php echo number_format($txn['TotalPaid'], 2); ?></span>
                                                </div>
                                                <div class="info-item">
                                                    <span class="info-label">Balance Amount:</span>
                                                    <span class="info-value <?php echo $balanceClass; ?>">₹<?php echo number_format($txn['BalanceAmount'], 2); ?></span>
                                                </div>
                                            </div>
                                        <?php } else { ?>
                                            <div class="no-data">No payments made yet</div>
                                            <div class="mt-2">
                                                <div class="info-item">
                                                    <span class="info-label">Total Due:</span>
                                                    <span class="info-value amount-negative">₹<?php echo number_format($txn['TotalDueAmount'], 2); ?></span>
                                                </div>
                                                <div class="info-item">
                                                    <span class="info-label">Outstanding:</span>
                                                    <span class="info-value amount-negative">₹<?php echo number_format($txn['BalanceAmount'], 2); ?></span>
                                                </div>
                                            </div>
                                        <?php } ?>
                                    </div>

                                    <!-- Bank Pledge Details -->
                                    <?php if ($txn['BankPledge']) { 
                                        $bp = $txn['BankPledge'];
                                        $bankTotalPaid = array_sum(array_column($txn['BankPayments'], 'PaymentAmount'));
                                        $bankBalance = $bp['PledgeAmount'] - $bankTotalPaid;
                                        ?>
                                        <div class="info-section">
                                            <div class="bank-pledge-section">
                                                <div class="info-title text-white"><i class="fas fa-university"></i> Bank Pledge Details</div>
                                                <div class="bank-info">
                                                    <div class="info-grid">
                                                        <div class="info-item">
                                                            <span class="info-label text-white">Bank:</span>
                                                            <span class="info-value text-white"><?php echo htmlspecialchars($bp['BankName']); ?></span>
                                                        </div>
                                                        <div class="info-item">
                                                            <span class="info-label text-white">Account:</span>
                                                            <span class="info-value text-white"><?php echo htmlspecialchars($bp['AccountNumber']); ?></span>
                                                        </div>
                                                        <div class="info-item">
                                                            <span class="info-label text-white">Pledge Amount:</span>
                                                            <span class="info-value text-white">₹<?php echo number_format($bp['PledgeAmount'], 2); ?></span>
                                                        </div>
                                                        <div class="info-item">
                                                            <span class="info-label text-white">Interest Rate:</span>
                                                            <span class="info-value text-white"><?php echo $bp['InterestRate']; ?>%</span>
                                                        </div>
                                                        <div class="info-item">
                                                            <span class="info-label text-white">Pledge Date:</span>
                                                            <span class="info-value text-white"><?php echo date('d-m-Y', strtotime($bp['PledgeDate'])); ?></span>
                                                        </div>
                                                        <div class="info-item">
                                                            <span class="info-label text-white">Maturity Date:</span>
                                                            <span class="info-value text-white"><?php echo date('d-m-Y', strtotime($bp['MaturityDate'])); ?></span>
                                                        </div>
                                                        <div class="info-item">
                                                            <span class="info-label text-white">Total Paid:</span>
                                                            <span class="info-value text-white">₹<?php echo number_format($bankTotalPaid, 2); ?></span>
                                                        </div>
                                                        <div class="info-item">
                                                            <span class="info-label text-white">Balance:</span>
                                                            <span class="info-value text-white">₹<?php echo number_format($bankBalance, 2); ?></span>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    <?php } ?>

                                    <!-- Redemption Details -->
                                    <?php if ($txn['Redemption']) { 
                                        $redemption = $txn['Redemption'];
                                        ?>
                                        <div class="info-section">
                                            <div class="info-title"><i class="fas fa-check-circle"></i> Redemption Details</div>
                                            <div class="info-grid">
                                                <div class="info-item">
                                                    <span class="info-label">Redemption Date:</span>
                                                    <span class="info-value"><?php echo date('d-m-Y', strtotime($redemption['RedemptionDate'])); ?></span>
                                                </div>
                                                <div class="info-item">
                                                    <span class="info-label">Final Payment:</span>
                                                    <span class="info-value amount-positive">₹<?php echo number_format($redemption['FinalPaymentAmount'], 2); ?></span>
                                                </div>
                                                <div class="info-item">
                                                    <span class="info-label">Discount Given:</span>
                                                    <span class="info-value amount-neutral">₹<?php echo number_format($redemption['DiscountGiven'], 2); ?></span>
                                                </div>
                                                <div class="info-item">
                                                    <span class="info-label">Receiver:</span>
                                                    <span class="info-value"><?php echo htmlspecialchars($redemption['ReceiverName']); ?></span>
                                                </div>
                                            </div>
                                        </div>
                                    <?php } ?>
                                </div>
                            </div>
                            <?php
                        }
                    } else { ?>
                        <div class="col-12">
                            <div class="alert alert-warning text-center">
                                <i class="fas fa-exclamation-triangle"></i> No records found matching your search criteria.
                            </div>
                        </div>
                    <?php } ?>
                </div>
            </div>
            <?php include_once('includes/footer.php'); ?>
        </div>
    </div>
    <?php include_once('includes/footer2.php'); ?>
</div>

<script>
function printJewelHistory(pawnId) {
    window.open('jewel-history-receipt.php?pawn_id=' + pawnId, '_blank', 'width=800,height=600');
}
</script>
</body>
</html>
