<?php
session_start();

// Check if config file exists
$configPath = __DIR__ . '/includes/config.php';
if (!file_exists($configPath)) {
    die("Error: Database configuration file not found. Please create includes/config.php");
}

include_once($configPath);

if (!isset($con) || !$con) {
    die("Error: Database connection failed. Please check your database configuration.");
}
// Load branding helper
include_once(__DIR__ . '/includes/brand.php');

if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$msg = "";
$userId = $_SESSION['uid'];

// Create uploads directory
$uploadDir = 'uploads/bulk_csv/';
if (!file_exists($uploadDir)) {
    mkdir($uploadDir, 0777, true);
}

// Handle CSV Upload
if (isset($_POST['upload_csv'])) {
    if (isset($_FILES['csv_file']) && $_FILES['csv_file']['error'] == 0) {
        $fileName = $_FILES['csv_file']['name'];
        $fileTmp = $_FILES['csv_file']['tmp_name'];
        $fileExt = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
        
        if ($fileExt === 'csv') {
            $uploadCode = "UP" . date('Y') . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
            
            // Create upload log
            $logQuery = "INSERT INTO tblbulkuploadlogs (UploadCode, FileName, UploadType, UploadedBy) 
                        VALUES ('$uploadCode', '$fileName', 'CSV', $userId)";
            mysqli_query($con, $logQuery);
            $uploadLogId = mysqli_insert_id($con);
            
            try {
                $file = fopen($fileTmp, 'r');
                $successCount = 0;
                $errorCount = 0;
                $errors = [];
                $totalRecords = 0;
                
                // Skip header row
                $header = fgetcsv($file);
                
                while (($row = fgetcsv($file)) !== FALSE) {
                    $totalRecords++;
                    
                    // Skip empty rows
                    if (empty($row[0])) continue;
                    
                    // Generate pawn code
                    $pawnCode = "BULK" . date('Y') . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
                    
                    $customerName = mysqli_real_escape_string($con, trim($row[0]));
                    $mobileNumber = mysqli_real_escape_string($con, trim($row[1]));
                    $itemType = mysqli_real_escape_string($con, trim($row[2]));
                    $itemDescription = mysqli_real_escape_string($con, trim($row[3]));
                    $itemWeight = floatval($row[4]);
                    $itemPurity = mysqli_real_escape_string($con, trim($row[5]));
                    $estimatedValue = floatval($row[6]);
                    $loanAmount = floatval($row[7]);
                    $interestRate = floatval($row[8]);
                    $loanDate = date('Y-m-d', strtotime(trim($row[9])));
                    $dueDate = date('Y-m-d', strtotime(trim($row[10])));
                    $status = mysqli_real_escape_string($con, trim($row[11]));
                    $notes = mysqli_real_escape_string($con, isset($row[12]) ? trim($row[12]) : '');
                    
                    // Validate required fields
                    if (empty($customerName) || empty($mobileNumber) || empty($itemType) || 
                        empty($itemDescription) || $itemWeight <= 0 || $estimatedValue <= 0 || 
                        $loanAmount <= 0 || $interestRate <= 0) {
                        $errors[] = "Row " . ($totalRecords + 1) . ": Missing required fields";
                        $errorCount++;
                        continue;
                    }
                    
                    // Check if customer exists, if not create
                    $customerQuery = mysqli_query($con, "SELECT ID FROM tblcustomers WHERE MobileNumber='$mobileNumber'");
                    if (mysqli_num_rows($customerQuery) > 0) {
                        $customer = mysqli_fetch_assoc($customerQuery);
                        $customerID = $customer['ID'];
                    } else {
                        // Create new customer
                        $customerCode = "CUST" . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
                        $insertCustomer = "INSERT INTO tblcustomers (CustomerCode, Name, MobileNumber, FatherHusbandName, IDProofType, IDProofNumber) 
                                         VALUES ('$customerCode', '$customerName', '$mobileNumber', 'N/A', 'Aadhaar Card', 'BULK_UPLOAD')";
                        if (mysqli_query($con, $insertCustomer)) {
                            $customerID = mysqli_insert_id($con);
                        } else {
                            $errors[] = "Row " . ($totalRecords + 1) . ": Failed to create customer - " . mysqli_error($con);
                            $errorCount++;
                            continue;
                        }
                    }
                    
                    // Insert pawn transaction
                    $insertPawn = "INSERT INTO tblpawntransactions 
                                  (PawnCode, CustomerID, ItemType, ItemDescription, ItemWeight, ItemPurity, EstimatedValue, 
                                   LoanAmount, OriginalLoanAmount, CurrentLoanAmount, InterestRate, LoanDate, DueDate, Status, 
                                   Notes, CreatedBy) 
                                  VALUES ('$pawnCode', $customerID, '$itemType', '$itemDescription', $itemWeight, '$itemPurity', 
                                         $estimatedValue, $loanAmount, $loanAmount, $loanAmount, $interestRate, '$loanDate', 
                                         '$dueDate', '$status', '$notes', $userId)";
                    
                    if (mysqli_query($con, $insertPawn)) {
                        $successCount++;
                    } else {
                        $errors[] = "Row " . ($totalRecords + 1) . ": " . mysqli_error($con);
                        $errorCount++;
                    }
                }
                
                fclose($file);
                
                // Update upload log
                $errorLog = implode("\n", $errors);
                $status = ($errorCount == 0) ? 'Completed' : 'Completed';
                $updateLog = "UPDATE tblbulkuploadlogs SET TotalRecords=$totalRecords, SuccessfulRecords=$successCount, 
                             FailedRecords=$errorCount, ErrorLog='$errorLog', Status='$status' WHERE ID=$uploadLogId";
                mysqli_query($con, $updateLog);
                
                $msg = "Upload completed! Success: $successCount, Errors: $errorCount";
                if (!empty($errors)) {
                    $msg .= "<br>First 10 errors:<br>" . implode("<br>", array_slice($errors, 0, 10));
                    if (count($errors) > 10) {
                        $msg .= "<br>... and " . (count($errors) - 10) . " more errors.";
                    }
                }
                
            } catch (Exception $e) {
                $msg = "Error reading CSV file: " . $e->getMessage();
                // Update log status to failed
                mysqli_query($con, "UPDATE tblbulkuploadlogs SET Status='Failed', ErrorLog='" . mysqli_real_escape_string($con, $e->getMessage()) . "' WHERE ID=$uploadLogId");
            }
        } else {
            $msg = "Please upload a valid CSV file (.csv)";
        }
    } else {
        $msg = "Please select a file to upload";
    }
}

// Handle Manual Entry
if (isset($_POST['add_manual'])) {
    $pawnCode = "MANUAL" . date('Y') . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
    $customerID = (int)$_POST['customer_id'];
    $itemType = mysqli_real_escape_string($con, $_POST['item_type']);
    $itemDescription = mysqli_real_escape_string($con, $_POST['item_description']);
    $itemWeight = (float)$_POST['item_weight'];
    $itemPurity = mysqli_real_escape_string($con, $_POST['item_purity']);
    $estimatedValue = (float)$_POST['estimated_value'];
    $loanAmount = (float)$_POST['loan_amount'];
    $interestRate = (float)$_POST['interest_rate'];
    $loanDate = mysqli_real_escape_string($con, $_POST['loan_date']);
    $dueDate = mysqli_real_escape_string($con, $_POST['due_date']);
    $status = mysqli_real_escape_string($con, $_POST['status']);
    $notes = mysqli_real_escape_string($con, $_POST['notes']);
    
    $query = "INSERT INTO tblpawntransactions (PawnCode, CustomerID, ItemType, ItemDescription, ItemWeight, ItemPurity, 
              EstimatedValue, LoanAmount, OriginalLoanAmount, CurrentLoanAmount, InterestRate, LoanDate, DueDate, Status, 
              Notes, CreatedBy) 
              VALUES ('$pawnCode', $customerID, '$itemType', '$itemDescription', $itemWeight, '$itemPurity', $estimatedValue, 
                     $loanAmount, $loanAmount, $loanAmount, $interestRate, '$loanDate', '$dueDate', '$status', '$notes', $userId)";
    
    if (mysqli_query($con, $query)) {
        // Log manual entry
        $uploadCode = "MAN" . date('Y') . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        $logQuery = "INSERT INTO tblbulkuploadlogs (UploadCode, FileName, TotalRecords, SuccessfulRecords, UploadType, Status, UploadedBy) 
                    VALUES ('$uploadCode', 'Manual Entry', 1, 1, 'Manual', 'Completed', $userId)";
        mysqli_query($con, $logQuery);
        
        $msg = "Manual entry added successfully!";
    } else {
        $msg = "Error adding manual entry: " . mysqli_error($con);
    }
}

// Handle Update
if (isset($_POST['update_entry'])) {
    $id = (int)$_POST['entry_id'];
    $customerID = (int)$_POST['customer_id'];
    $itemType = mysqli_real_escape_string($con, $_POST['item_type']);
    $itemDescription = mysqli_real_escape_string($con, $_POST['item_description']);
    $itemWeight = (float)$_POST['item_weight'];
    $itemPurity = mysqli_real_escape_string($con, $_POST['item_purity']);
    $estimatedValue = (float)$_POST['estimated_value'];
    $loanAmount = (float)$_POST['loan_amount'];
    $interestRate = (float)$_POST['interest_rate'];
    $loanDate = mysqli_real_escape_string($con, $_POST['loan_date']);
    $dueDate = mysqli_real_escape_string($con, $_POST['due_date']);
    $status = mysqli_real_escape_string($con, $_POST['status']);
    $notes = mysqli_real_escape_string($con, $_POST['notes']);
    
    $query = "UPDATE tblpawntransactions SET CustomerID=$customerID, ItemType='$itemType', 
              ItemDescription='$itemDescription', ItemWeight=$itemWeight, ItemPurity='$itemPurity', 
              EstimatedValue=$estimatedValue, LoanAmount=$loanAmount, CurrentLoanAmount=$loanAmount, 
              InterestRate=$interestRate, LoanDate='$loanDate', DueDate='$dueDate', Status='$status', 
              Notes='$notes' WHERE ID=$id";
    
    if (mysqli_query($con, $query)) {
        $msg = "Entry updated successfully!";
    } else {
        $msg = "Error updating entry: " . mysqli_error($con);
    }
}

// Handle Delete
if (isset($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    $query = "DELETE FROM tblpawntransactions WHERE ID=$id";
    if (mysqli_query($con, $query)) {
        $msg = "Entry deleted successfully!";
    } else {
        $msg = "Error deleting entry: " . mysqli_error($con);
    }
}

// Get entry for editing
$editEntry = null;
if (isset($_GET['edit'])) {
    $id = (int)$_GET['edit'];
    $result = mysqli_query($con, "SELECT p.*, c.Name as CustomerName FROM tblpawntransactions p 
                                  JOIN tblcustomers c ON p.CustomerID = c.ID WHERE p.ID=$id");
    $editEntry = mysqli_fetch_assoc($result);
}

// Pagination for bulk entries
$limit = 20;
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// Search functionality
$search = isset($_GET['search']) ? mysqli_real_escape_string($con, $_GET['search']) : '';
$whereClause = "";
if (!empty($search)) {
    $whereClause = "WHERE (p.PawnCode LIKE '%$search%' OR c.Name LIKE '%$search%' OR p.ItemType LIKE '%$search%')";
}

// Get total count
$countQuery = "SELECT COUNT(*) as total FROM tblpawntransactions p 
               JOIN tblcustomers c ON p.CustomerID = c.ID 
               $whereClause AND (p.PawnCode LIKE 'BULK%' OR p.PawnCode LIKE 'MANUAL%')";
$countResult = mysqli_query($con, $countQuery);
$totalRecords = mysqli_fetch_assoc($countResult)['total'];
$totalPages = ceil($totalRecords / $limit);

// Fetch bulk entries
$entriesQuery = mysqli_query($con, "SELECT p.*, c.Name as CustomerName, c.MobileNumber 
                                   FROM tblpawntransactions p 
                                   JOIN tblcustomers c ON p.CustomerID = c.ID 
                                   $whereClause AND (p.PawnCode LIKE 'BULK%' OR p.PawnCode LIKE 'MANUAL%')
                                   ORDER BY p.CreatedDate DESC LIMIT $limit OFFSET $offset");

// Get customers for dropdown
$customersQuery = mysqli_query($con, "SELECT ID, CustomerCode, Name, MobileNumber FROM tblcustomers WHERE Status='Active' ORDER BY Name");

// Get upload logs
$uploadLogsQuery = mysqli_query($con, "SELECT * FROM tblbulkuploadlogs ORDER BY UploadDate DESC LIMIT 10");
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?php echo page_title('Bulk Upload'); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/compact.css" rel="stylesheet">
    <style>
        .upload-section {
            background: #f8f9fc;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .manual-entry-section {
            background: #e3f2fd;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .upload-logs {
            background: #fff3cd;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid">
                    
                    <?php if($msg) { ?>
                        <div class="alert alert-info"><?php echo $msg; ?></div>
                    <?php } ?>
                    
                    <!-- CSV Upload Section -->
                    <div class="upload-section">
                        <h4 class="mb-3"><i class="fas fa-file-csv"></i> CSV Bulk Upload</h4>
                        <div class="row">
                            <div class="col-md-8">
                                <form method="POST" enctype="multipart/form-data">
                                    <div class="form-group">
                                        <label>Select CSV File (.csv)</label>
                                        <input type="file" name="csv_file" class="form-control-file" accept=".csv" required>
                                        <small class="text-muted">CSV format: Customer Name, Mobile Number, Item Type, Item Description, Weight, Purity, Estimated Value, Loan Amount, Interest Rate, Loan Date, Due Date, Status, Notes</small>
                                    </div>
                                    <button type="submit" name="upload_csv" class="btn btn-success">
                                        <i class="fas fa-upload"></i> Upload CSV
                                    </button>
                                </form>
                            </div>
                            <div class="col-md-4">
                                <a href="download_sample_csv.php" class="btn btn-info btn-block">
                                    <i class="fas fa-download"></i> Download Sample CSV
                                </a>
                                <small class="text-muted mt-2 d-block">
                                    Download the sample format and fill your data accordingly.
                                </small>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Upload Logs -->
                    <?php if (mysqli_num_rows($uploadLogsQuery) > 0) { ?>
                    <div class="upload-logs">
                        <h5 class="mb-3"><i class="fas fa-history"></i> Recent Upload History</h5>
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>Upload Code</th>
                                        <th>File Name</th>
                                        <th>Type</th>
                                        <th>Total</th>
                                        <th>Success</th>
                                        <th>Failed</th>
                                        <th>Status</th>
                                        <th>Date</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while($log = mysqli_fetch_assoc($uploadLogsQuery)) { ?>
                                    <tr>
                                        <td><?php echo $log['UploadCode']; ?></td>
                                        <td><?php echo htmlspecialchars($log['FileName']); ?></td>
                                        <td><span class="badge badge-info"><?php echo $log['UploadType']; ?></span></td>
                                        <td><?php echo $log['TotalRecords']; ?></td>
                                        <td><span class="text-success"><?php echo $log['SuccessfulRecords']; ?></span></td>
                                        <td><span class="text-danger"><?php echo $log['FailedRecords']; ?></span></td>
                                        <td>
                                            <span class="badge badge-<?php echo $log['Status'] == 'Completed' ? 'success' : ($log['Status'] == 'Failed' ? 'danger' : 'warning'); ?>">
                                                <?php echo $log['Status']; ?>
                                            </span>
                                        </td>
                                        <td><?php echo date('d/m/Y H:i', strtotime($log['UploadDate'])); ?></td>
                                    </tr>
                                    <?php } ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    <?php } ?>
                    
                    <!-- Manual Entry Section -->
                    <div class="manual-entry-section">
                        <h4 class="mb-3"><i class="fas fa-plus"></i> 
                            <?php echo $editEntry ? 'Edit Entry' : 'Manual Entry'; ?>
                        </h4>
                        <form method="POST">
                            <?php if($editEntry) { ?>
                                <input type="hidden" name="entry_id" value="<?php echo $editEntry['ID']; ?>">
                            <?php } ?>
                            
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label>Customer <span class="text-danger">*</span></label>
                                        <select name="customer_id" class="form-control" required>
                                            <option value="">Select Customer</option>
                                            <?php 
                                            mysqli_data_seek($customersQuery, 0);
                                            while($customer = mysqli_fetch_assoc($customersQuery)) { 
                                            ?>
                                                <option value="<?php echo $customer['ID']; ?>" 
                                                        <?php echo ($editEntry && $editEntry['CustomerID'] == $customer['ID']) ? 'selected' : ''; ?>>
                                                    <?php echo $customer['CustomerCode'] . ' - ' . $customer['Name'] . ' (' . $customer['MobileNumber'] . ')'; ?>
                                                </option>
                                            <?php } ?>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label>Item Type <span class="text-danger">*</span></label>
                                        <select name="item_type" class="form-control" required>
                                            <option value="">Select Item Type</option>
                                            <option value="Gold" <?php echo ($editEntry && $editEntry['ItemType'] == 'Gold') ? 'selected' : ''; ?>>Gold</option>
                                            <option value="Silver" <?php echo ($editEntry && $editEntry['ItemType'] == 'Silver') ? 'selected' : ''; ?>>Silver</option>
                                            <option value="Diamond" <?php echo ($editEntry && $editEntry['ItemType'] == 'Diamond') ? 'selected' : ''; ?>>Diamond</option>
                                            <option value="Platinum" <?php echo ($editEntry && $editEntry['ItemType'] == 'Platinum') ? 'selected' : ''; ?>>Platinum</option>
                                            <option value="Other" <?php echo ($editEntry && $editEntry['ItemType'] == 'Other') ? 'selected' : ''; ?>>Other</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-12">
                                    <div class="form-group">
                                        <label>Item Description <span class="text-danger">*</span></label>
                                        <textarea name="item_description" class="form-control" rows="2" required><?php echo $editEntry ? htmlspecialchars($editEntry['ItemDescription']) : ''; ?></textarea>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label>Weight (grams) <span class="text-danger">*</span></label>
                                        <input type="number" step="0.001" name="item_weight" class="form-control" required
                                               value="<?php echo $editEntry ? $editEntry['ItemWeight'] : ''; ?>">
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label>Purity</label>
                                        <input type="text" name="item_purity" class="form-control" placeholder="22K, 18K, etc."
                                               value="<?php echo $editEntry ? htmlspecialchars($editEntry['ItemPurity']) : ''; ?>">
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label>Estimated Value <span class="text-danger">*</span></label>
                                        <input type="number" step="0.01" name="estimated_value" class="form-control" required
                                               value="<?php echo $editEntry ? $editEntry['EstimatedValue'] : ''; ?>">
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label>Loan Amount <span class="text-danger">*</span></label>
                                        <input type="number" step="0.01" name="loan_amount" class="form-control" required
                                               value="<?php echo $editEntry ? $editEntry['LoanAmount'] : ''; ?>">
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label>Interest Rate (%) <span class="text-danger">*</span></label>
                                        <input type="number" step="0.01" name="interest_rate" class="form-control" required
                                               value="<?php echo $editEntry ? $editEntry['InterestRate'] : '2.00'; ?>">
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label>Loan Date <span class="text-danger">*</span></label>
                                        <input type="date" name="loan_date" class="form-control" required
                                               value="<?php echo $editEntry ? $editEntry['LoanDate'] : date('Y-m-d'); ?>">
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label>Due Date <span class="text-danger">*</span></label>
                                        <input type="date" name="due_date" class="form-control" required
                                               value="<?php echo $editEntry ? $editEntry['DueDate'] : date('Y-m-d', strtotime('+3 months')); ?>">
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="form-group">
                                        <label>Status</label>
                                        <select name="status" class="form-control">
                                            <option value="Active" <?php echo ($editEntry && $editEntry['Status'] == 'Active') ? 'selected' : ''; ?>>Active</option>
                                            <option value="Redeemed" <?php echo ($editEntry && $editEntry['Status'] == 'Redeemed') ? 'selected' : ''; ?>>Redeemed</option>
                                            <option value="Extended" <?php echo ($editEntry && $editEntry['Status'] == 'Extended') ? 'selected' : ''; ?>>Extended</option>
                                            <option value="Auctioned" <?php echo ($editEntry && $editEntry['Status'] == 'Auctioned') ? 'selected' : ''; ?>>Auctioned</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label>Notes</label>
                                <textarea name="notes" class="form-control" rows="2"><?php echo $editEntry ? htmlspecialchars($editEntry['Notes']) : ''; ?></textarea>
                            </div>
                            
                            <button type="submit" name="<?php echo $editEntry ? 'update_entry' : 'add_manual'; ?>" 
                                    class="btn btn-primary">
                                <i class="fas fa-save"></i> <?php echo $editEntry ? 'Update' : 'Add'; ?>
                            </button>
                            <?php if($editEntry) { ?>
                                <a href="bulk_pawn_upload.php" class="btn btn-secondary">Cancel</a>
                            <?php } ?>
                        </form>
                    </div>
                    
                    <!-- Search Section -->
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <form method="GET" class="d-flex">
                                <input type="text" name="search" class="form-control" 
                                       placeholder="Search by Pawn Code, Customer Name, Item Type" 
                                       value="<?php echo htmlspecialchars($search); ?>">
                                <button type="submit" class="btn btn-primary ml-2">
                                    <i class="fas fa-search"></i>
                                </button>
                                <a href="bulk_pawn_upload.php" class="btn btn-secondary ml-2">
                                    <i class="fas fa-refresh"></i>
                                </a>
                            </form>
                        </div>
                    </div>
                    
                    <!-- Bulk Entries List -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3 d-flex justify-content-between align-items-center">
                            <h6 class="m-0 font-weight-bold text-primary">Bulk Uploaded Entries</h6>
                            <small class="text-muted">Total: <?php echo $totalRecords; ?> entries</small>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-bordered">
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Pawn Code</th>
                                            <th>Customer</th>
                                            <th>Item Details</th>
                                            <th>Weight</th>
                                            <th>Loan Amount</th>
                                            <th>Date</th>
                                            <th>Status</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php 
                                        if (mysqli_num_rows($entriesQuery) > 0) {
                                            $i = $offset + 1; 
                                            while($entry = mysqli_fetch_assoc($entriesQuery)) { 
                                        ?>
                                            <tr>
                                                <td><?php echo $i++; ?></td>
                                                <td><?php echo htmlspecialchars($entry['PawnCode']); ?></td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($entry['CustomerName']); ?></strong><br>
                                                    <small><?php echo htmlspecialchars($entry['MobileNumber']); ?></small>
                                                </td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($entry['ItemType']); ?></strong><br>
                                                    <small><?php echo htmlspecialchars(substr($entry['ItemDescription'], 0, 50)) . '...'; ?></small>
                                                </td>
                                                <td><?php echo number_format($entry['ItemWeight'], 3); ?>g</td>
                                                <td>₹<?php echo number_format($entry['LoanAmount'], 2); ?></td>
                                                <td><?php echo date('d/m/Y', strtotime($entry['LoanDate'])); ?></td>
                                                <td>
                                                    <span class="badge badge-<?php 
                                                        echo $entry['Status'] == 'Active' ? 'success' : 
                                                            ($entry['Status'] == 'Redeemed' ? 'primary' : 
                                                            ($entry['Status'] == 'Extended' ? 'warning' : 'danger')); 
                                                    ?>">
                                                        <?php echo htmlspecialchars($entry['Status']); ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <a href="bulk_pawn_upload.php?edit=<?php echo $entry['ID']; ?>" 
                                                       class="btn btn-sm btn-primary" title="Edit">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <a href="bulk_pawn_upload.php?delete=<?php echo $entry['ID']; ?>" 
                                                       onclick="return confirm('Are you sure you want to delete?')" 
                                                       class="btn btn-sm btn-danger" title="Delete">
                                                        <i class="fas fa-trash"></i>
                                                    </a>
                                                </td>
                                            </tr>
                                        <?php 
                                            }
                                        } else { 
                                        ?>
                                            <tr>
                                                <td colspan="9" class="text-center">No bulk entries found</td>
                                            </tr>
                                        <?php } ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <!-- Pagination -->
                            <?php if ($totalPages > 1) { ?>
                                <nav aria-label="Page navigation">
                                    <ul class="pagination justify-content-center">
                                        <?php if ($page > 1) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?php echo ($page-1); ?>&search=<?php echo urlencode($search); ?>">Previous</a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php for ($i = 1; $i <= $totalPages; $i++) { ?>
                                            <li class="page-item <?php echo ($i == $page) ? 'active' : ''; ?>">
                                                <a class="page-link" href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>"><?php echo $i; ?></a>
                                            </li>
                                        <?php } ?>
                                        
                                        <?php if ($page < $totalPages) { ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?page=<?php echo ($page+1); ?>&search=<?php echo urlencode($search); ?>">Next</a>
                                            </li>
                                        <?php } ?>
                                    </ul>
                                </nav>
                            <?php } ?>
                        </div>
                    </div>
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    
    <?php include_once('includes/footer2.php');?>
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
</body>
</html>
