<?php
session_start();

// Check if config file exists
$configPath = __DIR__ . '/includes/config.php';
if (!file_exists($configPath)) {
    die("Error: Database configuration file not found. Please create includes/config.php");
}

include_once($configPath);
include_once(__DIR__ . '/includes/brand.php');

if (!isset($con) || !$con) {
    die("Error: Database connection failed. Please check your database configuration.");
}

if (!isset($_SESSION['uid'])) {
    header('location:logout.php');
    exit();
}

$msg = "";
$userId = $_SESSION['uid'];

// Search and pagination variables
$search = isset($_GET['search']) ? mysqli_real_escape_string($con, $_GET['search']) : '';
$status = isset($_GET['status']) ? mysqli_real_escape_string($con, $_GET['status']) : '';
$limit = 15;
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// Handle form submissions
if ($_POST) {
    if (isset($_POST['create_pledge'])) {
        $pledgeCode = "BP" . date('Y') . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        $pawnTransactionID = (int)$_POST['pawn_transaction_id'];
        $bankAccountID = (int)$_POST['bank_account_id'];
        $pledgeAmount = (float)$_POST['pledge_amount'];
        $interestRate = (float)$_POST['interest_rate'];
        $pledgeDate = mysqli_real_escape_string($con, $_POST['pledge_date']);
        $maturityDate = mysqli_real_escape_string($con, $_POST['maturity_date']);
        $bankLoanNumber = mysqli_real_escape_string($con, $_POST['bank_loan_number']);
        $notes = mysqli_real_escape_string($con, $_POST['notes']);
        
        $query = "INSERT INTO tblbankpledges (PledgeCode, PawnTransactionID, BankAccountID, PledgeAmount, InterestRate, PledgeDate, MaturityDate, BankLoanNumber, Notes) 
                  VALUES ('$pledgeCode', $pawnTransactionID, $bankAccountID, $pledgeAmount, $interestRate, '$pledgeDate', '$maturityDate', '$bankLoanNumber', '$notes')";
        
        if (mysqli_query($con, $query)) {
            // Update pawn transaction status
            mysqli_query($con, "UPDATE tblpawntransactions SET Status='Bank Pledged' WHERE ID=$pawnTransactionID");
            $msg = "Bank pledge created successfully! Pledge Code: $pledgeCode";
        } else {
            $msg = "Error creating bank pledge: " . mysqli_error($con);
        }
    }
    
    if (isset($_POST['add_payment'])) {
        $pledgeID = (int)$_POST['pledge_id'];
        $paymentAmount = (float)$_POST['payment_amount'];
        $paymentDate = mysqli_real_escape_string($con, $_POST['payment_date']);
        $paymentType = mysqli_real_escape_string($con, $_POST['payment_type']);
        $paymentMethod = mysqli_real_escape_string($con, $_POST['payment_method']);
        $transactionRef = mysqli_real_escape_string($con, $_POST['transaction_reference']);
        $receiptNumber = mysqli_real_escape_string($con, $_POST['receipt_number']);
        $notes = mysqli_real_escape_string($con, $_POST['notes']);
        
        $query = "INSERT INTO tblbankpledgepayments (BankPledgeID, PaymentAmount, PaymentDate, PaymentType, PaymentMethod, TransactionReference, ReceiptNumber, Notes, CreatedBy) 
                  VALUES ($pledgeID, $paymentAmount, '$paymentDate', '$paymentType', '$paymentMethod', '$transactionRef', '$receiptNumber', '$notes', $userId)";
        
        if (mysqli_query($con, $query)) {
            $msg = "Payment recorded successfully!";
        } else {
            $msg = "Error recording payment: " . mysqli_error($con);
        }
    }
    
    if (isset($_POST['redeem_pledge'])) {
        $pledgeID = (int)$_POST['pledge_id'];
        $redemptionAmount = (float)$_POST['redemption_amount'];
        $redemptionDate = mysqli_real_escape_string($con, $_POST['redemption_date']);
        $redemptionType = mysqli_real_escape_string($con, $_POST['redemption_type']);
        $paymentMethod = mysqli_real_escape_string($con, $_POST['payment_method']);
        $transactionRef = mysqli_real_escape_string($con, $_POST['transaction_reference']);
        $notes = mysqli_real_escape_string($con, $_POST['notes']);
        
        $query = "INSERT INTO tblbankpledgeredemptions (BankPledgeID, RedemptionAmount, RedemptionDate, RedemptionType, PaymentMethod, TransactionReference, Notes, CreatedBy) 
                  VALUES ($pledgeID, $redemptionAmount, '$redemptionDate', '$redemptionType', '$paymentMethod', '$transactionRef', '$notes', $userId)";
        
        if (mysqli_query($con, $query)) {
            // Update pledge status
            $newStatus = ($redemptionType == 'Full') ? 'Redeemed' : 'Partially Paid';
            mysqli_query($con, "UPDATE tblbankpledges SET Status='$newStatus' WHERE ID=$pledgeID");
            
            if ($redemptionType == 'Full') {
                // Update original pawn transaction status
                $pawnQuery = mysqli_query($con, "SELECT PawnTransactionID FROM tblbankpledges WHERE ID=$pledgeID");
                $pawnData = mysqli_fetch_assoc($pawnQuery);
                mysqli_query($con, "UPDATE tblpawntransactions SET Status='Bank Redeemed' WHERE ID=" . $pawnData['PawnTransactionID']);
            }
            
            $msg = "Redemption recorded successfully!";
        } else {
            $msg = "Error recording redemption: " . mysqli_error($con);
        }
    }
}

// Build WHERE clause for filtering
$whereConditions = [];
if (!empty($search)) {
    $whereConditions[] = "(bp.PledgeCode LIKE '%$search%' OR c.Name LIKE '%$search%' OR ba.BankName LIKE '%$search%' OR bp.BankLoanNumber LIKE '%$search%')";
}
if (!empty($status)) {
    $whereConditions[] = "bp.Status = '$status'";
}
$whereClause = !empty($whereConditions) ? "WHERE " . implode(" AND ", $whereConditions) : "";

// Get total count for pagination
$countQuery = "SELECT COUNT(*) as total FROM tblbankpledges bp 
               JOIN tblpawntransactions pt ON bp.PawnTransactionID = pt.ID
               JOIN tblcustomers c ON pt.CustomerID = c.ID
               JOIN tblbankaccounts ba ON bp.BankAccountID = ba.ID $whereClause";
$countResult = mysqli_query($con, $countQuery);
$totalRecords = mysqli_fetch_assoc($countResult)['total'];
$totalPages = ceil($totalRecords / $limit);

// Fetch bank pledges
$pledgesQuery = "SELECT bp.*, pt.PawnCode, c.Name as CustomerName, c.MobileNumber, ba.BankName, ba.AccountNumber,
                 (SELECT SUM(PaymentAmount) FROM tblbankpledgepayments WHERE BankPledgeID = bp.ID) as TotalPaid,
                 (SELECT SUM(RedemptionAmount) FROM tblbankpledgeredemptions WHERE BankPledgeID = bp.ID) as TotalRedeemed
                 FROM tblbankpledges bp 
                 JOIN tblpawntransactions pt ON bp.PawnTransactionID = pt.ID
                 JOIN tblcustomers c ON pt.CustomerID = c.ID
                 JOIN tblbankaccounts ba ON bp.BankAccountID = ba.ID 
                 $whereClause 
                 ORDER BY bp.CreatedDate DESC 
                 LIMIT $limit OFFSET $offset";
$pledgesResult = mysqli_query($con, $pledgesQuery);

// Get data for dropdowns
$pawnTransactionsQuery = mysqli_query($con, "SELECT pt.ID, pt.PawnCode, c.Name as CustomerName, c.MobileNumber FROM tblpawntransactions pt JOIN tblcustomers c ON pt.CustomerID = c.ID WHERE pt.Status IN ('Active', 'Extended') ORDER BY pt.PawnCode");
$bankAccountsQuery = mysqli_query($con, "SELECT ID, BankName, AccountNumber FROM tblbankaccounts WHERE Status='Active' ORDER BY BankName");
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?php echo page_title('Bank Pledge Management'); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <link href="css/compact.css" rel="stylesheet">
    <!-- Add Select2 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <style>
        .filter-section {
            background: #f8f9fc;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .pledge-card {
            border-left: 4px solid #4e73df;
            margin-bottom: 15px;
        }
        .status-active { border-left-color: #28a745; }
        .status-partially-paid { border-left-color: #ffc107; }
        .status-redeemed { border-left-color: #17a2b8; }
        .status-defaulted { border-left-color: #dc3545; }
    </style>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid">
                    
                    <?php if($msg) { ?>
                        <div class="alert alert-info alert-dismissible fade show">
                            <?php echo $msg; ?>
                            <button type="button" class="close" data-dismiss="alert">&times;</button>
                        </div>
                    <?php } ?>
                    
                    <!-- Quick Actions -->
                    <div class="row mb-4">
                        <div class="col-md-12">
                            <button class="btn btn-primary" data-toggle="modal" data-target="#createPledgeModal">
                                <i class="fas fa-plus"></i> Create New Pledge
                            </button>
                            <button class="btn btn-success" data-toggle="modal" data-target="#addPaymentModal">
                                <i class="fas fa-money-bill"></i> Add Payment
                            </button>
                            <button class="btn btn-info" data-toggle="modal" data-target="#redeemPledgeModal">
                                <i class="fas fa-hand-holding-usd"></i> Redeem Pledge
                            </button>
                        </div>
                    </div>
                    
                    <!-- Filter Section -->
                    <div class="filter-section">
                        <form method="GET" class="row">
                            <div class="col-md-4">
                                <input type="text" name="search" class="form-control" 
                                       placeholder="Search by pledge code, customer, bank..." 
                                       value="<?php echo htmlspecialchars($search); ?>">
                            </div>
                            <div class="col-md-3">
                                <select name="status" class="form-control">
                                    <option value="">All Status</option>
                                    <option value="Active" <?php echo ($status == 'Active') ? 'selected' : ''; ?>>Active</option>
                                    <option value="Partially Paid" <?php echo ($status == 'Partially Paid') ? 'selected' : ''; ?>>Partially Paid</option>
                                    <option value="Redeemed" <?php echo ($status == 'Redeemed') ? 'selected' : ''; ?>>Redeemed</option>
                                    <option value="Defaulted" <?php echo ($status == 'Defaulted') ? 'selected' : ''; ?>>Defaulted</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <button type="submit" class="btn btn-primary btn-block">
                                    <i class="fas fa-search"></i> Filter
                                </button>
                            </div>
                            <div class="col-md-2">
                                <a href="bank_pledge_management.php" class="btn btn-secondary btn-block">
                                    <i class="fas fa-refresh"></i> Reset
                                </a>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Bank Pledges List -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3">
                            <h6 class="m-0 font-weight-bold text-primary">Bank Pledges (<?php echo $totalRecords; ?>)</h6>
                        </div>
                        <div class="card-body">
                            <?php if (mysqli_num_rows($pledgesResult) > 0) { ?>
                                <?php while($pledge = mysqli_fetch_assoc($pledgesResult)) { 
                                    $statusClass = 'status-' . strtolower(str_replace(' ', '-', $pledge['Status']));
                                    // Calculate balance as PledgeAmount - TotalPaid
                                    $balanceAmount = $pledge['PledgeAmount'] - ($pledge['TotalPaid'] ?? 0);
                                ?>
                                    <div class="card pledge-card <?php echo $statusClass; ?>">
                                        <div class="card-body">
                                            <div class="row">
                                                <div class="col-md-3">
                                                    <h6 class="font-weight-bold text-primary"><?php echo $pledge['PledgeCode']; ?></h6>
                                                    <p class="mb-1"><strong>Pawn:</strong> <?php echo $pledge['PawnCode']; ?></p>
                                                    <p class="mb-1"><strong>Customer:</strong> <?php echo htmlspecialchars($pledge['CustomerName']); ?></p>
                                                    <p class="mb-0"><strong>Mobile:</strong> <?php echo htmlspecialchars($pledge['MobileNumber']); ?></p>
                                                </div>
                                                <div class="col-md-3">
                                                    <p class="mb-1"><strong>Bank:</strong> <?php echo htmlspecialchars($pledge['BankName']); ?></p>
                                                    <p class="mb-1"><strong>A/C:</strong> <?php echo htmlspecialchars($pledge['AccountNumber']); ?></p>
                                                    <p class="mb-0"><strong>Loan #:</strong> <?php echo htmlspecialchars($pledge['BankLoanNumber']); ?></p>
                                                </div>
                                                <div class="col-md-3">
                                                    <p class="mb-1"><strong>Pledge Amount:</strong> ₹<?php echo number_format($pledge['PledgeAmount'], 2); ?></p>
                                                    <p class="mb-1"><strong>Total Paid:</strong> ₹<?php echo number_format($pledge['TotalPaid'] ?? 0, 2); ?></p>
                                                    <p class="mb-1"><strong>Total Redeemed:</strong> ₹<?php echo number_format($pledge['TotalRedeemed'] ?? 0, 2); ?></p>
                                                    <p class="mb-0"><strong>Balance:</strong> ₹<?php echo number_format($balanceAmount, 2); ?></p>
                                                </div>
                                                <div class="col-md-3">
                                                    <p class="mb-1"><strong>Pledge Date:</strong> <?php echo date('d/m/Y', strtotime($pledge['PledgeDate'])); ?></p>
                                                    <p class="mb-1"><strong>Maturity:</strong> <?php echo date('d/m/Y', strtotime($pledge['MaturityDate'])); ?></p>
                                                    <p class="mb-1"><strong>Status:</strong> 
                                                        <span class="badge badge-<?php echo ($pledge['Status'] == 'Active') ? 'success' : (($pledge['Status'] == 'Redeemed') ? 'info' : 'warning'); ?>">
                                                            <?php echo $pledge['Status']; ?>
                                                        </span>
                                                    </p>
                                                    <div class="btn-group">
                                                        <a href="bank_pledge_details.php?id=<?php echo $pledge['ID']; ?>" class="btn btn-sm btn-primary">
                                                            <i class="fas fa-eye"></i> View
                                                        </a>
                                                        <button class="btn btn-sm btn-success" onclick="addPayment(<?php echo $pledge['ID']; ?>)">
                                                            <i class="fas fa-plus"></i> Payment
                                                        </button>
                                                        <?php if ($pledge['Status'] != 'Redeemed') { ?>
                                                            <button class="btn btn-sm btn-info" onclick="redeemPledge(<?php echo $pledge['ID']; ?>)">
                                                                <i class="fas fa-hand-holding-usd"></i> Redeem
                                                            </button>
                                                        <?php } ?>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php } ?>
                                
                                <!-- Pagination -->
                                <?php if ($totalPages > 1) { ?>
                                    <nav>
                                        <ul class="pagination justify-content-center">
                                            <?php if ($page > 1) { ?>
                                                <li class="page-item">
                                                    <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page-1])); ?>">Previous</a>
                                                </li>
                                            <?php } ?>
                                            
                                            <?php for ($i = max(1, $page-2); $i <= min($totalPages, $page+2); $i++) { ?>
                                                <li class="page-item <?php echo ($i == $page) ? 'active' : ''; ?>">
                                                    <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>"><?php echo $i; ?></a>
                                                </li>
                                            <?php } ?>
                                            
                                            <?php if ($page < $totalPages) { ?>
                                                <li class="page-item">
                                                    <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page+1])); ?>">Next</a>
                                                </li>
                                            <?php } ?>
                                        </ul>
                                    </nav>
                                <?php } ?>
                            <?php } else { ?>
                                <div class="text-center py-5">
                                    <i class="fas fa-university fa-3x text-muted mb-3"></i>
                                    <h5>No bank pledges found</h5>
                                    <p class="text-muted">Create your first bank pledge to get started.</p>
                                </div>
                            <?php } ?>
                        </div>
                    </div>
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    
    <!-- Create Pledge Modal -->
    <div class="modal fade" id="createPledgeModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Create New Bank Pledge</h5>
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Pawn Transaction <span class="text-danger">*</span></label>
                                    <select name="pawn_transaction_id" class="form-control select2-pawn" required style="width:100%">
                                        <option value="">Select Pawn Transaction</option>
                                        <?php 
                                        mysqli_data_seek($pawnTransactionsQuery, 0);
                                        while($pawn = mysqli_fetch_assoc($pawnTransactionsQuery)) { ?>
                                            <option value="<?php echo $pawn['ID']; ?>">
                                                <?php echo $pawn['PawnCode'] . ' - ' . htmlspecialchars($pawn['CustomerName']) . ' - ' . htmlspecialchars($pawn['MobileNumber']); ?>
                                            </option>
                                        <?php } ?>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Bank Account <span class="text-danger">*</span></label>
                                    <select name="bank_account_id" class="form-control" required>
                                        <option value="">Select Bank Account</option>
                                        <?php while($bank = mysqli_fetch_assoc($bankAccountsQuery)) { ?>
                                            <option value="<?php echo $bank['ID']; ?>">
                                                <?php echo htmlspecialchars($bank['BankName']) . ' - ' . htmlspecialchars($bank['AccountNumber']); ?>
                                            </option>
                                        <?php } ?>
                                    </select>
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Pledge Amount <span class="text-danger">*</span></label>
                                    <input type="number" name="pledge_amount" class="form-control" step="0.01" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Interest Rate (%) <span class="text-danger">*</span></label>
                                    <input type="number" name="interest_rate" class="form-control" step="0.01" required>
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Pledge Date <span class="text-danger">*</span></label>
                                    <input type="date" name="pledge_date" class="form-control" value="<?php echo date('Y-m-d'); ?>" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Maturity Date <span class="text-danger">*</span></label>
                                    <input type="date" name="maturity_date" class="form-control" required>
                                </div>
                            </div>
                        </div>
                        <div class="form-group">
                            <label>Bank Loan Number</label>
                            <input type="text" name="bank_loan_number" class="form-control">
                        </div>
                        <div class="form-group">
                            <label>Notes</label>
                            <textarea name="notes" class="form-control" rows="3"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                        <button type="submit" name="create_pledge" class="btn btn-primary">Create Pledge</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Add Payment Modal -->
    <div class="modal fade" id="addPaymentModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Add Payment</h5>
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="pledge_id" id="payment_pledge_id">
                        <div class="form-group">
                            <label>Payment Amount <span class="text-danger">*</span></label>
                            <input type="number" name="payment_amount" class="form-control" step="0.01" required>
                        </div>
                        <div class="form-group">
                            <label>Payment Date <span class="text-danger">*</span></label>
                            <input type="date" name="payment_date" class="form-control" value="<?php echo date('Y-m-d'); ?>" required>
                        </div>
                        <div class="form-group">
                            <label>Payment Type <span class="text-danger">*</span></label>
                            <select name="payment_type" class="form-control" required>
                                <option value="Interest">Interest</option>
                                <option value="Principal">Principal</option>
                                <option value="Penalty">Penalty</option>
                                <option value="Other">Other</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Payment Method <span class="text-danger">*</span></label>
                            <select name="payment_method" class="form-control" required>
                                <option value="Cash">Cash</option>
                                <option value="Bank Transfer">Bank Transfer</option>
                                <option value="Cheque">Cheque</option>
                                <option value="DD">DD</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Transaction Reference</label>
                            <input type="text" name="transaction_reference" class="form-control">
                        </div>
                        <div class="form-group">
                            <label>Receipt Number</label>
                            <input type="text" name="receipt_number" class="form-control">
                        </div>
                        <div class="form-group">
                            <label>Notes</label>
                            <textarea name="notes" class="form-control" rows="2"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                        <button type="submit" name="add_payment" class="btn btn-success">Add Payment</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Redeem Pledge Modal -->
    <div class="modal fade" id="redeemPledgeModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Redeem Pledge</h5>
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="pledge_id" id="redeem_pledge_id">
                        <div class="form-group">
                            <label>Redemption Amount <span class="text-danger">*</span></label>
                            <input type="number" name="redemption_amount" class="form-control" step="0.01" required>
                        </div>
                        <div class="form-group">
                            <label>Redemption Date <span class="text-danger">*</span></label>
                            <input type="date" name="redemption_date" class="form-control" value="<?php echo date('Y-m-d'); ?>" required>
                        </div>
                        <div class="form-group">
                            <label>Redemption Type <span class="text-danger">*</span></label>
                            <select name="redemption_type" class="form-control" required>
                                <option value="Full">Full Redemption</option>
                                <option value="Partial">Partial Redemption</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Payment Method <span class="text-danger">*</span></label>
                            <select name="payment_method" class="form-control" required>
                                <option value="Cash">Cash</option>
                                <option value="Bank Transfer">Bank Transfer</option>
                                <option value="Cheque">Cheque</option>
                                <option value="DD">DD</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Transaction Reference</label>
                            <input type="text" name="transaction_reference" class="form-control">
                        </div>
                        <div class="form-group">
                            <label>Notes</label>
                            <textarea name="notes" class="form-control" rows="2"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                        <button type="submit" name="redeem_pledge" class="btn btn-info">Process Redemption</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
    <!-- Add Select2 JS -->
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script>
    $(document).ready(function() {
        $('.select2-pawn').select2({
            dropdownParent: $('#createPledgeModal'),
            placeholder: "Search by Pawn Code, Name, or Mobile",
            allowClear: true
        });
    });
    
    function addPayment(pledgeId) {
        $('#payment_pledge_id').val(pledgeId);
        $('#addPaymentModal').modal('show');
    }
    
    function redeemPledge(pledgeId) {
        $('#redeem_pledge_id').val(pledgeId);
        $('#redeemPledgeModal').modal('show');
    }
    </script>
</body>
</html>
