<?php
session_start();
include_once('includes/config.php');

// Centralized shop name (with location)
$shopName = 'Jagadamba Opticals - Madurai';

// Determine user type and ID
$isAdmin = isset($_SESSION['aid']);
$userId = $isAdmin ? $_SESSION['aid'] : (isset($_SESSION['uid']) ? $_SESSION['uid'] : null);
$userType = $isAdmin ? 'admin' : (isset($_SESSION['usertype']) ? $_SESSION['usertype'] : null);

if (!$userId) {
    header('location:login.php');
    exit();
}

// CSRF token ensure
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
}

// Delete handler (kept same functionality)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_id'])) {
    $deleteId = (int)$_POST['delete_id'];
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $error_msg = 'Invalid request token.';
    } else if ($deleteId > 0) {
        if ($isAdmin) {
            $sqlDel = "DELETE FROM customer_prescriptions WHERE id=?";
            $stmtDel = $con->prepare($sqlDel);
            $stmtDel->bind_param('i', $deleteId);
        } else {
            $sqlDel = "DELETE FROM customer_prescriptions WHERE id=? AND user_id=?";
            $stmtDel = $con->prepare($sqlDel);
            $stmtDel->bind_param('ii', $deleteId, $userId);
        }
        if ($stmtDel && $stmtDel->execute()) {
            if ($stmtDel->affected_rows > 0) {
                $success_msg = 'Prescription deleted successfully.';
            } else {
                $error_msg = 'Delete not permitted or record not found.';
            }
        } else {
            $error_msg = 'Delete failed.';
        }
        if ($stmtDel) $stmtDel->close();
    }
}

// Status update (AJAX) handler
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_status') {
    header('Content-Type: application/json');
    $resp = ['success' => false, 'message' => 'Unknown error'];
    $allowed_status = ['Pending','In Progress','Ready','Delivered','Cancelled'];
    $prescription_id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
    $new_status = isset($_POST['new_status']) ? trim($_POST['new_status']) : '';
    $csrf = isset($_POST['csrf_token']) ? $_POST['csrf_token'] : '';
    if ($csrf !== $_SESSION['csrf_token']) {
        $resp['message'] = 'Invalid token';
    } elseif ($prescription_id <= 0) {
        $resp['message'] = 'Invalid ID';
    } elseif (!in_array($new_status, $allowed_status)) {
        $resp['message'] = 'Invalid status';
    } else {
        if ($isAdmin) {
            $sqlUp = "UPDATE customer_prescriptions SET order_status=?, updated_at=NOW() WHERE id=?";
            $stmtUp = $con->prepare($sqlUp);
            $stmtUp->bind_param('si', $new_status, $prescription_id);
        } else {
            $sqlUp = "UPDATE customer_prescriptions SET order_status=?, updated_at=NOW() WHERE id=? AND user_id=?";
            $stmtUp = $con->prepare($sqlUp);
            $stmtUp->bind_param('sii', $new_status, $prescription_id, $userId);
        }
        if ($stmtUp && $stmtUp->execute()) {
            if ($stmtUp->affected_rows > 0) {
                $resp = ['success' => true, 'message' => 'Status updated', 'new_status' => $new_status, 'id' => $prescription_id];
            } else {
                $resp['message'] = 'No change or not permitted';
            }
        } else {
            $resp['message'] = 'Update failed';
        }
        if (isset($stmtUp) && $stmtUp) $stmtUp->close();
    }
    echo json_encode($resp);
    exit();
}

// Handle export functionality
if (isset($_GET['export']) && $_GET['export'] == 'excel') {
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="staff_prescription_report_' . date('Y-m-d') . '.xls"');
    header('Pragma: no-cache');
    header('Expires: 0');
}

// Date and filter logic (default: today)
$todayDefault = date('Y-m-d');
$from = (isset($_GET['from']) && $_GET['from'] !== '') ? $_GET['from'] : $todayDefault;
$to = (isset($_GET['to']) && $_GET['to'] !== '') ? $_GET['to'] : $todayDefault;
$status_filter = isset($_GET['status']) ? $_GET['status'] : '';
$search = isset($_GET['search']) ? $_GET['search'] : '';
$frame_type_filter = isset($_GET['frame_type']) ? $_GET['frame_type'] : '';
$lens_type_filter = isset($_GET['lens_type']) ? $_GET['lens_type'] : '';

// Build query
$query = "SELECT * FROM customer_prescriptions WHERE DATE(created_at) BETWEEN ? AND ?";
$params = [$from, $to];
$types = "ss";

if ($status_filter && in_array($status_filter, ['Pending','In Progress','Ready','Delivered','Cancelled'])) {
    $query .= " AND order_status=?";
    $params[] = $status_filter;
    $types .= "s";
}

if ($search) {
    $query .= " AND (customer_name LIKE ? OR mobile LIKE ? OR order_number LIKE ?)";
    $search_param = "%$search%";
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
    $types .= "sss";
}

if ($frame_type_filter) {
    $query .= " AND frame_type=?";
    $params[] = $frame_type_filter;
    $types .= "s";
}

if ($lens_type_filter) {
    $query .= " AND lens_type=?";
    $params[] = $lens_type_filter;
    $types .= "s";
}

if (!$isAdmin) {
    $query .= " AND user_id=?";
    $params[] = $userId;
    $types .= "i";
}

$query .= " ORDER BY created_at DESC";
$stmt = $con->prepare($query);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();
$prescriptions = [];
while ($row = $result->fetch_assoc()) { $prescriptions[] = $row; }
$stmt->close();

// Only counts needed (no revenue / sums for staff view)
$total_orders = count($prescriptions);
$status_counts = array_count_values(array_column($prescriptions, 'order_status'));

// Excel export table (same columns)
if (isset($_GET['export']) && $_GET['export'] == 'excel') {
    echo '<table border="1">';
    echo '<tr>\n            <th>Order Number</th>\n            <th>Date</th>\n            <th>Customer Name</th>\n            <th>Mobile</th>\n            <th>Address</th>\n            <th>Right Eye SPH</th>\n            <th>Right Eye CYL</th>\n            <th>Right Eye AXIS</th>\n            <th>Right Eye ADD</th>\n            <th>Left Eye SPH</th>\n            <th>Left Eye CYL</th>\n            <th>Left Eye AXIS</th>\n            <th>Left Eye ADD</th>\n            <th>Right Contact SPH</th>\n            <th>Right Contact CYL</th>\n            <th>Right Contact AXIS</th>\n            <th>Left Contact SPH</th>\n            <th>Left Contact CYL</th>\n            <th>Left Contact AXIS</th>\n            <th>Frame Name</th>\n            <th>Lens Name</th>\n            <th>Frame Type</th>\n            <th>Lens Type</th>\n            <th>Total Amount</th>\n            <th>Discount Type</th>\n            <th>Discount Value</th>\n            <th>Discount Amount</th>\n            <th>Final Amount</th>\n            <th>Advance Amount</th>\n            <th>Balance Amount</th>\n            <th>Order Status</th>\n            <th>Delivery Date</th>\n            <th>Notes</th>\n          </tr>';
    foreach($prescriptions as $row) {
        $final_amount = $row['total_amount'] - ($row['discount_amount'] ?: 0);
        echo '<tr>';
        echo '<td>' . htmlspecialchars($row['order_number'] ?: 'N/A') . '</td>';
        echo '<td>' . date('d-M-Y', strtotime($row['created_at'])) . '</td>';
        echo '<td>' . htmlspecialchars($row['customer_name']) . '</td>';
        echo '<td>' . htmlspecialchars($row['mobile']) . '</td>';
        echo '<td>' . htmlspecialchars($row['address']) . '</td>';
        echo '<td>' . htmlspecialchars($row['right_eye_sph']) . '</td>';
        echo '<td>' . htmlspecialchars($row['right_eye_cyl']) . '</td>';
        echo '<td>' . htmlspecialchars($row['right_eye_axis']) . '</td>';
        echo '<td>' . htmlspecialchars($row['right_eye_addition'] ?: '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['left_eye_sph']) . '</td>';
        echo '<td>' . htmlspecialchars($row['left_eye_cyl']) . '</td>';
        echo '<td>' . htmlspecialchars($row['left_eye_axis']) . '</td>';
        echo '<td>' . htmlspecialchars($row['left_eye_addition'] ?: '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['right_contact_sph'] ?? '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['right_contact_cyl'] ?? '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['right_contact_axis'] ?? '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['left_contact_sph'] ?? '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['left_contact_cyl'] ?? '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['left_contact_axis'] ?? '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['frame_name']) . '</td>';
        echo '<td>' . htmlspecialchars($row['lens_name'] ?: '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['frame_type']) . '</td>';
        echo '<td>' . htmlspecialchars($row['lens_type']) . '</td>';
        echo '<td>' . number_format($row['total_amount'], 2) . '</td>';
        echo '<td>' . htmlspecialchars($row['discount_type'] ?: '-') . '</td>';
        echo '<td>' . number_format($row['discount_value'] ?: 0, 2) . '</td>';
        echo '<td>' . number_format($row['discount_amount'] ?: 0, 2) . '</td>';
        echo '<td>' . number_format($final_amount, 2) . '</td>';
        echo '<td>' . number_format($row['advance_amount'], 2) . '</td>';
        echo '<td>' . number_format($row['balance_amount'], 2) . '</td>';
        echo '<td>' . htmlspecialchars($row['order_status']) . '</td>';
        echo '<td>' . ($row['delivery_date'] ? date('d-M-Y', strtotime($row['delivery_date'])) : '') . '</td>';
        echo '<td>' . htmlspecialchars($row['notes']) . '</td>';
        echo '</tr>';
    }
    echo '</table>';
    exit();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title><?php echo htmlspecialchars($shopName); ?> | Staff Prescription Reports</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet">
<link href="css/sb-admin-2.min.css" rel="stylesheet">
<style>
body { background: #f8f9fc; }
.card { border-radius: 1rem; }
.form-control, .btn { border-radius: 0.5rem; }
.report-table { font-size: 13px; }
@media print { .no-print { display:none !important; } body { background:#fff !important; } }
.status-pill { display:inline-flex; align-items:center; gap:4px; padding:4px 8px; border:1px solid #e1e5e9; border-radius:20px; font-size:11px; text-decoration:none; margin:2px 4px 2px 0; }
.status-pill.active { background:#007bff; color:#fff; border-color:#007bff; }
.pill-count { background:rgba(0,0,0,0.1); padding:1px 4px; border-radius:8px; font-size:9px; font-weight:600; }
.status-pill.active .pill-count { background:rgba(255,255,255,.25); }
.quick-filter-container { background:#fff; padding:12px; border-radius:12px; box-shadow:0 2px 8px rgba(0,0,0,0.08); margin-bottom:16px; }
.filter-item { display:flex; align-items:center; background:#f8f9fa; border:1px solid #e9ecef; border-radius:8px; padding:6px 10px; }
.filter-input { border:none; background:transparent; outline:none; font-size:13px; }
.filter-btn { width:36px; height:36px; border-radius:8px; border:none; display:flex; align-items:center; justify-content:center; }
.primary-btn { background:#007bff; color:#fff; }
.reset-btn { background:#6c757d; color:#fff; }
.toggle-btn { background:#28a745; color:#fff; }
.toggle-btn.active { background:#dc3545; }
.table-actions .btn { margin-right:4px; }
.table-actions .btn:last-child { margin-right:0; }
</style>
</head>
<body id="page-top">
<div id="wrapper">
<?php include_once('includes/sidebar.php');?>
<div id="content-wrapper" class="d-flex flex-column">
<div id="content">
<?php include_once('includes/topbar.php');?>
<div class="container-fluid main-content">
    <div class="d-sm-flex align-items-center justify-content-between mb-4 no-print">
        <h1 class="h3 mb-0 text-gray-800"><i class="fas fa-user-md text-primary"></i> Staff Prescription Reports</h1>
        <div class="page-header-actions">
            <a href="customer-prescriptions.php" class="btn btn-secondary btn-sm"><i class="fas fa-arrow-left"></i> Back</a>
            <button onclick="window.print()" class="btn btn-info btn-sm"><i class="fas fa-print"></i> Print</button>
            <a href="?<?php echo http_build_query(array_merge($_GET, ['export' => 'excel'])); ?>" class="btn btn-success btn-sm"><i class="fas fa-file-excel"></i> Excel</a>
        </div>
    </div>

    <div class="quick-filter-container no-print mb-3">
        <form method="get" action="" class="quick-filter-form">
            <div class="filter-row">
                <div class="filter-group" style="display:flex; gap:12px; flex-wrap:wrap;">
                    <div class="filter-item date-range">
                        <input type="date" name="from" class="filter-input" value="<?php echo htmlspecialchars($from); ?>">
                        <span style="font-size:11px; margin:0 4px;">to</span>
                        <input type="date" name="to" class="filter-input" value="<?php echo htmlspecialchars($to); ?>">
                    </div>
                    <div class="filter-item" style="flex-grow:2; min-width:200px;">
                        <input type="text" name="search" class="filter-input" placeholder="Search customer, mobile, order..." value="<?php echo htmlspecialchars($search); ?>">
                    </div>
                    <div style="display:flex; gap:6px; align-items:center;">
                        <button type="submit" class="filter-btn primary-btn" title="Apply"><i class="fas fa-filter"></i></button>
                        <a href="staff-prescription-reports.php" class="filter-btn reset-btn" title="Reset"><i class="fas fa-undo"></i></a>
                        <button type="button" class="filter-btn toggle-btn" onclick="toggleAdvancedFilters()" title="More Filters"><i class="fas fa-sliders-h"></i></button>
                    </div>
                </div>
            </div>

            <div class="status-pills no-print" style="margin-top:6px;">
                <a href="?<?php echo http_build_query(array_merge($_GET, ['status' => ''])); ?>" class="status-pill <?php echo empty($status_filter) ? 'active' : ''; ?>">All <span class="pill-count"><?php echo $total_orders; ?></span></a>
                <?php foreach(['Pending','In Progress','Ready','Delivered','Cancelled'] as $status): $count = isset($status_counts[$status]) ? $status_counts[$status] : 0; if($count>0 || $status_filter===$status): ?>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['status' => $status])); ?>" class="status-pill <?php echo $status_filter===$status ? 'active' : ''; ?>"><?php echo $status; ?> <span class="pill-count"><?php echo $count; ?></span></a>
                <?php endif; endforeach; ?>
            </div>

            <div id="advancedFilters" style="display:none; margin-top:10px;">
                <div style="display:flex; gap:12px; flex-wrap:wrap;">
                    <select name="frame_type" class="form-control form-control-sm" style="max-width:180px;">
                        <option value="">All Frame Types</option>
                        <?php foreach(['Metal Frame','Cell Frame','Supra Frame','Sunglass','Rimless','Semi-Rimless','Others'] as $f): ?>
                            <option value="<?php echo $f; ?>" <?php if($frame_type_filter==$f) echo 'selected'; ?>><?php echo $f; ?></option>
                        <?php endforeach; ?>
                    </select>
                    <select name="lens_type" class="form-control form-control-sm" style="max-width:180px;">
                        <option value="">All Lens Types</option>
                    </select>
                </div>
            </div>
        </form>
    </div>

    <!-- Detail Table Only (no summary cards) -->
    <div class="card shadow">
        <div class="card-header bg-dark text-white py-2">
            <h6 class="m-0">Prescription List (<?php echo $total_orders; ?>)</h6>
        </div>
        <div class="card-body p-0">
            <?php if(isset($success_msg)): ?><div class="alert alert-success m-2 py-2 px-3 small"><?php echo htmlspecialchars($success_msg); ?></div><?php endif; ?>
            <?php if(isset($error_msg)): ?><div class="alert alert-danger m-2 py-2 px-3 small"><?php echo htmlspecialchars($error_msg); ?></div><?php endif; ?>
            <div class="table-responsive">
                <table class="table table-sm table-striped table-hover mb-0 report-table">
                    <thead class="thead-light">
                        <tr>
                            <th>#</th>
                            <th>Date</th>
                            <th>Order</th>
                            <th>Customer</th>
                            <th>Mobile</th>
                            <th>Frame</th>
                            <th>Lens</th>
                            <th>Total</th>
                            <th>Adv</th>
                            <th>Bal</th>
                            <th>Status</th>
                            <th class="no-print">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php if(empty($prescriptions)): ?>
                        <tr><td colspan="12" class="text-center text-muted small">No prescriptions found for selected filters.</td></tr>
                    <?php else: $i=1; foreach($prescriptions as $row): ?>
                        <tr>
                            <td><?php echo $i++; ?></td>
                            <td><?php echo date('d-M-Y', strtotime($row['created_at'])); ?></td>
                            <td><?php echo htmlspecialchars($row['order_number'] ?: 'N/A'); ?></td>
                            <td><?php echo htmlspecialchars($row['customer_name']); ?></td>
                            <td><?php echo htmlspecialchars($row['mobile']); ?></td>
                            <td><?php echo htmlspecialchars($row['frame_name']); ?></td>
                            <td><?php echo htmlspecialchars($row['lens_name']); ?></td>
                            <td><?php echo number_format($row['total_amount'],2); ?></td>
                            <td><?php echo number_format($row['advance_amount'],2); ?></td>
                            <td><?php echo number_format($row['balance_amount'],2); ?></td>
                            <td><span class="badge badge-pill badge-secondary" style="font-size:11px;"><?php echo htmlspecialchars($row['order_status']); ?></span></td>
                            <td class="no-print table-actions">
                                <div class="btn-group btn-group-sm">
                                    <a href="customer-prescription-details.php?id=<?php echo $row['id']; ?>" class="btn btn-info" title="View"><i class="fas fa-eye"></i></a>
                                    <a href="customer-prescriptions.php?edit=<?php echo $row['id']; ?>" class="btn btn-success" title="Edit"><i class="fas fa-edit"></i></a>
                                    <button onclick="sendPrescriptionWhatsApp('<?php echo htmlspecialchars($row['mobile']); ?>','<?php echo htmlspecialchars(addslashes($row['customer_name'])); ?>','<?php echo htmlspecialchars(addslashes($row['order_number'])); ?>','<?php echo htmlspecialchars(addslashes($row['order_status'])); ?>')" class="btn btn-success" style="background:#25D366; border-color:#25D366;" title="WhatsApp"><i class="fab fa-whatsapp"></i></button>
                                    <form method="post" action="" onsubmit="return confirmDelete();" style="display:inline-block;">
                                        <input type="hidden" name="delete_id" value="<?php echo $row['id']; ?>">
                                        <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                                        <button type="submit" class="btn btn-danger" title="Delete"><i class="fas fa-trash"></i></button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <div class="text-center mt-4 print-section d-none d-print-block">
        <hr>
        <p class="text-muted">Report generated on <?php echo date('d-M-Y h:i A'); ?> | <?php echo htmlspecialchars($shopName); ?> Staff Report</p>
    </div>
</div>
</div>
</div>
<?php include_once('includes/footer.php');?>
</div>
<?php include_once('includes/footer2.php');?>
<script src="vendor/jquery/jquery.min.js"></script>
<script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
<script src="vendor/jquery-easing/jquery.easing.min.js"></script>
<script src="js/sb-admin-2.min.js"></script>
<script>
function toggleAdvancedFilters(){
    const adv = document.getElementById('advancedFilters');
    const btn = document.querySelector('.toggle-btn');
    if(adv.style.display==='none'){ adv.style.display='block'; btn.classList.add('active'); btn.innerHTML='<i class="fas fa-times"></i>'; } else { adv.style.display='none'; btn.classList.remove('active'); btn.innerHTML='<i class="fas fa-sliders-h"></i>'; }
}
function sendPrescriptionWhatsApp(mobile, customerName, orderNumber, status){
    const shopName = <?php echo json_encode($shopName); ?>;
    const today = new Date().toLocaleDateString('en-GB');
    let statusLine='';
    switch((status||'').toLowerCase()){
        case 'pending': statusLine='Your order is received.'; break;
        case 'in progress': statusLine='Your order is being prepared.'; break;
        case 'ready': statusLine='Your glasses are ready for pickup.'; break;
        case 'delivered': statusLine='Your order has been delivered.'; break;
        case 'cancelled': statusLine='Order has been cancelled.'; break;
        default: statusLine='We are working on your order.';
    }
    const msg = `*${shopName}*\nHi ${customerName},\nOrder: ${orderNumber}\nDate: ${today}\n${statusLine}\nThank you.`;
    const clean = mobile.replace(/\D/g,'');
    const num = clean.length===10 ? '91'+clean : clean;
    window.open(`https://wa.me/${num}?text=${encodeURIComponent(msg)}`, '_blank');
}
function confirmDelete(){return confirm('Delete this prescription? This cannot be undone.');}
// Auto submit on date change
document.addEventListener('DOMContentLoaded',()=>{
    document.querySelectorAll('input[type=date]').forEach(i=> i.addEventListener('change',()=> i.form.submit()));
    document.querySelectorAll('select').forEach(s=> s.addEventListener('change',()=> s.form.submit()));
});
</script>
</body>
</html>
