<?php
session_start();
// Start output buffering early so we can safely clean before JSON responses
if (!ob_get_level()) { ob_start(); }
include_once('includes/config.php');

// Centralized shop name (with location)
$shopName = 'Jagadamba Opticals - Madurai';

// Determine user type and ID
$isAdmin = isset($_SESSION['aid']);
$userId = $isAdmin ? $_SESSION['aid'] : (isset($_SESSION['uid']) ? $_SESSION['uid'] : null);
$userType = $isAdmin ? 'admin' : (isset($_SESSION['usertype']) ? $_SESSION['usertype'] : null);

if (!$userId) {
    // For AJAX requests, return JSON instead of redirecting
    if (isset($_POST['action']) || (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest')) {
        if (ob_get_level()) ob_clean();
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'message' => 'Session expired. Please log in again.', 'redirect' => 'login.php']);
        exit();
    }
    header('location:login.php');
    exit();
}

// CSRF token ensure
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
}

// Delete handler (Soft delete to preserve order number sequence)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_id'])) {
    $deleteId = (int)$_POST['delete_id'];
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $error_msg = 'Invalid request token.';
    } else if ($deleteId > 0) {
        // Check if record has order number
        $checkSql = $isAdmin ? "SELECT order_number FROM customer_prescriptions WHERE id=?" : "SELECT order_number FROM customer_prescriptions WHERE id=? AND user_id=?";
        $checkStmt = $con->prepare($checkSql);
        if ($isAdmin) {
            $checkStmt->bind_param('i', $deleteId);
        } else {
            $checkStmt->bind_param('ii', $deleteId, $userId);
        }
        
        if ($checkStmt && $checkStmt->execute()) {
            $result = $checkStmt->get_result();
            if ($row = $result->fetch_assoc()) {
                if (!empty($row['order_number'])) {
                    // Soft delete: Mark as cancelled to preserve order number sequence
                    $deleteNote = "DELETED on " . date('Y-m-d H:i:s') . " - Order cancelled to preserve number sequence";
                    if ($isAdmin) {
                        $sqlSoft = "UPDATE customer_prescriptions SET order_status='Cancelled', notes=CONCAT(COALESCE(notes,''), '\n', ?), updated_at=NOW() WHERE id=?";
                        $stmtSoft = $con->prepare($sqlSoft);
                        $stmtSoft->bind_param('si', $deleteNote, $deleteId);
                    } else {
                        $sqlSoft = "UPDATE customer_prescriptions SET order_status='Cancelled', notes=CONCAT(COALESCE(notes,''), '\n', ?), updated_at=NOW() WHERE id=? AND user_id=?";
                        $stmtSoft = $con->prepare($sqlSoft);
                        $stmtSoft->bind_param('sii', $deleteNote, $deleteId, $userId);
                    }
                    
                    if ($stmtSoft && $stmtSoft->execute()) {
                        if ($stmtSoft->affected_rows > 0) {
                            $success_msg = 'Order marked as cancelled to preserve order number sequence.';
                        } else {
                            $error_msg = 'Delete not permitted or record not found.';
                        }
                        $stmtSoft->close();
                    } else {
                        $error_msg = 'Delete failed.';
                    }
                } else {
                    // Hard delete if no order number
                    if ($isAdmin) {
                        $sqlDel = "DELETE FROM customer_prescriptions WHERE id=?";
                        $stmtDel = $con->prepare($sqlDel);
                        $stmtDel->bind_param('i', $deleteId);
                    } else {
                        $sqlDel = "DELETE FROM customer_prescriptions WHERE id=? AND user_id=?";
                        $stmtDel = $con->prepare($sqlDel);
                        $stmtDel->bind_param('ii', $deleteId, $userId);
                    }
                    
                    if ($stmtDel && $stmtDel->execute()) {
                        if ($stmtDel->affected_rows > 0) {
                            $success_msg = 'Prescription deleted successfully.';
                        } else {
                            $error_msg = 'Delete not permitted or record not found.';
                        }
                        $stmtDel->close();
                    } else {
                        $error_msg = 'Delete failed.';
                    }
                }
            } else {
                $error_msg = 'Record not found.';
            }
            $checkStmt->close();
        } else {
            $error_msg = 'Database error occurred.';
        }
    }
}

// Status update (AJAX) handler
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_status') {
    // Clean output buffer to prevent any HTML/text before JSON
    if (ob_get_level()) ob_clean();
    // Start fresh buffer to capture any accidental output and discard it later
    ob_start();
    // Hide PHP warnings/notices from bleeding into JSON
    @ini_set('display_errors', '0');
    header('Content-Type: application/json');
    $resp = ['success' => false, 'message' => 'Unknown error'];
    $allowed_status = ['Pending','In Progress','Ready','Delivered','Cancelled'];
    $prescription_id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
    $new_status = isset($_POST['new_status']) ? trim($_POST['new_status']) : '';
    $csrf = isset($_POST['csrf_token']) ? $_POST['csrf_token'] : '';
    if ($csrf !== $_SESSION['csrf_token']) {
        $resp['message'] = 'Invalid token';
    } elseif ($prescription_id <= 0) {
        $resp['message'] = 'Invalid ID';
    } elseif (!in_array($new_status, $allowed_status)) {
        $resp['message'] = 'Invalid status';
    } else {
        if ($isAdmin) {
            $sqlUp = "UPDATE customer_prescriptions SET order_status=?, updated_at=NOW() WHERE id=?";
            $stmtUp = $con->prepare($sqlUp);
            $stmtUp->bind_param('si', $new_status, $prescription_id);
        } else {
            $sqlUp = "UPDATE customer_prescriptions SET order_status=?, updated_at=NOW() WHERE id=? AND user_id=?";
            $stmtUp = $con->prepare($sqlUp);
            $stmtUp->bind_param('sii', $new_status, $prescription_id, $userId);
        }
        if ($stmtUp && $stmtUp->execute()) {
            if ($stmtUp->affected_rows > 0) {
                $resp = ['success' => true, 'message' => 'Status updated', 'new_status' => $new_status, 'id' => $prescription_id];
            } else {
                $resp['message'] = 'No change or not permitted';
            }
        } else {
            $resp['message'] = 'Update failed';
        }
        if (isset($stmtUp) && $stmtUp) $stmtUp->close();
    }
    // If anything was output (warnings/notices), discard and return a clean JSON error
    $accidentalOutput = ob_get_clean();
    if (!empty($accidentalOutput) && empty($resp['success'])) {
        $resp = ['success' => false, 'message' => 'Server error. Please reload and try again.'];
    }
    echo json_encode($resp);
    exit();
}

// WhatsApp Single Message Handler (AJAX)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'send_whatsapp') {
    // Clean output buffer
    if (ob_get_level()) ob_clean();
    ob_start();
    @ini_set('display_errors', '0');
    header('Content-Type: application/json');
    
    $resp = ['success' => false, 'message' => 'Unknown error'];
    
    $prescription_id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
    $csrf = isset($_POST['csrf_token']) ? $_POST['csrf_token'] : '';
    
    if ($csrf !== $_SESSION['csrf_token']) {
        $resp['message'] = 'Invalid token';
    } elseif ($prescription_id <= 0) {
        $resp['message'] = 'Invalid prescription ID';
    } else {
        // Get prescription data
        if ($isAdmin) {
            $sqlGet = "SELECT * FROM customer_prescriptions WHERE id=?";
            $stmtGet = $con->prepare($sqlGet);
            $stmtGet->bind_param('i', $prescription_id);
        } else {
            $sqlGet = "SELECT * FROM customer_prescriptions WHERE id=? AND user_id=?";
            $stmtGet = $con->prepare($sqlGet);
            $stmtGet->bind_param('ii', $prescription_id, $userId);
        }
        
        if ($stmtGet && $stmtGet->execute()) {
            $result = $stmtGet->get_result();
            if ($prescData = $result->fetch_assoc()) {
                // Include WhatsApp functions
                include_once('includes/whatsapp-functions.php');
                
                $mobile = $prescData['mobile'];
                $customerName = $prescData['customer_name'];
                $orderNumber = $prescData['order_number'] ?: 'ORD-' . date('Ymd') . '-' . str_pad($prescription_id, 4, '0', STR_PAD_LEFT);
                
                // Send WhatsApp message
                $whatsappResult = sendPrescriptionWhatsApp($mobile, $customerName, $orderNumber, $prescData);
                
                // Log the WhatsApp activity
                logWhatsAppActivity($mobile, $customerName, $orderNumber, $whatsappResult['success'], $whatsappResult);
                
                if ($whatsappResult['success']) {
                    $resp = ['success' => true, 'message' => 'WhatsApp message sent successfully'];
                } else {
                    $resp = ['success' => false, 'message' => 'WhatsApp failed: ' . $whatsappResult['message']];
                }
            } else {
                $resp['message'] = 'Prescription not found';
            }
            $stmtGet->close();
        } else {
            $resp['message'] = 'Database error';
        }
    }
    
    $accidentalOutput = ob_get_clean();
    if (!empty($accidentalOutput) && empty($resp['success'])) {
        $resp = ['success' => false, 'message' => 'Server error. Please reload and try again.'];
    }
    echo json_encode($resp);
    exit();
}

// WhatsApp Bulk Message Handler (AJAX)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'send_bulk_whatsapp') {
    // Clean output buffer
    if (ob_get_level()) ob_clean();
    ob_start();
    @ini_set('display_errors', '0');
    header('Content-Type: application/json');
    
    $resp = ['success' => false, 'message' => 'Unknown error', 'results' => []];
    
    $prescription_ids = isset($_POST['prescription_ids']) ? $_POST['prescription_ids'] : [];
    $csrf = isset($_POST['csrf_token']) ? $_POST['csrf_token'] : '';
    
    if ($csrf !== $_SESSION['csrf_token']) {
        $resp['message'] = 'Invalid token';
    } elseif (empty($prescription_ids) || !is_array($prescription_ids)) {
        $resp['message'] = 'No prescriptions selected';
    } else {
        // Include WhatsApp functions
        include_once('includes/whatsapp-functions.php');
        
        $successCount = 0;
        $failedCount = 0;
        $results = [];
        
        foreach ($prescription_ids as $prescription_id) {
            $prescription_id = (int)$prescription_id;
            if ($prescription_id <= 0) continue;
            
            // Get prescription data
            if ($isAdmin) {
                $sqlGet = "SELECT * FROM customer_prescriptions WHERE id=?";
                $stmtGet = $con->prepare($sqlGet);
                $stmtGet->bind_param('i', $prescription_id);
            } else {
                $sqlGet = "SELECT * FROM customer_prescriptions WHERE id=? AND user_id=?";
                $stmtGet = $con->prepare($sqlGet);
                $stmtGet->bind_param('ii', $prescription_id, $userId);
            }
            
            if ($stmtGet && $stmtGet->execute()) {
                $result = $stmtGet->get_result();
                if ($prescData = $result->fetch_assoc()) {
                    $mobile = $prescData['mobile'];
                    $customerName = $prescData['customer_name'];
                    $orderNumber = $prescData['order_number'] ?: 'ORD-' . date('Ymd') . '-' . str_pad($prescription_id, 4, '0', STR_PAD_LEFT);
                    
                    // Send WhatsApp message
                    $whatsappResult = sendPrescriptionWhatsApp($mobile, $customerName, $orderNumber, $prescData);
                    
                    // Log the WhatsApp activity
                    logWhatsAppActivity($mobile, $customerName, $orderNumber, $whatsappResult['success'], $whatsappResult);
                    
                    if ($whatsappResult['success']) {
                        $successCount++;
                        $results[] = ['id' => $prescription_id, 'customer' => $customerName, 'mobile' => $mobile, 'success' => true, 'message' => 'Sent successfully'];
                    } else {
                        $failedCount++;
                        $results[] = ['id' => $prescription_id, 'customer' => $customerName, 'mobile' => $mobile, 'success' => false, 'message' => $whatsappResult['message']];
                    }
                } else {
                    $failedCount++;
                    $results[] = ['id' => $prescription_id, 'success' => false, 'message' => 'Prescription not found'];
                }
                $stmtGet->close();
            } else {
                $failedCount++;
                $results[] = ['id' => $prescription_id, 'success' => false, 'message' => 'Database error'];
            }
        }
        
        $resp = [
            'success' => true,
            'message' => "Bulk WhatsApp completed. Success: $successCount, Failed: $failedCount",
            'results' => $results,
            'summary' => ['success' => $successCount, 'failed' => $failedCount, 'total' => count($prescription_ids)]
        ];
    }
    
    $accidentalOutput = ob_get_clean();
    if (!empty($accidentalOutput) && empty($resp['success'])) {
        $resp = ['success' => false, 'message' => 'Server error. Please reload and try again.'];
    }
    echo json_encode($resp);
    exit();
}

// Payment handler (AJAX)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'record_payment') {
    // Clean output buffer to prevent any HTML/text before JSON
    if (ob_get_level()) ob_clean();
    // Start fresh buffer to capture any accidental output and discard it later
    ob_start();
    // Hide PHP warnings/notices from bleeding into JSON
    @ini_set('display_errors', '0');
    header('Content-Type: application/json');
    $resp = ['success' => false, 'message' => 'Unknown error'];
    
    $prescription_id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
    $payment_amount = isset($_POST['payment_amount']) ? (float)$_POST['payment_amount'] : 0;
    $payment_method = isset($_POST['payment_method']) ? trim($_POST['payment_method']) : 'Cash';
    $csrf = isset($_POST['csrf_token']) ? $_POST['csrf_token'] : '';
    
    if ($csrf !== $_SESSION['csrf_token']) {
        $resp['message'] = 'Invalid token';
    } elseif ($prescription_id <= 0) {
        $resp['message'] = 'Invalid prescription ID';
    } elseif ($payment_amount <= 0) {
        $resp['message'] = 'Payment amount must be greater than 0';
    } else {
        // Get current prescription data
        if ($isAdmin) {
            $sqlGet = "SELECT advance_amount, balance_amount, total_amount, discount_amount FROM customer_prescriptions WHERE id=?";
            $stmtGet = $con->prepare($sqlGet);
            $stmtGet->bind_param('i', $prescription_id);
        } else {
            $sqlGet = "SELECT advance_amount, balance_amount, total_amount, discount_amount FROM customer_prescriptions WHERE id=? AND user_id=?";
            $stmtGet = $con->prepare($sqlGet);
            $stmtGet->bind_param('ii', $prescription_id, $userId);
        }
        
        if ($stmtGet && $stmtGet->execute()) {
            $result = $stmtGet->get_result();
            if ($prescData = $result->fetch_assoc()) {
                $current_advance = (float)$prescData['advance_amount'];
                $current_balance = (float)$prescData['balance_amount'];
                
                // Calculate new amounts (balance is GENERATED; do not update it directly)
                $new_advance = $current_advance + $payment_amount;
                
                // Update prescription with new payment (only advance_amount)
                if ($isAdmin) {
                    $sqlUpdate = "UPDATE customer_prescriptions SET advance_amount=?, updated_at=NOW() WHERE id=?";
                    $stmtUpdate = $con->prepare($sqlUpdate);
                    $stmtUpdate->bind_param('di', $new_advance, $prescription_id);
                } else {
                    $sqlUpdate = "UPDATE customer_prescriptions SET advance_amount=?, updated_at=NOW() WHERE id=? AND user_id=?";
                    $stmtUpdate = $con->prepare($sqlUpdate);
                    $stmtUpdate->bind_param('dii', $new_advance, $prescription_id, $userId);
                }
                
                if ($stmtUpdate && $stmtUpdate->execute()) {
                    if ($stmtUpdate->affected_rows > 0) {
                        // Re-fetch recalculated balance from DB
                        if ($isAdmin) {
                            $sqlRefetch = "SELECT advance_amount, balance_amount FROM customer_prescriptions WHERE id=?";
                            $stmtRefetch = $con->prepare($sqlRefetch);
                            $stmtRefetch->bind_param('i', $prescription_id);
                        } else {
                            $sqlRefetch = "SELECT advance_amount, balance_amount FROM customer_prescriptions WHERE id=? AND user_id=?";
                            $stmtRefetch = $con->prepare($sqlRefetch);
                            $stmtRefetch->bind_param('ii', $prescription_id, $userId);
                        }
                        $new_balance_val = null;
                        if ($stmtRefetch && $stmtRefetch->execute()) {
                            $rs2 = $stmtRefetch->get_result();
                            if ($row2 = $rs2->fetch_assoc()) {
                                $new_balance_val = (float)$row2['balance_amount'];
                            }
                        }
                        if (isset($stmtRefetch) && $stmtRefetch) $stmtRefetch->close();
                        
                        $resp = [
                            'success' => true,
                            'message' => 'Payment recorded successfully',
                            'new_advance' => number_format($new_advance, 2),
                            'new_balance' => number_format($new_balance_val !== null ? $new_balance_val : max(0, $current_balance - $payment_amount), 2),
                            'payment_amount' => number_format($payment_amount, 2)
                        ];
                    } else {
                        $resp['message'] = 'No changes made or not permitted';
                    }
                    $stmtUpdate->close();
                } else {
                    $resp['message'] = 'Payment update failed';
                }
            } else {
                $resp['message'] = 'Prescription not found';
            }
            $stmtGet->close();
        } else {
            $resp['message'] = 'Database error';
        }
    }
    // If anything was output (warnings/notices), discard and return a clean JSON error
    $accidentalOutput = ob_get_clean();
    if (!empty($accidentalOutput) && empty($resp['success'])) {
        $resp = ['success' => false, 'message' => 'Server error. Please reload and try again.'];
    }
    echo json_encode($resp);
    exit();
}

// Handle export functionality
if (isset($_GET['export']) && $_GET['export'] == 'excel') {
    // Set headers for Excel download
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="prescription_report_' . date('Y-m-d') . '.xls"');
    header('Pragma: no-cache');
    header('Expires: 0');
}

// Date and filter logic
// Default filter changed to today's data only
$todayDefault = date('Y-m-d');
$from = (isset($_GET['from']) && $_GET['from'] !== '') ? $_GET['from'] : $todayDefault;
$to = (isset($_GET['to']) && $_GET['to'] !== '') ? $_GET['to'] : $todayDefault;
$status_filter = isset($_GET['status']) ? $_GET['status'] : '';
$search = isset($_GET['search']) ? $_GET['search'] : '';
$frame_type_filter = isset($_GET['frame_type']) ? $_GET['frame_type'] : '';
$lens_type_filter = isset($_GET['lens_type']) ? $_GET['lens_type'] : '';

// Build query (exclude soft-deleted records and include employee names)
$query = "SELECT cp.*, e.employee_name FROM customer_prescriptions cp 
          LEFT JOIN employees e ON cp.employee_id = e.id 
          WHERE DATE(cp.created_at) BETWEEN ? AND ? AND (cp.notes NOT LIKE '%DELETED%' OR cp.notes IS NULL)";
$params = [$from, $to];
$types = "ss";

if ($status_filter && in_array($status_filter, ['Pending','In Progress','Ready','Delivered','Cancelled'])) {
    $query .= " AND order_status=?";
    $params[] = $status_filter;
    $types .= "s";
}

if ($search) {
    $query .= " AND (customer_name LIKE ? OR mobile LIKE ? OR order_number LIKE ?)";
    $search_param = "%$search%";
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
    $types .= "sss";
}

if ($frame_type_filter) {
    $query .= " AND frame_type=?";
    $params[] = $frame_type_filter;
    $types .= "s";
}

if ($lens_type_filter) {
    $query .= " AND lens_type=?";
    $params[] = $lens_type_filter;
    $types .= "s";
}

if (!$isAdmin) {
    $query .= " AND cp.user_id=?";
    $params[] = $userId;
    $types .= "i";
}

$query .= " ORDER BY cp.created_at DESC";
$stmt = $con->prepare($query);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();
$prescriptions = [];
while ($row = $result->fetch_assoc()) {
    $prescriptions[] = $row;
}
$stmt->close();

// Calculate summary statistics
$total_orders = count($prescriptions);
$total_revenue = array_sum(array_column($prescriptions, 'total_amount'));
$total_discounts = array_sum(array_column($prescriptions, 'discount_amount'));
$total_outstanding = array_sum(array_column($prescriptions, 'balance_amount'));
$total_advances = array_sum(array_column($prescriptions, 'advance_amount'));
$status_counts = array_count_values(array_column($prescriptions, 'order_status'));
$frame_type_counts = array_count_values(array_column($prescriptions, 'frame_type'));
$lens_type_counts = array_count_values(array_column($prescriptions, 'lens_type'));

// If exporting to Excel, output table and exit
if (isset($_GET['export']) && $_GET['export'] == 'excel') {
    echo '<table border="1">';
    echo '<tr>
            <th>Order Number</th>
            <th>Date</th>
            <th>Customer Name</th>
            <th>Customer Age</th>
            <th>Customer Gender</th>
            <th>Mobile</th>
            <th>Handled By</th>
            <th>Address</th>
            <th>Right Eye SPH</th>
            <th>Right Eye CYL</th>
            <th>Right Eye AXIS</th>
            <th>Right Eye ADD</th>
            <th>Left Eye SPH</th>
            <th>Left Eye CYL</th>
            <th>Left Eye AXIS</th>
            <th>Left Eye ADD</th>
            <th>Right Contact SPH</th>
            <th>Right Contact CYL</th>
            <th>Right Contact AXIS</th>
            <th>Left Contact SPH</th>
            <th>Left Contact CYL</th>
            <th>Left Contact AXIS</th>
            <th>Frame Name</th>
            <th>Lens Name</th>
            <th>Frame Type</th>
            <th>Lens Type</th>
            <th>Frame Amount</th>
            <th>Lens Amount</th>
            <th>Total Amount</th>
            <th>Discount Type</th>
            <th>Discount Value</th>
            <th>Discount Amount</th>
            <th>Final Amount</th>
            <th>Advance Amount</th>
            <th>Balance Amount</th>
            <th>Order Status</th>
            <th>Delivery Date</th>
            <th>Notes</th>
          </tr>';
    
    foreach($prescriptions as $row) {
        $final_amount = $row['total_amount'] - ($row['discount_amount'] ?: 0);
        echo '<tr>';
        echo '<td>' . htmlspecialchars($row['order_number'] ?: 'N/A') . '</td>';
        echo '<td>' . date('d-M-Y', strtotime($row['created_at'])) . '</td>';
        echo '<td>' . htmlspecialchars($row['customer_name']) . '</td>';
        echo '<td>' . ($row['customer_age'] ?: 'N/A') . '</td>';
        echo '<td>' . htmlspecialchars($row['customer_sex'] ?? 'N/A') . '</td>';
        echo '<td>' . htmlspecialchars($row['mobile']) . '</td>';
        echo '<td>' . htmlspecialchars($row['employee_name'] ?? 'Not Assigned') . '</td>';
        echo '<td>' . htmlspecialchars($row['address']) . '</td>';
        echo '<td>' . htmlspecialchars($row['right_eye_sph']) . '</td>';
        echo '<td>' . htmlspecialchars($row['right_eye_cyl']) . '</td>';
        echo '<td>' . htmlspecialchars($row['right_eye_axis']) . '</td>';
        echo '<td>' . htmlspecialchars($row['right_eye_addition'] ?: '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['left_eye_sph']) . '</td>';
        echo '<td>' . htmlspecialchars($row['left_eye_cyl']) . '</td>';
        echo '<td>' . htmlspecialchars($row['left_eye_axis']) . '</td>';
        echo '<td>' . htmlspecialchars($row['left_eye_addition'] ?: '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['right_contact_sph'] ?? '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['right_contact_cyl'] ?? '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['right_contact_axis'] ?? '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['left_contact_sph'] ?? '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['left_contact_cyl'] ?? '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['left_contact_axis'] ?? '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['frame_name']) . '</td>';
        echo '<td>' . htmlspecialchars($row['lens_name'] ?: '-') . '</td>';
        echo '<td>' . htmlspecialchars($row['frame_type']) . '</td>';
        echo '<td>' . htmlspecialchars($row['lens_type']) . '</td>';
        echo '<td>' . number_format($row['frame_amount'] ?? 0, 2) . '</td>';
        echo '<td>' . number_format($row['lens_amount'] ?? 0, 2) . '</td>';
        echo '<td>' . number_format($row['total_amount'], 2) . '</td>';
        echo '<td>' . htmlspecialchars($row['discount_type'] ?: '-') . '</td>';
        echo '<td>' . number_format($row['discount_value'] ?: 0, 2) . '</td>';
        echo '<td>' . number_format($row['discount_amount'] ?: 0, 2) . '</td>';
        echo '<td>' . number_format($final_amount, 2) . '</td>';
        echo '<td>' . number_format($row['advance_amount'], 2) . '</td>';
        echo '<td>' . number_format($row['balance_amount'], 2) . '</td>';
        echo '<td>' . htmlspecialchars($row['order_status']) . '</td>';
        echo '<td>' . ($row['delivery_date'] ? date('d-M-Y', strtotime($row['delivery_date'])) : '') . '</td>';
        echo '<td>' . htmlspecialchars($row['notes']) . '</td>';
        echo '</tr>';
    }
    echo '</table>';
    exit();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title><?php echo htmlspecialchars($shopName); ?> | Prescription Reports</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <style>
        body { background: #f8f9fc; }
        .card { border-radius: 1rem; }
        .form-control, .btn { border-radius: 0.5rem; }
        .stats-card { 
            border-left: 4px solid #4e73df; 
            background: linear-gradient(45deg, #f8f9fc, #ffffff);
        }
        .stats-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
        }
        .report-table { font-size: 13px; }
        .print-section { background: white; }
        @media print {
            .no-print { display: none !important; }
            .card { box-shadow: none !important; border: 1px solid #ddd !important; }
            body { background: white !important; }
        }
        @media (max-width: 768px) {
            .form-inline .form-control, .form-inline .btn {
                margin-bottom: 0.5rem;
                width: 100%;
            }
        }
        
        /* Quick Filter Styles */
        .quick-filter-container {
            background: #fff;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
            padding: 12px;
            margin-bottom: 16px;
        }
        
        .quick-filter-form {
            width: 100%;
        }
        
        .filter-row {
            margin-bottom: 12px;
        }
        
        .filter-group {
            display: flex;
            align-items: center;
            gap: 12px;
            flex-wrap: wrap;
        }
        
        .filter-item {
            display: flex;
            align-items: center;
            background: #f8f9fa;
            border-radius: 8px;
            padding: 6px 10px;
            gap: 6px;
            border: 1px solid #e9ecef;
            min-height: 36px;
        }
        
        .filter-icon {
            color: #6c757d;
            font-size: 14px;
        }
        
        .filter-input {
            border: none;
            background: transparent;
            outline: none;
            font-size: 13px;
            color: #495057;
            min-width: 0;
        }
        
        .filter-input::placeholder {
            color: #adb5bd;
            font-size: 12px;
        }
        
        .date-range {
            flex-grow: 1;
            min-width: 200px;
        }
        
        .date-input {
            width: 100px;
            font-size: 12px;
        }
        
        .date-separator {
            font-size: 11px;
            color: #6c757d;
            margin: 0 4px;
        }
        
        .search-box {
            flex-grow: 2;
            min-width: 180px;
        }
        
        .search-input {
            width: 100%;
            min-width: 160px;
        }
        
        .filter-actions {
            display: flex;
            gap: 6px;
        }
        
        .filter-btn {
            width: 36px;
            height: 36px;
            border-radius: 8px;
            border: none;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.2s ease;
            text-decoration: none;
            font-size: 14px;
        }
        
        .primary-btn {
            background: #007bff;
            color: white;
        }
        
        .primary-btn:hover {
            background: #0056b3;
            transform: translateY(-1px);
        }
        
        .reset-btn {
            background: #6c757d;
            color: white;
        }
        
        .reset-btn:hover {
            background: #545b62;
            color: white;
            text-decoration: none;
        }
        
        .toggle-btn {
            background: #28a745;
            color: white;
        }
        
        .toggle-btn:hover {
            background: #1e7e34;
        }
        
        .toggle-btn.active {
            background: #dc3545;
        }
        
        /* Status Pills */
        .status-pills {
            display: flex;
            gap: 6px;
            flex-wrap: wrap;
            margin-bottom: 8px;
        }
        
        .status-pill {
            display: flex;
            align-items: center;
            gap: 4px;
            padding: 4px 8px;
            border-radius: 20px;
            font-size: 11px;
            font-weight: 500;
            text-decoration: none;
            transition: all 0.2s ease;
            border: 1px solid #e9ecef;
            background: #fff;
            color: #495057;
        }
        
        .status-pill:hover {
            text-decoration: none;
            transform: translateY(-1px);
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .status-pill.active {
            background: #007bff;
            color: white;
            border-color: #007bff;
        }
        
        .status-pill.status-pending {
            border-color: #ffc107;
            color: #856404;
        }
        
        .status-pill.status-pending.active {
            background: #ffc107;
            color: #212529;
        }
        
        .status-pill.status-in-progress {
            border-color: #17a2b8;
            color: #0c5460;
        }
        
        .status-pill.status-in-progress.active {
            background: #17a2b8;
            color: white;
        }
        
        .status-pill.status-ready {
            border-color: #28a745;
            color: #155724;
        }
        
        .status-pill.status-ready.active {
            background: #28a745;
            color: white;
        }
        
        .status-pill.status-delivered {
            border-color: #6c757d;
            color: #383d41;
        }
        
        .status-pill.status-delivered.active {
            background: #6c757d;
            color: white;
        }
        
        .status-pill.status-cancelled {
            border-color: #dc3545;
            color: #721c24;
        }
        
        .status-pill.status-cancelled.active {
            background: #dc3545;
            color: white;
        }
        
        .pill-count {
            background: rgba(0,0,0,0.1);
            color: inherit;
            padding: 1px 4px;
            border-radius: 8px;
            font-size: 9px;
            font-weight: 600;
        }
        
        .status-pill.active .pill-count {
            background: rgba(255,255,255,0.2);
        }
        
        /* Date Pills */
        .date-pills {
            display: flex;
            gap: 4px;
            flex-wrap: wrap;
            margin-bottom: 8px;
            padding-top: 4px;
            border-top: 1px solid #f1f3f4;
        }
        
        .date-pill {
            display: flex;
            align-items: center;
            gap: 4px;
            padding: 3px 8px;
            border-radius: 16px;
            font-size: 10px;
            font-weight: 500;
            text-decoration: none;
            transition: all 0.2s ease;
            border: 1px solid #e1e5e9;
            background: #f8f9fa;
            color: #495057;
            white-space: nowrap;
        }
        
        .date-pill:hover {
            text-decoration: none;
            background: #e9ecef;
            transform: translateY(-1px);
            box-shadow: 0 2px 4px rgba(0,0,0,0.08);
            color: #495057;
        }
        
        .date-pill.active {
            background: #28a745;
            color: white;
            border-color: #28a745;
            font-weight: 600;
        }
        
        .date-pill i {
            font-size: 9px;
        }
        
        /* Advanced Filters */
        .advanced-filters {
            border-top: 1px solid #e9ecef;
            padding-top: 12px;
            margin-top: 8px;
        }
        
        .advanced-row {
            display: flex;
            gap: 12px;
            flex-wrap: wrap;
        }
        
        .filter-select {
            flex: 1;
            min-width: 150px;
            padding: 6px 10px;
            border: 1px solid #e9ecef;
            border-radius: 6px;
            background: #f8f9fa;
            font-size: 12px;
            color: #495057;
        }
        
        .filter-select:focus {
            outline: none;
            border-color: #007bff;
            box-shadow: 0 0 0 2px rgba(0,123,255,0.1);
        }
        
        /* Mobile Optimizations */
        @media (max-width: 768px) {
            .quick-filter-container {
                padding: 8px;
                margin: 0 -8px 12px;
                border-radius: 0;
            }
            
            .filter-group {
                flex-direction: column;
                align-items: stretch;
                gap: 8px;
            }
            
            .filter-item {
                width: 100%;
            }
            
            .date-range {
                min-width: auto;
            }
            
            .date-input {
                width: 80px;
            }
            
            .search-box {
                min-width: auto;
            }
            
            .filter-actions {
                justify-content: center;
            }
            
            .status-pills {
                gap: 4px;
            }
            
            .status-pill {
                font-size: 10px;
                padding: 3px 6px;
            }
            
            .date-pills {
                gap: 3px;
            }
            
            .date-pill {
                font-size: 9px;
                padding: 2px 6px;
            }
            
            .advanced-row {
                flex-direction: column;
                gap: 8px;
            }
            
            .filter-select {
                min-width: auto;
            }
        }
        
        /* WhatsApp Button Styling */
        .btn-whatsapp {
            background-color: #25D366;
            border-color: #25D366;
            color: white;
        }
        
        .btn-whatsapp:hover {
            background-color: #128C7E;
            border-color: #128C7E;
            color: white;
        }
        
        .btn-whatsapp:focus {
            box-shadow: 0 0 0 0.2rem rgba(37, 211, 102, 0.5);
        }

        /* Bulk Actions Styling */
        .bulk-actions {
            display: flex;
            align-items: center;
            gap: 8px;
            background: #e3f2fd;
            padding: 8px 12px;
            border-radius: 8px;
            border: 1px solid #2196f3;
        }
        
        .selected-info {
            font-size: 12px;
            font-weight: 600;
            color: #1976d2;
            background: white;
            padding: 4px 8px;
            border-radius: 4px;
        }
        
        .bulk-checkbox {
            width: 18px;
            height: 18px;
            cursor: pointer;
        }
        
        .bulk-checkbox:checked {
            accent-color: #25D366;
        }
        
        .bulk-selected {
            background-color: #f0f8ff !important;
            border-left: 4px solid #25D366 !important;
        }
        
        .mobile-checkbox-wrapper {
            position: absolute;
            top: 8px;
            left: 8px;
            z-index: 10;
            background: rgba(255, 255, 255, 0.9);
            border-radius: 50%;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .mobile-checkbox {
            width: 20px;
            height: 20px;
            cursor: pointer;
        }
        
        .bulk-mode-only {
            transition: all 0.3s ease;
        }
        
        .bulk-results {
            font-size: 13px;
        }
        
        .result-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 6px 8px;
            margin-bottom: 4px;
            border-radius: 4px;
            background: #f8f9fa;
            border-left: 3px solid #dee2e6;
        }
        
        .result-item.success {
            background: #d4edda;
            border-left-color: #28a745;
        }
        
        .result-item.error {
            background: #f8d7da;
            border-left-color: #dc3545;
        }
        
        .result-item.processing {
            background: #fff3cd;
            border-left-color: #ffc107;
        }
        
        .customer-details {
            flex: 1;
        }
        
        .customer-name {
            font-weight: 600;
            color: #333;
        }
        
        .customer-mobile {
            font-size: 11px;
            color: #666;
        }
        
        .result-status {
            font-size: 12px;
            font-weight: 500;
        }
        
        .result-status.success {
            color: #155724;
        }
        
        .result-status.error {
            color: #721c24;
        }
        
        .result-status.processing {
            color: #856404;
        }
        
        @media (max-width: 768px) {
            .page-header-wrapper {
                flex-direction: column !important;
                align-items: flex-start !important;
            }
            
            .page-header-actions {
                width: 100%;
                margin-top: 12px;
            }
            
            .bulk-actions,
            .regular-actions {
                width: 100%;
                flex-wrap: wrap;
                gap: 6px;
            }
            
            .bulk-actions .btn,
            .regular-actions .btn {
                flex: 1 1 calc(50% - 3px);
                min-width: 120px;
                font-size: 12px;
                justify-content: center;
            }
            
            .selected-info {
                order: -1;
                flex: 1 1 100%;
                text-align: center;
                margin-bottom: 6px;
            }
        }
        
        /* 4 Inch Receipt Styles */
        .receipt-print {
            display: none;
        }
        
        @media print {
            .receipt-print-active .no-print,
            .receipt-print-active .main-content {
                display: none !important;
            }
            
            .receipt-print-active .receipt-print {
                display: block !important;
            }
            
            .receipt-print-active body {
                margin: 0 !important;
                padding: 0 !important;
                background: white !important;
                font-family: 'Courier New', monospace !important;
                font-size: 12px !important;
                line-height: 1.2 !important;
            }
            
            .receipt-print {
                width: 4in !important;
                max-width: 4in !important;
                margin: 0 !important;
                padding: 8px !important;
                background: white !important;
                color: black !important;
            }
            
            .receipt-header {
                text-align: center;
                border-bottom: 1px dashed #000;
                padding-bottom: 8px;
                margin-bottom: 8px;
            }
            
            .receipt-title {
                font-size: 16px !important;
                font-weight: bold !important;
                margin: 0 0 4px 0 !important;
            }
            
            .receipt-subtitle {
                font-size: 10px !important;
                margin: 0 !important;
            }
            
            .receipt-section {
                margin-bottom: 8px;
                border-bottom: 1px dashed #000;
                padding-bottom: 6px;
            }
            
            .receipt-row {
                display: flex;
                justify-content: space-between;
                margin-bottom: 2px;
                font-size: 11px;
            }
            
            .receipt-summary {
                font-weight: bold;
                text-align: center;
                margin: 8px 0;
            }
            
            .receipt-footer {
                text-align: center;
                font-size: 10px;
                margin-top: 8px;
                border-top: 1px dashed #000;
                padding-top: 6px;
            }
            
            .receipt-table {
                width: 100%;
                font-size: 10px;
                border-collapse: collapse;
            }
            
            .receipt-table td {
                padding: 2px 0;
                vertical-align: top;
            }
            
            .receipt-label {
                font-weight: bold;
                width: 35%;
            }
            
            .receipt-prescription {
                font-size: 9px;
                margin: 4px 0;
            }
        }
        
        /* Mobile Grid Styles - Amazon E-commerce Style */
        .mobile-grid { display: none; }
        .desktop-table { display: block; }
        
        .mobile-grid-container {
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
            margin: 0 -4px;
        }
        
        .mobile-card {
            width: calc(50% - 4px);
            background: #fff;
            border: 1px solid #e7e7e7;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.08);
            transition: all 0.2s ease;
            position: relative;
            overflow: hidden;
        }
        
        .mobile-card:hover {
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            transform: translateY(-1px);
        }
        
        .card-header-mobile {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            padding: 8px 10px 6px;
            background: #f8f9fa;
            border-bottom: 1px solid #e9ecef;
        }
        
        .order-info .order-number {
            font-size: 11px;
            font-weight: 600;
            color: #0066cc;
            line-height: 1.2;
        }
        
        .order-info .order-date {
            font-size: 9px;
            color: #666;
            margin-top: 1px;
        }
        
        .status-badge {
            font-size: 11px; /* increased */
            padding: 4px 8px;
            border-radius: 4px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.3px;
            cursor: pointer;
        }
        
        .status-pending { background: #fff3cd; color: #856404; }
        .status-in-progress { background: #d1ecf1; color: #0c5460; }
        .status-ready { background: #d4edda; color: #155724; }
        .status-delivered { background: #e2e3e5; color: #383d41; }
        .status-cancelled { background: #f8d7da; color: #721c24; }
        
        .card-body-mobile {
            padding: 8px 10px;
        }
        
        .customer-info {
            margin-bottom: 6px;
        }
        
        .customer-name {
            font-size: 14px; /* increased */
            font-weight: 600;
            color: #333;
            line-height: 1.3;
            margin-bottom: 2px;
        }
        
        .customer-mobile {
            font-size: 12px; /* increased */
            color: #666;
        }
        
        .product-info {
            margin-bottom: 6px;
        }
        
        .frame-name, .lens-name {
            font-size: 9px;
            color: #555;
            line-height: 1.3;
            margin-bottom: 1px;
        }
        
        .amount-section {
            background: #f8f9fa;
            margin: 0 -10px -8px;
            padding: 6px 10px 8px;
            border-top: 1px solid #e9ecef;
        }
        
        .price-row {
            display: flex;
            align-items: center;
            gap: 6px;
            margin-bottom: 3px;
            justify-content: flex-end; /* ensure content sits at right */
        }
        
        .total-amount {
            font-size: 18px; /* increased */
            font-weight: 700;
            color: #B12704;
            margin-left: auto; /* push to right edge */
            text-align: right;
        }
        
        .discount {
            font-size: 10px;
            color: #007600;
            font-weight: 500;
        }
        
        .balance-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .balance-label {
            font-size: 12px; /* increased */
            color: #666;
        }
        
        .balance-amount {
            font-size: 14px; /* increased */
            font-weight: 600;
            text-align: right;
            min-width: 70px;
        }
        
        .balance-amount.pending { color: #dc3545; }
        .balance-amount.paid { color: #28a745; }
        
        .card-actions {
            position: absolute;
            bottom: 4px;
            right: 4px;
            display: flex;
            gap: 6px; /* increased spacing between action icons */
            background: rgba(255,255,255,0.9);
            padding: 2px;
            border-radius: 4px;
            backdrop-filter: blur(4px);
        }
        
        .action-btn {
            width: 24px;
            height: 24px;
            border-radius: 4px;
            border: none;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 10px;
            cursor: pointer;
            transition: all 0.2s ease;
            text-decoration: none;
        }
        
        .view-btn { background: #17a2b8; color: white; }
        .edit-btn { background: #28a745; color: white; }
        .whatsapp-btn { background: #25D366; color: white; }
        .delete-btn { background: #dc3545; color: white; }
        
        .action-btn:hover {
            transform: scale(1.1);
            text-decoration: none;
            color: white;
        }
        
        .empty-state {
            width: 100%;
            text-align: center;
            padding: 40px 20px;
            color: #6c757d;
        }
        
        /* Responsive Layout Control */
        @media (max-width: 768px) {
            .desktop-table { display: none !important; }
            .mobile-grid { display: block !important; }
            
            .report-container {
                display: flex;
                flex-direction: column;
            }
            
            .detail-section { order: 1; }
            .summary-section { order: 2; }
            
            .mobile-card {
                width: calc(50% - 4px);
            }
            
            .form-inline .form-control, 
            .form-inline .btn {
                margin-bottom: 0.5rem;
                width: 100%;
            }
            
            .card-header h6 {
                font-size: 14px;
            }
            
            .stats-card .h5 {
                font-size: 18px;
            }
            
            .stats-card .text-xs {
                font-size: 10px;
            }
        }
        
        @media (max-width: 480px) {
            .mobile-card {
                width: 100%; /* full width on very small screens */
            }
            
            .container-fluid {
                padding-left: 8px;
                padding-right: 8px;
            }
        }

        /* Fix: ensure mobile action buttons do not cover balance/amount section */
        @media (max-width: 768px) {
            .mobile-card {
                padding-bottom: 40px; /* space for action buttons */
            }
            .mobile-card .amount-section {

        /* Added: clearer status badge readability & desktop action button spacing */
        .status-badge, .report-table .badge {
            font-size: 13px !important;
            font-weight: 600;
            letter-spacing: .2px;
        }
        .report-table .btn-group.btn-group-sm .btn { margin-right: 6px; }
        .report-table .btn-group.btn-group-sm .btn:last-child { margin-right: 0; }

        /* Responsive header action buttons */
        .page-header-wrapper { }
        .page-header-actions { display:flex; gap:8px; flex-wrap:wrap; }
        .page-header-actions .btn { display:flex; align-items:center; gap:4px; }
        @media (max-width: 768px){
            .page-header-wrapper { flex-direction:column !important; align-items:flex-start !important; }
            .page-header-actions { width:100%; padding-top:6px; }
            .page-header-actions .btn { flex:1 1 calc(50% - 8px); font-size:12px; justify-content:center; }
            .quick-filter-container { padding-top:4px; }
            .filter-row { flex-direction:column; }
            .filter-group { width:100%; }
            .filter-item.date-range { width:100%; }
            .filter-actions { order:3; }
        }
                margin: 0 -10px 0 !important; /* remove negative bottom margin that caused overlap */
            }
            .mobile-card .card-actions {
                bottom: 6px;
                right: 6px;
                box-shadow: 0 2px 4px rgba(0,0,0,0.15);
            }
        }
    </style>
</head>
<body id="page-top">
<div id="wrapper">
<?php include_once('includes/sidebar.php');?>
<div id="content-wrapper" class="d-flex flex-column">
<div id="content">
<?php include_once('includes/topbar.php');?>
<div class="container-fluid main-content">

    <!-- Page Heading -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4 no-print page-header-wrapper">
        <h1 class="h3 mb-0 text-gray-800">
            <i class="fas fa-chart-bar text-primary"></i> Prescription Reports
        </h1>
        <div class="page-header-actions">
            <!-- Bulk WhatsApp Actions (Initially Hidden) -->
            <div id="bulkActions" class="bulk-actions" style="display: none;">
                <div class="selected-info">
                    <span id="selectedCount">0</span> selected
                </div>
                <button onclick="sendBulkWhatsApp()" class="btn btn-whatsapp btn-sm" id="bulkWhatsAppBtn">
                    <i class="fab fa-whatsapp"></i> Send WhatsApp (<span id="bulkCount">0</span>)
                </button>
                <button onclick="clearSelection()" class="btn btn-outline-secondary btn-sm">
                    <i class="fas fa-times"></i> Clear
                </button>
            </div>
            
            <!-- Regular Actions -->
            <div id="regularActions" class="regular-actions">
                <button onclick="toggleBulkMode()" class="btn btn-primary btn-sm" id="bulkModeToggle">
                    <i class="fas fa-check-square"></i> Bulk Select
                </button>
                <a href="customer-prescriptions.php" class="btn btn-secondary btn-sm">
                    <i class="fas fa-arrow-left"></i> Back
                </a>
                <button onclick="window.print()" class="btn btn-info btn-sm">
                    <i class="fas fa-print"></i> Print
                </button>
                <button onclick="printReceipt()" class="btn btn-warning btn-sm">
                    <i class="fas fa-receipt"></i> Print 4" Receipt
                </button>
                <a href="?<?php echo http_build_query(array_merge($_GET, ['export' => 'excel'])); ?>" class="btn btn-success btn-sm">
                    <i class="fas fa-file-excel"></i> Excel
                </a>
            </div>
        </div>
    </div>

    <!-- Bulk Progress Modal -->
    <div class="modal fade" id="bulkProgressModal" tabindex="-1" role="dialog" aria-hidden="true" data-backdrop="static">
        <div class="modal-dialog modal-dialog-centered" role="document">
            <div class="modal-content">
                <div class="modal-header bg-success text-white py-2">
                    <h6 class="modal-title"><i class="fab fa-whatsapp mr-2"></i>Sending Bulk WhatsApp Messages</h6>
                </div>
                <div class="modal-body">
                    <div class="text-center mb-3">
                        <div class="progress mb-2">
                            <div id="bulkProgress" class="progress-bar bg-success" style="width: 0%"></div>
                        </div>
                        <div id="bulkProgressText">Preparing to send messages...</div>
                        <small class="text-muted" id="bulkProgressDetail">Please wait while we send messages with 15-second intervals.</small>
                    </div>
                    <div id="bulkResults" class="bulk-results" style="max-height: 300px; overflow-y: auto;">
                        <!-- Results will be populated here -->
                    </div>
                </div>
                <div class="modal-footer py-2">
                    <button type="button" class="btn btn-secondary" id="bulkProgressClose" onclick="closeBulkProgress()" disabled>Close</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Quick Filter Bar -->
    <div class="quick-filter-container no-print mb-3">
        <form method="get" action="" class="quick-filter-form">
            <!-- Main Filter Row -->
            <div class="filter-row">
                <div class="filter-group">
                    <div class="filter-item date-range">
                        <i class="fas fa-calendar-alt filter-icon"></i>
                        <input type="date" name="from" class="filter-input date-input" value="<?php echo htmlspecialchars($from); ?>" title="From Date">
                        <span class="date-separator">to</span>
                        <input type="date" name="to" class="filter-input date-input" value="<?php echo htmlspecialchars($to); ?>" title="To Date">
                    </div>
                    
                    <div class="filter-item search-box">
                        <i class="fas fa-search filter-icon"></i>
                        <input type="text" name="search" class="filter-input search-input" placeholder="Search customer, mobile, order..." value="<?php echo htmlspecialchars($search); ?>">
                    </div>
                    
                    <div class="filter-actions">
                        <button type="submit" class="filter-btn primary-btn">
                            <i class="fas fa-filter"></i>
                        </button>
                        <a href="prescription-reports.php" class="filter-btn reset-btn" title="Reset">
                            <i class="fas fa-undo"></i>
                        </a>
                        <button type="button" class="filter-btn toggle-btn" onclick="toggleAdvancedFilters()" title="More Filters">
                            <i class="fas fa-sliders-h"></i>
                        </button>
                    </div>
                </div>
            </div>
            
            <!-- Quick Status Filters -->
            <div class="status-pills">
                <a href="?<?php echo http_build_query(array_merge($_GET, ['status' => ''])); ?>" 
                   class="status-pill <?php echo empty($status_filter) ? 'active' : ''; ?>">
                    All <span class="pill-count"><?php echo $total_orders; ?></span>
                </a>
                <?php foreach(['Pending', 'In Progress', 'Ready', 'Delivered', 'Cancelled'] as $status): 
                    $count = isset($status_counts[$status]) ? $status_counts[$status] : 0;
                    if($count > 0 || $status_filter === $status):
                ?>
                <a href="?<?php echo http_build_query(array_merge($_GET, ['status' => $status])); ?>" 
                   class="status-pill status-<?php echo strtolower(str_replace(' ', '-', $status)); ?> <?php echo $status_filter === $status ? 'active' : ''; ?>">
                    <?php echo $status; ?> <span class="pill-count"><?php echo $count; ?></span>
                </a>
                <?php endif; endforeach; ?>
            </div>
            
            <!-- Quick Date Range Filters -->
            <div class="date-pills">
                <?php
                $today = date('Y-m-d');
                $yesterday = date('Y-m-d', strtotime('-1 day'));
                $thisWeekStart = date('Y-m-d', strtotime('monday this week'));
                $thisMonthStart = date('Y-m-01');
                $lastWeekStart = date('Y-m-d', strtotime('monday last week'));
                $lastWeekEnd = date('Y-m-d', strtotime('sunday last week'));
                $lastMonthStart = date('Y-m-01', strtotime('first day of last month'));
                $lastMonthEnd = date('Y-m-t', strtotime('last day of last month'));
                
                $dateRanges = [
                    'Today' => ['from' => $today, 'to' => $today],
                    'Yesterday' => ['from' => $yesterday, 'to' => $yesterday],
                    'This Week' => ['from' => $thisWeekStart, 'to' => $today],
                    'This Month' => ['from' => $thisMonthStart, 'to' => $today],
                    'Last Week' => ['from' => $lastWeekStart, 'to' => $lastWeekEnd],
                    'Last Month' => ['from' => $lastMonthStart, 'to' => $lastMonthEnd],
                    'Last 7 Days' => ['from' => date('Y-m-d', strtotime('-7 days')), 'to' => $today],
                    'Last 30 Days' => ['from' => date('Y-m-d', strtotime('-30 days')), 'to' => $today]
                ];
                
                foreach($dateRanges as $label => $range):
                    $isActive = ($from === $range['from'] && $to === $range['to']);
                    $queryParams = array_merge($_GET, ['from' => $range['from'], 'to' => $range['to']]);
                ?>
                <a href="?<?php echo http_build_query($queryParams); ?>" 
                   class="date-pill <?php echo $isActive ? 'active' : ''; ?>">
                    <i class="fas fa-calendar-day"></i>
                    <?php echo $label; ?>
                </a>
                <?php endforeach; ?>
            </div>
            
            <!-- Advanced Filters (Collapsible) -->
            <div class="advanced-filters" id="advancedFilters" style="display: none;">
                <div class="advanced-row">
                    <select name="frame_type" class="filter-select">
                        <option value="">All Frame Types</option>
                        <option value="Metal Frame" <?php if($frame_type_filter=='Metal Frame') echo 'selected'; ?>>Metal Frame</option>
                        <option value="Cell Frame" <?php if($frame_type_filter=='Cell Frame') echo 'selected'; ?>>Cell Frame</option>
                        <option value="Supra Frame" <?php if($frame_type_filter=='Supra Frame') echo 'selected'; ?>>Supra Frame</option>
                        <option value="Sunglass" <?php if($frame_type_filter=='Sunglass') echo 'selected'; ?>>Sunglass</option>
                        <option value="Rimless" <?php if($frame_type_filter=='Rimless') echo 'selected'; ?>>Rimless</option>
                        <option value="Semi-Rimless" <?php if($frame_type_filter=='Semi-Rimless') echo 'selected'; ?>>Semi-Rimless</option>
                        <option value="Others" <?php if($frame_type_filter=='Others') echo 'selected'; ?>>Others</option>
                    </select>
                    
                    <select name="lens_type" class="filter-select">
                        <option value="">All Lens Types</option>
                        <option value="Single Vision" <?php if($lens_type_filter=='Single Vision') echo 'selected'; ?>>Single Vision</option>
                        <option value="Bifocal" <?php if($lens_type_filter=='Bifocal') echo 'selected'; ?>>Bifocal</option>
                        <option value="Progressive" <?php if($lens_type_filter=='Progressive') echo 'selected'; ?>>Progressive</option>
                        <option value="Anti-Reflective" <?php if($lens_type_filter=='Anti-Reflective') echo 'selected'; ?>>Anti-Reflective</option>
                        <option value="Blue Light Filter" <?php if($lens_type_filter=='Blue Light Filter') echo 'selected'; ?>>Blue Light Filter</option>
                        <option value="Contact Lens" <?php if($lens_type_filter=='Contact Lens') echo 'selected'; ?>>Contact Lens</option>
                    </select>
                </div>
            </div>
        </form>
    </div>

    <div class="report-container">
    <!-- Summary Statistics -->
    <div class="row mb-4 print-section summary-section">
        <div class="col-12">
            <div class="card shadow">
                <div class="card-header bg-info text-white">
                    <h6 class="m-0 font-weight-bold">
                        <i class="fas fa-chart-pie"></i> Report Summary 
                        <small>(<?php echo date('d-M-Y', strtotime($from)); ?> to <?php echo date('d-M-Y', strtotime($to)); ?>)</small>
                    </h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-xl-3 col-md-6 mb-3">
                            <div class="card stats-card border-left-primary h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">Total Orders</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo number_format($total_orders); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-glasses fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-3 col-md-6 mb-3">
                            <div class="card stats-card border-left-success h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">Total Revenue</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($total_revenue, 2); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-rupee-sign fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-3 col-md-6 mb-3">
                            <div class="card stats-card border-left-warning h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">Total Discounts</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($total_discounts, 2); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-percentage fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-3 col-md-6 mb-3">
                            <div class="card stats-card border-left-danger h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">Outstanding</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($total_outstanding, 2); ?></div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-exclamation-triangle fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-3 col-md-6 mb-3">
                            <div class="card stats-card border-left-info h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">Net Revenue</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($total_revenue - $total_discounts, 2); ?></div>
                                            <div class="text-xs text-muted">After Discounts</div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-calculator fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-xl-3 col-md-6 mb-3">
                            <div class="card stats-card border-left-secondary h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-secondary text-uppercase mb-1">Total Advances</div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">₹<?php echo number_format($total_advances, 2); ?></div>
                                            <div class="text-xs text-muted">Collected</div>
                                        </div>
                                        <div class="col-auto">
                                            <i class="fas fa-hand-holding-usd fa-2x text-gray-300"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Status Breakdown (clickable) -->
                    <div class="row mt-3">
                        <div class="col-12">
                            <h6 class="font-weight-bold text-primary mb-3">Order Status Breakdown</h6>
                            <div class="row">
                                <?php foreach(['Pending', 'In Progress', 'Ready', 'Delivered', 'Cancelled'] as $status): 
                                    $count = isset($status_counts[$status]) ? $status_counts[$status] : 0;
                                    $percentage = $total_orders > 0 ? round(($count / $total_orders) * 100, 1) : 0;
                                    $linkParams = array_merge($_GET, ['status' => $status]);
                                ?>
                                <div class="col-md-2 mb-2">
                                    <a href="?<?php echo http_build_query($linkParams); ?>" class="text-center d-block p-2 border rounded small" style="text-decoration:none;">
                                        <div class="h6 font-weight-bold mb-0"><?php echo $count; ?></div>
                                        <div class="small <?php echo ($status_filter === $status) ? 'font-weight-bold text-primary' : 'text-muted'; ?>"><?php echo $status; ?></div>
                                        <div class="small text-info"><?php echo $percentage; ?>%</div>
                                    </a>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Detailed Report Table -->
    <div class="row print-section detail-section">
        <div class="col-12">
            <div class="card shadow">
                <div class="card-header bg-dark text-white">
                    <h6 class="m-0 font-weight-bold">
                        <i class="fas fa-table"></i> Detailed Prescription Report
                        <span class="badge badge-light ml-2"><?php echo count($prescriptions); ?> Records</span>
                    </h6>
                </div>
                <div class="card-body p-0">
                    <?php if(isset($success_msg)): ?>
                        <div class="alert alert-success m-2 py-2 px-3 small"><i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($success_msg); ?></div>
                    <?php endif; ?>
                    <?php if(isset($error_msg)): ?>
                        <div class="alert alert-danger m-2 py-2 px-3 small"><i class="fas fa-exclamation-triangle"></i> <?php echo htmlspecialchars($error_msg); ?></div>
                    <?php endif; ?>
                    
                    <!-- Desktop Table -->
                    <div class="table-responsive desktop-table">
                        <table class="table table-bordered table-striped mb-0 report-table">
                            <thead class="thead-dark">
                                <tr>
                                    <th class="no-print bulk-mode-only" style="display: none;">
                                        <input type="checkbox" class="bulk-checkbox" id="selectAll" onchange="toggleAllSelection()">
                                    </th>
                                    <th>Order #</th>
                                    <th>Date</th>
                                    <th>Customer</th>
                                    <th>Age/Gender</th>
                                    <th>Mobile</th>
                                    <th>Staff</th>
                                    <th>Frame & Lens</th>
                                    <th>Right Eye</th>
                                    <th>Left Eye</th>
                                    <th>Amount</th>
                                    <th>Discount</th>
                                    <th>Advance</th>
                                    <th>Balance</th>
                                    <th>Status</th>
                                    <th class="no-print">Action</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach($prescriptions as $row): ?>
                                <tr data-prescription-id="<?php echo $row['id']; ?>">
                                    <td class="no-print bulk-mode-only" style="display: none;">
                                        <input type="checkbox" class="bulk-checkbox prescription-checkbox" value="<?php echo $row['id']; ?>" 
                                               data-customer="<?php echo htmlspecialchars($row['customer_name']); ?>" 
                                               data-mobile="<?php echo htmlspecialchars($row['mobile']); ?>" 
                                               onchange="updateSelection()">
                                    </td>
                                    <td>
                                        <?php if($row['order_number']): ?>
                                            <small class="text-primary font-weight-bold"><?php echo htmlspecialchars($row['order_number']); ?></small>
                                        <?php else: ?>
                                            <small class="text-muted">N/A</small>
                                        <?php endif; ?>
                                        <?php 
                                            if(!empty($row['notes'])){
                                                $cleanNote = strip_tags($row['notes']);
                                                if(strlen($cleanNote) > 40){ $cleanNote = substr($cleanNote,0,37).'...'; }
                                                echo '<br><small class="text-muted" title="'.htmlspecialchars($row['notes']).'">'.htmlspecialchars($cleanNote).'</small>';
                                            }
                                        ?>
                                    </td>
                                    <td>
                                        <?php echo date('d-M-Y', strtotime($row['created_at'])); ?>
                                        <?php if($row['delivery_date']): ?>
                                            <br><small class="text-info">Del: <?php echo date('d-M-Y', strtotime($row['delivery_date'])); ?></small>
                                        <?php endif; ?>
                                    </td>
                                    <td><strong><?php echo htmlspecialchars($row['customer_name']); ?></strong></td>
                                    <td>
                                        <?php if($row['customer_age']): ?>
                                            <small class="text-primary"><?php echo $row['customer_age']; ?> yrs</small>
                                        <?php else: ?>
                                            <small class="text-muted">N/A</small>
                                        <?php endif; ?>
                                        <br>
                                        <small class="text-info">
                                            <i class="fas fa-venus-mars"></i>
                                            <?php echo htmlspecialchars($row['customer_sex'] ?? 'N/A'); ?>
                                        </small>
                                    </td>
                                    <td><?php echo htmlspecialchars($row['mobile']); ?></td>
                                    <td>
                                        <small class="text-muted">
                                            <i class="fas fa-user-tie"></i>
                                            <?php echo htmlspecialchars($row['employee_name'] ?? 'Not Assigned'); ?>
                                        </small>
                                    </td>
                                    <td>
                                        <strong><?php echo htmlspecialchars($row['frame_name']); ?></strong>
                                        <?php if($row['lens_name']): ?>
                                        <br><small class="text-info"><?php echo htmlspecialchars($row['lens_name']); ?></small>
                                        <?php endif; ?>
                                        <br><small><?php echo htmlspecialchars($row['frame_type'] . ' | ' . $row['lens_type']); ?></small>
                                        <?php if($row['frame_amount'] > 0 || $row['lens_amount'] > 0): ?>
                                        <br><small class="text-success">
                                            Frame: ₹<?php echo number_format($row['frame_amount'], 2); ?> |
                                            Lens: ₹<?php echo number_format($row['lens_amount'], 2); ?>
                                        </small>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <small>
                                            SPH: <?php echo htmlspecialchars($row['right_eye_sph'] ?: '-'); ?><br>
                                            CYL: <?php echo htmlspecialchars($row['right_eye_cyl'] ?: '-'); ?><br>
                                            AXIS: <?php echo htmlspecialchars($row['right_eye_axis'] ?: '-'); ?>
                                            <?php if($row['right_eye_addition']): ?>
                                            <br>ADD: <?php echo htmlspecialchars($row['right_eye_addition']); ?>
                                            <?php endif; ?>
                                            <?php if($row['lens_type'] == 'Contact Lens' && ($row['right_contact_sph'] || $row['right_contact_cyl'] || $row['right_contact_axis'])): ?>
                                            <br><strong class="text-success">Contact:</strong>
                                            <br>SPH: <?php echo htmlspecialchars($row['right_contact_sph'] ?: '-'); ?>
                                            <br>CYL: <?php echo htmlspecialchars($row['right_contact_cyl'] ?: '-'); ?>
                                            <br>AXIS: <?php echo htmlspecialchars($row['right_contact_axis'] ?: '-'); ?>
                                            <?php endif; ?>
                                        </small>
                                    </td>
                                    <td>
                                        <small>
                                            SPH: <?php echo htmlspecialchars($row['left_eye_sph'] ?: '-'); ?><br>
                                            CYL: <?php echo htmlspecialchars($row['left_eye_cyl'] ?: '-'); ?><br>
                                            AXIS: <?php echo htmlspecialchars($row['left_eye_axis'] ?: '-'); ?>
                                            <?php if($row['left_eye_addition']): ?>
                                            <br>ADD: <?php echo htmlspecialchars($row['left_eye_addition']); ?>
                                            <?php endif; ?>
                                            <?php if($row['lens_type'] == 'Contact Lens' && ($row['left_contact_sph'] || $row['left_contact_cyl'] || $row['left_contact_axis'])): ?>
                                            <br><strong class="text-success">Contact:</strong>
                                            <br>SPH: <?php echo htmlspecialchars($row['left_contact_sph'] ?: '-'); ?>
                                            <br>CYL: <?php echo htmlspecialchars($row['left_contact_cyl'] ?: '-'); ?>
                                            <br>AXIS: <?php echo htmlspecialchars($row['left_contact_axis'] ?: '-'); ?>
                                            <?php endif; ?>
                                        </small>
                                    </td>
                                    <td>₹<?php echo number_format($row['total_amount'], 2); ?></td>
                                    <td>
                                        <?php if($row['discount_amount'] > 0): ?>
                                        <small class="text-success">
                                            <?php echo ucfirst($row['discount_type']); ?>: 
                                            <?php if($row['discount_type'] == 'percentage'): ?>
                                                <?php echo number_format($row['discount_value'], 1); ?>%
                                            <?php else: ?>
                                                ₹<?php echo number_format($row['discount_value'], 2); ?>
                                            <?php endif; ?>
                                            <br>Amount: ₹<?php echo number_format($row['discount_amount'], 2); ?>
                                        </small>
                                        <?php else: ?>
                                        <small class="text-muted">No Discount</small>
                                        <?php endif; ?>
                                    </td>
                                    <td>₹<?php echo number_format($row['advance_amount'], 2); ?></td>
                                    <td class="<?php echo $row['balance_amount'] > 0 ? 'text-danger' : 'text-success'; ?>">
                                        ₹<?php echo number_format($row['balance_amount'], 2); ?>
                                    </td>
                                    <td>
                                        <span class="badge badge-<?php 
                                            if($row['order_status'] == 'Pending') echo 'warning';
                                            else if($row['order_status'] == 'In Progress') echo 'info';
                                            else if($row['order_status'] == 'Ready') echo 'success';
                                            else if($row['order_status'] == 'Delivered') echo 'secondary';
                                            else echo 'danger';
                                        ?>">
                                            <?php echo htmlspecialchars($row['order_status']); ?>
                                        </span>
                                    </td>
                                    <td class="no-print">
                                        <div class="btn-group btn-group-sm" role="group">
                                            <a href="customer-prescription-details.php?id=<?php echo $row['id']; ?>" class="btn btn-info" title="View"><i class="fas fa-eye"></i></a>
                                            <a href="customer-prescriptions.php?edit=<?php echo $row['id']; ?>" class="btn btn-success" title="Edit"><i class="fas fa-edit"></i></a>
                                            <?php if($row['balance_amount'] > 0): ?>
                                            <button onclick="openPaymentModal(<?php echo $row['id']; ?>, '<?php echo htmlspecialchars(addslashes($row['customer_name'])); ?>', <?php echo $row['balance_amount']; ?>)" class="btn btn-warning" title="Record Payment"><i class="fas fa-rupee-sign"></i></button>
                                            <?php endif; ?>
                                            <button onclick="sendPrescriptionWhatsApp('<?php echo $row['mobile']; ?>', '<?php echo htmlspecialchars(addslashes($row['customer_name'])); ?>', '<?php echo $row['order_number']; ?>', '<?php echo htmlspecialchars($row['order_status']); ?>')" class="btn btn-whatsapp" title="WhatsApp"><i class="fab fa-whatsapp"></i></button>
                                            <form method="post" onsubmit="return confirmDelete();" style="display:inline-block;">
                                                <input type="hidden" name="delete_id" value="<?php echo $row['id']; ?>">
                                                <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                                                <button type="submit" class="btn btn-danger" title="Delete"><i class="fas fa-trash"></i></button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                                
                                <?php if(empty($prescriptions)): ?>
                                <tr>
                                    <td colspan="12" class="text-center py-4">
                                        <i class="fas fa-search fa-2x text-muted mb-2"></i><br>
                                        <h6 class="text-muted">No prescriptions found</h6>
                                        <small class="text-muted">Try adjusting your search criteria</small>
                                    </td>
                                </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Mobile Grid -->
                    <div class="mobile-grid p-2">
                        <div class="mobile-grid-container">
                            <?php foreach($prescriptions as $row): ?>
                            <?php 
                                $final_amount = (float)$row['total_amount'] - (float)($row['discount_amount'] ?? 0); 
                                $statusClass = 'status-' . strtolower(str_replace(' ', '-', $row['order_status']));
                            ?>
                            <div class="mobile-card" data-id="<?php echo (int)$row['id']; ?>" data-prescription-id="<?php echo $row['id']; ?>">
                                <div class="bulk-mode-only mobile-checkbox-wrapper" style="display: none;">
                                    <input type="checkbox" class="bulk-checkbox prescription-checkbox mobile-checkbox" value="<?php echo $row['id']; ?>" 
                                           data-customer="<?php echo htmlspecialchars($row['customer_name']); ?>" 
                                           data-mobile="<?php echo htmlspecialchars($row['mobile']); ?>" 
                                           onchange="updateSelection()">
                                    <label for="mobile_checkbox_<?php echo $row['id']; ?>"></label>
                                </div>
                                <div class="card-header-mobile">
                                    <div class="order-info">
                                        <div class="order-number">#<?php echo htmlspecialchars($row['order_number'] ?: 'N/A'); ?>
                                            <?php 
                                                if(!empty($row['notes'])){
                                                    $cleanNote = strip_tags($row['notes']);
                                                    if(strlen($cleanNote) > 32){ $cleanNote = substr($cleanNote,0,29).'...'; }
                                                    echo '<br><small class="text-muted">'.htmlspecialchars($cleanNote).'</small>';
                                                }
                                            ?>
                                        </div>
                                        <div class="order-date">
                                            <?php echo date('d M', strtotime($row['created_at'])); ?>
                                            <?php if($row['delivery_date']): ?>
                                                <br><span class="text-info">Del <?php echo date('d M', strtotime($row['delivery_date'])); ?></span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <span class="status-badge <?php echo $statusClass; ?> js-status-open" data-id="<?php echo (int)$row['id']; ?>" data-status="<?php echo htmlspecialchars($row['order_status']); ?>">
                                        <?php echo htmlspecialchars($row['order_status']); ?>
                                    </span>
                                </div>
                                <div class="card-body-mobile">
                                    <div class="customer-info">
                                        <div class="customer-name"><?php echo htmlspecialchars($row['customer_name']); ?></div>
                                        <div class="customer-mobile"><i class="fas fa-phone"></i> <?php echo htmlspecialchars($row['mobile']); ?></div>
                                        <div class="customer-mobile"><i class="fas fa-user-tie"></i> <?php echo htmlspecialchars($row['employee_name'] ?? 'Not Assigned'); ?></div>
                                    </div>
                                    <div class="amount-section">
                                        <div class="price-row">
                                            <span class="total-amount">₹<?php echo number_format($final_amount, 2); ?></span>
                                        </div>
                                        <div class="balance-row" style="margin-bottom:2px;">
                                            <span class="balance-label">Discount</span>
                                            <span class="balance-amount text-success">₹<?php echo number_format($row['discount_amount'], 2); ?></span>
                                        </div>
                                        <div class="balance-row" style="margin-bottom:2px;">
                                            <span class="balance-label">Advance</span>
                                            <span class="balance-amount text-primary">₹<?php echo number_format($row['advance_amount'], 2); ?></span>
                                        </div>
                                        <div class="balance-row">
                                            <span class="balance-label">Balance</span>
                                            <span class="balance-amount <?php echo $row['balance_amount'] > 0 ? 'pending' : 'paid'; ?>">₹<?php echo number_format($row['balance_amount'], 2); ?></span>
                                        </div>
                                    </div>
                                </div>
                                <div class="card-actions">
                                    <a href="customer-prescription-details.php?id=<?php echo $row['id']; ?>" class="action-btn view-btn" title="View"><i class="fas fa-eye"></i></a>
                                    <a href="customer-prescriptions.php?edit=<?php echo $row['id']; ?>" class="action-btn edit-btn" title="Edit"><i class="fas fa-edit"></i></a>
                                    <?php if($row['balance_amount'] > 0): ?>
                                    <button onclick="openPaymentModal(<?php echo $row['id']; ?>, '<?php echo htmlspecialchars(addslashes($row['customer_name'])); ?>', <?php echo $row['balance_amount']; ?>)" class="action-btn" style="background:#ffc107; color:#212529;" title="Record Payment"><i class="fas fa-rupee-sign"></i></button>
                                    <?php endif; ?>
                                    <button onclick="sendPrescriptionWhatsApp('<?php echo $row['mobile']; ?>','<?php echo htmlspecialchars(addslashes($row['customer_name'])); ?>','<?php echo $row['order_number']; ?>','<?php echo htmlspecialchars($row['order_status']); ?>')" class="action-btn whatsapp-btn" title="WhatsApp"><i class="fab fa-whatsapp"></i></button>
                                </div>
                            </div>
                            <?php endforeach; ?>
                            
                            <?php if(empty($prescriptions)): ?>
                            <div class="empty-state">
                                <i class="fas fa-search fa-3x text-muted mb-3"></i>
                                <h6 class="text-muted">No prescriptions found</h6>
                                <small class="text-muted">Try adjusting your search criteria</small>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    </div> <!-- /.report-container -->

    <!-- Print Footer -->
    <div class="text-center mt-4 print-section d-none d-print-block">
        <hr>
        <p class="text-muted">
            Report generated on <?php echo date('d-M-Y h:i A'); ?> | <?php echo htmlspecialchars($shopName); ?> Management System
        </p>
    </div>

</div>

<!-- 4 Inch Receipt Print Section -->
<div class="receipt-print">
    <div class="receipt-header">
    <div class="receipt-title"><?php echo strtoupper($shopName); ?></div>
    <div class="receipt-subtitle">Madurai</div>
        <div class="receipt-subtitle">Prescription Summary Report</div>
        <div class="receipt-subtitle"><?php echo date('d-M-Y h:i A'); ?></div>
    </div>
    
    <div class="receipt-section">
        <div class="receipt-summary">REPORT SUMMARY</div>
        <div class="receipt-row">
            <span>Period:</span>
            <span><?php echo date('d-M-Y', strtotime($from)); ?> to <?php echo date('d-M-Y', strtotime($to)); ?></span>
        </div>
        <div class="receipt-row">
            <span>Total Orders:</span>
            <span><?php echo number_format($total_orders); ?></span>
        </div>
        <div class="receipt-row">
            <span>Total Revenue:</span>
            <span>₹<?php echo number_format($total_revenue, 2); ?></span>
        </div>
        <div class="receipt-row">
            <span>Total Discounts:</span>
            <span>₹<?php echo number_format($total_discounts, 2); ?></span>
        </div>
        <div class="receipt-row">
            <span>Net Revenue:</span>
            <span>₹<?php echo number_format($total_revenue - $total_discounts, 2); ?></span>
        </div>
        <div class="receipt-row">
            <span>Outstanding:</span>
            <span>₹<?php echo number_format($total_outstanding, 2); ?></span>
        </div>
        <div class="receipt-row">
            <span>Advances:</span>
            <span>₹<?php echo number_format($total_advances, 2); ?></span>
        </div>
    </div>
    
    <div class="receipt-section">
        <div class="receipt-summary">STATUS BREAKDOWN</div>
        <?php foreach(['Pending', 'In Progress', 'Ready', 'Delivered', 'Cancelled'] as $status): 
            $count = isset($status_counts[$status]) ? $status_counts[$status] : 0;
            if($count > 0):
        ?>
        <div class="receipt-row">
            <span><?php echo $status; ?>:</span>
            <span><?php echo $count; ?></span>
        </div>
        <?php endif; endforeach; ?>
    </div>
    
    <?php if(!empty($prescriptions) && count($prescriptions) <= 10): ?>
    <div class="receipt-section">
        <div class="receipt-summary">RECENT ORDERS</div>
        <?php foreach(array_slice($prescriptions, 0, 10) as $row): ?>
        <div style="margin-bottom: 6px; font-size: 9px;">
            <div style="font-weight: bold;"><?php echo htmlspecialchars($row['customer_name']); ?></div>
            <div><?php echo htmlspecialchars($row['mobile']); ?> | <?php echo date('d-M', strtotime($row['created_at'])); ?></div>
            <div><?php echo htmlspecialchars($row['frame_name']); ?></div>
            <div>₹<?php echo number_format($row['total_amount'], 2); ?> | <?php echo htmlspecialchars($row['order_status']); ?></div>
        </div>
        <?php endforeach; ?>
    </div>
    <?php endif; ?>
    
    <div class="receipt-footer">
        <div>Thank you!</div>
    <div><?php echo htmlspecialchars($shopName); ?> Management System</div>
        <div>Report ID: RPT-<?php echo date('YmdHis'); ?></div>
    </div>
</div>

</div>

</div>
</div>
<?php include_once('includes/footer.php');?>
</div>
</div>
<?php include_once('includes/footer2.php');?>

<!-- Payment Modal -->
<div class="modal fade" id="paymentModal" tabindex="-1" role="dialog" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered" role="document">
        <div class="modal-content">
            <div class="modal-header bg-success text-white py-2">
                <h6 class="modal-title"><i class="fas fa-rupee-sign mr-2"></i>Record Payment</h6>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>
            </div>
            <div class="modal-body">
                <form id="paymentForm">
                    <input type="hidden" name="action" value="record_payment">
                    <input type="hidden" name="id" id="paymentPrescriptionId">
                    <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                    
                    <div class="mb-3">
                        <div class="alert alert-info py-2 px-3 mb-3">
                            <strong>Customer:</strong> <span id="paymentCustomerName"></span><br>
                            <strong>Outstanding Balance:</strong> ₹<span id="paymentOutstandingAmount"></span>
                        </div>
                    </div>
                    
                    <div class="form-group mb-3">
                        <label class="small font-weight-bold mb-1">Payment Amount <span class="text-danger">*</span></label>
                        <div class="input-group">
                            <div class="input-group-prepend">
                                <span class="input-group-text">₹</span>
                            </div>
                            <input type="number" name="payment_amount" id="paymentAmount" class="form-control" step="0.01" min="0.01" required>
                        </div>
                        <small class="text-muted">Enter the amount received from customer</small>
                    </div>
                    
                    <div class="form-group mb-3">
                        <label class="small font-weight-bold mb-1">Payment Method</label>
                        <select name="payment_method" id="paymentMethod" class="form-control">
                            <option value="Cash">Cash</option>
                            <option value="Card">Card</option>
                            <option value="UPI">UPI</option>
                            <option value="Bank Transfer">Bank Transfer</option>
                            <option value="Cheque">Cheque</option>
                        </select>
                    </div>
                    
                    <div class="alert alert-danger py-2 px-3 d-none" id="paymentError"></div>
                    <div class="alert alert-success py-2 px-3 d-none" id="paymentSuccess"></div>
                </form>
            </div>
            <div class="modal-footer py-2">
                <button type="button" class="btn btn-secondary btn-sm" data-dismiss="modal">Cancel</button>
                <button type="button" id="recordPaymentBtn" class="btn btn-success btn-sm">
                    <i class="fas fa-save mr-1"></i>Record Payment
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Status Change Modal -->
<div class="modal fade" id="statusModal" tabindex="-1" role="dialog" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered" role="document">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white py-2">
                <h6 class="modal-title">Change Status</h6>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>
            </div>
            <div class="modal-body">
                <form id="statusForm">
                    <input type="hidden" name="action" value="update_status">
                    <input type="hidden" name="id" id="statusPrescriptionId">
                    <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                    <div class="form-group mb-2">
                        <label for="newStatus" class="small font-weight-bold">Select New Status</label>
                        <select name="new_status" id="newStatus" class="form-control form-control-sm" required>
                            <option value="Pending">Pending</option>
                            <option value="In Progress">In Progress</option>
                            <option value="Ready">Ready</option>
                            <option value="Delivered">Delivered</option>
                            <option value="Cancelled">Cancelled</option>
                        </select>
                    </div>
                    <div class="alert alert-danger py-1 px-2 d-none" id="statusError"></div>
                    <div class="alert alert-success py-1 px-2 d-none" id="statusSuccess"></div>
                </form>
            </div>
            <div class="modal-footer py-2">
                <button type="button" class="btn btn-secondary btn-sm" data-dismiss="modal">Close</button>
                <button type="button" id="saveStatusBtn" class="btn btn-primary btn-sm">Update</button>
            </div>
        </div>
    </div>
</div>
<script src="vendor/jquery/jquery.min.js"></script>
<script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
<script src="vendor/jquery-easing/jquery.easing.min.js"></script>
<script src="js/sb-admin-2.min.js"></script>

<script>
function printReceipt() {
    // Add receipt print class to body
    document.body.classList.add('receipt-print-active');
    
    // Print the page
    window.print();
    
    // Remove the class after printing (with a delay to ensure print dialog has opened)
    setTimeout(function() {
        document.body.classList.remove('receipt-print-active');
    }, 1000);
}

// Optional: Auto-configure printer settings for thermal printers
function configureThermalPrint() {
    if (window.matchMedia) {
        var mediaQueryList = window.matchMedia('print');
        mediaQueryList.addListener(function(mql) {
            if (mql.matches) {
                // Print mode - you can add additional configurations here
                console.log('Printing in thermal receipt mode');
            }
        });
    }
}

// Initialize thermal print configuration
document.addEventListener('DOMContentLoaded', function() {
    configureThermalPrint();
});

// WhatsApp functionality (Individual)
function sendPrescriptionWhatsApp(mobile, customerName, orderNumber, status) {
    // Ask for confirmation first
    if (!confirm(`Send WhatsApp message to ${customerName} (${mobile})?`)) {
        return;
    }
    
    // Show loading state
    const originalTitle = document.title;
    document.title = 'Sending WhatsApp message...';
    
    // Send via API for consistent delivery and logging
    const formData = new FormData();
    formData.append('action', 'send_whatsapp');
    formData.append('id', getIdFromRow(mobile, customerName)); // Helper to get prescription ID
    formData.append('csrf_token', '<?php echo $_SESSION['csrf_token']; ?>');
    
    fetch('prescription-reports.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(result => {
        document.title = originalTitle;
        if (result.success) {
            alert('✅ WhatsApp message sent successfully!');
        } else {
            alert('❌ Failed to send WhatsApp: ' + result.message);
        }
    })
    .catch(error => {
        document.title = originalTitle;
        console.error('WhatsApp Error:', error);
        alert('❌ Error sending WhatsApp: ' + error.message);
    });
}

// Helper function to get prescription ID from available data
function getIdFromRow(mobile, customerName) {
    // Try to find the prescription ID from the table/cards
    const checkboxes = document.querySelectorAll('.prescription-checkbox');
    for (let checkbox of checkboxes) {
        if (checkbox.dataset.mobile === mobile && checkbox.dataset.customer === customerName) {
            return checkbox.value;
        }
    }
    return null;
}

// Bulk WhatsApp functionality
let bulkWhatsAppInProgress = false;

function toggleBulkMode() {
    const bulkElements = document.querySelectorAll('.bulk-mode-only');
    const bulkActions = document.getElementById('bulkActions');
    const regularActions = document.getElementById('regularActions');
    const toggleBtn = document.getElementById('bulkModeToggle');
    
    const isCurrentlyVisible = bulkElements[0].style.display !== 'none';
    
    if (isCurrentlyVisible) {
        // Hide bulk mode
        bulkElements.forEach(el => el.style.display = 'none');
        bulkActions.style.display = 'none';
        regularActions.style.display = 'flex';
        toggleBtn.innerHTML = '<i class="fas fa-check-square"></i> Bulk Select';
        clearSelection();
    } else {
        // Show bulk mode
        bulkElements.forEach(el => el.style.display = 'table-cell');
        // For mobile checkboxes, use flex instead of table-cell
        document.querySelectorAll('.mobile-checkbox-wrapper').forEach(el => el.style.display = 'flex');
        bulkActions.style.display = 'flex';
        regularActions.style.display = 'none';
        toggleBtn.innerHTML = '<i class="fas fa-times"></i> Exit Bulk';
    }
}

function toggleAllSelection() {
    const selectAll = document.getElementById('selectAll');
    const checkboxes = document.querySelectorAll('.prescription-checkbox');
    
    checkboxes.forEach(checkbox => {
        checkbox.checked = selectAll.checked;
    });
    
    updateSelection();
}

function updateSelection() {
    const checkboxes = document.querySelectorAll('.prescription-checkbox:checked');
    const count = checkboxes.length;
    
    document.getElementById('selectedCount').textContent = count;
    document.getElementById('bulkCount').textContent = count;
    
    // Update UI state
    const bulkWhatsAppBtn = document.getElementById('bulkWhatsAppBtn');
    if (count > 0) {
        bulkWhatsAppBtn.disabled = false;
        
        // Add visual feedback to selected rows/cards
        document.querySelectorAll('[data-prescription-id]').forEach(element => {
            const id = element.dataset.prescriptionId;
            const checkbox = document.querySelector(`.prescription-checkbox[value="${id}"]`);
            
            if (checkbox && checkbox.checked) {
                element.classList.add('bulk-selected');
            } else {
                element.classList.remove('bulk-selected');
            }
        });
    } else {
        bulkWhatsAppBtn.disabled = true;
        document.querySelectorAll('[data-prescription-id]').forEach(element => {
            element.classList.remove('bulk-selected');
        });
    }
}

function clearSelection() {
    document.querySelectorAll('.prescription-checkbox').forEach(checkbox => {
        checkbox.checked = false;
    });
    document.getElementById('selectAll').checked = false;
    updateSelection();
}

function sendBulkWhatsApp() {
    if (bulkWhatsAppInProgress) {
        alert('Bulk WhatsApp is already in progress. Please wait.');
        return;
    }
    
    const selectedCheckboxes = document.querySelectorAll('.prescription-checkbox:checked');
    if (selectedCheckboxes.length === 0) {
        alert('Please select at least one prescription to send WhatsApp messages.');
        return;
    }
    
    const count = selectedCheckboxes.length;
    const estimatedTime = Math.ceil(count * 15 / 60); // Estimate in minutes
    
    if (!confirm(`Send WhatsApp messages to ${count} customers?\n\nThis will take approximately ${estimatedTime} minutes with 15-second intervals between messages.\n\nProceed?`)) {
        return;
    }
    
    bulkWhatsAppInProgress = true;
    
    // Show progress modal
    const modal = $('#bulkProgressModal');
    modal.modal('show');
    
    const progressBar = document.getElementById('bulkProgress');
    const progressText = document.getElementById('bulkProgressText');
    const progressDetail = document.getElementById('bulkProgressDetail');
    const resultsContainer = document.getElementById('bulkResults');
    
    // Reset progress
    progressBar.style.width = '0%';
    progressText.textContent = 'Starting bulk WhatsApp sending...';
    progressDetail.textContent = `Sending ${count} messages with 15-second intervals`;
    resultsContainer.innerHTML = '';
    document.getElementById('bulkProgressClose').disabled = true;
    
    // Prepare prescription IDs
    const prescriptionIds = Array.from(selectedCheckboxes).map(cb => cb.value);
    
    // Send messages one by one with delay
    sendBulkMessages(prescriptionIds, 0);
}

async function sendBulkMessages(prescriptionIds, currentIndex) {
    const totalCount = prescriptionIds.length;
    
    if (currentIndex >= totalCount) {
        // Completed all messages
        const progressText = document.getElementById('bulkProgressText');
        const progressDetail = document.getElementById('bulkProgressDetail');
        
        progressText.textContent = `Completed! Sent ${totalCount} WhatsApp messages.`;
        progressDetail.textContent = 'All messages have been processed.';
        document.getElementById('bulkProgressClose').disabled = false;
        bulkWhatsAppInProgress = false;
        
        // Clear selections after completion
        setTimeout(() => {
            clearSelection();
        }, 2000);
        
        return;
    }
    
    const prescriptionId = prescriptionIds[currentIndex];
    const progress = ((currentIndex + 1) / totalCount) * 100;
    
    // Update progress
    const progressBar = document.getElementById('bulkProgress');
    const progressText = document.getElementById('bulkProgressText');
    const progressDetail = document.getElementById('bulkProgressDetail');
    
    progressBar.style.width = progress + '%';
    progressText.textContent = `Sending message ${currentIndex + 1} of ${totalCount}`;
    
    // Get customer info for display
    const checkbox = document.querySelector(`.prescription-checkbox[value="${prescriptionId}"]`);
    const customerName = checkbox?.dataset.customer || 'Unknown Customer';
    const mobile = checkbox?.dataset.mobile || 'Unknown Mobile';
    
    progressDetail.textContent = `Sending to ${customerName} (${mobile})`;
    
    // Add result item (processing state)
    const resultsContainer = document.getElementById('bulkResults');
    const resultItem = document.createElement('div');
    resultItem.className = 'result-item processing';
    resultItem.id = `result-${prescriptionId}`;
    resultItem.innerHTML = `
        <div class="customer-details">
            <div class="customer-name">${customerName}</div>
            <div class="customer-mobile">${mobile}</div>
        </div>
        <div class="result-status processing">Sending...</div>
    `;
    resultsContainer.appendChild(resultItem);
    
    // Scroll to bottom to show latest result
    resultsContainer.scrollTop = resultsContainer.scrollHeight;
    
    try {
        // Send WhatsApp message
        const formData = new FormData();
        formData.append('action', 'send_whatsapp');
        formData.append('id', prescriptionId);
        formData.append('csrf_token', '<?php echo $_SESSION['csrf_token']; ?>');
        
        const response = await fetch('prescription-reports.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        // Update result item
        const statusElement = resultItem.querySelector('.result-status');
        if (result.success) {
            resultItem.className = 'result-item success';
            statusElement.className = 'result-status success';
            statusElement.textContent = '✓ Sent';
        } else {
            resultItem.className = 'result-item error';
            statusElement.className = 'result-status error';
            statusElement.textContent = '✗ Failed: ' + (result.message || 'Unknown error');
        }
        
    } catch (error) {
        // Update result item with error
        const statusElement = resultItem.querySelector('.result-status');
        resultItem.className = 'result-item error';
        statusElement.className = 'result-status error';
        statusElement.textContent = '✗ Error: ' + error.message;
    }
    
    // Wait 15 seconds before sending next message (except for the last one)
    if (currentIndex < totalCount - 1) {
        progressDetail.textContent = `Waiting 15 seconds before next message...`;
        
        // Countdown timer
        for (let i = 15; i > 0; i--) {
            await new Promise(resolve => setTimeout(resolve, 1000));
            if (i > 1) {
                progressDetail.textContent = `Waiting ${i - 1} seconds before next message...`;
            }
        }
    }
    
    // Send next message
    sendBulkMessages(prescriptionIds, currentIndex + 1);
}

function closeBulkProgress() {
    if (bulkWhatsAppInProgress) {
        if (!confirm('Bulk WhatsApp is still in progress. Are you sure you want to close this window?\n\nNote: The process will continue in the background.')) {
            return;
        }
    }
    
    $('#bulkProgressModal').modal('hide');
}

function createPrescriptionWhatsAppMessage(customerName, orderNumber, status) {
    const shopName = <?php echo json_encode($shopName); ?>;
    const today = new Date().toLocaleDateString('en-GB');

    let statusLine = '';
    switch((status||'').toLowerCase()) {
        case 'pending':
            statusLine = 'Your order is received and in queue.'; break;
        case 'in progress':
            statusLine = 'Your order is being prepared.'; break;
        case 'ready':
            statusLine = 'Your glasses are ready for pickup.'; break;
        case 'delivered':
            statusLine = 'Your order has been delivered. Hope you like it!'; break;
        case 'cancelled':
            statusLine = 'This order has been cancelled. Contact us if you need help.'; break;
        default:
            statusLine = 'We are working on your order.';
    }

    return `*${shopName}*\n` +
           `Hi ${customerName},\n` +
           `Order: ${orderNumber}\n` +
           `Date: ${today}\n` +
           `${statusLine}\n` +
           `Need help? Reply here.\n` +
           `Thank you.`;
}

function confirmDelete(){return confirm('Delete this prescription?\n\nNote: Records with order numbers will be marked as cancelled to preserve the order number sequence. Records without order numbers will be permanently deleted.');}

function toggleAdvancedFilters() {
    const advancedFilters = document.getElementById('advancedFilters');
    const toggleBtn = document.querySelector('.toggle-btn');
    
    if (advancedFilters.style.display === 'none') {
        advancedFilters.style.display = 'block';
        toggleBtn.classList.add('active');
        toggleBtn.innerHTML = '<i class="fas fa-times"></i>';
        toggleBtn.title = 'Hide Filters';
    } else {
        advancedFilters.style.display = 'none';
        toggleBtn.classList.remove('active');
        toggleBtn.innerHTML = '<i class="fas fa-sliders-h"></i>';
        toggleBtn.title = 'More Filters';
    }
}

// Auto-submit form on filter changes
document.addEventListener('DOMContentLoaded', function() {
    const form = document.querySelector('.quick-filter-form');
    const statusPills = document.querySelectorAll('.status-pill');
    
    // Auto-submit on date change
    const dateInputs = form.querySelectorAll('input[type="date"]');
    dateInputs.forEach(input => {
        input.addEventListener('change', () => {
            form.submit();
        });
    });
    
    // Auto-submit on advanced filter change
    const selects = form.querySelectorAll('select');
    selects.forEach(select => {
        select.addEventListener('change', () => {
            form.submit();
        });
    });
    
    // Highlight active filters
    const hasActiveFilters = '<?php echo (!empty($search) || !empty($status_filter) || !empty($frame_type_filter) || !empty($lens_type_filter)) ? "true" : "false"; ?>';
    if (hasActiveFilters === 'true') {
        document.querySelector('.toggle-btn').style.background = '#dc3545';
    }
});

// Status change (mobile + desktop future) using modal & AJAX
document.addEventListener('DOMContentLoaded', function(){
    const statusModal = $('#statusModal');
    let currentBadge = null;
    document.querySelectorAll('.js-status-open').forEach(badge => {
        badge.addEventListener('click', function(){
            currentBadge = this;
            const id = this.getAttribute('data-id');
            const status = this.getAttribute('data-status');
            document.getElementById('statusPrescriptionId').value = id;
            document.getElementById('newStatus').value = status;
            document.getElementById('statusError').classList.add('d-none');
            document.getElementById('statusSuccess').classList.add('d-none');
            statusModal.modal('show');
        });
    });
    document.getElementById('saveStatusBtn').addEventListener('click', function(){
        const form = document.getElementById('statusForm');
        const formData = new FormData(form);
        const errorEl = document.getElementById('statusError');
        const successEl = document.getElementById('statusSuccess');
        errorEl.classList.add('d-none');
        successEl.classList.add('d-none');
        this.disabled = true;
        this.textContent = 'Updating...';
        fetch('prescription-reports.php', {
                method:'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                },
                credentials: 'same-origin',
                redirect: 'follow'
            })
            .then(response => {
                if (response.redirected) {
                    throw new Error('Session expired or server error');
                }
                const contentType = response.headers.get('content-type');
                if (contentType && contentType.includes('application/json')) {
                    return response.json();
                } else {
                    throw new Error('Session expired or server error');
                }
            })
            .then(data => {
                if(data.success){
                    successEl.textContent = data.message;
                    successEl.classList.remove('d-none');
                    if(currentBadge){
                        currentBadge.textContent = data.new_status;
                        currentBadge.setAttribute('data-status', data.new_status);
                        // update color classes
                        currentBadge.className = 'status-badge js-status-open status-' + data.new_status.toLowerCase().replace(/ /g,'-');
                    }
                    setTimeout(()=>{ statusModal.modal('hide'); }, 800);
                } else {
                    errorEl.textContent = data.message || 'Update failed';
                    errorEl.classList.remove('d-none');
                    
                    // Handle session expiration
                    if (data.redirect) {
                        setTimeout(() => {
                            window.location.href = data.redirect;
                        }, 2000);
                    }
                }
            })
            .catch((error)=>{
                if (error.message.includes('Session expired')) {
                    errorEl.textContent = 'Session expired. Redirecting to login...';
                    setTimeout(() => {
                        window.location.href = 'login.php';
                    }, 2000);
                } else {
                    errorEl.textContent = 'Network error';
                }
                errorEl.classList.remove('d-none');
            })
            .finally(()=>{
                document.getElementById('saveStatusBtn').disabled = false;
                document.getElementById('saveStatusBtn').textContent = 'Update';
            });
    });
});

// Payment Modal Functions
function openPaymentModal(prescriptionId, customerName, outstandingAmount) {
    document.getElementById('paymentPrescriptionId').value = prescriptionId;
    document.getElementById('paymentCustomerName').textContent = customerName;
    document.getElementById('paymentOutstandingAmount').textContent = parseFloat(outstandingAmount).toFixed(2);
    
    // Suggest full outstanding amount as default
    document.getElementById('paymentAmount').value = parseFloat(outstandingAmount).toFixed(2);
    
    // Clear previous messages
    document.getElementById('paymentError').classList.add('d-none');
    document.getElementById('paymentSuccess').classList.add('d-none');
    
    $('#paymentModal').modal('show');
}

// Handle payment recording
document.addEventListener('DOMContentLoaded', function() {
    document.getElementById('recordPaymentBtn').addEventListener('click', function() {
        const form = document.getElementById('paymentForm');
        const formData = new FormData(form);
        const errorEl = document.getElementById('paymentError');
        const successEl = document.getElementById('paymentSuccess');
        const btn = this;
        
        // Clear previous messages
        errorEl.classList.add('d-none');
        successEl.classList.add('d-none');
        
        // Validate payment amount
        const paymentAmount = parseFloat(formData.get('payment_amount'));
        const outstandingAmount = parseFloat(document.getElementById('paymentOutstandingAmount').textContent);
        
        if (paymentAmount <= 0) {
            errorEl.textContent = 'Payment amount must be greater than 0';
            errorEl.classList.remove('d-none');
            return;
        }
        
        if (paymentAmount > outstandingAmount) {
            if (!confirm(`Payment amount (₹${paymentAmount.toFixed(2)}) is more than outstanding balance (₹${outstandingAmount.toFixed(2)}). Continue anyway?`)) {
                return;
            }
        }
        
        // Disable button and show loading
        btn.disabled = true;
        btn.innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i>Recording...';
        
        fetch('prescription-reports.php', {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json'
            },
            credentials: 'same-origin',
            redirect: 'follow'
        })
        .then(response => {
            if (response.redirected) {
                throw new Error('Session expired or server error');
            }
            // Check if response is JSON or HTML
            const contentType = response.headers.get('content-type');
            if (contentType && contentType.includes('application/json')) {
                return response.json();
            } else {
                // If HTML response, likely session expired
                throw new Error('Session expired or server error');
            }
        })
        .then(data => {
            if (data.success) {
                successEl.innerHTML = `<i class="fas fa-check-circle mr-1"></i>${data.message}<br><small>New Advance: ₹${data.new_advance} | New Balance: ₹${data.new_balance}</small>`;
                successEl.classList.remove('d-none');
                
                // Update the displayed balance amounts in the table/cards
                setTimeout(() => {
                    $('#paymentModal').modal('hide');
                    // Refresh the page to show updated amounts
                    window.location.reload();
                }, 1500);
            } else {
                errorEl.innerHTML = `<i class="fas fa-exclamation-triangle mr-1"></i>${data.message}`;
                errorEl.classList.remove('d-none');
                
                // Handle session expiration
                if (data.redirect) {
                    setTimeout(() => {
                        window.location.href = data.redirect;
                    }, 2000);
                }
            }
        })
        .catch(error => {
            console.error('Payment error:', error);
            if (error.message.includes('Session expired')) {
                errorEl.innerHTML = '<i class="fas fa-exclamation-triangle mr-1"></i>Session expired. Redirecting to login...';
                setTimeout(() => {
                    window.location.href = 'login.php';
                }, 2000);
            } else {
                errorEl.innerHTML = '<i class="fas fa-exclamation-triangle mr-1"></i>Network error occurred';
            }
            errorEl.classList.remove('d-none');
        })
        .finally(() => {
            btn.disabled = false;
            btn.innerHTML = '<i class="fas fa-save mr-1"></i>Record Payment';
        });
    });
    
    // Auto-focus on payment amount when modal opens
    $('#paymentModal').on('shown.bs.modal', function() {
        $('#paymentAmount').focus().select();
    });
    
    // Clear form when modal closes
    $('#paymentModal').on('hidden.bs.modal', function() {
        document.getElementById('paymentForm').reset();
        document.getElementById('paymentError').classList.add('d-none');
        document.getElementById('paymentSuccess').classList.add('d-none');
    });
});
</script>
</body>
</html>
