<?php
// WhatsApp Integration Functions
// This file contains functions to send WhatsApp messages via API

function sendWhatsAppMessage($mobile, $message, $instanceId = '685CEC22339F3', $accessToken = '677655e0f0f18') {
    // Clean and format mobile number
    $cleanMobile = preg_replace('/[^0-9]/', '', $mobile);
    
    // Remove leading zero if present
    if (substr($cleanMobile, 0, 1) === '0') {
        $cleanMobile = substr($cleanMobile, 1);
    }
    
    // Add country code if not present (assuming India +91)
    if (strlen($cleanMobile) === 10) {
        $cleanMobile = '91' . $cleanMobile;
    }
    
    // Validate mobile number format
    if (!preg_match('/^91\d{10}$/', $cleanMobile)) {
        return [
            'success' => false,
            'message' => 'Invalid mobile number format: ' . $mobile
        ];
    }
    
    // Prepare API call
    $encodedMessage = urlencode($message);
    $apiUrl = "https://wav5.algotechnosoft.com/api/send?instance_id={$instanceId}&access_token={$accessToken}&type=text&number={$cleanMobile}&message={$encodedMessage}";
    
    // Initialize cURL
    $curl = curl_init();
    
    curl_setopt_array($curl, [
        CURLOPT_URL => $apiUrl,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_USERAGENT => 'Opticals-WhatsApp-Bot/1.0'
    ]);
    
    $response = curl_exec($curl);
    $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    $error = curl_error($curl);
    
    curl_close($curl);
    
    if ($error) {
        return [
            'success' => false,
            'message' => 'cURL Error: ' . $error
        ];
    }
    
    if ($httpCode !== 200) {
        return [
            'success' => false,
            'message' => 'HTTP Error: ' . $httpCode
        ];
    }
    
    $responseData = json_decode($response, true);
    
    return [
        'success' => true,
        'response' => $responseData,
        'message' => 'WhatsApp message sent successfully'
    ];
}

function createPrescriptionWhatsAppMessage($customerName, $orderNumber, $prescriptionData = null, $shopName = 'Jagadamba Opticals') {
    $today = date('d-M-Y');
    
    if ($prescriptionData) {
        // Detailed message with prescription data
        $message = "👓 *{$shopName}* - Prescription Order\n\n";
        $message .= "Dear {$customerName},\n\n";
        $message .= "Your prescription order has been successfully recorded!\n\n";
        
        $message .= "📋 *ORDER DETAILS*\n";
        $message .= "• Order Number: {$orderNumber}\n";
        $message .= "• Date: {$today}\n";
        $message .= "• Status: {$prescriptionData['order_status']}\n";
        if (!empty($prescriptionData['delivery_date'])) {
            $deliveryDate = date('d-M-Y', strtotime($prescriptionData['delivery_date']));
            $message .= "• Expected Delivery: {$deliveryDate}\n";
        }
        $message .= "\n";
        
        $message .= "👁️ *PRESCRIPTION DETAILS*\n";
        $message .= "*Right Eye:*\n";
        $message .= "• SPH: " . ($prescriptionData['right_eye_sph'] ?: 'N/A') . "\n";
        $message .= "• CYL: " . ($prescriptionData['right_eye_cyl'] ?: 'N/A') . "\n";
        $message .= "• AXIS: " . ($prescriptionData['right_eye_axis'] ?: 'N/A') . "\n";
        $message .= "• ADD: " . ($prescriptionData['right_eye_addition'] ?: 'N/A') . "\n\n";
        
        $message .= "*Left Eye:*\n";
        $message .= "• SPH: " . ($prescriptionData['left_eye_sph'] ?: 'N/A') . "\n";
        $message .= "• CYL: " . ($prescriptionData['left_eye_cyl'] ?: 'N/A') . "\n";
        $message .= "• AXIS: " . ($prescriptionData['left_eye_axis'] ?: 'N/A') . "\n";
        $message .= "• ADD: " . ($prescriptionData['left_eye_addition'] ?: 'N/A') . "\n\n";
        
        $message .= "🔧 *PRODUCT DETAILS*\n";
        if (!empty($prescriptionData['frame_name'])) {
            $message .= "• Frame: {$prescriptionData['frame_name']}\n";
        }
        if (!empty($prescriptionData['lens_name'])) {
            $message .= "• Lens: {$prescriptionData['lens_name']}\n";
        }
        if (!empty($prescriptionData['frame_type'])) {
            $message .= "• Frame Type: {$prescriptionData['frame_type']}\n";
        }
        if (!empty($prescriptionData['lens_type'])) {
            $message .= "• Lens Type: {$prescriptionData['lens_type']}\n";
        }
        $message .= "\n";
        
        $message .= "💰 *BILLING DETAILS*\n";
        $frameAmount = floatval($prescriptionData['frame_amount']);
        $lensAmount = floatval($prescriptionData['lens_amount']);
        $discountValue = floatval($prescriptionData['discount_value'] ?: 0);
        $advanceAmount = floatval($prescriptionData['advance_amount'] ?: 0);
        
        $total = $frameAmount + $lensAmount - $discountValue;
        $balance = $total - $advanceAmount;
        
        $message .= "• Frame Amount: ₹" . number_format($frameAmount, 2) . "\n";
        $message .= "• Lens Amount: ₹" . number_format($lensAmount, 2) . "\n";
        if ($discountValue > 0) {
            $message .= "• Discount: -₹" . number_format($discountValue, 2) . "\n";
        }
        $message .= "• Total Amount: ₹" . number_format($total, 2) . "\n";
        $message .= "• Advance Paid: ₹" . number_format($advanceAmount, 2) . "\n";
        $message .= "• Balance Due: ₹" . number_format($balance, 2) . "\n\n";
        
        if (!empty($prescriptionData['notes'])) {
            $message .= "📝 *NOTES*\n{$prescriptionData['notes']}\n\n";
        }
    } else {
        // Simple message without detailed prescription data
        $message = "*{$shopName}*\n\n";
        $message .= "Hello {$customerName}!\n\n";
        $message .= "Thank you for choosing {$shopName}.\n\n";
        $message .= "📋 *Order Details:*\n";
        $message .= "Order Number: {$orderNumber}\n";
        $message .= "Date: {$today}\n\n";
        $message .= "Your prescription has been processed successfully.\n\n";
        $message .= "👁️ We will notify you once your glasses are ready for delivery.\n\n";
        $message .= "For any queries, please contact us.\n\n";
        $message .= "Thank you for your business! 🙏\n\n";
    }
    
    $message .= "📞 *CONTACT INFORMATION*\n";
    $message .= "• Phone: +91-XXXXXXXXXX\n";
    $message .= "• Address: [Your Shop Address]\n\n";
    
    $message .= "Thank you for choosing {$shopName}!\n";
    $message .= "Your Vision, Our Mission! 👓";
    
    return $message;
}

function sendPrescriptionWhatsApp($mobile, $customerName, $orderNumber, $prescriptionData = null) {
    $message = createPrescriptionWhatsAppMessage($customerName, $orderNumber, $prescriptionData);
    return sendWhatsAppMessage($mobile, $message);
}

function logWhatsAppActivity($mobile, $customerName, $orderNumber, $success, $response = null) {
    // Log WhatsApp activity to database or file
    $logData = [
        'timestamp' => date('Y-m-d H:i:s'),
        'mobile' => $mobile,
        'customer_name' => $customerName,
        'order_number' => $orderNumber,
        'success' => $success ? 1 : 0,
        'response' => $response ? json_encode($response) : null
    ];
    
    // You can implement database logging here if needed
    // For now, we'll just log to a file
    $logFile = __DIR__ . '/../logs/whatsapp.log';
    $logDir = dirname($logFile);
    
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $logEntry = date('Y-m-d H:i:s') . " - Mobile: {$mobile}, Customer: {$customerName}, Order: {$orderNumber}, Success: " . ($success ? 'Yes' : 'No') . "\n";
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}
?>