<?php
/**
 * SMS Functions for Bulk Campaign System
 * Handles SMS sending, logging, and validation
 */

// Provider configuration
// Options: 'demo' (no real send), 'http' (generic HTTP provider), 'twilio'
if (!defined('SMS_PROVIDER')) {
    define('SMS_PROVIDER', 'demo');
}

// Generic HTTP provider settings (configure if using SMS_PROVIDER='http')
// Example: Many gateways accept form data: to, sender_id, message, api_key
if (!defined('SMS_API_URL')) {
    define('SMS_API_URL', '');
}
if (!defined('SMS_HTTP_METHOD')) {
    define('SMS_HTTP_METHOD', 'POST'); // 'GET' or 'POST'
}
if (!defined('SMS_API_KEY')) {
    define('SMS_API_KEY', '');
}
if (!defined('SMS_SENDER_ID')) {
    define('SMS_SENDER_ID', '');
}
// Map field names to match your provider; adjust as needed
if (!defined('SMS_HTTP_FIELDS')) {
    // Sent as application/x-www-form-urlencoded for POST, query params for GET
    define('SMS_HTTP_FIELDS', json_encode([
        'api_key' => '{API_KEY}',
        'sender_id' => '{SENDER_ID}',
        'to' => '{MOBILE}',
        'message' => '{MESSAGE}'
    ]));
}

// Twilio settings (configure if using SMS_PROVIDER='twilio')
if (!defined('TWILIO_ACCOUNT_SID')) { define('TWILIO_ACCOUNT_SID', ''); }
if (!defined('TWILIO_AUTH_TOKEN')) { define('TWILIO_AUTH_TOKEN', ''); }
if (!defined('TWILIO_FROM_NUMBER')) { define('TWILIO_FROM_NUMBER', ''); }

/**
 * Direct SMS Sending Function (WhatsApp-style implementation)
 * Similar to working WhatsApp function - simple and direct
 * @param string $mobile Mobile number
 * @param string $message SMS content
 * @param string $apiUrl Your SMS provider URL
 * @param string $apiKey Your API key
 * @param string $senderId Your sender ID
 * @return array Result with success status and message
 */
function sendDirectSMS($mobile, $message, $apiUrl = '', $apiKey = '', $senderId = 'OPTICALS') {
    // Clean mobile number (same as WhatsApp)
    $cleanMobile = preg_replace('/[^0-9]/', '', $mobile);
    
    // Remove leading zero if present (same as WhatsApp)
    if (substr($cleanMobile, 0, 1) === '0') {
        $cleanMobile = substr($cleanMobile, 1);
    }
    
    // Add country code if not present (same as WhatsApp)
    if (strlen($cleanMobile) === 10) {
        $cleanMobile = '91' . $cleanMobile;
    }
    
    // Validate mobile number format (same as WhatsApp)
    if (!preg_match('/^91\d{10}$/', $cleanMobile)) {
        return [
            'success' => false,
            'message' => 'Invalid mobile number format: ' . $mobile
        ];
    }
    
    // Use demo SMS provider if no real API configured
    if (empty($apiUrl)) {
        return [
            'success' => false,
            'message' => 'SMS provider not configured - using demo mode'
        ];
    }
    
    // Prepare API URL with parameters (similar to WhatsApp)
    $encodedMessage = urlencode($message);
    $fullApiUrl = $apiUrl . "?api_key=" . urlencode($apiKey) . 
                  "&sender_id=" . urlencode($senderId) . 
                  "&to=" . urlencode($cleanMobile) . 
                  "&message=" . $encodedMessage;
    
    // Initialize cURL (exact same as WhatsApp)
    $curl = curl_init();
    
    curl_setopt_array($curl, [
        CURLOPT_URL => $fullApiUrl,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_USERAGENT => 'Opticals-SMS-Bot/1.0'
    ]);
    
    $response = curl_exec($curl);
    $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    $error = curl_error($curl);
    
    curl_close($curl);
    
    // Error handling (exact same as WhatsApp)
    if ($error) {
        return [
            'success' => false,
            'message' => 'cURL Error: ' . $error
        ];
    }
    
    if ($httpCode !== 200) {
        return [
            'success' => false,
            'message' => 'HTTP Error: ' . $httpCode
        ];
    }
    
    // Success (exact same pattern as WhatsApp)
    return [
        'success' => true,
        'response' => $response,
        'message' => 'SMS message sent successfully'
    ];
}

/**
 * Send SMS using API
 * @param string $mobile Mobile number
 * @param string $message SMS message content
 * @param string $customerName Customer name for logging
 * @return array Result with success status and message
 */
function sendSMS($mobile, $message, $customerName = '') {
    // Clean mobile number (remove spaces, dashes, etc.)
    $cleanMobile = preg_replace('/\D/', '', $mobile);
    
    // Validate mobile number
    if (strlen($cleanMobile) < 10) {
        return [
            'success' => false, 
            'message' => 'Invalid mobile number: ' . $mobile
        ];
    }
    
    // Format mobile number (add country code if needed)
    if (strlen($cleanMobile) == 10) {
        $cleanMobile = '91' . $cleanMobile; // Assuming India (+91)
    }
    
    // Validate message content
    if (empty(trim($message))) {
        return [
            'success' => false, 
            'message' => 'Message content is empty'
        ];
    }
    
    $trimmedMessage = trim($message);
    $provider = strtolower(SMS_PROVIDER);
    
    if ($provider === 'demo') {
        // Demo mode - simulate successful sending for testing UI
        // To enable real SMS, configure SMS_PROVIDER in config.php with actual provider credentials
        logSMSActivity($mobile, $trimmedMessage, $customerName, true, 'Demo mode - simulating successful SMS send');
        
        return [
            'success' => true, 
            'message' => 'SMS sent successfully (demo mode)'
        ];
    }
    
    if ($provider === 'twilio') {
        if (!TWILIO_ACCOUNT_SID || !TWILIO_AUTH_TOKEN || !TWILIO_FROM_NUMBER) {
            logSMSActivity($mobile, $trimmedMessage, $customerName, false, 'Twilio credentials missing');
            return ['success' => false, 'message' => 'Twilio not configured'];
        }
        // Twilio API
        $url = 'https://api.twilio.com/2010-04-01/Accounts/' . rawurlencode(TWILIO_ACCOUNT_SID) . '/Messages.json';
        $post = http_build_query([
            'From' => TWILIO_FROM_NUMBER,
            'To' => (strlen($cleanMobile) === 10 ? '+91' . $cleanMobile : '+' . $cleanMobile),
            'Body' => $trimmedMessage
        ]);
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_USERPWD, TWILIO_ACCOUNT_SID . ':' . TWILIO_AUTH_TOKEN);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        if ($httpCode >= 200 && $httpCode < 300) {
            logSMSActivity($mobile, $trimmedMessage, $customerName, true, 'Twilio OK');
            return ['success' => true, 'message' => 'SMS sent'];
        } else {
            logSMSActivity($mobile, $trimmedMessage, $customerName, false, 'Twilio HTTP ' . $httpCode . ' ' . substr((string)$response,0,200));
            return ['success' => false, 'message' => 'Twilio error HTTP ' . $httpCode];
        }
    }
    
    if ($provider === 'http') {
        if (!SMS_API_URL) {
            logSMSActivity($mobile, $trimmedMessage, $customerName, false, 'HTTP provider URL missing');
            return ['success' => false, 'message' => 'SMS API URL not configured'];
        }
        $fieldsTemplate = json_decode(SMS_HTTP_FIELDS, true) ?: [];
        $fields = [];
        foreach ($fieldsTemplate as $k => $v) {
            $replaced = str_replace(
                ['{API_KEY}','{SENDER_ID}','{MOBILE}','{MESSAGE}'],
                [SMS_API_KEY, SMS_SENDER_ID, $cleanMobile, $trimmedMessage],
                $v
            );
            $fields[$k] = $replaced;
        }
        $method = strtoupper(SMS_HTTP_METHOD);
        $ch = null;
        if ($method === 'GET') {
            $qs = http_build_query($fields);
            $url = SMS_API_URL . (strpos(SMS_API_URL,'?')===false?'?':'&') . $qs;
            $ch = curl_init($url);
        } else {
            $ch = curl_init(SMS_API_URL);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($fields));
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/x-www-form-urlencoded']);
        }
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlErr = curl_error($ch);
        curl_close($ch);
        if ($response === false) {
            logSMSActivity($mobile, $trimmedMessage, $customerName, false, 'cURL error: ' . $curlErr);
            return ['success' => false, 'message' => 'SMS HTTP request failed'];
        }
        if ($httpCode >= 200 && $httpCode < 300) {
            // You may want to parse provider-specific success flags here
            logSMSActivity($mobile, $trimmedMessage, $customerName, true, substr((string)$response,0,200));
            return ['success' => true, 'message' => 'SMS sent'];
        } else {
            logSMSActivity($mobile, $trimmedMessage, $customerName, false, 'HTTP ' . $httpCode . ' ' . substr((string)$response,0,200));
            return ['success' => false, 'message' => 'SMS API HTTP ' . $httpCode];
        }
    }
    
    // Unknown provider
    logSMSActivity($mobile, $trimmedMessage, $customerName, false, 'Unknown provider: ' . SMS_PROVIDER);
    return ['success' => false, 'message' => 'Unknown SMS provider'];
}

/**
 * Send bulk SMS with duplicate prevention
 * @param array $customers Array of customer data
 * @param string $messageTemplate Message template with placeholders
 * @return array Results for each customer
 */
function sendBulkSMS($customers, $messageTemplate) {
    $results = [];
    $processedMobiles = [];
    
    foreach ($customers as $customer) {
        $mobile = trim($customer['mobile']);
        $customerName = $customer['customer_name'] ?? 'Customer';
        
        // Skip if mobile number already processed (duplicate prevention)
        if (in_array($mobile, $processedMobiles)) {
            $results[] = [
                'customer_id' => $customer['id'],
                'customer_name' => $customerName,
                'mobile' => $mobile,
                'success' => false,
                'message' => 'Duplicate mobile number - skipped'
            ];
            continue;
        }
        
        // Process message template
        $personalizedMessage = personalizeSMSMessage($messageTemplate, $customer);
        
        // Send SMS
        $smsResult = sendSMS($mobile, $personalizedMessage, $customerName);
        
        $results[] = [
            'customer_id' => $customer['id'],
            'customer_name' => $customerName,
            'mobile' => $mobile,
            'success' => $smsResult['success'],
            'message' => $smsResult['message']
        ];
        
        // Mark mobile as processed
        if ($smsResult['success']) {
            $processedMobiles[] = $mobile;
        }
        
        // Add small delay to prevent API rate limiting
        usleep(500000); // 0.5 second delay
    }
    
    return $results;
}

/**
 * Personalize SMS message with customer data
 * @param string $template Message template
 * @param array $customer Customer data
 * @return string Personalized message
 */
function personalizeSMSMessage($template, $customer) {
    $placeholders = [
        '[Customer Name]' => $customer['customer_name'] ?? 'Valued Customer',
        '[CUSTOMER NAME]' => strtoupper($customer['customer_name'] ?? 'VALUED CUSTOMER'),
        '[customer name]' => strtolower($customer['customer_name'] ?? 'valued customer'),
        '[Mobile]' => $customer['mobile'] ?? '',
        '[Order Number]' => $customer['order_number'] ?? '',
        '[Status]' => $customer['status'] ?? ''
    ];
    
    $personalizedMessage = $template;
    foreach ($placeholders as $placeholder => $value) {
        $personalizedMessage = str_replace($placeholder, $value, $personalizedMessage);
    }
    
    return $personalizedMessage;
}

/**
 * Log SMS activity to database
 * @param string $mobile Mobile number
 * @param string $message SMS content
 * @param string $customerName Customer name
 * @param bool $success Whether SMS was sent successfully
 * @param string $response API response or error message
 */
function logSMSActivity($mobile, $message, $customerName, $success, $response = '') {
    global $con;
    
    // Create SMS logs table if it doesn't exist
    $createTable = "CREATE TABLE IF NOT EXISTS sms_logs (
        id INT AUTO_INCREMENT PRIMARY KEY,
        mobile VARCHAR(20) NOT NULL,
        message TEXT NOT NULL,
        customer_name VARCHAR(255),
        status ENUM('sent', 'failed') NOT NULL,
        response TEXT,
        user_id INT,
        sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_mobile (mobile),
        INDEX idx_status (status),
        INDEX idx_sent_at (sent_at)
    )";
    
    if (!$con->query($createTable)) {
        error_log("Failed to create sms_logs table: " . $con->error);
        return false;
    }
    
    // Insert log record
    $status = $success ? 'sent' : 'failed';
    $userId = $_SESSION['uid'] ?? $_SESSION['aid'] ?? null;
    
    $query = "INSERT INTO sms_logs (mobile, message, customer_name, status, response, user_id) VALUES (?, ?, ?, ?, ?, ?)";
    $stmt = $con->prepare($query);
    
    if ($stmt) {
        $stmt->bind_param("sssssi", $mobile, $message, $customerName, $status, $response, $userId);
        $success = $stmt->execute();
        $stmt->close();
        
        if (!$success) {
            error_log("Failed to log SMS activity: " . $con->error);
        }
        
        return $success;
    } else {
        error_log("Failed to prepare SMS log statement: " . $con->error);
        return false;
    }
}

/**
 * Get SMS statistics
 * @param string $dateFrom Start date (YYYY-MM-DD)
 * @param string $dateTo End date (YYYY-MM-DD)
 * @return array SMS statistics
 */
function getSMSStatistics($dateFrom = null, $dateTo = null) {
    global $conn;
    
    $whereClause = "WHERE 1=1";
    $params = [];
    $types = "";
    
    if ($dateFrom) {
        $whereClause .= " AND DATE(sent_at) >= ?";
        $params[] = $dateFrom;
        $types .= "s";
    }
    
    if ($dateTo) {
        $whereClause .= " AND DATE(sent_at) <= ?";
        $params[] = $dateTo;
        $types .= "s";
    }
    
    $query = "SELECT 
                COUNT(*) as total_sms,
                SUM(CASE WHEN status = 'sent' THEN 1 ELSE 0 END) as sent_count,
                SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed_count,
                COUNT(DISTINCT mobile) as unique_mobiles,
                DATE(MAX(sent_at)) as last_sms_date
              FROM sms_logs 
              $whereClause";
    
    $stmt = $con->prepare($query);
    
    if ($params) {
        $stmt->bind_param($types, ...$params);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    $stats = $result->fetch_assoc();
    $stmt->close();
    
    return [
        'total_sms' => (int)$stats['total_sms'],
        'sent_count' => (int)$stats['sent_count'],
        'failed_count' => (int)$stats['failed_count'],
        'success_rate' => $stats['total_sms'] > 0 ? round(($stats['sent_count'] / $stats['total_sms']) * 100, 2) : 0,
        'unique_mobiles' => (int)$stats['unique_mobiles'],
        'last_sms_date' => $stats['last_sms_date']
    ];
}

/**
 * Get recent SMS logs
 * @param int $limit Number of records to retrieve
 * @param int $offset Offset for pagination
 * @return array SMS log records
 */
function getRecentSMSLogs($limit = 50, $offset = 0) {
    global $conn;
    
    $query = "SELECT 
                mobile, 
                message, 
                customer_name, 
                status, 
                response, 
                sent_at 
              FROM sms_logs 
              ORDER BY sent_at DESC 
              LIMIT ? OFFSET ?";
    
    $stmt = $con->prepare($query);
    $stmt->bind_param("ii", $limit, $offset);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $logs = [];
    while ($row = $result->fetch_assoc()) {
        $logs[] = $row;
    }
    
    $stmt->close();
    return $logs;
}

/**
 * Validate SMS message content
 * @param string $message SMS message
 * @return array Validation result
 */
function validateSMSMessage($message) {
    $message = trim($message);
    $length = strlen($message);
    
    if (empty($message)) {
        return [
            'valid' => false,
            'message' => 'Message cannot be empty'
        ];
    }
    
    if ($length > 1000) {
        return [
            'valid' => false,
            'message' => 'Message is too long (maximum 1000 characters)'
        ];
    }
    
    // Check for potentially harmful content
    $forbiddenWords = ['spam', 'fraud', 'scam']; // Add more as needed
    foreach ($forbiddenWords as $word) {
        if (stripos($message, $word) !== false) {
            return [
                'valid' => false,
                'message' => 'Message contains forbidden content'
            ];
        }
    }
    
    return [
        'valid' => true,
        'message' => 'Message is valid',
        'length' => $length,
        'sms_count' => ceil($length / 160) // Estimate SMS count (160 chars per SMS)
    ];
}

/**
 * Format mobile number for display
 * @param string $mobile Raw mobile number
 * @return string Formatted mobile number
 */
function formatMobileNumber($mobile) {
    $clean = preg_replace('/\D/', '', $mobile);
    
    if (strlen($clean) >= 10) {
        $last10 = substr($clean, -10);
        return '+91 ' . substr($last10, 0, 5) . ' ' . substr($last10, 5);
    }
    
    return $mobile;
}
?>