<?php
session_start();
include_once('includes/config.php');

// Check if user is logged in
$isAdmin = isset($_SESSION['aid']);
$userId = $isAdmin ? $_SESSION['aid'] : (isset($_SESSION['uid']) ? $_SESSION['uid'] : null);

if (!$userId) {
    header('location:login.php');
    exit();
}

// CSRF token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
}

$success_msg = '';
$error_msg = '';

// Handle Create/Update Employee
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_employee'])) {
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $error_msg = 'Invalid request token.';
    } else {
        $employee_name = trim($_POST['employee_name']);
        $mobile = trim($_POST['mobile']);
        $status = $_POST['status'];
        $employee_id = $_POST['employee_id'] ? (int)$_POST['employee_id'] : 0;
        
        if (empty($employee_name)) {
            $error_msg = 'Employee name is required.';
        } elseif (empty($mobile)) {
            $error_msg = 'Mobile number is required.';
        } elseif (!preg_match('/^[6-9]\d{9}$/', $mobile)) {
            $error_msg = 'Please enter a valid 10-digit mobile number.';
        } else {
            if ($employee_id > 0) {
                // Update existing employee
                $sql = "UPDATE employees SET employee_name=?, mobile=?, status=?, updated_at=NOW() WHERE id=?";
                $stmt = $con->prepare($sql);
                $stmt->bind_param('sssi', $employee_name, $mobile, $status, $employee_id);
                
                if ($stmt->execute()) {
                    $success_msg = 'Employee updated successfully!';
                } else {
                    if ($con->errno === 1062) {
                        $error_msg = 'Mobile number already exists for another employee.';
                    } else {
                        $error_msg = 'Error updating employee: ' . $con->error;
                    }
                }
            } else {
                // Create new employee
                $sql = "INSERT INTO employees (employee_name, mobile, status) VALUES (?, ?, ?)";
                $stmt = $con->prepare($sql);
                $stmt->bind_param('sss', $employee_name, $mobile, $status);
                
                if ($stmt->execute()) {
                    $success_msg = 'Employee added successfully!';
                } else {
                    if ($con->errno === 1062) {
                        $error_msg = 'Mobile number already exists.';
                    } else {
                        $error_msg = 'Error adding employee: ' . $con->error;
                    }
                }
            }
        }
    }
}

// Handle Delete Employee
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_employee'])) {
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $error_msg = 'Invalid request token.';
    } else {
        $employee_id = (int)$_POST['employee_id'];
        
        // Check if employee has any prescriptions
        $checkSql = "SELECT COUNT(*) as count FROM customer_prescriptions WHERE employee_id = ?";
        $checkStmt = $con->prepare($checkSql);
        $checkStmt->bind_param('i', $employee_id);
        $checkStmt->execute();
        $result = $checkStmt->get_result();
        $count = $result->fetch_assoc()['count'];
        
        if ($count > 0) {
            // Soft delete - mark as inactive
            $sql = "UPDATE employees SET status='Inactive', updated_at=NOW() WHERE id=?";
            $stmt = $con->prepare($sql);
            $stmt->bind_param('i', $employee_id);
            
            if ($stmt->execute()) {
                $success_msg = 'Employee marked as inactive (has existing prescriptions).';
            } else {
                $error_msg = 'Error deactivating employee.';
            }
        } else {
            // Hard delete
            $sql = "DELETE FROM employees WHERE id=?";
            $stmt = $con->prepare($sql);
            $stmt->bind_param('i', $employee_id);
            
            if ($stmt->execute()) {
                $success_msg = 'Employee deleted successfully.';
            } else {
                $error_msg = 'Error deleting employee.';
            }
        }
    }
}

// Fetch employees
$employees_sql = "SELECT * FROM employees ORDER BY status DESC, employee_name ASC";
$employees_result = $con->query($employees_sql);
$employees = [];
while ($row = $employees_result->fetch_assoc()) {
    $employees[] = $row;
}

// Get employee for editing
$edit_employee = null;
if (isset($_GET['edit'])) {
    $edit_id = (int)$_GET['edit'];
    $edit_sql = "SELECT * FROM employees WHERE id = ?";
    $edit_stmt = $con->prepare($edit_sql);
    $edit_stmt->bind_param('i', $edit_id);
    $edit_stmt->execute();
    $edit_result = $edit_stmt->get_result();
    $edit_employee = $edit_result->fetch_assoc();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Jagadamba Opticals | Employee Management</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <style>
        body { background: #f8f9fc; }
        .card { border-radius: 1rem; box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15); }
        .form-control, .btn { border-radius: 0.5rem; }
        .table th, .table td { vertical-align: middle; font-size: 14px; }
        .badge-active { background-color: #28a745; }
        .badge-inactive { background-color: #6c757d; }
        .btn-group .btn { margin-right: 5px; }
        .btn-group .btn:last-child { margin-right: 0; }
        
        @media (max-width: 768px) {
            .btn-group {
                display: flex;
                flex-direction: column;
                gap: 5px;
            }
            .btn-group .btn {
                margin-right: 0;
                font-size: 12px;
            }
        }
    </style>
</head>
<body id="page-top">
<div id="wrapper">
<?php include_once('includes/sidebar.php');?>
<div id="content-wrapper" class="d-flex flex-column">
<div id="content">
<?php include_once('includes/topbar.php');?>
<div class="container-fluid">

    <!-- Page Heading -->
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h3 mb-0 text-gray-800">
            <i class="fas fa-users text-primary"></i> Employee Management
        </h1>
        <div>
            <a href="customer-prescriptions.php" class="btn btn-secondary btn-sm">
                <i class="fas fa-arrow-left"></i> Back to Prescriptions
            </a>
        </div>
    </div>

    <!-- Success/Error Messages -->
    <?php if($success_msg): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle mr-2"></i><?php echo htmlspecialchars($success_msg); ?>
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                <span aria-hidden="true">&times;</span>
            </button>
        </div>
    <?php endif; ?>

    <?php if($error_msg): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-triangle mr-2"></i><?php echo htmlspecialchars($error_msg); ?>
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                <span aria-hidden="true">&times;</span>
            </button>
        </div>
    <?php endif; ?>

    <div class="row">
        <!-- Employee Form -->
        <div class="col-lg-4 mb-4">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h6 class="m-0 font-weight-bold">
                        <i class="fas fa-user-plus mr-2"></i>
                        <?php echo $edit_employee ? 'Edit Employee' : 'Add New Employee'; ?>
                    </h6>
                </div>
                <div class="card-body">
                    <form method="POST" action="">
                        <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                        <input type="hidden" name="employee_id" value="<?php echo $edit_employee ? $edit_employee['id'] : ''; ?>">
                        
                        <div class="form-group">
                            <label class="small font-weight-bold">Employee Name <span class="text-danger">*</span></label>
                            <input type="text" name="employee_name" class="form-control" 
                                   value="<?php echo $edit_employee ? htmlspecialchars($edit_employee['employee_name']) : ''; ?>" 
                                   required maxlength="100" placeholder="Enter employee name">
                        </div>
                        
                        <div class="form-group">
                            <label class="small font-weight-bold">Mobile Number <span class="text-danger">*</span></label>
                            <input type="text" name="mobile" class="form-control" 
                                   value="<?php echo $edit_employee ? htmlspecialchars($edit_employee['mobile']) : ''; ?>" 
                                   required maxlength="15" pattern="[6-9][0-9]{9}" 
                                   placeholder="Enter 10-digit mobile number">
                            <small class="text-muted">Example: 9876543210</small>
                        </div>
                        
                        <div class="form-group">
                            <label class="small font-weight-bold">Status</label>
                            <select name="status" class="form-control">
                                <option value="Active" <?php echo ($edit_employee && $edit_employee['status'] == 'Active') ? 'selected' : ''; ?>>Active</option>
                                <option value="Inactive" <?php echo ($edit_employee && $edit_employee['status'] == 'Inactive') ? 'selected' : ''; ?>>Inactive</option>
                            </select>
                        </div>
                        
                        <div class="form-group mb-0">
                            <button type="submit" name="save_employee" class="btn btn-primary btn-block">
                                <i class="fas fa-save mr-2"></i>
                                <?php echo $edit_employee ? 'Update Employee' : 'Add Employee'; ?>
                            </button>
                            
                            <?php if($edit_employee): ?>
                            <a href="employees.php" class="btn btn-secondary btn-block mt-2">
                                <i class="fas fa-times mr-2"></i>Cancel Edit
                            </a>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Employee List -->
        <div class="col-lg-8">
            <div class="card">
                <div class="card-header bg-dark text-white">
                    <h6 class="m-0 font-weight-bold">
                        <i class="fas fa-list mr-2"></i>Employee List (<?php echo count($employees); ?>)
                    </h6>
                </div>
                <div class="card-body p-0">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover mb-0">
                            <thead class="thead-light">
                                <tr>
                                    <th>#</th>
                                    <th>Employee Name</th>
                                    <th>Mobile Number</th>
                                    <th>Status</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if(empty($employees)): ?>
                                <tr>
                                    <td colspan="6" class="text-center text-muted py-4">
                                        <i class="fas fa-users fa-3x mb-3 text-muted"></i><br>
                                        No employees found. Add your first employee above.
                                    </td>
                                </tr>
                                <?php else: ?>
                                    <?php foreach($employees as $index => $emp): ?>
                                    <tr class="<?php echo $emp['status'] == 'Inactive' ? 'table-secondary' : ''; ?>">
                                        <td><?php echo $index + 1; ?></td>
                                        <td>
                                            <strong><?php echo htmlspecialchars($emp['employee_name']); ?></strong>
                                            <?php if($emp['status'] == 'Inactive'): ?>
                                                <br><small class="text-muted">(Inactive)</small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <i class="fas fa-phone text-muted mr-1"></i>
                                            <?php echo htmlspecialchars($emp['mobile']); ?>
                                        </td>
                                        <td>
                                            <span class="badge badge-<?php echo $emp['status'] == 'Active' ? 'success' : 'secondary'; ?>">
                                                <?php echo $emp['status']; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php echo date('d-M-Y', strtotime($emp['created_at'])); ?>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <a href="employees.php?edit=<?php echo $emp['id']; ?>" 
                                                   class="btn btn-info" title="Edit">
                                                    <i class="fas fa-edit"></i>
                                                </a>
                                                
                                                <form method="POST" style="display: inline-block;" 
                                                      onsubmit="return confirm('Are you sure you want to delete this employee?\\n\\nNote: If employee has existing prescriptions, they will be marked as inactive instead of deleted.');">
                                                    <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                                                    <input type="hidden" name="employee_id" value="<?php echo $emp['id']; ?>">
                                                    <button type="submit" name="delete_employee" 
                                                            class="btn btn-danger" title="Delete">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

</div>
</div>
</div>
<?php include_once('includes/footer.php');?>
</div>
</div>
<?php include_once('includes/footer2.php');?>

<script src="vendor/jquery/jquery.min.js"></script>
<script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
<script src="vendor/jquery-easing/jquery.easing.min.js"></script>
<script src="js/sb-admin-2.min.js"></script>

<script>
$(document).ready(function() {
    // Auto-format mobile number input
    $('input[name="mobile"]').on('input', function() {
        let value = $(this).val().replace(/\D/g, ''); // Remove non-digits
        if (value.length > 10) {
            value = value.substring(0, 10); // Limit to 10 digits
        }
        $(this).val(value);
    });
    
    // Form validation
    $('form').on('submit', function(e) {
        const mobile = $('input[name="mobile"]').val();
        const name = $('input[name="employee_name"]').val().trim();
        
        if (name.length < 2) {
            alert('Employee name must be at least 2 characters long.');
            e.preventDefault();
            return false;
        }
        
        if (!/^[6-9]\d{9}$/.test(mobile)) {
            alert('Please enter a valid 10-digit mobile number starting with 6, 7, 8, or 9.');
            e.preventDefault();
            return false;
        }
    });
});
</script>
</body>
</html>