<?php
session_start();
include_once('includes/config.php');

// Determine user type and ID
$isAdmin = isset($_SESSION['aid']);
$userId = $isAdmin ? $_SESSION['aid'] : (isset($_SESSION['uid']) ? $_SESSION['uid'] : null);
$userType = $isAdmin ? 'admin' : (isset($_SESSION['usertype']) ? $_SESSION['usertype'] : null);

if (!$userId) {
    header('location:login.php');
    exit();
}

// --- FILTERS ---
function getDateRange($filter) {
    $today = date('Y-m-d');
    switch ($filter) {
        case 'today':
            return [$today, $today];
        case 'yesterday':
            $yesterday = date('Y-m-d', strtotime('-1 day'));
            return [$yesterday, $yesterday];
        case 'this_week':
            $start = date('Y-m-d', strtotime('monday this week'));
            $end = date('Y-m-d', strtotime('sunday this week'));
            return [$start, $end];
        case 'this_month':
            $start = date('Y-m-01');
            $end = date('Y-m-t');
            return [$start, $end];
        case 'this_year':
            $start = date('Y-01-01');
            $end = date('Y-12-31');
            return [$start, $end];
        case 'custom':
            $start = isset($_GET['from']) ? $_GET['from'] : $today;
            $end = isset($_GET['to']) ? $_GET['to'] : $today;
            return [$start, $end];
        default:
            return [null, null]; // all time
    }
}

// --- Excel Export ---
if (isset($_GET['export']) && $_GET['export'] == 'excel') {
    $filter = isset($_GET['filter']) ? $_GET['filter'] : 'this_month';
    list($from, $to) = getDateRange($filter);
    $where = "WHERE 1=1";
    if ($from && $to) {
        $where .= " AND DATE(created_at) BETWEEN '$from' AND '$to'";
    }
    
    // Status Filter
    $statusFilter = isset($_GET['status_filter']) ? $_GET['status_filter'] : '';
    if ($statusFilter !== '') {
        $where .= " AND order_status = '" . mysqli_real_escape_string($con, $statusFilter) . "'";
    }
    
    // Frame Type Filter
    $frameTypeFilter = isset($_GET['frame_type_filter']) ? $_GET['frame_type_filter'] : '';
    if ($frameTypeFilter !== '') {
        $where .= " AND frame_type = '" . mysqli_real_escape_string($con, $frameTypeFilter) . "'";
    }
    
    if (!$isAdmin) {
        $where .= " AND user_id = '$userId'";
    }
    
    $query = "SELECT 
                order_number,
                customer_name,
                mobile,
                address,
                CONCAT(right_eye_sph, '/', right_eye_cyl, '/', right_eye_axis) as right_eye_power,
                CONCAT(left_eye_sph, '/', left_eye_cyl, '/', left_eye_axis) as left_eye_power,
                right_eye_addition,
                left_eye_addition,
                frame_name,
                lens_name,
                frame_type,
                lens_type,
                total_amount,
                advance_amount,
                discount_type,
                discount_value,
                discount_amount,
                balance_amount,
                order_status,
                delivery_date,
                notes,
                created_at
              FROM customer_prescriptions 
              $where 
              ORDER BY created_at DESC";
    
    $records = mysqli_query($con, $query);

    header("Content-Type: application/vnd.ms-excel");
    header("Content-Disposition: attachment; filename=prescription_report.xls");
    echo "Order Number\tCustomer Name\tMobile\tAddress\tRight Eye\tLeft Eye\tRight ADD\tLeft ADD\tFrame Name\tLens Name\tFrame Type\tLens Type\tTotal Amount\tDiscount\tFinal Amount\tAdvance\tBalance\tStatus\tDelivery Date\tNotes\tCreated Date\n";
    while($row = mysqli_fetch_assoc($records)) {
        echo ($row['order_number'] ?: 'N/A')."\t";
        echo $row['customer_name']."\t";
        echo $row['mobile']."\t";
        echo $row['address']."\t";
        echo $row['right_eye_power']."\t";
        echo $row['left_eye_power']."\t";
        echo ($row['right_eye_addition'] ?: '-')."\t";
        echo ($row['left_eye_addition'] ?: '-')."\t";
        echo $row['frame_name']."\t";
        echo ($row['lens_name'] ?: '-')."\t";
        echo $row['frame_type']."\t";
        echo $row['lens_type']."\t";
        echo $row['total_amount']."\t";
        echo ($row['discount_amount'] ?: '0')."\t";
        echo ($row['total_amount'] - ($row['discount_amount'] ?: 0))."\t";
        echo $row['advance_amount']."\t";
        echo $row['balance_amount']."\t";
        echo $row['order_status']."\t";
        echo ($row['delivery_date'] ? $row['delivery_date'] : "")."\t";
        echo $row['notes']."\t";
        echo $row['created_at']."\n";
    }
    exit();
}

// --- Default filter: today ---
$filter = isset($_GET['filter']) ? $_GET['filter'] : 'today';
list($from, $to) = getDateRange($filter);

$where = "WHERE 1=1";
if ($from && $to) {
    $where .= " AND DATE(created_at) BETWEEN '$from' AND '$to'";
}

// Status Filter
$statusFilter = isset($_GET['status_filter']) ? $_GET['status_filter'] : '';
if ($statusFilter !== '') {
    $where .= " AND order_status = '" . mysqli_real_escape_string($con, $statusFilter) . "'";
}

// Frame Type Filter
$frameTypeFilter = isset($_GET['frame_type_filter']) ? $_GET['frame_type_filter'] : '';
if ($frameTypeFilter !== '') {
    $where .= " AND frame_type = '" . mysqli_real_escape_string($con, $frameTypeFilter) . "'";
}

// Search Filters
$searchCustomer = isset($_GET['search_customer']) ? trim($_GET['search_customer']) : '';
$searchMobile = isset($_GET['search_mobile']) ? trim($_GET['search_mobile']) : '';

if ($searchCustomer !== '') {
    $where .= " AND customer_name LIKE '%" . mysqli_real_escape_string($con, $searchCustomer) . "%'";
}
if ($searchMobile !== '') {
    $where .= " AND mobile LIKE '%" . mysqli_real_escape_string($con, $searchMobile) . "%'";
}

if (!$isAdmin) {
    $where .= " AND user_id = '$userId'";
}

// --- Sorting ---
$sortable = ['customer_name','mobile','frame_name','total_amount','balance_amount','order_status','created_at','delivery_date'];
$sort = isset($_GET['sort']) && in_array($_GET['sort'],$sortable) ? $_GET['sort'] : 'created_at';
$order = (isset($_GET['order']) && strtolower($_GET['order'])=='asc') ? 'ASC' : 'DESC';

$orderBy = "ORDER BY $sort $order";

// --- DATA QUERIES ---
$query = "SELECT cp.*, u.FullName as staff_name 
          FROM customer_prescriptions cp
          LEFT JOIN tblusers u ON cp.user_id = u.ID
          $where $orderBy";
$records = mysqli_query($con, $query);
$prescriptions = [];
while ($row = mysqli_fetch_assoc($records)) {
    $prescriptions[] = $row;
}

// --- STATISTICS ---
// Total counts
$totalQuery = "SELECT 
    COUNT(*) as total_prescriptions,
    SUM(total_amount) as total_sales,
    SUM(advance_amount) as total_advances,
    SUM(balance_amount) as total_outstanding
    FROM customer_prescriptions cp
    LEFT JOIN tblusers u ON cp.user_id = u.ID
    $where";
$totalResult = mysqli_query($con, $totalQuery);
$totals = mysqli_fetch_assoc($totalResult);

// Status counts
$statusQuery = "SELECT 
    order_status,
    COUNT(*) as count,
    SUM(total_amount) as amount
    FROM customer_prescriptions cp
    LEFT JOIN tblusers u ON cp.user_id = u.ID
    $where
    GROUP BY order_status";
$statusResult = mysqli_query($con, $statusQuery);
$statusCounts = [];
while ($row = mysqli_fetch_assoc($statusResult)) {
    $statusCounts[$row['order_status']] = $row;
}

// Frame type breakdown
$frameQuery = "SELECT 
    frame_type,
    COUNT(*) as count
    FROM customer_prescriptions cp
    LEFT JOIN tblusers u ON cp.user_id = u.ID
    $where
    GROUP BY frame_type";
$frameResult = mysqli_query($con, $frameQuery);
$frameCounts = [];
while ($row = mysqli_fetch_assoc($frameResult)) {
    $frameCounts[$row['frame_type']] = $row['count'];
}

// --- Helper for sort links ---
function sortLink($label, $col, $currentSort, $currentOrder, $params) {
    $order = ($currentSort == $col && $currentOrder == 'ASC') ? 'desc' : 'asc';
    $icon = '';
    if ($currentSort == $col) {
        $icon = $currentOrder == 'ASC' ? ' <i class="fas fa-sort-up"></i>' : ' <i class="fas fa-sort-down"></i>';
    }
    $url = $_SERVER['PHP_SELF'].'?'.http_build_query(array_merge($params, ['sort'=>$col,'order'=>$order]));
    return "<a href=\"$url\" class=\"text-dark\">$label$icon</a>";
}
$params = $_GET;
unset($params['sort'], $params['order']);

// Get available statuses and frame types for filter dropdown
$statuses = ['Pending', 'In Progress', 'Ready', 'Delivered', 'Cancelled'];
$frameTypes = ['Metal Frame', 'Cell Frame', 'Supra Frame', 'Sunglass', 'Rimless', 'Semi-Rimless', 'Others'];

// --- Helper for Indian number formatting ---
function formatIndianCurrency($number) {
    $number = (float)$number;
    $negative = $number < 0;
    $number = abs($number);
    if ($number < 100000) {
        $result = number_format($number, 2);
    } else if ($number < 10000000) {
        $result = number_format($number / 100000, 2) . ' Lakh';
    } else {
        $result = number_format($number / 10000000, 2) . ' Crore';
    }
    return ($negative ? '-' : '') . $result;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Jagadamba Opticals | Prescription Dashboard</title>
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <style>
        .card { border-radius: 1rem; }
        .card-header { border-radius: 1rem 1rem 0 0; }
        .summary-card { background: linear-gradient(90deg, #43cea2 0%, #185a9d 100%); color: #fff; }
        .summary-value { font-size: 2rem; font-weight: bold; }
        .filter-btn { margin: 0.2rem 0.2rem 0.2rem 0; }
        .table th, .table td { vertical-align: middle !important; }
        .prescription-card { 
            transition: all 0.3s ease; 
            border: 1px solid #e3e6f0; 
            border-radius: 0.75rem; 
            overflow: hidden; 
        }
        .prescription-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            border-color: #4e73df;
        }
        .badge-pending { background-color: #ffc107; color: #212529; }
        .badge-progress { background-color: #17a2b8; }
        .badge-ready { background-color: #28a745; }
        .badge-delivered { background-color: #6c757d; }
        .badge-cancelled { background-color: #dc3545; }
        .search-input {
            border-radius: 1.5rem;
            font-size: 1rem;
            padding: 0.5rem 1rem;
            margin-bottom: 0.5rem;
            width: 100%;
            border: 1px solid #43cea2;
        }
        @media (max-width: 767.98px) {
            .container, .container-fluid { padding-left: 0.5rem; padding-right: 0.5rem; }
            .card { margin-bottom: 1.2rem; }
            .table-responsive { font-size: 0.95rem; }
            .btn, .btn-block { font-size: 1rem; padding: 0.6rem 0.75rem; }
            .card-header h4, .card-header h5 { font-size: 1.1rem; }
            .search-input { font-size: 0.98rem; padding: 0.45rem 0.9rem; }
            .card-header form.form-row { flex-direction: column; }
            .card-header form.form-row > div { width: 100%; }
        }
    </style>
    <script>
    document.addEventListener("DOMContentLoaded", function() {
        // Auto submit for status and frame type filter (dropdowns only)
        var filterSelects = document.querySelectorAll('.filter-select');
        filterSelects.forEach(function(select) {
            select.addEventListener('change', function() {
                var form = select.closest('form');
                if(form) form.submit();
            });
        });
    });
    </script>
</head>
<body id="page-top">
    <div id="wrapper">
        <?php include_once('includes/sidebar.php');?>
        <div id="content-wrapper" class="d-flex flex-column">
            <div id="content">
                <?php include_once('includes/topbar.php');?>
                <div class="container-fluid mt-2 mb-2">
                   
                    <!-- Prescription Summary -->
                    <div class="row mb-4">
                        <div class="col-md-3 mb-2">
                            <div class="card summary-card shadow text-center" style="background: linear-gradient(90deg, #6a11cb 0%, #2575fc 100%);">
                                <div class="card-body">
                                    <div>Total Prescriptions</div>
                                    <div class="summary-value"><?php echo $totals['total_prescriptions'] ?: 0; ?></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3 mb-2">
                            <div class="card summary-card shadow text-center">
                                <div class="card-body">
                                    <div>Total Sales</div>
                                    <div class="summary-value">Rs. <?php echo formatIndianCurrency($totals['total_sales'] ?: 0); ?></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3 mb-2">
                            <div class="card summary-card shadow text-center" style="background: linear-gradient(90deg, #f7971e 0%, #ffd200 100%); color: #333;">
                                <div class="card-body">
                                    <div>Advances Collected</div>
                                    <div class="summary-value">Rs. <?php echo formatIndianCurrency($totals['total_advances'] ?: 0); ?></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3 mb-2">
                            <div class="card summary-card shadow text-center" style="background: linear-gradient(90deg, #43cea2 0%, #185a9d 100%);">
                                <div class="card-body">
                                    <div>Outstanding Balance</div>
                                    <div class="summary-value">Rs. <?php echo formatIndianCurrency($totals['total_outstanding'] ?: 0); ?></div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Order Status Summary -->
                    <div class="row mb-4">
                        <?php foreach($statuses as $status): ?>
                        <div class="col-md-2 col-sm-4 mb-2">
                            <div class="card border-left-primary shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="row no-gutters align-items-center">
                                        <div class="col mr-2">
                                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">
                                                <?php echo $status; ?>
                                            </div>
                                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                                <?php echo isset($statusCounts[$status]) ? $statusCounts[$status]['count'] : 0; ?>
                                            </div>
                                            <div class="text-xs text-muted">
                                                Rs. <?php echo isset($statusCounts[$status]) ? number_format($statusCounts[$status]['amount'], 0) : 0; ?>
                                            </div>
                                        </div>
                                        <div class="col-auto">
                                            <?php 
                                            $icon = 'fas fa-clock';
                                            $color = 'text-warning';
                                            if($status == 'In Progress') { $icon = 'fas fa-cog fa-spin'; $color = 'text-info'; }
                                            else if($status == 'Ready') { $icon = 'fas fa-check-circle'; $color = 'text-success'; }
                                            else if($status == 'Delivered') { $icon = 'fas fa-shipping-fast'; $color = 'text-secondary'; }
                                            else if($status == 'Cancelled') { $icon = 'fas fa-times-circle'; $color = 'text-danger'; }
                                            ?>
                                            <i class="<?php echo $icon; ?> fa-2x <?php echo $color; ?>"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                        
                        <!-- Frame Type Breakdown -->
                        <div class="col-md-2 col-sm-4 mb-2">
                            <div class="card border-left-success shadow h-100 py-2">
                                <div class="card-body">
                                    <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                        Frame Types
                                    </div>
                                    <?php foreach($frameCounts as $frameType => $count): ?>
                                    <div class="small text-muted"><?php echo $frameType; ?>: <?php echo $count; ?></div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Filter Options -->
                    <div class="card shadow mb-4">
                        <div class="card-header bg-gradient-primary text-white">
                            <form class="form-inline flex-wrap" method="get" action="">
                                <div class="date-row mb-2">
                                    <button type="submit" name="filter" value="all" class="btn btn-info filter-btn <?php if($filter=='all') echo 'font-weight-bold'; ?>">All Time</button>
                                    <button type="submit" name="filter" value="today" class="btn btn-primary filter-btn <?php if($filter=='today') echo 'font-weight-bold'; ?>">Today</button>
                                    <button type="submit" name="filter" value="yesterday" class="btn btn-primary filter-btn <?php if($filter=='yesterday') echo 'font-weight-bold'; ?>">Yesterday</button>
                                    <button type="submit" name="filter" value="this_week" class="btn btn-primary filter-btn <?php if($filter=='this_week') echo 'font-weight-bold'; ?>">This Week</button>
                                    <button type="submit" name="filter" value="this_month" class="btn btn-primary filter-btn <?php if($filter=='this_month') echo 'font-weight-bold'; ?>">This Month</button>
                                    <button type="submit" name="filter" value="this_year" class="btn btn-primary filter-btn <?php if($filter=='this_year') echo 'font-weight-bold'; ?>">This Year</button>
                                    <span class="ml-2 mr-2 font-weight-bold">Custom:</span>
                                    <input type="date" name="from" class="form-control filter-btn" value="<?php echo isset($_GET['from']) ? $_GET['from'] : ''; ?>">
                                    <input type="date" name="to" class="form-control filter-btn" value="<?php echo isset($_GET['to']) ? $_GET['to'] : ''; ?>">
                                    <button type="submit" name="filter" value="custom" class="btn btn-success filter-btn <?php if($filter=='custom') echo 'font-weight-bold'; ?>">Apply</button>
                                </div>
                                <div class="filter-row mb-2 w-100">
                                    <div class="d-flex flex-wrap align-items-center">
                                        <select name="status_filter" class="form-control filter-btn filter-select">
                                            <option value="">All Status</option>
                                            <?php foreach($statuses as $status) { ?>
                                                <option value="<?php echo $status; ?>" <?php if($statusFilter==$status) echo 'selected'; ?>><?php echo $status; ?></option>
                                            <?php } ?>
                                        </select>
                                        <select name="frame_type_filter" class="form-control filter-btn filter-select">
                                            <option value="">All Frame Types</option>
                                            <?php foreach($frameTypes as $frameType) { ?>
                                                <option value="<?php echo $frameType; ?>" <?php if($frameTypeFilter==$frameType) echo 'selected'; ?>><?php echo $frameType; ?></option>
                                            <?php } ?>
                                        </select>
                                        <input type="text" name="search_customer" class="form-control filter-btn" placeholder="Search Customer" value="<?php echo htmlspecialchars($searchCustomer); ?>" style="max-width: 150px;">
                                        <input type="text" name="search_mobile" class="form-control filter-btn" placeholder="Search Mobile" value="<?php echo htmlspecialchars($searchMobile); ?>" style="max-width: 150px;">
                                        <button type="submit" class="btn btn-info filter-btn"><i class="fas fa-filter"></i> Filter</button>
                                        <a href="?<?php echo http_build_query(array_merge($_GET, ['export'=>'excel'])); ?>" class="btn btn-warning filter-btn ml-auto"><i class="fas fa-file-excel"></i> Export Excel</a>
                                        <a href="customer-prescriptions.php" class="btn btn-success filter-btn"><i class="fas fa-plus"></i> Add New</a>
                                    </div>
                                </div>
                                <!-- Preserve other filters -->
                                <?php foreach(['sort','order'] as $keep) if(isset($_GET[$keep])) echo '<input type="hidden" name="'.$keep.'" value="'.htmlspecialchars($_GET[$keep]).'">'; ?>
                            </form>
                        </div>
                    </div>

                    <!-- Prescription Records -->
                    <div class="card shadow mb-4">
                        <div class="card-header py-3 d-flex justify-content-between align-items-center">
                            <h6 class="m-0 font-weight-bold text-primary">
                                <i class="fas fa-glasses"></i> Prescription Records (<?php echo count($prescriptions); ?>)
                            </h6>
                        </div>
                        <div class="card-body">
                            <?php if(empty($prescriptions)): ?>
                                <div class="text-center py-4">
                                    <i class="fas fa-glasses fa-3x text-muted mb-3"></i>
                                    <h5 class="text-muted">No prescriptions found</h5>
                                    <p class="text-muted">No prescriptions found for the selected filters</p>
                                    <a href="customer-prescriptions.php" class="btn btn-primary">
                                        <i class="fas fa-plus"></i> Add First Prescription
                                    </a>
                                </div>
                            <?php else: ?>
                                <div class="table-responsive">
                                    <table class="table table-bordered table-hover">
                                        <thead class="thead-light">
                                            <tr>
                                                <th>Order #</th>
                                                <th><?php echo sortLink('Customer', 'customer_name', $sort, $order, $params); ?></th>
                                                <th><?php echo sortLink('Mobile', 'mobile', $sort, $order, $params); ?></th>
                                                <th><?php echo sortLink('Frame', 'frame_name', $sort, $order, $params); ?></th>
                                                <th>Eye Power</th>
                                                <th><?php echo sortLink('Amount', 'total_amount', $sort, $order, $params); ?></th>
                                                <th><?php echo sortLink('Balance', 'balance_amount', $sort, $order, $params); ?></th>
                                                <th><?php echo sortLink('Status', 'order_status', $sort, $order, $params); ?></th>
                                                <th><?php echo sortLink('Date', 'created_at', $sort, $order, $params); ?></th>
                                                <?php if($isAdmin): ?><th>Staff</th><?php endif; ?>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach($prescriptions as $row): ?>
                                            <tr>
                                                <td>
                                                    <?php if($row['order_number']): ?>
                                                    <small class="text-primary font-weight-bold"><?php echo htmlspecialchars($row['order_number']); ?></small>
                                                    <?php else: ?>
                                                    <small class="text-muted">-</small>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($row['customer_name']); ?></strong>
                                                    <?php if($row['address']): ?>
                                                    <br><small class="text-muted"><?php echo htmlspecialchars(substr($row['address'], 0, 30)); ?>...</small>
                                                    <?php endif; ?>
                                                </td>
                                                <td><?php echo htmlspecialchars($row['mobile']); ?></td>
                                                <td>
                                                    <strong><?php echo htmlspecialchars($row['frame_name']); ?></strong>
                                                    <?php if($row['lens_name']): ?>
                                                    <br><small class="text-info"><?php echo htmlspecialchars($row['lens_name']); ?></small>
                                                    <?php endif; ?>
                                                    <br><small class="text-muted"><?php echo htmlspecialchars($row['frame_type']); ?> | <?php echo htmlspecialchars($row['lens_type']); ?></small>
                                                </td>
                                                <td>
                                                    <small>
                                                        <strong>R:</strong> <?php echo htmlspecialchars($row['right_eye_sph'] ?: '-'); ?>/<?php echo htmlspecialchars($row['right_eye_cyl'] ?: '-'); ?>/<?php echo htmlspecialchars($row['right_eye_axis'] ?: '-'); ?>
                                                        <?php if($row['right_eye_addition']): ?><br><strong>ADD:</strong> <?php echo htmlspecialchars($row['right_eye_addition']); ?><?php endif; ?>
                                                        <br><strong>L:</strong> <?php echo htmlspecialchars($row['left_eye_sph'] ?: '-'); ?>/<?php echo htmlspecialchars($row['left_eye_cyl'] ?: '-'); ?>/<?php echo htmlspecialchars($row['left_eye_axis'] ?: '-'); ?>
                                                        <?php if($row['left_eye_addition']): ?><br><strong>ADD:</strong> <?php echo htmlspecialchars($row['left_eye_addition']); ?><?php endif; ?>
                                                    </small>
                                                </td>
                                                <td class="text-right">
                                                    <strong>₹<?php echo number_format($row['total_amount'], 2); ?></strong>
                                                    <?php if($row['discount_amount'] > 0): ?>
                                                    <br><small class="text-success">Disc: -₹<?php echo number_format($row['discount_amount'], 2); ?></small>
                                                    <br><small class="text-info">Final: ₹<?php echo number_format($row['total_amount'] - $row['discount_amount'], 2); ?></small>
                                                    <?php endif; ?>
                                                    <br><small class="text-muted">Adv: ₹<?php echo number_format($row['advance_amount'], 2); ?></small>
                                                </td>
                                                <td class="text-right">
                                                    <strong class="<?php echo $row['balance_amount'] > 0 ? 'text-danger' : 'text-success'; ?>">
                                                        ₹<?php echo number_format($row['balance_amount'], 2); ?>
                                                    </strong>
                                                </td>
                                                <td>
                                                    <?php
                                                        $badgeClass = 'badge-warning';
                                                        if ($row['order_status'] == 'In Progress') $badgeClass = 'badge-info';
                                                        else if ($row['order_status'] == 'Ready') $badgeClass = 'badge-success';
                                                        else if ($row['order_status'] == 'Delivered') $badgeClass = 'badge-secondary';
                                                        else if ($row['order_status'] == 'Cancelled') $badgeClass = 'badge-danger';
                                                    ?>
                                                    <span class="badge <?php echo $badgeClass; ?>">
                                                        <?php echo htmlspecialchars($row['order_status']); ?>
                                                    </span>
                                                    <?php if($row['delivery_date']): ?>
                                                    <br><small class="text-muted">Due: <?php echo date('d-M-Y', strtotime($row['delivery_date'])); ?></small>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php echo date('d-M-Y', strtotime($row['created_at'])); ?>
                                                    <br><small class="text-muted"><?php echo date('H:i', strtotime($row['created_at'])); ?></small>
                                                </td>
                                                <?php if($isAdmin): ?>
                                                <td><small><?php echo htmlspecialchars($row['staff_name']); ?></small></td>
                                                <?php endif; ?>
                                                <td>
                                                    <a href="customer-prescription-details.php?id=<?php echo $row['id']; ?>" class="btn btn-sm btn-success" title="View Details">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                    <a href="customer-prescriptions.php?edit=<?php echo $row['id']; ?>" class="btn btn-sm btn-info" title="Edit">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <a href="print-voucher.php?id=<?php echo $row['id']; ?>" class="btn btn-sm btn-warning" title="Print" target="_blank">
                                                        <i class="fas fa-print"></i>
                                                    </a>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                 
                </div>
            </div>
            <?php include_once('includes/footer.php');?>
        </div>
    </div>
    <?php include_once('includes/footer2.php');?>
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>
    <script src="js/sb-admin-2.min.js"></script>
</body>
</html>