<?php
session_start();
include_once('../includes/config.php');

// Helper function to safely escape HTML content
function safe_htmlspecialchars($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// Check if user is logged in
if (!isset($_SESSION['uid'])) {
    header('location:../login.php');
    exit();
}

$userId = $_SESSION['uid'];
$userType = $_SESSION['usertype'];

// Get customer ID and date range
$customerId = $_GET['customer_id'] ?? '';
$dateFrom = $_GET['date_from'] ?? date('Y-m-01'); // First day of current month
$dateTo = $_GET['date_to'] ?? date('Y-m-d'); // Today

if (empty($customerId)) {
    header('location: customers.php');
    exit();
}

// Get customer details
$customerQuery = "SELECT * FROM tblcustomers WHERE ID = " . intval($customerId);
$customerResult = mysqli_query($con, $customerQuery);
if (mysqli_num_rows($customerResult) == 0) {
    header('location: customers.php');
    exit();
}
$customer = mysqli_fetch_assoc($customerResult);

// Get opening balance (balance before date range)
$openingBalanceQuery = "SELECT 
    COALESCE(SUM(
        CASE 
            WHEN TransactionType IN ('Invoice', 'Debit Note') THEN Amount
            WHEN TransactionType IN ('Payment', 'Credit Note') THEN -Amount
            ELSE 0
        END
    ), 0) as opening_balance
    FROM tblcustomer_accounts 
    WHERE CustomerID = " . intval($customerId) . " 
    AND TransactionDate < '$dateFrom'";

$openingResult = mysqli_query($con, $openingBalanceQuery);
$openingBalance = 0;
if ($openingRow = mysqli_fetch_assoc($openingResult)) {
    $openingBalance = floatval($openingRow['opening_balance']);
}

// Get transactions for the period
$transactionsQuery = "SELECT 
    ca.*,
    u.FullName as AddedByName
    FROM tblcustomer_accounts ca
    LEFT JOIN tblusers u ON ca.AddedBy = u.ID
    WHERE ca.CustomerID = " . intval($customerId) . "
    AND ca.TransactionDate BETWEEN '$dateFrom' AND '$dateTo'
    ORDER BY ca.TransactionDate ASC, ca.CreatedDate ASC";

$transactionsResult = mysqli_query($con, $transactionsQuery);

// Calculate running balance
$runningBalance = $openingBalance;
$transactions = [];
while ($row = mysqli_fetch_assoc($transactionsResult)) {
    if ($row['TransactionType'] == 'Invoice' || $row['TransactionType'] == 'Debit Note') {
        $runningBalance += $row['Amount'];
    } else {
        $runningBalance -= $row['Amount'];
    }
    $row['running_balance'] = $runningBalance;
    $transactions[] = $row;
}

$closingBalance = $runningBalance;

// Get period totals
$totalDebits = 0;
$totalCredits = 0;
foreach ($transactions as $transaction) {
    if ($transaction['TransactionType'] == 'Invoice' || $transaction['TransactionType'] == 'Debit Note') {
        $totalDebits += $transaction['Amount'];
    } else {
        $totalCredits += $transaction['Amount'];
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <title>Customer Ledger - <?php echo safe_htmlspecialchars($customer['CustomerName']); ?></title>
    
    <!-- Custom fonts for this template-->
    <link href="../vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="https://fonts.googleapis.com/css?family=Nunito:200,200i,300,300i,400,400i,600,600i,700,700i,800,800i,900,900i" rel="stylesheet">
    
    <!-- Custom styles for this template-->
    <link href="../css/sb-admin-2.min.css" rel="stylesheet">
    
    <style>
        body {
            font-size: 17px;
            font-family: 'Nunito', Arial, sans-serif;
            background: #f4f6fb;
            line-height: 1.2;
        }
        @media print {
            .no-print { display: none !important; }
            .sidebar, .topbar, .footer { display: none !important; }
            #content-wrapper { margin-left: 0 !important; }
            body { font-size: 15px; font-weight: bold; line-height: 1.1; }
            .container-fluid { padding: 0; }
            .card { border: none; box-shadow: none; }
            .page-break { page-break-before: always; }
            h1, h2, h3, h4, h5 { font-weight: bold; }
            .ledger-table { font-size: 15px; font-weight: bold; }
            .ledger-table th { font-size: 16px; font-weight: bold; }
            .company-header h2 { font-size: 28px; font-weight: bold; }
            .company-header p { font-size: 16px; font-weight: bold; }
            .amount-cell { font-weight: bold; }
        }

        .ledger-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 18px;
            margin-bottom: 18px;
            border-radius: 10px;
            font-size: 22px;
            letter-spacing: 1px;
        }

        .company-header {
            text-align: center;
            border-bottom: 3px solid #4e73df;
            padding-bottom: 12px;
            margin-bottom: 18px;
        }

        .company-header h2 {
            font-size: 28px;
            font-weight: bold;
            letter-spacing: 2px;
        }
        .company-header p {
            font-size: 16px;
            font-weight: 600;
        }

        .ledger-table {
            font-size: 15px;
            border-collapse: separate;
            border-spacing: 0;
            width: 100%;
            font-weight: 600;
            background: #fff;
            box-shadow: 0 2px 8px rgba(0,0,0,0.04);
            border-radius: 8px;
            overflow: hidden;
        }

        .ledger-table th,
        .ledger-table td {
            border: 1px solid #d1d3e2;
            padding: 7px 6px;
            text-align: left;
            font-weight: 600;
        }

        .ledger-table th {
            background-color: #e9ecef;
            font-size: 16px;
            font-weight: bold;
            text-align: center;
            letter-spacing: 1px;
        }

        .ledger-table tr {
            transition: background 0.2s;
        }
        .ledger-table tr:hover {
            background: #f3f6fb;
        }

        .amount-cell {
            text-align: right;
            font-family: 'Courier New', monospace;
            font-size: 16px;
        }

        .debit-amount {
            color: #e74a3b;
        }

        .credit-amount {
            color: #1cc88a;
        }

        .balance-positive {
            color: #e74a3b;
            font-weight: bold;
        }

        .balance-negative {
            color: #1cc88a;
            font-weight: bold;
        }

        .summary-box {
            background: #f8f9fc;
            border: 2px solid #4e73df;
            padding: 12px 8px;
            margin: 18px 0;
            border-radius: 7px;
            font-size: 16px;
            display: flex;
            justify-content: space-between;
            gap: 12px;
        }
        .summary-box > div {
            flex: 1;
            text-align: center;
        }

        .signature-section {
            margin-top: 30px;
            display: flex;
            justify-content: space-between;
        }

        .signature-box {
            text-align: center;
            border-top: 2px solid #000;
            padding-top: 7px;
            width: 180px;
            font-size: 14px;
        }

        .closing-balance-cell {
            background: #4e73df !important;
            color: #fff !important;
            font-size: 18px !important;
            font-weight: 900 !important;
            border: 2px solid #2e59d9 !important;
            letter-spacing: 1px;
        }

        @page {
            size: A4;
            margin: 0.7cm 0.7cm 0.7cm 0.7cm;
        }
    </style>
</head>

<body id="page-top">
    <!-- Page Wrapper -->
    <div id="wrapper">
        <!-- Content Wrapper -->
        <div id="content-wrapper" class="d-flex flex-column" style="margin-left: 0 !important;">
            <!-- Main Content -->
            <div id="content">
                <!-- Topbar -->
                <div class="no-print">
                    <?php include_once('includes/customer-topbar.php'); ?>
                </div>
                <!-- End of Topbar -->

                <!-- Begin Page Content -->
                <div class="container-fluid">
                    <!-- Page Controls -->
                    <div class="no-print">
                        <div class="d-sm-flex align-items-center justify-content-between mb-4">
                            <h1 class="h3 mb-0 text-gray-800">Customer Ledger Report</h1>
                            <div>
                                <button class="btn btn-primary btn-sm" onclick="window.print()">
                                    <i class="fas fa-print"></i> Print Report
                                </button>
                                <a href="customers.php" class="btn btn-secondary btn-sm">
                                    <i class="fas fa-arrow-left"></i> Back to Customers
                                </a>
                            </div>
                        </div>

                        <!-- Date Range Filter -->
                        <div class="card shadow mb-4">
                            <div class="card-body">
                                <form method="GET" class="row align-items-end">
                                    <input type="hidden" name="customer_id" value="<?php echo $customerId; ?>">
                                    <div class="col-md-3">
                                        <label>From Date</label>
                                        <input type="date" name="date_from" class="form-control" value="<?php echo $dateFrom; ?>">
                                    </div>
                                    <div class="col-md-3">
                                        <label>To Date</label>
                                        <input type="date" name="date_to" class="form-control" value="<?php echo $dateTo; ?>">
                                    </div>
                                    <div class="col-md-3">
                                        <button type="submit" class="btn btn-primary">
                                            <i class="fas fa-search"></i> Generate Report
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>

                    <!-- Ledger Report -->
                    <div class="card shadow mb-4">
                        <div class="card-body">
                            <!-- Company Header -->
                            <div class="company-header">
                                <h2 style="margin: 0; color: #4e73df;">IHITA SYSTEMS</h2>
                                <p style="margin: 5px 0;">Customer Account Ledger</p>
                                <p style="margin: 0; font-size: 14px;">Period: <?php echo date('d M Y', strtotime($dateFrom)); ?> to <?php echo date('d M Y', strtotime($dateTo)); ?></p>
                            </div>

                            <!-- Customer Information -->
                            <div style="margin-bottom: 20px;">
                                <div style="display: flex; justify-content: space-between; margin-bottom: 15px;">
                                    <div style="width: 48%;">
                                        <h4 style="color: #4e73df; margin-bottom: 10px;">Customer Details</h4>
                                        <p><strong>Customer Code:</strong> <?php echo safe_htmlspecialchars($customer['CustomerCode']); ?></p>
                                        <p><strong>Customer Name:</strong> <?php echo safe_htmlspecialchars($customer['CustomerName']); ?></p>
                                        <?php if ($customer['CompanyName']): ?>
                                            <p><strong>Company:</strong> <?php echo safe_htmlspecialchars($customer['CompanyName']); ?></p>
                                        <?php endif; ?>
                                        <?php if ($customer['Phone']): ?>
                                            <p><strong>Phone:</strong> <?php echo safe_htmlspecialchars($customer['Phone']); ?></p>
                                        <?php endif; ?>
                                        <?php if ($customer['Email']): ?>
                                            <p><strong>Email:</strong> <?php echo safe_htmlspecialchars($customer['Email']); ?></p>
                                        <?php endif; ?>
                                    </div>
                                    <div style="width: 48%;">
                                        <h4 style="color: #4e73df; margin-bottom: 10px;">Account Summary</h4>
                                        <p><strong>Credit Limit:</strong> ₹<?php echo number_format($customer['CreditLimit'], 2); ?></p>
                                        <p><strong>Current Balance:</strong> ₹<?php echo number_format($customer['CurrentBalance'], 2); ?></p>
                                        <p><strong>Report Generated:</strong> <?php echo date('d M Y, h:i A'); ?></p>
                                        <p><strong>Generated By:</strong> <?php echo safe_htmlspecialchars($_SESSION['fullname'] ?? 'System'); ?></p>
                                    </div>
                                </div>
                            </div>

                            <!-- Balance Summary -->
                            <div class="summary-box">
                                <div style="display: flex; justify-content: space-between;">
                                    <div><strong>Opening Balance:</strong> ₹<?php echo number_format($openingBalance, 2); ?></div>
                                    <div><strong>Total Debits:</strong> <span class="debit-amount">₹<?php echo number_format($totalDebits, 2); ?></span></div>
                                    <div><strong>Total Credits:</strong> <span class="credit-amount">₹<?php echo number_format($totalCredits, 2); ?></span></div>
                                    <div><strong>Closing Balance:</strong> ₹<?php echo number_format($closingBalance, 2); ?></div>
                                </div>
                            </div>

                            <!-- Transactions Table -->
                            <table class="ledger-table">
                                <thead>
                                    <tr>
                                        <th style="width: 13%;">Date</th>
                                        <th style="width: 18%;">Reference</th>
                                        <th style="width: 12%;">Type</th>
                                        <th style="width: 40%;">Description</th>
                                        <th style="width: 13%;">Debit (₹)</th>
                                        <th style="width: 13%;">Credit (₹)</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <!-- Opening Balance Row as first row in list -->
                                    <tr style="background-color: #f1f1f1; font-weight: bold;">
                                        <td><?php echo date('d-m-Y', strtotime($dateFrom . ' -1 day')); ?></td>
                                        <td>-</td>
                                        <td>Opening</td>
                                        <td>Opening Balance</td>
                                        <td class="amount-cell">
                                            <?php echo $openingBalance >= 0 ? number_format(abs($openingBalance), 2) : '-'; ?>
                                        </td>
                                        <td class="amount-cell">
                                            <?php echo $openingBalance < 0 ? number_format(abs($openingBalance), 2) : '-'; ?>
                                        </td>
                                    </tr>
                                    <!-- Transaction Rows -->
                                    <?php foreach ($transactions as $transaction): ?>
                                        <tr>
                                            <td><?php echo date('d-m-Y', strtotime($transaction['TransactionDate'])); ?></td>
                                            <td><?php echo safe_htmlspecialchars($transaction['InvoiceNumber']); ?></td>
                                            <td><?php echo safe_htmlspecialchars($transaction['TransactionType']); ?></td>
                                            <td>
                                                <?php echo safe_htmlspecialchars($transaction['Description']); ?>
                                                <?php if ($transaction['PaymentMode']): ?>
                                                    <br><small style="color: #666;">Mode: <?php echo safe_htmlspecialchars($transaction['PaymentMode']); ?></small>
                                                <?php endif; ?>
                                                <?php if ($transaction['ReferenceNumber']): ?>
                                                    <br><small style="color: #666;">Ref: <?php echo safe_htmlspecialchars($transaction['ReferenceNumber']); ?></small>
                                                <?php endif; ?>
                                            </td>
                                            <td class="amount-cell debit-amount">
                                                <?php 
                                                if ($transaction['TransactionType'] == 'Invoice' || $transaction['TransactionType'] == 'Debit Note') {
                                                    echo number_format($transaction['Amount'], 2);
                                                } else {
                                                    echo '-';
                                                }
                                                ?>
                                            </td>
                                            <td class="amount-cell credit-amount">
                                                <?php 
                                                if ($transaction['TransactionType'] == 'Payment' || $transaction['TransactionType'] == 'Credit Note') {
                                                    echo number_format($transaction['Amount'], 2);
                                                } else {
                                                    echo '-';
                                                }
                                                ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>

                                    <?php if (empty($transactions)): ?>
                                        <tr>
                                            <td colspan="6" style="text-align: center; color: #666; font-style: italic;">
                                                No transactions found for the selected period
                                            </td>
                                        </tr>
                                    <?php endif; ?>

                                    <!-- Totals Row -->
                                    <tr style="background-color: #e9ecef; font-weight: bold;">
                                        <td colspan="4" style="text-align:right;">Total</td>
                                        <td class="amount-cell debit-amount"><?php echo number_format($totalDebits, 2); ?></td>
                                        <td class="amount-cell credit-amount"><?php echo number_format($totalCredits, 2); ?></td>
                                    </tr>

                                </tbody>
                            </table>

                            <!-- Closing Balance Row (right aligned) -->
                            <div style="width:100%; margin-top:10px;">
                                <div style="display:flex; justify-content:flex-end;">
                                    <div style="min-width:340px;">
                                        <table class="ledger-table" style="width:100%; margin-bottom:0;">
                                            <tr>
                                                <td class="closing-balance-cell" style="text-align:right;">
                                                    Closing Balance: ₹<?php echo number_format(abs($closingBalance), 2); ?>
                                                    <?php echo $closingBalance >= 0 ? ' Dr' : ' Cr'; ?>
                                                </td>
                                            </tr>
                                        </table>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Notes Section -->
                            <div style="margin-top: 30px; font-size: 11px;">
                                <h5>Notes:</h5>
                                <ul style="margin-left: 20px;">
                                    <li>Dr (Debit) indicates amount receivable from customer</li>
                                    <li>Cr (Credit) indicates amount payable to customer</li>
                                    <li>All amounts are in Indian Rupees (₹)</li>
                                    <li>This is a computer-generated report</li>
                                </ul>
                            </div>

                            <!-- Signature Section -->
                            <div class="signature-section no-print">
                                <div class="signature-box">
                                    <div>Prepared By</div>
                                    <div style="margin-top: 30px;">__________________</div>
                                    <div style="font-size: 10px;"><?php echo safe_htmlspecialchars($_SESSION['fullname'] ?? 'System User'); ?></div>
                                </div>
                                <div class="signature-box">
                                    <div>Verified By</div>
                                    <div style="margin-top: 30px;">__________________</div>
                                    <div style="font-size: 10px;">Accounts Manager</div>
                                </div>
                                <div class="signature-box">
                                    <div>Approved By</div>
                                    <div style="margin-top: 30px;">__________________</div>
                                    <div style="font-size: 10px;">Authorized Signatory</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <!-- /.container-fluid -->
            </div>
            <!-- End of Main Content -->

            <!-- Footer -->
            <div class="no-print">
                <?php include_once('includes/customer-footer.php'); ?>
            </div>
            <!-- End of Footer -->
        </div>
        <!-- End of Content Wrapper -->
    </div>
    <!-- End of Page Wrapper -->

    <!-- Bootstrap core JavaScript-->
    <script src="../vendor/jquery/jquery.min.js"></script>
    <script src="../vendor/bootstrap/js/bootstrap.bundle.min.js"></script>

    <!-- Core plugin JavaScript-->
    <script src="../vendor/jquery-easing/jquery.easing.min.js"></script>

    <!-- Custom scripts for all pages-->
    <script src="../js/sb-admin-2.min.js"></script>
</body>
</html>
