// Customer Module JavaScript

$(document).ready(function() {
    // Initialize DataTables with custom options
    if ($.fn.DataTable) {
        $('.data-table').DataTable({
            "pageLength": 25,
            "ordering": true,
            "searching": true,
            "lengthChange": true,
            "info": true,
            "autoWidth": false,
            "responsive": true,
            "columnDefs": [
                { "orderable": false, "targets": -1 } // Disable ordering on last column (actions)
            ]
        });
    }

    // Auto-calculate totals
    function calculateTotals() {
        var amount = parseFloat($('#amount').val()) || 0;
        var taxAmount = parseFloat($('#tax_amount').val()) || 0;
        var totalAmount = amount + taxAmount;
        
        if ($('#total_amount').length) {
            $('#total_amount').val(totalAmount.toFixed(2));
        }
    }

    // Bind calculation events
    $('#amount, #tax_amount').on('input', calculateTotals);

    // Form validation
    $('form').on('submit', function(e) {
        var isValid = true;
        var errorMessages = [];

        // Validate required fields
        $(this).find('[required]').each(function() {
            if (!$(this).val().trim()) {
                isValid = false;
                $(this).addClass('is-invalid');
                errorMessages.push($(this).prev('label').text() + ' is required');
            } else {
                $(this).removeClass('is-invalid');
            }
        });

        // Validate email format
        $('input[type="email"]').each(function() {
            var email = $(this).val();
            var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (email && !emailRegex.test(email)) {
                isValid = false;
                $(this).addClass('is-invalid');
                errorMessages.push('Please enter a valid email address');
            }
        });

        // Validate amounts
        $('input[type="number"]').each(function() {
            var value = parseFloat($(this).val());
            if ($(this).val() && (isNaN(value) || value < 0)) {
                isValid = false;
                $(this).addClass('is-invalid');
                errorMessages.push($(this).prev('label').text() + ' must be a valid positive number');
            }
        });

        if (!isValid) {
            e.preventDefault();
            showAlert('Please fix the following errors:<br>' + errorMessages.join('<br>'), 'danger');
        }
    });

    // Clear validation on input
    $('input, select, textarea').on('input change', function() {
        $(this).removeClass('is-invalid');
    });

    // Auto-generate reference numbers
    $('#transaction_type').change(function() {
        var type = $(this).val();
        var timestamp = Date.now().toString().slice(-6);
        var prefix = '';
        
        switch(type) {
            case 'Invoice': prefix = 'INV-'; break;
            case 'Payment': prefix = 'PAY-'; break;
            case 'Credit Note': prefix = 'CN-'; break;
            case 'Debit Note': prefix = 'DN-'; break;
        }
        
        if (prefix && !$('#invoice_number').val()) {
            $('#invoice_number').val(prefix + timestamp);
        }
    });

    // Auto-set due date for invoices (30 days from transaction date)
    $('#transaction_type, #transaction_date').change(function() {
        if ($('#transaction_type').val() === 'Invoice' && $('#transaction_date').val()) {
            var transactionDate = new Date($('#transaction_date').val());
            transactionDate.setDate(transactionDate.getDate() + 30);
            if (!$('#due_date').val()) {
                $('#due_date').val(transactionDate.toISOString().split('T')[0]);
            }
        }
    });

    // Customer search with autocomplete
    if ($('#customer_search').length) {
        $('#customer_search').on('input', function() {
            var query = $(this).val();
            if (query.length >= 2) {
                $.ajax({
                    url: 'api/customer-search.php',
                    method: 'GET',
                    data: { q: query },
                    dataType: 'json',
                    success: function(data) {
                        var suggestions = '';
                        data.forEach(function(customer) {
                            suggestions += '<div class="suggestion-item" data-id="' + customer.ID + '">' +
                                         customer.CustomerCode + ' - ' + customer.CustomerName + '</div>';
                        });
                        $('#customer_suggestions').html(suggestions).show();
                    }
                });
            } else {
                $('#customer_suggestions').hide();
            }
        });

        $(document).on('click', '.suggestion-item', function() {
            var customerId = $(this).data('id');
            var customerName = $(this).text();
            $('#customer_id').val(customerId);
            $('#customer_search').val(customerName);
            $('#customer_suggestions').hide();
        });
    }

    // Payment amount validation
    $('#payment_amount').on('input', function() {
        var maxAmount = parseFloat($(this).attr('max'));
        var enteredAmount = parseFloat($(this).val());
        
        if (enteredAmount > maxAmount) {
            $(this).val(maxAmount);
            showAlert('Payment amount cannot exceed the outstanding balance', 'warning');
        }
    });

    // File upload preview
    $('input[type="file"]').change(function() {
        var file = this.files[0];
        if (file) {
            var fileSize = (file.size / 1024 / 1024).toFixed(2); // MB
            var fileName = file.name;
            var allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'application/pdf', 'application/msword'];
            
            if (fileSize > 5) {
                showAlert('File size should not exceed 5MB', 'danger');
                $(this).val('');
                return;
            }
            
            if (!allowedTypes.includes(file.type)) {
                showAlert('Only images, PDF, and Word documents are allowed', 'danger');
                $(this).val('');
                return;
            }
            
            $(this).next('.custom-file-label').text(fileName);
        }
    });

    // Confirm delete actions
    $('.delete-btn').click(function(e) {
        if (!confirm('Are you sure you want to delete this item? This action cannot be undone.')) {
            e.preventDefault();
        }
    });

    // Export functionality
    window.exportToExcel = function(tableId, filename) {
        var table = document.getElementById(tableId);
        var wb = XLSX.utils.table_to_book(table, {sheet: "Sheet1"});
        XLSX.writeFile(wb, filename + '.xlsx');
    };

    window.exportToPDF = function(elementId, filename) {
        var element = document.getElementById(elementId);
        html2pdf().from(element).save(filename + '.pdf');
    };

    // Print functionality
    window.printReport = function() {
        window.print();
    };

    // Advanced filters
    $('#advanced_filters').on('shown.bs.collapse', function() {
        $(this).find('input, select').first().focus();
    });

    // Quick date filters
    $('.quick-date-filter').click(function(e) {
        e.preventDefault();
        var days = parseInt($(this).data('days'));
        var endDate = new Date();
        var startDate = new Date();
        startDate.setDate(startDate.getDate() - days);
        
        $('#date_from').val(startDate.toISOString().split('T')[0]);
        $('#date_to').val(endDate.toISOString().split('T')[0]);
        
        // Auto-submit if in filter form
        if ($(this).closest('form').find('#auto_submit').length) {
            $(this).closest('form').submit();
        }
    });

    // Real-time balance calculation
    function updateCustomerBalance(customerId) {
        if (customerId) {
            $.ajax({
                url: 'api/get-customer-balance.php',
                method: 'GET',
                data: { customer_id: customerId },
                dataType: 'json',
                success: function(data) {
                    if (data.success) {
                        $('#customer_balance').text('₹' + data.balance.toLocaleString());
                        $('#credit_limit').text('₹' + data.credit_limit.toLocaleString());
                        
                        if (data.balance > data.credit_limit) {
                            $('#customer_balance').addClass('text-danger');
                            showAlert('Customer has exceeded credit limit!', 'warning');
                        } else {
                            $('#customer_balance').removeClass('text-danger');
                        }
                    }
                }
            });
        }
    }

    // Update balance when customer is selected
    $('#customer_id').change(function() {
        updateCustomerBalance($(this).val());
    });

    // Keyboard shortcuts
    $(document).keydown(function(e) {
        // Ctrl/Cmd + Enter to submit forms
        if ((e.ctrlKey || e.metaKey) && e.keyCode === 13) {
            $('form:visible').submit();
        }
        
        // Esc to close modals
        if (e.keyCode === 27) {
            $('.modal:visible').modal('hide');
        }
    });

    // Auto-save form data (draft)
    var autoSaveTimer;
    $('form input, form select, form textarea').on('input change', function() {
        clearTimeout(autoSaveTimer);
        autoSaveTimer = setTimeout(function() {
            saveFormDraft();
        }, 2000);
    });

    function saveFormDraft() {
        var formData = {};
        $('form input, form select, form textarea').each(function() {
            if ($(this).attr('name') && $(this).val()) {
                formData[$(this).attr('name')] = $(this).val();
            }
        });
        
        if (Object.keys(formData).length > 0) {
            localStorage.setItem('customer_form_draft', JSON.stringify(formData));
        }
    }

    function loadFormDraft() {
        var draft = localStorage.getItem('customer_form_draft');
        if (draft) {
            try {
                var formData = JSON.parse(draft);
                Object.keys(formData).forEach(function(name) {
                    $('[name="' + name + '"]').val(formData[name]);
                });
            } catch (e) {
                console.log('Error loading form draft:', e);
            }
        }
    }

    // Load draft on page load
    if ($('form').length && localStorage.getItem('customer_form_draft')) {
        loadFormDraft();
    }

    // Clear draft on successful submit
    $('form').on('submit', function() {
        if ($(this).find('.alert-success').length === 0) {
            localStorage.removeItem('customer_form_draft');
        }
    });

    // Utility function to show alerts
    function showAlert(message, type) {
        var alertHtml = '<div class="alert alert-' + type + ' alert-dismissible fade show" role="alert">' +
                       message +
                       '<button type="button" class="close" data-dismiss="alert" aria-label="Close">' +
                       '<span aria-hidden="true">&times;</span>' +
                       '</button>' +
                       '</div>';
        
        $('.alert').remove(); // Remove existing alerts
        $('.container-fluid').prepend(alertHtml);
        
        // Auto-hide after 5 seconds
        setTimeout(function() {
            $('.alert').fadeOut();
        }, 5000);
    }

    // Initialize tooltips
    $('[data-toggle="tooltip"]').tooltip();

    // Initialize popovers
    $('[data-toggle="popover"]').popover();

    // Smooth scrolling for anchor links
    $('a[href^="#"]').on('click', function(event) {
        var target = $(this.getAttribute('href'));
        if (target.length) {
            event.preventDefault();
            $('html, body').stop().animate({
                scrollTop: target.offset().top - 100
            }, 1000);
        }
    });
});

// Global utility functions
function formatCurrency(amount) {
    return '₹' + parseFloat(amount).toLocaleString('en-IN', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    });
}

function formatDate(dateString) {
    var date = new Date(dateString);
    return date.toLocaleDateString('en-IN', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
}

function generateInvoiceNumber(type) {
    var prefix = '';
    switch(type) {
        case 'Invoice': prefix = 'INV'; break;
        case 'Payment': prefix = 'PAY'; break;
        case 'Credit Note': prefix = 'CN'; break;
        case 'Debit Note': prefix = 'DN'; break;
    }
    
    var date = new Date();
    var year = date.getFullYear().toString().slice(-2);
    var month = ('0' + (date.getMonth() + 1)).slice(-2);
    var day = ('0' + date.getDate()).slice(-2);
    var timestamp = Date.now().toString().slice(-4);
    
    return prefix + year + month + day + timestamp;
}
