<?php
session_start();
include_once('includes/config.php');

// Determine user type and ID
$isAdmin = isset($_SESSION['aid']);
$userId = $isAdmin ? $_SESSION['aid'] : (isset($_SESSION['uid']) ? $_SESSION['uid'] : null);
$userType = $isAdmin ? 'admin' : (isset($_SESSION['usertype']) ? $_SESSION['usertype'] : null);

if (!$userId) {
    header('location:login.php');
    exit();
}

// Get prescription ID from URL
$prescriptionId = isset($_GET['id']) ? intval($_GET['id']) : 0;

if (!$prescriptionId) {
    header('location:customer-prescriptions.php');
    exit();
}

// Fetch prescription details
$query = "SELECT * FROM customer_prescriptions WHERE id = ?";
if (!$isAdmin) {
    $query .= " AND user_id = ?";
}

$stmt = $con->prepare($query);
if ($isAdmin) {
    $stmt->bind_param("i", $prescriptionId);
} else {
    $stmt->bind_param("ii", $prescriptionId, $userId);
}

$stmt->execute();
$result = $stmt->get_result();
$prescription = $result->fetch_assoc();

if (!$prescription) {
    header('location:customer-prescriptions.php');
    exit();
}

// Calculate amounts
$total_amount = $prescription['frame_amount'] + $prescription['lens_amount'];
$discount_amount = $prescription['discount_amount'] ?? 0;
$final_amount = $total_amount - $discount_amount;
$balance_amount = $final_amount - $prescription['advance_amount'];

// Determine if print mode
$isPrintMode = isset($_GET['print']) && $_GET['print'] == '1';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Prescription Details - <?php echo htmlspecialchars($prescription['order_number']); ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet">
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    <style>
        body { 
            background: #f8f9fc; 
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .prescription-details-container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            overflow: hidden;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        .prescription-header {
            background: linear-gradient(135deg, #4e73df 0%, #224abe 100%);
            color: white;
            padding: 15px;
            text-align: center;
            position: relative;
        }
        
        .prescription-header::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: repeating-linear-gradient(
                45deg,
                transparent,
                transparent 10px,
                rgba(255,255,255,0.1) 10px,
                rgba(255,255,255,0.1) 20px
            );
            animation: slide 20s linear infinite;
        }
        
        @keyframes slide {
            0% { transform: translate(-50%, -50%) rotate(0deg); }
            100% { transform: translate(-50%, -50%) rotate(360deg); }
        }
        
        .shop-logo {
            position: relative;
            z-index: 2;
        }
        
        .shop-name {
            font-size: 1.8rem;
            font-weight: bold;
            margin-bottom: 2px;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
            position: relative;
            z-index: 2;
        }
        
        .shop-tagline {
            font-size: 0.9rem;
            opacity: 0.9;
            position: relative;
            z-index: 2;
            margin-bottom: 5px;
        }
        
        .prescription-content {
            padding: 20px;
        }
        
        .order-info {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .info-card {
            background: #f8f9fc;
            padding: 12px;
            border-radius: 8px;
            border-left: 4px solid #4e73df;
        }
        
        .info-card h5 {
            color: #4e73df;
            margin-bottom: 8px;
            font-weight: 600;
            font-size: 0.9rem;
        }
        
        .info-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 4px;
            font-size: 0.85rem;
        }
        
        .info-label {
            font-weight: 500;
            color: #5a5c69;
        }
        
        .info-value {
            font-weight: 600;
            color: #2c3e50;
        }
        
        .power-section {
            margin: 20px 0;
        }
        
        .power-title {
            color: #4e73df;
            font-size: 1rem;
            font-weight: 600;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .power-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 15px;
            border-radius: 6px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }
        
        .power-table th {
            background: linear-gradient(135deg, #4e73df, #3d5cdb);
            color: white;
            padding: 8px;
            text-align: center;
            font-weight: 600;
            font-size: 0.8rem;
        }
        
        .power-table td {
            padding: 8px;
            text-align: center;
            border-bottom: 1px solid #e3e6f0;
            font-weight: 500;
            font-size: 0.85rem;
        }
        
        .power-table tbody tr:hover {
            background-color: #f8f9fc;
        }
        
        .eye-label {
            background: #f8f9fc;
            font-weight: 600;
            color: #4e73df;
        }
        
        .power-value {
            color: #2c3e50;
            font-family: 'Courier New', monospace;
        }
        
        .contact-lens-section {
            background: #e8f5e8;
            padding: 12px;
            border-radius: 8px;
            border-left: 4px solid #28a745;
            margin-top: 15px;
        }
        
        .frame-lens-info {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
            margin: 20px 0;
        }
        
        .frame-card, .lens-card {
            background: #f8f9fc;
            padding: 12px;
            border-radius: 8px;
            border-left: 4px solid #17a2b8;
        }
        
        .amount-summary {
            background: linear-gradient(135deg, #f8f9fc 0%, #e9ecef 100%);
            padding: 15px;
            border-radius: 8px;
            margin: 20px 0;
            border: 2px solid #4e73df;
        }
        
        .amount-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 6px;
            font-size: 0.9rem;
        }
        
        .amount-row.total {
            border-top: 2px solid #4e73df;
            padding-top: 8px;
            font-weight: bold;
            font-size: 1rem;
            color: #4e73df;
        }
        
        .status-badge {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 15px;
            font-weight: 600;
            text-transform: uppercase;
            font-size: 0.7rem;
        }
        
        .status-pending { background: #fff3cd; color: #856404; }
        .status-progress { background: #d1ecf1; color: #0c5460; }
        .status-ready { background: #d4edda; color: #155724; }
        .status-delivered { background: #d6d8db; color: #383d41; }
        .status-cancelled { background: #f8d7da; color: #721c24; }
        
        .action-buttons {
            display: flex;
            gap: 10px;
            justify-content: center;
            margin: 20px 0;
            flex-wrap: wrap;
        }
        
        .btn {
            padding: 8px 15px;
            border-radius: 6px;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 5px;
            transition: all 0.3s ease;
            font-size: 0.8rem;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.2);
            text-decoration: none;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #4e73df, #3d5cdb);
            color: white;
            border: none;
        }
        
        .btn-success {
            background: linear-gradient(135deg, #28a745, #20c997);
            color: white;
            border: none;
        }
        
        .btn-info {
            background: linear-gradient(135deg, #17a2b8, #138496);
            color: white;
            border: none;
        }
        
        .btn-secondary {
            background: linear-gradient(135deg, #6c757d, #545b62);
            color: white;
            border: none;
        }
        
        .btn-whatsapp {
            background: linear-gradient(135deg, #25D366, #128C7E);
            color: white;
            border: none;
        }
        
        .notes-section {
            background: #fff3cd;
            padding: 12px;
            border-radius: 8px;
            border-left: 4px solid #ffc107;
            margin: 15px 0;
        }
        
        /* Print Styles */
        @media print {
            body {
                background: white !important;
                -webkit-print-color-adjust: exact;
                color-adjust: exact;
                font-size: 12px;
                line-height: 1.2;
            }
            
            .prescription-details-container {
                box-shadow: none;
                max-width: none;
                width: 4in;
                margin: 0;
                padding: 0;
            }
            
            .prescription-header {
                background: #4e73df !important;
                color: white !important;
                padding: 8px !important;
                -webkit-print-color-adjust: exact;
            }
            
            .prescription-header::before {
                display: none;
            }
            
            .shop-name {
                font-size: 1.2rem !important;
                margin-bottom: 1px !important;
            }
            
            .shop-tagline {
                font-size: 0.7rem !important;
                margin-bottom: 2px !important;
            }
            
            .prescription-content {
                padding: 8px !important;
            }
            
            .order-info {
                grid-template-columns: 1fr;
                gap: 6px;
                margin-bottom: 8px;
            }
            
            .info-card {
                padding: 6px;
                margin-bottom: 4px;
            }
            
            .info-card h5 {
                font-size: 0.7rem;
                margin-bottom: 3px;
            }
            
            .info-row {
                margin-bottom: 2px;
                font-size: 0.7rem;
            }
            
            .power-section {
                margin: 6px 0;
            }
            
            .power-title {
                font-size: 0.8rem;
                margin-bottom: 4px;
            }
            
            .power-table th {
                background: #4e73df !important;
                color: white !important;
                -webkit-print-color-adjust: exact;
                padding: 3px;
                font-size: 0.6rem;
            }
            
            .power-table td {
                padding: 3px;
                font-size: 0.7rem;
            }
            
            .frame-lens-info {
                grid-template-columns: 1fr;
                gap: 6px;
                margin: 8px 0;
            }
            
            .frame-card, .lens-card {
                padding: 6px;
            }
            
            .amount-summary {
                padding: 6px;
                background: #f8f9fc !important;
                -webkit-print-color-adjust: exact;
                margin: 8px 0;
            }
            
            .amount-row {
                font-size: 0.7rem;
                margin-bottom: 2px;
            }
            
            .amount-row.total {
                font-size: 0.8rem;
                padding-top: 3px;
            }
            
            .action-buttons {
                display: none !important;
            }
            
            .notes-section {
                background: #fff3cd !important;
                -webkit-print-color-adjust: exact;
                padding: 6px;
                margin: 6px 0;
            }
            
            .contact-lens-section {
                background: #e8f5e8 !important;
                -webkit-print-color-adjust: exact;
                padding: 6px;
                margin-top: 6px;
            }
            
            .status-badge {
                padding: 2px 6px;
                font-size: 0.6rem;
            }
        }
        
        /* 4-inch specific styles */
        @media print and (max-width: 4in) {
            .prescription-details-container {
                width: 4in;
                font-size: 0.65rem;
            }
            
            .shop-name {
                font-size: 1rem !important;
            }
            
            .shop-tagline {
                font-size: 0.6rem !important;
            }
            
            .power-table {
                font-size: 0.55rem;
            }
            
            .power-table th,
            .power-table td {
                padding: 2px;
            }
            
            .info-card {
                padding: 4px;
            }
            
            .prescription-content {
                padding: 6px !important;
            }
            
            .info-row {
                font-size: 0.6rem;
            }
            
            .amount-row {
                font-size: 0.6rem;
            }
            
            .amount-row.total {
                font-size: 0.7rem;
            }
        }
        
        /* Mobile Responsive */
        @media (max-width: 768px) {
            .prescription-details-container {
                margin: 10px;
                border-radius: 10px;
            }
            
            .prescription-header {
                padding: 20px;
            }
            
            .shop-name {
                font-size: 1.8rem;
            }
            
            .prescription-content {
                padding: 20px;
            }
            
            .order-info,
            .frame-lens-info {
                grid-template-columns: 1fr;
                gap: 20px;
            }
            
            .power-table {
                font-size: 0.9rem;
            }
            
            .power-table th,
            .power-table td {
                padding: 10px 5px;
            }
            
            .action-buttons {
                flex-direction: column;
                align-items: center;
            }
            
            .btn {
                width: 100%;
                max-width: 300px;
                justify-content: center;
            }
        }
        
        .watermark {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%) rotate(-45deg);
            font-size: 4rem;
            color: rgba(78, 115, 223, 0.1);
            font-weight: bold;
            z-index: 1;
            pointer-events: none;
        }
        
        .prescription-content {
            position: relative;
            z-index: 2;
        }
    </style>
</head>
<body>
    <?php if (!$isPrintMode): ?>
    <div class="container-fluid mt-4">
        <div class="d-sm-flex align-items-center justify-content-between mb-4">
            <h1 class="h3 mb-0 text-gray-800">
                <i class="fas fa-file-medical"></i> Prescription Details
            </h1>
            <div>
                <a href="customer-prescriptions.php" class="btn btn-secondary">
                    <i class="fas fa-arrow-left"></i> Back to List
                </a>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <div class="prescription-details-container <?php echo $isPrintMode ? 'print-mode' : ''; ?>">
        <!-- Watermark -->
        <div class="watermark">JAGADAMBA</div>
        
        <!-- Header -->
        <div class="prescription-header">
            <div class="shop-logo">
                <i class="fas fa-glasses fa-2x mb-1"></i>
            </div>
            <div class="shop-name">Jagadamba Opticals</div>
            <div class="shop-tagline">Your Vision, Our Mission</div>
            <div class="mt-1">
                <small style="font-size: 0.7rem;">📍 Address • 📞 Phone • 📧 Email</small>
            </div>
        </div>

        <!-- Content -->
        <div class="prescription-content">
            <!-- Order Information -->
            <div class="order-info">
                <div class="info-card">
                    <h5><i class="fas fa-receipt"></i> Order Information</h5>
                    <div class="info-row">
                        <span class="info-label">Order Number:</span>
                        <span class="info-value"><?php echo htmlspecialchars($prescription['order_number']); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Order Date:</span>
                        <span class="info-value"><?php echo date('d/m/Y', strtotime($prescription['created_at'])); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Delivery Date:</span>
                        <span class="info-value">
                            <?php echo $prescription['delivery_date'] ? date('d/m/Y', strtotime($prescription['delivery_date'])) : 'Not Set'; ?>
                        </span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Status:</span>
                        <span class="info-value">
                            <span class="status-badge status-<?php echo strtolower(str_replace(' ', '', $prescription['order_status'])); ?>">
                                <?php echo htmlspecialchars($prescription['order_status']); ?>
                            </span>
                        </span>
                    </div>
                </div>

                <div class="info-card">
                    <h5><i class="fas fa-user"></i> Customer Information</h5>
                    <div class="info-row">
                        <span class="info-label">Name:</span>
                        <span class="info-value"><?php echo htmlspecialchars($prescription['customer_name']); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Mobile:</span>
                        <span class="info-value"><?php echo htmlspecialchars($prescription['mobile']); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Age:</span>
                        <span class="info-value"><?php echo $prescription['customer_age'] ?: 'Not specified'; ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Gender:</span>
                        <span class="info-value"><?php echo htmlspecialchars($prescription['customer_sex']); ?></span>
                    </div>
                    <?php if ($prescription['address']): ?>
                    <div class="info-row">
                        <span class="info-label">Address:</span>
                        <span class="info-value"><?php echo htmlspecialchars($prescription['address']); ?></span>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Eye Power Details -->
            <div class="power-section">
                <div class="power-title">
                    <i class="fas fa-eye"></i> Eye Power Prescription
                </div>
                
                <table class="power-table">
                    <thead>
                        <tr>
                            <th>Eye</th>
                            <th>SPH</th>
                            <th>CYL</th>
                            <th>AXIS</th>
                            <th>ADD</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td class="eye-label">Right Eye (OD)</td>
                            <td class="power-value"><?php echo $prescription['right_eye_sph'] ?: '-'; ?></td>
                            <td class="power-value"><?php echo $prescription['right_eye_cyl'] ?: '-'; ?></td>
                            <td class="power-value"><?php echo $prescription['right_eye_axis'] ?: '-'; ?></td>
                            <td class="power-value"><?php echo $prescription['right_eye_addition'] ?: '-'; ?></td>
                        </tr>
                        <tr>
                            <td class="eye-label">Left Eye (OS)</td>
                            <td class="power-value"><?php echo $prescription['left_eye_sph'] ?: '-'; ?></td>
                            <td class="power-value"><?php echo $prescription['left_eye_cyl'] ?: '-'; ?></td>
                            <td class="power-value"><?php echo $prescription['left_eye_axis'] ?: '-'; ?></td>
                            <td class="power-value"><?php echo $prescription['left_eye_addition'] ?: '-'; ?></td>
                        </tr>
                    </tbody>
                </table>

                <!-- Contact Lens Section (if applicable) -->
                <?php if ($prescription['lens_type'] == 'Contact Lens' && 
                         ($prescription['right_contact_sph'] || $prescription['left_contact_sph'])): ?>
                <div class="contact-lens-section">
                    <div class="power-title" style="color: #28a745;">
                        <i class="fas fa-circle"></i> Contact Lens Power
                    </div>
                    
                    <table class="power-table">
                        <thead>
                            <tr>
                                <th>Eye</th>
                                <th>SPH</th>
                                <th>CYL</th>
                                <th>AXIS</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td class="eye-label">Right Eye (OD)</td>
                                <td class="power-value"><?php echo $prescription['right_contact_sph'] ?: '-'; ?></td>
                                <td class="power-value"><?php echo $prescription['right_contact_cyl'] ?: '-'; ?></td>
                                <td class="power-value"><?php echo $prescription['right_contact_axis'] ?: '-'; ?></td>
                            </tr>
                            <tr>
                                <td class="eye-label">Left Eye (OS)</td>
                                <td class="power-value"><?php echo $prescription['left_contact_sph'] ?: '-'; ?></td>
                                <td class="power-value"><?php echo $prescription['left_contact_cyl'] ?: '-'; ?></td>
                                <td class="power-value"><?php echo $prescription['left_contact_axis'] ?: '-'; ?></td>
                            </tr>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>

            <!-- Frame and Lens Information -->
            <div class="frame-lens-info">
                <div class="frame-card">
                    <h5><i class="fas fa-glasses"></i> Frame Details</h5>
                    <div class="info-row">
                        <span class="info-label">Frame Name:</span>
                        <span class="info-value"><?php echo htmlspecialchars($prescription['frame_name'] ?: 'Not specified'); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Frame Type:</span>
                        <span class="info-value"><?php echo htmlspecialchars($prescription['frame_type'] ?: 'Not specified'); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Frame Amount:</span>
                        <span class="info-value">₹<?php echo number_format($prescription['frame_amount'], 2); ?></span>
                    </div>
                </div>

                <div class="lens-card">
                    <h5><i class="fas fa-eye"></i> Lens Details</h5>
                    <div class="info-row">
                        <span class="info-label">Lens Name:</span>
                        <span class="info-value"><?php echo htmlspecialchars($prescription['lens_name'] ?: 'Not specified'); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Lens Type:</span>
                        <span class="info-value"><?php echo htmlspecialchars($prescription['lens_type'] ?: 'Not specified'); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Lens Amount:</span>
                        <span class="info-value">₹<?php echo number_format($prescription['lens_amount'], 2); ?></span>
                    </div>
                </div>
            </div>

            <!-- Amount Summary -->
            <div class="amount-summary">
                <h5 style="color: #4e73df; margin-bottom: 20px;">
                    <i class="fas fa-calculator"></i> Amount Summary
                </h5>
                
                <div class="amount-row">
                    <span>Frame Amount:</span>
                    <span>₹<?php echo number_format($prescription['frame_amount'], 2); ?></span>
                </div>
                
                <div class="amount-row">
                    <span>Lens Amount:</span>
                    <span>₹<?php echo number_format($prescription['lens_amount'], 2); ?></span>
                </div>
                
                <div class="amount-row">
                    <span>Total Amount:</span>
                    <span>₹<?php echo number_format($total_amount, 2); ?></span>
                </div>
                
                <?php if ($discount_amount > 0): ?>
                <div class="amount-row" style="color: #28a745;">
                    <span>Discount (<?php echo $prescription['discount_type'] == 'percentage' ? $prescription['discount_value'].'%' : '₹'.number_format($prescription['discount_value'], 2); ?>):</span>
                    <span>-₹<?php echo number_format($discount_amount, 2); ?></span>
                </div>
                <?php endif; ?>
                
                <div class="amount-row total">
                    <span>Final Amount:</span>
                    <span>₹<?php echo number_format($final_amount, 2); ?></span>
                </div>
                
                <div class="amount-row">
                    <span>Advance Paid:</span>
                    <span>₹<?php echo number_format($prescription['advance_amount'], 2); ?></span>
                </div>
                
                <div class="amount-row total" style="color: <?php echo $balance_amount > 0 ? '#dc3545' : '#28a745'; ?>;">
                    <span>Balance Amount:</span>
                    <span>₹<?php echo number_format($balance_amount, 2); ?></span>
                </div>
            </div>

            <!-- Notes Section -->
            <?php if ($prescription['notes']): ?>
            <div class="notes-section">
                <h5><i class="fas fa-sticky-note"></i> Special Notes</h5>
                <p><?php echo nl2br(htmlspecialchars($prescription['notes'])); ?></p>
            </div>
            <?php endif; ?>

            <!-- Action Buttons (hidden in print mode) -->
            <?php if (!$isPrintMode): ?>
            <div class="action-buttons">
                <a href="customer-prescriptions.php?edit=<?php echo $prescription['id']; ?>" class="btn btn-primary">
                    <i class="fas fa-edit"></i> Edit Prescription
                </a>
                
                <button onclick="printPrescription()" class="btn btn-info">
                    <i class="fas fa-print"></i> Print (4 inch)
                </button>
                
                <button onclick="sendWhatsApp()" class="btn btn-whatsapp">
                    <i class="fab fa-whatsapp"></i> Send WhatsApp
                </button>
                
                <a href="print-voucher.php?id=<?php echo $prescription['id']; ?>" target="_blank" class="btn btn-success">
                    <i class="fas fa-receipt"></i> Print Voucher
                </a>
                
                <a href="customer-prescriptions.php" class="btn btn-secondary">
                    <i class="fas fa-arrow-left"></i> Back to List
                </a>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <?php if (!$isPrintMode): ?>
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script>
        function printPrescription() {
            // Open print version in new window
            const printWindow = window.open('<?php echo $_SERVER['PHP_SELF']; ?>?id=<?php echo $prescription['id']; ?>&print=1', '_blank');
            
            // Wait for content to load, then print
            printWindow.onload = function() {
                setTimeout(function() {
                    printWindow.print();
                    // Close after printing
                    printWindow.addEventListener('afterprint', function() {
                        printWindow.close();
                    });
                }, 500);
            };
        }

        function sendWhatsApp() {
            const customerName = '<?php echo addslashes($prescription['customer_name']); ?>';
            const orderNumber = '<?php echo $prescription['order_number']; ?>';
            const mobile = '<?php echo $prescription['mobile']; ?>';
            const finalAmount = '<?php echo number_format($final_amount, 2); ?>';
            const balanceAmount = '<?php echo number_format($balance_amount, 2); ?>';
            const deliveryDate = '<?php echo $prescription['delivery_date'] ? date('d/m/Y', strtotime($prescription['delivery_date'])) : 'TBD'; ?>';
            
            const message = `*Jagadamba Opticals*\n\n` +
                           `Hello ${customerName}!\n\n` +
                           `📋 *Prescription Details:*\n` +
                           `Order Number: ${orderNumber}\n` +
                           `Final Amount: ₹${finalAmount}\n` +
                           `Balance Amount: ₹${balanceAmount}\n` +
                           `Expected Delivery: ${deliveryDate}\n\n` +
                           `Thank you for choosing Jagadamba Opticals!\n` +
                           `Your vision is our mission 👓\n\n` +
                           `For any queries, please contact us.`;
            
            // Clean mobile number
            const cleanMobile = mobile.replace(/\D/g, '');
            let whatsappNumber = cleanMobile;
            if (cleanMobile.length === 10) {
                whatsappNumber = '91' + cleanMobile;
            }
            
            const whatsappUrl = `https://wa.me/${whatsappNumber}?text=${encodeURIComponent(message)}`;
            window.open(whatsappUrl, '_blank');
        }

        // Auto-print if in print mode
        <?php if ($isPrintMode): ?>
        window.onload = function() {
            setTimeout(function() {
                window.print();
                // Close window after printing
                window.addEventListener('afterprint', function() {
                    window.close();
                });
            }, 1000);
        };
        <?php endif; ?>
    </script>
    <?php endif; ?>
</body>
</html>
