<?php
session_start();
// Start output buffering to prevent stray output from breaking JSON
if (function_exists('ob_start')) { ob_start(); }
require_once 'includes/config.php';
require_once 'includes/whatsapp-functions.php';
require_once 'includes/sms-functions.php';

// Check if user is logged in
if (!isset($_SESSION['uid']) && !isset($_SESSION['aid'])) {
    header('Location: login.php');
    exit();
}

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    // Always output valid JSON without PHP error HTML
    ini_set('display_errors', '0');
    error_reporting(0);
    header('Content-Type: application/json');

    // Emit JSON on fatal errors
    register_shutdown_function(function(){
        $e = error_get_last();
        if ($e && in_array($e['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            if (function_exists('ob_get_length') && ob_get_length()) { ob_clean(); }
            echo json_encode(['success' => false, 'message' => 'Server error: ' . $e['message']]);
        }
    });
    
    // Verify CSRF token
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        echo json_encode(['success' => false, 'message' => 'Invalid CSRF token']);
        exit();
    }
    
    switch ($_POST['action']) {
        case 'send_bulk_sms':
            // Ensure we always return valid JSON (no PHP warnings/notices)
            ini_set('display_errors', '0');
            
            // customer_ids can come as a JSON string from FormData
            $customerIds = [];
            if (isset($_POST['customer_ids'])) {
                if (is_array($_POST['customer_ids'])) {
                    $customerIds = $_POST['customer_ids'];
                } else {
                    $decoded = json_decode($_POST['customer_ids'], true);
                    if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                        $customerIds = $decoded;
                    }
                }
            }

            // Normalize to array of integers (IDs)
            if (!is_array($customerIds)) {
                $customerIds = [];
            } else {
                $customerIds = array_values(array_filter(array_map(function($v){
                    return is_numeric($v) ? (int)$v : null;
                }, $customerIds), function($v){ return $v !== null; }));
            }

            $customMessage = isset($_POST['custom_message']) ? trim($_POST['custom_message']) : '';
            
            if (empty($customerIds)) {
                echo json_encode(['success' => false, 'message' => 'No customers selected']);
                exit();
            }
            
            if (empty($customMessage)) {
                echo json_encode(['success' => false, 'message' => 'Please enter a message']);
                exit();
            }
            
            // Get customer details and remove duplicates by mobile number
            $uniqueCustomers = [];
            $seenMobiles = [];
            
            foreach ($customerIds as $customerId) {
                $query = "SELECT id, customer_name, mobile FROM customer_prescriptions WHERE id = ?";
                if ($stmt = $con->prepare($query)) {
                    $stmt->bind_param("i", $customerId);
                    $stmt->execute();
                    $result = $stmt->get_result();
                    
                    if ($customer = $result->fetch_assoc()) {
                        $mobile = trim((string)$customer['mobile']);
                        
                        // Skip if mobile number already processed
                        if ($mobile !== '' && !in_array($mobile, $seenMobiles, true)) {
                            $uniqueCustomers[] = $customer;
                            $seenMobiles[] = $mobile;
                        }
                    }
                    $stmt->close();
                }
            }
            
            // Send SMS to unique customers
            $results = [];
            $successCount = 0;
            $failCount = 0;
            foreach ($uniqueCustomers as $customer) {
                // Personalize message for each customer
                $personalizedMessage = personalizeSMSMessage($customMessage, $customer);
                
                // Send SMS
                $smsResult = sendSMS($customer['mobile'], $personalizedMessage, $customer['customer_name']);
                
                $isOk = !empty($smsResult['success']);
                if ($isOk) { $successCount++; } else { $failCount++; }

                $results[] = [
                    'customer_id' => (int)$customer['id'],
                    'customer_name' => (string)$customer['customer_name'],
                    'mobile' => (string)$customer['mobile'],
                    'success' => (bool)$isOk,
                    'message' => (string)($smsResult['message'] ?? '')
                ];
                
                // Add small delay to prevent API overload
                usleep(100000); // 0.1 second delay between SMS
            }
            
            if (function_exists('ob_get_length') && ob_get_length()) { ob_clean(); }
            echo json_encode([
                'success' => true,
                'results' => $results,
                'total_success' => $successCount,
                'total_failed' => $failCount,
                'total_unique' => count($uniqueCustomers),
                'total_selected' => count($customerIds),
                'duplicates_removed' => count($customerIds) - count($uniqueCustomers)
            ]);
            exit();
            
        case 'get_customers':
            // Get latest prescription per (customer_name, mobile) and total count per mobile
            $query = "
                SELECT 
                    t.id,
                    t.customer_name,
                    t.mobile,
                    t.order_number,
                    t.order_status,
                    DATE(t.created_at) AS prescription_date,
                    c.cnt AS prescription_count
                FROM (
                    SELECT cp1.*
                    FROM customer_prescriptions cp1
                    INNER JOIN (
                        SELECT customer_name, mobile, MAX(created_at) AS max_created
                        FROM customer_prescriptions
                        WHERE mobile IS NOT NULL AND mobile <> ''
                        GROUP BY customer_name, mobile
                    ) latest
                    ON cp1.customer_name = latest.customer_name
                    AND cp1.mobile = latest.mobile
                    AND cp1.created_at = latest.max_created
                ) t
                INNER JOIN (
                    SELECT mobile, COUNT(*) AS cnt
                    FROM customer_prescriptions
                    WHERE mobile IS NOT NULL AND mobile <> ''
                    GROUP BY mobile
                ) c ON t.mobile = c.mobile
                ORDER BY t.created_at DESC
            ";

            $result = $con->query($query);
            if (!$result) {
                if (function_exists('ob_get_length') && ob_get_length()) { ob_clean(); }
                echo json_encode(['success' => false, 'message' => 'DB error: ' . $con->error]);
                exit();
            }
            $customers = [];
            
            while ($row = $result->fetch_assoc()) {
                // Cast to expected types
                $customers[] = [
                    'id' => (int)$row['id'],
                    'customer_name' => (string)$row['customer_name'],
                    'mobile' => (string)$row['mobile'],
                    'order_number' => (string)($row['order_number'] ?? ''),
                    'order_status' => (string)($row['order_status'] ?? ''),
                    'prescription_date' => (string)$row['prescription_date'],
                    'prescription_count' => (int)$row['prescription_count'],
                ];
            }
            
            if (function_exists('ob_get_length') && ob_get_length()) { ob_clean(); }
            echo json_encode(['success' => true, 'customers' => $customers]);
            exit();

        default:
            if (function_exists('ob_get_length') && ob_get_length()) { ob_clean(); }
            echo json_encode(['success' => false, 'message' => 'Unknown action']);
            exit();
    }
}

// This function is now handled by includes/sms-functions.php
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
    <meta name="description" content="">
    <meta name="author" content="">

    <title>Bulk SMS Campaign - Optical Management</title>

    <!-- Custom fonts for this template -->
    <link href="vendor/fontawesome-free/css/all.min.css" rel="stylesheet" type="text/css">
    <link href="https://fonts.googleapis.com/css?family=Nunito:200,200i,300,300i,400,400i,600,600i,700,700i,800,800i,900,900i" rel="stylesheet">

    <!-- Custom styles for this template -->
    <link href="css/sb-admin-2.min.css" rel="stylesheet">
    
    <!-- Custom DataTables Bootstrap 4 -->
    <link href="vendor/datatables/dataTables.bootstrap4.min.css" rel="stylesheet">

    <style>
        .campaign-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem;
            border-radius: 10px;
            margin-bottom: 2rem;
        }
        
        .message-composer {
            background: #f8f9fc;
            border: 2px solid #e3e6f0;
            border-radius: 10px;
            padding: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .message-textarea {
            border: 2px solid #d1d3e2;
            border-radius: 8px;
            padding: 1rem;
            font-size: 14px;
            line-height: 1.5;
            resize: vertical;
        }
        
        .message-textarea:focus {
            border-color: #5a5c69;
            box-shadow: 0 0 0 0.2rem rgba(90, 92, 105, 0.25);
        }
        
        .char-counter {
            font-size: 12px;
            color: #6c757d;
            text-align: right;
            margin-top: 0.5rem;
        }
        
        .char-counter.warning {
            color: #f6c23e;
        }
        
        .char-counter.danger {
            color: #e74a3b;
        }
        
        .customer-selection {
            background: white;
            border-radius: 10px;
            box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
        }
        
        .selection-controls {
            background: #f8f9fc;
            padding: 1rem;
            border-bottom: 1px solid #e3e6f0;
            border-radius: 10px 10px 0 0;
        }
        
        .bulk-selected {
            background-color: #e3f2fd !important;
            border-left: 4px solid #2196f3;
        }
        
        .duplicate-mobile {
            background-color: #fff3cd !important;
            border-left: 4px solid #ffc107;
        }
        
        .duplicate-badge {
            background-color: #ffc107;
            color: #212529;
            font-size: 10px;
            padding: 2px 6px;
            border-radius: 10px;
            margin-left: 5px;
        }
        
        .campaign-stats {
            background: linear-gradient(135deg, #36b9cc 0%, #258391 100%);
            color: white;
            padding: 1.5rem;
            border-radius: 8px;
            margin-bottom: 2rem;
        }
        
        .stat-item {
            text-align: center;
        }
        
        .stat-number {
            font-size: 2rem;
            font-weight: 700;
            display: block;
        }
        
        .stat-label {
            font-size: 0.9rem;
            opacity: 0.9;
        }
        
        .progress-modal .modal-body {
            max-height: 60vh;
            overflow-y: auto;
        }
        
        .result-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0.75rem;
            border-bottom: 1px solid #e3e6f0;
            border-radius: 5px;
            margin-bottom: 0.5rem;
        }
        
        .result-item.success {
            background-color: #d4edda;
            border-left: 4px solid #28a745;
        }
        
        .result-item.error {
            background-color: #f8d7da;
            border-left: 4px solid #dc3545;
        }
        
        .result-item.processing {
            background-color: #d1ecf1;
            border-left: 4px solid #17a2b8;
        }
        
        .customer-details {
            flex-grow: 1;
        }
        
        .customer-name {
            font-weight: 600;
            color: #2c3e50;
        }
        
        .customer-mobile {
            font-size: 0.9rem;
            color: #6c757d;
        }
        
        .result-status {
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
        }
        
        .result-status.success {
            background-color: #28a745;
            color: white;
        }
        
        .result-status.error {
            background-color: #dc3545;
            color: white;
        }
        
        .result-status.processing {
            background-color: #17a2b8;
            color: white;
        }
        
        @media (max-width: 768px) {
            .campaign-header {
                padding: 1rem;
                text-align: center;
            }
            
            .stat-item {
                margin-bottom: 1rem;
            }
            
            .message-composer {
                padding: 1rem;
            }
        }
    </style>
</head>

<body id="page-top">

    <!-- Page Wrapper -->
    <div id="wrapper">

        <!-- Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Content Wrapper -->
        <div id="content-wrapper" class="d-flex flex-column">

            <!-- Main Content -->
            <div id="content">

                <!-- Topbar -->
                <?php include 'includes/topbar.php'; ?>

                <!-- Begin Page Content -->
                <div class="container-fluid">

                    <!-- Campaign Header -->
                    <div class="campaign-header">
                        <div class="row align-items-center">
                            <div class="col-md-8">
                                <h1 class="h3 mb-2"><i class="fas fa-bullhorn mr-2"></i>Bulk SMS Campaign</h1>
                                <p class="mb-0">Send customized SMS messages to multiple customers. Duplicate mobile numbers are automatically skipped.</p>
                            </div>
                            <div class="col-md-4 text-md-right">
                                <i class="fas fa-comments fa-3x opacity-50"></i>
                            </div>
                        </div>
                    </div>

                    <!-- Campaign Stats -->
                    <div class="campaign-stats">
                        <div class="row">
                            <div class="col-md-3 col-6 stat-item">
                                <span class="stat-number" id="totalCustomers">0</span>
                                <span class="stat-label">Total Customers</span>
                            </div>
                            <div class="col-md-3 col-6 stat-item">
                                <span class="stat-number" id="selectedCustomers">0</span>
                                <span class="stat-label">Selected</span>
                            </div>
                            <div class="col-md-3 col-6 stat-item">
                                <span class="stat-number" id="uniqueMobiles">0</span>
                                <span class="stat-label">Unique Mobiles</span>
                            </div>
                            <div class="col-md-3 col-6 stat-item">
                                <span class="stat-number" id="duplicateCount">0</span>
                                <span class="stat-label">Duplicates</span>
                            </div>
                        </div>
                    </div>

                    <?php 
                    $currentProvider = defined('SMS_PROVIDER') ? SMS_PROVIDER : 'undefined';
                    if (strtolower($currentProvider) === 'demo'): 
                    ?>
                    <div class="alert alert-warning mt-3">
                        <i class="fas fa-exclamation-triangle mr-1"></i>
                        <strong>SMS is currently in Demo mode.</strong> Messages will NOT be sent. Configure your SMS provider to enable real sending.
                        <br><small>Current provider: <?php echo htmlspecialchars($currentProvider); ?></small>
                    </div>
                    <?php elseif ($currentProvider === 'undefined'): ?>
                    <div class="alert alert-danger mt-3">
                        <i class="fas fa-times-circle mr-1"></i>
                        <strong>SMS Provider not defined.</strong> Messages will not be sent.
                    </div>
                    <?php else: ?>
                    <div class="alert alert-info mt-3">
                        <i class="fas fa-info-circle mr-1"></i>
                        SMS Provider: <?php echo htmlspecialchars($currentProvider); ?>
                        <?php if ($currentProvider === 'http' && (!defined('SMS_API_URL') || empty(SMS_API_URL))): ?>
                        - <span class="text-warning">API URL not configured</span>
                        <?php endif; ?>
                    </div>
                    <?php endif; ?>

                    <!-- Message Composer -->
                    <div class="message-composer">
                        <div class="row">
                            <div class="col-12">
                                <h5 class="mb-3"><i class="fas fa-edit mr-2"></i>Compose Your Message</h5>
                                <textarea 
                                    id="customMessage" 
                                    class="form-control message-textarea" 
                                    rows="4" 
                                    placeholder="Type your custom message here... (e.g., Hello [Customer Name], we have special offers for you! Visit our store today.)"
                                    maxlength="500"
                                ></textarea>
                                <div class="char-counter" id="charCounter">0 / 500 characters</div>
                                
                                <div class="mt-3">
                                    <small class="text-muted">
                                        <i class="fas fa-info-circle mr-1"></i>
                                        <strong>Tip:</strong> Use [Customer Name] to personalize messages. The system will automatically replace it with each customer's name.
                                    </small>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Customer Selection -->
                    <div class="customer-selection">
                        <div class="selection-controls">
                            <div class="row align-items-center">
                                <div class="col-md-6">
                                    <h5 class="mb-0"><i class="fas fa-users mr-2"></i>Select Customers</h5>
                                </div>
                                <div class="col-md-6">
                                    <div class="btn-group float-md-right mt-2 mt-md-0" role="group">
                                        <button type="button" class="btn btn-primary btn-sm" onclick="selectAllCustomers()">
                                            <i class="fas fa-check-square mr-1"></i>Select All
                                        </button>
                                        <button type="button" class="btn btn-secondary btn-sm" onclick="clearAllSelections()">
                                            <i class="fas fa-square mr-1"></i>Clear All
                                        </button>
                                        <button type="button" class="btn btn-info btn-sm" onclick="loadCustomers()">
                                            <i class="fas fa-refresh mr-1"></i>Refresh
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="table-responsive">
                            <table class="table table-striped" id="customersTable">
                                <thead>
                                    <tr>
                                        <th width="50">
                                            <input type="checkbox" id="selectAllCheckbox" onchange="toggleAllCustomers()">
                                        </th>
                                        <th>Customer Name</th>
                                        <th>Mobile Number</th>
                                        <th>Last Prescription</th>
                                        <th>Total Prescriptions</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody id="customersTableBody">
                                    <tr>
                                        <td colspan="6" class="text-center py-4">
                                            <i class="fas fa-spinner fa-spin mr-2"></i>Loading customers...
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <!-- Send Campaign Button -->
                    <div class="text-center mt-4 mb-5">
                        <button type="button" class="btn btn-success btn-lg px-5" id="sendCampaignBtn" onclick="sendBulkSMS()" disabled>
                            <i class="fas fa-paper-plane mr-2"></i>Send SMS Campaign
                        </button>
                    </div>

                </div>
                <!-- /.container-fluid -->

            </div>
            <!-- End of Main Content -->

            <!-- Footer -->
            <?php include 'includes/footer.php'; ?>

        </div>
        <!-- End of Content Wrapper -->

    </div>
    <!-- End of Page Wrapper -->

    <!-- Progress Modal -->
    <div class="modal fade progress-modal" id="campaignProgressModal" tabindex="-1" role="dialog" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-paper-plane mr-2"></i>SMS Campaign Progress
                    </h5>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <div class="d-flex justify-content-between align-items-center mb-2">
                            <span id="campaignProgressText">Initializing SMS campaign...</span>
                            <span id="campaignProgressPercent">0%</span>
                        </div>
                        <div class="progress">
                            <div class="progress-bar progress-bar-striped progress-bar-animated bg-info" 
                                 role="progressbar" 
                                 style="width: 0%" 
                                 id="campaignProgress">
                            </div>
                        </div>
                        <div class="mt-2">
                            <small class="text-muted" id="campaignProgressDetail">Preparing to send messages...</small>
                        </div>
                    </div>
                    
                    <div id="campaignResults">
                        <!-- Results will be populated here -->
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" id="campaignProgressClose" onclick="closeCampaignProgress()" disabled>
                        Close
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Bootstrap core JavaScript-->
    <script src="vendor/jquery/jquery.min.js"></script>
    <script src="vendor/bootstrap/js/bootstrap.bundle.min.js"></script>

    <!-- Core plugin JavaScript-->
    <script src="vendor/jquery-easing/jquery.easing.min.js"></script>

    <!-- Custom scripts for all pages-->
    <script src="js/sb-admin-2.min.js"></script>

    <!-- Page level plugins -->
    <script src="vendor/datatables/jquery.dataTables.min.js"></script>
    <script src="vendor/datatables/dataTables.bootstrap4.min.js"></script>

    <script>
        let customers = [];
        let selectedCustomerIds = [];
        let campaignInProgress = false;

        // Initialize page
        $(document).ready(function() {
            loadCustomers();
            setupMessageCounter();
        });

        // Setup message character counter
        function setupMessageCounter() {
            const messageBox = document.getElementById('customMessage');
            const counter = document.getElementById('charCounter');
            
            messageBox.addEventListener('input', function() {
                const length = this.value.length;
                const maxLength = parseInt(this.getAttribute('maxlength'));
                
                counter.textContent = `${length} / ${maxLength} characters`;
                
                // Update counter color based on usage
                counter.className = 'char-counter';
                if (length > maxLength * 0.8) {
                    counter.classList.add('warning');
                }
                if (length > maxLength * 0.95) {
                    counter.classList.add('danger');
                }
                
                // Enable/disable send button
                updateSendButton();
            });
        }

        // Load customers from server
        function loadCustomers() {
            const formData = new FormData();
            formData.append('action', 'get_customers');
            formData.append('csrf_token', '<?php echo $_SESSION['csrf_token']; ?>');

            fetch('bulk-campaign.php', {
                method: 'POST',
                body: formData
            })
            .then(async (response) => {
                const text = await response.text();
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${text.slice(0, 200)}`);
                }
                try {
                    return JSON.parse(text);
                } catch (e) {
                    throw new Error('Invalid JSON received');
                }
            })
            .then(result => {
                if (result.success) {
                    customers = result.customers;
                    displayCustomers();
                    updateStats();
                } else {
                    alert('Error loading customers: ' + result.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error loading customers: ' + error.message);
            });
        }

        // Display customers in table
        function displayCustomers() {
            const tbody = document.getElementById('customersTableBody');
            
            if (customers.length === 0) {
                tbody.innerHTML = '<tr><td colspan="6" class="text-center py-4"><i class="fas fa-users-slash mr-2"></i>No customers found</td></tr>';
                return;
            }

            let html = '';
            const mobileCount = {};
            
            // Count mobile numbers to identify duplicates
            customers.forEach(customer => {
                const mobile = customer.mobile;
                mobileCount[mobile] = (mobileCount[mobile] || 0) + 1;
            });

            customers.forEach((customer, index) => {
                const isDuplicate = mobileCount[customer.mobile] > 1;
                const s = (customer.order_status || '').toLowerCase();
                let statusClass = 'secondary';
                if (s === 'pending') statusClass = 'warning';
                else if (s === 'in progress') statusClass = 'info';
                else if (s === 'ready') statusClass = 'primary';
                else if (s === 'delivered') statusClass = 'success';
                else if (s === 'cancelled') statusClass = 'dark';
                
                html += `
                    <tr class="${isDuplicate ? 'duplicate-mobile' : ''}" data-customer-id="${customer.id}">
                        <td>
                            <input type="checkbox" 
                                   class="customer-checkbox" 
                                   value="${customer.id}" 
                                   data-mobile="${customer.mobile}"
                                   onchange="updateSelection()">
                        </td>
                        <td>
                            <strong>${customer.customer_name}</strong>
                            ${isDuplicate ? '<span class="duplicate-badge">DUPLICATE</span>' : ''}
                        </td>
                        <td>${customer.mobile}</td>
                        <td>${customer.prescription_date}</td>
                        <td>
                            <span class="badge badge-primary">${customer.prescription_count}</span>
                        </td>
                        <td>
                            <span class="badge badge-${statusClass}">${customer.order_status || '—'}</span>
                        </td>
                    </tr>
                `;
            });
            
            tbody.innerHTML = html;
        }

        // Update selection and stats
        function updateSelection() {
            const checkboxes = document.querySelectorAll('.customer-checkbox:checked');
            selectedCustomerIds = Array.from(checkboxes).map(cb => cb.value);
            
            // Highlight selected rows
            document.querySelectorAll('[data-customer-id]').forEach(row => {
                const customerId = row.dataset.customerId;
                if (selectedCustomerIds.includes(customerId)) {
                    row.classList.add('bulk-selected');
                } else {
                    row.classList.remove('bulk-selected');
                }
            });
            
            updateStats();
            updateSendButton();
        }

        // Update statistics
        function updateStats() {
            const totalCustomers = customers.length;
            const selectedCount = selectedCustomerIds.length;
            
            // Get unique mobile numbers from selected customers
            const selectedMobiles = new Set();
            selectedCustomerIds.forEach(id => {
                const idStr = String(id);
                const customer = customers.find(c => String(c.id) === idStr);
                if (customer && customer.mobile) {
                    selectedMobiles.add(customer.mobile);
                }
            });
            
            const uniqueCount = selectedMobiles.size;
            const duplicateCount = selectedCount - uniqueCount;
            
            document.getElementById('totalCustomers').textContent = totalCustomers;
            document.getElementById('selectedCustomers').textContent = selectedCount;
            document.getElementById('uniqueMobiles').textContent = uniqueCount;
            document.getElementById('duplicateCount').textContent = duplicateCount;
        }

        // Toggle all customers
        function toggleAllCustomers() {
            const selectAll = document.getElementById('selectAllCheckbox');
            const checkboxes = document.querySelectorAll('.customer-checkbox');
            
            checkboxes.forEach(checkbox => {
                checkbox.checked = selectAll.checked;
            });
            
            updateSelection();
        }

        // Select all customers
        function selectAllCustomers() {
            document.getElementById('selectAllCheckbox').checked = true;
            toggleAllCustomers();
        }

        // Clear all selections
        function clearAllSelections() {
            document.getElementById('selectAllCheckbox').checked = false;
            document.querySelectorAll('.customer-checkbox').forEach(cb => cb.checked = false);
            updateSelection();
        }

        // Update send button state
        function updateSendButton() {
            const button = document.getElementById('sendCampaignBtn');
            const message = document.getElementById('customMessage').value.trim();
            const hasSelection = selectedCustomerIds.length > 0;
            const hasMessage = message.length > 0;
            
            button.disabled = !hasSelection || !hasMessage || campaignInProgress;
            
            if (!hasMessage) {
                button.innerHTML = '<i class="fas fa-edit mr-2"></i>Enter Message First';
            } else if (!hasSelection) {
                button.innerHTML = '<i class="fas fa-users mr-2"></i>Select Customers First';
            } else {
                const uniqueCount = document.getElementById('uniqueMobiles').textContent;
                button.innerHTML = `<i class="fas fa-paper-plane mr-2"></i>Send SMS to ${uniqueCount} Customers`;
            }
        }

        // Send bulk SMS
        function sendBulkSMS() {
            if (campaignInProgress) {
                alert('Campaign is already in progress. Please wait.');
                return;
            }

            const message = document.getElementById('customMessage').value.trim();
            const uniqueCount = parseInt(document.getElementById('uniqueMobiles').textContent);

            if (selectedCustomerIds.length === 0) {
                alert('Please select at least one customer.');
                return;
            }

            if (!message) {
                alert('Please enter a message to send.');
                return;
            }

            if (!confirm(`Send SMS to ${uniqueCount} unique customers?\n\nDuplicates will be automatically removed.\n\nProceed with campaign?`)) {
                return;
            }

            campaignInProgress = true;

            // Show progress modal
            $('#campaignProgressModal').modal('show');
            
            // Reset progress
            const progressBar = document.getElementById('campaignProgress');
            const progressText = document.getElementById('campaignProgressText');
            const progressDetail = document.getElementById('campaignProgressDetail');
            const progressPercent = document.getElementById('campaignProgressPercent');
            const resultsContainer = document.getElementById('campaignResults');
            
            progressBar.style.width = '0%';
            progressText.textContent = 'Starting SMS campaign...';
            progressDetail.textContent = `Sending to ${uniqueCount} unique customers`;
            progressPercent.textContent = '0%';
            resultsContainer.innerHTML = '';
            document.getElementById('campaignProgressClose').disabled = true;

            // Send campaign
            const formData = new FormData();
            formData.append('action', 'send_bulk_sms');
            formData.append('customer_ids', JSON.stringify(selectedCustomerIds));
            formData.append('custom_message', message);
            formData.append('csrf_token', '<?php echo $_SESSION['csrf_token']; ?>');

            fetch('bulk-campaign.php', {
                method: 'POST',
                body: formData
            })
            .then(async (response) => {
                const text = await response.text();
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${text.slice(0, 200)}`);
                }
                try {
                    return JSON.parse(text);
                } catch (e) {
                    throw new Error('Invalid JSON received');
                }
            })
            .then(result => {
                // Debug: Show what we received
                console.log('Campaign result:', result);
                console.log('Success count:', result.total_success);
                console.log('Failed count:', result.total_failed);
                
                if (result.success) {
                    processCampaignResults(result);
                } else {
                    progressText.textContent = 'Campaign failed';
                    progressDetail.textContent = result.message;
                    document.getElementById('campaignProgressClose').disabled = false;
                    campaignInProgress = false;
                }
            })
            .catch(error => {
                console.error('Campaign Error:', error);
                progressText.textContent = 'Campaign error';
                progressDetail.textContent = 'Error: ' + error.message;
                document.getElementById('campaignProgressClose').disabled = false;
                campaignInProgress = false;
            });
        }

        // Process campaign results
        function processCampaignResults(result) {
            const progressBar = document.getElementById('campaignProgress');
            const progressText = document.getElementById('campaignProgressText');
            const progressDetail = document.getElementById('campaignProgressDetail');
            const progressPercent = document.getElementById('campaignProgressPercent');
            const resultsContainer = document.getElementById('campaignResults');
            
            const totalSuccess = result.total_success ?? 0;
            const totalFailed = result.total_failed ?? 0;
            const totalSelected = result.total_selected ?? 0;
            const totalUnique = result.total_unique ?? 0;
            const duplicatesRemoved = result.duplicates_removed ?? 0;
            
            progressBar.style.width = '100%';
            progressPercent.textContent = '100%';
            
            if (totalSuccess > 0 && totalFailed === 0) {
                progressText.textContent = `Campaign completed! All ${totalSuccess} messages sent successfully.`;
                progressBar.className = 'progress-bar bg-success';
            } else if (totalSuccess === 0 && totalFailed > 0) {
                progressText.textContent = `Campaign completed! All ${totalFailed} messages failed.`;
                progressBar.className = 'progress-bar bg-danger';
            } else if (totalSuccess > 0 && totalFailed > 0) {
                progressText.textContent = `Campaign completed! ${totalSuccess} sent, ${totalFailed} failed.`;
                progressBar.className = 'progress-bar bg-warning';
            } else {
                progressText.textContent = `Campaign completed! No messages processed.`;
                progressBar.className = 'progress-bar bg-secondary';
            }
            
            progressDetail.textContent = `Selected: ${totalSelected}, Unique: ${totalUnique}, Success: ${totalSuccess}, Failed: ${totalFailed}, Duplicates removed: ${duplicatesRemoved}`;
            
            // Display results
            let resultsHtml = '';
            result.results.forEach(item => {
                const statusClass = item.success ? 'success' : 'error';
                resultsHtml += `
                    <div class="result-item ${statusClass}">
                        <div class="customer-details">
                            <div class="customer-name">${item.customer_name}</div>
                            <div class="customer-mobile">${item.mobile}</div>
                        </div>
                        <div class="result-status ${statusClass}">
                            ${item.success ? '✓ Sent' : '✗ Failed: ' + item.message}
                        </div>
                    </div>
                `;
            });
            
            resultsContainer.innerHTML = resultsHtml;
            document.getElementById('campaignProgressClose').disabled = false;
            campaignInProgress = false;
            
            // Clear selections after completion
            setTimeout(() => {
                clearAllSelections();
            }, 2000);
        }

        // Close campaign progress
        function closeCampaignProgress() {
            if (campaignInProgress) {
                if (!confirm('Campaign is still in progress. Are you sure you want to close this window?')) {
                    return;
                }
            }
            
            $('#campaignProgressModal').modal('hide');
        }
    </script>

</body>
</html>