<?php
/**
 * GST Reports Functions
 * Contains all reporting-related functions for the GST Notes module
 */

function getReportData($params) {
    $report_type = $params['report_type'] ?? 'summary';
    $note_type = $params['note_type'] ?? 'both'; // credit, debit, both
    $transaction_type = $params['transaction_type'] ?? 'both'; // sales, purchase, both
    $from_date = $params['from_date'] ?? date('Y-m-01');
    $to_date = $params['to_date'] ?? date('Y-m-t');
    $party_name = $params['party_name'] ?? '';
    $group_by = $params['group_by'] ?? 'date';
    
    $data = [];
    
    try {
        switch($report_type) {
            case 'summary':
                $data = getSummaryReport($note_type, $transaction_type, $from_date, $to_date);
                break;
            case 'detailed':
                $data = getDetailedReport($note_type, $transaction_type, $from_date, $to_date, $party_name);
                break;
            case 'gst_summary':
                $data = getGSTSummaryReport($note_type, $transaction_type, $from_date, $to_date, $group_by);
                break;
            case 'party_wise':
                $data = getPartyWiseReport($note_type, $transaction_type, $from_date, $to_date);
                break;
            case 'monthly':
                $data = getMonthlyReport($note_type, $transaction_type, $from_date, $to_date);
                break;
            default:
                throw new Exception("Invalid report type: " . $report_type);
        }
    } catch (Exception $e) {
        throw new Exception("Error generating report data: " . $e->getMessage());
    }
    
    return $data;
}

function getSummaryReport($note_type, $transaction_type, $from_date, $to_date) {
    $query = buildSummaryQuery($note_type, $transaction_type, $from_date, $to_date);
    $result = fetchAll($query['sql'], $query['params']);
    
    $summary = [
        'total_notes' => 0,
        'total_amount' => 0,
        'total_taxable' => 0,
        'total_cgst' => 0,
        'total_sgst' => 0,
        'total_igst' => 0,
        'breakdown' => [
            'credit_sales' => 0,
            'credit_purchase' => 0,
            'debit_sales' => 0,
            'debit_purchase' => 0
        ]
    ];
    
    foreach($result as $row) {
        $summary['total_notes'] += intval($row['count']);
        $summary['total_amount'] += floatval($row['total_amount']);
        $summary['total_taxable'] += floatval($row['taxable_amount']);
        $summary['total_cgst'] += floatval($row['cgst_amount']);
        $summary['total_sgst'] += floatval($row['sgst_amount']);
        $summary['total_igst'] += floatval($row['igst_amount']);
        
        $key = $row['note_category'] . '_' . $row['note_type'];
        if (isset($summary['breakdown'][$key])) {
            $summary['breakdown'][$key] = floatval($row['total_amount']);
        }
    }
    
    return $summary;
}

function getDetailedReport($note_type, $transaction_type, $from_date, $to_date, $party_name = '') {
    $tables = [];
    if($note_type === 'both' || $note_type === 'credit') {
        $tables[] = 'credit_notes';
    }
    if($note_type === 'both' || $note_type === 'debit') {
        $tables[] = 'debit_notes';
    }
    
    $results = [];
    
    foreach($tables as $table) {
        $note_category = str_replace('_notes', '', $table);
        
        $sql = "SELECT 
                    '$note_category' as note_category,
                    note_number,
                    note_date,
                    note_type,
                    party_name,
                    party_gstin,
                    original_invoice_no,
                    original_invoice_date,
                    reason,
                    taxable_amount,
                    cgst_rate,
                    cgst_amount,
                    sgst_rate,
                    sgst_amount,
                    igst_rate,
                    igst_amount,
                    total_amount,
                    created_at
                FROM $table 
                WHERE note_date BETWEEN ? AND ?";
        
        $params = [$from_date, $to_date];
        
        if($transaction_type !== 'both') {
            $sql .= " AND note_type = ?";
            $params[] = $transaction_type;
        }
        
        if($party_name) {
            $sql .= " AND party_name LIKE ?";
            $params[] = '%' . $party_name . '%';
        }
        
        $sql .= " ORDER BY note_date DESC, note_number DESC";
        
        $data = fetchAll($sql, $params);
        $results = array_merge($results, $data);
    }
    
    // Sort combined results by date
    usort($results, function($a, $b) {
        return strcmp($b['note_date'], $a['note_date']);
    });
    
    return $results;
}

function getGSTSummaryReport($note_type, $transaction_type, $from_date, $to_date, $group_by) {
    $tables = [];
    if($note_type === 'both' || $note_type === 'credit') {
        $tables[] = 'credit_notes';
    }
    if($note_type === 'both' || $note_type === 'debit') {
        $tables[] = 'debit_notes';
    }
    
    $group_fields = [
        'date' => 'DATE(note_date)',
        'month' => 'DATE_FORMAT(note_date, "%Y-%m")',
        'party' => 'party_name',
        'gst_rate' => 'CASE WHEN igst_rate > 0 THEN igst_rate ELSE (cgst_rate + sgst_rate) END'
    ];
    
    $group_field = $group_fields[$group_by] ?? $group_fields['date'];
    
    $results = [];
    
    foreach($tables as $table) {
        $note_category = str_replace('_notes', '', $table);
        
        $sql = "SELECT 
                    '$note_category' as note_category,
                    $group_field as group_key,
                    note_type,
                    COUNT(*) as note_count,
                    SUM(taxable_amount) as total_taxable,
                    SUM(cgst_amount) as total_cgst,
                    SUM(sgst_amount) as total_sgst,
                    SUM(igst_amount) as total_igst,
                    SUM(total_amount) as total_amount
                FROM $table 
                WHERE note_date BETWEEN ? AND ?";
        
        $params = [$from_date, $to_date];
        
        if($transaction_type !== 'both') {
            $sql .= " AND note_type = ?";
            $params[] = $transaction_type;
        }
        
        $sql .= " GROUP BY group_key, note_type, note_category
                 ORDER BY group_key DESC";
        
        $data = fetchAll($sql, $params);
        $results = array_merge($results, $data);
    }
    
    return $results;
}

function getPartyWiseReport($note_type, $transaction_type, $from_date, $to_date) {
    $tables = [];
    if($note_type === 'both' || $note_type === 'credit') {
        $tables[] = 'credit_notes';
    }
    if($note_type === 'both' || $note_type === 'debit') {
        $tables[] = 'debit_notes';
    }
    
    $results = [];
    
    foreach($tables as $table) {
        $note_category = str_replace('_notes', '', $table);
        
        $sql = "SELECT 
                    '$note_category' as note_category,
                    party_name,
                    party_gstin,
                    note_type,
                    COUNT(*) as note_count,
                    SUM(taxable_amount) as total_taxable,
                    SUM(cgst_amount) as total_cgst,
                    SUM(sgst_amount) as total_sgst,
                    SUM(igst_amount) as total_igst,
                    SUM(total_amount) as total_amount,
                    MIN(note_date) as first_note_date,
                    MAX(note_date) as last_note_date
                FROM $table 
                WHERE note_date BETWEEN ? AND ?";
        
        $params = [$from_date, $to_date];
        
        if($transaction_type !== 'both') {
            $sql .= " AND note_type = ?";
            $params[] = $transaction_type;
        }
        
        $sql .= " GROUP BY party_name, party_gstin, note_type, note_category
                 ORDER BY total_amount DESC";
        
        $data = fetchAll($sql, $params);
        $results = array_merge($results, $data);
    }
    
    return $results;
}

function getMonthlyReport($note_type, $transaction_type, $from_date, $to_date) {
    return getGSTSummaryReport($note_type, $transaction_type, $from_date, $to_date, 'month');
}

function buildSummaryQuery($note_type, $transaction_type, $from_date, $to_date) {
    $union_queries = [];
    $params = [];
    
    if($note_type === 'both' || $note_type === 'credit') {
        $sql = "SELECT 
                    'credit' as note_category,
                    note_type,
                    COUNT(*) as count,
                    SUM(taxable_amount) as taxable_amount,
                    SUM(cgst_amount) as cgst_amount,
                    SUM(sgst_amount) as sgst_amount,
                    SUM(igst_amount) as igst_amount,
                    SUM(total_amount) as total_amount
                FROM credit_notes 
                WHERE note_date BETWEEN ? AND ?";
        
        $current_params = [$from_date, $to_date];
        
        if($transaction_type !== 'both') {
            $sql .= " AND note_type = ?";
            $current_params[] = $transaction_type;
        }
        
        $sql .= " GROUP BY note_type";
        $union_queries[] = $sql;
        $params = array_merge($params, $current_params);
    }
    
    if($note_type === 'both' || $note_type === 'debit') {
        $sql = "SELECT 
                    'debit' as note_category,
                    note_type,
                    COUNT(*) as count,
                    SUM(taxable_amount) as taxable_amount,
                    SUM(cgst_amount) as cgst_amount,
                    SUM(sgst_amount) as sgst_amount,
                    SUM(igst_amount) as igst_amount,
                    SUM(total_amount) as total_amount
                FROM debit_notes 
                WHERE note_date BETWEEN ? AND ?";
        
        $current_params = [$from_date, $to_date];
        
        if($transaction_type !== 'both') {
            $sql .= " AND note_type = ?";
            $current_params[] = $transaction_type;
        }
        
        $sql .= " GROUP BY note_type";
        $union_queries[] = $sql;
        $params = array_merge($params, $current_params);
    }
    
    $final_sql = implode(' UNION ALL ', $union_queries);
    
    return ['sql' => $final_sql, 'params' => $params];
}

// Export functions (simplified versions)
function exportToCSV($params) {
    $data = getReportData($params);
    $report_type = $params['report_type'] ?? 'summary';
    
    $filename = 'gst_notes_' . $report_type . '_' . date('Y-m-d') . '.csv';
    
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = fopen('php://output', 'w');
    
    switch($report_type) {
        case 'detailed':
            fputcsv($output, [
                'Note Category', 'Note Number', 'Date', 'Type', 'Party Name', 'GSTIN',
                'Original Invoice', 'Invoice Date', 'Reason', 'Taxable Amount',
                'CGST Rate', 'CGST Amount', 'SGST Rate', 'SGST Amount',
                'IGST Rate', 'IGST Amount', 'Total Amount'
            ]);
            
            foreach($data as $row) {
                fputcsv($output, [
                    ucfirst($row['note_category']),
                    $row['note_number'],
                    $row['note_date'],
                    ucfirst($row['note_type']),
                    $row['party_name'],
                    $row['party_gstin'],
                    $row['original_invoice_no'],
                    $row['original_invoice_date'],
                    $row['reason'],
                    $row['taxable_amount'],
                    $row['cgst_rate'],
                    $row['cgst_amount'],
                    $row['sgst_rate'],
                    $row['sgst_amount'],
                    $row['igst_rate'],
                    $row['igst_amount'],
                    $row['total_amount']
                ]);
            }
            break;
            
        default:
            // Simple CSV for other reports
            fputcsv($output, ['Report Type', 'Value']);
            fputcsv($output, [$report_type, 'Generated at ' . date('Y-m-d H:i:s')]);
            break;
    }
    
    fclose($output);
}

function exportToPDF($params) {
    // Simple HTML output that can be printed as PDF
    $data = getReportData($params);
    $report_type = $params['report_type'] ?? 'summary';
    $title = 'GST Notes ' . ucfirst($report_type) . ' Report';
    
    header('Content-Type: text/html');
    echo "<html><head><title>$title</title></head><body>";
    echo "<h1>$title</h1>";
    echo "<p>Generated on: " . date('Y-m-d H:i:s') . "</p>";
    echo "<pre>" . print_r($data, true) . "</pre>";
    echo "</body></html>";
}

function exportToExcel($params) {
    // Simple Excel format (HTML table with Excel MIME type)
    $data = getReportData($params);
    $report_type = $params['report_type'] ?? 'summary';
    $filename = 'gst_notes_' . $report_type . '_' . date('Y-m-d') . '.xls';
    
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    echo "<table border='1'>";
    echo "<tr><th>GST Notes Report</th></tr>";
    echo "<tr><td>Generated: " . date('Y-m-d H:i:s') . "</td></tr>";
    echo "</table>";
}
?>
