<?php
// Handle AJAX requests for report data
if(isset($_POST['action'])) {
    // Include config and report functions
    if (!function_exists('fetchAll')) {
        require_once '../config.php';
    }
    if (!function_exists('getReportData')) {
        require_once '../report_functions.php';
    }
    
    try {
        header('Content-Type: application/json');
        
        switch($_POST['action']) {
            case 'get_report_data':
                $result = getReportData($_POST);
                echo json_encode($result);
                exit;
            case 'export_excel':
                exportToExcel($_POST);
                exit;
            case 'export_pdf':
                exportToPDF($_POST);
                exit;
            case 'export_csv':
                exportToCSV($_POST);
                exit;
        }
    } catch (Exception $e) {
        header('Content-Type: application/json');
        echo json_encode([
            'error' => true,
            'message' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ]);
        exit;
    }
}

// All reporting functions are now in ../report_functions.php
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>GST Notes Reports</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css" rel="stylesheet">
    <style>
        body {
            background-color: #f8f9fa;
        }
        .stats-card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            transition: transform 0.2s;
        }
        .stats-card:hover {
            transform: translateY(-2px);
        }
        .report-container {
            background: white;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .table th {
            background-color: #f8f9fa;
            font-weight: 600;
            border-top: none;
        }
        .btn-group-export .btn {
            border-radius: 0;
        }
        .btn-group-export .btn:first-child {
            border-top-left-radius: 0.375rem;
            border-bottom-left-radius: 0.375rem;
        }
        .btn-group-export .btn:last-child {
            border-top-right-radius: 0.375rem;
            border-bottom-right-radius: 0.375rem;
        }
        .filter-section {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .loading {
            display: none;
            text-align: center;
            padding: 50px;
        }
        .loading i {
            font-size: 3rem;
            color: #007bff;
            animation: spin 1s linear infinite;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="container-fluid mt-4">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h2><i class="fas fa-chart-bar text-primary me-2"></i>GST Notes Reports</h2>
                <small class="text-muted">Generate comprehensive reports for Credit and Debit Notes</small>
            </div>
            <div>
                <button type="button" class="btn btn-outline-secondary" onclick="window.history.back()">
                    <i class="fas fa-arrow-left me-1"></i>Back
                </button>
            </div>
        </div>

        <!-- Filters -->
        <div class="filter-section">
            <form id="reportForm">
                <div class="row">
                    <div class="col-md-3 mb-3">
                        <label for="reportType" class="form-label">Report Type</label>
                        <select class="form-select" id="reportType" name="report_type">
                            <option value="summary">Summary Report</option>
                            <option value="detailed">Detailed Report</option>
                            <option value="gst_summary">GST Summary</option>
                            <option value="party_wise">Party Wise Report</option>
                            <option value="monthly">Monthly Report</option>
                        </select>
                    </div>
                    <div class="col-md-2 mb-3">
                        <label for="noteType" class="form-label">Note Type</label>
                        <select class="form-select" id="noteType" name="note_type">
                            <option value="both">Both</option>
                            <option value="credit">Credit Notes</option>
                            <option value="debit">Debit Notes</option>
                        </select>
                    </div>
                    <div class="col-md-2 mb-3">
                        <label for="transactionType" class="form-label">Transaction</label>
                        <select class="form-select" id="transactionType" name="transaction_type">
                            <option value="both">Both</option>
                            <option value="sales">Sales</option>
                            <option value="purchase">Purchase</option>
                        </select>
                    </div>
                    <div class="col-md-3 mb-3">
                        <label for="dateRange" class="form-label">Date Range</label>
                        <input type="text" class="form-control" id="dateRange" name="date_range" 
                               value="<?= date('01/m/Y') ?> - <?= date('t/m/Y') ?>">
                    </div>
                    <div class="col-md-2 mb-3">
                        <label class="form-label">&nbsp;</label>
                        <div>
                            <button type="button" class="btn btn-primary w-100" onclick="generateReport()">
                                <i class="fas fa-search me-1"></i>Generate
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="row" id="additionalFilters" style="display: none;">
                    <div class="col-md-3 mb-3">
                        <label for="partyName" class="form-label">Party Name (Optional)</label>
                        <input type="text" class="form-control" id="partyName" name="party_name" 
                               placeholder="Search by party name...">
                    </div>
                    <div class="col-md-2 mb-3" id="groupBySection" style="display: none;">
                        <label for="groupBy" class="form-label">Group By</label>
                        <select class="form-select" id="groupBy" name="group_by">
                            <option value="date">Date</option>
                            <option value="month">Month</option>
                            <option value="party">Party</option>
                            <option value="gst_rate">GST Rate</option>
                        </select>
                    </div>
                </div>
            </form>
        </div>

        <!-- Export Buttons -->
        <div class="d-flex justify-content-between align-items-center mb-3" id="exportSection" style="display: none;">
            <div>
                <span class="text-muted" id="resultCount"></span>
            </div>
            <div class="btn-group btn-group-export">
                <button type="button" class="btn btn-success btn-sm" onclick="exportReport('excel')">
                    <i class="fas fa-file-excel me-1"></i>Excel
                </button>
                <button type="button" class="btn btn-danger btn-sm" onclick="exportReport('pdf')">
                    <i class="fas fa-file-pdf me-1"></i>PDF
                </button>
                <button type="button" class="btn btn-info btn-sm" onclick="exportReport('csv')">
                    <i class="fas fa-file-csv me-1"></i>CSV
                </button>
            </div>
        </div>

        <!-- Loading -->
        <div class="loading" id="loading">
            <i class="fas fa-spinner"></i>
            <p class="mt-3">Generating report...</p>
        </div>

        <!-- Report Results -->
        <div class="report-container p-4" id="reportResults" style="display: none;">
            <div id="reportContent"></div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/moment/moment.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script>

    <script>
        // Initialize date range picker
        $('#dateRange').daterangepicker({
            ranges: {
                'Today': [moment(), moment()],
                'Yesterday': [moment().subtract(1, 'days'), moment().subtract(1, 'days')],
                'Last 7 Days': [moment().subtract(6, 'days'), moment()],
                'Last 30 Days': [moment().subtract(29, 'days'), moment()],
                'This Month': [moment().startOf('month'), moment().endOf('month')],
                'Last Month': [moment().subtract(1, 'month').startOf('month'), moment().subtract(1, 'month').endOf('month')],
                'This Quarter': [moment().startOf('quarter'), moment().endOf('quarter')],
                'This Year': [moment().startOf('year'), moment().endOf('year')]
            },
            opens: 'left',
            format: 'DD/MM/YYYY'
        });

        // Show/hide additional filters based on report type
        $('#reportType').change(function() {
            const reportType = $(this).val();
            
            if(reportType === 'detailed') {
                $('#additionalFilters').show();
                $('#groupBySection').hide();
            } else if(reportType === 'gst_summary') {
                $('#additionalFilters').show();
                $('#groupBySection').show();
            } else {
                $('#additionalFilters').hide();
                $('#groupBySection').hide();
            }
        });

        let currentReportData = null;

        function generateReport() {
            const formData = new FormData($('#reportForm')[0]);
            
            // Parse date range
            const dateRange = $('#dateRange').val().split(' - ');
            formData.append('from_date', moment(dateRange[0], 'DD/MM/YYYY').format('YYYY-MM-DD'));
            formData.append('to_date', moment(dateRange[1], 'DD/MM/YYYY').format('YYYY-MM-DD'));
            formData.append('action', 'get_report_data');

            $('#loading').show();
            $('#reportResults').hide();
            $('#exportSection').hide();

            $.ajax({
                url: '',
                method: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                dataType: 'json',
                success: function(response) {
                    $('#loading').hide();
                    
                    // Check if response contains an error
                    if (response.error) {
                        alert('Error generating report: ' + response.message);
                        return;
                    }
                    
                    currentReportData = response;
                    displayReport(response, formData.get('report_type'));
                    $('#reportResults').show();
                    $('#exportSection').show();
                    updateResultCount(response, formData.get('report_type'));
                },
                error: function(xhr, status, error) {
                    $('#loading').hide();
                    console.log('XHR Response:', xhr.responseText);
                    
                    let errorMessage = 'Error generating report: ' + error;
                    
                    // Try to parse the response to get a better error message
                    try {
                        let response = JSON.parse(xhr.responseText);
                        if (response.error) {
                            errorMessage = 'Error: ' + response.message;
                        }
                    } catch (e) {
                        // If it's not JSON, check if it's HTML with an error
                        if (xhr.responseText.includes('<!DOCTYPE') || xhr.responseText.includes('<html>')) {
                            errorMessage = 'Server error occurred. Please check the browser console for details.';
                        }
                    }
                    
                    alert(errorMessage);
                }
            });
        }

        function displayReport(data, reportType) {
            let html = '';
            
            switch(reportType) {
                case 'summary':
                    html = displaySummaryReport(data);
                    break;
                case 'detailed':
                    html = displayDetailedReport(data);
                    break;
                case 'gst_summary':
                case 'monthly':
                    html = displayGSTSummaryReport(data);
                    break;
                case 'party_wise':
                    html = displayPartyWiseReport(data);
                    break;
            }
            
            $('#reportContent').html(html);
        }

        function displaySummaryReport(data) {
            return `
                <h4><i class="fas fa-chart-pie me-2"></i>Summary Report</h4>
                <div class="row">
                    <div class="col-md-6">
                        <div class="card stats-card">
                            <div class="card-body">
                                <h5>Overall Statistics</h5>
                                <table class="table table-sm">
                                    <tr><th>Total Notes</th><td class="text-end">${data.total_notes}</td></tr>
                                    <tr><th>Total Amount</th><td class="text-end">₹${numberFormat(data.total_amount)}</td></tr>
                                    <tr><th>Total Taxable</th><td class="text-end">₹${numberFormat(data.total_taxable)}</td></tr>
                                    <tr><th>Total CGST</th><td class="text-end">₹${numberFormat(data.total_cgst)}</td></tr>
                                    <tr><th>Total SGST</th><td class="text-end">₹${numberFormat(data.total_sgst)}</td></tr>
                                    <tr><th>Total IGST</th><td class="text-end">₹${numberFormat(data.total_igst)}</td></tr>
                                </table>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card stats-card">
                            <div class="card-body">
                                <h5>Breakdown by Type</h5>
                                <table class="table table-sm">
                                    <tr><th>Credit Notes - Sales</th><td class="text-end">₹${numberFormat(data.breakdown.credit_sales || 0)}</td></tr>
                                    <tr><th>Credit Notes - Purchase</th><td class="text-end">₹${numberFormat(data.breakdown.credit_purchase || 0)}</td></tr>
                                    <tr><th>Debit Notes - Sales</th><td class="text-end">₹${numberFormat(data.breakdown.debit_sales || 0)}</td></tr>
                                    <tr><th>Debit Notes - Purchase</th><td class="text-end">₹${numberFormat(data.breakdown.debit_purchase || 0)}</td></tr>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            `;
        }

        function displayDetailedReport(data) {
            let html = `
                <h4><i class="fas fa-list me-2"></i>Detailed Report</h4>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>Category</th>
                                <th>Note Number</th>
                                <th>Date</th>
                                <th>Type</th>
                                <th>Party Name</th>
                                <th>GSTIN</th>
                                <th>Original Invoice</th>
                                <th class="text-end">Taxable</th>
                                <th class="text-end">CGST</th>
                                <th class="text-end">SGST</th>
                                <th class="text-end">IGST</th>
                                <th class="text-end">Total</th>
                            </tr>
                        </thead>
                        <tbody>
            `;

            let totalTaxable = 0, totalCgst = 0, totalSgst = 0, totalIgst = 0, totalAmount = 0;

            data.forEach(row => {
                html += `
                    <tr>
                        <td><span class="badge bg-${row.note_category === 'credit' ? 'success' : 'danger'}">${row.note_category.toUpperCase()}</span></td>
                        <td><strong>${row.note_number}</strong></td>
                        <td>${formatDate(row.note_date)}</td>
                        <td><span class="badge bg-${row.note_type === 'sales' ? 'primary' : 'warning'}">${row.note_type.toUpperCase()}</span></td>
                        <td>${row.party_name}</td>
                        <td>${row.party_gstin || 'N/A'}</td>
                        <td>${row.original_invoice_no}</td>
                        <td class="text-end">₹${numberFormat(row.taxable_amount)}</td>
                        <td class="text-end">₹${numberFormat(row.cgst_amount)}</td>
                        <td class="text-end">₹${numberFormat(row.sgst_amount)}</td>
                        <td class="text-end">₹${numberFormat(row.igst_amount)}</td>
                        <td class="text-end"><strong>₹${numberFormat(row.total_amount)}</strong></td>
                    </tr>
                `;

                totalTaxable += parseFloat(row.taxable_amount);
                totalCgst += parseFloat(row.cgst_amount);
                totalSgst += parseFloat(row.sgst_amount);
                totalIgst += parseFloat(row.igst_amount);
                totalAmount += parseFloat(row.total_amount);
            });

            html += `
                        </tbody>
                        <tfoot>
                            <tr class="table-dark">
                                <th colspan="7">TOTAL</th>
                                <th class="text-end">₹${numberFormat(totalTaxable)}</th>
                                <th class="text-end">₹${numberFormat(totalCgst)}</th>
                                <th class="text-end">₹${numberFormat(totalSgst)}</th>
                                <th class="text-end">₹${numberFormat(totalIgst)}</th>
                                <th class="text-end">₹${numberFormat(totalAmount)}</th>
                            </tr>
                        </tfoot>
                    </table>
                </div>
            `;

            return html;
        }

        function displayGSTSummaryReport(data) {
            let html = `
                <h4><i class="fas fa-chart-bar me-2"></i>GST Summary Report</h4>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>Category</th>
                                <th>Group</th>
                                <th>Type</th>
                                <th class="text-end">Count</th>
                                <th class="text-end">Taxable</th>
                                <th class="text-end">CGST</th>
                                <th class="text-end">SGST</th>
                                <th class="text-end">IGST</th>
                                <th class="text-end">Total</th>
                            </tr>
                        </thead>
                        <tbody>
            `;

            data.forEach(row => {
                html += `
                    <tr>
                        <td><span class="badge bg-${row.note_category === 'credit' ? 'success' : 'danger'}">${row.note_category.toUpperCase()}</span></td>
                        <td>${row.group_key}</td>
                        <td><span class="badge bg-${row.note_type === 'sales' ? 'primary' : 'warning'}">${row.note_type.toUpperCase()}</span></td>
                        <td class="text-end">${row.note_count}</td>
                        <td class="text-end">₹${numberFormat(row.total_taxable)}</td>
                        <td class="text-end">₹${numberFormat(row.total_cgst)}</td>
                        <td class="text-end">₹${numberFormat(row.total_sgst)}</td>
                        <td class="text-end">₹${numberFormat(row.total_igst)}</td>
                        <td class="text-end"><strong>₹${numberFormat(row.total_amount)}</strong></td>
                    </tr>
                `;
            });

            html += '</tbody></table></div>';
            return html;
        }

        function displayPartyWiseReport(data) {
            let html = `
                <h4><i class="fas fa-users me-2"></i>Party Wise Report</h4>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>Category</th>
                                <th>Party Name</th>
                                <th>GSTIN</th>
                                <th>Type</th>
                                <th class="text-end">Count</th>
                                <th class="text-end">Taxable</th>
                                <th class="text-end">CGST</th>
                                <th class="text-end">SGST</th>
                                <th class="text-end">IGST</th>
                                <th class="text-end">Total</th>
                            </tr>
                        </thead>
                        <tbody>
            `;

            data.forEach(row => {
                html += `
                    <tr>
                        <td><span class="badge bg-${row.note_category === 'credit' ? 'success' : 'danger'}">${row.note_category.toUpperCase()}</span></td>
                        <td>${row.party_name}</td>
                        <td>${row.party_gstin || 'N/A'}</td>
                        <td><span class="badge bg-${row.note_type === 'sales' ? 'primary' : 'warning'}">${row.note_type.toUpperCase()}</span></td>
                        <td class="text-end">${row.note_count}</td>
                        <td class="text-end">₹${numberFormat(row.total_taxable)}</td>
                        <td class="text-end">₹${numberFormat(row.total_cgst)}</td>
                        <td class="text-end">₹${numberFormat(row.total_sgst)}</td>
                        <td class="text-end">₹${numberFormat(row.total_igst)}</td>
                        <td class="text-end"><strong>₹${numberFormat(row.total_amount)}</strong></td>
                    </tr>
                `;
            });

            html += '</tbody></table></div>';
            return html;
        }

        function updateResultCount(data, reportType) {
            let count = 0;
            if(reportType === 'summary') {
                count = data.total_notes || 0;
                $('#resultCount').text(`Total ${count} notes found`);
            } else if(Array.isArray(data)) {
                count = data.length;
                $('#resultCount').text(`${count} records found`);
            }
        }

        function exportReport(format) {
            if(!currentReportData) {
                alert('Please generate a report first');
                return;
            }

            const formData = new FormData($('#reportForm')[0]);
            const dateRange = $('#dateRange').val().split(' - ');
            formData.append('from_date', moment(dateRange[0], 'DD/MM/YYYY').format('YYYY-MM-DD'));
            formData.append('to_date', moment(dateRange[1], 'DD/MM/YYYY').format('YYYY-MM-DD'));
            formData.append('action', 'export_' + format);

            // Create a temporary form to submit
            const tempForm = $('<form>', {
                method: 'POST',
                action: '',
                style: 'display: none;'
            });

            formData.forEach((value, key) => {
                tempForm.append($('<input>', {
                    type: 'hidden',
                    name: key,
                    value: value
                }));
            });

            $('body').append(tempForm);
            tempForm.submit();
            tempForm.remove();
        }

        // Utility functions
        function numberFormat(num) {
            return parseFloat(num).toLocaleString('en-IN', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            });
        }

        function formatDate(dateStr) {
            return moment(dateStr).format('DD-MM-YYYY');
        }

        // Initialize
        $(document).ready(function() {
            // Generate initial summary report
            generateReport();
        });
    </script>
</body>
</html>
