<?php
// GST Return Reports - Integration with main billing system
// This file provides GSTR-1 and GSTR-2 reports for Credit/Debit Notes

// Handle AJAX requests for GST return data
if(isset($_POST['action']) && $_POST['action'] === 'get_gstr_data') {
    header('Content-Type: application/json');
    
    $return_type = $_POST['return_type'] ?? 'gstr1';
    $from_date = $_POST['from_date'] ?? date('Y-m-01');
    $to_date = $_POST['to_date'] ?? date('Y-m-t');
    
    $data = [];
    
    switch($return_type) {
        case 'gstr1':
            $data = getGSTR1Data($from_date, $to_date);
            break;
        case 'gstr2':
            $data = getGSTR2Data($from_date, $to_date);
            break;
        case 'gstr3b':
            $data = getGSTR3BData($from_date, $to_date);
            break;
    }
    
    echo json_encode($data);
    exit;
}

function getGSTR1Data($from_date, $to_date) {
    // GSTR-1 - Outward Supplies (Sales)
    // Credit Notes and Debit Notes for Sales transactions
    
    $data = [
        'cdnr' => [], // Credit/Debit notes - Registered persons
        'cdnur' => [], // Credit/Debit notes - Unregistered persons  
        'summary' => [
            'total_credit_notes' => 0,
            'total_debit_notes' => 0,
            'total_credit_amount' => 0,
            'total_debit_amount' => 0,
            'net_effect' => 0
        ]
    ];
    
    // Get Credit Notes for Sales (GSTR-1 - Outward)
    $credit_notes = fetchAll("
        SELECT * FROM credit_notes 
        WHERE note_type = 'sales' 
        AND note_date BETWEEN ? AND ?
        ORDER BY party_gstin, note_date
    ", [$from_date, $to_date]);
    
    // Get Debit Notes for Sales (GSTR-1 - Outward) 
    $debit_notes = fetchAll("
        SELECT * FROM debit_notes 
        WHERE note_type = 'sales' 
        AND note_date BETWEEN ? AND ?
        ORDER BY party_gstin, note_date
    ", [$from_date, $to_date]);
    
    // Process Credit Notes
    foreach($credit_notes as $note) {
        $is_registered = !empty($note['party_gstin']) && strlen($note['party_gstin']) === 15;
        $key = $is_registered ? 'cdnr' : 'cdnur';
        
        $gst_rate = 0;
        if($note['igst_rate'] > 0) {
            $gst_rate = $note['igst_rate'];
        } else {
            $gst_rate = $note['cgst_rate'] + $note['sgst_rate'];
        }
        
        $data[$key][] = [
            'note_type' => 'C',
            'note_number' => $note['note_number'],
            'note_date' => $note['note_date'],
            'party_name' => $note['party_name'],
            'party_gstin' => $note['party_gstin'],
            'original_invoice_no' => $note['original_invoice_no'],
            'original_invoice_date' => $note['original_invoice_date'],
            'reason' => $note['reason'],
            'taxable_value' => $note['taxable_amount'],
            'gst_rate' => $gst_rate,
            'cgst_amount' => $note['cgst_amount'],
            'sgst_amount' => $note['sgst_amount'],
            'igst_amount' => $note['igst_amount'],
            'total_amount' => $note['total_amount']
        ];
        
        $data['summary']['total_credit_notes']++;
        $data['summary']['total_credit_amount'] += $note['total_amount'];
    }
    
    // Process Debit Notes
    foreach($debit_notes as $note) {
        $is_registered = !empty($note['party_gstin']) && strlen($note['party_gstin']) === 15;
        $key = $is_registered ? 'cdnr' : 'cdnur';
        
        $gst_rate = 0;
        if($note['igst_rate'] > 0) {
            $gst_rate = $note['igst_rate'];
        } else {
            $gst_rate = $note['cgst_rate'] + $note['sgst_rate'];
        }
        
        $data[$key][] = [
            'note_type' => 'D',
            'note_number' => $note['note_number'],
            'note_date' => $note['note_date'],
            'party_name' => $note['party_name'],
            'party_gstin' => $note['party_gstin'],
            'original_invoice_no' => $note['original_invoice_no'],
            'original_invoice_date' => $note['original_invoice_date'],
            'reason' => $note['reason'],
            'taxable_value' => $note['taxable_amount'],
            'gst_rate' => $gst_rate,
            'cgst_amount' => $note['cgst_amount'],
            'sgst_amount' => $note['sgst_amount'],
            'igst_amount' => $note['igst_amount'],
            'total_amount' => $note['total_amount']
        ];
        
        $data['summary']['total_debit_notes']++;
        $data['summary']['total_debit_amount'] += $note['total_amount'];
    }
    
    // Calculate net effect (Debit - Credit for outward supplies)
    $data['summary']['net_effect'] = $data['summary']['total_debit_amount'] - $data['summary']['total_credit_amount'];
    
    return $data;
}

function getGSTR2Data($from_date, $to_date) {
    // GSTR-2 - Inward Supplies (Purchase)
    // Credit Notes and Debit Notes for Purchase transactions
    
    $data = [
        'cdnr' => [], // Credit/Debit notes - Registered persons
        'cdnur' => [], // Credit/Debit notes - Unregistered persons
        'summary' => [
            'total_credit_notes' => 0,
            'total_debit_notes' => 0,
            'total_credit_amount' => 0,
            'total_debit_amount' => 0,
            'net_effect' => 0
        ]
    ];
    
    // Get Credit Notes for Purchase (GSTR-2 - Inward)
    $credit_notes = fetchAll("
        SELECT * FROM credit_notes 
        WHERE note_type = 'purchase' 
        AND note_date BETWEEN ? AND ?
        ORDER BY party_gstin, note_date
    ", [$from_date, $to_date]);
    
    // Get Debit Notes for Purchase (GSTR-2 - Inward)
    $debit_notes = fetchAll("
        SELECT * FROM debit_notes 
        WHERE note_type = 'purchase' 
        AND note_date BETWEEN ? AND ?
        ORDER BY party_gstin, note_date
    ", [$from_date, $to_date]);
    
    // Process Credit Notes
    foreach($credit_notes as $note) {
        $is_registered = !empty($note['party_gstin']) && strlen($note['party_gstin']) === 15;
        $key = $is_registered ? 'cdnr' : 'cdnur';
        
        $gst_rate = 0;
        if($note['igst_rate'] > 0) {
            $gst_rate = $note['igst_rate'];
        } else {
            $gst_rate = $note['cgst_rate'] + $note['sgst_rate'];
        }
        
        $data[$key][] = [
            'note_type' => 'C',
            'note_number' => $note['note_number'],
            'note_date' => $note['note_date'],
            'party_name' => $note['party_name'],
            'party_gstin' => $note['party_gstin'],
            'original_invoice_no' => $note['original_invoice_no'],
            'original_invoice_date' => $note['original_invoice_date'],
            'reason' => $note['reason'],
            'taxable_value' => $note['taxable_amount'],
            'gst_rate' => $gst_rate,
            'cgst_amount' => $note['cgst_amount'],
            'sgst_amount' => $note['sgst_amount'],
            'igst_amount' => $note['igst_amount'],
            'total_amount' => $note['total_amount']
        ];
        
        $data['summary']['total_credit_notes']++;
        $data['summary']['total_credit_amount'] += $note['total_amount'];
    }
    
    // Process Debit Notes
    foreach($debit_notes as $note) {
        $is_registered = !empty($note['party_gstin']) && strlen($note['party_gstin']) === 15;
        $key = $is_registered ? 'cdnr' : 'cdnur';
        
        $gst_rate = 0;
        if($note['igst_rate'] > 0) {
            $gst_rate = $note['igst_rate'];
        } else {
            $gst_rate = $note['cgst_rate'] + $note['sgst_rate'];
        }
        
        $data[$key][] = [
            'note_type' => 'D',
            'note_number' => $note['note_number'],
            'note_date' => $note['note_date'],
            'party_name' => $note['party_name'],
            'party_gstin' => $note['party_gstin'],
            'original_invoice_no' => $note['original_invoice_no'],
            'original_invoice_date' => $note['original_invoice_date'],
            'reason' => $note['reason'],
            'taxable_value' => $note['taxable_amount'],
            'gst_rate' => $gst_rate,
            'cgst_amount' => $note['cgst_amount'],
            'sgst_amount' => $note['sgst_amount'],
            'igst_amount' => $note['igst_amount'],
            'total_amount' => $note['total_amount']
        ];
        
        $data['summary']['total_debit_notes']++;
        $data['summary']['total_debit_amount'] += $note['total_amount'];
    }
    
    // Calculate net effect (Credit - Debit for inward supplies)
    $data['summary']['net_effect'] = $data['summary']['total_credit_amount'] - $data['summary']['total_debit_amount'];
    
    return $data;
}

function getGSTR3BData($from_date, $to_date) {
    // GSTR-3B Summary for Credit/Debit Notes
    
    $data = [
        'outward_supplies' => [
            'credit_notes' => 0,
            'debit_notes' => 0,
            'net_adjustment' => 0
        ],
        'inward_supplies' => [
            'credit_notes' => 0,
            'debit_notes' => 0,
            'net_adjustment' => 0
        ],
        'tax_summary' => [
            'cgst' => 0,
            'sgst' => 0,
            'igst' => 0,
            'total' => 0
        ],
        'net_liability' => 0
    ];
    
    // Outward Supplies (Sales)
    $sales_credit = fetchOne("
        SELECT 
            COALESCE(SUM(total_amount), 0) as total,
            COALESCE(SUM(cgst_amount), 0) as cgst,
            COALESCE(SUM(sgst_amount), 0) as sgst,
            COALESCE(SUM(igst_amount), 0) as igst
        FROM credit_notes 
        WHERE note_type = 'sales' AND note_date BETWEEN ? AND ?
    ", [$from_date, $to_date]);
    
    $sales_debit = fetchOne("
        SELECT 
            COALESCE(SUM(total_amount), 0) as total,
            COALESCE(SUM(cgst_amount), 0) as cgst,
            COALESCE(SUM(sgst_amount), 0) as sgst,
            COALESCE(SUM(igst_amount), 0) as igst
        FROM debit_notes 
        WHERE note_type = 'sales' AND note_date BETWEEN ? AND ?
    ", [$from_date, $to_date]);
    
    // Inward Supplies (Purchase)
    $purchase_credit = fetchOne("
        SELECT 
            COALESCE(SUM(total_amount), 0) as total,
            COALESCE(SUM(cgst_amount), 0) as cgst,
            COALESCE(SUM(sgst_amount), 0) as sgst,
            COALESCE(SUM(igst_amount), 0) as igst
        FROM credit_notes 
        WHERE note_type = 'purchase' AND note_date BETWEEN ? AND ?
    ", [$from_date, $to_date]);
    
    $purchase_debit = fetchOne("
        SELECT 
            COALESCE(SUM(total_amount), 0) as total,
            COALESCE(SUM(cgst_amount), 0) as cgst,
            COALESCE(SUM(sgst_amount), 0) as sgst,
            COALESCE(SUM(igst_amount), 0) as igst
        FROM debit_notes 
        WHERE note_type = 'purchase' AND note_date BETWEEN ? AND ?
    ", [$from_date, $to_date]);
    
    // Calculate adjustments
    $data['outward_supplies']['credit_notes'] = $sales_credit['total'];
    $data['outward_supplies']['debit_notes'] = $sales_debit['total'];
    $data['outward_supplies']['net_adjustment'] = $sales_debit['total'] - $sales_credit['total'];
    
    $data['inward_supplies']['credit_notes'] = $purchase_credit['total'];
    $data['inward_supplies']['debit_notes'] = $purchase_debit['total'];
    $data['inward_supplies']['net_adjustment'] = $purchase_credit['total'] - $purchase_debit['total'];
    
    // Tax summary (Net effect on tax liability)
    $data['tax_summary']['cgst'] = ($sales_debit['cgst'] - $sales_credit['cgst']) - ($purchase_credit['cgst'] - $purchase_debit['cgst']);
    $data['tax_summary']['sgst'] = ($sales_debit['sgst'] - $sales_credit['sgst']) - ($purchase_credit['sgst'] - $purchase_debit['sgst']);
    $data['tax_summary']['igst'] = ($sales_debit['igst'] - $sales_credit['igst']) - ($purchase_credit['igst'] - $purchase_debit['igst']);
    $data['tax_summary']['total'] = $data['tax_summary']['cgst'] + $data['tax_summary']['sgst'] + $data['tax_summary']['igst'];
    
    // Net liability adjustment
    $data['net_liability'] = $data['tax_summary']['total'];
    
    return $data;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>GST Returns - Credit & Debit Notes</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css" rel="stylesheet">
    <style>
        body { background-color: #f8f9fa; }
        .gst-container {
            background: white;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            padding: 30px;
            margin-bottom: 20px;
        }
        .return-tabs .nav-link {
            color: #666;
            border: 2px solid transparent;
            border-radius: 10px;
            margin-right: 10px;
            font-weight: 600;
        }
        .return-tabs .nav-link.active {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-color: transparent;
        }
        .summary-card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            transition: transform 0.2s;
        }
        .summary-card:hover { transform: translateY(-2px); }
        .table th { 
            background-color: #f8f9fa; 
            font-weight: 600; 
            border-top: none;
            font-size: 0.875rem;
        }
        .gstr-section {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .loading {
            display: none;
            text-align: center;
            padding: 50px;
        }
        .loading i {
            font-size: 3rem;
            color: #007bff;
            animation: spin 1s linear infinite;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        .tax-positive { color: #dc3545; }
        .tax-negative { color: #28a745; }
        .export-json {
            background: #17a2b8;
            color: white;
            border: none;
        }
    </style>
</head>
<body>
    <div class="container-fluid mt-4">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h2><i class="fas fa-file-alt text-primary me-2"></i>GST Returns - Credit & Debit Notes</h2>
                <small class="text-muted">Generate GSTR-1, GSTR-2, and GSTR-3B reports for Credit/Debit Notes</small>
            </div>
            <div>
                <button type="button" class="btn btn-outline-secondary" onclick="window.history.back()">
                    <i class="fas fa-arrow-left me-1"></i>Back to Reports
                </button>
            </div>
        </div>

        <div class="gst-container">
            <!-- Date Range Selection -->
            <div class="row mb-4">
                <div class="col-md-4">
                    <label for="returnPeriod" class="form-label">Return Period</label>
                    <input type="text" class="form-control" id="returnPeriod" 
                           value="<?= date('01/m/Y') ?> - <?= date('t/m/Y') ?>">
                </div>
                <div class="col-md-2">
                    <label class="form-label">&nbsp;</label>
                    <div>
                        <button type="button" class="btn btn-primary w-100" onclick="loadGSTData()">
                            <i class="fas fa-sync me-1"></i>Refresh
                        </button>
                    </div>
                </div>
            </div>

            <!-- GST Return Tabs -->
            <ul class="nav nav-pills return-tabs mb-4" id="gstTabs" role="tablist">
                <li class="nav-item" role="presentation">
                    <button class="nav-link active" id="gstr1-tab" data-bs-toggle="pill" 
                            data-bs-target="#gstr1" type="button" role="tab">
                        <i class="fas fa-arrow-up me-1"></i>GSTR-1 (Outward)
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="gstr2-tab" data-bs-toggle="pill" 
                            data-bs-target="#gstr2" type="button" role="tab">
                        <i class="fas fa-arrow-down me-1"></i>GSTR-2 (Inward)
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="gstr3b-tab" data-bs-toggle="pill" 
                            data-bs-target="#gstr3b" type="button" role="tab">
                        <i class="fas fa-calculator me-1"></i>GSTR-3B Summary
                    </button>
                </li>
            </ul>

            <!-- Loading -->
            <div class="loading" id="loading">
                <i class="fas fa-spinner"></i>
                <p class="mt-3">Loading GST return data...</p>
            </div>

            <!-- Tab Content -->
            <div class="tab-content" id="gstTabContent">
                <!-- GSTR-1 -->
                <div class="tab-pane fade show active" id="gstr1" role="tabpanel">
                    <div id="gstr1Content"></div>
                </div>

                <!-- GSTR-2 -->
                <div class="tab-pane fade" id="gstr2" role="tabpanel">
                    <div id="gstr2Content"></div>
                </div>

                <!-- GSTR-3B -->
                <div class="tab-pane fade" id="gstr3b" role="tabpanel">
                    <div id="gstr3bContent"></div>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/moment/moment.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script>

    <script>
        // Initialize date range picker
        $('#returnPeriod').daterangepicker({
            ranges: {
                'This Month': [moment().startOf('month'), moment().endOf('month')],
                'Last Month': [moment().subtract(1, 'month').startOf('month'), moment().subtract(1, 'month').endOf('month')],
                'This Quarter': [moment().startOf('quarter'), moment().endOf('quarter')],
                'Last Quarter': [moment().subtract(1, 'quarter').startOf('quarter'), moment().subtract(1, 'quarter').endOf('quarter')],
                'This Year': [moment().startOf('year'), moment().endOf('year')],
                'Last Year': [moment().subtract(1, 'year').startOf('year'), moment().subtract(1, 'year').endOf('year')]
            },
            opens: 'left',
            format: 'DD/MM/YYYY'
        });

        let gstData = {};

        function loadGSTData() {
            const dateRange = $('#returnPeriod').val().split(' - ');
            const fromDate = moment(dateRange[0], 'DD/MM/YYYY').format('YYYY-MM-DD');
            const toDate = moment(dateRange[1], 'DD/MM/YYYY').format('YYYY-MM-DD');

            $('#loading').show();

            // Load all GST return types
            const promises = [
                loadGSTRData('gstr1', fromDate, toDate),
                loadGSTRData('gstr2', fromDate, toDate),
                loadGSTRData('gstr3b', fromDate, toDate)
            ];

            Promise.all(promises).then(() => {
                $('#loading').hide();
                updateAllTabs();
            }).catch(error => {
                $('#loading').hide();
                alert('Error loading GST data: ' + error);
            });
        }

        function loadGSTRData(returnType, fromDate, toDate) {
            return new Promise((resolve, reject) => {
                $.ajax({
                    url: '',
                    method: 'POST',
                    data: {
                        action: 'get_gstr_data',
                        return_type: returnType,
                        from_date: fromDate,
                        to_date: toDate
                    },
                    dataType: 'json',
                    success: function(response) {
                        gstData[returnType] = response;
                        resolve(response);
                    },
                    error: function(xhr, status, error) {
                        reject(error);
                    }
                });
            });
        }

        function updateAllTabs() {
            if(gstData.gstr1) updateGSTR1Tab(gstData.gstr1);
            if(gstData.gstr2) updateGSTR2Tab(gstData.gstr2);
            if(gstData.gstr3b) updateGSTR3BTab(gstData.gstr3b);
        }

        function updateGSTR1Tab(data) {
            const html = `
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="summary-card card text-center">
                            <div class="card-body">
                                <h5 class="text-danger">Credit Notes</h5>
                                <h3>${data.summary.total_credit_notes}</h3>
                                <p class="text-muted">₹${numberFormat(data.summary.total_credit_amount)}</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="summary-card card text-center">
                            <div class="card-body">
                                <h5 class="text-success">Debit Notes</h5>
                                <h3>${data.summary.total_debit_notes}</h3>
                                <p class="text-muted">₹${numberFormat(data.summary.total_debit_amount)}</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="summary-card card text-center">
                            <div class="card-body">
                                <h5>Net Effect</h5>
                                <h3 class="${data.summary.net_effect >= 0 ? 'text-success' : 'text-danger'}">
                                    ₹${numberFormat(Math.abs(data.summary.net_effect))}
                                </h3>
                                <p class="text-muted">${data.summary.net_effect >= 0 ? 'Additional Tax' : 'Tax Reduction'}</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="summary-card card text-center">
                            <div class="card-body">
                                <button class="btn export-json btn-sm w-100" onclick="downloadJSON('gstr1', gstData.gstr1)">
                                    <i class="fas fa-download me-1"></i>Export JSON
                                </button>
                                <button class="btn btn-success btn-sm w-100 mt-2" onclick="exportToExcel('gstr1')">
                                    <i class="fas fa-file-excel me-1"></i>Export Excel
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                ${generateCDNRTable(data.cdnr, 'Registered Persons')}
                ${data.cdnur.length > 0 ? generateCDNRTable(data.cdnur, 'Unregistered Persons') : ''}
            `;
            
            $('#gstr1Content').html(html);
        }

        function updateGSTR2Tab(data) {
            const html = `
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="summary-card card text-center">
                            <div class="card-body">
                                <h5 class="text-danger">Credit Notes</h5>
                                <h3>${data.summary.total_credit_notes}</h3>
                                <p class="text-muted">₹${numberFormat(data.summary.total_credit_amount)}</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="summary-card card text-center">
                            <div class="card-body">
                                <h5 class="text-success">Debit Notes</h5>
                                <h3>${data.summary.total_debit_notes}</h3>
                                <p class="text-muted">₹${numberFormat(data.summary.total_debit_amount)}</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="summary-card card text-center">
                            <div class="card-body">
                                <h5>Net ITC Effect</h5>
                                <h3 class="${data.summary.net_effect >= 0 ? 'text-success' : 'text-danger'}">
                                    ₹${numberFormat(Math.abs(data.summary.net_effect))}
                                </h3>
                                <p class="text-muted">${data.summary.net_effect >= 0 ? 'ITC Reversal' : 'ITC Claim'}</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="summary-card card text-center">
                            <div class="card-body">
                                <button class="btn export-json btn-sm w-100" onclick="downloadJSON('gstr2', gstData.gstr2)">
                                    <i class="fas fa-download me-1"></i>Export JSON
                                </button>
                                <button class="btn btn-success btn-sm w-100 mt-2" onclick="exportToExcel('gstr2')">
                                    <i class="fas fa-file-excel me-1"></i>Export Excel
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                ${generateCDNRTable(data.cdnr, 'Registered Persons')}
                ${data.cdnur.length > 0 ? generateCDNRTable(data.cdnur, 'Unregistered Persons') : ''}
            `;
            
            $('#gstr2Content').html(html);
        }

        function updateGSTR3BTab(data) {
            const html = `
                <div class="row">
                    <div class="col-md-6">
                        <div class="gstr-section">
                            <h5><i class="fas fa-arrow-up text-primary me-2"></i>Outward Supplies Impact</h5>
                            <table class="table table-sm">
                                <tr>
                                    <th>Credit Notes</th>
                                    <td class="text-end text-danger">₹${numberFormat(data.outward_supplies.credit_notes)}</td>
                                </tr>
                                <tr>
                                    <th>Debit Notes</th>
                                    <td class="text-end text-success">₹${numberFormat(data.outward_supplies.debit_notes)}</td>
                                </tr>
                                <tr class="table-active">
                                    <th>Net Adjustment</th>
                                    <td class="text-end fw-bold ${data.outward_supplies.net_adjustment >= 0 ? 'text-success' : 'text-danger'}">
                                        ₹${numberFormat(Math.abs(data.outward_supplies.net_adjustment))}
                                    </td>
                                </tr>
                            </table>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="gstr-section">
                            <h5><i class="fas fa-arrow-down text-warning me-2"></i>Inward Supplies Impact</h5>
                            <table class="table table-sm">
                                <tr>
                                    <th>Credit Notes</th>
                                    <td class="text-end text-success">₹${numberFormat(data.inward_supplies.credit_notes)}</td>
                                </tr>
                                <tr>
                                    <th>Debit Notes</th>
                                    <td class="text-end text-danger">₹${numberFormat(data.inward_supplies.debit_notes)}</td>
                                </tr>
                                <tr class="table-active">
                                    <th>Net ITC Adjustment</th>
                                    <td class="text-end fw-bold ${data.inward_supplies.net_adjustment >= 0 ? 'text-success' : 'text-danger'}">
                                        ₹${numberFormat(Math.abs(data.inward_supplies.net_adjustment))}
                                    </td>
                                </tr>
                            </table>
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-md-8">
                        <div class="gstr-section">
                            <h5><i class="fas fa-calculator text-info me-2"></i>Tax Liability Impact</h5>
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Tax Type</th>
                                        <th class="text-end">Net Impact</th>
                                        <th class="text-end">Liability Adjustment</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        <td><strong>CGST</strong></td>
                                        <td class="text-end ${data.tax_summary.cgst >= 0 ? 'tax-positive' : 'tax-negative'}">
                                            ₹${numberFormat(Math.abs(data.tax_summary.cgst))}
                                        </td>
                                        <td class="text-end">
                                            ${data.tax_summary.cgst >= 0 ? 'Additional' : 'Reduction'}
                                        </td>
                                    </tr>
                                    <tr>
                                        <td><strong>SGST</strong></td>
                                        <td class="text-end ${data.tax_summary.sgst >= 0 ? 'tax-positive' : 'tax-negative'}">
                                            ₹${numberFormat(Math.abs(data.tax_summary.sgst))}
                                        </td>
                                        <td class="text-end">
                                            ${data.tax_summary.sgst >= 0 ? 'Additional' : 'Reduction'}
                                        </td>
                                    </tr>
                                    <tr>
                                        <td><strong>IGST</strong></td>
                                        <td class="text-end ${data.tax_summary.igst >= 0 ? 'tax-positive' : 'tax-negative'}">
                                            ₹${numberFormat(Math.abs(data.tax_summary.igst))}
                                        </td>
                                        <td class="text-end">
                                            ${data.tax_summary.igst >= 0 ? 'Additional' : 'Reduction'}
                                        </td>
                                    </tr>
                                    <tr class="table-dark">
                                        <td><strong>TOTAL</strong></td>
                                        <td class="text-end fw-bold">
                                            <strong>₹${numberFormat(Math.abs(data.tax_summary.total))}</strong>
                                        </td>
                                        <td class="text-end fw-bold">
                                            <strong>${data.tax_summary.total >= 0 ? 'Additional Liability' : 'Liability Reduction'}</strong>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="gstr-section text-center">
                            <h5>Net Tax Liability Effect</h5>
                            <div class="display-4 ${data.net_liability >= 0 ? 'text-danger' : 'text-success'} mb-3">
                                ₹${numberFormat(Math.abs(data.net_liability))}
                            </div>
                            <p class="lead">
                                ${data.net_liability >= 0 ? 
                                  '<span class="badge bg-danger">Additional Tax Payable</span>' : 
                                  '<span class="badge bg-success">Tax Liability Reduced</span>'}
                            </p>
                            <div class="mt-3">
                                <button class="btn export-json btn-sm" onclick="downloadJSON('gstr3b', gstData.gstr3b)">
                                    <i class="fas fa-download me-1"></i>Export JSON
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            $('#gstr3bContent').html(html);
        }

        function generateCDNRTable(data, title) {
            if(!data || data.length === 0) return '';
            
            let html = `
                <div class="gstr-section">
                    <h5>${title} (${data.length} entries)</h5>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover table-sm">
                            <thead>
                                <tr>
                                    <th>Type</th>
                                    <th>Note Number</th>
                                    <th>Date</th>
                                    <th>Party</th>
                                    <th>GSTIN</th>
                                    <th>Original Invoice</th>
                                    <th class="text-end">Taxable</th>
                                    <th class="text-end">GST Rate</th>
                                    <th class="text-end">CGST</th>
                                    <th class="text-end">SGST</th>
                                    <th class="text-end">IGST</th>
                                    <th class="text-end">Total</th>
                                </tr>
                            </thead>
                            <tbody>
            `;
            
            data.forEach(row => {
                html += `
                    <tr>
                        <td>
                            <span class="badge bg-${row.note_type === 'C' ? 'danger' : 'success'}">
                                ${row.note_type === 'C' ? 'Credit' : 'Debit'}
                            </span>
                        </td>
                        <td><strong>${row.note_number}</strong></td>
                        <td>${formatDate(row.note_date)}</td>
                        <td>${row.party_name}</td>
                        <td><small>${row.party_gstin || 'N/A'}</small></td>
                        <td><small>${row.original_invoice_no}</small></td>
                        <td class="text-end">₹${numberFormat(row.taxable_value)}</td>
                        <td class="text-end">${row.gst_rate}%</td>
                        <td class="text-end">₹${numberFormat(row.cgst_amount)}</td>
                        <td class="text-end">₹${numberFormat(row.sgst_amount)}</td>
                        <td class="text-end">₹${numberFormat(row.igst_amount)}</td>
                        <td class="text-end"><strong>₹${numberFormat(row.total_amount)}</strong></td>
                    </tr>
                `;
            });
            
            html += '</tbody></table></div></div>';
            return html;
        }

        function downloadJSON(returnType, data) {
            const filename = `${returnType.toUpperCase()}_Credit_Debit_Notes_${moment().format('YYYY-MM-DD')}.json`;
            const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
            const url = URL.createObjectURL(blob);
            
            const a = document.createElement('a');
            a.href = url;
            a.download = filename;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
        }

        function exportToExcel(returnType) {
            // This would typically integrate with your existing Excel export functionality
            alert(`Excel export for ${returnType.toUpperCase()} would be implemented here`);
        }

        // Utility functions
        function numberFormat(num) {
            return parseFloat(num).toLocaleString('en-IN', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            });
        }

        function formatDate(dateStr) {
            return moment(dateStr).format('DD-MM-YYYY');
        }

        // Initialize
        $(document).ready(function() {
            loadGSTData();
        });
    </script>
</body>
</html>
