<?php
// Simple test page to debug reporting functions
require_once 'config.php';

// Set error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<!DOCTYPE html>
<html>
<head>
    <title>GST Reports Debug Test</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .success { color: green; }
        .error { color: red; }
        .info { color: blue; }
        pre { background: #f5f5f5; padding: 10px; border: 1px solid #ddd; overflow: auto; }
    </style>
</head>
<body>
    <h2>GST Reports Debug Test</h2>";

echo "<h3>1. Database Connection Test</h3>";
try {
    if ($mysqli->connect_error) {
        throw new Exception("Connection failed: " . $mysqli->connect_error);
    }
    echo "<p class='success'>✓ Database connection OK</p>";
    
    // Test table existence
    $tables = ['credit_notes', 'debit_notes'];
    foreach ($tables as $table) {
        $result = $mysqli->query("SELECT COUNT(*) as count FROM $table");
        if ($result) {
            $count = $result->fetch_assoc()['count'];
            echo "<p class='success'>✓ Table '$table' has $count records</p>";
        } else {
            echo "<p class='error'>✗ Table '$table' query failed</p>";
        }
    }
} catch (Exception $e) {
    echo "<p class='error'>✗ Database error: " . $e->getMessage() . "</p>";
}

echo "<h3>2. Core Function Tests</h3>";
try {
    // Test fetchAll function
    $test_data = fetchAll("SELECT COUNT(*) as total FROM credit_notes UNION ALL SELECT COUNT(*) as total FROM debit_notes");
    echo "<p class='success'>✓ fetchAll() function working</p>";
    echo "<pre>fetchAll result: " . print_r($test_data, true) . "</pre>";
    
    // Test fetchOne function  
    $test_one = fetchOne("SELECT COUNT(*) as count FROM credit_notes");
    echo "<p class='success'>✓ fetchOne() function working</p>";
    echo "<pre>fetchOne result: " . print_r($test_one, true) . "</pre>";
    
} catch (Exception $e) {
    echo "<p class='error'>✗ Core function error: " . $e->getMessage() . "</p>";
}

echo "<h3>3. Report Function Tests</h3>";

// Test if we can include the reports.php functions
ob_start(); // Capture any output
$reports_file = __DIR__ . '/pages/reports.php';
if (file_exists($reports_file)) {
    // Read the file content and extract functions only
    $content = file_get_contents($reports_file);
    
    // Extract just the function definitions
    if (preg_match_all('/function\s+(\w+)\s*\([^}]*\}(?:[^{}]*\{[^}]*\})*[^{}]*\}/ms', $content, $matches)) {
        foreach ($matches[0] as $func) {
            if (strpos($func, 'getReportData') !== false || 
                strpos($func, 'getSummaryReport') !== false ||
                strpos($func, 'getDetailedReport') !== false) {
                // Safely eval the function (only for testing)
                try {
                    eval($func);
                } catch (Exception $e) {
                    echo "<p class='error'>Function definition error: " . $e->getMessage() . "</p>";
                }
            }
        }
    }
}
ob_end_clean(); // Discard captured output

// Now test the functions if they exist
if (function_exists('getSummaryReport')) {
    try {
        $from_date = date('Y-m-01');
        $to_date = date('Y-m-t');
        
        echo "<p class='info'>Testing getSummaryReport() with dates: $from_date to $to_date</p>";
        $summary = getSummaryReport('both', 'both', $from_date, $to_date);
        echo "<p class='success'>✓ getSummaryReport() working</p>";
        echo "<pre>Summary result: " . print_r($summary, true) . "</pre>";
        
    } catch (Exception $e) {
        echo "<p class='error'>✗ getSummaryReport() error: " . $e->getMessage() . "</p>";
    }
} else {
    echo "<p class='error'>✗ getSummaryReport() function not found</p>";
}

echo "<h3>4. AJAX Simulation Test</h3>";
try {
    // Simulate the AJAX call
    $_POST['action'] = 'get_report_data';
    $_POST['report_type'] = 'summary';
    $_POST['note_type'] = 'both';
    $_POST['transaction_type'] = 'both';
    $_POST['from_date'] = date('Y-m-01');
    $_POST['to_date'] = date('Y-m-t');
    
    // Test the AJAX handler logic manually
    if (function_exists('getReportData')) {
        $result = getReportData($_POST);
        echo "<p class='success'>✓ AJAX simulation successful</p>";
        echo "<pre>AJAX result: " . print_r($result, true) . "</pre>";
        
        // Test JSON encoding
        $json = json_encode($result);
        if ($json === false) {
            echo "<p class='error'>✗ JSON encoding failed: " . json_last_error_msg() . "</p>";
        } else {
            echo "<p class='success'>✓ JSON encoding successful</p>";
            echo "<pre>JSON result: " . substr($json, 0, 200) . (strlen($json) > 200 ? '...' : '') . "</pre>";
        }
    } else {
        echo "<p class='error'>✗ getReportData() function not available for testing</p>";
    }
    
} catch (Exception $e) {
    echo "<p class='error'>✗ AJAX simulation error: " . $e->getMessage() . "</p>";
    echo "<p class='error'>Stack trace: " . $e->getTraceAsString() . "</p>";
}

echo "<h3>5. Simple Direct Query Test</h3>";
try {
    // Direct query without functions
    $query = "SELECT 
        'credit' as note_category,
        note_type,
        COUNT(*) as count,
        SUM(taxable_amount) as taxable_amount,
        SUM(cgst_amount) as cgst_amount,
        SUM(sgst_amount) as sgst_amount,
        SUM(igst_amount) as igst_amount,
        SUM(total_amount) as total_amount
    FROM credit_notes 
    WHERE note_date BETWEEN ? AND ?
    GROUP BY note_type";
    
    $from_date = date('Y-m-01');
    $to_date = date('Y-m-t');
    
    $stmt = $mysqli->prepare($query);
    $stmt->bind_param('ss', $from_date, $to_date);
    $stmt->execute();
    $result = $stmt->get_result();
    $data = $result->fetch_all(MYSQLI_ASSOC);
    
    echo "<p class='success'>✓ Direct query test successful</p>";
    echo "<pre>Direct query result: " . print_r($data, true) . "</pre>";
    
} catch (Exception $e) {
    echo "<p class='error'>✗ Direct query error: " . $e->getMessage() . "</p>";
}

echo "<hr>";
echo "<h3>Recommendations:</h3>";
echo "<ul>";
echo "<li>If database connection failed, check WAMP server is running</li>";
echo "<li>If functions are missing, check reports.php file structure</li>";
echo "<li>If JSON encoding failed, check for circular references or invalid data</li>";
echo "<li>Check browser network tab for actual AJAX request/response</li>";
echo "</ul>";

echo "<p><a href='index.php'>← Back to GST Notes</a> | <a href='test_connection.php'>Full System Test</a></p>";

echo "</body></html>";

$mysqli->close();
?>
